// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   PoseAssembly.cc
/// @brief  Insert/delete residues in a Pose maintaining fold tree.
/// @brief  Insertions may be either continuous or inserted with a break.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// unit headers
#include <epigraft/design/PoseAssembly.hh>

// package headers
#include <epigraft/design/ConnectionResult.hh>
#include <epigraft/design/PoseAssemblyInstruction.hh>
#include <epigraft/design/design_functions.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/ResidueRange.hh>

// rosetta headers
#include <aaproperties_pack.h>
#include <fold_tree.h>
#include <jumping_util.h>
#include <param.h>
#include <param_aa.h>
#include <pose.h>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray3D.hh>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <map>
#include <sstream>


namespace epigraft {
namespace design {


/// @brief internal struct for sorting peptide edges of fold tree as fold tree does not give needed ordering
struct PeptideEdge {
	// data
	Integer start;
	Integer stop;

	// '<'
	inline
	bool
	operator <( PeptideEdge const & rval ) const
	{
		return std::min( start, stop ) < std::min( rval.start, rval.stop );
	}
};


/// @brief return instructions for given residue
std::set< PoseAssemblyInstruction >
PoseAssembly::instructions_for_residue(
	Integer const & res
) const
{
	std::set< PoseAssemblyInstruction > instruction_queue;

	// check insert left
	std::set< PoseAssemblyInstruction >::const_iterator iter = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT ) );
	if ( iter != instructions_.end() ) {
		instruction_queue.insert( *iter );
	}

	// check keep
	iter = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
	if ( iter != instructions_.end() ) {
		instruction_queue.insert( *iter );
	}

	// check insert right
	iter = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_RIGHT ) );
	if ( iter != instructions_.end() ) {
		instruction_queue.insert( *iter );
	}

	return instruction_queue;
}


/// @brief calculate old residue to new residue map from existing instructions
/// @note  deleted old residues are mapped to -1
std::map< Integer, Integer >
PoseAssembly::map_old_to_new() const
{
	std::map< Integer, Integer > old_to_new;
	Integer counter = 0; // track new residue numbers

	for ( std::set< PoseAssemblyInstruction >::const_iterator i = instructions_.begin(), ie = instructions_.end(); i != ie; ++i ) {
		PoseAssemblyInstruction const & instruction = *i;

		switch( instruction.action() ) {
			case PoseAssemblyInstruction::KEEP:
				{
					Integer const & residue = instruction.residue();

					// increment counter
					++counter;
					old_to_new[ residue ] = counter;
				}
				break;
			default: // insertion
				{
					// setup for each insert residue
					for ( Integer insert = 1, last_insert = instruction.size(); insert <= last_insert; ++insert ) {
						// increment counter
						++counter;
					}
				}
				break;
		}
	}

	// add all deleted residues
	for ( Integer res = 1; res <= base_total_residues_; ++res ) {
		if ( old_to_new.find( res ) == old_to_new.end() ) {
			old_to_new[ res ] = -1;
		}
	}

	assert( (Integer)old_to_new.size() == base_total_residues_ );

	return old_to_new;
}

/// @brief calculate new residue to old residue map from existing instructions
/// @note  inserted new residues are mapped to -1
std::map< Integer, Integer >
PoseAssembly::map_new_to_old() const
{
	std::map< Integer, Integer > new_to_old;
	Integer counter = 0; // track new residue numbers

	for ( std::set< PoseAssemblyInstruction >::const_iterator i = instructions_.begin(), ie = instructions_.end(); i != ie; ++i ) {
		PoseAssemblyInstruction const & instruction = *i;

		switch( instruction.action() ) {
			case PoseAssemblyInstruction::KEEP:
				{
					Integer const & residue = instruction.residue();

					// increment counter
					++counter;
					new_to_old[ counter ] = residue;
				}
				break;
			default: // insertion
				{
					// setup for each insert residue
					for ( Integer insert = 1, last_insert = instruction.size(); insert <= last_insert; ++insert ) {
						// increment counter
						++counter;
						new_to_old[ counter ] = -1;
					}
				}
				break;
		}
	}

	return new_to_old;
}


/// @brief calculate old cutpoint to new cutpoint map from existing instructions
std::map< Integer, Integer >
PoseAssembly::map_cut_old_to_new(
	Fold_tree const & f
) const
{
	FArray1D< bool > const & is_cutpoint = f.get_is_cutpoint();
	std::set< PoseAssemblyInstruction>::const_iterator ie = instructions_.end(); // cache

	std::map< Integer, Integer > cut_old_to_new;

	Integer offset = 0;
	for ( Integer res = 1; res <= base_total_residues_; ++res ) {
		std::set< PoseAssemblyInstruction >::const_iterator i = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );

		// deletion
		if ( i == ie ) {
			--offset;
		}

		// handle left insert
		i = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT ) );
		if ( i != ie ) {
			offset += i->size();
		}

		// handle right insert
		i = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_RIGHT ) );

		// right insert, insert without break case (pushes cutpoint to right)
		if ( i != ie && !i->break_on_insertion() ) {
			offset += i->size();
		}

		// mark cutpoint
		if ( is_cutpoint( res ) ) {
			cut_old_to_new[ res ] = res + offset;
		}

		// right insert, insert with break case
		if ( i != ie && i->break_on_insertion() ) {
			offset += i->size();
		}
	}

	return cut_old_to_new;
}


/// @brief apply instructions and create new Pose/fold tree
/// @note full atom set true in output Pose
/// @warning all cutpoints in new Pose will be idealized
void
PoseAssembly::apply(
	Pose const & input,
	Pose & output,
	bool const & idealize_cutpoints
) const
{
	using pose_ns::Edge;

	assert( base_total_residues_ == input.total_residue() );

	// create mapping for one-letter code to rosetta residue
	std::map< char, Integer > one_letter_to_residue = get_one_letter_to_rosetta_residue();

	// count number of total residues in new output Pose
	Integer total_residue = 0;
	for ( std::set< PoseAssemblyInstruction >::const_iterator i = instructions_.begin(), ie = instructions_.end(); i != ie; ++i ) { // count keep/insertions
		PoseAssemblyInstruction const & instruction = *i;

		switch( instruction.action() ) {
			case PoseAssemblyInstruction::KEEP:
				++total_residue;
				break;
			case PoseAssemblyInstruction::INSERT_LEFT:
				total_residue += instruction.size();
				break;
			case PoseAssemblyInstruction::INSERT_RIGHT:
				total_residue += instruction.size();
				break;
		}
	}

	// process new fold tree
	Fold_tree f = make_fold_tree( input.fold_tree() );

	// new data for output Pose
	output.set_fold_tree( f );
	output.set_fullatom_flag( true, false ); // booleans: fullatom, repack
	FArray3D_float full_coord( 3, param::MAX_ATOM(), total_residue, 0.0 ); // initialize all coordinates as zero

	// process residues and fill new Pose incrementally
	FArray3D_float const & input_full_coord = input.full_coord();
	Integer counter = 0; // track new residue numbers
	std::map< Integer, Integer > old_to_new_residue;
	std::map< Integer, Integer > new_to_old_residue; // -1 here indicates insert residue, no
	utility::vector1< ResidueRange > inserts; // keep track of inserts to setup proper bonds/angles later

	for ( std::set< PoseAssemblyInstruction >::const_iterator i = instructions_.begin(), ie = instructions_.end(); i != ie; ++i ) {
		PoseAssemblyInstruction const & instruction = *i;

		switch( instruction.action() ) {
			case PoseAssemblyInstruction::KEEP:
				{
					Integer const & residue = instruction.residue();

					// increment counter
					++counter;
					old_to_new_residue[ residue ] = counter;
					new_to_old_residue[ counter ] = residue;

					// transfer identity information
					output.set_res( counter, input.res( residue ) );
					output.set_res_variant( counter, input.res_variant( residue ) );
					output.set_secstruct( counter, input.secstruct( residue ) );
					output.set_name( counter, input.name( residue ) );

					// fill full_coord
					Integer const aa = input.res( residue );
					Integer const aav = input.res_variant( residue );
					for ( Integer atom = 1, last_atom = aaproperties_pack::natoms( aa, aav ); atom <= last_atom; ++atom ) {
						for ( Integer i = 1; i <= 3; ++i ) {
							full_coord( i, atom, counter ) = input_full_coord( i, atom, residue );
						}
					}
				}
				break;
			default: // insertion
				{
					// setup for each insert residue
					char const * ss_chars = instruction.ss_string().c_str();
					char const * residue_chars = instruction.residue_string().c_str();
					for ( Integer insert = 1, last_insert = instruction.size(); insert <= last_insert; ++insert ) {
						// increment counter
						++counter;
						new_to_old_residue[ counter ] = -1;

						// transfer identity information
						output.set_res( counter, one_letter_to_residue[ residue_chars[ insert - 1 ] ] ); // gly insert
						output.set_res_variant( counter, 1 ); // default (starting) variant
						output.set_secstruct( counter, ss_chars[ insert - 1 ] ); // degenerate for now
					}

					// track insert
					inserts.push_back( ResidueRange( counter - instruction.size() + 1, counter ) );
				}
				break;
		}
	}

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, total_residue );
	full_coord_to_Eposition( total_residue, full_coord, Eposition );

	// set coordinates -- check missing is turned on below, this will give warning messages due to
	// missing coordinates for insert residues, but otherwise everything is fine
	output.set_coords( false, Eposition, full_coord, true ); // booleans: ideal_pose, check_missing

	// NOTE: the *ORDER* in which idealization is done in the next few code blocks is important!
	//       (both overall order and the order things are done in each block)
	//       otherwise much messiness is incurred in logic attempting to recover proper phi/psi/omega
	//       and bond geometry
	//       (1) set ideal bonds and torsions for all _new_ residues
	//       (2) set ideal bonds/angles surrounding peptide bonds of residues adjacent to deletions
	//       (3) set ideal bonds/angles surrounding peptide bonds of residues adjacent to inserts
	//       (4) set ideal bonds/angles surrounding peptide bonds of cutpoints
	// set ideal bonds and torsions
	for ( utility::vector1< ResidueRange >::const_iterator i = inserts.begin(), ie = inserts.end(); i != ie; ++i ) {
		ResidueRange const & insert_range = *i;
		output.insert_ideal_bonds( insert_range.begin(), insert_range.end() ); // bonds
		insert_initial_torsions( output, insert_range ); // angles
	}

	// set ideal bonds for residues adjacent to deletions
	for ( Integer i = 1, ie = input.total_residue(); i <= ie; ++i ) {
		if ( instructions_.find( PoseAssemblyInstruction( i, PoseAssemblyInstruction::KEEP ) ) == instructions_.end() ) {
			Integer ia = i;

			// FIRST look left
			ia = i - 1;
			if ( ia >= 1 && ( old_to_new_residue.find( ia ) != old_to_new_residue.end() ) ) {
				Integer const left_residue = old_to_new_residue[ ia ];
				// the n-side phi in the next function is potentially fake, depending upon the instructions given
				// (e.g. if a section is deleted without insertion ), but this will be resolved when the right
				// side is handled when "looking right" in the next 'if' block
				idealize_cutpoint_retaining_psiphi( output, left_residue, input.psi( ia ), output.phi( left_residue + 1 ) );
			}

			// SECOND look right
			ia = i + 1;
			if ( ia <= input.total_residue() && ( old_to_new_residue.find( ia ) != old_to_new_residue.end() ) ) {
				Integer const right_residue = old_to_new_residue[ ia ];
				idealize_cutpoint_retaining_psiphi( output, right_residue - 1, output.psi( right_residue - 1 ), input.phi( ia ) );
			}
		}
	}

	// insert ideal bonds for residues adjacent to inserts
	for ( utility::vector1< ResidueRange >::const_iterator i = inserts.begin(), ie = inserts.end(); i != ie; ++i ) {
		ResidueRange const & insert_range = *i;

		std::map< Integer, Integer >::const_iterator not_found = new_to_old_residue.end();
		std::map< Integer, Integer >::const_iterator on_left = new_to_old_residue.find( insert_range.begin() - 1 ); // look left
		std::map< Integer, Integer >::const_iterator on_right = new_to_old_residue.find( insert_range.end() + 1 ); // look right

		// handle gluing on left of insert
		if ( on_left != not_found && on_left->second != -1 ) {
			idealize_cutpoint_retaining_psiphi( output, insert_range.begin() - 1, input.psi( on_left->second ), output.phi( insert_range.begin() ) );
		}

		// handle gluing on right of insert
		if ( on_right != not_found && on_right->second != -1 ) {
			idealize_cutpoint_retaining_psiphi( output, insert_range.end(), output.psi( insert_range.end() ), input.phi( on_right->second ) );
		}
	}

	// create idealization information for idealizing actual cutpoints
	std::map< Integer, Real > cut_to_torsion;
	FArray1D< bool > const & is_output_cutpoint = output.fold_tree().get_is_cutpoint();
	for ( Integer i = 2, ie = output.total_residue() - 1; i <= ie; ++i ) {
		if ( is_output_cutpoint( i ) ) {
			Integer const nside = i;
			Integer const cside = i + 1;

			// check to see which residues are old and new, for old residues take from input
			// and for new residues take from output ( -1 in new_to_old_residue indicates new residue )
			Real const nside_psi = new_to_old_residue[ nside ] == -1 ? output.psi( nside ) : input.psi( new_to_old_residue[ nside ] );
			Real const cside_phi = new_to_old_residue[ cside ] == -1 ? output.phi( cside ) : input.phi( new_to_old_residue[ cside ] );

			// store
			cut_to_torsion[ nside ] = nside_psi;
			cut_to_torsion[ cside ] = cside_phi;
		}
	}

	// idealize both old cutpoints and new cutpoints created by insertions/deletions
	if ( idealize_cutpoints ) {
		idealize_all_cutpoints( output, cut_to_torsion );
	}

	// gather insert positions to fix their hydrogens, temporarily designing
	// alanine as glycine to workaround weird ala hydrogen explosion, also
	// do glycine as alanine for safety
	std::set< Integer > is_ala, is_gly;
	FArray2D< bool > design_matrix( param::MAX_AA(), output.total_residue(), false );
//	FArray1D< bool > allow_repack( output.total_residue(), false );
	for ( utility::vector1< ResidueRange >::const_iterator i = inserts.begin(), ie = inserts.end(); i != ie; ++i ) {
		ResidueRange const & insert_range = *i;
		for ( Integer j = insert_range.begin(), je = insert_range.end(); j <= je; ++j ) {
//			allow_repack( j ) = true;
			if ( output.res( j ) == param_aa::aa_ala ) { // ala as gly
				is_ala.insert( j );
				design_matrix( param_aa::aa_gly, j ) = true;
			} else if ( output.res( j ) == param_aa::aa_gly ) {
				is_gly.insert( j );
				design_matrix( param_aa::aa_ala, j ) = true;
			} else {
				design_matrix( output.res( j ), j ) = true;
			}
		}
	}

	// fix insert hydrogens
	epigraft::design::design( output, design_matrix );
//	output.repack( allow_repack, false ); // boolean: no point in including wrong current position

	// fix ala and gly
	design_matrix = false;
	for ( std::set< Integer >::const_iterator i = is_ala.begin(), ie = is_ala.end(); i != ie; ++i ) {
		design_matrix( param_aa::aa_ala, *i ) = true;
	}
	for ( std::set< Integer >::const_iterator i = is_gly.begin(), ie = is_gly.end(); i != ie; ++i ) {
		design_matrix( param_aa::aa_gly, *i ) = true;
	}
	epigraft::design::design( output, design_matrix );

	// reset pdb info
	char const input_chain_id = input.pdb_info().res_chain( 1 );
	for ( Integer i = 1, ie = output.total_residue(); i <= ie; ++i ) {
		output.pdb_info().set_pdb_chain( i, input_chain_id );
		output.pdb_info().set_pdb_res( i, i );
		output.pdb_info().set_pdb_insert_let( i, ' ' );
	}
	output.pdb_info().set_use_pdb_numbering( input.pdb_info().use_pdb_numbering() );

}


/// @brief count peptide degree of vertices in fold tree
utility::vector1< Integer >
PoseAssembly::count_peptide_degree(
	Fold_tree const & input_fold_tree
)
{
	using pose_ns::Edge;

	// count total residue
	Integer total_residue = PoseAssembly::total_residue( input_fold_tree );;

	utility::vector1< Integer > peptide_degree( total_residue, 0 ); // number of peptide edges containing vertex

	for ( Fold_tree::const_iterator e = input_fold_tree.begin(), ee = input_fold_tree.end(); e != ee; ++e ) {
		Edge const & edge = *e;

		// record peptide degree of all vertices (number of peptide edges containing vertex) -- this is used to
		// lookup all peptide degree one vertices later
		if ( edge.label == Fold_tree::PEPTIDE ) {
			++peptide_degree[ edge.start ];
			++peptide_degree[ edge.stop ];
		}
	}

	return peptide_degree;
}


/// @brief report status
std::string
PoseAssembly::to_string() const
{
	std::ostringstream ss;

	ss << "PoseAssembly: base_total_residues = " << base_total_residues_ << "\n";
	for ( std::set< PoseAssemblyInstruction >::const_iterator i = instructions_.begin(), ie = instructions_.end(); i != ie; ++i ) {
		ss << i->to_string() << "\n";
	}

	return ss.str();
}


/// @brief using instructions_, make fold tree from input Pose fold tree
PoseAssembly::Fold_tree
PoseAssembly::make_fold_tree(
	Fold_tree const & input_fold_tree
) const
{
	using pose_ns::Edge; // fold tree edge, defined in fold_tree.h

	// sanity check first, make sure no jump residues have been removed (currently not supported)
	// TODO: handle removal of jump residues by heuristic (e.g. check of allow_bb_move on boths sides and shift jump in opposite direction)
	check_for_jump_removal( input_fold_tree );

	// record original start point of input fold tree
	Integer const input_reorder = input_fold_tree.begin()->start;
	check_for_root_removal( input_reorder ); // sanity check to make sure root has not been removed

	// record data for later use
	Integer total_residue = PoseAssembly::total_residue( input_fold_tree );
	Integer largest_label = PoseAssembly::largest_label( input_fold_tree );

	// record original peptide degree for each vertex
	utility::vector1< Integer > original_peptide_degree = PoseAssembly::count_peptide_degree ( input_fold_tree ); // number of peptide edges containing vertex

	// take into account any deletions that may remove and therefore change peptide degree
	utility::vector1< Integer > peptide_degree = original_peptide_degree;
	for ( Integer i = 1, ie = peptide_degree.size(); i <= ie; ++i ) {
		// test and handle deletions
		if ( original_peptide_degree[ i ] == 1 && instructions_.find( PoseAssemblyInstruction( i, PoseAssemblyInstruction::KEEP ) ) == instructions_.end() ) {
			peptide_degree[ i ] = -1;

			// find adjacent residues that will be newly peptide degree 1
			// first cycle backwards
			std::set< PoseAssemblyInstruction >::const_iterator iter;
			Integer res = i;
			do {
				--res;
				iter = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
			} while ( res > 0 && iter == instructions_.end() );
			peptide_degree[ iter->residue() ] = 1;
			// next cycle forwards
			res = i;
			do {
				++res;
				iter = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
			} while ( res <= total_residue && iter == instructions_.end() );
			peptide_degree[ iter->residue() ] = 1;
		}
	}

	// make copy of input fold tree and re-order at 1 so instructions can be applied
	Fold_tree ift = input_fold_tree;
	ift.set_allow_reorder( true ); // ensure fold tree is allowed to be re-ordered
	ift.reorder( 1 );

	Fold_tree f; // output fold tree

	// Gather all peptide edges -- these need to be sorted by smallest element of the range.
	// Unfortunately the fold tree doesn't give the ordering needed so we do it here.
	std::set< PeptideEdge > ordered_peptide_edges; // PeptideEdge is internal struct, defined at top of this file
	for ( Fold_tree::const_iterator e = ift.begin(), ee = ift.end(); e != ee; ++e ) {
		Edge const & edge = *e;

		if ( edge.label == Fold_tree::PEPTIDE ) {
			PeptideEdge pe;
			pe.start = edge.start;
			pe.stop = edge.stop;
			ordered_peptide_edges.insert( pe );
		}
	}

	// first handle peptide edges
	Integer offset = 0; // incrementally track differences after each instruction
	utility::vector1< bool > processed_res( total_residue, false );
	std::map< Integer, Integer > old_to_new_residue; // map from old residue -> new residue to update jumps later in the procedure
	for ( std::set< PeptideEdge >::const_iterator e = ordered_peptide_edges.begin(), ee = ordered_peptide_edges.end(); e != ee; ++e ) {
		PeptideEdge const & edge = *e;

		// (u, v) tracks the result of operations on the parent edge, which could be shrunk or shifted
		// right if there is an insertion.
		// u < v is enforced, operations reversed below when necessary.
		Integer edge_start = edge.start;
		Integer edge_stop = edge.stop;
		bool reverse = false;
		if ( edge_start > edge_stop ) {
			reverse = true;
			Integer tmp = edge_start;
			edge_start = edge_stop;
			edge_stop = tmp;
		}
		Integer u = edge_start + offset;
		Integer v = edge_stop + offset;

		// search for instructions over edge range
		for ( Integer res = edge_start; res <= edge_stop; ++res ) {

			// skip processed vertices
			if ( processed_res[ res ] ) {
				continue;
			}
			processed_res[ res ] = true; // go ahead and flip bit so that residue removal is also caught

			if ( instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) ) == instructions_.end() ) { // residue removal (no instructions)
				--v;
				--offset;
				continue;
			}

			// shifted residue numbering
			Integer shifted_residue = res + offset;
			bool need_to_reshift_residue = false;

			// left/right instructions?
			std::set< PoseAssemblyInstruction >::const_iterator has_left = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT ) );
			std::set< PoseAssemblyInstruction >::const_iterator has_right = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_RIGHT ) );

			// NOTE: the order of the following procedure/cases below matters!

			// handle peptide degree one vertex at the left extreme of a peptide chain
			if ( peptide_degree[ res ] == 1 && shifted_residue == u && has_left != instructions_.end() ) {
					PoseAssemblyInstruction const & instruction = *has_left;
					u += instruction.size(); // shift 'u' _before_ adding the new insertion edge
					v += instruction.size();
					f.add_edge( u, u - instruction.size(), Fold_tree::PEPTIDE ); // new insertion edge
					offset += instruction.size();
					need_to_reshift_residue = true;
			}

			// handle insertions internal to a peptide chain
			// - we consider insertions here in pairs: [ prior_res/insert_right, res/insert_left ]
			// - directionality of any new left/right insertion edge for peptide degree 1 is to
			//   always move away from the old edge
			if ( peptide_degree[ res ] != 1 || ( peptide_degree[ res ] == 1 && ( res + offset ) != u && ( res + offset ) == v ) ) {
				// find first prior residue that has a right insertion
				std::set< PoseAssemblyInstruction >::const_iterator prior_iter = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) ); // start from current instruction and go backwards
				--prior_iter; // step back one to start (i.e. move away from keep)
				while ( prior_iter->action() != PoseAssemblyInstruction::INSERT_RIGHT && prior_iter->action() != PoseAssemblyInstruction::KEEP ) {
					--prior_iter;
				}
				PoseAssemblyInstruction const & prior_instruction = *prior_iter;

				// grab left insertion of current residue
				std::set< PoseAssemblyInstruction >::const_iterator iter = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT ) );
				if ( iter == instructions_.end() ) { // "dummy" keep instruction
					iter = instructions_.find( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
				}
				PoseAssemblyInstruction const & instruction = *iter;

				// simple case, no breaks, done in pairs
				if ( !prior_instruction.break_on_insertion() && !instruction.break_on_insertion() ) {
					// handle right insertion from prior residue (can be further away than just res - 1 due to deletions)
					if ( prior_instruction.action() == PoseAssemblyInstruction::INSERT_RIGHT ) {
						v += prior_instruction.size();
						offset += prior_instruction.size();
						need_to_reshift_residue = true;
					}

					// handle only left insertion of current residue
					if ( instruction.action() == PoseAssemblyInstruction::INSERT_LEFT ) {
						v += instruction.size();
						offset += instruction.size();
						need_to_reshift_residue = true;
					}
				} else { // next we process if insertions have breaks
					bool new_edges_added = false;
					Integer first_branch_vertex = res - 1 + offset;
					Integer second_branch_vertex = res + offset;

					// handle right insertion from prior residue (can be further away than just res - 1 due to deletions)
					if ( prior_instruction.action() == PoseAssemblyInstruction::INSERT_RIGHT ) {
						second_branch_vertex += prior_instruction.size();
						v += prior_instruction.size();
						offset += prior_instruction.size();
						need_to_reshift_residue = true;

						f.add_edge( first_branch_vertex, first_branch_vertex + prior_instruction.size(), Fold_tree::PEPTIDE );
						new_edges_added = true;
					}

					// handle only left insertion of current residue
					if ( instruction.action() == PoseAssemblyInstruction::INSERT_LEFT ) {
						second_branch_vertex += instruction.size();
						v += instruction.size();
						offset += instruction.size();
						need_to_reshift_residue = true;

						f.add_edge( second_branch_vertex, second_branch_vertex - instruction.size(), Fold_tree::PEPTIDE );
						new_edges_added = true;
					}

					// complete the new part of the tree
					if ( new_edges_added ) {
						// add new jump
						f.add_edge( first_branch_vertex, second_branch_vertex, ++largest_label );

						// only add left branch peptide connection and update 'u'
						if ( reverse ) {
							f.add_edge( first_branch_vertex, u, Fold_tree::PEPTIDE );
						} else {
							f.add_edge( u, first_branch_vertex, Fold_tree::PEPTIDE );
						}
						u = second_branch_vertex;
					}
				}
			}

			// handle peptide degree vertex at the right extreme of a peptide chain
			if ( peptide_degree[ res ] == 1 && ( res + offset ) == v && has_right != instructions_.end() ) {
				PoseAssemblyInstruction const & instruction = *has_right;
				f.add_edge( v, v + instruction.size(), Fold_tree::PEPTIDE ); // new insertion edge
				offset += instruction.size();
			}

			// record new reside numbering
			if ( need_to_reshift_residue ) {
				shifted_residue = res + offset;
			}
			old_to_new_residue[ res ] = shifted_residue; // record new residue number
		}

		// add new edge
		if ( u <= v ) { // edge still exists
			if ( reverse ) {
				f.add_edge( v, u, Fold_tree::PEPTIDE );
			} else {
				f.add_edge( u, v, Fold_tree::PEPTIDE );
			}
		}

	}

	// next handle jump edges
	for ( Fold_tree::const_iterator e = ift.begin(), ee = ift.end(); e != ee; ++e ) {
		Edge const & edge = *e;
		if ( edge.label != Fold_tree::PEPTIDE ) {
			f.add_edge( old_to_new_residue[ edge.start ], old_to_new_residue[ edge.stop ], edge.label );
		}
	}

	// safety, pre-finalize fold tree before removing extra peptide degree 2 vertices
	f.reorder( old_to_new_residue[ input_reorder ] );

	// remove extra peptide degree 2 vertices
	f.delete_extra_vertices();

	// finalize fold tree
	f.reorder( old_to_new_residue[ input_reorder ] );

	return f;
}


/// @brief connect two Poses by specified jump between two residues and give a single Pose with merged
/// @brief fold tree
/// @param[in] left Pose (will be first in output Pose)
/// @param[in] right Pose (will be second in output Pose)
/// @param[in] residue in left Pose to jump from
/// @param[in] residue in right Pose to jump to
/// @param[out] output Pose
/// @return returns false if disconnection failed
/// @note  output Pose fold tree is rooted at left_pose's root
/// @note  full atom set true in output Pose
/// @warning This will idealize all cutpoints in resulting Pose with respect to cutpoint torsions
/// @warning in original pose.
ConnectionResult
PoseAssembly::connect (
	Pose const & left_pose,
	Pose const & right_pose,
	Integer const & left_jump_residue,
	Integer const & right_jump_residue,
	Pose & output,
	bool const & idealize_cutpoints
)
{
	using pose_ns::Edge;

	// cache total residue and offsets
	Integer const total_residue = left_pose.total_residue() + right_pose.total_residue();
	Integer const offset = left_pose.total_residue();

	// grab fold trees
	Fold_tree left_fold_tree = left_pose.fold_tree();
	Fold_tree right_fold_tree = right_pose.fold_tree();

	// offset for fold tree modification
	Integer const jump_offset = largest_label( left_fold_tree );

	// handle fold tree first
	Fold_tree output_fold_tree;

	// loop over left fold tree edges and insert edges, breaking edge where necessary due to jump
	for ( Fold_tree::const_iterator e = left_fold_tree.begin(), ee = left_fold_tree.end(); e != ee; ++e ) {
		Edge const & edge = *e;
		Integer const small_vertex = std::min( edge.start, edge.stop );
		Integer const big_vertex = std::max( edge.start, edge.stop );

		if ( edge.label == Fold_tree::PEPTIDE && small_vertex < left_jump_residue && left_jump_residue < big_vertex ) { // case to break edge
			output_fold_tree.add_edge( edge.start, left_jump_residue, edge.label );
			output_fold_tree.add_edge( left_jump_residue, edge.stop, edge.label );
		} else {
			output_fold_tree.add_edge( edge.start, edge.stop, edge.label );
		}
	}

	// loop over right fold tree edges and insert edges with offset, breaking edge where necessary due to jump
	for ( Fold_tree::const_iterator e = right_fold_tree.begin(), ee = right_fold_tree.end(); e != ee; ++e ) {
		Edge const & edge = *e;
		Integer const small_vertex = std::min( edge.start, edge.stop );
		Integer const big_vertex = std::max( edge.start, edge.stop );

		Integer const new_start = edge.start + offset;
		Integer const new_stop = edge.stop + offset;
		Integer const new_label = ( edge.label == Fold_tree::PEPTIDE ) ? edge.label : edge.label + jump_offset;

		if ( edge.label == Fold_tree::PEPTIDE && small_vertex < right_jump_residue && right_jump_residue < big_vertex ) { // case to break edge
			output_fold_tree.add_edge( new_start, right_jump_residue + offset, new_label );
			output_fold_tree.add_edge( right_jump_residue + offset, new_stop, new_label );
		} else {
			output_fold_tree.add_edge( new_start, new_stop, new_label );
		}

	}

	// add jump edge
	Integer const connect_start = left_jump_residue;
	Integer const connect_stop = right_jump_residue + offset;
	Integer const connect_label = largest_label( output_fold_tree ) + 1;
	output_fold_tree.add_edge( connect_start, connect_stop, connect_label );

	// reorder according to left_fold_tree root
	output_fold_tree.reorder( left_fold_tree.begin()->start );

	// setup
	output.set_fold_tree( output_fold_tree );
	output.set_fullatom_flag( true, false ); // booleans: fullatom, repack

	// transfer identity information
	transfer_identity_info_between_Poses( left_pose, ResidueRange( 1, left_pose.total_residue() ), output, 1, true ); // boolean: keep pdb info
	transfer_identity_info_between_Poses( right_pose, ResidueRange( 1, right_pose.total_residue() ), output, 1 + offset, true ); // boolean: keep pdb info

	// finally transfer all coordinates
	FArray3D_float full_coord( 3, param::MAX_ATOM(), total_residue, 0.0 ); // initialize all coordinates as zero
	Integer index = 1; // next residue to insert at
	fill_full_coord_from_Pose( left_pose, ResidueRange( 1, left_pose.total_residue() ), full_coord, index ); // index is modified here
	fill_full_coord_from_Pose( right_pose, ResidueRange( 1, right_pose.total_residue() ), full_coord, index ); // index is modified here

	// sanity check
	assert( index == total_residue + 1 );

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, total_residue );
	full_coord_to_Eposition( total_residue, full_coord, Eposition );

	// finalize Pose
	output.set_coords( false, Eposition, full_coord, false ); // booleans: ideal_pose, check_missing

	// idealize cutpoints
	if ( idealize_cutpoints ) {
		std::map< Integer, Real > cut_to_torsion;
		FArray1D< bool > const & is_output_cutpoint = output.fold_tree().get_is_cutpoint();
		for ( Integer i = 2, ie = output.total_residue() - 1; i <= ie; ++i ) {
			if ( is_output_cutpoint( i ) ) {
				Integer const nside = i;
				Integer const cside = i + 1;

				// take from either left or right pose, depending on where the new residue came from
				// offset = left_pose.total_residue
				Real const nside_psi = nside <= offset ? left_pose.psi( nside ) : right_pose.psi( nside - offset );
				Real const cside_phi = cside <= offset ? left_pose.phi( cside ) : right_pose.phi( cside - offset );

				idealize_cutpoint_retaining_psiphi( output, i, nside_psi, cside_phi );
			}
		}
	}

	// return ConnectionResult data
	return ConnectionResult( jump_offset, offset, connect_start, connect_stop, connect_label, ResidueRange( 1 + left_pose.total_residue(), total_residue ) );
}


/// @brief disconnect a Pose from the input Pose keeping fold tree intact
/// @param[in] input Pose
/// @param[in] ConnectionResult indicating connection to sever
/// @param[out] output Pose
/// @return returns false if disconnection failed
/// @note Edge directions (i.e. reversals) are handled correctly, in case the input old tree
/// @note has been re-rooted elsewhere at some point in time.
/// @note Full atom set true in output Pose.
/// @warning Remember to manually update any other ConnectionResult objects tied to this Pose
/// @warning with .disconnection_update() calls!  This is currently not handled automatically.
/// @warning This will idealize all cutpoints in resulting Pose with respect to cutpoint torsions
/// @warning in original pose.
bool
PoseAssembly::disconnect(
	Pose const & input,
	ConnectionResult const & connection,
	Pose & output,
	bool const & idealize_cutpoints
)
{
	using pose_ns::Edge;

	Fold_tree input_fold_tree = input.fold_tree();

	// run through existing edges in fold tree and make sure everything is ok, currently we don't want to be
	// severing a Pose that's being used as a jump to other Poses as there is not yet a facility to handle this
	for ( Fold_tree::const_iterator e = input_fold_tree.begin(), ee = input_fold_tree.end(); e != ee; ++e ) {
		Edge const & edge = *e;

		if ( edge.label != Fold_tree::PEPTIDE && edge.label != connection.jump_label() ) {
			if ( ( connection.segment_range().contains( edge.start ) && !connection.segment_range().contains( edge.stop ) ) ||
			     ( !connection.segment_range().contains( edge.start ) && connection.segment_range().contains( edge.stop ) ) ) {
				std::cerr << "PoseAssembly WARNING: you have requested removal of a Pose that had other Poses connected to it.  This is undefined -- refusing to remove the Pose section at "
				          << connection.segment_range().to_string() << "!" << std::endl;
				return false;
			}
		}
	}

	// new counts
	Integer const offset = connection.segment_range().length();
	Integer const total_residue = input.total_residue() - offset;

	// output fold tree
	Fold_tree output_fold_tree;

	// first handle fold tree
	for ( Fold_tree::const_iterator e = input_fold_tree.begin(), ee = input_fold_tree.end(); e != ee; ++e ) {
		Edge const & edge = *e;
		ResidueRange const edge_range( std::min( edge.start, edge.stop ), std::max( edge.start, edge.stop ) );

		// update any jump labels and add edges that don't belong to severed Pose
		if ( !edge_range.within( connection.segment_range() ) && edge.label != connection.jump_label() ) {
			Integer const new_edge_label = edge.label > connection.jump_label() ? edge.label - 1 : edge.label;

			if ( edge_range > connection.segment_range() ) { // edge needs to be shifted left
				output_fold_tree.add_edge( edge.start - offset, edge.stop - offset, new_edge_label  );
			} else {
				output_fold_tree.add_edge( edge.start, edge.stop, new_edge_label );
			}
		}
	}

	// In the next few blocks, we attempt to merge edge that was split upon
	// connection by looking for edges incident to the jump_start residue.
	// If there are *only* two peptide edges incident to the jump_start
	// residue, then merge these edges.
	// Usage of std::set< ResidueRange > is ok, as edges are proper and do
	// not overlap.
	bool do_merge = true; // assume true
	std::set< ResidueRange > candidate_edges_to_merge;
	for ( Fold_tree::const_iterator e = output_fold_tree.begin(), ee = output_fold_tree.end(); e != ee; ++e ) {
		Edge const & edge = *e;
		if ( edge.start == connection.jump_start() || edge.stop == connection.jump_start() ) {
			if ( edge.label != Fold_tree::PEPTIDE ) { // there's a jump edge incident, so merge is impossible
				do_merge = false;
				break;
			} else {
				candidate_edges_to_merge.insert( ResidueRange( std::min( edge.start, edge.stop ), std::max( edge.start, edge.stop ) ) );
			}
		}
	}

	// check if there are two peptide edges, and if so, merge them
	if ( do_merge && candidate_edges_to_merge.size() == 2 ) {
		// grab edges
		std::set< ResidueRange >::const_iterator left_edge = candidate_edges_to_merge.begin();
		std::set< ResidueRange >::const_iterator right_edge = ++( candidate_edges_to_merge.begin() );
		assert( left_edge->end() == right_edge->begin() );

		// remove edges
		output_fold_tree.delete_unordered_edge( left_edge->begin(), left_edge->end(), Fold_tree::PEPTIDE );
		output_fold_tree.delete_unordered_edge( right_edge->begin(), right_edge->end(), Fold_tree::PEPTIDE );

		// add merged edge
		output_fold_tree.add_edge( left_edge->begin(), right_edge->end(), Fold_tree::PEPTIDE );
	}

	// reorder following input fold tree root
	output_fold_tree.reorder( input_fold_tree.begin()->start );

	// setup output Pose
	output.set_fold_tree( output_fold_tree );
	output.set_fullatom_flag( true, false ); // booleans: fullatom, repack
	FArray3D_float full_coord( 3, param::MAX_ATOM(), total_residue, 0.0 ); // initialize all coordinates as zero

	// tracking index
	Integer index = 1; // next residue to insert at

	// handle transfer of left range
	if ( connection.segment_range().begin() > 1 ) {
		ResidueRange left_range( 1, connection.segment_range().begin() - 1 );
		transfer_identity_info_between_Poses( input, left_range, output, 1, true ); // boolean: keep pdb info
		fill_full_coord_from_Pose( input, left_range, full_coord, index ); // index is modified here
	}

	// handle transfer of right range
	if ( connection.segment_range().end() < input.total_residue() ) {
		ResidueRange right_range( connection.segment_range().end() + 1, input.total_residue() );
		transfer_identity_info_between_Poses( input, right_range, output, index, true ); // boolean: keep pdb info
		fill_full_coord_from_Pose( input, right_range, full_coord, index ); // index is modified here
	}

	// sanity check
	assert( index == total_residue + 1 );

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, total_residue );
	full_coord_to_Eposition( total_residue, full_coord, Eposition );

	// finalize Pose
	output.set_coords( false, Eposition, full_coord, false ); // booleans: ideal_pose, check_missing

	// idealize cutpoints
	if ( idealize_cutpoints ) {
		std::map< Integer, Real > cut_to_torsion;
		FArray1D< bool > const & is_output_cutpoint = output.fold_tree().get_is_cutpoint();
		Integer const removed_segment_length = connection.segment_range().length();
		for ( Integer i = 2, ie = output.total_residue() - 1; i <= ie; ++i ) {
			if ( is_output_cutpoint( i ) ) {
				Integer const nside = i;
				Integer const cside = i + 1;

				// take torsion from input pose, using correct index w/ respect to removed segment range
				Real const nside_psi = nside < connection.segment_range().begin() ? input.psi( nside ) : input.psi( nside + removed_segment_length );
				Real const cside_phi = cside < connection.segment_range().begin() ? input.phi( cside ) : input.phi( cside + removed_segment_length );

				idealize_cutpoint_retaining_psiphi( output, i, nside_psi, cside_phi );
			}
		}
	}

	return true;
}


/// @brief sanity check for jump removal (currently disallowed due to ambiguity)
// TODO: handle removal of jump residues by heuristic (e.g. check of allow_bb_move on sides and shift jump in opposite direction)
void
PoseAssembly::check_for_jump_removal(
	Fold_tree const & input_fold_tree
) const
{
	using pose_ns::Edge;

	for ( Fold_tree::const_iterator e = input_fold_tree.begin(), ee = input_fold_tree.end(); e != ee; ++e ) {
		Edge const & edge = *e;

		if ( edge.label != Fold_tree::PEPTIDE ) { // jump edge
			bool failure = false;
			Integer failure_residue = 0;

			// check jump start
			if ( instructions_.find( PoseAssemblyInstruction( edge.start, PoseAssemblyInstruction::KEEP ) ) == instructions_.end() ) {
				failure = true;
				failure_residue = edge.start;
			}
			// check jump stop
			if ( instructions_.find( PoseAssemblyInstruction( edge.stop, PoseAssemblyInstruction::KEEP ) ) == instructions_.end() ) {
				failure = true;
				failure_residue = edge.stop;
			}

			if ( failure ) {
				std::ostringstream ss;
				ss << "PoseAssembly: fold tree construction failed due to requested removal of jump residue ";
				ss << failure_residue << ".  Automatic shift of jump can be ambiguous and is currently not supported.";
				utility::exit( __FILE__, __LINE__, ss.str() );
			}
		}
	}
}


/// @brief sanity check for (old fold tree ) root removal (currently disallowed due to ambiguity)
void
PoseAssembly::check_for_root_removal(
	Integer const & old_root
) const
{
	bool failure = instructions_.find( PoseAssemblyInstruction( old_root, PoseAssemblyInstruction::KEEP ) ) == instructions_.end();
	if ( failure ) {
		std::ostringstream ss;
		ss << "PoseAssembly:: fold tree construction failed due to requsted removal of root residue ";
		ss << old_root << ".  Automatic shift of root can be ambiguous and is currently not supported.";
		utility::exit( __FILE__, __LINE__, ss.str() );
	}
}


/// @brief insert initial torsions, phi/psi/omega = -150/150/180
void
PoseAssembly::insert_initial_torsions(
	Pose & pose,
	ResidueRange const & range
)
{
	for ( Integer i = range.begin(), ie = range.end(); i <= ie; ++i ) {
		pose.set_phi( i, param::init_phi );
		pose.set_psi( i, param::init_psi );
		pose.set_omega( i, param::init_omega );
	}
}


/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values
/// @details if values don't exist in map, then regular idealization occurs
/// @note  this is useful when it's necessary to retain the existing torsions of the cut
void
PoseAssembly::idealize_cutpoint_retaining_psiphi(
	Pose & pose,
	Integer const & cut,
	Real const & nside_psi,
	Real const & cside_phi
)
{
	// idealize
	pose.insert_ideal_bonds_at_cutpoint( cut );

	// set n-side psi
	pose.set_psi( cut, nside_psi );

	// set c-side phi
	pose.set_phi( cut + 1, cside_phi );
}


/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values
/// @details if values don't exist in map, then regular idealization occurs
/// @note  this is useful when it's necessary to retain the existing torsions of the cut
void
PoseAssembly::idealize_cutpoint_retaining_psiphi(
	Pose & pose,
	Integer const & cut,
	std::map< Integer, Real > const & original_torsions
)
{
	// idealize
	pose.insert_ideal_bonds_at_cutpoint( cut );

	// grab original n-side psi if available and set
	std::map< Integer, Real >::const_iterator ns_psi = original_torsions.find( cut );
	if ( ns_psi != original_torsions.end() ) {
		pose.set_psi( cut, ns_psi->second );
	}

	// grab original c-side phi if available and set
	std::map< Integer, Real >::const_iterator cs_phi = original_torsions.find( cut + 1 );
	if ( cs_phi != original_torsions.end() ) {
		pose.set_phi( cut + 1, cs_phi->second );
	}
}


/// @brief idealize cutpoints but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values
/// @details if values don't exist in map, then regular idealization occurs
/// @note  this is useful when it's necessary to retain the existing torsions of the cuts
void
PoseAssembly::idealize_all_cutpoints(
	Pose & pose,
	std::map< Integer, Real > const & original_torsions
)
{
	FArray1D< bool > const & is_cutpoint = pose.fold_tree().get_is_cutpoint();
	for ( Integer i = 2, ie = pose.total_residue() - 1; i <= ie; ++i ) {
		if ( is_cutpoint( i ) ) {
			idealize_cutpoint_retaining_psiphi( pose, i, original_torsions );
		}
	}
}


/// @brief create one letter character -> Rosetta param_aa residue type map
/// @warning this function will have problems if residue one letter names are not unique,
/// @warning such as for nucleic acids, see aa_name1 in param_aa.cc
std::map< char, Integer >
PoseAssembly::get_one_letter_to_rosetta_residue() const
{
	std::map< char, Integer > oltr;

	for ( Integer i = 1, ie = param_aa::aa_name1.size1(); i <= ie; ++i ) {
		oltr[ param_aa::aa_name1( i ) ] = i;
	}

	return oltr;
}


} // namespace design
} // namespace epigraft
