// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   PoseAssembly.hh
/// @brief  Insert/delete residues in a Pose maintaining fold tree.
/// @brief  Insertions may be either continuous or inserted with a break.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_design_PoseAssembly_HH_
#define INCLUDED_epigraft_design_PoseAssembly_HH_


// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/design/ConnectionResult.fwd.hh>
#include <epigraft/design/PoseAssemblyInstruction.hh>
#include <epigraft/ResidueRange.hh>

// rosetta headers
#include <fold_tree.h>
#include <param_aa.h>
#include <pose.h>

// c++ headers
#include <set>
#include <map>


namespace epigraft {
namespace design {


/// @brief  Insert/delete residues in a Pose maintaining fold tree.
/// @brief  Insertions may be either continuous or inserted with a break.
class PoseAssembly {


	public: // typedefs

		typedef pose_ns::Fold_tree Fold_tree;


	public: // construct/destruct

		/// @brief default constructor
		inline
		PoseAssembly()
		{}

		/// @brief nres constructor
		inline
		PoseAssembly(
			Integer const & base_total_residues
		) : base_total_residues_( base_total_residues )
		{
			reset(); // load default instruction set (keep all residues)
		}

		/// @brief copy constructor
		inline
		PoseAssembly(
			PoseAssembly const & a
		) : base_total_residues_( a.base_total_residues_ ),
		    instructions_( a.instructions_ )
		{}

		/// @brief default destructor
		inline
		~PoseAssembly()
		{}


	public: // assignment

		/// @brief copy assignment
		inline
		PoseAssembly &
		operator =( PoseAssembly const & a )
		{
			if ( this != &a ) {
				base_total_residues_ = a.base_total_residues_;
				instructions_ = a.instructions_;
			}
			return *this;
		}


	public: // One-Pose modification methods (insert/delete)

		/// @brief keep residue
		inline
		void
		keep(
			Integer const & res
		)
		{
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
		}

		/// @brief remove residue or insertion w/ respect to residue
		/// @note  removal of a residue 'KEEP' action will result in removal of any insertions anchored
		/// @note  to the residue
		inline
		void
		remove(
			Integer const & res,
			PoseAssemblyInstruction::Action const & action = PoseAssemblyInstruction::KEEP
		)
		{
			instructions_.erase( PoseAssemblyInstruction( res, action ) );
			if ( action == PoseAssemblyInstruction::KEEP ) {
				instructions_.erase( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT ) );
				instructions_.erase( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_RIGHT ) );
			}
		}


		/// @brief remove a range of residues
		inline
		void
		remove(
			ResidueRange const & range
		)
		{
			for ( Integer i = range.begin(), ie = range.end(); i <= ie; ++i ) {
				remove( i );
			}
		}


		/// @brief remove instruction
		/// @note  removal of a residue 'KEEP' action will result in removal of any insertions anchored
		/// @note  to the residue
		inline
		void
		remove(
			PoseAssemblyInstruction const & instruction
		)
		{
			instructions_.erase( instruction );
			if ( instruction.action() == PoseAssemblyInstruction::KEEP ) {
				instructions_.erase( PoseAssemblyInstruction( instruction.residue(), PoseAssemblyInstruction::INSERT_LEFT ) );
				instructions_.erase( PoseAssemblyInstruction( instruction.residue(), PoseAssemblyInstruction::INSERT_RIGHT ) );
			}
		}

		/// @brief insert #residues to left of residue
		/// @note  an anchor residue is always necessary, so if that residue is not 'KEEP'
		/// @note  then this function will also add 'KEEP' status
		inline
		void
		insert_left(
			Integer const & res,
			Integer const & size,
			bool const & break_on_insertion = false,
			char const & ss_type = 'D',
			char const & residue_type = 'G'
		)
		{
			// remove existing
			instructions_.erase( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT ) );

			// add
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT, size, break_on_insertion, ss_type, residue_type ) );
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
		}

		/// @brief insert #residues to left of residue
		/// @note  an anchor residue is always necessary, so if that residue is not 'KEEP'
		/// @note  then this function will also add 'KEEP' status
		inline
		void
		insert_left(
			Integer const & res,
			Integer const & size,
			bool const & break_on_insertion,
			std::string const & ss_string,
			std::string const & residue_string
		)
		{
			// remove existing
			instructions_.erase( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT ) );

			// add
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_LEFT, size, break_on_insertion, ss_string, residue_string ) );
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
		}

		/// @brief insert #residues to right of residue
		/// @note  an anchor residue is always necessary, so if that residue is not 'KEEP'
		/// @note  then this function will also add 'KEEP' status
		inline
		void
		insert_right(
			Integer const & res,
			Integer const & size,
			bool const & break_on_insertion = false,
			char const & ss_type = 'D',
			char const & residue_type = 'G'
		)
		{
			// remove existing
			instructions_.erase( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_RIGHT ) );

			// add
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_RIGHT, size, break_on_insertion, ss_type, residue_type ) );
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
		}

		/// @brief insert #residues to right of residue
		/// @note  an anchor residue is always necessary, so if that residue is not 'KEEP'
		/// @note  then this function will also add 'KEEP' status
		inline
		void
		insert_right(
			Integer const & res,
			Integer const & size,
			bool const & break_on_insertion,
			std::string const & ss_string,
			std::string const & residue_string
		)
		{
			// remove existing
			instructions_.erase( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_RIGHT ) );

			// add
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::INSERT_RIGHT, size, break_on_insertion, ss_string, residue_string ) );
			instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
		}

		/// @brief add instruction
		/// @note  this will override any existing instruction with the same residue & action
		/// @note  when doing an insertion an anchor residue is always necessary, so if that residue is
		/// @note  not 'KEEP' then this function will also add 'KEEP' status
		inline
		void
		add_instruction(
			PoseAssemblyInstruction const & instruction
		)
		{
			// remove existing
			instructions_.erase( instruction );

			// add
			instructions_.insert( instruction );
			if ( instruction.action() != PoseAssemblyInstruction::KEEP ) {
				instructions_.insert( PoseAssemblyInstruction( instruction.residue(), PoseAssemblyInstruction::KEEP ) );
			}
		}

		/// @brief return instructions for given residue
		std::set< PoseAssemblyInstruction >
		instructions_for_residue(
			Integer const & res
		) const;


	public: // One-Pose modify

		/// @brief calculate old residue to new residue map from existing instructions
		/// @note  deleted old residues are mapped to -1
		std::map< Integer, Integer >
		map_old_to_new() const;

		/// @brief calculate new residue to old residue map from existing instructions
		/// @note  inserted new residues are mapped to -1
		std::map< Integer, Integer >
		map_new_to_old() const;

		/// @brief calculate old cutpoint to new cutpoint map from existing instructions
		std::map< Integer, Integer >
		map_cut_old_to_new(
			Fold_tree const & f
		) const;

		/// @brief apply instructions and create new Pose/fold tree
		/// @note full atom set true in output Pose
		/// @param[in] idealize_cutpoints  if true all cutpoints in new Pose will be idealized
		/// @warning residues adjacent to inserts/deletions will have their connecting bonds/angles idealized
		void
		apply(
			Pose const & input,
			Pose & output,
			bool const & idealize_cutpoints = true
		) const;


	public: // Two-Pose modify

		/// @brief connect two Poses by specified jump between two residues and give a single Pose with merged
		/// @brief fold tree
		/// @param[in] left Pose (will be first in output Pose)
		/// @param[in] right Pose (will be second in output Pose)
		/// @param[in] residue in left Pose to jump from
		/// @param[in] residue in right Pose to jump to
		/// @param[out] output Pose
		/// @param[in] idealize_cutpoints  idealize cutpoints in resulting pose w/ respect to cutpoint torsions
		/// @return returns false if disconnection failed
		/// @note  output Pose fold tree is rooted at left_pose's root
		static
		ConnectionResult
		connect(
			Pose const & left_pose,
			Pose const & right_pose,
			Integer const & jump_from_left,
			Integer const & jump_to_right,
			Pose & output,
			bool const & idealize_cutpoints = true
		);

		/// @brief disconnect a Pose from the input Pose keeping fold tree intact
		/// @param[in] input Pose
		/// @param[in] ConnectionResult indicating connection to sever
		/// @param[out] output Pose
		/// @param[in] idealize_cutpoints  idealize cutpoints in resulting pose w/ respect to cutpoint torsions
		/// @note Edge directions (i.e. reversals) are handled correctly, in case the input old tree
		/// @note has been re-rooted elsewhere at some point in time.
		/// @return returns false if disconnection failed
		/// @warning Remember to manually update any other ConnectionResult objects tied to this Pose
		/// @warning with .disconnection_update() calls!  This is currently not handled automatically.
		static
		bool
		disconnect(
			Pose const & input,
			ConnectionResult const & connection,
			Pose & output,
			bool const & idealize_cutpoints = true
		);


	public: // accessors

		/// @brief base total residues object was initialized with
		Integer const &
		base_total_residues() const
		{
			return base_total_residues_;
		}


	public: // gather fold tree data

		/// @brief find total residue from fold tree
		static
		inline
		Integer
		total_residue(
			Fold_tree const & f
		)
		{
			using pose_ns::Edge;

			Integer total_residue = 0;

			for ( Fold_tree::const_iterator e = f.begin(), ee = f.end(); e != ee; ++e ) {
				Edge const & edge = *e;

				// total residue
				total_residue = std::max( total_residue, std::max( edge.start, edge.stop ) );
			}

			return total_residue;
		}

		/// @brief find largest jump label in fold tree
		static
		inline
		Integer
		largest_label(
			Fold_tree const & f
		)
		{
			using pose_ns::Edge;

			Integer largest_label = 0;

			for ( Fold_tree::const_iterator e = f.begin(), ee = f.end(); e != ee; ++e ) {
				Edge const & edge = *e;

				largest_label = std::max( largest_label, edge.label );
			}

			return largest_label;
		}

		/// @brief count peptide degree of vertices in fold tree
		static
		utility::vector1< Integer >
		count_peptide_degree(
			Fold_tree const & input_fold_tree
		);


	public: // initialize/reset

		/// @brief reset object to default instruction set (keep all residues)
		inline
		void
		reset()
		{
			instructions_.clear();
			for ( Integer res = 1; res <= base_total_residues_; ++res ) {
				instructions_.insert( PoseAssemblyInstruction( res, PoseAssemblyInstruction::KEEP ) );
			}
		}


	public: // status

		/// @brief report status
		std::string
		to_string() const;


	private: // fold tree manipulation

		/// @brief using instructions_, make fold tree from input Pose fold tree
		Fold_tree
		make_fold_tree(
			Fold_tree const & input_fold_tree
		) const;

		/// @brief sanity check for jump removal (currently disallowed due to ambiguity)
		void
		check_for_jump_removal(
			Fold_tree const & input_fold_tree
		) const;

		/// @brief sanity check for (old fold tree ) root removal (currently disallowed due to ambiguity)
		void
		check_for_root_removal(
			Integer const & old_root
		) const;


	private: // geometry setup

		/// @brief insert initial torsions, phi/psi/omega = -150/150/180
		static
		void
		insert_initial_torsions(
			Pose & pose,
			ResidueRange const & range
		);

		/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
		/// @brief using given values
		/// @details if values don't exist in map, then regular idealization occurs
		/// @note  this is useful when it's necessary to retain the existing torsions of the cut
		static
		void
		idealize_cutpoint_retaining_psiphi(
			Pose & pose,
			Integer const & cut,
			Real const & nside_psi,
			Real const & cside_phi
		);

		/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
		/// @brief using given values
		/// @details if values don't exist in map, then regular idealization occurs
		/// @note  this is useful when it's necessary to retain the existing torsions of the cut
		static
		void
		idealize_cutpoint_retaining_psiphi(
			Pose & pose,
			Integer const & cut,
			std::map< Integer, Real > const & original_torsions
		);

		/// @brief idealize cutpoints but retain psi on the n-side and phi on the c-side of the cut
		/// @brief using given values
		/// @details if values don't exist in map, then regular idealization occurs
		/// @note  this is useful when it's necessary to retain the existing torsions of the cuts
		static
		void
		idealize_all_cutpoints(
			Pose & pose,
			std::map< Integer, Real > const & original_torsions
		);


	private: // convenience functions

		/// @brief create one letter character -> Rosetta param_aa residue type map
		/// @warning this function will have problems if residue one letter names are not unique,
		/// @warning such as for nucleic acids, see aa_name1 in param_aa.cc
		std::map< char, Integer >
		get_one_letter_to_rosetta_residue() const;


	private: // data

		Integer base_total_residues_;
		std::set< PoseAssemblyInstruction > instructions_;

};


} // namespace design
} // namespace epigraft


#endif /*INCLUDED_epigraft_design_PoseAssembly_HH_*/
