// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   PoseAssemblyInstruction.hh
/// @brief  Instructions for PoseAssembly operation, immutable.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_design_PoseAssemblyInstruction_HH_
#define INCLUDED_epigraft_design_PoseAssemblyInstruction_HH_

// package headers
#include <epigraft/design/design_types.hh>

// rosetta headers
#include <param_aa.h>

// c++ headers
#include <sstream>


namespace epigraft {
namespace design {


/// @brief  Instructions for PoseAssembly operation, immutable.
class PoseAssemblyInstruction {

	public: // enums
	
		enum Action {
			INSERT_LEFT = -1,
			KEEP = 0,
			INSERT_RIGHT = 1
		};


	public: // construct/destruct

		/// @brief default constructor
		inline
		PoseAssemblyInstruction()
		{}
		
		/// brief  simple constructor
		/// @note  not necessary to indicate size if action = 'KEEP', by default break on insertion is false,
		/// @note  the residue type field is optional and only takes into effect when the action is 'INSERT_LEFT'
		/// @note  or 'INSERT_RIGHT'
		inline
		PoseAssemblyInstruction(
			Integer const & residue,
			Action const & action,
			Integer const & size = 0,
			bool const & break_on_insertion = false,
			char const & ss_type = 'D',
			char const & residue_type = 'G'
		) : residue_( residue ),
		    action_( action ),
		    size_( size ),
		    break_on_insertion_( break_on_insertion ),
		    ss_string_( size, ss_type ),
		    residue_string_( size, residue_type )
		{}

		/// brief  string constructor
		/// @details both strings always run N->C
		/// @note  not necessary to indicate size if action = 'KEEP', by default break on insertion is false,
		/// @note  the residue type field is optional and only takes into effect when the action is 'INSERT_LEFT'
		/// @note  or 'INSERT_RIGHT'
		inline
		PoseAssemblyInstruction(
			Integer const & residue,
			Action const & action,
			Integer const & size,
			bool const & break_on_insertion,
			std::string const & ss_string,
			std::string const & residue_string
		) : residue_( residue ),
		    action_( action ),
		    size_( size ),
		    break_on_insertion_( break_on_insertion ),
		    ss_string_( ss_string ),
		    residue_string_( residue_string )
		{
				assert( ss_string_.length() == ( (std::size_t)size_ ) && residue_string_.length() == ( (std::size_t)size_ ) );
		}
		
		/// @brief copy constructor
		inline
		PoseAssemblyInstruction(
			PoseAssemblyInstruction const & i
		) : residue_( i.residue_ ),
		    action_( i.action_ ),
		    size_( i.size_ ),
		    break_on_insertion_( i.break_on_insertion_ ),
		    ss_string_( i.ss_string_ ),
		    residue_string_( i.residue_string_ )
		{}
		
		/// @brief default destructor
		inline
		~PoseAssemblyInstruction()
		{}


	public: // copy assignment

		/// @brief copy assignment
		inline
		PoseAssemblyInstruction &
		operator =( PoseAssemblyInstruction const & i )
		{
			if ( this != &i ) {
				residue_ = i.residue_;
				action_ = i.action_;
				size_ = i.size_;
				break_on_insertion_ = i.break_on_insertion_;
				ss_string_ = i.ss_string_;
				residue_string_ = i.residue_string_;
			}
			return *this;
		}


	public: // member operator
	
		/// @brief this < rvalue, only compares residue and action
		inline
		bool
		operator <( PoseAssemblyInstruction const & rvalue ) const
		{
			if ( residue_ < rvalue.residue_ ) {
				return true;
			} else if ( residue_ == rvalue.residue_ && action_ < rvalue.action_ ) {
				return true;
			}
			return false;
		}
		
		/// @brief this > rvalue, only compares residue and action
		inline
		bool
		operator >( PoseAssemblyInstruction const & rvalue ) const
		{
			if ( residue_ > rvalue.residue_ ) {
				return true;
			} else if ( residue_ == rvalue.residue_ && action_ > rvalue.action_ ) {
				return true;
			}
			return false;
		}
		
		/// @brief this == rvalue, only compares residue and action
		inline
		bool
		operator ==( PoseAssemblyInstruction const & rvalue ) const
		{
			return residue_ == rvalue.residue_ && action_ == rvalue.action_;
		}
		
		/// @brief this != rvalue, only compares residue and action
		inline
		bool
		operator !=( PoseAssemblyInstruction const & rvalue ) const
		{
			return residue_ != rvalue.residue_ || action_ != rvalue.action_;
		}


	public: // accessors
	
		/// @brief residue number
		inline
		Integer const &
		residue() const
		{
			return residue_;
		}
		
		/// @brief action
		inline
		Action const &
		action() const
		{
			return action_;
		}
		
		/// @brief size of action
		inline
		Integer const &
		size() const
		{
			return size_;
		}
		
		/// @brief break on insertion?
		inline
		bool const &
		break_on_insertion() const
		{
			return break_on_insertion_;
		}
		
		/// @brief string containing one-letter secondary structure codes (H,E,L,D), will be the
		/// @brief same length as 'size'
		/// @details string always runs N->C
		inline
		std::string const &
		ss_string() const
		{
			return ss_string_;
		}
		
		/// @brief string containing one-letter residue codes, will be the same length as 'size'
		/// @details string always runs N->C
		inline
		std::string const &
		residue_string() const
		{
			return residue_string_;
		}

		/// @brief report status
		inline
		std::string
		to_string() const
		{
			std::ostringstream ss;
			ss << "res = " << residue_ << " act = " << action_ << " size = " << size_ << " break_on_insert = " << break_on_insertion_
			   << " ss = " << ss_string_ << " rs = " << residue_string_;
			return ss.str();
		}

	private: // data
	
		Integer residue_;
		Action action_;
		Integer size_; // size of action
		bool break_on_insertion_;
		std::string ss_string_; // string always runs N->C
		std::string residue_string_; // string always runs N->C

};


} // namespace design
} // namespace epigraft


#endif /*INCLUDED_epigraft_design_PoseAssemblyInstruction_HH_*/
