// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   Vall.hh
/// @brief  Holds Vall data.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_design_Vall_HH_
#define INCLUDED_epigraft_design_Vall_HH_

// package headers
#include <epigraft/design/design_types.hh>

// Rosetta headers
#include <fragments_ns.h>

// utility headers
#include <utility/vector1.hh>


namespace epigraft {
namespace design {


class Vall {


	public: // construct

		Vall()
		{}

		inline
		Vall( Size const & reserve ) :
			pdb_( reserve ),
			chain_( reserve ),
			resseq_( reserve ),
			sequence_( reserve ),
			secstruct_( reserve ),
			phi_( reserve ),
			psi_( reserve ),
			omega_( reserve )
		{}

		inline
		Vall( Vall const & v ) :
			pdb_( v.pdb_ ),
			chain_( v.chain_ ),
			resseq_( v.resseq_ ),
			sequence_( v.sequence_ ),
			secstruct_( v.secstruct_ ),
			phi_( v.phi_ ),
			psi_( v.psi_ ),
			omega_( v.omega_ )
		{}

		inline
		~Vall()
		{}


	public: // assignment

		inline
		Vall &
		operator =( Vall const & v )
		{
			if ( this != &v ) {
				pdb_ = v.pdb_;
				chain_ = v.chain_;
				resseq_ = v.resseq_;
				sequence_ = v.sequence_;
				secstruct_ = v.secstruct_;
				phi_ = v.phi_;
				psi_ = v.psi_;
				omega_ = v.omega_;
			}

			return *this;
		}


	public: // methods

		inline
		void
		conserve_memory()
		{
			utility::vector1< String >( pdb_ ).swap( pdb_ );
			utility::vector1< char >( chain_ ).swap( chain_ );
			utility::vector1< Integer >( resseq_ ).swap( resseq_ );

			utility::vector1< char >( sequence_ ).swap( sequence_ );
			utility::vector1< char >( secstruct_ ).swap( secstruct_ );

			utility::vector1< Real >( phi_ ).swap( phi_ );
			utility::vector1< Real >( psi_ ).swap( psi_ );
			utility::vector1< Real >( omega_ ).swap( omega_ );
		}

		inline
		Size
		size() const
		{
			return secstruct_.size();
		}


	public: // methods

		inline
		String const &
		pdb( Size const & i ) const
		{
			return pdb_[ i ];
		}

		inline
		char const &
		chain( Size const & i ) const
		{
			return chain_[ i ];
		}

		inline
		Integer const &
		resseq( Size const & i ) const
		{
			return resseq_[ i ];
		}

		inline
		char const &
		sequence( Size const & i ) const
		{
			return sequence_[ i ];
		}

		inline
		char const &
		secstruct( Size const & i ) const
		{
			return secstruct_[ i ];
		}

		inline
		Real const &
		phi( Size const & i ) const
		{
			return phi_[ i ];
		}

		inline
		Real const &
		psi( Size const & i ) const
		{
			return psi_[ i ];
		}

		inline
		Real const &
		omega( Size const & i ) const
		{
			return omega_[ i ];
		}


	public: // methods

		void
		copy_to_global_fragment_array(
			Size const & position,
			Size const & width,
			Integer const & res,
			Integer const & fragment_number,
			Integer const & size_bin
		);


	public: // methods

		inline
		void
		push_back(
			String const & pdb,
			char const & chain,
			Integer const & resseq,
			char const & sequence,
			char const & secstruct,
			Real const & phi,
			Real const & psi,
			Real const & omega
		)
		{
			pdb_.push_back( pdb );
			chain_.push_back( chain );
			resseq_.push_back( resseq );
			sequence_.push_back( sequence );
			secstruct_.push_back( secstruct );
			phi_.push_back( phi );
			psi_.push_back( psi );
			omega_.push_back( omega );
		}


	private: // data

		utility::vector1< String > pdb_;
		utility::vector1< char > chain_;
		utility::vector1< Integer > resseq_;

		utility::vector1< char > sequence_;
		utility::vector1< char > secstruct_;

		utility::vector1< Real > phi_;
		utility::vector1< Real > psi_;
		utility::vector1< Real > omega_;

};


} // namespace design
} // namespace epigraft


#endif /* INCLUDED_epigraft_design_Vall_HH_ */
