// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   decimated_multigraft.cc
/// @brief  Performs closure and design for an epigraft match hit.
///         Decimated for benchmarking.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bruno Correia (bcorreia@u.washington.edu)

// unit headers
#include <epigraft/design/decimated_multigraft.hh>

// package headers
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/Checkpoint.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/epigraft_io.hh>
#include <epigraft/design/DesignFileExport.hh>
#include <epigraft/design/EpitopeScaffold.hh>
#include <epigraft/design/ESBundle.hh>
#include <epigraft/design/GraftInfo.hh>
#include <epigraft/design/MultiGraftStats.hh>
#include <epigraft/design/OutputFilename.hh>
#include <epigraft/design/PoseAssembly.hh>
#include <epigraft/design/ccd_functions.hh>
#include <epigraft/design/design_io.hh>
#include <epigraft/design/fragment_functions.hh>
#include <epigraft/design/loop_functions.hh>
#include <epigraft/design/multigraft_checkpoint.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/match_functions.hh>
//#include <epigraft/match/match_io.hh>
#include <epigraft/match/rescore_matches.hh>

// rosetta headers
#include <decoy_features.h>
#include <disulfides.h>
#include <files_paths.h>
#include <fragments_pose.h>
#include <InteractionGraphBase.h>
#include <jumping_loops.h>
#include <PackerTask.h>
#include <param_aa.h>
#include <pose.h>
#include <pose_io.h>
#include <random_numbers.h>
#include <RotamerSet.h>
#include <score.h>
#include <vall_data.h>
#include <vdw.h>

// utility headers
#include <utility/file/FileName.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/vector1.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>

// C++ headers
#include <cstdio>
#include <ctime>
#include <fstream>
#include <iostream>
#include <limits>
#include <map>
#include <set>
#include <sstream>
#include <string>
#include <utility>

namespace epigraft {
namespace design {


/// @brief decimated MultiGraft top-level protocol
void
decimated_multigraft(
	GraftOptions const & options
)
{
	using epigraft::match::MatchResult;
	using namespace pose_ns;

	// Rosetta setup
	disulfides::options::find_disulf = true;
	disulfides::options::norepack_disulf = true;
	score_set_vary_omega( false );
//	select_atomvdw( "highres" );
	std::cout << "STATE: find_disulf is ON\n";
	std::cout << "STATE: norepack_disulf is ON\n";
	std::cout << "STATE: score_set_vary_omega is OFF\n";
//	std::cout << "STATE: centroid level vdw is HIGHRES" << std::endl;

	// override any rosetta global options
	if ( options.override_rosetta_pdb_output_path ) {
		files_paths::pdb_out_path = options.pdb_output_path;
	}

	// initialize decoy features
	decoy_features_ns::set_df_flag( true ); // force decoy features flag here instead of depending on command line option
	decoy_features_ns::decoy_features_initialize( false ); // boolean: init_using_command_line_option
	decoy_features_ns::set_df_flag( false ); // temporarily turn off, flag must be governed by design_io.cc::output_pdb()


	// load native complex from file, NOTE: assume Ab comes first!
	AntibodyComplex native( options.native_complex_filename, options.nres_Ab, options.Ab_first, true ); // booleans: Ab_first, refold_sidechains_from_chi

	Real const wt_ddG = native.Ab_exists() ?
	                    native.complex().score( Score_weight_map( score12 ) )
	                      - ( native.Ab().score( Score_weight_map( score12 ) ) +
	                          native.antigen().score( Score_weight_map( score12 ) ) ) :
	                    std::numeric_limits< Real >::infinity();

	// read list of epitope (loop) ranges
	// TODO: optional -- error checking to see if any of the subranges lie out of bounds relative to various criteria
	utility::vector1< LoopInfo > loops_to_scan;
	std::set< Integer > user_specified_superposition_residues;
	load_epitope_subranges( options.loop_ranges_filename, loops_to_scan, user_specified_superposition_residues, options.use_SS_align );

	// sanity check for epitope (loop) ranges and epitope input structure
	if ( !epitope_and_epitope_subranges_one_to_one( loops_to_scan, native.antigen() ) ) {
		utility::exit( __FILE__, __LINE__, "ERROR: please make sure the epitope in your antibody complex structure includes exactly the residues specified in the 'full_range'(s) of the loop ranges file" );
	}

	// load keep natro file if present
	std::set< Integer > keep_natro_residues;
	if ( options.use_keep_natro ) {
		keep_natro_residues = parse_keep_natro_file( options.keep_natro_filename );
	}

	// load match results
	std::map< std::string, utility::vector1< GraftInfo > > pdb_to_graft_info;
	graft_info_from_file( options.input_filename, loops_to_scan, pdb_to_graft_info, options.use_old_graft_info_format );

	// scaffold filenames
	utility::vector1< std::string > scaffold_filenames;
	for ( std::map< std::string, utility::vector1< GraftInfo > >::const_iterator entry = pdb_to_graft_info.begin(), entry_e = pdb_to_graft_info.end(); entry != entry_e; ++entry ) {
		scaffold_filenames.push_back( entry->first ); // scaffold filename
	}

	// open output file
	std::string OUTPUT_FILE = files_paths::score_path + options.output_filename;
	utility::io::ozstream outfile;
	utility::io::izstream testfile( OUTPUT_FILE );
	if ( testfile.fail() ) {
		testfile.close();
		outfile.open( OUTPUT_FILE );
	} else {
		testfile.close();
		outfile.open_append( OUTPUT_FILE );
	}

	// create iterator index outside of loop due to potential checkpointing
	Size scaffold_id = 1;
	Size match_counter = 1;

	// stage for checkpointing
	MultiGraftStage stage = ONGOING;

	// init/recover checkpoint info
	Checkpoint outer_checkpoint;
	if ( options.use_checkpoint ) {
		outer_checkpoint.open( options.checkpoint_filename );
		outer_checkpoint.recover_last_id_processed();

		if ( outer_checkpoint.current_id() > scaffold_filenames.size() ) {
			// at max or somehow over count, so we're done
			utility::exit( __FILE__, __LINE__, "checkpoint indicates all files processed" );
		}

		scaffold_id = outer_checkpoint.current_id() > 0 ? outer_checkpoint.current_id() : 1;
		match_counter = outer_checkpoint.current_sub_id() > 0 ? outer_checkpoint.current_sub_id() : 1;
	}

	// run over each scaffold
	for ( Size last_scaffold_id = scaffold_filenames.size(); scaffold_id <= last_scaffold_id; ++scaffold_id ) {

		// find scaffold filename
		std::string const & scaffold_filename = scaffold_filenames[ scaffold_id ];

		// status
		std::cout << "MultiGraft: processing scaffold " << scaffold_filename << std::endl;

		// load pdb from either individual file or complex file
		Pose scaffold;
		if ( !options.input_pdb_has_Ab ) {
			pose_from_pdb( scaffold, files_paths::start_path + scaffold_filename, true, false, true ); // boolean: fullatom, ideal_pose, read_all_chains
		} else {
			AntibodyComplex ac( files_paths::start_path + scaffold_filename, options.nres_Ab, options.Ab_first, false );
			scaffold = ac.antigen();
		}

		// score scaffold for rudimentary stability output (wild-type scaffold vs grafted scaffold)
		Real const wt_scaffold_score = scaffold.score( Score_weight_map( score12 ) );

		// run over each match result and run closure+design schedule
		utility::vector1< GraftInfo > & graft_infos = pdb_to_graft_info[ scaffold_filename ];
		for ( Size match_counter_end = graft_infos.size(); match_counter <= match_counter_end; ++match_counter ) {

			// checkpointing: the rest of the run is slow enough that open/update/close is ok
			if ( options.use_checkpoint ) {
				outer_checkpoint.set_current_id( scaffold_id );
				outer_checkpoint.set_current_sub_id( match_counter );
				outer_checkpoint.update();
			}

			// prepare output filename
			std::ostringstream middle_path;
			std::ostringstream output_pdb_prefix;
			if ( options.use_batch_id ) {
				utility::file::FileName full_path( scaffold_filename ); // used to get base filename & path
				output_pdb_prefix << options.batch_id << "_" << base_pdb_filename( full_path.base() ) << ".m" << match_counter;
			} else {
				output_pdb_prefix << base_pdb_filename( scaffold_filename ) << ".m" << match_counter;
			}
			middle_path << output_pdb_prefix.str();

			// grab graft info for processing
			GraftInfo & graft_info = graft_infos[ match_counter ];

			// need to rescore since match results may not have transformation matrix or may want alternate transformation matrix due to input_pdb_has_Ab
			rescore_match( options, native, scaffold, graft_info.match_result(), scaffold_filename, user_specified_superposition_residues );

			// lock graft info to perform range checks
			graft_info.lock();

			// instantiate initial epitope scaffold object
			if ( !options.idealize_loop_geometry ) {
				std::cout << "! WARNING: Keeping native bond/angle geometry around loops.  This mode is primarily for testing, check your solutions otherwise." << std::endl;
				std::cout << "! Mode not yet ready for general use, exiting." << std::endl;
				std::exit( 1 );
			}
			EpitopeScaffold epitope_scaffold( scaffold, native.antigen(), graft_info, keep_natro_residues, options.micromanage_termini, options.idealize_loop_geometry );

			// setup EpitopeScaffold options
			epitope_scaffold.set_chainbreak_criterion( 0.2 );
			epitope_scaffold.set_local_rama_criterion( options.max_local_rama );
			epitope_scaffold.set_allow_Ab_repack( true );
			epitope_scaffold.set_use_sequence_biased_fragments( options.use_sequence_biased_fragments ); // default false
			epitope_scaffold.set_use_variable_length_fragments( options.use_variable_length_fragments ); // default false
			epitope_scaffold.set_number_of_fragments( options.number_of_fragments ); // default 200

			// set and alter Pose to reflect moveable/grow/remove closure residues as well as
			// any changes from double -> single break specified in GraftInfo
			epitope_scaffold.alter_closure_sites();

			// set antibody
			if ( native.Ab_exists() ) {
				epitope_scaffold.set_Ab( native.Ab() ); // antibody reoriented, but not connected
			}

			// status
			// TODO: consider removing this status output
			std::cout << "MultiGraft: potential primary closures:" << std::endl;
			std::set< LoopClosureInfo > primary_closures = epitope_scaffold.primary_closures_to_attempt();
			for ( std::set< LoopClosureInfo >::const_iterator a = primary_closures.begin(), ae = primary_closures.end(); a != ae; ++a ) {
				std::cout << "   " << a->to_string() << std::endl;
			}
			std::cout << "MultiGraft: potential secondary closures:" << std::endl;
			std::set< LoopClosureInfo > secondary_closures = epitope_scaffold.secondary_closures_to_attempt();
			for ( std::set< LoopClosureInfo >::const_iterator a = secondary_closures.begin(), ae = secondary_closures.end(); a != ae; ++a ) {
				std::cout << "   " << a->to_string() << std::endl;
			}

			// switch to closure residue type
			epitope_scaffold.convert_to_residue_type( options.closure_residue_type ); // ala by default, unless -close_as_GLY

			// connect Ab
			epitope_scaffold.connect_Ab();

			// prime log directory
			std::ostringstream log_dir;
			log_dir << files_paths::pdb_out_path << middle_path.str() << '/' << "log" << '/';
			recursively_create_directory( log_dir.str() );

			// prime build directory for storing centroid closures
			std::ostringstream build_dir;
			build_dir << files_paths::pdb_out_path << middle_path.str() << '/' << "centroid_build" << '/';
			recursively_create_directory( build_dir.str() );

			// see if there are any files in directory
			Size n_closed = count_n_files( build_dir.str() );
			std::cout << "* STATUS " << n_closed << " files found in " << build_dir.str() << " directory" << std::endl;

			// checkpoint, could point to either build stage or design-refine stage
			Integer checkpoint_index = 0;
			if ( options.use_checkpoint ) {
				checkpoint_peek( options.checkpoint_filename_inner, stage, checkpoint_index );
			}

			// build 'n' structures first
			Size build_attempts = checkpoint_index;
			Size max_build_attempts = static_cast< Size >( options.nstruct * options.build_attempt_multiplier );
			while( n_closed < static_cast< Size >( options.nstruct ) && build_attempts < max_build_attempts ) {
				std::cout << "* BUILD nstruct: " << n_closed << " / " << options.nstruct << "   attempts: " << build_attempts << " / " << max_build_attempts << std::endl;

				EpitopeScaffold work_es = epitope_scaffold;

				// centroid closure
				build_stage( options, work_es, native.Ab(), stage, build_attempts );

				// dump structure if closed
				if ( work_es.all_loops_closed() ) {
					++n_closed;

					std::ostringstream out_fn;
					out_fn << build_dir.str() << output_pdb_prefix.str() << '.' << n_closed << ".build.pdb.gz";

					Pose output_complex;
					work_es.extract_complex( output_complex );
					output_pdb( out_fn.str(), output_complex, true ); // boolean: include_decoy_features
				}
			}

			// prime build directory for storing design-refine
			std::ostringstream dr_dir;
			dr_dir << files_paths::pdb_out_path << middle_path.str() << '/' << "design_refine" << '/';
			recursively_create_directory( dr_dir.str().c_str() );

			// now design-refine all of them
			Size f = count_n_files( dr_dir.str() );
			std::vector< std::string > build_base_filenames = dir( build_dir.str() );
			assert( static_cast< Size >( options.nstruct ) == build_base_filenames.size() );
			for ( Size fe = build_base_filenames.size(); f < fe; ++f ) {
				std::cout << "* DR struct: " << f << std::endl;
				// load the es from disk
				EpitopeScaffold work_es = epitope_scaffold;
				reload_epitope_scaffold( build_dir.str() + build_base_filenames[ f ], native.Ab(), work_es, true ); // boolean: also_replace_archive
				work_es.connect_Ab();

				// full-atom design-refine cycles
				dr_stage( options, work_es, native.Ab(), stage, checkpoint_index );

				// output filenames
				std::ostringstream out_fn;
				out_fn << dr_dir.str() << output_pdb_prefix.str() << '.' << ( f + 1 ) << ".dr.pdb.gz";
				std::ostringstream log_fn;
				log_fn << log_dir.str() << output_pdb_prefix.str() << '.' << ( f + 1 ) << ".log";

				// output pdb and log
				final_output( work_es, log_fn.str(), out_fn.str(), scaffold_filename, match_counter, wt_scaffold_score, wt_ddG );

				// dump structure if finished
//				if ( work_es.all_loops_closed() ) {
//					std::ostringstream out_fn;
//					out_fn << build_dir.str() << output_pdb_prefix.str() << '.' << ( f + 1 ) << ".dr.pdb.gz";
//
//					Pose output_complex;
//					work_es.extract_complex( output_complex );
//					output_pdb( out_fn.str(), output_complex, true ); // boolean: include_decoy_features
//				}
			}

		} // foreach match result

		// need to reset match counter
		match_counter = 1;

	} // foreach scaffold

	// close output file
	outfile.close();

	// checkpointing: the rest of the run is slow enough that open/update/close is ok
	if ( options.use_checkpoint ) {
		outer_checkpoint.set_current_id( scaffold_filenames.size() + 1 );
		outer_checkpoint.set_current_sub_id( 0 );
		outer_checkpoint.update();
		outer_checkpoint.close();
	}

	std::cout << "MultiGraft: finished" << std::endl;

}


/// @brief centroid level build stage
void
build_stage(
	GraftOptions const & options,
	EpitopeScaffold & work_es,
	Pose const & native_ab,
	MultiGraftStage & stage,
	Size & build_attempts
)
{
	// checkpointing
	if ( stage == CLOSURE ) {
		std::cout << "* build CHECKPOINTED starting from stage " << stage << " from file " << ( options.checkpoint_filename + ".build.pdb.gz" ) << std::endl;
		reload_epitope_scaffold( options.checkpoint_filename + ".build.pdb.gz", native_ab, work_es, false ); // boolean: also_replace_archive
		work_es.load_loop_checkpoint( options.checkpoint_filename + ".build_loops" );
		work_es.convert_to_residue_type( options.closure_residue_type ); // ala by default, unless -close_as_GLY
		work_es.connect_Ab();
	} else {
		work_es.clear_closed_during_trajectory();
	}

	work_es.set_repick_fragments( true );

	stage = CLOSURE;
	Size n_open_loops;
	Size local_attempts = 0;
//	Size max_local_attempts = static_cast< Size >( options.build_attempt_multiplier );
	Size max_build_attempts = static_cast< Size >( options.nstruct * options.build_attempt_multiplier );
	while ( !work_es.all_loops_closed() && build_attempts < max_build_attempts ) {
		std::cout << "* build:   " << local_attempts << "   "
		          << build_attempts << " / " << max_build_attempts << std::endl;

		n_open_loops = work_es.n_open_loops();

		// reset state of broken loops every so often to get away from trapped minima
		if ( local_attempts % options.closure_reset_period == 0 ) {
			std::cout << "* randomizing broken closure site phi-psi and cutpoints" << std::endl;

			// randomize cutpoints
			work_es.randomize_broken_closure_site_cutpoints();

			// randomize phi/psi of moveable residues except for the lever side of the primary
			// if N2C or C2N
			work_es.randomize_broken_closure_site_phipsi( false ); // boolean: also_randomize_lever
		}

		// if user requests completely degenerate fragment insertion
		if ( options.allow_any_ss_during_fragment_insertion ) {
			work_es.alter_secondary_structure_of_moveable_residues( 'D' );
		}

		work_es.switch_to_centroid();
		work_es.build_all_simul( true, EpitopeScaffold::NORMAL_BUILD ); // boolean: ignore closure state
		work_es.set_repick_fragments( false );

		// checkpoint only if necessary
		if ( !work_es.all_loops_closed() && work_es.n_open_loops() < n_open_loops ) {
			// make a copy so we ensure work_es is not perturbed
			EpitopeScaffold scratch_es = work_es;
			scratch_es.disconnect_Ab();
			scratch_es.switch_to_fullatom();
			scratch_es.recover_native_epitope_scaffold_sidechains();
			scratch_es.connect_Ab();
			scratch_es.pose().dump_pdb( options.checkpoint_filename + ".build.pdb.gz" );
			scratch_es.save_loop_checkpoint( options.checkpoint_filename + ".build_loops" );
			n_open_loops = work_es.n_open_loops(); // check work_es, not scratch_es
		}

		// increment counters
		++local_attempts;
		++build_attempts;

		// write total attempts
		if ( options.use_checkpoint ) {
			checkpoint_save( options.checkpoint_filename_inner, stage, build_attempts );
		}
	}

	if ( work_es.all_loops_closed() ) { // prepare for full-atom design/refine
		work_es.disconnect_Ab();

		// now switch to fullatom (note that this repacks since fullatom data is invalidated,
		// but since all-gly or all-ala at this point there's no problem)
		work_es.switch_to_fullatom();

		// revive native epitope sidechains and Ab sidechains if present
		work_es.recover_native_epitope_scaffold_sidechains();

		work_es.connect_Ab();

		if ( options.use_checkpoint ) {
			remove( std::string( options.checkpoint_filename + ".build.pdb.gz" ).c_str() );
			remove( std::string( options.checkpoint_filename + ".build_loops" ).c_str() );
		}
	}

	stage = ONGOING;

	// clear checkpoint
	if ( options.use_checkpoint ) {
		checkpoint_save( options.checkpoint_filename_inner, stage, build_attempts );
	}
}


/// @brief full-atom design-refine stage
void
dr_stage(
	GraftOptions const & options,
	EpitopeScaffold & work_es,
	Pose const & native_ab,
	MultiGraftStage & stage,
	Size const & checkpoint_index
)
{
	Size cycle = 0;

	if ( options.use_checkpoint && ( stage == DESIGN || stage == REFINE ) ) {
		cycle = checkpoint_index;
		std::cout << "* dr CHECKPOINTED starting from stage " << stage << " from file " << ( options.checkpoint_filename + ".dr.pdb.gz" ) << std::endl;
		reload_epitope_scaffold( options.checkpoint_filename + ".dr.pdb.gz", native_ab, work_es, false ); // boolean: also_replace_archive
		work_es.connect_Ab();
		work_es.refresh_cached_design_positions();
		work_es.load_loop_checkpoint( options.checkpoint_filename + ".refine_loops" );
	} else {
		work_es.clear_closed_during_trajectory();
	}

	// 3 cycles of design-refine
	for ( ; cycle < 3; ++cycle ) {
		std::cout << "* dr cycle: " << cycle << std::endl;

		if ( options.use_checkpoint && ( stage == DESIGN || stage == REFINE ) ) {
			checkpoint_save( options.checkpoint_filename_inner, stage, cycle );
		}

		stage = DESIGN;
		work_es.design();

		stage = REFINE;
//		Size refine_attempts = 0;
		work_es.clear_closed_during_trajectory();
		work_es.refine_all_simul( true, EpitopeScaffold::CLASSIC_REFINE_TWO ); // accept any chainbreak for now
//		while ( !work_es.all_loops_closed() && refine_attempts < static_cast< Size >( options.max_refine_attempts_per_round ) ) {
//			work_es.refine_all_simul( true, EpitopeScaffold::CLASSIC_REFINE_TWO );
//			++refine_attempts;
//		}

		if ( options.use_checkpoint ) {
			work_es.pose().dump_pdb( options.checkpoint_filename + ".dr.pdb.gz" );
			work_es.save_loop_checkpoint( options.checkpoint_filename + ".refine_loops" );
		}
	}

	if ( options.use_checkpoint ) {
		remove( std::string( options.checkpoint_filename + ".dr.pdb.gz" ).c_str() );
		remove( std::string( options.checkpoint_filename + ".refine_loops" ).c_str() );
	}

	stage = ONGOING;

	// clear checkpoint
	if ( options.use_checkpoint ) {
		checkpoint_save( options.checkpoint_filename_inner, stage, 0 );
	}
}


/// @brief output final structure and log
void
final_output(
	EpitopeScaffold const & es,
	std::string const & log_filename,
	std::string const & es_pdb_filename,
	std::string const & scaffold_filename,
	Size const match_counter,
	Real const wt_scaffold_score,
	Real const wt_ddG
)
{
	using namespace pose_ns;

	// first output the final structure
	Pose output_complex;
	es.extract_complex( output_complex );
	Real const es_complex_score = output_complex.score( Score_weight_map( score12 ) );
	output_pdb( es_pdb_filename, output_complex, true ); // boolean: include_decoy_features

	// temp cache
	GraftInfo const & graft_info = es.graft_info();

	// next output the log
	utility::io::ozstream logfile( log_filename );

	logfile << "##### " << es_pdb_filename << '\n';
	logfile << "##### " << scaffold_filename << "  match  " << match_counter << '\n';
	logfile << graft_info.match_result().to_string( scaffold_filename, "", "#####" );
	logfile << graft_info.to_string( scaffold_filename, "####", false );
	logfile << "#\n";

	logfile << es.to_string();
	logfile << "#\n";

	// epitope scaffold scores
	Pose es_design;
	es.extract_epitope_scaffold( es_design );
	Real const es_design_score = es_design.score( Score_weight_map( score12 ) );
	Pose es_ab;
	es.extract_Ab( es_ab );
	Real const es_ab_score = es_ab.score( Score_weight_map( score12 ) );
	Real const es_ddG = es_complex_score - ( es_design_score + es_ab_score );

	logfile << "# * Design Scores\n";
	logfile << "#\n";
	logfile << "# design_scaffold = " << es_design_score << "\n";
	logfile << "# design_ab = " << es_ab_score << "\n";
	logfile << "# design_ddG = " << es_ddG << "\n";
	logfile << "#\n";

	// rudimentary stability comparison
	logfile << "# * Stability Scores\n";
	logfile << "#\n";
	logfile << "# WT_scaffold = " << wt_scaffold_score << "\n";
	logfile << "# WT_ddG_native_complex = " << wt_ddG << "\n";
	logfile << "#\n";

	logfile.close();
}


} // namespace design
} // namespace epigraft
