// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   design_functions.hh
/// @brief  Miscellaneous design functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


#ifndef INCLUDED_epigraft_design_design_functions_HH_
#define INCLUDED_epigraft_design_design_functions_HH_

// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/ResidueRange.hh>

// roostock headers
#include <rootstock/Octree.hh>

// rosetta headers
#include <InteractionGraphBase.h>
#include <PackerTask.h>
#include <pose.h>
#include <RotamerSet.h>

// C++ headers
#include <iostream>
#include <set>


namespace epigraft {
namespace design {


/// @brief copy sidechains from full-atom sc-pose to full-atom bb-pose
/// @note  range copied in bb-pose and sc-pose is the same
/// @warning both Poses must be full-atom before calling this!
void
copy_sidechains(
	Pose & bb_pose,
	Pose & sc_pose,
	ResidueRange const & range
);


/// @brief copy sidechains from full-atom sc-pose to full-atom bb-pose
/// @details Iterator is iterator from a container that stores type integer
/// @note  range copied in bb-pose and sc-pose is the same
/// @warning both Poses must be full-atom before calling this!
template< typename Iterator >
inline
void
copy_sidechains(
	Pose & bb_pose,
	Pose & sc_pose,
	Iterator const & begin,
	Iterator const & end
)
{
	// make sure both bb and sc Poses are fullatom
	if ( !( bb_pose.fullatom() && sc_pose.fullatom() ) ) {
		std::cerr << "ERROR -- epigraft::design::copy_sidechains() called with a Pose that is not full-atom, refusing to copy!" << std::endl;
		return;
	}

	// cache
	FArray3D_float const & sc_pose_full_coord = sc_pose.full_coord();

	// copy sidechains
	for ( Iterator i = begin; i != end; ++i ) {
		Integer const & res = *i;
		bb_pose.copy_sidechain( res, sc_pose.res( res ), sc_pose.res_variant( res ), sc_pose_full_coord( 1, 1, res ) );
	}
}


/// @brief design using design matrix
void
design(
	Pose & input,
	FArray2D< bool > const & design_matrix
);


/// @brief design using design matrix, returning design
///  objects so they can be re-used in subsequent design
///  call
void
design(
	Pose & input,
	FArray2D< bool > const & design_matrix,
	PackerTask & task,
	pack::InteractionGraphBase * & ig,
	RotamerSet & rs,
	FArray1D_int & current_rot_index,
	FArray1D_float & ligenergy1b_old
);


/// @brief design using Task and given objects
void
design(
	Pose & input,
	PackerTask & task,
	pack::InteractionGraphBase * & ig,
	RotamerSet & rs,
	FArray1D_int & current_rot_index,
	FArray1D_float & ligenergy1b_old,
	bool setup = false
);



/// @brief mutate range to residue type
/// @brief param[in] keep_gly will mutate all but glycines if true
void
mutate_range(
	Pose & pose,
	ResidueRange const & range,
	Integer const & aa,
	bool const & keep_gly = false
);


/// @brief mutate range to residue type
/// @brief param[in] keep_gly will mutate all but glycines if true
template< typename BooleanType >
void
mutate_range(
	Pose & pose,
	BooleanType const & mutate_residue,
	Integer const & aa,
	bool const & keep_gly = false
)
{
	FArray2D< bool > design_matrix( param::MAX_AA(), pose.total_residue(), false );

	// lock chi
	pose.set_allow_chi_move( false );

	// set residue identities
	for ( Size i = 1, ie = mutate_residue.size(); i <= ie; ++i ) {
		if ( mutate_residue[ i ] ) {
			if ( pose.res( i ) != param_aa::aa_gly || ( !keep_gly ) ) {
				design_matrix( aa, i ) = true;
			}

			pose.set_allow_chi_move( i, true );
		}
	}

	// repack to get right coordinates
	epigraft::design::design( pose, design_matrix );

	// re-lock chi
	pose.set_allow_chi_move( false );
}


/// @brief fill octree with sidechain atoms only
void
fill_octree_sc(
	rootstock::Octree< AtomPoint > & oc,
	Pose const & input
);


/// @brief fill octree with sidechain atoms only given range
void
fill_octree_sc(
	rootstock::Octree< AtomPoint > & oc,
	Pose const & input,
	ResidueRange const & range
);


/// @brief given a query residue, return the residues that are nearby (in the octree)
std::set< Integer >
near_residues(
	Real const & distance_cutoff,
	rootstock::Octree< AtomPoint > const & oc,
	Pose const & query,
	Integer const & query_residue
);


/// @brief given a query residue's sidechain, return the residues that are nearby (in the octree)
std::set< Integer >
near_residues_to_sc(
	Real const & distance_cutoff,
	rootstock::Octree< AtomPoint > const & oc,
	Pose const & query,
	Integer const & query_residue
);


/// @brief given a query structure, return the residues that are nearby (in the octree)
std::set< Integer >
near_residues_to_sc(
	Real const & distance_cutoff,
	rootstock::Octree< AtomPoint > const & oc,
	Pose const & query
);


/// @brief given a query structure, return the residues that are nearby on the query structure
std::set< Integer >
near_residues_to_sc_in_query(
	Real const & distance_cutoff,
	rootstock::Octree< AtomPoint > const & oc,
	Pose const & query
);


} // namespace design
} // namespace epigraft


#endif /*INCLUDED_epigraft_design_design_functions_HH_*/
