// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   design_io.cc
/// @brief  Design i/o functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/design/design_io.hh>

// package headers
#include <epigraft/design/GraftInfo.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/epigraft_io.hh>

// Rosetta headers
#include <decoy_features.h>
#include <score.h>

// utility headers
#include <utility/file/FileName.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/vector1.hh>

// ObjexxFCL headers
#include <ObjexxFCL/string.functions.hh>

// C++ headers
#include <map>
#include <set>
#include <sstream>
#include <string>


namespace epigraft {
namespace design {


/// @brief read keep natural rotamer file for epitope
/// @note  "keep natro" file indicates epitope positions whose sidechains
/// @note  are to be kept/not allowed to be designed
/// @warning currently does not handle insertion codes -- only integer residue numbering!
std::set< Integer >
parse_keep_natro_file(
	String const & filename
)
{
	utility::io::izstream infile ( filename );

	if ( !infile ) {
		std::ostringstream ss;
		ss << "parse_keep_natro_file: cannot open file " << filename << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}

	std::set< Integer > keep_natro_residues;

	String line;
	utility::vector1< String > entries;
	while ( getline( infile, line ) ) {
		entries.clear();
		split_string( line, entries ); // grab all columns from input file

		if ( entries[ 1 ].substr( 0 , 1 ) == "#" || entries.size() == 0 ) { // skip comment lines
			continue;
		}

		Integer res;
		std::istringstream( entries[ 1 ] ) >> res;
		keep_natro_residues.insert( res );
	}

	infile.close();

	return keep_natro_residues;
}


/// @brief read complementarity design residue file
/// @details residue file contains list of residues on the antibody
/// @details in *native* antibody numbering (including insertion code),
/// @details one residue per line
std::set< String >
parse_complementarity_residue_file(
	String const & filename
)
{
	utility::io::izstream infile ( filename );

	if ( !infile ) {
		std::ostringstream ss;
		ss << "parse_complementarity_residue_file: cannot open file " << filename << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}

	std::set< String > residues;

	String line;
	utility::vector1< String > entries;
	while ( getline( infile, line ) ) {
		entries.clear();
		split_string( line, entries ); // grab all columns from input file

		if ( entries[ 1 ].substr( 0 , 1 ) == "#" || entries.size() == 0 ) { // skip comment lines
			continue;
		}

		residues.insert( entries[ 1 ] );
	}

	infile.close();

	return residues;
}


/// @brief wrapper for Pose dump_pdb -- checks if directory path exists first and makes it if necessary
void
output_pdb(
	String const & filename,
	Pose & pose,
	bool const & include_decoy_features,
	bool const & use_pdb_info
)
{
	using namespace pose_ns;

	utility::file::FileName fn( filename );
	recursively_create_directory( fn.path() );

	pose.pdb_info().set_use_pdb_numbering( use_pdb_info );

	// open file
	utility::io::ozstream out( filename );

	// score
	pose.dump_scored_pdb( out, Score_weight_map( score12 ) );

	if ( include_decoy_features ) {
		// we control the flag locally since decoy features has
		// a hook into make_pdb, which typically we don't want to trigger
		decoy_features_ns::set_df_flag( true );

		// DecoyFeatures code needs misc to be current.  This should
		// happen in the previous scoring call, but we reload as a
		// precaution.
		pose.copy_to_misc();

		// load decoy features from misc
		// the buried polar code in DecoyFeatures relies on the fullatom
		// energy arrays to be current, so these steps are done *after*
		// the scoring call provided by Pose::dump_scored_pdb()
		//
		// there is at least one, if not more, logic bugs in DecoyFeatures
		// so initializing both current & native must happen for this to work
		decoy_features_ns::nativeDF.initFromMisc();
		decoy_features_ns::currentDF.initFromMisc();

		// output DecoyFeatures
		decoy_features_ns::currentDF.print( out );

		decoy_features_ns::set_df_flag( false );
	}

	// close file
	out.close();
}


/// @brief match result from file (from output table) and corresponding graft info
/// @brief fill in as much data as possible from given file
void
graft_info_from_file(
	String const & filename,
	utility::vector1< LoopInfo > loops_to_scan,
	std::map< String, utility::vector1< GraftInfo > > & pdb_to_graft_info,
	bool const & using_old_graft_info_format
)
{
	using std::istringstream;
	using epigraft::match::MatchComponent;
	using epigraft::match::MatchResult;
	using epigraft::match::align::AlignmentSystem;
	using epigraft::conformation::DihedralInfo;

	utility::io::izstream infile ( filename );
	if ( !infile ) {
		std::ostringstream ss;
		ss << "match_results_from_file: cannot open file " << filename << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}

	// cache loop offsets for loop id computation later in the procedure
	utility::vector1< int > loop_offsets;
	for ( Integer loop_id = 1, last_loop_id = loops_to_scan.size(); loop_id <= last_loop_id; ++loop_id ) {
		loop_offsets.push_back( compute_loop_offset( loops_to_scan, loop_id ) );
	}

	String line;

	bool past_beginning = false;
	MatchResult match_result;
	String pdb_filename;
	utility::vector1< String > entries;
	utility::vector1< String > graft_info_lines;
	Size dollar_sign = 0, mr_entry_size = 0;
	while ( getline( infile, line ) ) {

		entries.clear();
		split_string( line, entries ); // grab all columns from input file for match result

		if ( entries[ 1 ].substr( 0 , 1 ) == "#" ) { // skip comment lines
			continue;
		}

		// find index of dollar sign (indicates presence of graft info)
		// 0 implies not found
		dollar_sign = index_of_dollar( entries );

		// compute number of entries corresponding to match result
		mr_entry_size = entries.size();
		if ( dollar_sign ) {
			mr_entry_size = dollar_sign - 1;
		}

		if ( mr_entry_size < 7 ) {
			std::ostringstream ss;
			ss << "match_results_from_file: following line in match results file missing required number of columns:" << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}

		// decide if this is a primary component or secondary component
		bool processing_primary = false;
		if ( *( entries.begin()->begin() ) != '*' ) { // primary

			if ( past_beginning ) {
				assert( match_result.components.size() == graft_info_lines.size() );

				// finalize handling of the prior match and store it via graft info
				GraftInfo graft_info( match_result );

				// add proper graft info ranges etc
				for ( Size i = 1, ie = graft_info_lines.size(); i <= ie; ++i ) {
					if ( graft_info_lines[ i ] != "#" ) {
						if ( using_old_graft_info_format ) {
							graft_info.add_info_from_line( match_result.components[ i ].scaffold_gap_range.begin(),
							                               match_result.components[ i ].scaffold_gap_range.end(),
							                               graft_info_lines[ i ], dollar_sign + 1 );
						} else {
							Size const dollar_sign_index = graft_info_lines[ i ].find( '$' );
							graft_info.add_encoded_info_from_line( match_result.components[ i ].scaffold_gap_range.begin(),
			                                                       match_result.components[ i ].scaffold_gap_range.end(),
			                                                       graft_info_lines[ i ].substr( dollar_sign_index + 1, graft_info_lines[ i ].length() - dollar_sign_index - 1 ) );
						}
					}
				}
				graft_info_lines.clear(); // will be start of new match, so clear

				// store graft info
				pdb_to_graft_info[ pdb_filename ].push_back( graft_info );
			} else {
				past_beginning = true;
			}

			// now do new MatchResult starting from new primary
			pdb_filename = entries[ 1 ];
			processing_primary = true;
			match_result = MatchResult();

			// alignment system
			match_result.system_name = entries[ 3 ];
			String sys = match_result.system_name.substr( 0, 3 );
			if ( sys == "N2C" ) {
				match_result.system_type = AlignmentSystem::N2C;

				String center = match_result.system_name.substr( 4, 2 );
				if ( center == "N" ) {
					match_result.alignment_center = AlignmentSystem::N;
				} else if ( center == "CA" ) {
					match_result.alignment_center = AlignmentSystem::CA;
				} else if ( center == "C" ) {
					match_result.alignment_center = AlignmentSystem::C;
				}
			} else if ( sys == "C2N" ) {
				match_result.system_type = AlignmentSystem::C2N;

				String center = match_result.system_name.substr( 4, 2 );
				if ( center == "N" ) {
					match_result.alignment_center = AlignmentSystem::N;
				} else if ( center == "CA" ) {
					match_result.alignment_center = AlignmentSystem::CA;
				} else if ( center == "C" ) {
					match_result.alignment_center = AlignmentSystem::C;
				}
			} else if ( sys == "E" ) {
				match_result.system_type = AlignmentSystem::ENDPOINT;
				match_result.alignment_center = AlignmentSystem::DOUBLEBREAK;
			} else if ( sys == "S" ) {
				match_result.system_type = AlignmentSystem::SUPERPOSITION;
				match_result.alignment_center = AlignmentSystem::DOUBLEBREAK;
			} else if ( sys == "SS" ) {
				match_result.system_type = AlignmentSystem::SPECIFIC_SUPERPOSITION;
				match_result.alignment_center = AlignmentSystem::DOUBLEBREAK;
			}

			if ( mr_entry_size >= 27 ) { // transformation matrix exists, fill it in
				// remember that FArrays are column-major:
				// | 0 4  8 12 |
				// | 1 5  9 13 |
				// | 2 6 10 14 |
				// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 |
				istringstream( entries[ 16 ] ) >> match_result.transformation_matrix[ 0 ];
				istringstream( entries[ 17 ] ) >> match_result.transformation_matrix[ 1 ];
				istringstream( entries[ 18 ] ) >> match_result.transformation_matrix[ 2 ];
				match_result.transformation_matrix[ 3 ] = 0.0;
				istringstream( entries[ 19 ] ) >> match_result.transformation_matrix[ 4 ];
				istringstream( entries[ 20 ] ) >> match_result.transformation_matrix[ 5 ];
				istringstream( entries[ 21 ] ) >> match_result.transformation_matrix[ 6 ];
				match_result.transformation_matrix[ 7 ] = 0.0;
				istringstream( entries[ 22 ] ) >> match_result.transformation_matrix[ 8 ];
				istringstream( entries[ 23 ] ) >> match_result.transformation_matrix[ 9 ];
				istringstream( entries[ 24 ] ) >> match_result.transformation_matrix[ 10 ];
				match_result.transformation_matrix[ 11 ] = 0.0;
				istringstream( entries[ 25 ] ) >> match_result.transformation_matrix[ 12 ];
				istringstream( entries[ 26 ] ) >> match_result.transformation_matrix[ 13 ];
				istringstream( entries[ 27 ] ) >> match_result.transformation_matrix[ 14 ];
				match_result.transformation_matrix[ 15 ] = 1.0;
			}
		} // primary

		// now store loop info in component
		MatchComponent component;
		istringstream( entries[ 2 ] ) >> component.loop_id;

		Integer gap_begin;
		istringstream( entries[ 4 ] ) >> gap_begin;
		Integer gap_end;
		istringstream( entries[ 5 ] ) >> gap_end;
		component.scaffold_gap_range = ResidueRange( gap_begin, gap_end );

		Integer native_loop_begin;
		istringstream( entries[ 6 ] ) >> native_loop_begin;
		Integer native_loop_end;
		istringstream( entries[ 7 ] ) >> native_loop_end;
		component.native_loop_subrange = ResidueRange( native_loop_begin, native_loop_end );

		// Important: component.loop_subrange is _not_ filled in here, this is done in external post-processing

		// for 'S' and 'SS' systems, make sure the number of residues is the same
		if ( processing_primary && ( match_result.system_type == AlignmentSystem::SUPERPOSITION ||
		     match_result.system_type == AlignmentSystem::SPECIFIC_SUPERPOSITION ) ) {
		     	if ( component.scaffold_gap_range.length() != component.native_loop_subrange.length() ) {
					std::ostringstream ss;
					ss << "match_results_from_file: S/SS system indicated in the following line, but there's a mismatch in the number of residues in gap versus loop:" << std::endl;
					ss << line << std::endl;
					utility::exit( __FILE__, __LINE__, ss.str() );
		     	}
		}

		// now correct loop ids so they match those is available LoopInfo
		// and fill in proper loop subranges w/ respect to all loops
		// note that loop id is currently artificial and is just meant to match
		// the index of the loop in its vector
		for ( Integer loop_id = 1, last_loop_id = loops_to_scan.size(); loop_id <= last_loop_id; ++loop_id ) {
			LoopInfo const & loop = loops_to_scan[ loop_id ];

			if ( loop.full_native_range().contains( component.native_loop_subrange ) ) {
				component.loop_id = loop_id;
				component.loop_subrange = loop.convert_native_to_internal( component.native_loop_subrange ) + loop_offsets[ loop_id ];
			}

		}

		// component dihedrals?
		if ( mr_entry_size >= 15 ) {
			for ( Integer i = 8; i <= 15; ++i ) {
				if ( entries[ i ] != "-" ) {
					replace( entries[ i ].begin(), entries[ i ].end(), ':', ' ' ); // dihedral information delimited by ':'

					istringstream dss( entries[ i ] );

					Integer residue;
					Integer a_t; // of type DihedralInfo::AngleType
					Real angle;

					dss >> residue;
					dss >> a_t;
					dss >> angle;

					component.dihedrals.insert( DihedralInfo( (DihedralInfo::AngleType)a_t, residue, angle ) );
				}
			}
		}

		match_result.components.push_back( component );

		// store processed line
		if ( dollar_sign ) {
			graft_info_lines.push_back( line ); // for creating graft info
		} else {
			graft_info_lines.push_back( "#" );
		}

	} // foreach line

	// handle last entry
	if ( past_beginning ) {
		assert( match_result.components.size() == graft_info_lines.size() );

		// finalize handling of the prior match and store it via graft info
		GraftInfo graft_info( match_result );

		// add proper graft info ranges etc
		for ( Size i = 1, ie = graft_info_lines.size(); i <= ie; ++i ) {
			if ( graft_info_lines[ i ] != "#" ) {
				if ( using_old_graft_info_format ) {
					graft_info.add_info_from_line( match_result.components[ i ].scaffold_gap_range.begin(),
					                               match_result.components[ i ].scaffold_gap_range.end(),
					                               graft_info_lines[ i ], dollar_sign + 1 );
				} else {
					Size const dollar_sign_index = graft_info_lines[ i ].find( '$' );
					graft_info.add_encoded_info_from_line( match_result.components[ i ].scaffold_gap_range.begin(),
	                                                       match_result.components[ i ].scaffold_gap_range.end(),
	                                                       graft_info_lines[ i ].substr( dollar_sign_index + 1, graft_info_lines[ i ].length() - dollar_sign_index - 1 ) );
				}
			}
		}
		graft_info_lines.clear(); // will be start of new match, so clear

		// store graft info
		pdb_to_graft_info[ pdb_filename ].push_back( graft_info );
	}

	infile.close(); // close input file

}


/// @brief helper function, search for index of "$" in vector1 of strings
Size
index_of_dollar(
	utility::vector1< String > const & v
)
{
	for ( Size i = 1, ie = v.size(); i <= ie; ++i ) {
		if ( v[ i ] == "$" ) {
			return i;
		}
	}

	return 0; // if not found
}


/// @brief   remove all spaces from string
/// @details first removes all leading/trailing all types of whitespace, then
///          removes all spaces (just spaces, not including other whitespace,
///          like tabs)
String
collapsed_spaces(
	String const & s_in
)
{
	// strip whitespace first
	String s = ObjexxFCL::stripped_whitespace( s_in );

	return collapsed_character( s, ' ' );
}


/// @brief   remove all copies of character from string
/// @details removes all copies of character 'c' found in string
String
collapsed_character(
	String const & s_in,
	char const & c
)
{
	// now run through and construct new string, skipping spaces
	std::ostringstream ss;
	for ( String::const_iterator i = s_in.begin(), ie = s_in.end(); i != ie; ++i ) {
		if ( *i != c ) {
			ss << (*i);
		}
	}

	return ss.str();
}


} // namespace design
} // namespace epigraft
