// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   fragment_functions.cc
/// @brief  Miscellaneous fragment functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Possu Huang (possu@u.washington.edu)


// unit headers
#include <epigraft/design/fragment_functions.hh>
#include <epigraft/design/loop_functions.hh>
#include <epigraft/design/LoopClosureInfo.hh>
#include <epigraft/design/PoseAssembly.hh>
#include <epigraft/design/Vall.hh>

// rosetta headers
#include <after_opts.h>
#include <files_paths.h>
#include <fragments_ns.h>
#include <fragments_pose.h>
#include <pose.h>
#include <pose_io.h>
#include <score.h>
#include <score_data.h>
#include <score_ns.h>

// utility headers
#include <utility/vector1.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray4D.hh>

// C++ headers
#include <algorithm>
#include <iostream>
#include <set>
#include <sstream>
#include <vector>


namespace epigraft {
namespace design {


/// @brief internal struct for heap sorting fragments
struct FragmentScore {
	Size library_index;
	Size position;

	Real match_score;
	Real chainbreak_score;

	FragmentScore() {}
	FragmentScore( Size const & l, Size const & p, Real const & m, Real const & c ) :
		library_index( l ), position( p ), match_score( m ), chainbreak_score( c )
		{}
	~FragmentScore() {}

	inline
	bool
	operator <( FragmentScore const & rval ) const
	{
		return (
			match_score < rval.match_score ? true :
			rval.match_score < match_score ? false :
			chainbreak_score < rval.chainbreak_score
		);
	}
};


/// @brief build 1-mer from 3-mer, for use with Vall only as not all classical fragment data fields filled
void
build_1mer_from_3mer(
	Integer const & total_residue
)
{
	using namespace fragments;

	// is 3-mer library present?
	int bin3 = 0;
	for ( Integer i = 1, ie = files_paths::n_frag_files(); i <= ie; ++i ) {
		if ( files_paths::frag_sizes( i ) == 3 ) {
			bin3 = i;
		}
	}

	// generate 1-mer library from 3-mer library if present
	if ( bin3 > 0 ) {
		Integer const bin1 = 1;
		files_paths::frag_sizes( bin1 ) = 1;

		for ( Integer i = 1; i <= total_residue; ++i ) {
			Integer pos3 = 1; // keep central res of 3-mer
			Integer begin3 = i - 1;

			if ( i == 1 ) {
				begin3 = i; // first res of first frag
				pos3 = 0;
			} else if ( i == total_residue ) {
				begin3 = i - 2; // last res of last frag
				pos3 = 2;
			}

			align_depth( i, bin1 ) = align_depth( begin3, bin3 );

			for ( Integer j = 1, je = align_depth( i, bin1 ); j <= je; ++j ) {
				align_phi(i,j,0,bin1) = align_phi( begin3, j, pos3, bin3 );
				align_psi(i,j,0,bin1) = align_psi( begin3, j, pos3, bin3 );
				align_omega(i,j,0,bin1) = align_omega( begin3, j, pos3, bin3 );
				ss_type(i,j,0,bin1) = ss_type( begin3, j, pos3, bin3 );
			}

		} // foreach residue
	} // bin3 > 0
}


/// @brief parse Vall
void
parse_vall(
	String filename,
	utility::vector1< Vall > & library
)
{
	utility::io::izstream data ( filename );
	if ( !data ) {
		utility::exit( __FILE__, __LINE__, "can't open file: " + filename );
	}

	char line[250];
	Real phi,psi,omega;
	Integer resseq;
	char pdb[5],chain,seq,ss;

	pdb[ 4 ] = NULL;

	String prior_pdb;

	utility::vector1< Vall >::iterator last;

	Size count = 0;
	while ( data ) {
		data.getline( line, 250 );
		if ( data.eof() ) {
			break;
		}

		std::sscanf( line,     "%4s",    pdb );
		std::sscanf( line+4 ,  "%1c", &chain );
		std::sscanf( line+10 , "%5d", &resseq );

		std::sscanf( line+6 , "%1c", &seq );
		std::sscanf( line+8 , "%1c", &ss );

		std::sscanf( line+52, "%9f", &phi );
		std::sscanf( line+61, "%9f", &psi );
		std::sscanf( line+70, "%9f", &omega );

		if ( prior_pdb != pdb ) {
			// tighten memory
			if ( library.size() > 1 ) {
				last->conserve_memory();
			}

			// new pdb
			prior_pdb = pdb;
			library.push_back( Vall() );
			last = --library.end();
		}

		last->push_back( pdb, chain, resseq, seq, ss, phi, psi, omega );

		++count;
		if ( count % 100000 == 0 ) {
			std::cout << "* Parsing Vall @ line: " << count << std::endl;
		}
	}

	data.close();

	Size entries = 0;

	for ( utility::vector1< Vall >::iterator i = library.begin(), ie = library.end(); i != ie; ++i ) {
		entries += i->size();
	}

	std::cout << "* Parsed " << library.size() << " pdbs containing " << entries << " entries from Vall " << filename << std::endl;
}


/// @brief screen fragments sorting by secondary structure and chain break
void
screen_fragments(
	Pose const & pose,
	std::set< LoopClosureInfo > const & closures,
	String const & ss_string,
	String const & aa_string,
	Real const & linear_threshold,
	Integer const & n_fragments
)
{
	using std::push_heap;
	using std::sort_heap;

	static utility::vector1< Vall > library;

	if ( library.size() == 0 ) {
		std::string filename;
		stringafteroption( "vall", "vall.dat.pc60_2002-12-27_v01.2", filename );
		filename = files_paths::data_path + filename;
		parse_vall( filename, library );
	}

	Pose scratch;
	scratch = pose;

	std::vector< FragmentScore > heap;
	Real score = 0.0;
	Real chainbreak = 0.0;

	Real ss_penalty = 1.0;

	for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {

		heap.clear();

		Size const loop_left = *( c->moveable_residues().begin() );
		Size const loop_right = *( --( c->moveable_residues().end() ) );
		Size const loop_width = c->moveable_residues_width();
		Size const cutpoint = c->cut();

		std::cout << "* picking fragments for loop [ " << loop_left << ", " << loop_right << " ]  cut = " << cutpoint << std::endl;

		String desired_ss = ss_string.substr( loop_left - 1, loop_width );
		String desired_seq = aa_string.substr( loop_left - 1, loop_width );

		for ( Size i = 1, ie = library.size(); i <= ie; ++i ) {
			Vall const & v = library[ i ];

			if ( v.size() < loop_width ) {
				std::cout << "*      skipping library index  " << i << " size of " << v.size() << " too small for fragment width" << std::endl;
				continue;
			}

			for ( Size p = 1, pe = v.size() - loop_width + 1; p <= pe; ++p ) {

				score = 0.0;
				for ( Size k = 0, ke = loop_width; k < ke; ++k ) {

					char const ss = v.secstruct( p + k );
					char const seq = v.sequence( p + k );
					Real const phi = v.phi( p + k );
					Real const psi = v.psi( p + k );
					Real const omega = v.omega( p + k );

					if ( ss != desired_ss[ k ] ) {
						score += ss_penalty;
					}

					// TODO: aa penalty?

					scratch.set_phi( loop_left + k, phi );
					scratch.set_psi( loop_left + k, psi );
					scratch.set_omega( loop_left + k, omega );

					// TODO: set secstruct and others?
				}

				chainbreak = score_cut_in_Pose_linear( scratch, cutpoint, 1 );

				heap.push_back( FragmentScore( i, p, score, chainbreak ) );
				push_heap( heap.begin(), heap.end() );
			}

//			std::cout << "*      finished library index  " << i << "  best =  " << heap.begin()->match_score << "  " << heap.begin()->chainbreak_score << std::endl;
			std::cout << "*      finished library index  " << i << std::endl;
		}

		sort_heap( heap.begin(), heap.end() );

		// statistics
		for ( std::vector< FragmentScore >::const_iterator h = heap.begin(), he = heap.end(); h != he; ++h ) {
			if ( h->match_score > 0 ) {
				break;
			}

			std::cout << "$  " << h->library_index << "  " << h->position << "  " << h->match_score << "  " << h->chainbreak_score << std::endl;
		}

	}
}


/// @brief debugging routine, grabs fragment positions in Vall from text file
void
read_fragment_index(
	String const & filename,
//	utility::vector1< Vall > const & library,
	Pose const & pose
)
{
	static utility::vector1< Vall > library;

	if ( library.size() == 0 ) {
		std::string filename;
		stringafteroption( "vall", "vall.dat.pc60_2002-12-27_v01.2", filename );
		filename = files_paths::data_path + filename;
		parse_vall( filename, library );
	}

	// make a working copy of the pose
	Pose scratch;
	scratch = pose;

	utility::io::izstream in( filename );

	String line;
	std::istringstream iss;

	// first line indicates position and width of fragment
	getline( in, line );
	iss.clear();
	iss.str( line );

	Size res;
	Size width;

	iss >> res >> width;

	Size n_fragments = 200; // read the top 200 fragments
	Integer size_bin = 3; // typically the largest bin

	char dollar;
	Size library_index, position;
	std::ostringstream pdb_name;
	for ( Size i = 0; i < n_fragments; ++i ) {
		getline( in, line );
		iss.clear();
		iss.str( line );

		iss >> dollar >> library_index >> position;

		library[ library_index ].copy_to_global_fragment_array( position, width, res, i + 1, size_bin );

//		for ( Size k = 0; k < width; ++k ) {
//			scratch.set_phi( res + k, library[ library_index ].phi( position + k ) );
//			scratch.set_psi( res + k, library[ library_index ].psi( position + k ) );
//			scratch.set_omega( res + k, library[ library_index ].omega( position + k ) );
//		}
//
//		pdb_name.str( " ");
//		pdb_name << i << "." << library_index << "_" << position << ".pdb";
//		scratch.dump_pdb( pdb_name.str() );
	}

	in.close();
}


/// @brief SCRATCH ROUTINE: screen fragments by quadratic chainbreak
///  and output results for all fragments
void
scratch_screen_fragments()
{
	using pose_ns::Score_weight_map;

	String const input_filename = stringafteroption( "input" );
	String const output_filename = stringafteroption( "output" );

	Integer const loop_length = intafteroption( "loop_length" );
	Integer const insert_after = intafteroption( "insert_after" );

	Pose input, scratch;
	pose_from_pdb( input, input_filename, true, false );

	Score_weight_map weight_map;
	weight_map.set_weight( pose_ns::FA_REP, 1.0 );

	input.score( weight_map );
	Real const base_rep = input.get_0D_score( pose_ns::FA_REP );

	std::cout << "*** BASE REP = " << base_rep << std::endl;

	PoseAssembly assembly( input.total_residue() );
	assembly.insert_right( insert_after, loop_length, true, 'L', 'A' );
	assembly.apply( input, scratch );

	// read vall
	static utility::vector1< Vall > library;

	if ( library.size() == 0 ) {
		std::string filename;
		stringafteroption( "vall", "vall.dat.pc60_2002-12-27_v01.2", filename );
		filename = files_paths::data_path + filename;
		parse_vall( filename, library );
	}

	// open output file
	utility::io::ozstream out( output_filename );

	// screen fragments
	std::cout << "* screening for " << loop_length << "-mer loop after residue " << insert_after << std::endl;

	Integer const loop_left = insert_after + 1;
	Integer const loop_width = loop_length;
	Integer const cutpoint = insert_after + loop_length;

	Real score = 0.0;
	Real chainbreak = 0.0;
	char * ss = new char[ loop_width + 1 ];
	ss[ loop_width ] = NULL;

	Size const library_start_index = intafteroption( "library_index_start", 1 );
	Size const library_length_minimum = intafteroption( "library_length_minimum", 1 );

	for ( Size i = library_start_index, ie = library.size(); i <= ie; ++i ) {
		Vall const & v = library[ i ];

		if ( v.size() < library_length_minimum || v.size() < loop_width ) {
			std::cout << "*      SKIPPING library index  " << i << "  length of " << v.size() << " too small" << std::endl;
			continue;
		}

		for ( Size p = 1, pe = v.size() - loop_width + 1; p <= pe; ++p ) {

			for ( Size k = 0, ke = loop_width; k < ke; ++k ) {

//				char const ss = v.secstruct( p + k );
//				char const seq = v.sequence( p + k );
				ss[ k ] = v.secstruct( p + k );
				Real const phi = v.phi( p + k );
				Real const psi = v.psi( p + k );
				Real const omega = v.omega( p + k );

				scratch.set_phi( loop_left + k, phi );
				scratch.set_psi( loop_left + k, psi );
				scratch.set_omega( loop_left + k, omega );
			}

			chainbreak = score_cut_in_Pose_quadratic( scratch, cutpoint, 1 );
			score = scratch.score( weight_map ) - base_rep;
			out << "$  " << i << "  " << p << "  " << score << "  " << chainbreak << "  " << ss << '\n';
		}

		std::cout << "*      finished library index  " << i << std::endl;
	}

	out << "# DONE\n";
	out.close();

	delete[]  ss;
}


} // namespace design
} // namespace epigraft
