// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   loop_functions.cc
/// @brief  Loop closure and movement functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)
/// @author Possu Huang (possu@u.washington.edu)

// unit headers
#include <epigraft/design/loop_functions.hh>

// package headers
#include <epigraft/design/design_constants.hh>
#include <epigraft/design/LoopClosureInfo.hh>
#include <epigraft/design/PoseAssembly.hh>
#include <epigraft/design/ccd_functions.hh>
#include <epigraft/design/design_functions.hh>
#include <epigraft/design/fragment_functions.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/epigraft_functions.hh>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// Rosetta headers
#include <cst_set.h>
#include <fragments.h>
#include <fragments_ns.h>
#include <fragments_pose.h>
#include <jumping_loops.h>
#include <jumping_ns.h>
#include <jumping_refold.h>
#include <jumping_util.h>
#include <loop_relax.h> // eventually remove this and put necessary functions within epigraft/design
#include <loops.h>
#include <misc.h>
#include <minimize.h>
#include <param_aa.h>
#include <pose.h>
#include <pose_loops.h>
#include <ramachandran.h>
#include <random_numbers.h>
#include <score.h>
#include <score_data.h>
#include <score_ns.h>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>

// numeric headers
#include <numeric/xyzVector.hh>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <algorithm>
#include <iostream>
#include <map>
#include <set>
#include <sstream>
#include <string>
#include <utility>


namespace epigraft {
namespace design {


/// @brief   (centroid level) build loop
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details allow_bb_move must be set outside _before_ entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed. Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
build_loop(
	Pose & pose,
	LoopClosureInfo const & closure_info,
	Integer const & outer_cycles,
	bool const & use_fragment_insertion,
	String const & ss_string,
	String const & aa_string,
	Integer const & nfrag,
	bool const & use_boost
)
{
	using namespace pose_ns;

	// setup weights for centroid level remodeling
	// TODO: wrap weights for ease of use
	Score_weight_map weight_map;
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( RG, 1.0 );

	// additional scoring setup
	score_set_cst_mode(3);

	// cutoffs for fragment picking and operations
	Integer cutoff9 = 8; // 16
	Integer cutoff3 = 2; // 6
	Integer cutoff1 = 0; // 3

	// offsets for fragment insertion
	Integer const frag_offset = 0;

	// local variables
	Integer loop_length = closure_info.moveable_residues_width(); // cache

	bool use_fragments = use_fragment_insertion &&
	                     !closure_info.is_closed() &&
	                     ( ( loop_length > cutoff9 ) ||
	                       ( loop_length > cutoff3 ) ||
	                       ( loop_length > cutoff1 ) );

	 // fragment insertion, if requested
	if ( use_fragments ) {
		// number of fragments to use
		Integer const n_fragments = nfrag;

		// weights for fragment picking
		Real sequence_weight = 1.0;
		Real secondary_structure_weight = 1.0;
		Real bigbin_weight = 0.0; // not used

		// create secondary structure string
		std::ostringstream ss_ss;
		if ( ss_string == "" ) {
			for ( Integer r = 1, re = pose.total_residue(); r <= re; ++ r ) {
				ss_ss << pose.secstruct( r );
			}
		} else {
			ss_ss << ss_string;
		}

		// strings indicating desired fragments
		String sequence_str = aa_string == "" ? String( pose.total_residue(), '.' ) : aa_string; // all "." if not used
		String secondary_structure_str( ss_ss.str() );
		String bigbin_str( pose.total_residue(), '.' ); // all "." so not used

		// reset global fragment arrays
		fragments::reset_fragment_arrays_used_by_Vall();

		// pick 9-mers
		if ( loop_length > cutoff9 ) {
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                9, n_fragments,
			                true, true, true, 3 ); // booleans: exclude gly, pro, cis-peptide
		}

		// pick 3-mers
		if ( loop_length > cutoff3 || loop_length > cutoff1)  {
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                3, n_fragments,
			                true, true, true, 2 ); // booleans: exclude gly, pro, cis-peptide
		}

		// build 1-mers
		if ( loop_length > cutoff1 ) {
			build_1mer_from_3mer( pose.total_residue() );
		}

		choose_frag_set_top_N_frags( n_fragments ); // use top 'n' fragments for each size
	} // if using fragment insertion

	// force centroid level
	bool const prior_fullatom_flag = pose.fullatom();
	pose.set_fullatom_flag( false, false ); // boolean: fullatom, repack

	// perturb initial loop conformation if not closed
	if ( use_fragments && !closure_info.is_closed() ) {
		insert_random_frags( 3, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
	}

	// monte carlo cycles
	Integer n_fragment_moves = outer_cycles;
	Integer cycles2 = n_fragment_moves;
	Real const cycle_ratio = std::max( 0.5f, get_looprlx_cycle_ratio() );
	Integer const cycles3 = std::max( 50, (int)( 10 * loop_length * cycle_ratio ) );
	Integer const max_boost_cycles = 100;
	Integer const max_cycles_after_boost = cycles2 + max_boost_cycles;

	// monte carlo
	pose.score( weight_map );
	Monte_carlo mc( pose, weight_map, 2 );

	// chainbreak weighting
	Real const final_weight = 5.0;
	Real const delta_weight = final_weight / cycles2;
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 ); // for scoring chainbreak of the current loop

	// sanity check for ccd
	bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
	utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
	if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
	     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
		do_ccd = false;
	}

	// build procedure
	for ( Integer c2 = 1, overlap; c2 <= cycles2; ++c2 ) { // 10 cycles of chainbreak ramping up to 'final_weight'

		// statistics
		Integer frag_accepts = 0, frag_trials = 0, ccd_accepts = 0, ccd_trials = 0;

		// initialize
		overlap = 0;
		pose = mc.low_pose(); // recover low Pose

		// set chainbreak weights
		if ( closure_info.loop_range().begin() != 1 && closure_info.loop_range().end() != pose.total_residue() ) {
			weight_map.set_weight( CHAINBREAK, c2 * delta_weight );
			if ( c2 > cycles2 / 2 ) {
				overlap = 1;
				weight_map.set_weight( CHAINBREAK_OVERLAP, overlap );
			}

			// score just the current loop's chainbreak
			FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();
			cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
			weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );
		}

		// mc reset using new chainbreak weights
		pose.score( weight_map );
		mc.set_weight_map( weight_map );
		mc.reset( pose );

		for ( Integer c3 = 1, size; c3 <= cycles3; ++c3 ) { // 30 cycles of fragment insertion + ccd move

			if ( use_fragments && ( !do_ccd || overlap == 0 || ran3() * cycles2 > 2 ) ) { // fragment insertion

				// 9-mer
				if ( loop_length > cutoff9 ) {
					size = 9;
					choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
					bool const accepted( mc.boltzmann( pose ) );
					if ( accepted ) {
						++frag_accepts;
					}
					++frag_trials;
				}

				// 3-mer
				if ( loop_length > cutoff3 ) {
					size = 3;
					choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
					bool const accepted( mc.boltzmann( pose ) );
					if ( accepted ) {
						++frag_accepts;
					}
					++frag_trials;
				}

				// 1-mer
				if ( loop_length > cutoff1 ) {
					size = 1;
					choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
					bool const accepted( mc.boltzmann( pose ) );
					if ( accepted ) {
						++frag_accepts;
					}
					++frag_trials;
				}

			} else { // ccd move

				pose_to_misc( pose );
				ccd_moves_obeying_nonideality( 10, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );
//				per_torsion_ccd_moves_obeying_nonideality( 20, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );

				// attempt acceptance
				bool const accepted( mc.boltzmann( pose ) );
				if ( accepted ) {
					++ccd_accepts;
				}
				++ccd_trials;

			} // ccd move

		} // c3

		std::cout << "cycle2: " << c2;
		std::cout << " | frag: " << frag_accepts << " / " << frag_trials << " = " << (float)frag_accepts / (float)( frag_trials == 0 ? 1 : frag_trials );
		std::cout << " | ccd: " << ccd_accepts << " / " << ccd_trials << " = " << (float)ccd_accepts / (float)( ccd_trials == 0 ? 1 : ccd_trials );
		std::cout << std::endl;

		// boost build cycle if chainbreak within range
		if ( use_boost && do_ccd && c2 == cycles2 ) {
			Real chainbreak_score = score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ); // overlap = 1

			if ( 0.2 < chainbreak_score && chainbreak_score < 10.0 ) {
				std::cout << "boosting closure cycle by +5" << std::endl;
				if ( cycles2 <= max_cycles_after_boost ) {
					cycles2 += 5;
				} else {
					std::cout << "boost cycle exceeds max of " << max_boost_cycles << " additional cycles, terminating" << std::endl;
					break;
				}
			}
		} // boost build cycle

	} // c2

	// recover low
	pose = mc.low_pose();

	// attempt drop in loop closure state
//	ccd_loop_closure_obeying_nonideality( pose, closure_info );
//	mc.boltzmann( pose );

	// recover low again
//	pose = mc.low_pose();

	// store linear score
	if ( do_ccd ) {
		closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
	} else {
		closure_info.set_chainbreak_score( 0.0 ); // chainbreak nonexistent since true begin/end of chain
	}

	// rama score for moveable residues
	score_rama_for_moveable( pose, closure_info );

	// restore any prior status
	pose.set_fullatom_flag( prior_fullatom_flag, false ); // boolean: fullatom, repack
}


/// @brief (centroid level) build loops simultaneously
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
build_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles,
	bool const & use_fragment_insertion,
	Real const & chainbreak_criterion,
	Real const & rama_moveable_criterion,
	String const & ss_string,
	String const & aa_string,
	Integer const & nfrag,
	bool const & use_variable_frag,
	bool const & repick_frag,
	bool const & use_boost
)
{
	using namespace pose_ns;

	// setup weights for centroid level remodeling
	// TODO: wrap weights for ease of use
	Score_weight_map weight_map;
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( RG, 1.0 );

	// additional scoring setup
	score_set_cst_mode(3);

	// cutoffs for fragment picking and operations
	Integer cutoff_full = use_variable_frag ? 3 : 8; // 9-mers by default, originally cutoff == 16
	Integer cutoff3 = 2; // 6
	Integer cutoff1 = 0; // 3

	// offsets for fragment insertion
	Integer const frag_offset = 0;

	if ( use_fragment_insertion && repick_frag ) {
		// update max res manually, as this may not necessarily be done beforehand
		pose_update_MAX_RES( pose );

		// reset global fragment arrays
		fragments::reset_fragment_arrays_used_by_Vall();
	}

	// secondary structure string for fragment insertion
	std::ostringstream ss_ss;
	if ( ss_string == "" ) {
		for ( Integer r = 1, re = pose.total_residue(); r <= re; ++ r ) {
			ss_ss << pose.secstruct( r );
		}
	} else {
		ss_ss << ss_string;
	}


	if ( repick_frag ) {
		bool fragments_were_picked = false; // see if fragments were actually selected

		// number of fragments to use
		Integer const n_fragments = nfrag;

		// strings indicating desired fragments
		String sequence_str = aa_string == "" ? String( pose.total_residue(), '.' ) : aa_string; // all "." if not used
		String secondary_structure_str( ss_ss.str() );
		String bigbin_str( pose.total_residue(), '.' ); // all "." so not used

		// weights for fragment picking
		Real sequence_weight = 1.0;
		Real secondary_structure_weight = 1.0;
		Real bigbin_weight = 0.0; // not used

		// now we setup 9-mers and 3-mers, unfortunately due to the way the fragment picking routine is setup,
		// we are forced to cycle through the loops each time, first setting up 9-mers and then setting up 3-mers
		// it's not possible to do this in a single 'for' loop

		// setup 9-mers for each loop
		for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
			LoopClosureInfo const & closure_info = *i;

			Integer loop_length = closure_info.moveable_residues_width(); // cache

			// actually use fragments for this loop?  check length
			// no closure check here -- we need to pick fragments even for
			// closed loops in case the loop is broken again
			bool const use_9mers = use_fragment_insertion && ( loop_length > cutoff_full );

			fragments_were_picked = fragments_were_picked || use_9mers;

			if ( use_9mers ) { // pick 9-mers
				Integer const fullmer_size = use_variable_frag ? loop_length : 9;
				get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
								sequence_weight, secondary_structure_weight, bigbin_weight,
								closure_info.loop_range().begin(), closure_info.loop_range().end(),
								fullmer_size, n_fragments,
								true, true, true, 3 ); // booleans: exclude gly, pro, cis-peptide
			} // if use_fragments

		} // foreach loop (9-mer setup)

		// setup 3-mers for each loop
		for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
			LoopClosureInfo const & closure_info = *i;

			Integer loop_length = closure_info.moveable_residues_width(); // cache

			// actually use fragments for this loop?  check length
			// no closure check here -- we need to pick fragments even for
			// closed loops in case the loop is broken again
			bool const use_3mers = use_fragment_insertion &&
									   ( ( loop_length > cutoff3 ) ||
										 ( loop_length > cutoff1 ) );

			fragments_were_picked = fragments_were_picked || use_3mers;

			if ( use_3mers ) { // pick 3-mers
				get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
								sequence_weight, secondary_structure_weight, bigbin_weight,
								closure_info.loop_range().begin(), closure_info.loop_range().end(),
								3, n_fragments,
								true, true, true, 2 ); // booleans: exclude gly, pro, cis-peptide
			} // if use_fragments

		} // foreach loop (3-mer setup)

		// if applicable, build 1-mers and indicate usage of top fragments
		if ( fragments_were_picked ) {
			build_1mer_from_3mer( pose.total_residue() );
			choose_frag_set_top_N_frags( n_fragments ); // use top 'n' fragments for each size
		}
	}

	// force centroid level
	bool const prior_fullatom_flag = pose.fullatom();
	pose.set_fullatom_flag( false, false ); // boolean: fullatom, repack

	// perturb initial loop conformations for those that are not closed
//	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
//		LoopClosureInfo const & closure_info = *i;
//		Integer const loop_length = closure_info.moveable_residues_width(); // cache
//
//		// actually use fragments for this loop?  check length and closure status
//		bool const use_fragments = use_fragment_insertion &&
//		                           !closure_info.is_closed() &&
//		                           ( ( loop_length > cutoff9 ) ||
//		                             ( loop_length > cutoff3 ) ||
//		                             ( loop_length > cutoff1 ) );
//		if ( use_fragments ) {
//			insert_random_frags( 3, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
//		}
//	}

	// outer monte carlo cycles
	Integer n_fragment_moves = outer_cycles;
	Integer cycles2 = n_fragment_moves;
	Real const cycle_ratio = std::max( 0.5f, get_looprlx_cycle_ratio() );
	Integer const max_boost_cycles = 100;
	Integer const max_cycles_after_boost = cycles2 + max_boost_cycles;

	// chainbreak weighting
	Real const final_weight = 5.0;
	Real const delta_weight = final_weight / cycles2;
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 ); // for scoring chainbreak of the current loop
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// set initial chainbreak weights
	weight_map.set_weight( CHAINBREAK, 0.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 0 );

	// turn on ALL cutpoint weights to score all loops simultaneously
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// setup monte carlo
	pose.score( weight_map );
	Monte_carlo mc( pose, weight_map, 2 );

	// now cycle through set of loops randomly
	for ( Integer c2 = 1, overlap; c2 <= cycles2; ++c2 ) { // 10 cycles of chainbreak ramping up to 'final_weight'

		// create random order of closures to attempt
		utility::vector1< LoopClosureInfo > randomized_closures_to_attempt( closures_to_attempt.begin(), closures_to_attempt.end() );
		std::random_shuffle( randomized_closures_to_attempt.begin(), randomized_closures_to_attempt.end() );

		// build each loop
		for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
			LoopClosureInfo const & closure_info = *c;
			Integer loop_length = closure_info.moveable_residues_width(); // cache

			// sanity check for ccd
			bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
			utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
			if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
			     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
				do_ccd = false;
			}

			// store linear score
			if ( do_ccd ) {
				closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
			} else {
				closure_info.set_chainbreak_score( 0.0 ); // chainbreak nonexistent since true begin/end of chain
			}

			// store rama score for moveable residues
			score_rama_for_moveable( pose, closure_info );

			// use scores to decide on closure status
			closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
					                    closure_info.ramas_are_not_above( rama_moveable_criterion ) );

			// skip this loop if it's closed
			if ( closure_info.is_closed() ) {
				continue;
			}

			// actually use fragments for this loop?  check length and closure status
			bool const use_fragments = use_fragment_insertion &&
			                           !closure_info.is_closed() &&
			                           ( ( loop_length > cutoff_full ) ||
			                             ( loop_length > cutoff3 ) ||
			                             ( loop_length > cutoff1 ) );

			// inner cycles
			Integer const cycles3 = std::max( 50, (int)( 10 * loop_length * cycle_ratio ) );

			// statistics
			Integer frag_accepts = 0, frag_trials = 0, ccd_accepts = 0, ccd_trials = 0;

			// initialize
			overlap = 0;
			pose = mc.low_pose(); // recover low Pose

			// set chainbreak weights
			if ( closure_info.loop_range().begin() != 1 && closure_info.loop_range().end() != pose.total_residue() ) {
				weight_map.set_weight( CHAINBREAK, c2 * delta_weight );
				if ( c2 > cycles2 / 2 ) {
					overlap = 1;
				}
				weight_map.set_weight( CHAINBREAK_OVERLAP, overlap );
			}

			// mc reset using new chainbreak weights
			pose.score( weight_map );
			mc.set_weight_map( weight_map );
			mc.reset( pose );

			for ( Integer c3 = 1, size; c3 <= cycles3; ++c3 ) { // 30 cycles of fragment insertion + ccd move

				if ( use_fragments && ( !do_ccd || overlap == 0 || ran3() * cycles2 > 2 ) ) { // fragment insertion

					// 9-mer or full-mer
					if ( loop_length > cutoff_full ) {
						size = use_variable_frag ? loop_length : 9;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

					// 3-mer
					if ( loop_length > cutoff3 ) {
						size = 3;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

					// 1-mer
					if ( loop_length > cutoff1 ) {
						size = 1;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

				} else { // ccd move

					pose_to_misc( pose );
					ccd_moves_obeying_nonideality( 10, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );
//					per_torsion_ccd_moves_obeying_nonideality( 20, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );

					// attempt acceptance
					if ( mc.boltzmann( pose ) ) {
						++ccd_accepts;
					}
					++ccd_trials;

				} // ccd move

			} // foreach inner cycle (c3)

			// recover low
			pose = mc.low_pose();

			// attempt drop in loop closure state
//			ccd_loop_closure_obeying_nonideality( pose, closure_info, 100, true );
//			per_torsion_ccd_moves_obeying_nonideality( 100, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );
//			mc.boltzmann( pose );

			// recover low again
//			pose = mc.low_pose();

			// closure status
//			std::cout << "** " << closure_info.to_string() << std::endl;

			// trial status
//			std::cout << "cycle2: " << c2;
//			std::cout << " | frag: " << frag_accepts << " / " << frag_trials << " = " << (float)frag_accepts / (float)( frag_trials == 0 ? 1 : frag_trials );
//			std::cout << " | ccd: " << ccd_accepts << " / " << ccd_trials << " = " << (float)ccd_accepts / (float)( ccd_trials == 0 ? 1 : ccd_trials );
//			std::cout << std::endl;

		} // foreach loop


		// boost build cycle if chainbreak within range
		if ( use_boost && c2 == cycles2 ) {

			bool within_tolerance = false;
			for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
				LoopClosureInfo const & closure_info = *c;

				within_tolerance = within_tolerance || ( chainbreak_criterion < closure_info.chainbreak_score() && closure_info.chainbreak_score() < 10.0 );
			}

			if ( within_tolerance ) {
				if ( cycles2 <= max_cycles_after_boost ) {
					cycles2 += 5;
				} else {
					std::cout << "boost cycle exceeds max of " << max_boost_cycles << " additional cycles, terminating" << std::endl;
					break;
				}
			}

		} // if ( c2 == cycles2 )

	} // foreach outer cycle (cycle2)

	// need to rescore all loops and store in LoopClosureInfo objects that were passed in
	// (since objects used above are temporary)
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
		if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
		     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		} else {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );

		// use scores to decide on closure status
		closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
				                    closure_info.ramas_are_not_above( rama_moveable_criterion ) );
	}

	// restore any prior status
	pose.set_fullatom_flag( prior_fullatom_flag, false ); // boolean: fullatom, repack
}


/// @brief (centroid level) build loops simultaneously (arm cycler version)
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
build_loops_simul_arm(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	std::set< Integer > const & arm_residues,
	Integer const & outer_cycles,
	bool const & use_fragment_insertion,
	Real const & chainbreak_criterion,
	Real const & rama_moveable_criterion,
	String const & ss_string,
	String const & aa_string,
	Integer const & nfrag,
	bool const & use_variable_frag,
	bool const & use_boost
)
{
	using namespace pose_ns;

	// setup weights for centroid level remodeling
	// TODO: wrap weights for ease of use
	Score_weight_map weight_map;
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( RG, 1.0 );
	weight_map.set_weight( HB_SRBB, 0.1 );
	weight_map.set_weight( HB_LRBB, 0.1 );

	// additional scoring setup
	score_set_cst_mode(3);

	// cutoffs for fragment picking and operations
	Integer cutoff_full = use_variable_frag ? 3 : 8; // 9-mers by default, originally cutoff == 16
	Integer cutoff3 = 2; // 6
	Integer cutoff1 = 0; // 3

	// offsets for fragment insertion
	Integer const frag_offset = 0;

	if ( use_fragment_insertion ) {
		// update max res manually, as this may not necessarily be done beforehand
		pose_update_MAX_RES( pose );

		// reset global fragment arrays
		fragments::reset_fragment_arrays_used_by_Vall();
	}

	// secondary structure string for fragment insertion
	std::ostringstream ss_ss;
	if ( ss_string == "" ) {
		for ( Integer r = 1, re = pose.total_residue(); r <= re; ++ r ) {
			ss_ss << pose.secstruct( r );
		}
	} else {
		ss_ss << ss_string;
	}

	bool fragments_were_picked = false; // see if fragments were actually selected

	// number of fragments to use
	Integer const n_fragments = nfrag;

	// strings indicating desired fragments
	String sequence_str = aa_string == "" ? String( pose.total_residue(), '.' ) : aa_string; // all "." if not used
	String secondary_structure_str( ss_ss.str() );
	String bigbin_str( pose.total_residue(), '.' ); // all "." so not used

	// weights for fragment picking
	Real sequence_weight = 1.0;
	Real secondary_structure_weight = 1.0;
	Real bigbin_weight = 0.0; // not used

	// now we setup 9-mers and 3-mers, unfortunately due to the way the fragment picking routine is setup,
	// we are forced to cycle through the loops each time, first setting up 9-mers and then setting up 3-mers
	// it's not possible to do this in a single 'for' loop

	// setup 9-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		// actually use fragments for this loop?  check length
		// no closure check here -- we need to pick fragments even for
		// closed loops in case the loop is broken again
		bool const use_9mers = use_fragment_insertion && ( loop_length > cutoff_full );

		fragments_were_picked = fragments_were_picked || use_9mers;

		if ( use_9mers ) { // pick 9-mers
			Integer const fullmer_size = use_variable_frag ? loop_length : 9;
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                fullmer_size, n_fragments,
			                true, true, true, 3 ); // booleans: exclude gly, pro, cis-peptide
		} // if use_fragments

	} // foreach loop (9-mer setup)

	// setup 3-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		// actually use fragments for this loop?  check length
		// no closure check here -- we need to pick fragments even for
		// closed loops in case the loop is broken again
		bool const use_3mers = use_fragment_insertion &&
		                           ( ( loop_length > cutoff3 ) ||
		                             ( loop_length > cutoff1 ) );

		fragments_were_picked = fragments_were_picked || use_3mers;

		if ( use_3mers ) { // pick 3-mers
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                3, n_fragments,
			                true, true, true, 2 ); // booleans: exclude gly, pro, cis-peptide
		} // if use_fragments

	} // foreach loop (3-mer setup)

	// if applicable, build 1-mers and indicate usage of top fragments
	if ( fragments_were_picked ) {
		build_1mer_from_3mer( pose.total_residue() );
		choose_frag_set_top_N_frags( n_fragments ); // use top 'n' fragments for each size
	}

	// force centroid level
	bool const prior_fullatom_flag = pose.fullatom();
	pose.set_fullatom_flag( false, false ); // boolean: fullatom, repack

	// perturb initial loop conformations for those that are not closed
//	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
//		LoopClosureInfo const & closure_info = *i;
//		Integer const loop_length = closure_info.moveable_residues_width(); // cache
//
//		// actually use fragments for this loop?  check length and closure status
//		bool const use_fragments = use_fragment_insertion &&
//		                           !closure_info.is_closed() &&
//		                           ( ( loop_length > cutoff9 ) ||
//		                             ( loop_length > cutoff3 ) ||
//		                             ( loop_length > cutoff1 ) );
//		if ( use_fragments ) {
//			insert_random_frags( 3, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
//		}
//	}

	// outer monte carlo cycles
	Integer n_fragment_moves = outer_cycles;
	Integer cycles2 = n_fragment_moves;
	Real const cycle_ratio = std::max( 0.5f, get_looprlx_cycle_ratio() );
	Integer const max_boost_cycles = 100;
	Integer const max_cycles_after_boost = cycles2 + max_boost_cycles;

	// chainbreak weighting
	Real const final_weight = 5.0;
	Real const delta_weight = final_weight / cycles2;
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 ); // for scoring chainbreak of the current loop
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// set initial chainbreak weights
	weight_map.set_weight( CHAINBREAK, 0.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 0 );

	// turn on ALL cutpoint weights to score all loops simultaneously
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// setup monte carlo
	pose.score( weight_map );
	Monte_carlo mc( pose, weight_map, 2 );

	utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );

	// track which loops are within chain and not at chain extremes
	std::map< ResidueRange, bool > loop_within_chain;
	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt.begin(), ce = closures_to_attempt.end(); c != ce; ++c ) {
		bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
		if ( ( ( peptide_degree[ c->loop_range().begin() ] == 1 )  && ( c->cut() == ( c->loop_range().begin() - 1 ) ) ) ||
		     ( ( peptide_degree[ c->loop_range().end() ] == 1 )  && ( c->cut() == ( c->loop_range().end() ) ) ) ) {
			do_ccd = false;
		}

		loop_within_chain[ c->loop_range() ] = do_ccd;
	}

	// figure out the number of inner cycles
	Integer cycles3 = 0;
	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt.begin(), ce = closures_to_attempt.end(); c != ce; ++c ) {
		cycles3 += std::max( 50, (int)( 10 * c->moveable_residues_width() * cycle_ratio ) );
	}

	// statistics
	Integer frag_accepts = 0, frag_trials = 0, ccd_accepts = 0, ccd_trials = 0;

	// now cycle through set of loops randomly
	for ( Integer c2 = 1, overlap; c2 <= cycles2; ++c2 ) { // 10 cycles of chainbreak ramping up to 'final_weight'

		// initialize
		pose = mc.low_pose(); // recover low Pose

		// set chainbreak weights
		overlap = c2 > cycles2 / 2 ? 1 : 0;
		weight_map.set_weight( CHAINBREAK_OVERLAP, overlap );
		for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt.begin(), ce = closures_to_attempt.end(); c != ce; ++c ) {
			if ( c->loop_range().begin() != 1 && c->loop_range().end() != pose.total_residue() ) {
				weight_map.set_weight( CHAINBREAK, c2 * delta_weight );
			}
		}

		// mc reset using new chainbreak weights
		pose.score( weight_map );
		mc.set_weight_map( weight_map );
		mc.reset( pose );

		// turn arm residues off if past cycles threshold (also prevents
		// ccd incompatibility w/ respect to arm)
		if ( c2 > cycles2 / 2 ) {
			for ( std::set< Integer >::const_iterator r = arm_residues.begin(), re = arm_residues.end(); r != re; ++r ) {
				pose.set_allow_bb_move( *r, false );
			}
		}

		for ( Integer c3 = 1, size; c3 <= cycles3; ++c3 ) {

			// create random order of closures to attempt
			utility::vector1< LoopClosureInfo > randomized_closures_to_attempt( closures_to_attempt.begin(), closures_to_attempt.end() );
			std::random_shuffle( randomized_closures_to_attempt.begin(), randomized_closures_to_attempt.end() );

			// build each loop
			for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
				LoopClosureInfo const & closure_info = *c;
				Integer loop_length = closure_info.moveable_residues_width(); // cache

				// sanity check for ccd
				bool do_ccd = loop_within_chain.find( closure_info.loop_range() )->second; // if false, this indicates that we're at the true begin/end of a chain

				// store linear score
				if ( do_ccd ) {
					closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
				} else {
					closure_info.set_chainbreak_score( 0.0 ); // chainbreak nonexistent since true begin/end of chain
				}

				// store rama score for moveable residues
				score_rama_for_moveable( pose, closure_info );

				// use scores to decide on closure status
				closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
						                    closure_info.ramas_are_not_above( rama_moveable_criterion ) );

				// skip this loop if it's closed
				if ( closure_info.is_closed() ) {
					continue;
				}

				// actually use fragments for this loop?  check length and closure status
				bool const use_fragments = use_fragment_insertion &&
				                           !closure_info.is_closed() &&
				                           ( ( loop_length > cutoff_full ) ||
				                             ( loop_length > cutoff3 ) ||
				                             ( loop_length > cutoff1 ) );

				if ( use_fragments && ( !do_ccd || overlap == 0 || ran3() * cycles2 > 2 ) ) { // fragment insertion

					// 9-mer or full-mer
					if ( loop_length > cutoff_full ) {
						size = use_variable_frag ? loop_length : 9;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

					// 3-mer
					if ( loop_length > cutoff3 ) {
						size = 3;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

					// 1-mer
					if ( loop_length > cutoff1 ) {
						size = 1;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

				} else { // ccd move

					pose_to_misc( pose );
					ccd_moves_obeying_nonideality( loop_length, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );
//					per_torsion_ccd_moves_obeying_nonideality( 20, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );

					// attempt acceptance
					if ( mc.boltzmann( pose ) ) {
						++ccd_accepts;
					}
					++ccd_trials;

				} // ccd move

			}

		}

		// recover low
		pose = mc.low_pose();

			// closure status
//			std::cout << "** " << closure_info.to_string() << std::endl;

			// trial status
//			std::cout << "cycle2: " << c2;
//			std::cout << " | frag: " << frag_accepts << " / " << frag_trials << " = " << (float)frag_accepts / (float)( frag_trials == 0 ? 1 : frag_trials );
//			std::cout << " | ccd: " << ccd_accepts << " / " << ccd_trials << " = " << (float)ccd_accepts / (float)( ccd_trials == 0 ? 1 : ccd_trials );
//			std::cout << std::endl;


		// boost build cycle if chainbreak within range
//		if ( use_boost && c2 == cycles2 ) {
//
//			bool within_tolerance = false;
//			for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
//				LoopClosureInfo const & closure_info = *c;
//
//				within_tolerance = within_tolerance || ( chainbreak_criterion < closure_info.chainbreak_score() && closure_info.chainbreak_score() < 10.0 );
//			}
//
//			if ( within_tolerance ) {
//				if ( cycles2 <= max_cycles_after_boost ) {
//					cycles2 += 5;
//				} else {
//					std::cout << "boost cycle exceeds max of " << max_boost_cycles << " additional cycles, terminating" << std::endl;
//					break;
//				}
//			}
//
//		} // if ( c2 == cycles2 )

	} // foreach outer cycle (cycle2)

	// need to rescore all loops and store in LoopClosureInfo objects that were passed in
	// (since objects used above are temporary)
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		if ( loop_within_chain.find( closure_info.loop_range() )->second ) {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		} else {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );

		// use scores to decide on closure status
		closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
				                    closure_info.ramas_are_not_above( rama_moveable_criterion ) );
	}

	// restore any prior status
	pose.set_fullatom_flag( prior_fullatom_flag, false ); // boolean: fullatom, repack
}


/// @brief (centroid level) adaptive build loops simultaneously
/// @brief by using fragments + 2-torsion ccd + 1-torsion ccd
/// @brief with percentage of time spent between the three governed by
/// @brief chainbreak score
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
adaptive_build_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles,
	bool const & use_fragment_insertion,
	Real const & chainbreak_criterion,
	Real const & rama_moveable_criterion,
	String const & ss_string,
	String const & aa_string,
	Integer const & nfrag,
	bool const & use_variable_frag
)
{
	using namespace pose_ns;

	// linear chainbreak thresholds for establishing closure schedule
	Real threshold_two_torsion_start = 20.0;
	Real threshold_two_torsion_stop = 5.0;
	Real threshold_one_torsion_start = 10.0;

	// levels of closure moves
	// procedure needs to be designed so that at any point the sum of probabilities should never exceed 1
	// in the current scheme, fragment_background + two_torsion_max + one_torsion_max = 1.0
	Real fragment_insertion_background_probability = 0.5;
	Real two_torsion_min_probability = 0.0;
	Real two_torsion_max_probability = 0.5;
	Real one_torsion_min_probability = 0.0;
	Real one_torsion_max_probability = 0.5;

	// slope/intercept governing closure moves
//	Real two_torsion_increase_slope = ( two_torsion_max_probability - two_torsion_min_probability ) / ( threshold_one_torsion_start - threshold_two_torsion_start );
//	Real two_torsion_increase_intercept = -two_torsion_increase_slope * threshold_two_torsion_start + two_torsion_min_probability;
	Real two_torsion_falloff_slope = ( two_torsion_min_probability - two_torsion_max_probability ) / ( threshold_two_torsion_stop - threshold_one_torsion_start );
	Real two_torsion_falloff_intercept = -two_torsion_falloff_slope * threshold_one_torsion_start + two_torsion_max_probability;
	Real one_torsion_increase_slope = ( one_torsion_max_probability - one_torsion_min_probability ) / ( threshold_two_torsion_stop - threshold_one_torsion_start );
	Real one_torsion_increase_intercept = -one_torsion_increase_slope * threshold_one_torsion_start + one_torsion_min_probability;

	// setup weights for centroid level remodeling
	// TODO: wrap weights for ease of use
	Score_weight_map weight_map;
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( RG, 1.0 );

	// additional scoring setup
	score_set_cst_mode(3);

	// cutoffs for fragment picking and operations
	Integer cutoff_full = use_variable_frag ? 3 : 8; // 16
	Integer cutoff3 = 2; // 6
	Integer cutoff1 = 0; // 3

	// offsets for fragment insertion
	Integer const frag_offset = 0;

	if ( use_fragment_insertion ) {
		// update max res manually, as this may not necessarily be done beforehand
		pose_update_MAX_RES( pose );

		// reset global fragment arrays
		fragments::reset_fragment_arrays_used_by_Vall();
	}

	// secondary structure string for fragment insertion
	std::ostringstream ss_ss;
	if ( ss_string == "" ) {
		for ( Integer r = 1, re = pose.total_residue(); r <= re; ++ r ) {
			ss_ss << pose.secstruct( r );
		}
	} else {
		ss_ss << ss_string;
	}

	bool fragments_were_picked = false; // see if fragments were actually selected

	// number of fragments to use
	Integer const n_fragments = nfrag;

	// strings indicating desired fragments
	String sequence_str = aa_string == "" ? String( pose.total_residue(), '.' ) : aa_string; // all "." if not used
	String secondary_structure_str( ss_ss.str() );
	String bigbin_str( pose.total_residue(), '.' ); // all "." so not used

	// weights for fragment picking
	Real sequence_weight = 1.0;
	Real secondary_structure_weight = 1.0;
	Real bigbin_weight = 0.0; // not used

	// now we setup 9-mers and 3-mers, unfortunately due to the way the fragment picking routine is setup,
	// we are forced to cycle through the loops each time, first setting up 9-mers and then setting up 3-mers
	// it's not possible to do this in a single 'for' loop

	// setup 9-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		// actually use fragments for this loop?  check length
		// no closure check here -- we need to pick fragments even for
		// closed loops in case the loop is broken again
		bool const use_9mers = use_fragment_insertion && ( loop_length > cutoff_full );

		fragments_were_picked = fragments_were_picked || use_9mers;

		if ( use_9mers ) { // pick 9-mers
			Integer const fullmer_size = use_variable_frag ? loop_length : 9;
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                fullmer_size, n_fragments,
			                true, true, true, 3 ); // booleans: exclude gly, pro, cis-peptide
		} // if use_fragments

	} // foreach loop (9-mer setup)

	// setup 3-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		// actually use fragments for this loop?  check length
		// no closure check here -- we need to pick fragments even for
		// closed loops in case the loop is broken again
		bool const use_3mers = use_fragment_insertion &&
		                           ( ( loop_length > cutoff3 ) ||
		                             ( loop_length > cutoff1 ) );

		fragments_were_picked = fragments_were_picked || use_3mers;

		if ( use_3mers ) { // pick 3-mers
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                3, n_fragments,
			                true, true, true, 2 ); // booleans: exclude gly, pro, cis-peptide
		} // if use_fragments

	} // foreach loop (3-mer setup)

	// if applicable, build 1-mers and indicate usage of top fragments
	if ( fragments_were_picked ) {
		build_1mer_from_3mer( pose.total_residue() );
		choose_frag_set_top_N_frags( n_fragments ); // use top 'n' fragments for each size
	}

	// force centroid level
	bool const prior_fullatom_flag = pose.fullatom();
	pose.set_fullatom_flag( false, false ); // boolean: fullatom, repack

	// perturb initial loop conformations for those that are not closed
//	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
//		LoopClosureInfo const & closure_info = *i;
//		Integer const loop_length = closure_info.moveable_residues_width(); // cache
//
//		// actually use fragments for this loop?  check length and closure status
//		bool const use_fragments = use_fragment_insertion &&
//		                           !closure_info.is_closed() &&
//		                           ( ( loop_length > cutoff9 ) ||
//		                             ( loop_length > cutoff3 ) ||
//		                             ( loop_length > cutoff1 ) );
//		if ( use_fragments ) {
//			insert_random_frags( 3, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
//		}
//	}

	// outer monte carlo cycles
	Integer n_fragment_moves = outer_cycles;
	Integer cycles2 = n_fragment_moves;
	Real const cycle_ratio = std::max( 0.5f, get_looprlx_cycle_ratio() );

	// chainbreak weighting
	Real const final_weight = 5.0;
	Real const delta_weight = final_weight / cycles2;
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 ); // for scoring chainbreak of the current loop
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// set initial chainbreak weights
	weight_map.set_weight( CHAINBREAK, 0.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1 );

	// turn on ALL cutpoint weights to score all loops simultaneously
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// monte carlo
	pose.score( weight_map );
	Monte_carlo mc( pose, weight_map, 2.0 );

	// now cycle through set of loops randomly
	for ( Integer c2 = 1; c2 <= cycles2; ++c2 ) { // 10 cycles of chainbreak ramping up to 'final_weight'

		// create random order of closures to attempt
		utility::vector1< LoopClosureInfo > randomized_closures_to_attempt( closures_to_attempt.begin(), closures_to_attempt.end() );
		std::random_shuffle( randomized_closures_to_attempt.begin(), randomized_closures_to_attempt.end() );

		// build each loop
		for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
			LoopClosureInfo const & closure_info = *c;
			Integer loop_length = closure_info.moveable_residues_width(); // cache

			// sanity check for ccd
			bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
			utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
			if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
			     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
				do_ccd = false;
			}

			// store linear score
			if ( do_ccd ) {
				closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
			} else {
				closure_info.set_chainbreak_score( 0.0 ); // chainbreak nonexistent since true begin/end of chain
			}

			// store rama score for moveable residues
			score_rama_for_moveable( pose, closure_info );

			// use scores to decide on closure status
			closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
					                    closure_info.ramas_are_not_above( rama_moveable_criterion ) );

			// skip this loop if it's closed
			if ( closure_info.is_closed() ) {
				continue;
			}

			// actually use fragments for this loop?  check length and closure status
			bool const use_fragments = use_fragment_insertion &&
			                           !closure_info.is_closed() &&
			                           ( ( loop_length > cutoff_full ) ||
			                             ( loop_length > cutoff3 ) ||
			                             ( loop_length > cutoff1 ) );

			// inner cycles
			Integer const cycles3 = std::max( 50, (int)( 10 * loop_length * cycle_ratio ) );

			// statistics
			Integer frag_accepts = 0, frag_trials = 0;
			Integer two_ccd_accepts = 0, two_ccd_trials = 0;
			Integer one_ccd_accepts = 0, one_ccd_trials = 0;

			// initialize
			pose = mc.low_pose(); // recover low Pose

			// set chainbreak weights
			if ( closure_info.loop_range().begin() != 1 && closure_info.loop_range().end() != pose.total_residue() ) {
				weight_map.set_weight( CHAINBREAK, c2 * delta_weight );
			}

			// mc reset using new chainbreak weights
			pose.score( weight_map );
			mc.set_weight_map( weight_map );
			mc.reset( pose );

			// temporaries for establishing closure schedule
			Real current_chainbreak;
			Real two_torsion_ccd_probability;
			Real one_torsion_ccd_probability;
			Real fragment_insertion_limit, two_torsion_ccd_limit, one_torsion_ccd_limit;

			for ( Integer c3 = 1, size; c3 <= cycles3; ++c3 ) { // 30 cycles of fragment insertion + ccd move

				// establish closure schedule
				current_chainbreak = score_cut_in_Pose_linear( pose, closure_info.cut(), 1 );
				if ( current_chainbreak <= threshold_two_torsion_stop ) {
					one_torsion_ccd_probability = current_chainbreak * one_torsion_increase_slope + one_torsion_increase_intercept;
					two_torsion_ccd_probability = 0.0;
				} else if ( current_chainbreak <= threshold_one_torsion_start && current_chainbreak > threshold_two_torsion_stop ) {
					one_torsion_ccd_probability = current_chainbreak * one_torsion_increase_slope + one_torsion_increase_intercept;
					two_torsion_ccd_probability = current_chainbreak * two_torsion_falloff_slope + two_torsion_falloff_intercept;
				} else if ( current_chainbreak <= threshold_two_torsion_start && current_chainbreak > threshold_one_torsion_start ) {
					one_torsion_ccd_probability = 0.0;
					two_torsion_ccd_probability = current_chainbreak * two_torsion_falloff_slope + two_torsion_falloff_intercept;
				} else {
					one_torsion_ccd_probability = 0.0;
					two_torsion_ccd_probability = 0.0;
				}

				// closure schedule ranges
				fragment_insertion_limit = std::max( fragment_insertion_background_probability, ONE - two_torsion_ccd_probability - one_torsion_ccd_probability );
				two_torsion_ccd_limit = fragment_insertion_limit + two_torsion_ccd_probability;
				one_torsion_ccd_limit = two_torsion_ccd_limit + one_torsion_ccd_probability;
				if ( !use_fragments ) { // then divide time between two and one torsion ccd where applicable
					fragment_insertion_limit = 0.0;
					two_torsion_ccd_limit = two_torsion_ccd_probability / ( two_torsion_ccd_probability + one_torsion_ccd_probability );
					one_torsion_ccd_limit = one_torsion_ccd_probability / ( two_torsion_ccd_probability + one_torsion_ccd_probability );
				}

				Real move_selection = ran3(); // type of move, to be based on closure schedule ranges

				if ( use_fragments && ( !do_ccd || move_selection < fragment_insertion_limit ) ) { // fragment insertion

					// 9-mer
					if ( loop_length > cutoff_full ) {
						size = use_variable_frag ? loop_length : 9;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

					// 3-mer
					if ( loop_length > cutoff3 ) {
						size = 3;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

					// 1-mer
					if ( loop_length > cutoff1 ) {
						size = 1;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

				} else if ( do_ccd && move_selection < two_torsion_ccd_limit ) { // 2-torsion ccd move

					pose_to_misc( pose );
					ccd_moves_obeying_nonideality( loop_length, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );

					// attempt acceptance
					if ( mc.boltzmann( pose ) ) {
						++two_ccd_accepts;
					}
					++two_ccd_trials;

				} else if ( do_ccd && move_selection < one_torsion_ccd_limit ) { // 1-torsion ccd move

					pose_to_misc( pose );
					per_torsion_ccd_moves_obeying_nonideality( loop_length, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );

					// attempt acceptance
					if ( mc.boltzmann( pose ) ) {
						++one_ccd_accepts;
					}
					++one_ccd_trials;

				}

			} // foreach inner cycle (c3)

			// recover low
			pose = mc.low_pose();

			// attempt drop in loop closure state
//			ccd_loop_closure_obeying_nonideality( pose, closure_info, 100, true );
//			per_torsion_ccd_moves_obeying_nonideality( 100, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );
//			mc.boltzmann( pose );

			// recover low again
//			pose = mc.low_pose();

			// closure status
//			std::cout << "** " << closure_info.to_string() << std::endl;

			// trial status
//			std::cout << "cycle2: " << c2;
//			std::cout << " | frag: " << frag_accepts << " / " << frag_trials << " = " << (float)frag_accepts / (float)( frag_trials == 0 ? 1 : frag_trials );
//			std::cout << " | two_ccd: " << two_ccd_accepts << " / " << two_ccd_trials << " = " << (float)two_ccd_accepts / (float)( two_ccd_trials == 0 ? 1 : two_ccd_trials );
//			std::cout << " | one_ccd: " << one_ccd_accepts << " / " << one_ccd_trials << " = " << (float)one_ccd_accepts / (float)( one_ccd_trials == 0 ? 1 : one_ccd_trials );
//			std::cout << std::endl;

		} // foreach loop

	} // foreach outer cycle (cycle2)

	// need to rescore all loops and store in LoopClosureInfo objects that were passed in
	// (since objects used above are temporary)
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
		if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
		     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		} else {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );

		// use scores to decide on closure status
		closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
				                    closure_info.ramas_are_not_above( rama_moveable_criterion ) );
	}

	// restore any prior status
	pose.set_fullatom_flag( prior_fullatom_flag, false ); // boolean: fullatom, repack
}



/// @brief (centroid level) screen build loops simultaneously by
/// @brief fragment + min (without vdw) and mc (with vdw)
/// @details allow_bb_move is governed *inside* this procedure, so recommend
/// @details setting all allow_bb_move to false beforehand, as the state will not be kept.
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
/// @warning This procedure ALLOWS secondary structure movement during minimization!
void
screen_build_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles,
	Real const & chainbreak_criterion,
	Real const & rama_moveable_criterion,
	Real const & semi_closed_threshold,
	String const & ss_string,
	String const & aa_string,
	Integer const & nfrag
)
{
	using namespace pose_ns;

	// setup weights for closure and eval stages
	Score_weight_map weight_map;
	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( RG, 1.0 );

	// setup weights for minimization stage
	Score_weight_map min_weight_map;
	min_weight_map.set_weight( RAMACHANDRAN, 0.1 );
	min_weight_map.set_weight( CHAINBREAK, 1.0 );
	min_weight_map.set_weight( CHAINBREAK_OVERLAP, 1 );

	// additional scoring setup
	score_set_cst_mode(3);

	// cutoffs for fragment picking and operations
	Integer cutoff9 = 8; // 16
	Integer cutoff3 = 2; // 6
	Integer cutoff1 = 0; // 3

	// offsets for fragment insertion
	Integer const frag_offset = 0;

	// update max res manually, as this may not necessarily be done beforehand
	pose_update_MAX_RES( pose );

	// reset global fragment arrays
	fragments::reset_fragment_arrays_used_by_Vall();

	// secondary structure string for fragment insertion
	std::ostringstream ss_ss;
	if ( ss_string == "" ) {
		for ( Integer r = 1, re = pose.total_residue(); r <= re; ++ r ) {
			ss_ss << pose.secstruct( r );
		}
	} else {
		ss_ss << ss_string;
	}

	bool fragments_were_picked = false; // see if fragments were actually selected

	// number of fragments to use
	Integer const n_fragments = nfrag;

	// strings indicating desired fragments
	String sequence_str = aa_string == "" ? String( pose.total_residue(), '.' ) : aa_string; // all "." if not used
	String secondary_structure_str( ss_ss.str() );
	String bigbin_str( pose.total_residue(), '.' ); // all "." so not used

	// weights for fragment picking
	Real sequence_weight = 1.0;
	Real secondary_structure_weight = 1.0;
	Real bigbin_weight = 0.0; // not used

	// now we setup 9-mers and 3-mers, unfortunately due to the way the fragment picking routine is setup,
	// we are forced to cycle through the loops each time, first setting up 9-mers and then setting up 3-mers
	// it's not possible to do this in a single 'for' loop

	// setup 9-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		// actually use fragments for this loop?  check length
		// no closure check here -- we need to pick fragments even for
		// closed loops in case the loop is broken again
		bool const use_9mers = loop_length > cutoff9;

		fragments_were_picked = fragments_were_picked || use_9mers;

		if ( use_9mers ) { // pick 9-mers
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                9, n_fragments,
			                true, true, true, 3 ); // booleans: exclude gly, pro, cis-peptide
		} // if use_fragments

	} // foreach loop (9-mer setup)

	// setup 3-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		// actually use fragments for this loop?  check length
		// no closure check here -- we need to pick fragments even for
		// closed loops in case the loop is broken again
		bool const use_3mers = ( loop_length > cutoff3 ) || ( loop_length > cutoff1 );

		fragments_were_picked = fragments_were_picked || use_3mers;

		if ( use_3mers ) { // pick 3-mers
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                3, n_fragments,
			                true, true, true, 2 ); // booleans: exclude gly, pro, cis-peptide
		} // if use_fragments

	} // foreach loop (3-mer setup)

	// if applicable, build 1-mers and indicate usage of top fragments
	if ( fragments_were_picked ) {
		build_1mer_from_3mer( pose.total_residue() );
		choose_frag_set_top_N_frags( n_fragments ); // use top 'n' fragments for each size
	} else {
		// problem, print warning and return
		std::cerr << "WARNING: build_loops_simul_with_fragments() : no fragments picked, procedure exiting" << std::endl;
		return;
	}

	// force centroid level
	bool const prior_fullatom_flag = pose.fullatom();
	pose.set_fullatom_flag( false, false ); // boolean: fullatom, repack

	// minimization parameters
	minimize_reset();
	minimize_exclude_sstype( false, false ); // allow secondary structure movements
	bool const prior_pose_loop_flag = get_pose_loop_flag();
	set_pose_loop_flag( true ); // need proper score compatible with derivative

	// outer monte carlo cycles
	Integer n_fragment_moves = outer_cycles;
	Integer cycles2 = n_fragment_moves;
	Real const cycle_ratio = std::max( 0.5f, get_looprlx_cycle_ratio() );

	// chainbreak weighting
	Real const final_weight = 5.0;
	Real const delta_weight = final_weight / cycles2;
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 ); // for scoring chainbreak of the current loop
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// set initial chainbreak weights
	weight_map.set_weight( CHAINBREAK, 0.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1 );

	// turn on ALL cutpoint weights to score all loops simultaneously during closure
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );
	min_weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// setup monte carlo
	pose.score( weight_map );
	Monte_carlo mc( pose, weight_map, 2 );

	// now cycle through set of loops randomly and do fragment insertion
	for ( Integer c2 = 1; c2 <= cycles2; ++c2 ) { // 10 cycles of chainbreak ramping up to 'final_weight'

		// create random order of closures to attempt
		utility::vector1< LoopClosureInfo > randomized_closures_to_attempt( closures_to_attempt.begin(), closures_to_attempt.end() );
		std::random_shuffle( randomized_closures_to_attempt.begin(), randomized_closures_to_attempt.end() );

		// build each loop
		for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
			LoopClosureInfo const & closure_info = *c;
			Integer loop_length = closure_info.moveable_residues_width(); // cache

			// initialize
			pose = mc.low_pose(); // recover low Pose

			// setup inital proper allow bb move, need to do this after recovering low since we
			// change the bb move settings during minimization
			setup_proper_allow_bb_move( pose, closures_to_attempt );

			// sanity check for ccd
			bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
			utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
			if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
			     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
				do_ccd = false;
			}

			// store linear score
			if ( do_ccd ) {
				closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
			} else {
				closure_info.set_chainbreak_score( 0.0 ); // chainbreak nonexistent since true begin/end of chain
			}

			// store rama score for moveable residues
			score_rama_for_moveable( pose, closure_info );

			// use scores to decide on closure status
			closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
					                    closure_info.ramas_are_not_above( rama_moveable_criterion ) );

			// skip this loop if it's closed
			if ( closure_info.is_closed() ) {
				continue;
			}

			// actually use fragments for this loop?  check length and closure status
			bool const use_fragments = !closure_info.is_closed() &&
			                           ( ( loop_length > cutoff9 ) ||
			                             ( loop_length > cutoff3 ) ||
			                             ( loop_length > cutoff1 ) );

			// inner cycles
			Integer const cycles3 = std::max( 50, (int)( 10 * loop_length * cycle_ratio ) );

			// statistics
			Integer frag_accepts = 0, frag_trials = 0, min_accepts = 0, min_trials = 0;

			// set chainbreak weights
			if ( closure_info.loop_range().begin() != 1 && closure_info.loop_range().end() != pose.total_residue() ) {
				weight_map.set_weight( CHAINBREAK, c2 * delta_weight );
			}

			// mc reset using new chainbreak weights
			pose.score( weight_map );
			mc.set_weight_map( weight_map );
			mc.reset( pose );

			for ( Integer c3 = 1, size; c3 <= cycles3; ++c3 ) { // 30 cycles of fragment insertion

				if ( use_fragments && ( c3 == 1 || score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) > semi_closed_threshold
				                        || ran3() < 0.7 ) ) { // the boolean expressions must be in this order otherwise fall through will screw things up!

					// 9-mer
					if ( loop_length > cutoff9 ) {
						size = 9;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

					// 3-mer
					if ( loop_length > cutoff3 ) {
						size = 3;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

					// 1-mer
					if ( loop_length > cutoff1 ) {
						size = 1;
						choose_offset_frag( size, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), frag_offset );
						if ( mc.boltzmann( pose ) ) {
							++frag_accepts;
						}
						++frag_trials;
					}

				} else if ( do_ccd && score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) > chainbreak_criterion ) {
					pose.set_allow_bb_move( false ); // lock residues flanking open chainbreaks
//					for ( std::set< LoopClosureInfo >::const_iterator a = closures_to_attempt.begin(), ae = closures_to_attempt.end(); a != ae; ++a ) {
//						if ( a->is_closed() ) {
//							a->setup_allow_bb_move( pose );
//						}
//					}
					closure_info.setup_allow_bb_move( pose ); // open residues for current chainbreak

					pose.main_minimize( min_weight_map, "linmin" );

					if ( mc.boltzmann( pose ) ) { // note that this boltzmann call uses the full eval (vdw+rg+rama) ) {
						++min_accepts;
					}
					++min_trials;

					// reset proper allow bb move
					setup_proper_allow_bb_move( pose, closures_to_attempt );
				}

			} // foreach inner cycle (c3)


			// closure status
//			std::cout << "** " << closure_info.to_string() << std::endl;

			// trial status
//			std::cout << "cycle2: " << c2;
//			std::cout << " | frag: " << frag_accepts << " / " << frag_trials << " = " << (float)frag_accepts / (float)( frag_trials == 0 ? 1 : frag_trials );
//			std::cout << " | min: " << min_accepts << " / " << min_trials << " = " << (float)min_accepts / (float)( min_trials == 0 ? 1 : min_trials );
//			std::cout << std::endl;
//			std::cout << "    lc = " << score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) << std::endl;
		} // foreach loop

	} // foreach outer cycle (cycle2)

	// need to rescore all loops and store in LoopClosureInfo objects that were passed in
	// (since objects used above are temporary)
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
		if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
		     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		} else {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );

		// use scores to decide on closure status
		closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
				                    closure_info.ramas_are_not_above( rama_moveable_criterion ) );
	}


	// restore any prior status
	pose.set_fullatom_flag( prior_fullatom_flag, false ); // boolean: fullatom, repack

	set_pose_loop_flag( prior_pose_loop_flag ); // reset
}



/// @brief (centroid level) build loops simultaneously using only fragments
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @param[in] ccd_type  0 = no ccd, 1 = 1-torsion ccd move, 2 = 2-torsion ccd move, 3 = ccd closure
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
build_loops_simul_with_fragments(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles,
	Real const & chainbreak_criterion,
	Real const & rama_moveable_criterion,
	Integer const & ccd_type,
	String const & ss_string,
	String const & aa_string,
	Integer const & nfrag
)
{
	using namespace fragments;
	using namespace pose_ns;

	// setup weights for centroid level remodeling
	// TODO: wrap weights for ease of use
	Score_weight_map weight_map;
	weight_map.set_weight( VDW, 1.0 );
//	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( RG, 1.0 );

	// additional scoring setup
	score_set_cst_mode(3);

	// offsets for fragment insertion
	Integer const frag_offset = 0;

	// update max res manually, as this may not necessarily be done beforehand
	pose_update_MAX_RES( pose );

	// reset global fragment arrays
	fragments::reset_fragment_arrays_used_by_Vall();

	// secondary structure string for fragment insertion
	std::ostringstream ss_ss;
	if ( ss_string == "" ) {
		for ( Integer r = 1, re = pose.total_residue(); r <= re; ++ r ) {
			ss_ss << pose.secstruct( r );
		}
	} else {
		ss_ss << ss_string;
	}

	bool fragments_were_picked = false; // see if fragments were actually selected

	// number of fragments to use
	Integer const n_fragments = nfrag;

	// strings indicating desired fragments
	String sequence_str = aa_string == "" ? String( pose.total_residue(), '.' ) : aa_string; // all "." if not used
	String secondary_structure_str( ss_ss.str() );
	String bigbin_str( pose.total_residue(), '.' ); // all "." so not used

	// weights for fragment picking
	Real sequence_weight = 1.0;
	Real secondary_structure_weight = 1.0;
	Real bigbin_weight = 0.0; // not used

	// setup full-mer for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		if ( loop_length > 3 ) {
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
							sequence_weight, secondary_structure_weight, bigbin_weight,
							closure_info.loop_range().begin(), closure_info.loop_range().end(),
							loop_length, n_fragments,
							true, true, true, 3 ); // booleans: exclude gly, pro, cis-peptide
		}
	}

	// setup 3-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
						sequence_weight, secondary_structure_weight, bigbin_weight,
						closure_info.loop_range().begin(), closure_info.loop_range().end(),
						3, n_fragments,
						true, true, true, 2 ); // booleans: exclude gly, pro, cis-peptide

	} // foreach loop (3-mer setup)

	// if applicable, build 1-mers and indicate usage of top fragments
	build_1mer_from_3mer( pose.total_residue() );
	choose_frag_set_top_N_frags( n_fragments ); // use top 'n' fragments for each size

	// force centroid level
	bool const prior_fullatom_flag = pose.fullatom();
	pose.set_fullatom_flag( false, false ); // boolean: fullatom, repack

	bool const prior_pose_loop_flag = get_pose_loop_flag();
	set_pose_loop_flag( true ); // need proper score compatible with derivative

	// outer monte carlo cycles
	std::vector< std::pair< Integer, LoopClosureInfo const * > > moveable_residues;
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		for ( std::set< Integer >::const_iterator j = closure_info.moveable_residues().begin(), je = closure_info.moveable_residues().end(); j != je; ++j ) {
			moveable_residues.push_back( std::make_pair< Integer, LoopClosureInfo const * >( *j, &closure_info ) );
		}
	}
	Integer switch_point = 180;

	// chainbreak weighting
	Real const final_weight = 5.0;
	Real const delta_weight = final_weight / 180.0;
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 ); // for scoring chainbreak of the current loop
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// set initial chainbreak weights
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1 );

	// turn on ALL cutpoint weights to score all loops simultaneously
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// setup monte carlo
	pose.score( weight_map );
	Monte_carlo mc( pose, weight_map, 2 );

	Integer frag_accepts = 0, frag_trials = 0;
	Integer ccd_accepts = 0, ccd_trials = 0;

	// create random order of closures to attempt
	for ( Integer i = 1; i <= 3*n_fragments; ++i ) {
		utility::vector1< LoopClosureInfo > randomized_closures_to_attempt( closures_to_attempt.begin(), closures_to_attempt.end() );
		std::random_shuffle( randomized_closures_to_attempt.begin(), randomized_closures_to_attempt.end() );

		for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
			LoopClosureInfo const & closure_info = *c;

			if ( !closure_info.is_closed() ) {
				Integer loop_length = closure_info.moveable_residues_width();
				Integer res = *closure_info.moveable_residues().begin();

				Integer const size_bin( get_index_by_frag_size( loop_length ) );
				Integer nn_num = static_cast< Integer >( ran3() * align_depth( res, size_bin ) ) + 1;

				for ( Integer r = res, re = res + loop_length; r < re; ++r ) {
					if ( pose.get_allow_bb_move( r ) ) {
						pose.set_phi( res, align_phi( r, nn_num, 0, size_bin ) );
						pose.set_psi( res, align_psi( r, nn_num, 0, size_bin ) );
						pose.set_omega( res, align_omega( r, nn_num, 0, size_bin ) );
						pose.set_secstruct( res, ss_type( r, nn_num, 0, size_bin ) );
					}
				}
				if ( mc.boltzmann( pose ) ) {
					++frag_accepts;
				}
				++frag_trials;
			}
		}
	}

	pose = mc.low_pose();

	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt.begin(), ce = closures_to_attempt.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure_info = *c;

		utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
		if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
			 ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		} else {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );

		// use scores to decide on closure status
		closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
									closure_info.ramas_are_not_above( rama_moveable_criterion ) );
	}

	// trial status
	std::cout << " | frag: " << frag_accepts << " / " << frag_trials << " = " << (float)frag_accepts / (float)( frag_trials == 0 ? 1 : frag_trials );
	std::cout << std::endl;

//	// insert 1-mers
//	for ( Integer i = 1; i <= 360; ++i ) {
//		for ( Integer j = 1, je = moveable_residues.size(); j <= je; ++j ) {
//			// select random residue
//			std::pair< Integer, LoopClosureInfo const * > const & r2l = moveable_residues[ static_cast< Size >( ran3() * moveable_residues.size() ) ];
//			if ( !r2l.second->is_closed() ) {
//				Integer const res = r2l.first;
//
//				// pick a random 1-mer and insert it
//				Integer const size_bin( get_index_by_frag_size( 1 ) );
//				Integer nn_num = static_cast< Integer >( ran3() * align_depth( res, size_bin ) ) + 1;
//				pose.set_phi( res, align_phi( res, nn_num, 0, size_bin ) );
//				pose.set_psi( res, align_psi( res, nn_num, 0, size_bin ) );
//				pose.set_omega( res, align_omega( res, nn_num, 0, size_bin ) );
//				pose.set_secstruct( res, ss_type( res, nn_num, 0, size_bin ) );
//				if ( mc.boltzmann( pose ) ) {
//					++frag_accepts;
//				}
//				++frag_trials;
//			}
//		}
//
//		if ( i > switch_point ) {
//			// create random order of closures to attempt
//			utility::vector1< LoopClosureInfo > randomized_closures_to_attempt( closures_to_attempt.begin(), closures_to_attempt.end() );
//			std::random_shuffle( randomized_closures_to_attempt.begin(), randomized_closures_to_attempt.end() );
//
//			// ccd
//			for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
//				LoopClosureInfo const & closure_info = *c;
//
//				Integer loop_length = closure_info.moveable_residues_width(); // cache
//
//				// sanity check for ccd
//				bool do_ccd = !closure_info.is_closed(); // if false, this indicates that we're at the true begin/end of a chain
//				utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
//				if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
//					 ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
//					do_ccd = false;
//				}
//
//				if ( ccd_type > 0 && do_ccd ) {
//					switch( ccd_type ) {
//						case 3: // ccd closure
//							ccd_loop_closure_obeying_nonideality( pose, closure_info, loop_length * 10 );
//							break;
//						case 2: // two-torsion ccd move
//							ccd_moves_obeying_nonideality( loop_length, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );
//							break;
//						case 1: // one-torsion ccd move
//							per_torsion_ccd_moves_obeying_nonideality( loop_length, pose, closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut() );
//							break;
//						default:
//							// should never get here
//							break;
//					}
//
//					if ( mc.boltzmann( pose ) ) {
//						++ccd_accepts;
//					}
//					++ccd_trials;
//				}
//
//				pose = mc.low_pose();
//			}
//
//			// trial status
//			std::cout << " | ccd: " << ccd_accepts << " / " << ccd_trials << " = " << (float)ccd_accepts / (float)( ccd_trials == 0 ? 1 : ccd_trials );
//			std::cout << std::endl;
//		} // switch_point
//
//		// recover low
////		pose = mc.low_pose();
//
//		// mc reset using new chainbreak weights
//		for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt.begin(), ce = closures_to_attempt.end(); c != ce; ++c ) {
//			// set chainbreak weights
//			if ( c->loop_range().begin() != 1 && c->loop_range().end() != pose.total_residue() ) {
//				weight_map.set_weight( CHAINBREAK, i * delta_weight );
//			}
//		}
//		pose.score( weight_map );
//		mc.set_weight_map( weight_map );
//		mc.reset( pose );
//
//		for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt.begin(), ce = closures_to_attempt.end(); c != ce; ++c ) {
//			LoopClosureInfo const & closure_info = *c;
//
//			utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
//			if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
//				 ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
//				closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
//			} else {
//				closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
//			}
//
//			// rama score for moveable residues
//			score_rama_for_moveable( pose, closure_info );
//
//			// use scores to decide on closure status
//			closure_info.set_is_closed( closure_info.chainbreak_score() <= chainbreak_criterion &&
//										closure_info.ramas_are_not_above( rama_moveable_criterion ) );
//
//			std::cout << closure_info.to_string() << std::endl;
//		}
//
//		// trial status
//		std::cout << " | frag: " << frag_accepts << " / " << frag_trials << " = " << (float)frag_accepts / (float)( frag_trials == 0 ? 1 : frag_trials );
//		std::cout << std::endl;
//
//	}


	// restore any prior status
	pose.set_fullatom_flag( prior_fullatom_flag, false ); // boolean: fullatom, repack

	set_pose_loop_flag( prior_pose_loop_flag ); // reset
}


/// @brief fragment picking test for build loops simul procedures
/// @details procedure does nothing except pick fragments
void
fragment_test_build_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt
)
{
	using namespace pose_ns;

	// setup weights for centroid level remodeling
	// TODO: wrap weights for ease of use
	Score_weight_map weight_map;
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( RAMACHANDRAN, 0.1 );
	weight_map.set_weight( RG, 1.0 );

	// additional scoring setup
	score_set_cst_mode(3);

	// cutoffs for fragment picking and operations
	Integer cutoff9 = 8; // 16
	Integer cutoff3 = 2; // 6
	Integer cutoff1 = 0; // 3

	// offsets for fragment insertion
//	Integer const frag_offset = 0;

	bool const use_fragment_insertion = true;

	if ( use_fragment_insertion ) {
		// update max res manually, as this may not necessarily be done beforehand
		pose_update_MAX_RES( pose );

		// reset global fragment arrays
		fragments::reset_fragment_arrays_used_by_Vall();
	}

	// secondary structure string for fragment insertion
	std::ostringstream ss_ss;
	for ( Integer r = 1, re = pose.total_residue(); r <= re; ++ r ) {
		ss_ss << pose.secstruct( r );
	}

	bool fragments_were_picked = false; // see if fragments were actually selected

	// strings indicating desired fragments
	String sequence_str( pose.total_residue(), '.' ); // all "." so not used
	String secondary_structure_str( ss_ss.str() );
	String bigbin_str( pose.total_residue(), '.' ); // all "." so not used

	// weights for fragment picking
	Real sequence_weight = 1.0;
	Real secondary_structure_weight = 1.0;
	Real bigbin_weight = 0.0; // not used

	// now we setup 9-mers and 3-mers, unfortunately due to the way the fragment picking routine is setup,
	// we are forced to cycle through the loops each time, first setting up 9-mers and then setting up 3-mers
	// it's not possible to do this in a single 'for' loop

	// setup 9-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		// actually use fragments for this loop?  check length
		// no closure check here -- we need to pick fragments even for
		// closed loops in case the loop is broken again
		bool const use_9mers = use_fragment_insertion && ( loop_length > cutoff9 );

		fragments_were_picked = fragments_were_picked || use_9mers;

		if ( use_9mers ) { // pick 9-mers
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                9, 200,
			                true, true, true, 3 ); // booleans: exclude gly, pro, cis-peptide
		} // if use_fragments

	} // foreach loop (9-mer setup)

	// setup 3-mers for each loop
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		Integer loop_length = closure_info.moveable_residues_width(); // cache

		// actually use fragments for this loop?  check length
		// no closure check here -- we need to pick fragments even for
		// closed loops in case the loop is broken again
		bool const use_3mers = use_fragment_insertion &&
		                           ( ( loop_length > cutoff3 ) ||
		                             ( loop_length > cutoff1 ) );

		fragments_were_picked = fragments_were_picked || use_3mers;

		if ( use_3mers ) { // pick 3-mers
			get_vall_frags( sequence_str, secondary_structure_str, bigbin_str,
			                sequence_weight, secondary_structure_weight, bigbin_weight,
			                closure_info.loop_range().begin(), closure_info.loop_range().end(),
			                3, 200,
			                true, true, true, 2 ); // booleans: exclude gly, pro, cis-peptide
		} // if use_fragments

	} // foreach loop (3-mer setup)

	// if applicable, build 1-mers and indicate usage of top fragments
	if ( fragments_were_picked ) {
		build_1mer_from_3mer( pose.total_residue() );
		choose_frag_set_top_N_frags( 200 ); // use top 200 fragments for each size
	}
}


/// @brief   (full atom) refine loop
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details allow_bb_move must be set outside _before_ entering this procedure
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues)
/// @details Loops that are marked as not closed will have their cutpoints idealized.
void
refine_loop(
	Pose & pose,
	LoopClosureInfo const & closure_info,
	Integer const & outer_cycles,
	bool const & fast_refine,
	std::set< Integer > const & fixed_chi_residues
)
{
	using namespace pose_ns;

	// check full-atom flag
	if ( !pose.fullatom() ) {
		std::cerr << "epigraft::design::refine_loop() : begin called with a non full-atom Pose, refusing to refine!" << std::endl;
		return;
	}

	// minimizer parameters
	minimize_reset();
	minimize_set_vary_chi( true );
	minimize_set_tolerance( 0.001 );

	// score weights
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );

	// initial chainbreak weights -- scores only the chainbreak of the current loop
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 ); // for scoring chainbreak of the current loop
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();
	cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// repack behavior
	bool const include_existing_sc_in_repack = true;

	// setup initial allow chi move
	std::set< LoopClosureInfo > closures; // scratch, only used for setting proper chi move
	closures.insert( closure_info );
	setup_proper_allow_chi_move( pose, closures, 5.5, fixed_chi_residues );

	// initial repack to allow things to settle down a bit due to any strangeness of loop
	pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true

	// inital score
	pose.score( weight_map );

	// monte carlo cycles, outer_cycles is user defined
	bool fast_inner_cycle = fast_refine;
	int const cycle_number = std::max( 2, (int)( closure_info.moveable_residues_width() * get_looprlx_cycle_ratio() ) );
	int const inner_cycles = fast_inner_cycle ? std::min( 20, cycle_number ) : std::min( 200, 10*cycle_number );

	// monte carlo parameters
	Real const init_temp = 1.5;
	Real const last_temp = 0.5;
	Real const gamma = std::pow( ( last_temp/init_temp ), 1.0f/( outer_cycles*inner_cycles ) );
	Monte_carlo mc( pose, weight_map, init_temp );

	// sanity check for ccd
	bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
	utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
	if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
	     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
		do_ccd = false;
	}

	// begin closure attempt
	int small_accepted = 0, shear_accepted = 0, ccd_accepted = 0; // diagnostics
	int n_trial = 0; // diagnostics
	Real temperature = init_temp;
	for ( int i = 1, ii = i + 3; i <= outer_cycles; ++i, ++ii ) {

		// chainbreak ramping
		weight_map.set_weight( CHAINBREAK, std::min( (Real)i, (Real)3.0 ) ); // don't allow chainbreak weight to climb above 3.0
		mc.set_weight_map( weight_map );

		// recover low
		pose = mc.low_pose();

		// show score info for outer cycle
		std::cout << "outer_cycle: " << i << "  ";
		pose.show_scores( std::cout );
		std::cout << std::endl;

		// inner cycles
		for ( int j = 1; j <= inner_cycles; ++j ) {
			n_trial++;

			// alter temperature
			temperature *= gamma;
			mc.set_temperature( temperature );

			// small + shear trial (perturb configuration)
			pose_small_moves( pose, 1 ); // 1 random small move
			if ( mc.boltzmann( pose ) ) {
				++small_accepted;
			}
			pose_shear_moves( pose, 1 ); // 1 random shear move
			if ( mc.boltzmann( pose ) ) {
				++shear_accepted;
			}

			// ccd
			if ( do_ccd ) {
				ccd_loop_closure_obeying_nonideality( pose, closure_info );
			}
			if ( mc.boltzmann( pose ) ) {
				++ccd_accepted;
			}

		} // inner_cycles

		// minimize
		pose.main_minimize( weight_map, "dfpmin" );
		mc.boltzmann( pose );

	} // outer_cycles


	// recover low
	pose = mc.low_pose();

	std::cout << "refine_loops_simul -- final scores:" << std::endl;
	pose.show_scores( std::cout );
	std::cout << std::endl;
	std::cout << "refine_loops_simul -- small_accepted/trial/ratio:  "
	          << small_accepted << "/" << n_trial << "/"
	          << float( small_accepted )/float( n_trial )
	          << std::endl
	          << "                      shear_accepted/trial/ratio:  "
	          << shear_accepted << "/" << n_trial << "/"
	          << float( shear_accepted )/float( n_trial )
	          << std::endl
	          << "                      ccd_accepted/trial/ratio:  "
	          << ccd_accepted << "/" << n_trial << "/"
	          << float( ccd_accepted )/float( n_trial )
	          << std::endl;

	// store linear score
	if ( do_ccd ) {
		closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
	} else {
		closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
	}

	// rama score for moveable residues
	score_rama_for_moveable( pose, closure_info );

}


/// @brief (full atom) refine loops simultaneously
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
refine_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles,
	bool const & fast_refine,
	std::set< Integer > const & fixed_chi_residues
)
{
	using namespace pose_ns;

	// check full-atom flag
	if ( !pose.fullatom() ) {
		std::cerr << "epigraft::design::refine_loops_simul() : begin called with a non full-atom Pose, refusing to refine!" << std::endl;
		return;
	}

	// minimizer parameters
	minimize_reset();
	minimize_set_vary_chi( true );
	minimize_set_tolerance( 0.001 );

	// score weights
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );

	// chainbreak weighting
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 );
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// turn on ALL cutpoint weights to score all loops simultaneously
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// repack behavior
	bool const include_existing_sc_in_repack = true;

	// setup initial allow chi move
	setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

	// initial repack of allowed move region to settle down a bit due to any strangeness of loop
	pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true

	// initial score
	pose.score( weight_map );

	// monte carlo cycles, outer_cycles is user defined
	bool fast_inner_cycle = fast_refine;

	// monte carlo parameters
	Real const init_temp = 1.5;
	Real const last_temp = 0.5;
	Monte_carlo mc( pose, weight_map, init_temp );

	// set cycles based on length of loop
	int const cycle_number = std::max( 2, (int)( get_looprlx_cycle_ratio() ) );
	int const inner_cycles = fast_inner_cycle ? std::min( 20, cycle_number ) : std::min( 200, 10*cycle_number );
	Real const gamma = std::pow( ( last_temp/init_temp ), static_cast< Real >( 1.0 )/( outer_cycles*inner_cycles ) );

	int move_accepted = 0; // diagnostics
	int n_trial = 0; // diagnostics
	Real temperature = init_temp;
	for ( int i = 1, ii = i + 3; i <= outer_cycles; ++i, ++ii ) {

		// create random order of closures to attempt
		utility::vector1< LoopClosureInfo > randomized_closures_to_attempt( closures_to_attempt.begin(), closures_to_attempt.end() );
		std::random_shuffle( randomized_closures_to_attempt.begin(), randomized_closures_to_attempt.end() );

		// chainbreak ramping
		weight_map.set_weight( CHAINBREAK, std::min( (Real)i, (Real)3.0 ) ); // don't allow chainbreak weight to climb above 3.0
		mc.set_weight_map( weight_map );

		// recover low
		pose = mc.low_pose();

		// show score info for outer cycle
		std::cout << "outer_cycle: " << i << "  ";
		pose.show_scores( std::cout );
		std::cout << std::endl;

		// build each loop
		for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
			LoopClosureInfo const & closure_info = *c;

			// sanity check for ccd
			bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
			utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
			if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
			     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
				do_ccd = false;
			}

			// (re-)setup proper chi move for inner cycles (e.g. minimize)
			setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

			// inner cycles
			for ( int j = 1; j <= inner_cycles; ++j ) {
				n_trial++;

				// alter temperature
				temperature *= gamma;
				mc.set_temperature( temperature );

				// small + shear trial (perturb configuration)
				pose_small_moves( pose, 1 ); // 1 random small move
				pose_shear_moves( pose, 1 ); // 1 random shear move

				// ccd
				if ( do_ccd ) {
					ccd_loop_closure_obeying_nonideality( pose, closure_info );
				}

				// accept/reject
				if ( mc.boltzmann( pose ) ) {
					++move_accepted;
				}

				// repack
				if ( mod( j, 20 ) == 0 || j == inner_cycles ) {
					// setup proper chi move for repack
					setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

					pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true
					pose.score( weight_map );
					mc.boltzmann( pose );
				}
			} // inner_cycles

			// minimize
			pose.main_minimize( weight_map, "dfpmin" );
			mc.boltzmann( pose );

			// recover low
			pose = mc.low_pose();

			// store linear score
			if ( do_ccd ) {
				closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
			} else {
				closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
			}

			// rama score for moveable residues
			score_rama_for_moveable( pose, closure_info );

			// closure status
//			std::cout << "** " << closure_info.to_string() << std::endl;

		} // foreach loop

	} // outer_cycles

	// recover low
	pose = mc.low_pose();

	std::cout << "refine_loops_simul -- final scores:" << std::endl;
	pose.show_scores( std::cout );
	std::cout << std::endl;
	std::cout << "refine_loops_simul -- move_accepted/trial/ratio:  "
	          << move_accepted << "/" << n_trial << "/"
	          << float( move_accepted )/float( n_trial )
	          << std::endl;

	// need to rescore all loops and store in LoopClosureInfo objects that were passed in
	// (since objects used above are temporary)
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
		if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
		     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		} else {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );
	}

}


/// @brief full atom minimize loop simultaneously + repack
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
minrepack_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & cycles,
	std::set< Integer > const & fixed_chi_residues,
	Real const & chainbreak_weight
)
{
	using namespace pose_ns;

	// check full-atom flag
	if ( !pose.fullatom() ) {
		std::cerr << "epigraft::design::minrepack_loops_simul() : called with a non full-atom Pose, refusing to minimize!" << std::endl;
		return;
	}

	// minimizer parameters
	minimize_reset();
	minimize_set_vary_chi( true );
	minimize_set_tolerance( 0.001 );
//	minimize_set_tolerance( 1.0 );

	// score weights
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );

	// chainbreak weighting
	weight_map.set_weight( CHAINBREAK, chainbreak_weight ); // default 3.0, need constant HIGH chainbreak weighting
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 );
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// turn on ALL cutpoint weights to score all loops simultaneously
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// repack behavior
	bool const include_existing_sc_in_repack = true;

	// setup initial allow chi move
	setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

	// initial score
	pose.score( weight_map );

	// monte carlo parameters
	Real const init_temp = 1.5;
	Real const last_temp = 0.5;
	Real const gamma = std::pow( ( last_temp/init_temp ), static_cast< Real >( 1.0 )/( cycles ) ); // temperature scaling
	Monte_carlo mc( pose, weight_map, init_temp );

	Real temperature = init_temp;
	for ( int i = 1; i <= cycles; ++i ) {
		// show score info for outer cycle
		std::cout << "start of cycle: " << i << "  ";
		pose.show_scores( std::cout );
		std::cout << std::endl;

		// alter temperature
		temperature *= gamma;
		mc.set_temperature( temperature );

		// setup proper chi move for repack
		setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

		// repack
		pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true
		pose.score( weight_map );
		mc.boltzmann( pose );

		// minimize
		pose.main_minimize( weight_map, "dfpmin" );
		mc.boltzmann( pose );

		// recover low
		pose = mc.low_pose();
	}

	std::cout << "minrepack_loops_simul -- final scores:" << std::endl;
	pose.show_scores( std::cout );
	std::cout << std::endl;

	// need to score all loops and store in LoopClosureInfo objects that were passed in
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
		if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
		     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		} else {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );
	}
}


/// @brief (full atom) refine loops simultaneously, original pose loops protocol
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
refine_loops_simul_classic(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles,
	bool const & fast_refine,
	std::set< Integer > const & fixed_chi_residues
)
{
	using namespace pose_ns;

	// check full-atom flag
	if ( !pose.fullatom() ) {
		std::cerr << "epigraft::design::refine_loops_simul_classic() : begin called with a non full-atom Pose, refusing to refine!" << std::endl;
		return;
	}

	// minimizer parameters
	minimize_reset();
	minimize_set_vary_chi( true );
//	minimize_set_tolerance( 1.0 );

	// score weights
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );

	// chainbreak weighting
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 );
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// turn on ALL cutpoint weights to score all loops simultaneously
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// repack behavior
	bool const include_existing_sc_in_repack = true;

	// setup initial allow chi move
	setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

	// initial repack of allowed move region to settle down a bit due to any strangeness of loop
	pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true

	// initial score
	pose.score( weight_map );

	// monte carlo cycles, outer_cycles is user defined
	bool fast_inner_cycle = fast_refine;

	// monte carlo parameters
	Real const init_temp = 1.5;
	Real const last_temp = 0.5;
	Monte_carlo mc( pose, weight_map, init_temp );

	// set cycles based on length of loop
	int const cycle_number = std::max( 2, (int)( get_looprlx_cycle_ratio() ) );
	int const inner_cycles = fast_inner_cycle ? std::min( 20, cycle_number ) : std::min( 200, 10*cycle_number );
	Real const gamma = std::pow( ( last_temp/init_temp ), static_cast< Real >( 1.0 )/( outer_cycles*inner_cycles ) );

	int small_accepted = 0, shear_accepted = 0; // diagnostics
	int n_trial = 0; // diagnostics
	Real temperature = init_temp;
	for ( int i = 1, ii = i + 3; i <= outer_cycles; ++i, ++ii ) {

		// create random order of closures to attempt
		utility::vector1< LoopClosureInfo > randomized_closures_to_attempt( closures_to_attempt.begin(), closures_to_attempt.end() );
		std::random_shuffle( randomized_closures_to_attempt.begin(), randomized_closures_to_attempt.end() );

		// chainbreak ramping
		weight_map.set_weight( CHAINBREAK, std::min( (Real)i, (Real)3.0 ) ); // don't allow chainbreak weight to climb above 3.0
		mc.set_weight_map( weight_map );

		// recover low
		pose = mc.low_pose();

		// show score info for outer cycle
		std::cout << "outer_cycle: " << i << "  ";
		pose.show_scores( std::cout );
		std::cout << std::endl;

		// build each loop
		for ( utility::vector1< LoopClosureInfo >::const_iterator c = randomized_closures_to_attempt.begin(), ce = randomized_closures_to_attempt.end(); c != ce; ++c ) {
			LoopClosureInfo const & closure_info = *c;

			// sanity check for ccd
			bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
			utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
			if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
			     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
				do_ccd = false;
			}

			// (re-)setup proper chi move for inner cycles (e.g. minimize)
			setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

			// inner cycles
			for ( int j = 1; j <= inner_cycles; ++j ) {
				n_trial++;

				// alter temperature
				temperature *= gamma;
				mc.set_temperature( temperature );

				// small trial (perturb configuration)
				pose_small_moves( pose, 1 ); // 1 random small move
				// ccd
				if ( do_ccd ) {
					ccd_loop_closure_obeying_nonideality( pose, closure_info );
				}
				// minimize
				pose.main_minimize( weight_map, "dfpmin" );
				if ( mc.boltzmann( pose ) ) {
					++small_accepted;
				}

				// shear trial (perturb configuration)
				pose_shear_moves( pose, 1 ); // 1 random shear move
				// ccd
				if ( do_ccd ) {
					ccd_loop_closure_obeying_nonideality( pose, closure_info );
				}
				// minimize
				pose.main_minimize( weight_map, "dfpmin" );
				if ( mc.boltzmann( pose ) ) {
					++shear_accepted;
				}

				// repack
				if ( mod( j, 20 ) == 0 || j == inner_cycles ) {
					// setup proper chi move for repack
					setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

					pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true
					pose.score( weight_map );
					mc.boltzmann( pose );
				}
			} // inner_cycles

			// recover low
			pose = mc.low_pose();

			// store linear score
			if ( do_ccd ) {
				closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
			} else {
				closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
			}

			// rama score for moveable residues
			score_rama_for_moveable( pose, closure_info );

			// closure status
//			std::cout << "** " << closure_info.to_string() << std::endl;

		} // foreach loop

	} // outer_cycles

	// recover low
	pose = mc.low_pose();

	std::cout << "refine_loops_simul_classic -- final scores:" << std::endl;
	pose.show_scores( std::cout );
	std::cout << std::endl;
	std::cout << "refine_loops_simul_classic -- small_accepted/trial/ratio:  "
	          << small_accepted << "/" << n_trial << "/"
	          << float( small_accepted )/float( n_trial )
	          << std::endl
	          << "                      shear_accepted/trial/ratio:  "
	          << shear_accepted << "/" << n_trial << "/"
	          << float( shear_accepted )/float( n_trial )
	          << std::endl;

	// need to rescore all loops and store in LoopClosureInfo objects that were passed in
	// (since objects used above are temporary)
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
		if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
		     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		} else {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );
	}

}


/// @brief (full atom) refine loops simultaneously, one-at-a-time original pose loops protocol
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
refine_loops_simul_classic_two(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles,
	bool const & fast_refine,
	std::set< Integer > const & fixed_chi_residues
)
{
	using namespace pose_ns;

	// check full-atom flag
	if ( !pose.fullatom() ) {
		std::cerr << "epigraft::design::refine_loops_simul_classic_two() : begin called with a non full-atom Pose, refusing to refine!" << std::endl;
		return;
	}

	// minimizer parameters
	minimize_reset();
	minimize_set_vary_chi( true );
//	minimize_set_tolerance( 1.0 );

	// score weights
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );

	// chainbreak weighting
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	FArray1D< Real > cutpoint_weight( pose.fold_tree().get_num_fold_tree_cutpoint(), 0.0 );
	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();

	// turn on ALL cutpoint weights to score all loops simultaneously
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;
		cutpoint_weight( cutpoint_map( closure_info.cut() ) ) = closure_info.cutpoint_weight();
	}
	weight_map.set_1D_weight( CUT_WEIGHT, cutpoint_weight );

	// repack behavior
	bool const include_existing_sc_in_repack = true;

	// setup initial allow chi move
	setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

	// initial repack of allowed move region to settle down a bit due to any strangeness of loop
	pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true

	// initial score
	pose.score( weight_map );

	// monte carlo parameters
	Real const init_temp = 1.5;
	Real const last_temp = 0.5;
	Monte_carlo mc( pose, weight_map, init_temp );

	// monte carlo cycles, outer_cycles is user defined
	bool fast_inner_cycle = fast_refine;
	int const cycle_number = std::max( 2, (int)( closures_to_attempt.size() * get_looprlx_cycle_ratio() ) );
	int const inner_cycles = fast_inner_cycle ? std::min( 20, cycle_number ) : std::min( 200, 10*cycle_number );
	Real const gamma = std::pow( ( last_temp/init_temp ), static_cast< Real >( 1.0 )/( outer_cycles*inner_cycles ) );

	// sanity check for ccd
	bool do_ccd = true; // if false, this indicates that we're at the true begin/end of a chain
	utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );

	std::vector< LoopClosureInfo > closures( closures_to_attempt.begin(), closures_to_attempt.end() );

	int small_accepted = 0, shear_accepted = 0; // diagnostics
	int n_trial = 0; // diagnostics
	Real temperature = init_temp;
	for ( int i = 1, ii = i + 3; i <= outer_cycles; ++i, ++ii ) {

		// chainbreak ramping
		weight_map.set_weight( CHAINBREAK, std::min( (Real)i, (Real)3.0 ) ); // don't allow chainbreak weight to climb above 3.0
		mc.set_weight_map( weight_map );

		// recover low
		pose = mc.low_pose();

		// show score info for outer cycle
		std::cout << "outer_cycle: " << i << "  ";
		pose.show_scores( std::cout );
		std::cout << std::endl;

		// (re-)setup proper chi move for inner cycles (e.g. minimize)
		setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

		// inner cycles
		for ( int j = 1; j <= inner_cycles; ++j ) {
			n_trial++;

			// alter temperature
			temperature *= gamma;
			mc.set_temperature( temperature );

			// small trial (perturb configuration)
			LoopClosureInfo const & closure_info_small = closures[ static_cast< Size >( ran3() * closures.size() ) ];
			do_ccd = true;
			if ( ( ( peptide_degree[ closure_info_small.loop_range().begin() ] == 1 )  && ( closure_info_small.cut() == ( closure_info_small.loop_range().begin() - 1 ) ) ) ||
				 ( ( peptide_degree[ closure_info_small.loop_range().end() ] == 1 )  && ( closure_info_small.cut() == ( closure_info_small.loop_range().end() ) ) ) ) {
				do_ccd = false;
			}
			pose.set_allow_bb_move( false );
			closure_info_small.setup_allow_bb_move( pose );
			pose_small_moves( pose, 1 ); // 1 random small move
			// ccd
			if ( do_ccd ) {
				ccd_loop_closure_obeying_nonideality( pose, closure_info_small );
			}
			// minimize
			pose.main_minimize( weight_map, "dfpmin" );
			if ( mc.boltzmann( pose ) ) {
				++small_accepted;
			}

			// shear trial (perturb configuration)
			LoopClosureInfo const & closure_info_shear = closures[ static_cast< Size >( ran3() * closures.size() ) ];
			do_ccd = true;
			if ( ( ( peptide_degree[ closure_info_shear.loop_range().begin() ] == 1 )  && ( closure_info_shear.cut() == ( closure_info_shear.loop_range().begin() - 1 ) ) ) ||
				 ( ( peptide_degree[ closure_info_shear.loop_range().end() ] == 1 )  && ( closure_info_shear.cut() == ( closure_info_shear.loop_range().end() ) ) ) ) {
				do_ccd = false;
			}
			pose.set_allow_bb_move( false );
			closure_info_shear.setup_allow_bb_move( pose );
			pose_shear_moves( pose, 1 ); // 1 random shear move
			// ccd
			if ( do_ccd ) {
				ccd_loop_closure_obeying_nonideality( pose, closure_info_shear );
			}
			// minimize
			pose.main_minimize( weight_map, "dfpmin" );
			if ( mc.boltzmann( pose ) ) {
				++shear_accepted;
			}

			// repack
			if ( mod( j, 20 ) == 0 || j == inner_cycles ) {
				// setup proper chi move for repack
				setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

				pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true
				pose.score( weight_map );
				mc.boltzmann( pose );
			}
		} // inner_cycles

		// recover low
		pose = mc.low_pose();

	} // outer_cycles

	// recover low
	pose = mc.low_pose();

	std::cout << "refine_loops_simul_classic_two -- final scores:" << std::endl;
	pose.show_scores( std::cout );
	std::cout << std::endl;
	std::cout << "refine_loops_simul_classic_two -- small_accepted/trial/ratio:  "
	          << small_accepted << "/" << n_trial << "/"
	          << float( small_accepted )/float( n_trial )
	          << std::endl
	          << "                      shear_accepted/trial/ratio:  "
	          << shear_accepted << "/" << n_trial << "/"
	          << float( shear_accepted )/float( n_trial )
	          << std::endl;

	// need to rescore all loops and store in LoopClosureInfo objects that were passed in
	// (since objects used above are temporary)
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		utility::vector1< Integer > peptide_degree = PoseAssembly::count_peptide_degree( pose.fold_tree() );
		if ( ( ( peptide_degree[ closure_info.loop_range().begin() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().begin() - 1 ) ) ) ||
		     ( ( peptide_degree[ closure_info.loop_range().end() ] == 1 )  && ( closure_info.cut() == ( closure_info.loop_range().end() ) ) ) ) {
			closure_info.set_chainbreak_score( 0 ); // chainbreak nonexistent since true begin/end of chain
		} else {
			closure_info.set_chainbreak_score( score_cut_in_Pose_linear( pose, closure_info.cut(), 1 ) ); // overlap = 1
		}

		// rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );
	}

}


/// @brief (full atom) refine loops simultaneously using constraints to anchor structure
/// @details non-moveable residues (both loop and outside loop) are anchored by coordinate constraints
/// @details function should really only be called with a sealed fold tree / fully closed structure
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
refine_loops_simul_constrained(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & cycles,
	std::set< Integer > const & fixed_chi_residues
)
{
	using namespace pose_ns;

	// check full-atom flag
	if ( !pose.fullatom() ) {
		std::cerr << "epigraft::design::minrepack_loops_simul() : called with a non full-atom Pose, refusing to minimize!" << std::endl;
		return;
	}

	// cache
	FArray3D_float const & full_coord = pose.full_coord();

//	// setup constraints to tether all chainbreaks by constraining peptide bond geometry
//	// bb heavy atoms are N,CA,C,O indexed from 1 to 4, respectively, in full_coord
//	cst_set_ns::Cst_set constraints;
//	for ( std::set< LoopClosureInfo >::const_iterator c = closures_to_attempt.begin(), ce = closures_to_attempt.end(); c != ce; ++c ) {
//		LoopClosureInfo const & closure = *c;
//
//		// create atom ids
//		kin::Atom_id left_CA( 2, closure.cut() );
//		kin::Atom_id left_C( 3, closure.cut() );
//		kin::Atom_id right_N( 1, closure.cut() + 1 );
//		kin::Atom_id right_CA( 2, closure.cut() + 1 );
//
//		// grab atom coordinates
//		numeric::xyzVector< Real > left_CA_crd( &full_coord( 1, 2, closure.cut() ) );
//		numeric::xyzVector< Real > left_C_crd( &full_coord( 1, 3, closure.cut() ) );
//		numeric::xyzVector< Real > right_N_crd( &full_coord( 1, 1, closure.cut() + 1 ) );
//		numeric::xyzVector< Real > right_CA_crd( &full_coord( 1, 2, closure.cut() + 1 ) );
//
//		// add constraints surrounding peptide bond
//		constraints.add_atompair_constraint( left_C, right_N, cst_set_ns::Cst( distance( left_C_crd, right_N_crd ) ) );
//		constraints.add_atompair_constraint( left_CA, right_N, cst_set_ns::Cst( distance( left_CA_crd, right_N_crd ) ) );
//		constraints.add_atompair_constraint( left_C, right_CA, cst_set_ns::Cst( distance( left_C_crd, right_CA_crd ) ) );
//		constraints.add_atompair_constraint( left_CA, right_CA, cst_set_ns::Cst( distance( left_CA_crd, right_CA_crd ) ) );
//	}

	// setup coordinate constraints for backbone heavy atoms (N, CA, C, O)
	cst_set_ns::Cst_set constraints;
	for ( Integer res = 1, last_res = pose.total_residue(); res <= last_res; ++res ) {

		// allow_bb_move governs constraints
		if ( !pose.get_allow_bb_move( res ) ) {
			for ( Integer atom = 1; atom <= 4; ++atom ) { // fullatom N, CA, C, O
				constraints.add_coordinate_constraint( kin::Atom_id( atom, res ), numeric::xyzVector< float >( &full_coord( 1, atom, res ) ) );
			}
		}
	}

	// set constraints
	pose.set_constraints( constraints );

	// minimizer parameters
	minimize_reset(); // nothing except phi-psi and rb (rb is disallowed at Pose level below)
	minimize_set_tolerance( 0.001 );
	minimize_set_vary_chi( true );

	// score weights
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );
	weight_map.set_weight( COORD_CST, 100.0 );

	// repack behavior
	bool const include_existing_sc_in_repack = true;

	// initial repack of allowed move region to settle down a bit due to any strangeness of loop
	setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );
	pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true

	pose.score( weight_map );

	// monte carlo
	Monte_carlo mc( pose, weight_map, 1.0 );

	// do refine
	for ( Integer i = 1; i <= cycles; ++i ) {
		// show score info for outer cycle
		std::cout << "start of cycle: " << i << "  ";
		pose.show_scores( std::cout );
		std::cout << std::endl;

		// setup appropriate chi move
		setup_proper_allow_chi_move( pose, closures_to_attempt, 5.5, fixed_chi_residues );

		pose.main_minimize( weight_map, "dfpmin" );
		mc.boltzmann( pose );

		pose.full_repack( include_existing_sc_in_repack ); // only repacks things where allow_chi_move is set to true
		pose.score( weight_map );
		mc.boltzmann( pose );
	}

	// recover low
	pose = mc.low_pose();

	std::cout << "refine_loops_simul_constrained -- final scores:" << std::endl;
	pose.show_scores( std::cout );
	std::cout << std::endl;

	// clear constraints
	pose.reset_constraints();

	// need to rescore all loops and store in LoopClosureInfo objects that were passed in
	// (since objects used above are temporary)
	for ( std::set< LoopClosureInfo >::const_iterator i = closures_to_attempt.begin(), ie = closures_to_attempt.end(); i != ie; ++i ) {
		LoopClosureInfo const & closure_info = *i;

		// no chainbreak score (since Pose is sealed), just rama score for moveable residues
		score_rama_for_moveable( pose, closure_info );
	}
}


/// @brief ccd loop closure, obeys non-ideal bond geometry
/// @details  weights (chainbreak/overlap/etc) must be setup outside of this routine
void
ccd_loop_closure_obeying_nonideality(
	Pose & pose,
	LoopClosureInfo const & closure_info,
	Integer const & ccd_cycles,
	bool const & use_greedy
)
{
	// ccd closure parameters
//	Integer const ccd_cycles = 100;
	Real const ccd_tol    = 0.01; // closed loop criterion

	// ccd rama parameters
	bool const rama_check = true; // always rama check
	Real const max_rama_score_increase =  2.0; // dummy number when rama_check is false
	Real const max_total_delta_helix   = 10.0; // max overall angle changes for a helical residue
	Real const max_total_delta_strand  = 50.0; // max overall angle changes for a strand residue
	Real const max_total_delta_loop    = 75.0; // max overall angle changes for a loop residue

	// ccd closure output
	Real forward_deviation, backward_deviation; // loop closure msd
	Real torsion_delta, rama_delta; // torsion and rama score changes, averaged by loop_size

	// ccd close
	if ( !use_greedy ) {
		fast_ccd_loop_closure_obeying_nonideality( pose,
		                       closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut(),
		                       ccd_cycles, ccd_tol,
		                       rama_check, max_rama_score_increase,
		                       max_total_delta_helix, max_total_delta_strand, max_total_delta_loop,
		                       forward_deviation, backward_deviation, torsion_delta, rama_delta );
	} else {
		greedy_ccd_loop_closure_obeying_nonideality( pose,
		                       closure_info.loop_range().begin(), closure_info.loop_range().end(), closure_info.cut(),
		                       ccd_cycles, ccd_tol,
		                       rama_check, max_rama_score_increase,
		                       max_total_delta_helix, max_total_delta_strand, max_total_delta_loop,
		                       forward_deviation, backward_deviation, torsion_delta, rama_delta );
	}
}


/// @brief score cut in Pose linear chainbreak
Real
score_cut_in_Pose_linear(
	Pose const & pose,
	Integer const & cut,
	Integer const & overlap
)
{
	using namespace param_aa;
	Real chainbreak_score = 0.0;

	if ( !pose.is_protein( cut ) || !pose.is_protein( cut + 1 ) ) {
		return DESIGN_INFINITY;
	}

	// initialize
	FArray3D< Real > const & Eposition = pose.Eposition();
	pose.copy_to_misc(); // unfortunately Pose::get_overlap_Eposition() needs explicit misc loading call to work correctly
	FArray4D< Real > const & jmp_overlap_Eposition( pose.get_overlap_Eposition( overlap ) );

	// the next two are necessary because the 3rd dimension of jmp_overlap_Eposition is
	// dynamic range based indexing
	Integer istart = std::max( 1 - overlap, 1 - cut );
	Integer istop  = std::min( overlap, pose.total_residue() - cut );

	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();
	Integer const ncut = cutpoint_map( cut );

	// compute score
	for ( int i = istart; i <= istop; ++i ) {
		int seqpos = cut + i;
		for ( int j = 1; j <= 5; ++j ) {
			if ( j != 3 || pose.res( seqpos ) != aa_gly ) { // no gly c-beta
				chainbreak_score +=
					std::sqrt( square( Eposition( 1, j, seqpos ) - jmp_overlap_Eposition( 1, j, i, ncut ) ) +
					           square( Eposition( 2, j, seqpos ) - jmp_overlap_Eposition( 2, j, i, ncut ) ) +
					           square( Eposition( 3, j, seqpos ) - jmp_overlap_Eposition( 3, j, i, ncut ) ) );
			} // no gly c-beta
		} // j
	} // i

	// NOTE: The original linear chainbreak score in jmp_chainbreak_score divided by 20 before
	//       returning the value... some magic number which is unclear.  For the moment we don't
	//       want that behavior.
	return chainbreak_score;
}


/// @brief score cut in Pose linear chainbreak
Real
score_cut_in_Pose_quadratic(
	Pose const & pose,
	Integer const & cut,
	Integer const & overlap
)
{
	using namespace param_aa;
	Real chainbreak_score = 0.0;

	if ( !pose.is_protein( cut ) || !pose.is_protein( cut + 1 ) ) {
		return DESIGN_INFINITY;
	}

	// initialize
	FArray3D< Real > const & Eposition = pose.Eposition();
	pose.copy_to_misc(); // unfortunately Pose::get_overlap_Eposition() needs explicit misc loading call to work correctly
	FArray4D< Real > const & jmp_overlap_Eposition( pose.get_overlap_Eposition( overlap ) );

	// the next two are necessary because the 3rd dimension of jmp_overlap_Eposition is
	// dynamic range based indexing
	Integer istart = std::max( 1 - overlap, 1 - cut );
	Integer istop  = std::min( overlap, pose.total_residue() - cut );

	FArray1D< Integer > const & cutpoint_map = pose.fold_tree().get_cutpoint_map();
	Integer const ncut = cutpoint_map( cut );

	// compute score
	for ( int i = istart; i <= istop; ++i ) {
		int seqpos = cut + i;
		for ( int j = 1; j <= 5; ++j ) {
			if ( j != 3 || pose.res( seqpos ) != aa_gly ) { // no gly c-beta
				chainbreak_score +=
					square( Eposition( 1, j, seqpos ) - jmp_overlap_Eposition( 1, j, i, ncut ) ) +
					square( Eposition( 2, j, seqpos ) - jmp_overlap_Eposition( 2, j, i, ncut ) ) +
					square( Eposition( 3, j, seqpos ) - jmp_overlap_Eposition( 3, j, i, ncut ) );
			} // no gly c-beta
		} // j
	} // i

	return chainbreak_score;
}


/// @brief score local rama for moveable residues in LoopClosureInfo
void
score_rama_for_moveable(
	Pose const & pose,
	LoopClosureInfo const & closure
)
{
	Real residue_rama, drama_dphi, drama_dpsi;
	for ( std::set< Integer >::const_iterator i = closure.moveable_residues().begin(), ie = closure.moveable_residues().end(); i != ie; ++i ) {
		Integer const & res_number = *i;

		eval_rama_score_residue( pose.res( res_number ), pose.phi( res_number ),
		                         pose.psi( res_number), pose.secstruct( res_number ),
		                         residue_rama, drama_dphi, drama_dpsi );
		closure.set_rama_moveable( res_number, residue_rama );
	}
}


/// @brief score local composite rama for moveable residues in LoopClosureInfo
/// @details scores rama for a position assuming A, G, and V, and then takes
/// @details the minimum of the three scores
void
score_composite_rama_for_moveable(
	Pose const & pose,
	LoopClosureInfo const & closure
)
{
	Real residue_rama, drama_dphi, drama_dpsi;
	Real ala_rama, gly_rama, val_rama;
	for ( std::set< Integer >::const_iterator i = closure.moveable_residues().begin(), ie = closure.moveable_residues().end(); i != ie; ++i ) {
		Integer const & res_number = *i;

		eval_rama_score_residue( param_aa::aa_ala, pose.phi( res_number ),
		                         pose.psi( res_number), pose.secstruct( res_number ),
		                         ala_rama, drama_dphi, drama_dpsi );
		eval_rama_score_residue( param_aa::aa_gly, pose.phi( res_number ),
		                         pose.psi( res_number), pose.secstruct( res_number ),
		                         gly_rama, drama_dphi, drama_dpsi );
		eval_rama_score_residue( param_aa::aa_val, pose.phi( res_number ),
		                         pose.psi( res_number), pose.secstruct( res_number ),
		                         val_rama, drama_dphi, drama_dpsi );
		residue_rama = std::min( std::min( ala_rama, gly_rama ), val_rama );
		closure.set_rama_moveable( res_number, residue_rama );
	}
}


/// @brief find neighboring residues to a set of loops within the given distance cutoff
/// @note  default cutoff is 5.5
std::set< Integer >
find_neighbor_residues_of_loops(
	Pose const & pose,
	std::set< LoopClosureInfo > closures,
	Real const & cutoff
)
{
	using rootstock::BoundingBox;
	using rootstock::Octree;
	using epigraft::AtomPoint;

	Real const max_distance_cutoff = std::max( cutoff, (Real)6.0 );

	std::set< Integer > all_near_residues;

	// bounding box and octree
	BoundingBox bb = bounding_box( pose );
	Octree< AtomPoint > octree_sc( max_distance_cutoff, bb.lower_corner(), bb.upper_corner() );
	fill_octree_sc( octree_sc, pose );

	// run through loops, find residues near moveable residues of loop
	for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure_info = *c;

		for ( std::set< Integer >::const_iterator r = closure_info.moveable_residues().begin(), re = closure_info.moveable_residues().end(); r != re; ++r ) {
			std::set< Integer > nr = near_residues( cutoff, octree_sc, pose, *r );
			all_near_residues.insert( nr.begin(), nr.end() );
		}
	}

	return all_near_residues;
}


/// @brief setup proper allow chi move with respect to loops
/// @details default distance cutoff 5.5 and no fixed chi residues
void
setup_proper_allow_chi_move(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures,
	Real const & cutoff,
	std::set< Integer > const & fixed_chi_residues
)
{
	// reset everything to false
	pose.set_allow_chi_move( false );

	// first set all moveable loop residues to allow chi move
	for ( std::set< LoopClosureInfo >::const_iterator c = closures.begin(), ce = closures.end(); c != ce; ++c ) {
		LoopClosureInfo const & closure_info = *c;
		closure_info.setup_allow_chi_move( pose );
	}

	// next set all nearby residues within cutoff to allow chi move
	std::set< Integer > all_near_residues = find_neighbor_residues_of_loops( pose, closures, cutoff );
	for ( std::set< Integer >::const_iterator i = all_near_residues.begin(), ie = all_near_residues.end(); i != ie ; ++i ) {
		pose.set_allow_chi_move( *i, true );
	}

	// finally set any fixed chi residues
	for ( std::set< Integer >::const_iterator i = fixed_chi_residues.begin(), ie = fixed_chi_residues.end(); i != ie ; ++i ) {
		pose.set_allow_chi_move( *i, false );
	}
}


/// @brief setup proper allow bb move for set of loops
void
setup_proper_allow_bb_move(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures
)
{
	// reset allow bb move
	pose.set_allow_bb_move( false );

	// setup bb move
	for ( std::set< LoopClosureInfo >::const_iterator a = closures.begin(), ae = closures.end(); a != ae; ++a ) {
		a->setup_allow_bb_move( pose );
	}
}


/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values
/// @details if values don't exist in map, then regular idealization occurs
/// @note  this is useful when it's necessary to retain the existing torsions of the cut
void
idealize_cutpoint_retaining_psiphi(
	Pose & pose,
	Integer const & cut,
	std::map< Integer, Real > const & original_torsions
)
{
	// idealize
	pose.insert_ideal_bonds_at_cutpoint( cut );

	// grab original n-side psi if available and set
	std::map< Integer, Real >::const_iterator ns_psi = original_torsions.find( cut );
	if ( ns_psi != original_torsions.end() ) {
		pose.set_psi( cut, ns_psi->second );
	}

	// grab original c-side phi if available and set
	std::map< Integer, Real >::const_iterator cs_phi = original_torsions.find( cut + 1 );
	if ( cs_phi != original_torsions.end() ) {
		pose.set_phi( cut + 1, cs_phi->second );
	}
}


/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values
/// @note  this is useful when it's necessary to retain the existing torsions of the cut
void
idealize_cutpoint_retaining_psiphi(
	Pose & pose,
	Integer const & cut,
	Real const & nside_psi,
	Real const & cside_phi
)
{
	// idealize
	pose.insert_ideal_bonds_at_cutpoint( cut );

	// set original n-side psi
	pose.set_psi( cut, nside_psi );

	// set original c-side phi
	pose.set_phi( cut + 1, cside_phi );
}


} // using namespace design
} // using namespace epigraft
