// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   loop_functions.cc
/// @brief  Loop closure and movement functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)
/// @author Possu Huang (possu@u.washington.edu)

#ifndef INCLUDED_epigraft_design_loop_functions_HH_
#define INCLUDED_epigraft_design_loop_functions_HH_

// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/design/design_constants.hh>
#include <epigraft/design/LoopClosureInfo.fwd.hh>

// Rosetta headers
#include <pose.h>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>

// utility headers
#include <utility/vector1.fwd.hh>

// C++ headers
#include <set>
#include <map>


namespace epigraft {
namespace design {


/// @brief   (centroid level) build loop
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details allow_bb_move must be set outside _before_ entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed. Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
build_loop(
	Pose & pose,
	LoopClosureInfo const & closure_info,
	Integer const & outer_cycles = 10,
	bool const & use_fragment_insertion = true,
	String const & ss_string = "",
	String const & aa_string = "",
	Integer const & nfrag = 200,
	bool const & use_boost = false
);


/// @brief (centroid level) build loops simultaneously
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
build_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles = 10,
	bool const & use_fragment_insertion = true,
	Real const & chainbreak_criterion = 0.0,
	Real const & rama_moveable_criterion = DESIGN_INFINITY,
	String const & ss_string = "",
	String const & aa_string = "",
	Integer const & nfrag = 200,
	bool const & use_variable_frag = false,
	bool const & repick_frag = true,
	bool const & use_boost = false
);


/// @brief (centroid level) build loops simultaneously (arm cycler version)
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
build_loops_simul_arm(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	std::set< Integer > const & arm_residues,
	Integer const & outer_cycles = 10,
	bool const & use_fragment_insertion = true,
	Real const & chainbreak_criterion = 0.0,
	Real const & rama_moveable_criterion = DESIGN_INFINITY,
	String const & ss_string = "",
	String const & aa_string = "",
	Integer const & nfrag = 200,
	bool const & use_variable_frag = false,
	bool const & use_boost = false
);


/// @brief (centroid level) adaptive build loops simultaneously
/// @brief by using fragments + 2-torsion ccd + 1-torsion ccd
/// @brief with percentage of time spent between the three governed by
/// @brief chainbreak score
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
adaptive_build_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles = 10,
	bool const & use_fragment_insertion = true,
	Real const & chainbreak_criterion = 0.0,
	Real const & rama_moveable_criterion = DESIGN_INFINITY,
	String const & ss_string = "",
	String const & aa_string = "",
	Integer const & nfrag = 200,
	bool const & use_variable_frag = false
);


/// @brief (centroid level) screen build loops simultaneously by
/// @brief fragment + min (without vdw) and mc (with vdw)
/// @details allow_bb_move is governed *inside* this procedure, so recommend
/// @details setting all allow_bb_move to false beforehand, as the state will not be kept.
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
/// @warning This procedure ALLOWS secondary structure movement during minimization!
void
screen_build_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles = 10,
	Real const & chainbreak_criterion = 0.0,
	Real const & rama_moveable_criterion = DESIGN_INFINITY,
	Real const & semi_closed_threshold = 10.0,
	String const & ss_string = "",
	String const & aa_string = "",
	Integer const & nfrag = 200
);


/// @brief (centroid level) build loops simultaneously using only fragments
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details Loops that are marked as not closed will have their initial loop conformations
/// @details perturbed.  Fragment insertion will not be applied to any loops that are marked
/// @details as closed.  If ss string not given, fragments are picked via pose secondary structure.
/// @details If aa string not given, fragments are picked without sequence bias.
/// @details As the procedure moves through a single cycle and randomly shuffles through
/// @details the loops, any loops that fall within the chainbreak criterion are
/// @details considered tentatively closed, and no operations are performed until a
/// @details subsequent cycle where the loop happens to break.
/// @param[in] ccd_type  0 = no ccd, 1 = 1-torsion ccd move, 2 = 2-torsion ccd move, 3 = ccd closure
/// @warning Loops may re-break within this procedure depending upon fold-tree connectivity!
/// @warning after this procedure, any full-atom data in Pose will be invalid!
void
build_loops_simul_with_fragments(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles = 10,
	Real const & chainbreak_criterion = 0.0,
	Real const & rama_moveable_criterion = DESIGN_INFINITY,
	Integer const & ccd_type = 0,
	String const & ss_string = "",
	String const & aa_string = "",
	Integer const & nfrag = 200
);


/// @brief fragment picking test for build loops simul procedures
/// @details procedure does nothing except pick fragments
void
fragment_test_build_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt
);


/// @brief   (full atom) refine loop
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details allow_bb_move must be set outside _before_ entering this procedure
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues)
/// @details Loops that are marked as not closed will have their cutpoints idealized.
void
refine_loop(
	Pose & pose,
	LoopClosureInfo const & closure_info,
	Integer const & outer_cycles = 3,
	bool const & fast_refine = false,
	std::set< Integer > const & fixed_chi_residues = std::set< Integer >()
);


/// @brief (full atom) refine loops simultaneously
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
refine_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles = 3,
	bool const & fast_refine = false,
	std::set< Integer > const & fixed_chi_residues = std::set< Integer >()
);


/// @brief full atom minimize loop simultaneously + repack
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
minrepack_loops_simul(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & cycles = 5,
	std::set< Integer > const & fixed_chi_residues = std::set< Integer >(),
	Real const & chainbreak_weight = 3.0
);


/// @brief (full atom) refine loops simultaneously, original pose loops protocol
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
refine_loops_simul_classic(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles = 3,
	bool const & fast_refine = false,
	std::set< Integer > const & fixed_chi_residues = std::set< Integer >()
);


/// @brief (full atom) refine loops simultaneously, one-at-a-time original pose loops protocol
/// @details Cutpoints must be idealized _before_ entering this procedure.
/// @details Cutpoint weights governed by weight in LoopClosureInfo objects.
/// @details allow_bb_move must be set _outside_ before entering this procedure.
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
refine_loops_simul_classic_two(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & outer_cycles = 3,
	bool const & fast_refine = false,
	std::set< Integer > const & fixed_chi_residues = std::set< Integer >()
);


/// @brief (full atom) refine loops simultaneously using constraints to anchor structure
/// @details non-moveable residues (both loop and outside loop) are anchored by coordinate constraints
/// @details function should really only be called with a sealed fold tree / fully closed structure
/// @details allow_bb_move must be set _outside_ before entering this procedure
/// @details allow_chi_move is governed internally (allow repack w/in 5.5 ang of
/// @details all loops, and with respect to fixed_chi_residues).
void
refine_loops_simul_constrained(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures_to_attempt,
	Integer const & cycles = 5,
	std::set< Integer > const & fixed_chi_residues = std::set< Integer >()
);


/// @brief ccd loop closure, obeys non-ideal bond geometry
/// @details  weights (chainbreak/overlap/etc) must be setup outside of this routine
void
ccd_loop_closure_obeying_nonideality(
	Pose & pose,
	LoopClosureInfo const & closure_info,
	Integer const & ccd_cycles = 100,
	bool const & use_greedy = false
);


/// @brief score cut in Pose linear chainbreak
Real
score_cut_in_Pose_linear(
	Pose const & structure,
	Integer const & cut,
	Integer const & overlap
);


/// @brief score cut in Pose quadratic chainbreak
Real
score_cut_in_Pose_quadratic(
	Pose const & structure,
	Integer const & cut,
	Integer const & overlap
);


/// @brief score local rama for moveable residues in LoopClosureInfo
void
score_rama_for_moveable(
	Pose const & pose,
	LoopClosureInfo const & closure
);


/// @brief score local composite rama for moveable residues in LoopClosureInfo
/// @details scores rama for a position assuming A, G, and V, and then takes
/// @details the minimum of the three scores
void
score_composite_rama_for_moveable(
	Pose const & pose,
	LoopClosureInfo const & closure
);


/// @brief find neighboring residues to a set of loops within the given distance cutoff
/// @note  default cutoff is 5.5
std::set< Integer >
find_neighbor_residues_of_loops(
	Pose const & pose,
	std::set< LoopClosureInfo > closures,
	Real const & cutoff = 5.5
);


/// @brief setup proper allow chi move with respect to loops
/// @details default distance cutoff 5.5 and no fixed chi residues
void
setup_proper_allow_chi_move(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures,
	Real const & cutoff = 5.5,
	std::set< Integer > const & fixed_chi_residues = std::set< Integer >()
);


/// @brief setup proper allow bb move for set of loops
void
setup_proper_allow_bb_move(
	Pose & pose,
	std::set< LoopClosureInfo > const & closures
);


/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values
/// @details if values don't exist in map, then regular idealization occurs
/// @note  this is useful when it's necessary to retain the existing torsions of the cut
void
idealize_cutpoint_retaining_psiphi(
	Pose & pose,
	Integer const & cut,
	std::map< Integer, Real > const & original_torsions
);


/// @brief idealize cutpoint but retain psi on the n-side and phi on the c-side of the cut
/// @brief using given values
/// @note  this is useful when it's necessary to retain the existing torsions of the cut
void
idealize_cutpoint_retaining_psiphi(
	Pose & pose,
	Integer const & cut,
	Real const & nside_psi,
	Real const & cside_phi
);


} // using namespace design
} // using namespace epigraft


#endif /*INCLUDED_epigraft_design_loop_functions_HH_*/
