// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   multigraft.cc
/// @brief  Performs closure and design for an epigraft match hit.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)

// unit headers
#include <epigraft/design/multigraft.hh>

// package headers
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/Checkpoint.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/epigraft_io.hh>
#include <epigraft/design/DesignFileExport.hh>
#include <epigraft/design/EpitopeScaffold.hh>
#include <epigraft/design/ESBundle.hh>
#include <epigraft/design/GraftInfo.hh>
#include <epigraft/design/MultiGraftStats.hh>
#include <epigraft/design/OutputFilename.hh>
#include <epigraft/design/PoseAssembly.hh>
#include <epigraft/design/ccd_functions.hh>
#include <epigraft/design/design_io.hh>
#include <epigraft/design/fragment_functions.hh>
#include <epigraft/design/loop_functions.hh>
#include <epigraft/design/multigraft_checkpoint.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/match_functions.hh>
//#include <epigraft/match/match_io.hh>
#include <epigraft/match/rescore_matches.hh>

// rosetta headers
#include <decoy_features.h>
#include <disulfides.h>
#include <files_paths.h>
#include <fragments_pose.h>
#include <InteractionGraphBase.h>
#include <jumping_loops.h>
#include <PackerTask.h>
#include <param_aa.h>
#include <pose.h>
#include <pose_io.h>
#include <random_numbers.h>
#include <RotamerSet.h>
#include <score.h>
#include <vall_data.h>
#include <vdw.h>

// utility headers
#include <utility/file/FileName.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/vector1.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>

// C++ headers
#include <ctime>
#include <fstream>
#include <iostream>
#include <limits>
#include <map>
#include <set>
#include <sstream>
#include <string>
#include <utility>

namespace epigraft {
namespace design {


/// @brief MultiGraft top-level protocol
void
multigraft(
	GraftOptions const & options
)
{
	using epigraft::match::MatchResult;
	using namespace pose_ns;

	// Rosetta setup
	disulfides::options::find_disulf = true;
	disulfides::options::norepack_disulf = true;
	score_set_vary_omega( false );
//	select_atomvdw( "highres" );
	std::cout << "STATE: find_disulf is ON\n";
	std::cout << "STATE: norepack_disulf is ON\n";
	std::cout << "STATE: score_set_vary_omega is OFF\n";
//	std::cout << "STATE: centroid level vdw is HIGHRES" << std::endl;

	// override any rosetta global options
	if ( options.override_rosetta_pdb_output_path ) {
		files_paths::pdb_out_path = options.pdb_output_path;
	}

	// initialize decoy features
	decoy_features_ns::set_df_flag( true ); // force decoy features flag here instead of depending on command line option
	decoy_features_ns::decoy_features_initialize( false ); // boolean: init_using_command_line_option
	decoy_features_ns::set_df_flag( false ); // temporarily turn off, flag must be governed by design_io.cc::output_pdb()

	if (options.test_mode) {
		test();
		return;
	}

	// load native complex from file, NOTE: assume Ab comes first!
	AntibodyComplex native( options.native_complex_filename, options.nres_Ab, options.Ab_first, true ); // booleans: Ab_first, refold_sidechains_from_chi

	// read list of epitope (loop) ranges
	// TODO: optional -- error checking to see if any of the subranges lie out of bounds relative to various criteria
	utility::vector1< LoopInfo > loops_to_scan;
	std::set< Integer > user_specified_superposition_residues;
	load_epitope_subranges( options.loop_ranges_filename, loops_to_scan, user_specified_superposition_residues, options.use_SS_align );

	// sanity check for epitope (loop) ranges and epitope input structure
	if ( !epitope_and_epitope_subranges_one_to_one( loops_to_scan, native.antigen() ) ) {
		utility::exit( __FILE__, __LINE__, "ERROR: please make sure the epitope in your antibody complex structure includes exactly the residues specified in the 'full_range'(s) of the loop ranges file" );
	}

	// load keep natro file if present
	std::set< Integer > keep_natro_residues;
	if ( options.use_keep_natro ) {
		keep_natro_residues = parse_keep_natro_file( options.keep_natro_filename );
	}

	// TODO: load list of residues for complementarity design if requested
	std::set< std::string > complementarity_residues;
	if ( options.complementarity_design ) {
		complementarity_residues = parse_complementarity_residue_file( options.complementarity_residue_file );
	}

	// load match results
	std::map< std::string, utility::vector1< GraftInfo > > pdb_to_graft_info;
	graft_info_from_file( options.input_filename, loops_to_scan, pdb_to_graft_info, options.use_old_graft_info_format );

	// scaffold filenames
	utility::vector1< std::string > scaffold_filenames;
	for ( std::map< std::string, utility::vector1< GraftInfo > >::const_iterator entry = pdb_to_graft_info.begin(), entry_e = pdb_to_graft_info.end(); entry != entry_e; ++entry ) {
		scaffold_filenames.push_back( entry->first ); // scaffold filename
	}

	// open output file
	std::string OUTPUT_FILE = files_paths::score_path + options.output_filename;
	utility::io::ozstream outfile;
	utility::io::izstream testfile( OUTPUT_FILE );
	if ( testfile.fail() ) {
		testfile.close();
		outfile.open( OUTPUT_FILE );
	} else {
		testfile.close();
		outfile.open_append( OUTPUT_FILE );
	}

	// create iterator index outside of loop due to potential checkpointing
	Size scaffold_id = 1;
	Size match_counter = 1;

	// for outer checkpointing
	Checkpoint checkpoint;
	if ( options.use_checkpoint ) {
		checkpoint.open( options.checkpoint_filename );
		checkpoint.recover_last_id_processed();

		if ( checkpoint.current_id() > scaffold_filenames.size() ) {
			// at max or somehow over count, so we're done
			utility::exit( __FILE__, __LINE__, "checkpoint indicates all files processed" );
		}

		scaffold_id = checkpoint.current_id() > 0 ? checkpoint.current_id() : 1;
		match_counter = checkpoint.current_sub_id() > 0 ? checkpoint.current_sub_id() : 1;
	}

	// inner checkpointing
	bool restarted_from_inner_checkpoint = false;
	MultiGraftStage stage = ONGOING;
	Integer checkpoint_index = 0;
	if ( options.use_checkpoint ) {
		checkpoint_peek( options.checkpoint_filename_inner, stage, checkpoint_index );
		restarted_from_inner_checkpoint = ( stage != ONGOING );
	}

	// run over each scaffold
	for ( Size last_scaffold_id = scaffold_filenames.size(); scaffold_id <= last_scaffold_id; ++scaffold_id ) {

		// find scaffold filename
		std::string const & scaffold_filename = scaffold_filenames[ scaffold_id ];

		// status
		std::cout << "MultiGraft: processing scaffold " << scaffold_filename << std::endl;

		// load pdb from either individual file or complex file
		Pose scaffold;
		if ( !options.input_pdb_has_Ab ) {
			pose_from_pdb( scaffold, files_paths::start_path + scaffold_filename, true, false, true ); // boolean: fullatom, ideal_pose, read_all_chains
		} else {
			AntibodyComplex ac( files_paths::start_path + scaffold_filename, options.nres_Ab, options.Ab_first, false );
			scaffold = ac.antigen();
		}

		// score scaffold for rudimentary stability output (wild-type scaffold vs grafted scaffold)
		Real const wt_scaffold_score = scaffold.score( Score_weight_map( score12 ) );

		// run over each match result and run closure+design schedule
		utility::vector1< GraftInfo > & graft_infos = pdb_to_graft_info[ scaffold_filename ];
		for ( Size match_counter_end = graft_infos.size(); match_counter <= match_counter_end; ++match_counter ) {

			// checkpointing: the rest of the run is slow enough that open/update/close is ok
			if ( options.use_checkpoint ) {
				checkpoint.open();
				checkpoint.set_current_id( scaffold_id );
				checkpoint.set_current_sub_id( match_counter );
				checkpoint.update();
				checkpoint.close();
			}

			// prepare output filename
			OutputFilename output_pdb_name;
			if ( options.use_batch_id ) {
				utility::file::FileName full_path( scaffold_filename ); // used to get base filename & path

				std::ostringstream oss;
				oss << full_path.path() << options.batch_id << "_" << base_pdb_filename( full_path.base() ) << ".m" << match_counter;
				output_pdb_name = OutputFilename( files_paths::pdb_out_path, oss.str() );
			} else {
				std::ostringstream oss;
				oss << base_pdb_filename( scaffold_filename ) << ".m" << match_counter;
				output_pdb_name = OutputFilename( files_paths::pdb_out_path, oss.str() );
			}

			// grab graft info for processing
			GraftInfo & graft_info = graft_infos[ match_counter ];

			// need to rescore since match results may not have transformation matrix or may want alternate transformation matrix due to input_pdb_has_Ab
			rescore_match( options, native, scaffold, graft_info.match_result(), scaffold_filename, user_specified_superposition_residues );

			// lock graft info to perform range checks
			graft_info.lock();

			// instantiate initial epitope scaffold object
			if ( !options.idealize_loop_geometry ) {
				std::cout << "! WARNING: Keeping native bond/angle geometry around loops.  This mode is primarily for testing, check your solutions otherwise." << std::endl;
				std::cout << "! Mode not yet ready for general use, exiting." << std::endl;
				std::exit( 1 );
			}
			EpitopeScaffold epitope_scaffold( scaffold, native.antigen(), graft_info, keep_natro_residues, options.micromanage_termini, options.idealize_loop_geometry );

			// options: build loop
			epitope_scaffold.set_use_fragment_insertion( options.use_fragment_insertion ); // default true
			epitope_scaffold.set_use_sequence_biased_fragments( options.use_sequence_biased_fragments ); // default false
			epitope_scaffold.set_use_variable_length_fragments( options.use_variable_length_fragments ); // default false
			epitope_scaffold.set_number_of_fragments( options.number_of_fragments ); // default 200
			epitope_scaffold.set_build_cycles( options.build_cycles ); // default: 10

			// options: refine loop
			epitope_scaffold.set_use_fast_refine( options.use_fast_refine ); // default: false
			epitope_scaffold.set_refine_cycles( options.refine_cycles ); // default: 5

			// options: closure criteria
			epitope_scaffold.set_chainbreak_criterion( options.max_chainbreak_score ); // default 0.008
			epitope_scaffold.set_local_rama_criterion( options.max_local_rama ); // default 5.0

			// options: design procedure
			epitope_scaffold.set_allow_epitope_repack( options.repack_epitope ); // default: false
			epitope_scaffold.set_allow_epitope_design( options.design_epitope ); // default: false
			epitope_scaffold.set_allow_Ab_repack( options.repack_Ab ); // default: false
			epitope_scaffold.set_allow_ALLAA_at_inter_design_positions( options.allow_AA_at_inter_design ); // default: false
			epitope_scaffold.set_inter_design_position_cutoff( options.inter_design_cutoff ); // default: 4.0
			epitope_scaffold.set_intra_design_position_cutoff( options.intra_design_cutoff ); // default: 4.0
			epitope_scaffold.set_Ab_repack_cutoff( options.Ab_repack_cutoff ); // default: 4.5

			// options: complementarity design procedure
			epitope_scaffold.set_complementarity_design_cycles( options.complementarity_design_cycles );
			epitope_scaffold.set_complementarity_design_position_cutoff( options.complementarity_design_cutoff );
			epitope_scaffold.set_complementarity_shell_cutoff( options.complementarity_shell_cutoff );
			epitope_scaffold.set_complementarity_shell_repack( options.complementarity_shell_repack );
			epitope_scaffold.set_complementarity_shell_redesign( options.complementarity_shell_redesign );

			// optimize initial epitope versus scaffold geometry?
			if ( options.epitope_rb_optimize &&
			     !( options.use_checkpoint && stage != ONGOING ) ) { // default: false
				epitope_scaffold.epitope_rb_optimize( options.epitope_rb_cycles, 100, options.epitope_rb_min ); // defaults: 1, false
			}

			// grow type?  default is ALA since 20/Mar/08
			if ( options.grow_as_GLY ) {
				epitope_scaffold.set_grow_residue_type( 'G' );
			}

			if ( options.do_restart && options.post_modify_restart ) {
				std::cout << "* reloading base epitope scaffold from user specified restart file" << std::endl;
				if ( native.Ab_exists() ) {
					reload_epitope_scaffold( options.restart_from, native.Ab(), epitope_scaffold );
				} else {
					Pose fake_ab;
					reload_epitope_scaffold( options.restart_from, fake_ab, epitope_scaffold );
				}
				std::cout << "* post-modifying restart" << std::endl;
			}

			// set and alter Pose to reflect moveable/grow/remove closure residues as well as
			// any changes from double -> single break specified in GraftInfo
			epitope_scaffold.alter_closure_sites();

			// randomize phi/psi of moveable residues except for the lever side of the primary
			// if N2C or C2N
			if ( options.randomize_moveable_phipsi ) {
				std::cout << "* randomizing broken closure site phi-psi" << std::endl;
				epitope_scaffold.randomize_broken_closure_site_phipsi( false ); // boolean: also_randomize_lever
			}

			// reload if restarting from either checkpoint or user provided coordinates
			if ( options.use_checkpoint && stage != ONGOING ) {
				std::cout << "* reloading base epitope scaffold from checkpoint" << std::endl;
				if ( native.Ab_exists() ) {
					checkpoint_load_base( options.checkpoint_filename_inner, native.Ab(), epitope_scaffold );
				} else {
					Pose fake_ab;
					checkpoint_load_base( options.checkpoint_filename_inner, fake_ab, epitope_scaffold );
				}
			} else if ( options.do_restart && !options.post_modify_restart ) {
				std::cout << "* reloading base epitope scaffold from user specified restart file" << std::endl;
				if ( native.Ab_exists() ) {
					reload_epitope_scaffold( options.restart_from, native.Ab(), epitope_scaffold );
				} else {
					Pose fake_ab;
					reload_epitope_scaffold( options.restart_from, fake_ab, epitope_scaffold );
				}
			}

			// set antibody
			if ( native.Ab_exists() ) {
				epitope_scaffold.set_Ab( native.Ab() ); // antibody reoriented, but not connected
			}

			// status
			// TODO: consider removing this status output
			std::cout << "MultiGraft: potential primary closures:" << std::endl;
			std::set< LoopClosureInfo > primary_closures = epitope_scaffold.primary_closures_to_attempt();
			for ( std::set< LoopClosureInfo >::const_iterator a = primary_closures.begin(), ae = primary_closures.end(); a != ae; ++a ) {
				std::cout << "   " << a->to_string() << std::endl;
			}
			std::cout << "MultiGraft: potential secondary closures:" << std::endl;
			std::set< LoopClosureInfo > secondary_closures = epitope_scaffold.secondary_closures_to_attempt();
			for ( std::set< LoopClosureInfo >::const_iterator a = secondary_closures.begin(), ae = secondary_closures.end(); a != ae; ++a ) {
				std::cout << "   " << a->to_string() << std::endl;
			}

			// first convert es to gly or ala
			epitope_scaffold.convert_to_residue_type( options.closure_residue_type ); // ala by default, unless -close_as_GLY

			// rb optimize Ab?  typically done in superposition scaffolds
			if ( ( options.Ab_epitope_optimize || options.Ab_epitope_optimize_including_rb ) &&
			    !( options.use_checkpoint && stage != ONGOING ) ) { // default: false
				// first recover native sidechains of epitope
				epitope_scaffold.recover_native_epitope_sidechains();

				// rb optimize Ab
				epitope_scaffold.connect_Ab();
				epitope_scaffold.Ab_epitope_optimize( true, options.Ab_epitope_optimize_including_rb ); // boolean: chi_minimize
				epitope_scaffold.repack_interface();
				epitope_scaffold.Ab_epitope_optimize( true, options.Ab_epitope_optimize_including_rb ); // boolean: chi_minimize
				epitope_scaffold.archive_identity_equivalent_scaffold_sidechains(); // save state

				// disconnect
				epitope_scaffold.disconnect_Ab();

				// now must re-convert to gly/ala (flip epitope back)
				epitope_scaffold.convert_to_residue_type( options.closure_residue_type ); // ala by default, unless -close_as_GLY
			} else if ( options.epitope_optimize && !( options.use_checkpoint && stage != ONGOING ) ) {
				// first recover native sidechains of epitope
				epitope_scaffold.recover_native_epitope_sidechains();

				// run optimization
				epitope_scaffold.epitope_optimize();

				// archive epitope state
				epitope_scaffold.archive_identity_equivalent_scaffold_sidechains(); // save state

				// now must re-convert to gly/ala (flip epitope back)
				epitope_scaffold.convert_to_residue_type( options.closure_residue_type ); // ala by default, unless -close_as_GLY
			}

			// do closure schedule with Ab attached?
			if ( options.graft_with_Ab ) { // default: false
				epitope_scaffold.connect_Ab();
			}

			// find initial complementarity design positions, this needs to occur after closure
			// site alteration
			if ( options.complementarity_design ) {
				epitope_scaffold.mark_complementarity_Ab_positions( complementarity_residues );
			}

			// checkpoint if necessary
			if ( options.use_checkpoint && stage == ONGOING ) {
				std::cout << "* saving base epitope scaffold to checkpoint" << std::endl;
				EpitopeScaffold base_es( epitope_scaffold );
				base_es.connect_Ab();
				base_es.recover_all_native_sidechains();
				checkpoint_save_base( options.checkpoint_filename_inner, base_es );
			}

			// prepare containers for closed epitope-scaffolds, best designs, etc
			// Poses kept here do not need to be deleted because they will be
			// assigned to ESBundle structs and manually deleted later on
			std::multimap< Real, ESBundle > closed_epitope_scaffolds; // score -> ESBundle containing epitope scaffolds
			std::multimap< Real, ESBundle > designed_epitope_scaffolds; // score -> ESBundle containing epitope scaffolds

			// reload saved history from checkpoint if necessary
			if ( options.use_checkpoint && stage != ONGOING ) {
				if ( native.Ab_exists() ) {
					checkpoint_load( options.checkpoint_filename_inner, native.Ab(), epitope_scaffold, stage, checkpoint_index, closed_epitope_scaffolds, designed_epitope_scaffolds );
				} else {
					Pose fake_ab;
					checkpoint_load( options.checkpoint_filename_inner, fake_ab, epitope_scaffold, stage, checkpoint_index, closed_epitope_scaffolds, designed_epitope_scaffolds );
				}
			}

			// prepare statistics for this scaffold + match result
			MultiGraftStats statistics;

			// TESTING
//			epigraft::design::screen_fragments( epitope_scaffold.pose(), epitope_scaffold.closures_to_attempt(), epitope_scaffold.loop_closure_ss_string(), epitope_scaffold.loop_closure_seq_string() );
//			epigraft::design::read_fragment_index( "screened.txt", epitope_scaffold.pose() );
//			std::exit( 0 );

			// closure attempts
			if ( options.build_loops || options.refine_loops ) {
				closure_stage( options, output_pdb_name, epitope_scaffold, closed_epitope_scaffolds, statistics, stage, checkpoint_index );
			} else if ( stage == ONGOING ){
				// assumes restarting without build/refine, add current es
				Real const fullatom_score = epitope_scaffold.fullatom_score_no_breaks();
				closed_epitope_scaffolds.insert( std::make_pair( fullatom_score, ESBundle( 1, -1, new EpitopeScaffold( epitope_scaffold ) ) ) ); // -1 since no design id yet
			}

			// TODO: complementarity design
			if ( options.complementarity_design ) {
				complementarity_design_stage( options, closed_epitope_scaffolds, stage, checkpoint_index );
			}

			// run through design
			if ( options.design_after_closure && closed_epitope_scaffolds.size() > 0 ) { // default: false
				design_stage( options, closed_epitope_scaffolds, designed_epitope_scaffolds, stage, checkpoint_index );
			}

			// refine loop with Ab in place after closure -- note that closure state always ignored
			if ( options.refine_with_Ab_after_design ) {
				refine_stage( options, designed_epitope_scaffolds, statistics, stage, checkpoint_index );
			}

			// quality control -- refine design without antibody and see what happens
			if ( options.design_after_closure && options.do_QC && designed_epitope_scaffolds.size() > 0 ) {
				quality_control_stage( options, designed_epitope_scaffolds, statistics, stage, checkpoint_index );
			}

			// overall statistics output
			outfile << "##### " << scaffold_filename << "  match  " << match_counter << '\n';
			outfile << graft_info.match_result().to_string( scaffold_filename, "", "#####" );
			outfile << graft_info.to_string( scaffold_filename, "####", options.use_old_graft_info_format );
			if ( options.use_checkpoint && restarted_from_inner_checkpoint ) {
				outfile << "# NOTE: This run was checkpointed, so the overall statistics gathering was interrupted and not complete!" << std::endl;
			}
			outfile << statistics.to_string();
			restarted_from_inner_checkpoint = false; // reset

			// per design output
			if ( options.design_after_closure ) { // gone to design/refine stage

				// run through designs
				for ( std::multimap< Real, ESBundle >::const_iterator esi = designed_epitope_scaffolds.begin(), esi_e = designed_epitope_scaffolds.end(); esi != esi_e; ++esi ) {
					ESBundle const & esb = esi->second;

					recursively_create_directory_for_filename( output_pdb_name.name_from_ESBundle( esb ) );
					utility::io::ozstream individual_log( output_pdb_name.name_from_ESBundle( esb ) + ".multigraft.log" );
					individual_log << "##### " << scaffold_filename << "  match  " << match_counter << '\n';
					individual_log << graft_info.match_result().to_string( scaffold_filename, "", "#####" );
					individual_log << graft_info.to_string( scaffold_filename, "####", options.use_old_graft_info_format );

					// spacer
					individual_log << "#\n";

					// design output name
					individual_log << "### " << output_pdb_name.name_from_ESBundle( esb, true) << "\n"; // boolean: just base filename

					if ( options.Ab_epitope_optimize ) {
						individual_log << "# NOTICE -- Ab-epitope interaction versus epitope-scaffold was optimized before closure";
						if ( options.Ab_epitope_optimize_including_rb ) {
							individual_log << ", including rb" << "\n";
						} else {
							individual_log << '\n';
						}
					}

					if ( options.graft_with_Ab ) {
						individual_log << "# NOTICE -- closure was performed with Ab" << "\n";
					}

					// query refine w/Ab after design
					if ( options.refine_with_Ab_after_design ) {
						if ( esb.refined_es->all_loops_closed() ) {
							individual_log << "# NOTICE -- this design succeeded refinement with Ab after design.  QC will be of REFINED designed structure." << "\n";
						} else {
							individual_log << "# WARNING -- this design failed refinement with Ab after design!  QC will be of ORIGINAL designed structure." << "\n";
						}
					}

					// status and scores
					individual_log << esb.closure_and_rama_string();
					individual_log << "#\n";
					individual_log << "# * Design Positions\n";
					individual_log << "#\n";
					individual_log << esb.designed_es->design_positions_to_string();
					individual_log << "#\n";
					individual_log << esb.score_string();

					// rudimentary stability output
					individual_log << "#\n";
					individual_log << "# * Stability Scores\n";
					individual_log << "#\n";
					individual_log << "# Score12_WT_scaffold = " << wt_scaffold_score << "\n";

					individual_log << "#\n";

					// qc output
					if ( options.do_QC ) {
						individual_log << "# * QC Scores\n";
					}

//					if ( !options.graft_with_Ab && options.do_QC ) {
//						EpitopeScaffold scratch_es = *esb.closed_es;
//						scratch_es.connect_Ab();
//						Real const scratch_ddG = scratch_es.ddG();
//						scratch_es.repack_interface();
//						Real const scratch_ddG_after_repack_interface = scratch_es.ddG();
//
//						individual_log << "# ddG after closure / after repack interface = " << scratch_ddG << " / " << scratch_ddG_after_repack_interface << "\n";
//					}

					if ( options.do_QC ) {
						EpitopeScaffold scratch_es = *esb.qc_es;
						scratch_es.connect_Ab();
						Real const scratch_ddG = scratch_es.ddG();
						scratch_es.repack_interface();
						Real const scratch_ddG_after_repack_interface = scratch_es.ddG();

						individual_log << "# final QC ddG / after repack interface = " << scratch_ddG << " / " << scratch_ddG_after_repack_interface << "\n";
					}

					// close individual log
					individual_log.close();

					// output design pdb (already has Ab)
					// grab epitope-scaffold and complex for output
					Pose output_complex;
					esb.designed_es->extract_complex( output_complex );
					output_pdb( output_pdb_name.name_from_ESBundle( esb ) + ".design.pdb.gz", output_complex, true ); // boolean: include_decoy_features

					// output finalized (currently just refined w/Ab) pdb
					if ( options.refine_with_Ab_after_design && esb.refined_es != NULL ) {
						Pose scratch_complex;
						esb.refined_es->extract_complex( scratch_complex );
						output_pdb( output_pdb_name.name_from_ESBundle( esb ) + ".refine_with_Ab_after_design.pdb.gz", scratch_complex, true ); // boolean: include_decoy_features
					}

					// output qc pdb
					if ( esb.qc_es != NULL ) {
						Pose scratch_complex;
						esb.qc_es->connect_Ab();
						esb.qc_es->extract_complex( scratch_complex );
						output_pdb( output_pdb_name.name_from_ESBundle( esb ) + ".qc.pdb.gz", scratch_complex, true ); // boolean: include_decoy_features
					}

					// export design files
					if ( options.export_blueprint ) {
						esb.designed_es->export_blueprint( output_pdb_name.name_from_ESBundle( esb ) + ".blueprint" );
					}
					if ( options.export_resfile ) {
						esb.designed_es->export_resfile( output_pdb_name.name_from_ESBundle( esb ) + ".resfile" );
					}

					std::cout << std::flush;
				}

			} else if ( options.build_loops || options.refine_loops ) { // only closure stage

				for ( std::multimap< Real, ESBundle >::const_iterator esi = closed_epitope_scaffolds.begin(), esi_e = closed_epitope_scaffolds.end(); esi != esi_e; ++esi ) {
					ESBundle const & esb = esi->second;

					utility::io::ozstream individual_log( output_pdb_name.name_from_ESBundle( esb ) + ".multigraft.log" );
					individual_log << "##### " << scaffold_filename << "  match  " << match_counter << '\n';
					individual_log << graft_info.match_result().to_string( scaffold_filename, "", "#####" );
					individual_log << graft_info.to_string( scaffold_filename, "####", options.use_old_graft_info_format );

					// design output name
					individual_log << "### " << output_pdb_name.name_from_ESBundle( esb, true ) << "\n"; // boolean: just base name

					if ( options.Ab_epitope_optimize ) {
						individual_log << "# NOTICE -- Ab-epitope interaction versus epitope-scaffold was optimized before closure";
					}
					if ( options.Ab_epitope_optimize_including_rb ) {
						individual_log << ", including rb" << "\n";
					} else {
						individual_log << "\n";
					}

					if ( options.graft_with_Ab ) {
						individual_log << "# NOTICE -- closure was performed with Ab" << "\n";
					}

					// status
					individual_log << "#\n";
					individual_log << esb.closed_es->to_string();

					// scores
					individual_log << "#" << std::endl;
					if ( esb.closed_es->Ab_is_connected() ) {
						if ( options.refine_loops ) {
							individual_log << "# Score12_with_Ab = " << esb.closed_es->fullatom_score_no_breaks() << "\n";
							individual_log << "# Score12 = " << esb.closed_es->scaffold_fullatom_score_no_breaks() << "\n";
						} else if ( options.build_loops ) {
							individual_log << "# CentroidScore_with_Ab = " << esb.closed_es->centroid_score_no_breaks() << "\n";
							individual_log << "# CentroidScore = " << esb.closed_es->scaffold_centroid_score_no_breaks() << "\n";
						}
						individual_log << "# Score12_ddG = " << esb.closed_es->ddG() << "\n";
					} else {
						if ( options.refine_loops ) {
							individual_log << "# Score12 = " << esb.closed_es->fullatom_score_no_breaks() << "\n";
						} else if ( options.build_loops ) {
							individual_log << "# CentroidScore = " << esb.closed_es->centroid_score_no_breaks() << "\n";
						}
					}

					// close individual log
					individual_log.close();

					// output pdb
					Pose output_complex;
					esb.closed_es->extract_complex( output_complex );
					output_pdb( output_pdb_name.name_from_ESBundle( esb ) + ".pdb.gz", output_complex, true ); // boolean: include_decoy_features

					std::cout << std::flush;
				}
			}

			// dump predesign if requested
			if ( options.dump_predesign ) {
				std::cout << "MultiGraft: dumping predesign" << std::endl;
				epitope_scaffold.connect_Ab();
				epitope_scaffold.recover_all_native_sidechains();
				output_pdb( options.predesign_filename, epitope_scaffold.pose() );
				epitope_scaffold.disconnect_Ab();
			}

			// destroy all bundles and reclaim memory
			// we DO NOT use the .destroy_contents() function in ESBundle
			// because there are multiple overlapping pointers, so instead we
			// collect everything and then delete
			std::set< EpitopeScaffold * > es_to_delete;
			for ( std::multimap< Real, ESBundle >::iterator i = designed_epitope_scaffolds.begin(), ie = designed_epitope_scaffolds.end(); i != ie; ++i ) {
				ESBundle & esb = i->second;
				es_to_delete.insert( esb.closed_es );
				es_to_delete.insert( esb.designed_es );
				es_to_delete.insert( esb.refined_es );
				es_to_delete.insert( esb.qc_es );
			}
			for ( std::multimap< Real, ESBundle >::iterator i = closed_epitope_scaffolds.begin(), ie = closed_epitope_scaffolds.end(); i != ie; ++i ) {
				ESBundle & esb = i->second;
				es_to_delete.insert( esb.closed_es );
				es_to_delete.insert( esb.designed_es );
				es_to_delete.insert( esb.refined_es );
				es_to_delete.insert( esb.qc_es );
			}

			// remove NULL pointer
			es_to_delete.erase( NULL );

			// now delete
			for ( std::set< EpitopeScaffold * >::iterator i = es_to_delete.begin(), ie = es_to_delete.end(); i != ie; ++i ) {
				delete *i;
			}

			// safety
			designed_epitope_scaffolds.clear();
			closed_epitope_scaffolds.clear();

			// clear checkpoint file
			checkpoint_touch( options.checkpoint_filename_inner );

		} // foreach match result

		// need to reset match counter
		match_counter = 1;

	} // foreach scaffold

	// close output file
	outfile.close();

	// checkpointing: the rest of the run is slow enough that open/update/close is ok
	if ( options.use_checkpoint ) {
		checkpoint.open();
		checkpoint.set_current_id( scaffold_filenames.size() + 1 );
		checkpoint.set_current_sub_id( 0 );
		checkpoint.update();
		checkpoint.close();
	}

	std::cout << "MultiGraft: finished" << std::endl;

}


/// @brief top-level closure
void
closure_stage(
	GraftOptions const & options,
	OutputFilename const & output_pdb_name,
	EpitopeScaffold const & original_es,
	std::multimap< Real, ESBundle > & closed_epitope_scaffolds,
	MultiGraftStats & statistics,
	MultiGraftStage & stage,
	Integer & checkpoint_index
)
{
	if ( stage != ONGOING && stage != CLOSURE ) {
		return;
	}
	stage = CLOSURE;

	time_t time_start = time( NULL );
	time_t time_current = time_start, time_elapsed = 0;

	std::cout << "MultiGraft: starting closure attempts" << std::endl;

	EpitopeScaffold::BuildType build_type = EpitopeScaffold::NORMAL_BUILD;
	if ( options.build_with_arm ) {
		build_type = EpitopeScaffold::ARM_BUILD;
	} else if ( options.build_with_adaptive ) {
		build_type = EpitopeScaffold::ADAPTIVE_BUILD;
	} else if ( options.build_with_screen ) {
		build_type = EpitopeScaffold::SCREEN_BUILD;
	} else if ( options.build_with_fragments_only ) {
		switch ( options.test_ccd_type ) {
			case 3:
				build_type = EpitopeScaffold::FRAGMENT_FAST_CCD_BUILD;
				break;
			case 2:
				build_type = EpitopeScaffold::FRAGMENT_TWO_TORSION_CCD_BUILD;
				break;
			case 1:
				build_type = EpitopeScaffold::FRAGMENT_ONE_TORSION_CCD_BUILD;
				break;
			case 0:
				build_type = EpitopeScaffold::FRAGMENT_NO_CCD_BUILD;
				break;
			default:
				std::cerr << "WARNING: UNKNOWN CCD TYPE, using two torsion ccd" << std::endl;
				build_type = EpitopeScaffold::FRAGMENT_TWO_TORSION_CCD_BUILD;
				break;
		}
	}

	EpitopeScaffold::RefineType refine_type = EpitopeScaffold::NORMAL_REFINE;
	if ( options.refine_with_minrepack ) {
		refine_type = EpitopeScaffold::MIN_REPACK_REFINE;
	} else if ( options.refine_with_classic ) {
		refine_type = EpitopeScaffold::CLASSIC_REFINE;
	} else if ( options.refine_with_constraints ) {
		refine_type = EpitopeScaffold::CONSTRAINED_REFINE;
	}

	EpitopeScaffold work_es( original_es );

	Integer i = checkpoint_index ? checkpoint_index : 1;
	for ( ; i <= options.closure_attempts; ++i ) {

		time_current = time( NULL );
		time_elapsed = time_current - time_start;
		if ( options.use_checkpoint && ( time_elapsed >= options.checkpoint_interval_seconds ) ) {
			checkpoint_save( options.checkpoint_filename_inner, stage, i, closed_epitope_scaffolds, std::multimap< Real, ESBundle >() );

			time_start = time_current; // reset time
			time_elapsed = 0;
		}

		std::cout << "MultiGraft: closure attempt " << i << std::endl;

		// build loops, epitope scaffold is fullatom up until the switch to centroid
		// call, at which point it loses all full_coord data
		if ( options.build_loops ) { // default: false

			// If using 'D', to grow then reset ss, otherwise memory of the first
			// types of random fragments are retained.
			if ( work_es.grow_ss_type() == 'D' && !options.scan_randomize_ss ) {
				work_es.reset_grown_residue_ss();
			}

			// reset state of broken loops every so often to get away from trapped minima
			if ( ( i - 1 ) % options.closure_reset_period == 0 ) {
				std::cout << "* resetting backbone of broken loops\n";

				work_es.reset_broken_closure_site_backbone();

				// do scan if requested
				if ( options.use_scan_behavior ) {
					if ( options.scan_randomize_moveable ) {
						std::cout << "* scan randomizing moveable residues of broken loops\n";
						work_es.randomize_broken_closure_site_moveable_residues();
					} else if ( options.scan_randomize_cutpoints ) {
						work_es.randomize_broken_closure_site_cutpoints();
					}

					if ( options.scan_randomize_ss ) {
						std::cout << "* scan randomizing secondary structure of broken loops\n";
						work_es.randomize_broken_closure_site_moveable_residue_secondary_structure( options.scan_ss_minimum_content, options.scan_ss_type );
					}
				}

				std::cout << std::flush;
			}

			// if user requests completely degenerate fragment insertion
			if ( options.allow_any_ss_during_fragment_insertion ) {
				work_es.alter_secondary_structure_of_moveable_residues( 'D' );
			}

			// rb perturb non-closed secondary components (governed by graft info)
			if ( work_es.graft_info().perturbing_any_secondary_component_rb() ) {
				std::cout << "* performing rb perturbation of epitope secondaries" << std::endl;

				// need to force reconnect Ab on primary component
				if ( work_es.Ab_is_connected() ) {
					work_es.disconnect_Ab( false ); // no auto-archive necessary here
					work_es.connect_Ab( true ); // boolean: only_connect_to_primary
				}

				work_es.gaussian_perturb_epitope_secondary_components( options.translation_perturb_magnitude, options.rotation_perturb_magnitude );

				// reset connection to default closest between epitope and Ab
				if ( work_es.Ab_is_connected() ) {
					work_es.disconnect_Ab( false ); // no auto-archive necessary here
					work_es.connect_Ab();
				}
			}

			work_es.switch_to_centroid();
			work_es.build_all_simul( true, build_type ); // boolean: ignore closure state

			// record build attempt
			statistics.record_closure_attempt( "Build", work_es );

			// dump pdb if requested
			if ( work_es.all_loops_closed() ) {
				if ( options.dump_closed || options.dump_all_closure_attempt_structures ) {
					work_es.recover_all_native_sidechains(); //bc crap to output pdbs with full atom
					output_pdb( output_pdb_name.name_with_closure( i ) + ".closure_build.pdb.gz", work_es.pose() );
					work_es.convert_to_residue_type( options.closure_residue_type );
				}
			} else if ( work_es.some_loops_closed() ) {
				if ( options.dump_someclosed || options.dump_all_closure_attempt_structures ) {
					work_es.recover_all_native_sidechains(); //bc crap to output pdbs with full atom
					output_pdb( output_pdb_name.name_with_closure( i ) + ".partial_closure_build.pdb.gz", work_es.pose() );
					work_es.convert_to_residue_type( options.closure_residue_type );
				}
			} else {
				if ( options.dump_all_closure_attempt_structures ) {
					work_es.recover_all_native_sidechains(); //bc crap to output pdbs with full atom
					output_pdb( output_pdb_name.name_with_closure( i ) + ".attempted_closure_build.pdb.gz", work_es.pose() );
					work_es.convert_to_residue_type( options.closure_residue_type );
				}
			}
		}

		// detach Ab first if connected, as we don't want to waste time repacking everything when
		// switching to fullatom
		bool const ab_was_connected = work_es.Ab_is_connected();
		if ( ab_was_connected ) {
			work_es.disconnect_Ab();
		}

		// now switch to fullatom (note that this repacks since fullatom data is invalidated,
		// but since all-gly or all-ala at this point there's no problem)
		work_es.switch_to_fullatom();

		// revive native epitope sidechains
		work_es.recover_native_epitope_sidechains();

		// need to revive Ab with sidechains (note that sidechains get wiped out during centroid mode)
		if ( ab_was_connected ) {
			work_es.connect_Ab();
		}

		// refine loops -- note that closure state always ignored
		if ( options.force_refine || ( options.refine_loops && work_es.all_loops_closed() ) ) { // default: false

			if ( refine_type != EpitopeScaffold::CONSTRAINED_REFINE ) {
				work_es.clear_closed_during_trajectory();
			}
			work_es.refine_all_simul( true, refine_type ); // boolean: ignore_closure_state, use_minrepack_procedure

			// record refine attempt
			statistics.record_closure_attempt( "Refine", work_es );

			// save state, since sidechains may have moved during refine loops
			work_es.archive_identity_equivalent_scaffold_sidechains();
			work_es.archive_Ab_sidechains();

			// dump pdb if requested
			if ( work_es.all_loops_closed() ) {
				if ( options.dump_closed || options.dump_refined || options.dump_all_closure_attempt_structures ) {
					work_es.recover_all_native_sidechains(); //bc crap to output pdbs with full atom
					output_pdb( output_pdb_name.name_with_closure( i ) + ".closure_refine.pdb.gz", work_es.pose() );
				}
			} else if ( work_es.some_loops_closed() ) {
				if ( options.dump_someclosed || options.dump_all_closure_attempt_structures ) {
					work_es.recover_all_native_sidechains(); //bc crap to output pdbs with full atom
					output_pdb( output_pdb_name.name_with_closure( i ) + ".partial_closure_refine.pdb.gz", work_es.pose() );
				}
			} else {
				if ( options.dump_all_closure_attempt_structures ) {
					work_es.recover_all_native_sidechains(); //bc crap to output pdbs with full atom
					output_pdb( output_pdb_name.name_with_closure( i ) + ".attempted_closure_refine.pdb.gz", work_es.pose() );
				}
			}

		}

		// re-convert back to gly or hybrid ala-gly
		work_es.convert_to_residue_type( options.closure_residue_type );

		// handle closure states, use partially closed epitope-scaffold or reset
		if ( work_es.all_loops_closed() ) { // if all loops closed, store in list

			// store by fullatom score
			Real const fullatom_score = work_es.fullatom_score_no_breaks();
			closed_epitope_scaffolds.insert( std::make_pair( fullatom_score, ESBundle( i, -1, new EpitopeScaffold( work_es ) ) ) ); // -1 since no design_id yet

			// now reset to the the original epitope scaffold
			work_es = original_es;

//				} else if ( work_es.some_loops_closed() ) { // if at least one loop closed, this as the new starting point
		}

		// only store a certain number of best closures based on total fullatom score
		if ( closed_epitope_scaffolds.size() > (std::size_t)options.store_n_best_closures ) {
			// kick out the last one
			closed_epitope_scaffolds.rbegin()->second.destroy_contents();
			closed_epitope_scaffolds.erase( --closed_epitope_scaffolds.end() );
		}

	}

	// report status on closure attempts
	std::cout << "MultiGraft: build closures " << statistics.grouped_status( "Build" ) << std::endl;
	std::cout << "MultiGraft: refine closures " <<  statistics.grouped_status( "Refine" ) << std::endl;

	stage = ONGOING;
	checkpoint_index = 0;
}


/// @brief   top-level complementarity design
/// @details currently creates one complementarity design per closed scaffold
void
complementarity_design_stage(
	GraftOptions const & options,
	std::multimap< Real, ESBundle > & closed_epitope_scaffolds,
	MultiGraftStage & stage,
	Integer & checkpoint_index
)
{
	if ( stage != ONGOING && stage != COMPLEMENTARITY_DESIGN ) {
		return;
	}
	stage = COMPLEMENTARITY_DESIGN;

	std::cout << "MultiGraft: starting complementarity design" << std::endl;

	time_t time_start = time( NULL );
	time_t time_current = time_start, time_elapsed = 0;

	// run through each closed epitope-scaffold, try designs
	std::multimap< Real, ESBundle >::iterator esi = closed_epitope_scaffolds.begin();
	for ( Integer i = 0; i < checkpoint_index; ++i ) {
		++esi;
	}

	Integer finished = checkpoint_index;
	for ( std::multimap< Real, ESBundle >::iterator esi_e = closed_epitope_scaffolds.end(); esi != esi_e; ++esi ) {
		ESBundle & esb = esi->second;

		time_current = time( NULL );
		time_elapsed = time_current - time_start;
		if ( options.use_checkpoint && ( time_elapsed >= options.checkpoint_interval_seconds ) ) {
			checkpoint_save( options.checkpoint_filename_inner, stage, finished, closed_epitope_scaffolds, std::multimap< Real, ESBundle >() );

			time_start = time_current; // reset time
			time_elapsed = 0;
		}

		std::multimap< Real, EpitopeScaffold > ddG_to_es;
		for ( Integer attempt = 1; attempt <= options.complementarity_design_attempts; ++attempt ) {
			EpitopeScaffold work_es( *esb.closed_es );
			work_es.connect_Ab();
			work_es.complementarity_design( options.complementarity_rb );
			work_es.archive_all_complementarity_design_sidechains();
			ddG_to_es.insert( std::make_pair( work_es.ddG(), work_es ) );
		}

		// grab epitope scaffold with best ddG
		*esb.closed_es = ddG_to_es.begin()->second;

		++finished;
	}

	stage = ONGOING;
	checkpoint_index = 0;
}


/// @brief top-level design
void
design_stage(
	GraftOptions const & options,
	std::multimap< Real, ESBundle > & closed_epitope_scaffolds,
	std::multimap< Real, ESBundle > & designed_epitope_scaffolds,
	MultiGraftStage & stage,
	Integer & checkpoint_index
)
{
	if ( stage != ONGOING && stage != DESIGN ) {
		return;
	}
	stage = DESIGN;

	time_t time_start = time( NULL );
	time_t time_current = time_start, time_elapsed = 0;

	std::cout << "MultiGraft: starting design" << std::endl;

	// run through each closed epitope-scaffold, try designs
	std::multimap< Real, ESBundle >::iterator esi = closed_epitope_scaffolds.begin();
	for ( Integer i = 0; i < checkpoint_index; ++i ) {
		++esi;
	}

	std::pair< Real, ESBundle > top_design_per_backbone; // used only when tracking top design
	top_design_per_backbone.first = std::numeric_limits< Real >::infinity();

	Integer finished = checkpoint_index;
	for ( std::multimap< Real, ESBundle >::iterator esi_e = closed_epitope_scaffolds.end(); esi != esi_e; ++esi ) {
		ESBundle & esb = esi->second;

		// cached design objects
		PackerTask * task = NULL;
		pack::InteractionGraphBase * ig = NULL;
		RotamerSet * rs = NULL;
		FArray1D_int * current_rot_index = NULL;
		FArray1D_float * ligenergy1b_old = NULL;

		// design iterations
		for ( Integer i = 1; i <= options.design_attempts; ++i ) {

			time_current = time( NULL );
			time_elapsed = time_current - time_start;
			if ( options.use_checkpoint && ( time_elapsed >= options.checkpoint_interval_seconds ) ) {
				checkpoint_save( options.checkpoint_filename_inner, stage, finished, closed_epitope_scaffolds, designed_epitope_scaffolds );

				time_start = time_current; // reset time
				time_elapsed = 0;
			}

			// temporary work
			EpitopeScaffold work_es = *esb.closed_es;

			// recover native epitope and scaffold sidechains
			work_es.recover_native_epitope_scaffold_sidechains();

			// design -- antibody is attached after this!
			work_es.connect_Ab();

			// cached design objects
			if ( i == 1 ) {
				task = new PackerTask( work_es.pose() );
				rs = new RotamerSet();
				current_rot_index = new FArray1D_int( work_es.pose().total_residue() );
				ligenergy1b_old = new FArray1D_float();

				FArray2D< bool > design_info = work_es.make_design_info();
				work_es.design( design_info, task, ig, rs, current_rot_index, ligenergy1b_old );
			} else {
				work_es.make_design_info(); // artificially invoke
				work_es.design( task, ig, rs, current_rot_index, ligenergy1b_old );
			}

//			work_es.design();

			// recheck closure criteria (sidechain identity change will change
			// rama score)
			work_es.recheck_closure_criteria();

			// store design by fullatom score
			Real const fullatom_score = work_es.fullatom_score_no_breaks();

			if ( options.store_top_design_per_backbone ) {
				// here we accumulate to get the top design for this backbone
				// which is then stored in 'designed_epitope_scaffolds'
				if ( top_design_per_backbone.first > fullatom_score ) {
					top_design_per_backbone = std::make_pair( fullatom_score, ESBundle( esb.closure_id, i, esb.closed_es, new EpitopeScaffold( work_es ) ) );
				}

			} else { // here we track all designs regardless of backbone
				designed_epitope_scaffolds.insert( std::make_pair( fullatom_score, ESBundle( esb.closure_id, i, esb.closed_es, new EpitopeScaffold( work_es ) ) ) );

				// only store a certain number of best designs based on total fullatom score
				if ( designed_epitope_scaffolds.size() > (std::size_t)options.store_n_best_designs ) {
					// delete only the designed_es in the bundle, nothing else
					delete designed_epitope_scaffolds.rbegin()->second.designed_es;

					// remove from best list
					designed_epitope_scaffolds.erase( --designed_epitope_scaffolds.end() );
				}
			}

		} // design after closure

		// if only tracking top design per backbone, insert it into storage
		if ( options.store_top_design_per_backbone && top_design_per_backbone.first != std::numeric_limits< Real >::infinity() ) {
			designed_epitope_scaffolds.insert( top_design_per_backbone );

			// only store a certain number of best designs based on total fullatom score
			if ( designed_epitope_scaffolds.size() > (std::size_t)options.store_n_best_designs ) {
				// delete only the designed_es in the bundle, nothing else
				delete designed_epitope_scaffolds.rbegin()->second.designed_es;

				// remove from best list
				designed_epitope_scaffolds.erase( --designed_epitope_scaffolds.end() );
			}
		}

		if ( task != NULL ) {
			delete task;
			delete ig;
			delete rs;
			delete current_rot_index;
			delete ligenergy1b_old;
		}

		++finished;
	}

	stage = ONGOING;
	checkpoint_index = 0;
}


/// @brief top-level refine
void
refine_stage(
	GraftOptions const & options,
	std::multimap< Real, ESBundle > & designed_epitope_scaffolds,
	MultiGraftStats & statistics,
	MultiGraftStage & stage,
	Integer & checkpoint_index
)
{
	if ( stage != ONGOING && stage != REFINE ) {
		return;
	}
	stage = REFINE;

	time_t time_start = time( NULL );
	time_t time_current = time_start, time_elapsed = 0;

	EpitopeScaffold::RefineType refine_type = EpitopeScaffold::NORMAL_REFINE;
	if ( options.refine_with_minrepack ) {
		refine_type = EpitopeScaffold::MIN_REPACK_REFINE;
	} else if ( options.refine_with_classic ) {
		refine_type = EpitopeScaffold::CLASSIC_REFINE;
	} else if ( options.refine_with_constraints ) {
		refine_type = EpitopeScaffold::CONSTRAINED_REFINE;
	}

	std::multimap< Real, ESBundle >::iterator esi = designed_epitope_scaffolds.begin();
	for ( Integer i = 0; i < checkpoint_index; ++i ) {
		++esi;
	}

	std::cout << "MultiGraft: starting refine with Ab after design" << std::endl;
	Integer finished = checkpoint_index;
	for ( std::multimap< Real, ESBundle >::iterator esi_e = designed_epitope_scaffolds.end(); esi != esi_e; ++esi ) {

		time_current = time( NULL );
		time_elapsed = time_current - time_start;
		if ( options.use_checkpoint && ( time_elapsed >= options.checkpoint_interval_seconds ) ) {
			checkpoint_save( options.checkpoint_filename_inner, stage, finished, std::multimap< Real, ESBundle >(), designed_epitope_scaffolds );

			time_start = time_current; // reset time
			time_elapsed = 0;
		}

		ESBundle & esb = esi->second;

		// temporary work es
		EpitopeScaffold work_es = *esb.designed_es;

		// refinement _with_ antibody
		if ( refine_type != EpitopeScaffold::CONSTRAINED_REFINE ) {
			work_es.clear_closed_during_trajectory();
		}
		work_es.refine_all_simul( true, refine_type ); // boolean: ignore_closure_state, use_minrepack_procedure

		// store
		esb.refined_es = new EpitopeScaffold( work_es );

		// record refine attempt
		statistics.record_closure_attempt( "Refine with Ab after design", work_es );

		++finished;
	}

	stage = ONGOING;
	checkpoint_index = 0;
}


/// @brief top-level quality control
void
quality_control_stage(
	GraftOptions const & options,
	std::multimap< Real, ESBundle > & designed_epitope_scaffolds,
	MultiGraftStats & statistics,
	MultiGraftStage & stage,
	Integer & checkpoint_index
)
{
	if ( stage != ONGOING && stage != QC ) {
		return;
	}
	stage = QC;

	time_t time_start = time( NULL );
	time_t time_current = time_start, time_elapsed = 0;

	EpitopeScaffold::RefineType refine_type = EpitopeScaffold::NORMAL_REFINE;
	if ( options.refine_with_minrepack ) {
		refine_type = EpitopeScaffold::MIN_REPACK_REFINE;
	} else if ( options.refine_with_classic ) {
		refine_type = EpitopeScaffold::CLASSIC_REFINE;
	} else if ( options.refine_with_constraints ) {
		refine_type = EpitopeScaffold::CONSTRAINED_REFINE;
	}

	std::multimap< Real, ESBundle >::iterator esi = designed_epitope_scaffolds.begin();
	for ( Integer i = 0; i < checkpoint_index; ++i ) {
		++esi;
	}

	std::cout << "MultiGraft: performing QC refinement attempt (without Ab)" << std::endl;
	Integer finished = checkpoint_index;
	for ( std::multimap< Real, ESBundle >::iterator esi_e = designed_epitope_scaffolds.end(); esi != esi_e; ++esi ) {

		time_current = time( NULL );
		time_elapsed = time_current - time_start;
		if ( options.use_checkpoint && ( time_elapsed >= options.checkpoint_interval_seconds ) ) {
			checkpoint_save( options.checkpoint_filename_inner, stage, finished, std::multimap< Real, ESBundle >(), designed_epitope_scaffolds );

			time_start = time_current; // reset time
			time_elapsed = 0;
		}

		ESBundle & esb = esi->second;

		// temporary work es
		EpitopeScaffold work_es;
		if ( esb.refined_es != NULL ) {
			work_es = *esb.refined_es;
		} else {
			work_es = *esb.designed_es;
		}

		// remove antibody
		work_es.disconnect_Ab();

		// refinement _without_ antibody to check for movement
		if ( refine_type != EpitopeScaffold::CONSTRAINED_REFINE ) {
			work_es.clear_closed_during_trajectory();
		}
		work_es.refine_all_simul( true, refine_type ); // boolean: ignore_closure_state, use_minrepack_procedure

		// store
		esb.qc_es = new EpitopeScaffold( work_es );

		// record refine attempt
		statistics.record_closure_attempt( "QC refine without Ab", work_es );

		++finished;
	}

	stage = ONGOING;
	checkpoint_index = 0;
}


void
test()
{
	epigraft::design::scratch_screen_fragments();
}


void
old_test()
{
	using pose_ns::Fold_tree;

	disulfides::options::find_disulf = false;
	disulfides::options::norepack_disulf = false;
	score_set_vary_omega( false );

	utility::io::ozstream out( "ARGH.gz" );
	out << "yahoo" << std::endl;
	out.flush();
	out << "uguu" << std::endl;
	out.close();

	out.open_append( "ARGH.gz" );
	out << "wahoo" << std::endl;
	out << std::flush;
	out << "gao" << std::endl;
	out.close();

	return;

	utility::io::izstream infile( "list.txt" );
	utility::io::ozstream outfile( "linear_scores.txt" );

	std::string line;
	utility::vector1< std::string > entries;
	while ( getline( infile, line ) ) {
		entries.clear();
		split_string( line, entries ); // grab all columns from input file
		std::cout << entries[1] << std::endl;

		Pose pose;
		pose_from_pdb( pose, entries[1], true, false, true ); // boolean: fullatom, ideal_pose, read_all_chains

//		Integer buffer = 5;
//		for ( Integer i = buffer, ie = pose.total_residue() - buffer + 1; i <= ie; ++i ) {
		for ( Integer i = 3, ie = pose.total_residue() - 3; i <= ie; ++i ) {
//		for ( Integer i = 271; i < 272; ++i ) {
			// scratch pose
			Pose scratch;
			scratch = pose;

//			scratch.dump_pdb( "zero1.pdb" );

			// fold tree
			Fold_tree aft;
			aft.add_edge( 1, i - 1, Fold_tree::PEPTIDE );
			aft.add_edge( i - 1, i, Fold_tree::PEPTIDE );
			aft.add_edge( i - 1, i + 1, 1 );
			aft.add_edge( i + 1, pose.total_residue(), Fold_tree::PEPTIDE );

//			aft.add_edge( 1, i - 3, Fold_tree::PEPTIDE );
//			aft.add_edge( i - 3, i, Fold_tree::PEPTIDE );
//			aft.add_edge( i - 3, i + 4, 1 );
//			aft.add_edge( i + 4, i + 1, Fold_tree::PEPTIDE );
//			aft.add_edge( i + 4, pose.total_residue(), Fold_tree::PEPTIDE );
			aft.reorder( 1 );
			scratch.set_fold_tree( aft );

			scratch.set_fullatom_flag( true, false ); // force booleans: fullatom, repack
			scratch.set_coords( false, pose.Eposition(), pose.full_coord(), true );

//			std::cout << "SCARLET " << scratch.ideal_backbone() << std::endl;

			Real zero1_score = score_cut_in_Pose_linear( scratch, i, 1 );

//			scratch.dump_pdb( "zero2.pdb" );

//			Real zero2_score = score_cut_in_Pose_linear( scratch, i, 1 );

//			idealize_cutpoint_retaining_cside_phi( scratch, i );
//			scratch.insert_ideal_bonds_at_cutpoint( i );
//			scratch.insert_ideal_bonds( i, i + 1 );

			scratch.set_allow_bb_move( false );
			scratch.set_allow_bb_move( i, true );
//			for ( Integer j = i - 2, je = i + 3; j <= je; ++j ) {
//				scratch.set_allow_bb_move( j, true );
//			}

//			Real forward_deviation, backward_deviation; // actually loop closure msd, both dirs
//			Real torsion_delta; // actually torsion and rama score changes, averaged by loop_size
//			Real rama_delta;

//			scratch.dump_pdb( "start.pdb" );

//			Real before_score = score_cut_in_Pose_linear( scratch, i, 1 );
//			ccd_moves( 50, scratch, i-1, i+1, i );
//			ccd_moves_obeying_nonideality( 50, scratch, i-1, i+1, i );
//			fast_ccd_loop_closure( scratch, i-1, i+1, i, 100,
//					 0.01, false, 10000, 10000,
//					 10000, 10000, forward_deviation,
//					 backward_deviation, torsion_delta, rama_delta );
//			fast_ccd_loop_closure_obeying_nonideality( scratch, i-1, i+1, i, 100,
//					 0.01, false, 10000, 10000,
//					 10000, 10000, forward_deviation,
//					 backward_deviation, torsion_delta, rama_delta );

//			Real after_score = score_cut_in_Pose_linear( scratch, i, 1 );

//			Fold_tree simple_tree;
//			simple_tree.simple_tree( scratch.total_residue() );
//			simple_tree.reorder( 1 );
//			scratch.set_fold_tree( simple_tree );
//
//			scratch.dump_pdb( "after_ccd.pdb" );

			// score
//			outfile << i << "   " << zero1_score << "   " << zero2_score << "   " << before_score << "   " << after_score << std::endl;
			outfile << i << " " << zero1_score << std::endl;
		}
	}

	infile.close();
	outfile.close();
}


void
older_test()
{
	using namespace pose_ns;

	Pose to_build;
	pose_from_pdb( to_build, files_paths::start_path + "to_build.pdb", true, false, true ); // boolean: fullatom, ideal_pose, read_all_chains

//	Fold_tree aft;
//	aft.add_edge( 1, 29, Fold_tree::PEPTIDE );
//	aft.reorder( 1 );

	PoseAssembly assembly( to_build.total_residue() );
	assembly.remove( ResidueRange( 30, 104 ) );
	assembly.insert_right( 29, 2, true, "LL", "GG" );
	assembly.insert_left( 105, 1, false, "L", "G" );

	Pose built0;
	assembly.apply( to_build, built0 );

	Fold_tree ft; // new fold tree for wider jump interval
	ft.add_edge( 1, 25, Fold_tree::PEPTIDE );
	ft.add_edge( 25, 31, Fold_tree::PEPTIDE );
	ft.add_edge( 25, 33, 1 );
	ft.add_edge( 33, 32, Fold_tree::PEPTIDE );
	ft.add_edge( 33, built0.total_residue(), Fold_tree::PEPTIDE );
	ft.reorder( 1 );

	built0.set_fold_tree( ft );

	built0.set_allow_bb_move( false );
	built0.set_allow_chi_move( false );
	built0.set_allow_jump_move( false );

	for ( Integer i = 1; i <= 100; ++i) {
		Pose built;
		built = built0;

		LoopClosureInfo closure( ResidueRange( 26, 32 ), 31, 6, 1 );
		std::set< LoopClosureInfo > cset;
		cset.insert( closure );
		std::set< Integer > fixed_chi;
		closure.setup_allow_bb_move( built );

		build_loops_simul( built, cset, 10, true, 0.008, 3.0 );
		built.recover_sidechain( built0 );

		std::cout << " Statistics Iteration " << i << " :\n";
		closure = *cset.begin();
		std::cout << closure.to_string() << std::endl;
		if ( closure.is_closed() ) {
			// only set allow chi move for residues neighboring the changed region
			std::set< Integer > near_residues = find_neighbor_residues_of_loops( built, cset, 5.5 );
			closure.setup_allow_chi_move( built );
			for ( std::set< Integer >::const_iterator r = near_residues.begin(), re = near_residues.end(); r != re; ++r ) {
				built.set_allow_chi_move( *r, true );
			}
			built.full_repack( true ); // boolean: rotamers exist

			std::ostringstream ss;
			ss << "fully_closed." << i << ".pdb";
			built.dump_scored_pdb( ss.str(), Score_weight_map( score12 ) );
		}
	}

//	refine_loops_simul( built, cset, 20, false, fixed_chi );
}


} // namespace design
} // namespace epigraft
