// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   multigraft_checkpoint.cc
/// @brief  Checkpointing for multigraft
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// unit headers
#include <epigraft/design/multigraft_checkpoint.hh>
#include <epigraft/design/EpitopeScaffold.hh>
#include <epigraft/design/ESBundle.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/epigraft_io.hh>

// rosetta headers
#include <pose.h>
#include <pose_io.h>
#include <refold.h>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// utility headers
#include <utility/io/izstream.hh>

// c++ headers
#include <fstream>
#include <iomanip>
#include <map>
#include <sstream>
#include <string>
#include <utility>
#include <vector>

namespace epigraft {
namespace design {


/// @brief loads epitope scaffold from a pdb file; requires preconstructed EpitopeScaffold
/// @details antibody is SECOND in pdb file
void
reload_epitope_scaffold(
	std::string const & pdb_file,
	Pose const & original_antibody,
	EpitopeScaffold & epitope_scaffold,
	bool const & also_replace_archive
)
{
	// read pose
	Pose restart;
	pose_from_pdb( restart, pdb_file, true, false, true ); // boolean: fullatom, ideal_pose, read_all_chains

	// disconnect Ab from epitope scaffold if necessary
	if ( epitope_scaffold.Ab_is_connected() ) {
		epitope_scaffold.disconnect_Ab( false ); // no auto-archive necessary since we'll be replacing it
	}

	// handle external pose with antibody in file
	assert( restart.total_residue() >= epitope_scaffold.pose().total_residue() );
	bool const file_has_ab = restart.total_residue() > epitope_scaffold.pose().total_residue();

	// now do loading
	AntibodyComplex es_complex;
	if ( file_has_ab ) {
		es_complex = AntibodyComplex( restart, original_antibody.total_residue(), false ); // antibody is SECOND

		// first replace pose in epitope scaffold
		epitope_scaffold.replace_structure( es_complex.antigen(), also_replace_archive );

		// second find rigid body transformation from original to new antibody
		FArray3D_float const & original_Ab_full_coord = original_antibody.full_coord();
		FArray3D_float const & restart_Ab_full_coord = es_complex.Ab().full_coord();
		FArray2D_float transformation_matrix = ObjexxFCL::FArray2D< Real >::identity( 4 );
		get_GL_matrix( original_Ab_full_coord.a( 1, 1, 1 ), original_Ab_full_coord.a( 1, 1, 2 ), original_Ab_full_coord.a( 1, 1, 3 ),
		               restart_Ab_full_coord.a( 1, 1, 1 ), restart_Ab_full_coord.a( 1, 1, 2 ), restart_Ab_full_coord.a( 1, 1, 3 ),
		               transformation_matrix ); // original_Ab moves onto restart_Ab

		// reset transformation matrix
		epitope_scaffold.set_Ab_transform( transformation_matrix );

		// make copy of original antibody and recover restart sidechains so that
		// we have the restart sidechains on an antibody in the original rigid body
		// orientation
		Pose original_antibody_with_restart_sidechains;
		original_antibody_with_restart_sidechains = original_antibody;
		original_antibody_with_restart_sidechains.recover_sidechain( es_complex.Ab() );

		epitope_scaffold.set_Ab( original_antibody_with_restart_sidechains );
	} else {
		// just replace pose in epitope scaffold
		epitope_scaffold.replace_structure( restart, also_replace_archive );
	}

}


/// @brief save checkpoint during multigraft
void
checkpoint_save(
	std::string const & checkpoint_filename,
	MultiGraftStage const & current_stage,
	Integer const & checkpoint_index,
	std::multimap< Real, ESBundle > const & closed_epitope_scaffolds,
	std::multimap< Real, ESBundle > const & designed_epitope_scaffolds
)
{
	std::cout << "* checkpointing... " << std::endl;

	std::ofstream out( checkpoint_filename.c_str() );
	out.setf( std::ios::fixed, std::ios::floatfield ); // formatting for reals
	out.precision( 15 ); // precision for reals

	out << "stage " << current_stage << '\n';
	out << "index " << checkpoint_index << '\n';

	if ( !closed_epitope_scaffolds.empty() ) {
		Integer counter = 0;

		for ( std::multimap< Real, ESBundle >::const_iterator i = closed_epitope_scaffolds.begin(), ie = closed_epitope_scaffolds.end(); i != ie; ++i ) {
			++counter;
			std::ostringstream pdb_prefix;
			pdb_prefix << checkpoint_filename << ".checkpoint.ces."<< counter;
			out << "ces " << i->second.closure_id << ' ' << i->second.design_id << ' ' << pdb_prefix.str() << ' ' << i->first << '\n';
			i->second.checkpoint_save( pdb_prefix.str() );
		}
	}

	if ( !designed_epitope_scaffolds.empty() ) {
		Integer counter = 0;

		for ( std::multimap< Real, ESBundle >::const_iterator i = designed_epitope_scaffolds.begin(), ie = designed_epitope_scaffolds.end(); i != ie; ++i ) {
			++counter;
			std::ostringstream pdb_prefix;
			pdb_prefix << checkpoint_filename << ".checkpoint.des."<< counter;
			out << "des " << i->second.closure_id << ' ' << i->second.design_id << ' ' << pdb_prefix.str() << ' ' << i->first << '\n';
			i->second.checkpoint_save( pdb_prefix.str() );
		}
	}

	out.close();

	std::cout << "* checkpointing saved " << closed_epitope_scaffolds.size() << " closed bundles and "
              << designed_epitope_scaffolds.size() << " designed bundles" << std::endl;
}


/// @brief reload checkpoint contents during multigraft
void
checkpoint_load(
	std::string const & checkpoint_filename,
	Pose const & original_antibody,
	EpitopeScaffold const & base_es,
	MultiGraftStage & current_stage,
	Integer & checkpoint_index,
	std::multimap< Real, ESBundle > & closed_epitope_scaffolds,
	std::multimap< Real, ESBundle > & designed_epitope_scaffolds
)
{
	using std::istringstream;
	using std::make_pair;

	std::cout << "* attempting to restart from checkpoint... " << std::endl;

	std::ifstream in( checkpoint_filename.c_str() );
	if ( in.fail() ) { // no checkpoint file exists
		in.close();
		current_stage = ONGOING;
		checkpoint_index = 0;
		std::cout << "* no checkpoint file found, starting from beginning" << std::endl;
		return;
	}

	std::string line;
	std::vector< std::string> entry;

	while ( getline( in, line ) ) {

		split_string( line, entry );

		if ( entry[ 0 ] == "stage" ) { // current stage

			Integer cs;
			istringstream( entry[ 1 ] ) >> cs;
			current_stage = static_cast< MultiGraftStage >( cs );

		} else if ( entry[ 0 ] == "index" ) { // checkpoint index

			istringstream( entry[ 1 ] ) >> checkpoint_index;

		} else if ( entry[ 0 ] == "ces" ) { // closed epitope scaffold

			ESBundle bundle;
			istringstream( entry[ 1 ] ) >> bundle.closure_id;
			istringstream( entry[ 2 ] ) >> bundle.design_id;

			bundle.checkpoint_load( entry[ 3 ], original_antibody, base_es );

			Real score;
			istringstream( entry[ 4 ] ) >> score;

			closed_epitope_scaffolds.insert( make_pair( score, bundle ) );

		} else if ( entry[ 0 ] == "des" ) { // designed epitope scaffold

			ESBundle bundle;
			istringstream( entry[ 1 ] ) >> bundle.closure_id;
			istringstream( entry[ 2 ] ) >> bundle.design_id;

			bundle.checkpoint_load( entry[ 3 ], original_antibody, base_es );

			Real score;
			istringstream( entry[ 4 ] ) >> score;

			designed_epitope_scaffolds.insert( make_pair( score, bundle ) );
		}

		entry.clear();
	}

	in.close();

	std::cout << "* checkpointing loaded " << closed_epitope_scaffolds.size() << " closed bundles and "
	          << designed_epitope_scaffolds.size() << " designed bundles" << std::endl;
}


/// @brief check if a checkpoint file exists and if there's stage information
void
checkpoint_peek(
	std::string const & checkpoint_filename,
	MultiGraftStage & current_stage,
	Integer & checkpoint_index
)
{
	std::cout << "* searching for checkpoint file to obtain stage... ";

	current_stage = ONGOING;
	checkpoint_index = 0;

	std::ifstream in( checkpoint_filename.c_str() );
	if ( in.fail() ) { // no checkpoint file exists
		in.close();
		std::cout << "not found" << std::endl;
		return;
	}

	std::string line;
	std::vector< std::string > entry;

	while ( getline( in, line ) ) {

		split_string( line, entry );

		if ( entry[ 0 ] == "stage" ) { // current stage
			Integer cs;
			std::istringstream( entry[ 1 ] ) >> cs;
			current_stage = static_cast< MultiGraftStage >( cs );
		} else if ( entry[ 0 ] == "index" ) { // current index
			std::istringstream( entry[ 1 ] ) >> checkpoint_index;
		}

	}

	in.close();

	std::cout << "found, protocol will restart from stage " << current_stage << std::endl;

}


/// @brief touch checkpoint file with basic data
void
checkpoint_touch(
	std::string const & checkpoint_filename,
	MultiGraftStage const & current_stage,
	Integer const & checkpoint_index
)
{
	std::ofstream out( checkpoint_filename.c_str() );

	out << "stage " << current_stage << '\n';
	out << "index " << checkpoint_index << '\n';

	out.close();
}


/// @brief save base epitope scaffold (initial position)
void
checkpoint_save_base(
	std::string const & checkpoint_filename,
	EpitopeScaffold const & base_es
)
{
	base_es.pose().dump_pdb( checkpoint_filename + ".checkpoint.base_es.pdb.gz" );
}


/// @brief reload base epitope scaffold (initial position)
bool
checkpoint_load_base(
	std::string const & checkpoint_filename,
	Pose const & original_antibody,
	EpitopeScaffold & base_es
)
{
	utility::io::izstream in;
	in.open( checkpoint_filename + ".checkpoint.base_es.pdb.gz" );

	if ( in.fail() ) {
		in.close();
		return false;
	}
	in.close();

	reload_epitope_scaffold( checkpoint_filename + ".checkpoint.base_es.pdb.gz", original_antibody, base_es );

	return true;
}


}
}
