// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   multigraft_checkpoint.hh
/// @brief  Checkpointing for multigraft
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_design_multigraft_checkpoint_HH_
#define INCLUDED_epigraft_design_multigraft_checkpoint_HH_

// package headers
#include <epigraft/design/design_types.hh>
#include <epigraft/design/ESBundle.fwd.hh>
#include <epigraft/design/EpitopeScaffold.fwd.hh>

// rosetta headers
#include <pose.h>

// c++ headers
#include <map>
#include <string>


namespace epigraft {
namespace design {


/// @brief loads epitope scaffold from a pdb file; requires preconstructed EpitopeScaffold
void
reload_epitope_scaffold(
	std::string const & pdb_file,
	Pose const & original_antibody,
	EpitopeScaffold & es,
	bool const & also_replace_archive = true
);


/// @brief save checkpoint during multigraft
void
checkpoint_save(
	std::string const & checkpoint_filename,
	MultiGraftStage const & current_stage,
	Integer const & checkpoint_index,
	std::multimap< Real, ESBundle > const & closed_epitope_scaffolds = std::multimap< Real, ESBundle >(),
	std::multimap< Real, ESBundle > const & designed_epitope_scaffolds = std::multimap< Real, ESBundle >()
);


/// @brief reload checkpoint contents during multigraft
void
checkpoint_load(
		std::string const & checkpoint_filename,
		Pose const & original_antibody,
		EpitopeScaffold const & base_es,
		MultiGraftStage & current_stage,
		Integer & checkpoint_index,
		std::multimap< Real, ESBundle > & closed_epitope_scaffolds,
		std::multimap< Real, ESBundle > & designed_epitope_scaffolds
);


/// @brief check if a checkpoint file exists and if there's stage information
void
checkpoint_peek(
	std::string const & checkpoint_filename,
	MultiGraftStage & current_stage,
	Integer & checkpoint_index
);


/// @brief touch checkpoint file with basic data
void
checkpoint_touch(
	std::string const & checkpoint_filename,
	MultiGraftStage const & current_stage = ONGOING,
	Integer const & checkpoint_index = 0
);


/// @brief save base epitope scaffold (initial position)
void
checkpoint_save_base(
	std::string const & checkpoint_filename,
	EpitopeScaffold const & base_es
);


/// @brief reload base epitope scaffold (initial position)
bool
checkpoint_load_base(
	std::string const & checkpoint_filename,
	Pose const & original_antibody,
	EpitopeScaffold & base_es
);


}
}


#endif /*INCLUDED_epigraft_design_multigraft_checkpoint_HH_*/
