// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7630 $
//  $Date: 2006-03-10 09:37:52 -0800 (Fri, 10 Mar 2006) $
//  $Author: stuartm $

// epigraft headers
#include <epigraft/epi_graft.h> // bills
#include <epigraft/epigraft_functions.hh>
#include <epigraft/option_check.hh>
#include <epigraft/design/decimated_multigraft.hh>
#include <epigraft/design/multigraft.hh>
#include <epigraft/match/epigraft_match.hh>
#include <epigraft/prediction/fragment_assembly.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/design/GraftInfo.hh>
#include <epigraft/glycan/glycan_test.hh>

// Rosetta Headers
#include <aaproperties_pack.h>
#include <count_pair.h>
#include <design.h> // bills
#include <input_pdb.h>  // dihedral
#include <diagnostics_rosetta.h>
#include <disulfides.h> // cys_res_in_disulf
#include <after_opts.h>
#include <fast_pairenergy.h> // fast_pairenergy
#include <files_paths.h>
#include <jumping_refold.h>
#include <knots.h> // rank
#include <loops_ns.h> // loop_bool::trim
#include <misc.h> // damn
#include <pack.h> // pack_rotamers,num_from_res1
#include <pack_geom_inline.h> // distance_bk
#include <PackerTask.h>
#include <param.h> // MAX_POS
#include <param_pack.h> // MAX_POS
#include <pdbstatistics_pack.h> // hydrogen_interaction_cutoff
#include <pose.h>
#include <pose_io.h>
#include <pose_loops.h>
#include <pose_loops_input_ns.h>
#include <read_aaproperties.h> // atom_name_from_atom_num
#include <refold.h> // get_GL_matrix
#include <rms.h> // charlie rms routines
#include <runlevel.h>
#include <smallmove.h>
#include <score.h>

// rootstock headers
#include <rootstock/rootstock_types.hh>
#include <rootstock/BoundingBox.hh>
#include <rootstock/ClosedInterval.hh>
#include <rootstock/Octree.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh> // this might be needed for Atom
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/formatted.io.hh>

// numeric headers
#include <numeric/xyzVector.hh>

// utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <iostream>
#include <set>
#include <string>
#include <vector>


/////////////////////////////////////////////////////////////////////////////////////////////////////
void
epi_graft()
{
	// set rosetta global options
	disulfides::options::find_disulf = true;
	disulfides::options::norepack_disulf = true;

	// first check to see if there are any known command line options
	if ( !epigraft::options_are_valid() ) {
		utility::exit( __FILE__, __LINE__, "ERROR: unknown options" );
	}

	epigraft::GraftOptions options;

	if ( truefalseoption( "help" ) ) {
		if ( truefalseoption( "external_graft_info" ) ) {
			std::cout << std::endl;
			std::cout << epigraft::design::GraftInfo::external_graft_info_help() << std::endl;
		} else {
			std::cout << std::endl;
			std::cout << options.usage() << std::endl;
		}
		return;
	}

	options.get_from_rosetta();

	if ( options.match_mode ) {

		if ( truefalseoption( "test" ) ) {
			epigraft::match::test( options );
			return;
		}

		epigraft::match::epigraft_match( options );
		return;

	} else if ( options.multigraft_mode ) {

		epigraft::design::multigraft( options );
		return;

	} else if ( options.decimated_multigraft_mode ) {

		epigraft::design::decimated_multigraft( options );
		return;

	} else if ( options.scarlet ) { // experimental section

//		epigraft::glycan::test();
		epigraft::prediction::mper();
		return;
	}

	////////////////////////////////////////////////////////////////////////////
	//general protocol to identify potential graft sites in proteins
	//
	//   this reports closure_rms, intra_clash, inter_clash
	//

	if ( truefalseoption("epi_graft_match") ) {
		std::cerr << "old functionality disabled, please run -epi_graft -help to see new options" << std::endl;
		return;
	}


	///////////////////////////////////////////////////////////////////////////
	//special protocol to identify potential graft sites in scaffolds that already have
	// one loop superposed ( or grafted )
	//
	//   this reports closure_rms
	//

	if ( truefalseoption("rough_match_for_input_superpos") ) {
		std::cerr << "old functionality disabled, please run -epi_graft -help to see new options" << std::endl;
		return;
	}


	////////////////////////////////////////////////////////////////////////////
	//
	//main protocol for actually making/closing/designing a graft
	//
	//

	if ( truefalseoption("epi_graft_design") ) {

		if ( truefalseoption("add_on") ) {

			//
			//protocol for grafting a second loop onto a scaffold with 1 existing loop
			//
			add_on_epi_graft_design();

		} else {

			//
			//grafting of the first loop
			//

			epi_graft_design();
		}
		return;
	}


}


/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
void
epi_graft_design()
{
	using namespace pose_ns;

	// scorefxn
	Score_weight_map weight_map( score12 );

	//
	//set basic parameters controlling pose behavior
	//
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	bool const read_all_chains( true );
	bool const check_missing( false ); //use this for pose.set_coords throughout
	bool const coords_init( true ); //used in pose_from_misc


	//debug option
	//
	bool const output_debug_pdbs=truefalseoption("output_debug_pdbs");

	//
	//Before you read in pdbs, set disulf options automatically
	//
	disulfides::options::find_disulf = true;
	disulfides::options::norepack_disulf = true;

	//
	//Before you read in pdbs, set no_optH
	//
	//	files_paths::no_optH = truefalseoption("no_optH");

	//
	//file to output scores
	//
	std::string output_filename;
	//vds	stringafteroption( "output_file", "none", output_filename );
	stringafteroption( "output_file", "graft.out", output_filename );
	if( output_filename == "none" ) {
		std::cout << "Need -output_file <filename> "
              << "on command-line " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
 	}
	//
	//open the output file
	// ( leave it open since we write scores about 1/s or faster )
	utility::io::ozstream outfile;
	open_output_file( output_filename, outfile );

	//
	//checkpoint file for running on clusters
	//
	std::string checkpoint_filename;
	std::string checkpoint_file_w_path;
	stringafteroption( "checkpoint", "none", checkpoint_filename );
	bool using_checkpoint = false;
 	if ( checkpoint_filename == "none" ) {
		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "no checkpoint_filename, NOT using checkpointing" << std::endl;
		}
	} else {
		using_checkpoint = true;
		checkpoint_file_w_path = files_paths::score_path + checkpoint_filename;
	}

	//
	//Do you want pdbs output for design/refine?
	//
	bool output_pdbs = truefalseoption("output_pdbs");

	// read in the native complex of epitope-Ab
	Pose nat_cmplx_pose;
	pose_from_pdb( nat_cmplx_pose, stringafteroption("native_complex"), fullatom, ideal_pose,
								 read_all_chains );

	//	nat_cmplx_pose.dump_pdb( "nat_cmplx_pose.pdb" );

	// flip symmetric sidechains
	{
		float const start_score( nat_cmplx_pose.score( weight_map ) );
		nat_cmplx_pose.refold_sidechains_from_chi();
		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "score-delta from sidechain flip: " <<
				nat_cmplx_pose.score( weight_map ) - start_score << std::endl;
		}
	}

	int const nres_nat_cmplx( nat_cmplx_pose.total_residue() );
	int nres_Ab(0);
	// set length of partner1:
	// get from commandline
	intafteroption( "nres_Ab", 0, nres_Ab );
	if ( nres_Ab == 0 ) {
		std::cout << "must specify -nres_Ab <nres> on commandline" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//save nres for full-length peptide graft
	//
	int const nres_nat_pep = nres_nat_cmplx - nres_Ab;

	//make a pose that is Ab only and score it
	//Assuming that Ab is first in the input nat_cmplx_pose
	if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
		std::cout << "Assuming that Ab is first in the input nat_cmplx_pose !! " << std::endl;
	}
	Pose Ab_pose;
	Ab_pose.simple_fold_tree( nres_Ab );
	Ab_pose.set_fullatom_flag( true, false /*repack*/);
	//		Ab_pose.copy_segment( nres_Ab, nat_cmplx_pose, 1, 1 /*begin_src*/);
	copy_segment_to_new_pose( nat_cmplx_pose, 1 /*begin_src*/, nres_Ab /*end_src*/, Ab_pose );
	Ab_pose.score( weight_map );
	//	Ab_pose.dump_pdb("Ab_pose.pdb");

	float const fa_rep_Ab ( Ab_pose.get_0D_score( FA_REP ) );

	//make a pose that is epitope only and score it
	Pose nat_pep_pose;
	nat_pep_pose.simple_fold_tree( nres_nat_pep );
	nat_pep_pose.set_fullatom_flag( true, false /*repack*/);
	//		nat_pep_pose.copy_segment( nres_nat_pep, nat_cmplx_pose, 1, nres_Ab+1 /*begin_src*/ );
	copy_segment_to_new_pose( nat_cmplx_pose, nres_Ab+1 /*begin_src*/, nres_nat_cmplx /*end_src*/, nat_pep_pose );
	nat_pep_pose.score( weight_map );
	float const fa_rep_nat_pep ( nat_pep_pose.get_0D_score( FA_REP ) );

	/*	nat_pep_pose.dump_pdb("nat_pep_omega_notfixed.pdb");
	nat_pep_pose.set_omega( 9, 180. );
	nat_pep_pose.score( weight_map );
	nat_pep_pose.dump_pdb("nat_pep_omega_fixed.pdb");
	return;
	*/

	//Read in epitope positions at which native rotamers will be retained.
	// Notes: These are epitope positions that interact with Ab.
	//        These must be distinguished from epitope positions that point away
	//        from the Ab and can be repacked/redesigned.
	//        The list file must contain rosetta numbering not pdb numbering !
	//
	int n_natro_in_epitope = 0;
	FArray1D_int list_natro_in_epitope( n_natro_in_epitope );
	FArray1D_bool natro_in_epitope( nres_nat_pep, false );
	get_natro_in_epitope( nres_nat_pep, n_natro_in_epitope,
												list_natro_in_epitope,
												natro_in_epitope );


	//make a pose with all-atom Ab and glycine for epitope
	Pose nat_cmplx_gly_pose;
	nat_cmplx_gly_pose = nat_cmplx_pose;
	int gly_begin = nres_Ab + 1;
	int gly_end = nres_nat_cmplx;
	convert_resnum_range_to_gly( nat_cmplx_gly_pose, gly_begin,  gly_end );
	nat_cmplx_gly_pose.score( weight_map );
	float const fa_rep_nat_complex_gly ( nat_cmplx_gly_pose.get_0D_score( FA_REP ) );

	//	nat_cmplx_gly_pose.dump_pdb( "nat_cmplx_gly_pose.pdb");

	//make a pose that is just the native epitope (peptide) all-gly
	Pose nat_pep_gly_pose;
	nat_pep_gly_pose.simple_fold_tree( nres_nat_pep );
	nat_pep_gly_pose.set_fullatom_flag( true, false /*repack*/);
	//		nat_pep_gly_pose.copy_segment( nres_nat_pep, nat_cmplx_gly_pose, 1, nres_Ab+1 /*begin_src*/ );
	copy_segment_to_new_pose( nat_cmplx_gly_pose, nres_Ab+1 /*begin_src*/, nres_nat_cmplx /*end_src*/, nat_pep_gly_pose );
	nat_pep_gly_pose.score( weight_map );
	//	nat_pep_gly_pose.dump_pdb( "nat_pep_gly_pose.pdb");
	float const fa_rep_nat_peptide_gly ( nat_pep_gly_pose.get_0D_score( FA_REP ) );



	////////////////////////////////////////////////////////////////////
	//native: now measure clash between epitope backbone and Ab all-atom
	////////////////////////////////////////////////////////////////////

	float const ddg_fa_rep_nat_gly ( fa_rep_nat_complex_gly - fa_rep_nat_peptide_gly - fa_rep_Ab );

	if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
		std::cout << "native complex w/all-gly epitope:: "
							<< "farep_cmplx, farep_Ab, farep_peptide, ddg_farep "
							<< fa_rep_nat_complex_gly << " "
							<< fa_rep_Ab << " "
							<< fa_rep_nat_peptide_gly << " "
							<< ddg_fa_rep_nat_gly << std::endl;
	}
	////////////////////////////////////////////////////////////////////
	//These are the parameters describing the graft that should be
	//read in from one line of a file
	////////////////////////////////////////////////////////////////////

	//
	//The basic assumption is that the scaff will nearly close the graft already
	//and that the residues that can be aligned in space are
	//
	//   graft_resnum_1   <---->  scaff_gap_resnum_1
	//   graft_resnum_2   <---->  scaff_gap_resnum_2
	//
	//When we finally put the graft on, we can only keep one member of
	//of each aligned pair, because they will overlap in space.
	//

	/*
	std::string graft_dir = "n2c";             // n2c or c2n...if n2c, then Nterm is superposed
	std::string graft_align_system = "N";     // N or CA or C
	int const graft_resnum_1 = 3; //671; //(low resnum in peptide we want to graft into a scaffold...in sequential numbering)
	int const graft_resnum_2 = 12; //680;  //( resnum_1 < resnum_2 always !)
	std::string scaff_pdb5code = "1z6na";
	int const	scaff_gap_resnum_1 = 138;               // n-term of gap in scaffold
	int const scaff_gap_resnum_2 = 147;               // c-term of gap in scaffold
	*/

	//
	//Read in list of scaffold pdbs...also reads information about the graft for each scaffold
	//
	// each line of input file has 7 fields:
	//graft_dir graft_align_system graft_resnum_1 graft_resnum_2 scaff_gap_resnum_1 scaff_gap_resnum_2 pdbname
	FArray1D_string list_graft_dir;            //n2c or c2n
	FArray1D_string list_graft_align_system;   //N or CA or C ( for normal graft ) or S ( for graft based on epitope-graft superposition ) or E (endpoint superposition)
	FArray2D_int list_graft_resnums;           //i=1:graft_resnum_1      i=2:graft_resnum_2       i=3,4:graft_resnum_1,_2 in native epitope numbering  j=scaff_id_num
	FArray2D_int list_scaff_gap_resnums;       //i=1:scaff_gap_resnum_1  i=2:scaff_gap_resnum_2   j=scaff_id_num
	FArray1D_string list_scaff_pdb;
	read_list_scaff_info_for_epi_graft_design( list_graft_dir,
																						 list_graft_align_system,
																						 list_graft_resnums,
																						 list_scaff_gap_resnums,
																						 list_scaff_pdb );
	int const n_scaff = list_scaff_pdb.size();

	//if allow rb moves,
	//then we release any initial superposition and we must close both sides of graft.
	bool const allow_rb_moves_for_epitope = false;  //if allow rb moves,

	for ( int i_scaff = 1; i_scaff <= n_scaff; i_scaff++ ) {

		reset_global_loop_info(); // required, otherwise there is confusion from prior iteration

		if ( using_checkpoint ) {
			bool already_scored = check_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			if ( already_scored ) continue;
		}

		std::string graft_dir = list_graft_dir( i_scaff );
		std::string graft_align_system = list_graft_align_system( i_scaff );
		int const graft_resnum_1 = list_graft_resnums( 1, i_scaff );
		int const graft_resnum_2 = list_graft_resnums( 2, i_scaff );
		int const scaff_gap_resnum_1 = list_scaff_gap_resnums( 1, i_scaff );
		int const scaff_gap_resnum_2 = list_scaff_gap_resnums( 2, i_scaff );
		std::string scaff_base_filename = list_scaff_pdb( i_scaff );
		if ( has_suffix( scaff_base_filename, ".pdb.gz" ) ) scaff_base_filename.erase( scaff_base_filename.length() - 7 ); // strip terminal .pdb.gz
		if ( has_suffix( scaff_base_filename, ".pdb" ) ) scaff_base_filename.erase( scaff_base_filename.length() - 4 ); // strip terminal .pdb


		////////////////////////////////////////////////////////////////////
		// Parameters derived from those read in
		//

		int const	scaff_stem_resnum_1 = scaff_gap_resnum_1 + 1;               // n-term stem in gap in scaffold
		int const scaff_stem_resnum_2 = scaff_gap_resnum_2 - 1;               // c-term setm in gap in scaffold
		int const nres_graft = graft_resnum_2 - graft_resnum_1 + 1;
		//
		//nres_gap_in_scaff tells how many residues can fit in gap if
		//both gap residues from the scaff are deleted.
		//or if the terminal residues on the graft are deleted.
		//
		int const nres_gap_in_scaff = scaff_gap_resnum_2 - scaff_gap_resnum_1 + 1;

		if ( runlevel_ns::runlevel > runlevel_ns::silent ) {
			if ( ! ( nres_gap_in_scaff == nres_graft ) ) {
				std::cout << "WARNING: size mismatch between graft: " << nres_graft << " and gap: "
									<< nres_gap_in_scaff << __FILE__ << " " <<  __LINE__ << std::endl;
				//			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

		//
		//Determine if this is a
		//(1) "normal" graft with one termini superposed to scaff and the other terminus needing closure
		//(2) "S" graft in which rb orientation of epitope w.r.t. scaff is fixed by backbone superposition over entire epitope, and both ends of graft need closure
		//(3) "E" graft in which rb orientation of epitope w.r.t. scaff is fixed by backbone superposition of both ends of epitope, and both ends of graft need closure
		//

		////////////////////////////////////////////////////////////////////
		//		load scaffold into pose
		//

		std::string scaff_filename = files_paths::start_path + list_scaff_pdb( i_scaff );
		//files_paths::start_path + files_paths::start_file

		utility::io::izstream infile ( scaff_filename );
		if ( ! infile ) {
			std::cout	<< " cannot open file " << scaff_filename
								<< std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		std::cout << "successfully read scaff " << SS( scaff_filename ) << " with dir: " << graft_dir
							<< " and system: " << graft_align_system << std::endl;

		Pose scaff_pose;
		pose_from_pdb( scaff_pose, scaff_filename, fullatom, ideal_pose, read_all_chains );
		int const nres_scaff = scaff_pose.total_residue();

		//		scaff_pose.dump_pdb("scaff_pose.pdb");

		//	construct a gly-only version of scaffold  -- scaff_gly_pose
		//
		Pose scaff_gly_pose;
		scaff_gly_pose = scaff_pose;
		gly_begin = 1;
		gly_end = nres_scaff;
		convert_resnum_range_to_gly( scaff_gly_pose, gly_begin, gly_end );
		//		scaff_gly_pose.dump_pdb("scaff_gly_pose.pdb");

		//	construct gly-only version with graft-site removed -- scaff_gly_w_gap_pose
		//
		//  Note:  scaff_gap_resnum_1 =< gap =< scaff_gap_resnum_2
		//         gap DOES include the scaff_gap_resnum_x
		//
		Pose scaff_gly_w_gap_pose;
		int gap_begin = scaff_gap_resnum_1; //scaff_gap_resnum_1 + 1;
		int gap_end = scaff_gap_resnum_2;   //scaff_gap_resnum_2 - 1;
		make_pose_with_gap( scaff_gly_pose, gap_begin, gap_end, scaff_gly_w_gap_pose );
		//		scaff_gly_w_gap_pose.dump_pdb("scaff_gly_w_gap_pose.pdb");
		int const nres_scaff_w_gap = scaff_gly_w_gap_pose.total_residue();

		if ( nres_scaff_w_gap != ( nres_scaff - ( gap_end - gap_begin + 1 ) ) ) {
			std::cout << "error on nres_scaff_w_gap " << nres_scaff_w_gap << " "
								<< nres_scaff << " " << gap_end << " " << gap_begin << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		//	score scaff_gly_w_gap and save the fa_rep
		//
		scaff_gly_w_gap_pose.score( weight_map );
		float const fa_rep_scaff_gly_w_gap ( scaff_gly_w_gap_pose.get_0D_score( FA_REP ) );

		////////////////////////////////////////////////////////////////////
		//The next 3 steps are:
		//
		//(1) Orient the full-length all-gly peptide onto the graftsite (gap in scaff)
		//       must do the orienting on full-length peptide b/c input resnums
		//       control which atoms to superpose
		//(2) trim native peptide to the right residue range for this graft
		//(3) evalute repE interaction between peptide and scaff.
		//
		////////////////////////////////////////////////////////////////////

		//
		//if graft_align_system="S" then we need to make alignment by superpos rather than GL
		//

		//
		// matrix to reorient peptide coordinates onto scaffold coordinates
		//we'll use this orient re-orient the antibody also, down below.
		//
		FArray2D_float Mgl( 4, 4 );

		if ( graft_align_system == "S" ) {

			//
			//Do the superposition of epitope onto scaffold
			//
			//need to superpose NCACO backbone of epitope onto scaffold over full range:
			//epitope:  graft_resnum_1      -   graft_resnum_2
			//scaff:    scaff_gap_resnum_1  -   scaff_gap_resnum_2

			Pose nat_pep_gly_oriented_pose;
			nat_pep_gly_oriented_pose = nat_pep_gly_pose;

			superimpose_using_ncaco(
															scaff_gly_pose /*pose1*/,
															scaff_gap_resnum_1 /*pose1_first*/,
															scaff_gap_resnum_2 /*pose1_last*/,
															nat_pep_gly_oriented_pose /*pose2*/, //pose2 will be changed, superimposed onto pose1
															graft_resnum_1  /*pose2_first*/,
															graft_resnum_2 /*pose2_last*/
															);


			if ( output_debug_pdbs ) {
				std::cout << "dumping 2 pdbs so you can see if superpose worked correctly: " << std::endl;
				std::cout << "nat_pep_gly_oriented_pose.pdb and scaff_gly_pose.pdb" << std::endl;
				nat_pep_gly_oriented_pose.dump_pdb("nat_pep_gly_oriented_pose.pdb");
				scaff_gly_pose.dump_pdb("scaff_gly_pose.pdb");
			}

			//
			//Compute Mgl matrix for this superposition
			//
			//To be consistent with normal grafting for "N","CA",or "C",
			//we will compute the Mgl for this superimpose
			//and we'll use the Mgl to reconstruct the superposition
			//and make the oriented antibody below.
			//we don't have to do it this way
			//
			FArray1D_float coords_cent( 3 ),coords_forw( 3 ),coords_back( 3 );
			FArray1D_float coords_new_cent( 3 ),coords_new_forw( 3 ),coords_new_back( 3 );
			for ( int k = 1; k <= 3; k++ ) {
				coords_back( k ) = nat_pep_gly_pose.full_coord()( k, 1, 1 );//N
				coords_cent( k ) = nat_pep_gly_pose.full_coord()( k, 2, 1 );//CA
				coords_forw( k ) = nat_pep_gly_pose.full_coord()( k, 4, 1 );//C
				coords_new_back( k ) = nat_pep_gly_oriented_pose.full_coord()( k, 1, 1 );//N
				coords_new_cent( k ) = nat_pep_gly_oriented_pose.full_coord()( k, 2, 1 );//CA
				coords_new_forw( k ) = nat_pep_gly_oriented_pose.full_coord()( k, 4, 1 );//C
			}

			get_GL_matrix( coords_forw,   coords_cent,   coords_back,
										 coords_new_forw, coords_new_cent, coords_new_back, Mgl );
		} else if ( graft_align_system == "E" ) {
			// superimpose only the heavy atoms (N, CA, C, O) of the two endpoints
			Pose peptide_end_residues;
			epigraft::prepare_pose( peptide_end_residues, 2);
			FArray1D_bool keep( nat_pep_gly_pose.total_residue(), false );
			keep( graft_resnum_1 ) = true;
			keep( graft_resnum_2 ) = true;
			copy_info_to_trim_pose( nat_pep_gly_pose, keep, peptide_end_residues );

			Pose scaffold_end_residues;
			epigraft::prepare_pose( scaffold_end_residues, 2);
			keep.redimension( scaff_gly_pose.total_residue() );
			keep = false;
			keep( scaff_gap_resnum_1 ) = true;
			keep( scaff_gap_resnum_2 ) = true;
			copy_info_to_trim_pose( scaff_gly_pose, keep, scaffold_end_residues );

			superimpose_using_ncaco( scaffold_end_residues, 1, 2, peptide_end_residues, 1, 2 );

			// compute Mgl matrix for superposition -- code copied from prior conditional
			FArray1D_float coords_cent( 3 ),coords_forw( 3 ),coords_back( 3 );
			FArray1D_float coords_new_cent( 3 ),coords_new_forw( 3 ),coords_new_back( 3 );
			for ( int k = 1; k <= 3; k++ ) {
				coords_back( k ) = nat_pep_gly_pose.full_coord()( k, 1, graft_resnum_1 );//N
				coords_cent( k ) = nat_pep_gly_pose.full_coord()( k, 2, graft_resnum_1 );//CA
				coords_forw( k ) = nat_pep_gly_pose.full_coord()( k, 4, graft_resnum_1 );//C
				coords_new_back( k ) = peptide_end_residues.full_coord()( k, 1, 1 );//N
				coords_new_cent( k ) = peptide_end_residues.full_coord()( k, 2, 1 );//CA
				coords_new_forw( k ) = peptide_end_residues.full_coord()( k, 4, 1 );//C
			}

			get_GL_matrix( coords_forw, coords_cent, coords_back,
			               coords_new_forw, coords_new_cent, coords_new_back, Mgl );
		} else {

			//
			//(1)  Orient the full-length all-gly peptide onto the graftsite (gap in scaff)
			//
			//align 3 atoms on graft peptide with 3 atoms on scaff
			//the three atoms we use depend on the "graft_align_sys" and the "graft_dir"
			//the 3 atoms are referred to as "cent","forw", and "back"


			FArray1D_float coords_pep_cent( 3 ),coords_pep_forw( 3 ),coords_pep_back( 3 );
			FArray1D_float coords_scaff_cent( 3 ),coords_scaff_forw( 3 ),coords_scaff_back( 3 );

			get_alignment_coords( nat_pep_gly_pose,
														scaff_gly_pose,
														graft_align_system,
														graft_dir,
														graft_resnum_1,
														graft_resnum_2,
														scaff_gap_resnum_1,
														scaff_gap_resnum_2,
														coords_pep_cent,    //outputs from here down
														coords_pep_forw,
														coords_pep_back,
														coords_scaff_cent,
														coords_scaff_forw,
														coords_scaff_back );

			////////////////////////////////////////////////////////////////////
			// get matrix to reorient peptide coordinates onto scaffold coordinates
			// P2 mapped to Q2...P2->P3 vector mapped to Q2->Q3 vector
			//
			//		FArray2D_float Mgl( 4, 4 );
			get_GL_matrix( coords_pep_forw,   coords_pep_cent,   coords_pep_back,
										 coords_scaff_forw, coords_scaff_cent, coords_scaff_back, Mgl );
		}


		////////////////////////////////////////////////////////////////////
		//make a copy of the peptide gly coordinates
		//then re-orient these coords
		//

		FArray3D_float coords_pep_gly_oriented ( nat_pep_gly_pose.full_coord() );

		// apply the transformation to the peptide
		// apply to coords_pep_gly_oriented
		for ( int ires = 1; ires <= nres_nat_pep; ires++ ) {
			int const aa( nat_pep_gly_pose.res( ires ) );
			int const aav( nat_pep_gly_pose.res_variant( ires ));
			for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
				GL_rot_in_place( Mgl, coords_pep_gly_oriented( 1, iatom, ires ) );
			}
		}

		////////////////////////////////////////////////////////////////////
		//make a pose of just the oriented peptide
		//
		Pose pep_gly_oriented_pose;
		pep_gly_oriented_pose.simple_fold_tree( nres_nat_pep );
		pep_gly_oriented_pose.set_fullatom_flag( true, false );// set fullatom and do not repack
		FArray3D_float Epos_pep_gly_oriented( 3, param::MAX_POS, nres_nat_pep );//MAX_POS = 5
		for ( int i = 1; i <= nres_nat_pep; ++i ) {
			pep_gly_oriented_pose.set_phi        ( i, nat_pep_gly_pose.phi(i) );
			pep_gly_oriented_pose.set_psi        ( i, nat_pep_gly_pose.psi(i) );
			pep_gly_oriented_pose.set_omega      ( i, nat_pep_gly_pose.omega(i) );
			pep_gly_oriented_pose.set_secstruct  ( i, nat_pep_gly_pose.secstruct(i) );
			pep_gly_oriented_pose.set_name       ( i, nat_pep_gly_pose.name(i) );
			pep_gly_oriented_pose.set_res        ( i, nat_pep_gly_pose.res( i ) );
			pep_gly_oriented_pose.set_res_variant( i, nat_pep_gly_pose.res_variant( i ) );
			for ( int k = 1; k <= 3; ++k ) {
				Epos_pep_gly_oriented(k,1,i) = coords_pep_gly_oriented(k,1,i);
				Epos_pep_gly_oriented(k,2,i) = coords_pep_gly_oriented(k,2,i);
				Epos_pep_gly_oriented(k,4,i) = coords_pep_gly_oriented(k,3,i);
				Epos_pep_gly_oriented(k,5,i) = coords_pep_gly_oriented(k,4,i);
				Epos_pep_gly_oriented(k,3,i) = coords_pep_gly_oriented(k,5,i);
			}
		}
		pep_gly_oriented_pose.set_coords( ideal_pose, Epos_pep_gly_oriented, coords_pep_gly_oriented, check_missing );
		if ( output_debug_pdbs ) pep_gly_oriented_pose.dump_pdb( "pep_gly_oriented_pose.pdb" );

		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << " done orienting peptide, now to trim pep->graft" << std::endl;
		}
		////////////////////////////////////////////////////////////////////
		//1. trim the full length all-gly peptide to the correct
		//   sub-range for the graft (call this the "graft_gly")
		//2. make a pose for oriented graft_gly
		//
		//
		//Note: General rule is to include the terminal residues of the graft
		//      Use the corresponding gap residues from scaff instead.
		//      This means we do keep graft_resnum_1,graft_resnum_2
		//      we will NOT use scaff_gap_resnum_1, scaff_gap_resnum_2 at
		//      those spatial positions.

		//
		//graft all-gly
		//
		Pose graft_gly_pose;
		graft_gly_pose.simple_fold_tree( nres_graft );
		graft_gly_pose.set_fullatom_flag( true, false /*repack*/ );
		copy_segment_to_new_pose( pep_gly_oriented_pose /*src_pose*/,
															graft_resnum_1 /*begin_src*/,
															graft_resnum_2 /*end_src*/,
															graft_gly_pose /*pose_out*/);

		if ( output_debug_pdbs ) graft_gly_pose.dump_pdb( "graft_gly_pose.pdb" );

		//
		//Score the graft_notermini_gly_pose to get fa_rep
		//

		graft_gly_pose.score( weight_map );
		float const fa_rep_graft_gly ( graft_gly_pose.get_0D_score( FA_REP ) );

		//next we want to make a pose that has the graft + scaff_w_gap in all-gly
		//then score fa_rep for that pose and substract fa_rep for scaff_w_gap alone

		int const nres_scaff_w_graft = nres_graft + nres_scaff_w_gap;

		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "nres_scaff_w_graft, nres_graft, nres_scaff_w_gap = "
								<< nres_scaff_w_graft << " "
								<< nres_graft << " "
								<< nres_scaff_w_gap << std::endl;

			std::cout << "nres_scaff, scaff_gap_resnum_1, scaff_gap_resnum_2 "
								<< nres_scaff << " " << scaff_gap_resnum_1 << " "
								<< scaff_gap_resnum_2 << std::endl;
		}

		Pose scaff_w_graft_gly_pose_old;
		make_scaff_w_graft_gly_pose ( scaff_gly_pose,
																	graft_gly_pose,
																	scaff_gap_resnum_1,
																	scaff_gap_resnum_2,
																	nres_scaff_w_graft,
																	scaff_w_graft_gly_pose_old );//output


    Pose scaff_w_graft_gly_pose;
    new_make_scaff_w_graft_gly_pose( scaff_gly_pose,
                                     pep_gly_oriented_pose,
                                     graft_align_system,
                                     graft_dir,
                                     graft_resnum_1,
                                     graft_resnum_2,
                                     scaff_gap_resnum_1,
                                     scaff_gap_resnum_2,
                                     nres_scaff_w_graft,
                                     scaff_w_graft_gly_pose );//output



		scaff_w_graft_gly_pose.score( weight_map );
		float const fa_rep_scaff_w_graft_gly ( scaff_w_graft_gly_pose.get_0D_score( FA_REP ) );
		if ( output_debug_pdbs ) {
      scaff_w_graft_gly_pose_old.dump_pdb("scaff_w_graft_gly_pose_old.pdb");
			std::string tmp_filename = "scaff_w_graft_gly_pose_" + string_of( i_scaff ) + ".pdb";
			scaff_w_graft_gly_pose.dump_pdb(tmp_filename);
		}

    ////////////////////////////////////////////////////////////////////
    //scaffold: now measure "intra" clash between graft and scaff backbones only
    ////////////////////////////////////////////////////////////////////

		float const fa_rep_intra_gly ( fa_rep_scaff_w_graft_gly - fa_rep_graft_gly - fa_rep_scaff_gly_w_gap );

		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "fa_rep_intra_gly, fa_rep_scaff_w_graft_gly, fa_rep_graft_gly, fa_rep_scaff_gly_w_gap "
								<< fa_rep_intra_gly << " " << fa_rep_scaff_w_graft_gly << " " << fa_rep_graft_gly << " "
								<< fa_rep_scaff_gly_w_gap << std::endl;
		}
		///////////////////////////////////////////////////////////////////////////////
		//Ask if this intra clash is below the loose threshold
		//Using a loose threshold initially prior to closing the chainbreak(s)
		//

		float max_fa_rep_intra_gly_before_close_default( 3000. ); //needs to be tested
		float max_fa_rep_intra_gly_before_close;
		realafteroption("max_fa_rep_intra_gly_before_close", max_fa_rep_intra_gly_before_close_default, max_fa_rep_intra_gly_before_close );

		//
		// if not satisfy loose_max, continue to next graft/scaff combo
		//
		if ( fa_rep_intra_gly > max_fa_rep_intra_gly_before_close ) {
			std::cout << "fa_rep_intra_gly:" << fa_rep_intra_gly << " is higher than cutoff: " << max_fa_rep_intra_gly_before_close
								<< " going to next graft/scaff combo" << std::endl;

			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}
			continue;
		}

		///////////////////////////////////////////////////////////////////////////////
		//Now want to measure fa_rep_inter_gly
		//  This is the clash between all-atom Ab and all-gly scaff_w_gap
		//
		//For this we must orient the Ab the same way we have oriented the graft
		//Then we can either (1) make a pose of oriented Ab+ scaff_w_gap and score it
		//            or     (2) directly measure repE between Ab and scaff_w_gap (faster?)
		//

		//make a copy of the Ab coords to orient
		//
		FArray3D_float coords_Ab_oriented ( Ab_pose.full_coord() );

		// apply the transformation to the Ab ( use same matrix "Mgl" applied to epitope )
		//
		for ( int ires = 1; ires <= nres_Ab; ires++ ) {
			int const aa( Ab_pose.res( ires ) );
			int const aav( Ab_pose.res_variant( ires ));
			for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
				GL_rot_in_place( Mgl, coords_Ab_oriented( 1, iatom, ires ) );
			}
		}

		//
		//make a pose of Ab_oriented
		//
		Pose Ab_oriented_pose;
		Ab_oriented_pose = Ab_pose;

		//make an Eposition array that is consistent with
		//transformed fcoord array.
		//
		FArray3D_float Epos_Ab_oriented( 3, param::MAX_POS, nres_Ab );//MAX_POS = 5

		for ( int i = 1; i <= nres_Ab; ++i ) {
			for ( int k = 1; k <= 3; ++k ) {
				Epos_Ab_oriented(k,1,i) = coords_Ab_oriented(k,1,i);
				Epos_Ab_oriented(k,2,i) = coords_Ab_oriented(k,2,i);
				Epos_Ab_oriented(k,4,i) = coords_Ab_oriented(k,3,i);
				Epos_Ab_oriented(k,5,i) = coords_Ab_oriented(k,4,i);
				Epos_Ab_oriented(k,3,i) = coords_Ab_oriented(k,5,i);
			}
		}

		//Put oriented coords in the pose
		//
		Ab_oriented_pose.set_coords( ideal_pose, Epos_Ab_oriented, coords_Ab_oriented, check_missing );

		//
		//make a pose of Ab (all-atom) + scaff_w_graft (all-gly)
		// ( this can be revised to use construct_pose_complex_from_p1_p2 )
		// must test it out to make sure gives same result
		//

		Pose Ab_scaff_w_graft_gly_pose;
		make_Ab_scaff_w_graft_gly_pose( Ab_oriented_pose,
																		scaff_w_graft_gly_pose,
																		Ab_scaff_w_graft_gly_pose );
		//pose_to_misc( Ab_scaff_w_graft_gly_pose );
		if ( output_debug_pdbs ) {
			std::string tmp_filename = "Ab_scaff_w_graft_gly_pose_" + string_of( i_scaff ) + ".pdb";
			Ab_scaff_w_graft_gly_pose.dump_pdb( tmp_filename );
		}

		//
		//Score the Ab_scaff_w_graft_gly_pose to get fa_rep
		//

		Ab_scaff_w_graft_gly_pose.score( weight_map );
		float const fa_rep_Ab_scaff_w_graft_gly ( Ab_scaff_w_graft_gly_pose.get_0D_score( FA_REP ) );
		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << Ab_scaff_w_graft_gly_pose.show_scores() << std::endl;
		}
    ////////////////////////////////////////////////////////////////////
    //measure "inter" clash between Ab all-atom and scaff_w_graft backbone
    ////////////////////////////////////////////////////////////////////

		float const fa_rep_inter_gly ( fa_rep_Ab_scaff_w_graft_gly - fa_rep_Ab - fa_rep_scaff_w_graft_gly );

		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "fa_rep_inter_gly, fa_rep_Ab_scaff_w_graft_gly, fa_rep_Ab, fa_rep_scaff_w_graft_gly "
								<< fa_rep_inter_gly << " " << fa_rep_Ab_scaff_w_graft_gly << " "
								<< fa_rep_Ab << " " << fa_rep_scaff_w_graft_gly << std::endl;
		}

		///////////////////////////////////////////////////////////////////////////////
		//Ask if this inter clash is below the loose threshold
		//Using a loose threshold initially prior to closing the chainbreak(s)
		//
		float const max_fa_rep_inter_gly_before_close_default( 3000. ); //needs to be tested
		float max_fa_rep_inter_gly_before_close;
		realafteroption("max_fa_rep_inter_gly_before_close",max_fa_rep_inter_gly_before_close_default,max_fa_rep_inter_gly_before_close);

		//NOTE NOTE NOTE: this is where we could enter into the rb_move procedure
		//use rb_moves for cases in which intra_clash is fine but inter_clash
		//needs to be improved.
		//to test we need another, slightly looser threshold for inter_clash
		//if max_fa_rep_inter_gly_before_close < fa_inter < max_fa_rep_inter_gly_before_close_II
		//then we try rb_moves to see if we can maintain low intra clash
		//but also get low inter clash
		//
		//if we do try them, then we need to record that rb_moves have been used
		bool const rb_moves_used = false;


		//
		// if not satisfy loose_max, continue to next graft/scaff combo
		//
		if ( fa_rep_inter_gly > max_fa_rep_inter_gly_before_close ) {
			std::cout << "fa_rep_inter_gly: " << fa_rep_inter_gly << "is higher than cutoff: " << max_fa_rep_inter_gly_before_close
								<< "going to next graft/scaff combo" << std::endl;

			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

			continue;
		}

		///////////////////////////////////////////////////////////////////////////////
		//If we get here then we are ready to carry out closure/design protocols
		//
		//Protocol I : close first (using all-gly), then design, then refine
		//Protocol II (not implemented): design first, then close...then design again, then refine?
		//
		//We will generate about 100 decoys for each protocol,
		//All decoys will pass through a final trimming design stage
		//then we output the best 3 or 10 decoys
		//
		// To evaluate the best decoys we need to weigh
		//    (1) epitope-scaffold/Ab intE
		//    (2) ddg_stability of epitope-scaffold relative to native
		//
		//For closure, we'll use Chu's pose_loops protocol
		//this has the advantage that it can accomodate more aggressive
		//grafting in which neither side is fully closed.
		//

    //
    //Parameters controlling how many graft or scaff
    //residues can move to close the graft
    //
    int n_graft_res_allowed_bbmove_nterm = 0;
    int n_graft_res_allowed_bbmove_cterm = 0;
    int n_scaff_gap_res_allowed_bbmove_nterm = 3;
    int n_scaff_gap_res_allowed_bbmove_cterm = 3;
    intafteroption("n_graft_res_allowed_bbmove_nterm",0,n_graft_res_allowed_bbmove_nterm);
    intafteroption("n_graft_res_allowed_bbmove_cterm",0,n_graft_res_allowed_bbmove_cterm);
    intafteroption("n_scaff_gap_res_allowed_bbmove_nterm",3,n_scaff_gap_res_allowed_bbmove_nterm);
    intafteroption("n_scaff_gap_res_allowed_bbmove_cterm",3,n_scaff_gap_res_allowed_bbmove_cterm);
    if ( n_scaff_gap_res_allowed_bbmove_nterm < 1 ) {
			std::cout << "command-line input error: n_scaff_gap_res_allowed_bbmove_nterm MUST be >= 1" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }
    if ( n_scaff_gap_res_allowed_bbmove_cterm < 1 ) {
			std::cout << "command-line input error: n_scaff_gap_res_allowed_bbmove_cterm MUST be >= 1" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }


		//
		//Protocol I : close using all-gly, then design, then refine
		//

		//
		//if we used rb_moves (above, to reduce inter-clash)
		//   then we have to close on both sides of graft
		//else if no rb_moves but we do want lever_moves
		//   then we move on both sides but only close on one side
		//else if no rb_moves and no lever_moves,
		//   then we move on one side to close on that side
		//
		//
		int n_loops;
		if ( rb_moves_used ) {
			n_loops = 2;
		} else {
			if ( graft_align_system == "S" || graft_align_system == "E" ) { // if we get rb orientation of epitope via superposition then we have chainbreaks at both ends of graft.
				n_loops = 2;
			} else {
				n_loops = 1;
			}
		}

		//Use lever moves or not?
		//

		bool use_lever_moves ( true );
		if ( truefalseoption("no_lever_moves") ) use_lever_moves = false;
		if ( n_loops == 2 ) use_lever_moves = false;

		//
		//Setup loop info
		//
		FArray1D_int loop_begin( n_loops );
		FArray1D_int loop_end( n_loops );
		FArray1D_int cut_point( n_loops );

		//
		//list_disallow_bbmove is used with lever moves
		//to set allow_bbmove=false at positions within graft
		//
		//this is necessary with lever moves b/c the loop
		//region spans across the entire gap
		//
		int nres_disallow_bbmove = 0;
		FArray1D_int list_disallow_bbmove( 0 );//initially dimension 0

		//if graft has diff num of residues than native scaffold gap,
		//then residue numbering on cterminal side of gap will change
		//
		int const resnum_shift_due_to_graft = nres_scaff_w_graft - nres_scaff;

		setup_loops( rb_moves_used,
								 use_lever_moves,
								 graft_dir,
								 graft_align_system,
								 scaff_gap_resnum_1,
								 scaff_gap_resnum_2,
								 n_graft_res_allowed_bbmove_nterm,
								 n_graft_res_allowed_bbmove_cterm,
								 n_scaff_gap_res_allowed_bbmove_nterm,
								 n_scaff_gap_res_allowed_bbmove_cterm,
								 resnum_shift_due_to_graft,
								 n_loops,
								 loop_begin,
								 loop_end,
								 cut_point,
								 nres_disallow_bbmove,
								 list_disallow_bbmove
								 );
		///////////////////////////////////////////////////////////////////////////////
		//close the loops
		///////////////////////////////////////////////////////////////////////////////

		float chainbreak_score = 1000.;
		float max_chainbreak_score_after_close = 0.1;
		realafteroption("max_chainbreak_score_after_close",max_chainbreak_score_after_close,max_chainbreak_score_after_close);

		int max_closure_attempts = 3;
		intafteroption("max_closure_attempts",max_closure_attempts,max_closure_attempts);

		Pose scaff_w_closed_graft_gly_pose;
		int count_closure_attempts = 0;
		while ( chainbreak_score > max_chainbreak_score_after_close && count_closure_attempts < max_closure_attempts ) {

			scaff_w_closed_graft_gly_pose = scaff_w_graft_gly_pose;

			close_graft( n_loops,
									 loop_begin,
									 loop_end,
									 cut_point,
									 nres_disallow_bbmove,
									 list_disallow_bbmove,
									 scaff_w_closed_graft_gly_pose,
									 chainbreak_score );

			++count_closure_attempts;
			if ( runlevel_ns::runlevel > runlevel_ns::silent ) {
				std::cout << "Finished closure attempt # " << count_closure_attempts << " with chainbreak_score = " << chainbreak_score << std::endl;
				if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
					if ( chainbreak_score > max_chainbreak_score_after_close && count_closure_attempts < max_closure_attempts ) {
						std::cout << "chainbreak_score > max_chainbreak_score_after_close....try again " << std::endl;
					}
				}
			}
		}

		if ( chainbreak_score > max_chainbreak_score_after_close ) {
			std::cout << "After max_closure_attempts (" << max_closure_attempts
								<< "), chainbreak_score > max_chainbreak_score_after_close....skip to next "
								<< std::endl;

			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

			continue;
		} else {
			std::cout << "chainbreak score ok ! after " << count_closure_attempts << " closure attempts" << std::endl;
		}

    //
    //Quick measurement of how far the graft moves in space during closure
    //
    float graft_move_distance_during_closure = 0.0;
    {
      int const ires_center_of_graft = int( 0.5*( scaff_gap_resnum_1 + scaff_gap_resnum_2+resnum_shift_due_to_graft ));
      graft_move_distance_during_closure =
        distance( numeric::xyzVector_float( &( scaff_w_graft_gly_pose.full_coord()(1,2,ires_center_of_graft))),
                  numeric::xyzVector_float( &( scaff_w_closed_graft_gly_pose.full_coord()(1,2,ires_center_of_graft))));
    }

		//
		//Check that omegas are not crazy.
		//
    {
      for ( int ires = 1; ires <= nres_scaff_w_graft; ires++ ) {
        if ( std::abs(scaff_w_closed_graft_gly_pose.omega( ires )) < 175.) {
					std::cout << "WARNING bad omega in scaff_w_closed_graft_gly_pose. ires, omega = " << ires << " " << scaff_w_closed_graft_gly_pose.omega( ires ) << std::endl;
        }
      }
    }


    ///////////////////////////////////////////////////////////////////////////////
		//measure "inter" clash again, now that we have closed the graft.
    ///////////////////////////////////////////////////////////////////////////////


		scaff_w_closed_graft_gly_pose.score( weight_map );
		float const fa_rep_scaff_w_closed_graft_gly ( scaff_w_closed_graft_gly_pose.get_0D_score( FA_REP ) );
		if ( runlevel_ns::runlevel > runlevel_ns::silent ) {
			std::cout << "fa_rep_scaff_w_closed_graft_gly = " << fa_rep_scaff_w_closed_graft_gly << std::endl;
		}

		//////////////////////////////////////////////////////////////////////////////////////////////
		///////////  NEED TO MOVE ANTIBODY TO NEW LOCATION HERE !!! //////////////////////////////////
		///// ( during closure, graft has moved relative to scaffold, so Ab must move with graft ) ///
		//////////////////////////////////////////////////////////////////////////////////////////////

		//
		//make Ab_scaff_w_closed_graft_gly_pose to measure inter clash
		//

		//////////////////////////////////////////////////////////////////////////////////////////////////
		//remake the complex of Ab-scaff...graft has moved so Ab must move same way, relative to scaff
		//////////////////////////////////////////////////////////////////////////////////////////////////

		Pose Ab_scaff_w_closed_graft_gly_pose;
		Pose Ab_oriented_after_closure_pose;
		int const resnum_first_disallow_bbmove = scaff_gap_resnum_1 + n_graft_res_allowed_bbmove_nterm;

		get_Ab_scaff_refine_pose( Ab_scaff_w_graft_gly_pose,        //Ab_scaff_design_pose
															Ab_oriented_pose,                 //Ab_design_pose
															scaff_w_closed_graft_gly_pose,    //scaff_refine_pose
															resnum_first_disallow_bbmove,
															Ab_oriented_after_closure_pose /*output*/, //Ab_refine_pose
															Ab_scaff_w_closed_graft_gly_pose /*output*/);  //Ab_scaff_refine_pose


		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "Ab_scaff_w_closed_graft_gly_pose.total_residue() = " << Ab_scaff_w_closed_graft_gly_pose.total_residue() << std::endl;
			std::cout << " nres_Ab, nres_scaff_w_graft " << nres_Ab << " " << nres_scaff_w_graft << std::endl;
		}

		Ab_scaff_w_closed_graft_gly_pose.new_score_pose(); //clear all existing score info
		Ab_scaff_w_closed_graft_gly_pose.score( weight_map );
		std::cout << "scores: " << Ab_scaff_w_closed_graft_gly_pose.show_scores() << std::endl;

		//		Ab_scaff_w_closed_graft_gly_pose.dump_pdb("Ab_scaff_w_closed_graft_gly_pose_keytest.pdb");

		float const fa_rep_Ab_scaff_w_closed_graft_gly ( Ab_scaff_w_closed_graft_gly_pose.get_0D_score( FA_REP ) );
		float const fa_rep_inter_gly_after_close ( fa_rep_Ab_scaff_w_closed_graft_gly - fa_rep_Ab - fa_rep_scaff_w_closed_graft_gly );

		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "fa_rep_inter_gly_after_close, fa_rep_Ab_scaff_w_closed_graft_gly, fa_rep_Ab, fa_rep_scaff_w_closed_graft_gly " << std::endl;
			std::cout << fa_rep_inter_gly_after_close << " " << fa_rep_Ab_scaff_w_closed_graft_gly << " "
								<< fa_rep_Ab << " " << fa_rep_scaff_w_closed_graft_gly << std::endl;
		}

    ///////////////////////////////////////////////////////////////////////////////
    //Ask if this inter clash is below the tight ( post-close ) threshold
    //
    float max_fa_rep_inter_gly_after_close( 1000. ); //needs to be tested
		realafteroption("max_fa_rep_inter_gly_after_close",max_fa_rep_inter_gly_after_close,max_fa_rep_inter_gly_after_close);

    //
    // if not satisfy, continue to next graft/scaff combo
    //
    if ( fa_rep_inter_gly_after_close > max_fa_rep_inter_gly_after_close ) {
			std::cout << "fa_rep_inter_gly_after_close: " << fa_rep_inter_gly_after_close
								<< "is higher than cutoff: " << max_fa_rep_inter_gly_after_close
                << "going to next graft/scaff combo" << std::endl;

			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

      continue;
		}

		//*******************************************************************************************
		//  !!!!!!!!!!!!!!! ********************* !!!!!!!!!!!!!!!!!!
		//  WE SHOULD MEASURE INTRA CLASH AFTER CLOSING TOO...
		//  IT MAY BE TOO LARGE !!!! ( compare to clash in wt scaffold )
		//*******************************************************************************************

		if ( output_debug_pdbs ) Ab_oriented_pose.dump_pdb("Ab_oriented1.pdb");

		/////////////////////////////////////////////////////////////////////////////////////////////////////
		//make a new Ab_oriented pose that includes movement of graft during closure
		//( we are just over-writing a pre-existing pose )
		////////////////////////////////////////////////////////////////////////////////////////////////////
		copy_segment_to_new_pose( Ab_scaff_w_closed_graft_gly_pose /*src_pose*/,
															1 /*begin_src*/,
															nres_Ab /*end_src*/,
															Ab_oriented_pose /*pose_out*/);
		if ( output_debug_pdbs ) Ab_oriented_pose.dump_pdb("Ab_oriented2.pdb");


		///////////////////////////////////////////////////////////////////////////////////////////////////
		//Make a new Mgl that transforms nat_cmplx coords for Ab,epitope to the coordinate system of the newly closed graft.
		//( this over-writes the existing Mgl that is from nat_cmplx coords to unclosed-graft)
		//////////////////////////////////////////////////////////////////////////////////////////////////
		FArray1D_float coords_cent( 3 ),coords_forw( 3 ),coords_back( 3 );
		FArray1D_float coords_new_cent( 3 ),coords_new_forw( 3 ),coords_new_back( 3 );
		for ( int k = 1; k <= 3; k++ ) {
			coords_back( k ) = nat_cmplx_pose.full_coord()( k, 1, 1 );//N
			coords_cent( k ) = nat_cmplx_pose.full_coord()( k, 2, 1 );//CA
			coords_forw( k ) = nat_cmplx_pose.full_coord()( k, 4, 1 );//C
			coords_new_back( k ) = Ab_oriented_pose.full_coord()( k, 1, 1 );//N
			coords_new_cent( k ) = Ab_oriented_pose.full_coord()( k, 2, 1 );//CA
			coords_new_forw( k ) = Ab_oriented_pose.full_coord()( k, 4, 1 );//C
		}

		get_GL_matrix( coords_forw,   coords_cent,   coords_back,
									 coords_new_forw, coords_new_cent, coords_new_back, Mgl );


		////////////////////////////////////////////////////////////////////
		//
		//make a pose of oriented fullatom peptide to get native rotamers
		//in the correct orientation wrt scaffold so they can be transferred
		//
		////////////////////////////////////////////////////////////////////

		////////////////////////////////////////////////////////////////////
		//make a copy of the peptide coordinates
		//then re-orient these coords using Mgl computed above for peptide
		//

		FArray3D_float coords_pep_oriented ( nat_pep_pose.full_coord() );

		// apply the transformation to the peptide
		// apply to coords_pep_oriented
		for ( int ires = 1; ires <= nres_nat_pep; ires++ ) {
			int const aa( nat_pep_pose.res( ires ) );
			int const aav( nat_pep_pose.res_variant( ires ));
			for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
				GL_rot_in_place( Mgl, coords_pep_oriented( 1, iatom, ires ) );
			}
		}

		////////////////////////////////////////////////////////////////////
		//Here setup the new pose

		Pose pep_oriented_pose;
		pep_oriented_pose.simple_fold_tree( nres_nat_pep );
		pep_oriented_pose.set_fullatom_flag( true, false );// set fullatom and do not repack
		FArray3D_float Epos_pep_oriented( 3, param::MAX_POS, nres_nat_pep );//MAX_POS = 5
		for ( int i = 1; i <= nres_nat_pep; ++i ) {
			pep_oriented_pose.set_phi        ( i, nat_pep_pose.phi(i) );
			pep_oriented_pose.set_psi        ( i, nat_pep_pose.psi(i) );
			pep_oriented_pose.set_omega      ( i, nat_pep_pose.omega(i) );
			pep_oriented_pose.set_secstruct  ( i, nat_pep_pose.secstruct(i) );
			pep_oriented_pose.set_name       ( i, nat_pep_pose.name(i) );
			pep_oriented_pose.set_res        ( i, nat_pep_pose.res( i ) );
			pep_oriented_pose.set_res_variant( i, nat_pep_pose.res_variant( i ) );
			for ( int k = 1; k <= 3; ++k ) {
				Epos_pep_oriented(k,1,i) = coords_pep_oriented(k,1,i);
				Epos_pep_oriented(k,2,i) = coords_pep_oriented(k,2,i);
				Epos_pep_oriented(k,4,i) = coords_pep_oriented(k,3,i);
				Epos_pep_oriented(k,5,i) = coords_pep_oriented(k,4,i);
				Epos_pep_oriented(k,3,i) = coords_pep_oriented(k,5,i);
			}
		}
		pep_oriented_pose.set_coords( ideal_pose, Epos_pep_oriented, coords_pep_oriented, check_missing );
		//		pep_oriented_pose.dump_pdb( "pep_oriented_pose.pdb" );

		//
		//graft with native rotamers
		// ( this is shorter than the full peptide )
		//
		Pose graft_pose;
		graft_pose.simple_fold_tree( nres_graft );
		graft_pose.set_fullatom_flag( true, false );// set fullatom and do not repack
		copy_segment_to_new_pose( pep_oriented_pose /*src_pose*/,
															graft_resnum_1 /*begin_src*/,
															graft_resnum_2 /*end_src*/,
															graft_pose /*pose_out*/);

		//		graft_pose.dump_pdb( "graft_pose.pdb" );

		//
		//Now we must put on epitope and scaff native rotamers
		//and then identify design positions.
		//

		Pose scaff_w_closed_graft_natro_pose;
		scaff_w_closed_graft_natro_pose = scaff_w_closed_graft_gly_pose;


    ////////////////////////////////////////////////////////////////////////////////////////////////
		//
		//First transfer all rotamers from scaffold
		//
    ////////////////////////////////////////////////////////////////////////////////////////////////

		FArray1D_bool natro_scaff( nres_scaff, false );
		FArray1D_bool natro_scaff_w_graft( nres_scaff_w_graft, false );

		natro_scaff_w_graft = false; //initialize all positions to false
		for ( int ires = 1; ires <= scaff_gap_resnum_1 - 1; ires++ ) {
			natro_scaff( ires ) = true;
			natro_scaff_w_graft( ires ) = true;
		}
		for ( int ires = scaff_gap_resnum_2 + 1; ires <= nres_scaff; ires++ ) {
			natro_scaff( ires ) = true;
		}
		for ( int ires = scaff_gap_resnum_2 + 1 + resnum_shift_due_to_graft; ires <= nres_scaff_w_graft; ires++ ) {
			natro_scaff_w_graft( ires ) = true;
		}

		//logic check
		{
			int nres_scaff_test = scaff_pose.total_residue();
			int nres_scaff_w_graft_test = scaff_w_closed_graft_natro_pose.total_residue();
			if ( !(nres_scaff_test == nres_scaff ) ) {
				std::cout << "Error nres_scaff_test != nres_scaff..." << nres_scaff_test << " " << nres_scaff << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			if ( !(nres_scaff_w_graft_test == nres_scaff_w_graft ) ) {
				std::cout << "Error nres_scaff_w_graft_test != nres_scaff_w_graft..." << nres_scaff_w_graft_test << " " << nres_scaff_w_graft << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
		//done logic check
		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "about to transfer rots to scaffold" << std::endl;
		}
		transfer_native_rotamers( scaff_pose, natro_scaff, natro_scaff_w_graft, scaff_w_closed_graft_natro_pose );

		if ( output_debug_pdbs ) scaff_w_closed_graft_natro_pose.dump_pdb("scaff_w_closed_graft_natro_pose_scaffrotonly.pdb");

		if ( runlevel_ns::runlevel > runlevel_ns::silent ) {
			std::cout << "done transferring rots to scaffold" << std::endl;
		}
    ////////////////////////////////////////////////////////////////////////////////////////////////
		//
		//Now transfer select rotamers within the graft
		//    Only transfer rotamers that interact with Ab.
		//    These positions have been specified by input file, are stored in bool array natro_in_epitope.
		//
		//Note: the graft termini WILL  have natro !!
    ////////////////////////////////////////////////////////////////////////////////////////////////

		//
    //What subset of natro_in_epitope will be used in graft?
    //

		//keep track of epitope natro positions within the graft alone
    int n_natro_in_graft = 0;
    FArray1D_int list_natro_in_graft( n_natro_in_graft );//initially dimension 0
		FArray1D_bool natro_graft( nres_graft, false );

		//keep track of epitope natro positions within the scaff_w_graft
		FArray1D_int list_natro_epitope_in_scaff_w_graft( n_natro_in_graft );//initially dimension 0
		FArray1D_bool natro_epitope_in_scaff_w_graft( nres_scaff_w_graft, false );

		//graft goes from
		//    graft_resnum_1          to  graft_resnum_2                                   in graft numbering
		//    scaff_gap_resnum_1      to  scaff_gap_resnum_2 + resnum_shift_due_to_graft   in scaff_w_graft numbering

		for ( int ires = 1; ires <= nres_graft; ires++ ) {
			//ires_epitope tells position within epitope
			int ires_epitope       = ires - 1 + graft_resnum_1;
			int ires_scaff_w_graft = ires - 1 + scaff_gap_resnum_1;

			if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
				std::cout << " ires, ires_epitope, ires_scaff_w_graft, natro_in_epitope(ires_epitope) [ " << ires << " ][ "
									<< ires_epitope << " ][ " << ires_scaff_w_graft << " ][ " << natro_in_epitope( ires_epitope ) << " ]" << std::endl;
			}
			if ( natro_in_epitope( ires_epitope ) ) {
				++n_natro_in_graft;

				//logic check
				if ( ires_scaff_w_graft > scaff_gap_resnum_2 + resnum_shift_due_to_graft ) {
					std::cout << "Error setting up natro_epitope_in_scaff_w_graft " << std::endl;
					std::cout << "ires_scaff_w_graft > scaff_gap_resnum_2 + resnum_shift_due_to_graft" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}

				//keep track of epitope natro positions within the graft alone
				list_natro_in_graft.redimension( n_natro_in_graft );
				list_natro_in_graft( n_natro_in_graft ) = ires;
				natro_graft( ires ) = true;

				//keep track of epitope natro positions within the scaff_w_graft
				list_natro_epitope_in_scaff_w_graft.redimension( n_natro_in_graft );
				list_natro_epitope_in_scaff_w_graft( n_natro_in_graft ) = ires_scaff_w_graft;
				natro_epitope_in_scaff_w_graft( ires_scaff_w_graft ) = true;

			}

			//Note the positions that are not natro_in_epitope will just be GLY unless we make them something else.
			// else {
			//	++n_ala_in_graft;
			//	list_ala_in_graft.redimension( n_ala_in_graft );
			//	list_ala_in_graft( n_ala_in_graft ) = ires;
			//			}
		}

		if ( runlevel_ns::runlevel > runlevel_ns::silent ) {
			std::cout << "about to transfer rots to graft" << std::endl;
		}
		transfer_native_rotamers( graft_pose, natro_graft, natro_epitope_in_scaff_w_graft, scaff_w_closed_graft_natro_pose );

		if ( output_debug_pdbs ) scaff_w_closed_graft_natro_pose.dump_pdb("scaff_w_closed_graft_natro_pose.pdb");

    ////////////////////////////////////////////////////////////////////////////////////////////////
		//Now construct a pose of scaff_w_graft_natro and Ab_oriented -- this is the "template" for design
		// Ab is first !!
    ////////////////////////////////////////////////////////////////////////////////////////////////

		Pose Ab_scaff_w_graft_natro_pose;
		construct_pose_complex_from_p1_p2( Ab_oriented_pose, scaff_w_closed_graft_natro_pose, Ab_scaff_w_graft_natro_pose );

		//NOTE! in the add_on_design module I use both nres_Ab_scaff and nres_Ab_scaff_w_graft
		//But here I am use nres_Ab_scaff to mean nres_Ab_scaff_w_graft
		//Crazy.
		int const nres_Ab_scaff = Ab_scaff_w_graft_natro_pose.total_residue();

		//logic check
		//
		if ( !( nres_Ab_scaff == nres_Ab + nres_scaff_w_gap + nres_graft ) ) {
			std::cout << "Logic Error nres_Ab_scaff !=nres_Ab + nres_scaff_w_gap + nres_graft" << std::endl;
			std::cout << "nres_Ab_scaff = " << nres_Ab_scaff << std::endl;
			std::cout << "nres_Ab + nres_scaff_w_gap + nres_graft = " << nres_Ab + nres_scaff_w_gap + nres_graft << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		if ( output_debug_pdbs ) Ab_scaff_w_graft_natro_pose.dump_pdb("Ab_scaff_w_graft_natro_pose.pdb");

    ////////////////////////////////////////////////////////////////////////////////////////////////
		//Identify design positions
    ////////////////////////////////////////////////////////////////////////////////////////////////

		// NOTE intra design positions are those near any part of the "loop" region that allows bbmove

		//
		//figure out maximal range of resnums over which backbone was allowed to move
		//
		int begin_move_bb = 999999;//large number
		int end_move_bb = 0; //small number
		for ( int i_loop = 1; i_loop <= n_loops; i_loop++ ) {
			if ( loop_begin( i_loop ) < begin_move_bb ) begin_move_bb = loop_begin( i_loop );
			if ( loop_end( i_loop ) > end_move_bb ) end_move_bb = loop_end( i_loop );
		}


		FArray2D_bool design_matrix_Ab_scaff( param::MAX_AA(), nres_Ab_scaff, false );
		prepare_grafting_protocol_design_matrix( Ab_scaff_w_graft_natro_pose,     //input
																						 nres_Ab,                   //input
																						 begin_move_bb,             //beginning of move_bb region
																						 end_move_bb,               //end       of move_bb region
																						 natro_epitope_in_scaff_w_graft,  //input  epitope_natro_start_in_scaff,
																						 design_matrix_Ab_scaff );        //output

		{
			bool found( false );
			for ( int i = 1; i <= nres_Ab_scaff; i++ ) {
				for ( int aa = 1; aa <= param::MAX_AA(); aa++ ) {
					if ( design_matrix_Ab_scaff( aa, i ) ) {
						if (runlevel_ns::runlevel > runlevel_ns::quiet) {
							std::cout << "design_matrix_Ab_scaff is true at ires, aa " << i << " " << aa << " " << SS(param_aa::param_aa_data::aa_name3(aa)) << std::endl;
						}
						found = true;
					}
				}
			}
			if ( !found ) std::cout << "nothing is true in design_matrix_Ab_scaff!!!" << std::endl;
		}

		//
		//how many design iterations at each position?
		//( default is 3 )
		int n_design_iterations = 3;
		intafteroption( "n_design_iterations", n_design_iterations, n_design_iterations );

		//
		//Get scores for native scaffold so we can compute ddg_stability for scaffold
		// *** We should probably just repack the scaff ***

		scaff_pose.score( weight_map );
		float score_scaff_native   ( scaff_pose.get_0D_score( SCORE ) );
		float fa_rep_scaff_native  ( scaff_pose.get_0D_score( FA_REP ) );
		float fa_atr_scaff_native  ( scaff_pose.get_0D_score( FA_ATR ) );
		float fa_sol_scaff_native  ( scaff_pose.get_0D_score( FA_SOL ) );
		float fa_dun_scaff_native  ( scaff_pose.get_0D_score( FA_DUN ) );
		float hbsc_scaff_native    ( scaff_pose.get_0D_score( HB_SC ) );
		float fa_prob_scaff_native ( scaff_pose.get_0D_score( FA_PROB ) );

		//
    //arrays to store score values for each design iteration
    //

		//Only keep the Ab_scaff pose and store total energy in an array
		//rank the total E at end
		//output best x by rank
		//for each output, extract scaff and Ab poses and
		//compute ddg_bind and ddg_stability on the fly
		//

		//total scores
		//
    FArray1D_float score_Ab_scaff_design( n_design_iterations, 0.0 );
    FArray1D_int rank_to_index_score_Ab_scaff_design( n_design_iterations, 0 );

    //vector or FArray of pointers to poses
    //
    //    std::vector< Pose* > pose_list;
    FArray1D< Pose* > pose_array_Ab_scaff_design( n_design_iterations );

    for ( int i_design = 1; i_design <= n_design_iterations; i_design++ ) {

			Pose Ab_scaff_design_pose;
			Ab_scaff_design_pose = Ab_scaff_w_graft_natro_pose;
			design_using_design_matrix( Ab_scaff_design_pose, design_matrix_Ab_scaff );

			//save score for  this design
			//
			Ab_scaff_design_pose.score( weight_map );
      score_Ab_scaff_design( i_design ) = Ab_scaff_design_pose.get_0D_score( SCORE );
			std::cout << "i_design, score " << i_design << " " << score_Ab_scaff_design( i_design ) << std::endl;

			//dump design pdb
			if ( output_debug_pdbs ) Ab_scaff_design_pose.dump_pdb( "Ab_scaff_design_pose_" + string_of( i_scaff ) + "_" + string_of( i_design ) + ".pdb" );

      //allocate memory for new poses
      //
      Pose * pose_ptr_Ab_scaff_design( new Pose );

      //attach pointers to actual poses
      //either from misc
      //pose_from_misc( *pose_ptr, true, false, true );
      //or get it directly
      *pose_ptr_Ab_scaff_design = Ab_scaff_design_pose;

      //store the pose pointer in an array
      //
      pose_array_Ab_scaff_design( i_design ) = pose_ptr_Ab_scaff_design;

			//initialize the rank_to_index arrays
			//
			rank_to_index_score_Ab_scaff_design( i_design ) = i_design;

    }//design iterations

    //rank the ddg values
    //( this sets rank_to_index... in ascending order )
    rank( n_design_iterations, score_Ab_scaff_design, rank_to_index_score_Ab_scaff_design );

		/*
    int best_index_Ab_scaff_design = rank_to_index_score_Ab_scaff_design( 1 );
    float best_score_Ab_scaff_design = score_Ab_scaff_design( best_index_Ab_scaff_design );

		std::cout << "best_index, best_score for Ab_scaff_design: "
              << best_index_Ab_scaff_design << " " << best_score_Ab_scaff_design << std::endl;

    //Extract design pose with best score and repack pose with best score
    //extract from std vector
    //    Pose & best_pose( *pose_list[ best_index] ); // 0->
    //extract from FArray
    Pose & best_Ab_scaff_design_pose( *pose_array_Ab_scaff_design( best_index_Ab_scaff_design ) ); // 1->

    best_Ab_scaff_design_pose.score( weight_map );
    best_score_Ab_scaff_design  = best_Ab_scaff_design_pose.get_0D_score( SCORE );
    float  fa_rep_Ab_scaff_design ( best_Ab_scaff_design_pose.get_0D_score( FA_REP ) );
    float  fa_atr_Ab_scaff_design ( best_Ab_scaff_design_pose.get_0D_score( FA_ATR ) );
    float  fa_sol_Ab_scaff_design ( best_Ab_scaff_design_pose.get_0D_score( FA_SOL ) );
    float  fa_dun_Ab_scaff_design ( best_Ab_scaff_design_pose.get_0D_score( FA_DUN ) );
    float  hbsc_Ab_scaff_design ( best_Ab_scaff_design_pose.get_0D_score( HB_SC ) );
    float  fa_prob_Ab_scaff_design ( best_Ab_scaff_design_pose.get_0D_score( FA_PROB ) );
		*/

		/*
			std::string graft_dir = "n2c";             // n2c or c2n...if n2c, then Nterm is superposed
			std::string graft_align_system = "N";     // N or CA or C
			int const graft_resnum_1 = 3; //671; //(low resnum in peptide we want to graft into a scaffold...in sequential numbering)
			int const graft_resnum_2 = 12; //680;  //( resnum_1 < resnum_2 always !)
			std::string scaff_pdb5code = "1z6na";
			int const	scaff_gap_resnum_1 = 138;               // n-term of gap in scaffold
			int const scaff_gap_resnum_2 = 147;               // c-term of gap in scaffold
		*/

		//
    //output design pdbs and scores, mutations
    //

		//		std::string filename_base = scaff_pdb5code + "_" + string_of( nres_scaff_w_graft ) + "_" +
		//	graft_dir + "_" + graft_align_system + "_" + string_of( graft_resnum_1 ) + "-" + string_of( graft_resnum_2 ) +
		//	"_" + string_of( scaff_gap_resnum_1 ) + "_" + string_of( scaff_gap_resnum_2 );

		std::string filename_base = scaff_base_filename + "_" + lead_zero_string_of( nres_scaff_w_graft, 4 ) + "_" +
			graft_dir + "_" + graft_align_system + "_" + lead_zero_string_of( graft_resnum_1, 3 ) + "-" + lead_zero_string_of( graft_resnum_2, 3 ) +
			"_" + lead_zero_string_of( scaff_gap_resnum_1, 4 ) + "-" + lead_zero_string_of( scaff_gap_resnum_2, 4 );

		//
		//How many designs to output?
		//

		int n_design_output = 1;
		intafteroption("n_design_output",n_design_output,n_design_output);

		if ( n_design_output > n_design_iterations ) {
			n_design_output = n_design_iterations;
			std::cout << " requested n_design_output is > n_design_iterations, resetting n_design_output" << std::endl;
		}

		for ( int i_rank = 1; i_rank <= n_design_output; i_rank++ ) {

			//note index corresponds to i_design
			int const i_design = rank_to_index_score_Ab_scaff_design( i_rank );

			Pose & Ab_scaff_out_pose( *pose_array_Ab_scaff_design( i_design ) ); // 1->

			//
			//Output energies: total E, ddg-bind, and ddg-stability for scaffold
			//

			Ab_scaff_out_pose.score( weight_map );
      float score_out = Ab_scaff_out_pose.get_0D_score( SCORE );
			if ( std::abs( score_out - score_Ab_scaff_design( i_design ) ) > 0.01 ) {
				std::cout << "!!!!!!!!!!!! WARNING WARNING WARNING WARNING !!!!!!!!!!!!!!!" << std::endl;
				std::cout << "ERROR score disagreement between score_out " << score_out
									<< " and score_Ab_scaff_design( i_design ) " << score_Ab_scaff_design( i_design ) << std::endl;
				std::cout << "!!!!!!!!!!!! WARNING WARNING WARNING WARNING !!!!!!!!!!!!!!!" << std::endl;
			}
      float  fa_rep_Ab_scaff_design  ( Ab_scaff_out_pose.get_0D_score( FA_REP ) );
      float  fa_atr_Ab_scaff_design  ( Ab_scaff_out_pose.get_0D_score( FA_ATR ) );
      float  fa_sol_Ab_scaff_design  ( Ab_scaff_out_pose.get_0D_score( FA_SOL ) );
      float  fa_dun_Ab_scaff_design  ( Ab_scaff_out_pose.get_0D_score( FA_DUN ) );
      float  hbsc_Ab_scaff_design    ( Ab_scaff_out_pose.get_0D_score( HB_SC ) );
      float  fa_prob_Ab_scaff_design ( Ab_scaff_out_pose.get_0D_score( FA_PROB ) );

			//
			//Here extract scaff and Ab poses and compute ddb_bind and ddg_stability
			//

			Pose scaff_design_pose;
			scaff_design_pose.simple_fold_tree( nres_scaff_w_graft );
			scaff_design_pose.set_fullatom_flag( true, false /*repack*/ );
			//			scaff_design_pose.copy_segment( nres_scaff_w_graft, Ab_scaff_out_pose, 1, nres_Ab+1 /*begin_src*/ );
			copy_segment_to_new_pose( Ab_scaff_out_pose /*src_pose*/,
																nres_Ab+1 /*begin_src*/,
																nres_Ab_scaff /*end_src*/,
																scaff_design_pose /*pose_out*/);
			scaff_design_pose.score( weight_map );
			//			scaff_design_pose.dump_pdb("scaff_design_pose.pdb");

			Pose Ab_design_pose;
			Ab_design_pose.simple_fold_tree( nres_Ab );
			Ab_design_pose.set_fullatom_flag( true, false /*repack*/ );
			//					Ab_design_pose.copy_segment( nres_Ab, Ab_scaff_out_pose, 1, 1 /*begin_src*/ );
			copy_segment_to_new_pose( Ab_scaff_out_pose /*src_pose*/,
																1                 /*begin_src*/,
																nres_Ab           /*end_src*/,
																Ab_design_pose    /*pose_out*/);
			Ab_design_pose.score( weight_map );
			//			Ab_design_pose.dump_pdb("Ab_design_pose.pdb");

			//extract scaff_design_pose and Ab_design_pose
			//score each

      scaff_design_pose.score( weight_map );
      float score_scaff_design   ( scaff_design_pose.get_0D_score( SCORE ) );
      float fa_rep_scaff_design  ( scaff_design_pose.get_0D_score( FA_REP ) );
      float fa_atr_scaff_design  ( scaff_design_pose.get_0D_score( FA_ATR ) );
      float fa_sol_scaff_design  ( scaff_design_pose.get_0D_score( FA_SOL ) );
      float fa_dun_scaff_design  ( scaff_design_pose.get_0D_score( FA_DUN ) );
      float hbsc_scaff_design    ( scaff_design_pose.get_0D_score( HB_SC ) );
      float fa_prob_scaff_design ( scaff_design_pose.get_0D_score( FA_PROB ) );

      Ab_design_pose.score( weight_map );
      float score_Ab_design   ( Ab_design_pose.get_0D_score( SCORE )  );
      float fa_rep_Ab_design  ( Ab_design_pose.get_0D_score( FA_REP ) );
      float fa_atr_Ab_design  ( Ab_design_pose.get_0D_score( FA_ATR ) );
      float fa_sol_Ab_design  ( Ab_design_pose.get_0D_score( FA_SOL ) );
      float fa_dun_Ab_design  ( Ab_design_pose.get_0D_score( FA_DUN ) );
      float hbsc_Ab_design    ( Ab_design_pose.get_0D_score( HB_SC ) );
      float fa_prob_Ab_design ( Ab_design_pose.get_0D_score( FA_PROB ) );

      //ddG-bind total score and components
      //
			float ddg_bind_Ab_scaff         = score_Ab_scaff_design( i_design ) - score_scaff_design - score_Ab_design;
      float ddg_bind_Ab_scaff_fa_rep  = fa_rep_Ab_scaff_design  - fa_rep_scaff_design  - fa_rep_Ab_design;
      float ddg_bind_Ab_scaff_fa_atr  = fa_atr_Ab_scaff_design  - fa_atr_scaff_design  - fa_atr_Ab_design;
      float ddg_bind_Ab_scaff_fa_sol  = fa_sol_Ab_scaff_design  - fa_sol_scaff_design  - fa_sol_Ab_design;
      float ddg_bind_Ab_scaff_fa_dun  = fa_dun_Ab_scaff_design  - fa_dun_scaff_design  - fa_dun_Ab_design;
      float ddg_bind_Ab_scaff_hbsc    = hbsc_Ab_scaff_design    - hbsc_scaff_design    - hbsc_Ab_design;
      float ddg_bind_Ab_scaff_fa_prob = fa_prob_Ab_scaff_design - fa_prob_scaff_design - fa_prob_Ab_design;

			//ddG-stability total score and components
			//
			float ddg_scaff                 = score_scaff_design   - score_scaff_native;
      float ddg_scaff_fa_rep          = fa_rep_scaff_design  - fa_rep_scaff_native;
      float ddg_scaff_fa_atr          = fa_atr_scaff_design  - fa_atr_scaff_native;
      float ddg_scaff_fa_sol          = fa_sol_scaff_design  - fa_sol_scaff_native;
      float ddg_scaff_fa_dun          = fa_dun_scaff_design  - fa_dun_scaff_native;
      float ddg_scaff_hbsc            = hbsc_scaff_design    - hbsc_scaff_native;
      float ddg_scaff_fa_prob         = fa_prob_scaff_design - fa_prob_scaff_native;

			std::string filename;
			std::string design_id_str = right_string_of( i_design, 3, '0' );
			int const i_refine = 0;
			std::string refine_id_str = right_string_of( i_refine, 3, '0' );
			std::string filename_base2 = filename_base + "_" + design_id_str + "_" + refine_id_str;
			get_filename_with_energies( filename_base2, score_Ab_scaff_design( i_design ), ddg_bind_Ab_scaff, filename );

			if ( output_pdbs ) {
				Ab_scaff_out_pose.pdb_info().set_use_pdb_numbering( true );
				Ab_scaff_out_pose.dump_pdb( files_paths::pdb_out_path + filename );
			}

			outfile   << SS( "ddg_bind_Ab_scaff: " )
								<< SS( "ides" )
								<< SS( "     ddg" )
								<< SS( "     atr" )
								<< SS( "     rep" )
								<< SS( "     sol" )
								<< SS( "     dun" )
								<< SS( "    hbsc" )
								<< SS( "    prob" )
								<< SS( "   scaff" )
								<< std::endl;

			outfile   << SS( "ddg_bind_Ab_scaff: " )
								<< I( 5, i_design ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff        ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_atr ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_rep ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_sol ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_dun ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_hbsc   ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_prob) << " "
								<< filename
								<< std::endl;

			outfile   << SS( "ddg_scaff        : " )
								<< SS( "ides" )
								<< SS( "     ddg" )
								<< SS( "     atr" )
								<< SS( "     rep" )
								<< SS( "     sol" )
								<< SS( "     dun" )
								<< SS( "    hbsc" )
								<< SS( "    prob" )
								<< SS( " cl_move" )
								<< SS( "   scaff" )
								<< std::endl;

			outfile   << SS( "ddg_scaff        : " )
								<< I( 5, i_design ) << " "
								<< F( 8, 2, ddg_scaff        ) << " "
								<< F( 8, 2, ddg_scaff_fa_atr ) << " "
								<< F( 8, 2, ddg_scaff_fa_rep ) << " "
								<< F( 8, 2, ddg_scaff_fa_sol ) << " "
								<< F( 8, 2, ddg_scaff_fa_dun ) << " "
								<< F( 8, 2, ddg_scaff_hbsc   ) << " "
								<< F( 8, 2, ddg_scaff_fa_prob) << " "
								<< F( 8, 2, graft_move_distance_during_closure ) << " "
								<< filename
								<< std::endl;

			//
			//output list of mutations
			//

			//regions in scaff to specify mutations:
			//1 -> scaff_gap_resum_1-1                                                  nterm of graft
			//scaff_gap_resnum_1 -> scaff_gap_resum_2+resnum_shift_due_to_graft         within graft
			//scaff_gap_resum_2+1+resnum_shift_due_to_graft -> nres_scaff_w_graft       cterm of graft

			for ( int ires = 1; ires < scaff_gap_resnum_1; ires++ ) {
				int ires_scaff_in_cmplx = ires + nres_Ab; //scaff positions in cmplx of Ab+scaff, Ab is first
				int const aa_wt = scaff_pose.res( ires );
				int const aa = Ab_scaff_out_pose.res( ires_scaff_in_cmplx );
				if ( aa != aa_wt ) {
					std::string const aa3_wt = param_aa::param_aa_data::aa_name3( aa_wt );
					std::string const aa3 = param_aa::param_aa_data::aa_name3( aa );
					outfile   << "Mutation:" << I( 4, ires ) << " " << SS( aa3_wt ) << " --> " << I( 4, ires_scaff_in_cmplx ) << SS( aa3 ) << " "
										<< filename << " nterm_to_graft" << std::endl;
				}
			}

			//
			//*For now just consider all positions within the graft as a mutation*
			//
			//But we could make this better:
			//For outputting mutations we should in principle consider whether there is
			//an insertion or a deletion due to the graft
			//and whether the insertion/deletion is on the n or c side of the graft
			//then we can consider other positions as point mutations and it is possible that
			//some positions will remain WT
			//

			for ( int ires = scaff_gap_resnum_1; ires <= scaff_gap_resnum_2+resnum_shift_due_to_graft; ires++ ) {

				int ires_scaff_in_cmplx = ires + nres_Ab; //scaff positions in cmplx of Ab+scaff, Ab is first
				int const aa = Ab_scaff_out_pose.res( ires_scaff_in_cmplx );
					std::string const aa3 = param_aa::param_aa_data::aa_name3( aa );
					outfile   << "Mutation:" << SS( "XXXX" ) << " " << SS( "XXX" ) << " --> " << I( 4, ires_scaff_in_cmplx ) << SS( aa3 ) << " "
										<< filename << " within_graft" << std::endl;
			}

			for ( int ires = scaff_gap_resnum_2+1+resnum_shift_due_to_graft; ires <= nres_scaff_w_graft; ires++ ) {

				//ires_scaff is scaff positions in cmplx of Ab+scaff, Ab is first
				int ires_scaff_in_cmplx = ires + nres_Ab;
				int ires_scaff_no_graft = ires - resnum_shift_due_to_graft;//ires in wt scaffold


				int const aa_wt = scaff_pose.res( ires_scaff_no_graft );
				int const aa = Ab_scaff_out_pose.res( ires_scaff_in_cmplx );
				if ( aa != aa_wt ) {
					std::string const aa3_wt = param_aa::param_aa_data::aa_name3( aa_wt );
					std::string const aa3 = param_aa::param_aa_data::aa_name3( aa );
					outfile   << "Mutation:" << I( 4, ires ) << " " << SS( aa3_wt ) << " --> " << I( 4, ires_scaff_in_cmplx ) << SS( aa3 ) << " "
										<< filename << " graft_to_cterm" << std::endl;
				}
			}
		}

		//
		//loop refinement step
		//
		//---------- is it necessary?
		//---------- should we minimize the surrounding scaffold backbone ?
		//---------- will we need to relax/minimize the entire scaffold backbone ?


		//
		//make list of the unique designs
		//
		//Note this is stupid, we should avoid saving duplicate designs
		//in the pose_array above

		int n_unique_designs = 0;
		FArray1D_int list_unique_designs_by_rank( n_design_iterations, 0 );
		float Eprev = 99999.;
		for ( int i_rank = 1; i_rank <= n_design_iterations; i_rank++ ) {
			float Edes = score_Ab_scaff_design( i_rank );
			if ( Edes != Eprev ) {
				list_unique_designs_by_rank( ++n_unique_designs ) = i_rank;
			}
		}
		list_unique_designs_by_rank.redimension( n_unique_designs );

		//
		//How many designs to refine?
		//

		int n_designs_to_refine = 1;
		intafteroption( "n_designs_to_refine", n_designs_to_refine, n_designs_to_refine );

		if ( n_designs_to_refine > n_unique_designs ) {
			std::cout << "WARNING: n_designs_to_refine > n_unique_designs ... resetting n_designs_to_refine to n_unique_designs = " << n_unique_designs << std::endl;
			n_designs_to_refine = n_unique_designs;
		}

		for ( int i_design_to_refine = 1; i_design_to_refine <= n_designs_to_refine; i_design_to_refine++ ) {

			int i_rank = list_unique_designs_by_rank( i_design_to_refine );

			//index to the array of designed poses
			int const i_design = rank_to_index_score_Ab_scaff_design( i_rank );

			Pose & Ab_scaff_design_pose( *pose_array_Ab_scaff_design( i_design ) ); // 1->

			//
			//Output energies: total E, ddg-bind, and ddg-stability for scaffold
			//

			Ab_scaff_design_pose.score( weight_map );
      float score_out = Ab_scaff_design_pose.get_0D_score( SCORE );
			if ( std::abs( score_out - score_Ab_scaff_design( i_design ) ) > 0.01 ) {
				std::cout << "!!!!!!!!!!!! WARNING WARNING WARNING WARNING !!!!!!!!!!!!!!!" << std::endl;
				std::cout << "ERROR score disagreement between score_out " << score_out
									<< " and score_Ab_scaff_design( i_design ) " << score_Ab_scaff_design( i_design ) << std::endl;
				std::cout << "!!!!!!!!!!!! WARNING WARNING WARNING WARNING !!!!!!!!!!!!!!!" << std::endl;
			}
      float  fa_rep_Ab_scaff_design  ( Ab_scaff_design_pose.get_0D_score( FA_REP ) );
      float  fa_atr_Ab_scaff_design  ( Ab_scaff_design_pose.get_0D_score( FA_ATR ) );
      float  fa_sol_Ab_scaff_design  ( Ab_scaff_design_pose.get_0D_score( FA_SOL ) );
      float  fa_dun_Ab_scaff_design  ( Ab_scaff_design_pose.get_0D_score( FA_DUN ) );
      float  hbsc_Ab_scaff_design    ( Ab_scaff_design_pose.get_0D_score( HB_SC ) );
      float  fa_prob_Ab_scaff_design ( Ab_scaff_design_pose.get_0D_score( FA_PROB ) );

			//
			//Here extract scaff and Ab poses and compute ddb_bind and ddg_stability
			//

			Pose scaff_design_pose;
			scaff_design_pose.simple_fold_tree( nres_scaff_w_graft );
			scaff_design_pose.set_fullatom_flag( true, false /*repack*/ );
			//			scaff_design_pose.copy_segment( nres_scaff_w_graft, Ab_scaff_design_pose, 1, nres_Ab+1 /*begin_src*/ );
			copy_segment_to_new_pose( Ab_scaff_design_pose /*src_pose*/,
																nres_Ab+1 /*begin_src*/,
																nres_Ab_scaff /*end_src*/,
																scaff_design_pose /*pose_out*/);
			scaff_design_pose.score( weight_map );
			if ( output_debug_pdbs ) scaff_design_pose.dump_pdb("scaff_design_pose_before_refine_" + string_of( i_scaff ) + "_" + string_of( i_design ) + ".pdb");
			if ( output_debug_pdbs ) Ab_scaff_design_pose.dump_pdb("Ab_scaff_design_pose_before_refine_" + string_of( i_scaff ) + "_" + string_of( i_design ) + ".pdb");

			Pose Ab_design_pose;
			Ab_design_pose.simple_fold_tree( nres_Ab );
			Ab_design_pose.set_fullatom_flag( true, false /*repack*/ );
			//					Ab_design_pose.copy_segment( nres_Ab, Ab_scaff_design_pose, 1, 1 /*begin_src*/ );
			copy_segment_to_new_pose( Ab_scaff_design_pose /*src_pose*/,
																1                 /*begin_src*/,
																nres_Ab           /*end_src*/,
																Ab_design_pose    /*pose_out*/);
			Ab_design_pose.score( weight_map );
			//			Ab_design_pose.dump_pdb("Ab_design_pose.pdb");

			//
			//Now do the loop refinement or any other desired refinement
			//

			int n_refine_iterations = 1;
			intafteroption( "n_refine_iterations", n_refine_iterations, n_refine_iterations );

			int const max_refine_iterations = 50;

			if ( n_refine_iterations > max_refine_iterations ) {
				std::cout << "WARNING: requested n_refine_iterations > max_refine_iterations, resetting to max = " << max_refine_iterations << std::endl;
				n_refine_iterations = max_refine_iterations;
			}

			for ( int i_refine = 1; i_refine <= n_refine_iterations; i_refine++ ) {

				Pose scaff_refine_pose;
				scaff_refine_pose = scaff_design_pose;

				float chainbreak_score_refine = 1000.;
				close_graft( n_loops,
										 loop_begin,
										 loop_end,
										 cut_point,
										 nres_disallow_bbmove,
										 list_disallow_bbmove,
										 scaff_refine_pose,
										 chainbreak_score_refine );

				if ( chainbreak_score_refine > max_chainbreak_score_after_close ) {
					std::cout << "!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!" << std::endl;
					std::cout << "!!WARNING!! After refine: chainbreak_score_refine = " << chainbreak_score_refine << " > max_chainbreak_score_after_close" << std::endl;
					std::cout << "!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!" << std::endl;
				}

				if ( output_debug_pdbs ) scaff_refine_pose.dump_pdb( "scaff_refine_pose_" + string_of( i_design ) + "_" + string_of( i_refine ) +"_.pdb");

				//
				//remake the complex of Ab-scaff...graft has moved so Ab must move same way, relative to scaff
				//
				Pose Ab_refine_pose;
				Pose Ab_scaff_refine_pose;
				//int const resnum_first_disallow_bbmove = scaff_gap_resnum_1 + n_graft_res_allowed_bbmove_nterm + 1;

				get_Ab_scaff_refine_pose( Ab_scaff_design_pose,
																	Ab_design_pose,
																	scaff_refine_pose,
																	resnum_first_disallow_bbmove,
																	Ab_refine_pose       /*output*/,
																	Ab_scaff_refine_pose /*output*/);//could reduce the input arguments



        //
        //Quick measurement of how far the graft moves in space during refinement
        //
        float graft_move_distance_during_refine = 0.0;
        float graft_move_distance_total = 0.0;
        {
          int const ires_center_of_graft = int( 0.5*( scaff_gap_resnum_1 + scaff_gap_resnum_2+resnum_shift_due_to_graft ));
          graft_move_distance_during_refine =
            distance( numeric::xyzVector_float( &( scaff_design_pose.full_coord()(1,2,ires_center_of_graft))),
                      numeric::xyzVector_float( &( scaff_refine_pose.full_coord()(1,2,ires_center_of_graft))));
          graft_move_distance_total =
            distance( numeric::xyzVector_float( &( scaff_w_graft_gly_pose.full_coord()(1,2,ires_center_of_graft))),
                      numeric::xyzVector_float( &( scaff_refine_pose.full_coord()(1,2,ires_center_of_graft))));

					std::cout << "*** graft_move_distance_total = " << F( 8, 2, graft_move_distance_total ) << "***" << std::endl;
        }

				//
				//Output energies: total E, ddg-bind, and ddg-stability for scaffold
				//

				Ab_scaff_refine_pose.score( weight_map );
				float  score_Ab_scaff_refine    ( Ab_scaff_refine_pose.get_0D_score( SCORE ) );
				float  fa_rep_Ab_scaff_refine  ( Ab_scaff_refine_pose.get_0D_score( FA_REP ) );
				float  fa_atr_Ab_scaff_refine  ( Ab_scaff_refine_pose.get_0D_score( FA_ATR ) );
				float  fa_sol_Ab_scaff_refine  ( Ab_scaff_refine_pose.get_0D_score( FA_SOL ) );
				float  fa_dun_Ab_scaff_refine  ( Ab_scaff_refine_pose.get_0D_score( FA_DUN ) );
				float  hbsc_Ab_scaff_refine    ( Ab_scaff_refine_pose.get_0D_score( HB_SC ) );
				float  fa_prob_Ab_scaff_refine ( Ab_scaff_refine_pose.get_0D_score( FA_PROB ) );

				//usual weight_map
				//
				scaff_refine_pose.score( weight_map );
				float score_scaff_refine   ( scaff_refine_pose.get_0D_score( SCORE ) );
				float fa_rep_scaff_refine  ( scaff_refine_pose.get_0D_score( FA_REP ) );
				float fa_atr_scaff_refine  ( scaff_refine_pose.get_0D_score( FA_ATR ) );
				float fa_sol_scaff_refine  ( scaff_refine_pose.get_0D_score( FA_SOL ) );
				float fa_dun_scaff_refine  ( scaff_refine_pose.get_0D_score( FA_DUN ) );
				float hbsc_scaff_refine    ( scaff_refine_pose.get_0D_score( HB_SC ) );
				float fa_prob_scaff_refine ( scaff_refine_pose.get_0D_score( FA_PROB ) );

				Ab_refine_pose.score( weight_map );
				float score_Ab_refine   ( Ab_refine_pose.get_0D_score( SCORE )  );
				float fa_rep_Ab_refine  ( Ab_refine_pose.get_0D_score( FA_REP ) );
				float fa_atr_Ab_refine  ( Ab_refine_pose.get_0D_score( FA_ATR ) );
				float fa_sol_Ab_refine  ( Ab_refine_pose.get_0D_score( FA_SOL ) );
				float fa_dun_Ab_refine  ( Ab_refine_pose.get_0D_score( FA_DUN ) );
				float hbsc_Ab_refine    ( Ab_refine_pose.get_0D_score( HB_SC ) );
				float fa_prob_Ab_refine ( Ab_refine_pose.get_0D_score( FA_PROB ) );

				//ddG-bind total score and components
				//
				float ddg_bind_Ab_scaff_refine         = score_Ab_scaff_refine   - score_scaff_refine - score_Ab_refine;
				float ddg_bind_Ab_scaff_fa_rep_refine  = fa_rep_Ab_scaff_refine  - fa_rep_scaff_refine  - fa_rep_Ab_refine;
				float ddg_bind_Ab_scaff_fa_atr_refine  = fa_atr_Ab_scaff_refine  - fa_atr_scaff_refine  - fa_atr_Ab_refine;
				float ddg_bind_Ab_scaff_fa_sol_refine  = fa_sol_Ab_scaff_refine  - fa_sol_scaff_refine  - fa_sol_Ab_refine;
				float ddg_bind_Ab_scaff_fa_dun_refine  = fa_dun_Ab_scaff_refine  - fa_dun_scaff_refine  - fa_dun_Ab_refine;
				float ddg_bind_Ab_scaff_hbsc_refine    = hbsc_Ab_scaff_refine    - hbsc_scaff_refine    - hbsc_Ab_refine;
				float ddg_bind_Ab_scaff_fa_prob_refine = fa_prob_Ab_scaff_refine - fa_prob_scaff_refine - fa_prob_Ab_refine;

				//ddG-stability total score and components
				//
				float ddg_scaff_refine                 = score_scaff_refine   - score_scaff_native;
				float ddg_scaff_fa_rep_refine          = fa_rep_scaff_refine  - fa_rep_scaff_native;
				float ddg_scaff_fa_atr_refine          = fa_atr_scaff_refine  - fa_atr_scaff_native;
				float ddg_scaff_fa_sol_refine          = fa_sol_scaff_refine  - fa_sol_scaff_native;
				float ddg_scaff_fa_dun_refine          = fa_dun_scaff_refine  - fa_dun_scaff_native;
				float ddg_scaff_hbsc_refine            = hbsc_scaff_refine    - hbsc_scaff_native;
				float ddg_scaff_fa_prob_refine         = fa_prob_scaff_refine - fa_prob_scaff_native;


				std::string filename;
				std::string design_id_str = right_string_of( i_design, 3, '0' );
				std::string refine_id_str = right_string_of( i_refine, 3, '0' );
				std::string filename_base2 = filename_base + "_" + design_id_str + "_" + refine_id_str;

				get_filename_with_energies( filename_base2, score_Ab_scaff_refine, ddg_bind_Ab_scaff_refine, filename );

				if ( output_pdbs ) {
					Ab_scaff_refine_pose.pdb_info().set_use_pdb_numbering( true );
					Ab_scaff_refine_pose.dump_pdb( files_paths::pdb_out_path + filename );
				}

			outfile   << SS( "ddg_bind_Ab_scaff_refine:" )
								<< SS( "ides" )
								<< SS( "iref" )
								<< SS( "     ddg" )
								<< SS( "     atr" )
								<< SS( "     rep" )
								<< SS( "     sol" )
								<< SS( "     dun" )
								<< SS( "    hbsc" )
								<< SS( "    prob" )
								<< SS( "   scaff" )
								<< std::endl;

			outfile   << SS( "ddg_bind_Ab_scaff_refine: " )
								<< I( 4, i_design ) << " "
								<< I( 4, i_refine ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_refine        ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_atr_refine ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_rep_refine ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_sol_refine ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_dun_refine ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_hbsc_refine   ) << " "
								<< F( 8, 2, ddg_bind_Ab_scaff_fa_prob_refine) << " "
								<< filename
								<< std::endl;

			outfile   << SS( "ddg_scaff_refine        :" )
								<< SS( "ides" )
								<< SS( "iref" )
								<< SS( "     ddg" )
								<< SS( "     atr" )
								<< SS( "     rep" )
								<< SS( "     sol" )
								<< SS( "     dun" )
								<< SS( "    hbsc" )
								<< SS( "    prob" )
								<< SS( "    chbk" )
								<< SS( " cl_move" )
								<< SS( " re_move" )
								<< SS( " cl_move" )
								<< SS( " totmove" )
								<< SS( "   scaff" )
								<< std::endl;

			outfile   << SS( "ddg_scaff_refine        : " )
								<< I( 4, i_design ) << " "
								<< I( 4, i_refine ) << " "
								<< F( 8, 2, ddg_scaff_refine        ) << " "
								<< F( 8, 2, ddg_scaff_fa_atr_refine ) << " "
								<< F( 8, 2, ddg_scaff_fa_rep_refine ) << " "
								<< F( 8, 2, ddg_scaff_fa_sol_refine ) << " "
								<< F( 8, 2, ddg_scaff_fa_dun_refine ) << " "
								<< F( 8, 2, ddg_scaff_hbsc_refine   ) << " "
								<< F( 8, 2, ddg_scaff_fa_prob_refine) << " "
								<< F( 8, 4, chainbreak_score_refine ) << " "
								<< F( 8, 2, graft_move_distance_during_closure ) << " "
								<< F( 8, 2, graft_move_distance_during_refine ) << " "
								<< F( 8, 2, graft_move_distance_total ) << " "
								<< filename
								<< std::endl;


			}//refine iterations

		}//i_design_to_refine

    //
    //Delete the array of pointers
    //
    for ( int i_design = 1; i_design <= n_design_iterations; ++i_design ) {
      //      delete pose_list[k];
      delete pose_array_Ab_scaff_design( i_design );
    }

		//
		//checkpoint for clusters
		//
		if ( using_checkpoint ) {
			update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
		}

	}//i_scaff

	//
	//Close the output file
	//
	outfile.close();
	outfile.clear();
}
///////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////
void
add_on_epi_graft_design()
{

  using namespace pose_ns;

  // scorefxn
  Score_weight_map weight_map( score12 );

  //
  //set basic parameters controlling pose behavior
  //
  bool const fullatom( true );
  bool const ideal_pose( false ); // non-ideal backbone geometry
  bool const read_all_chains( true );
  bool const check_missing( false ); //use this for pose.set_coords throughout
	//  bool const coords_init( true ); //used in pose_from_misc

	//
	//atom numbers
	//

  int atomnum_N,atomnum_CA,atomnum_C,atomnum_O,atomnum_CB;
  if ( fullatom ) {
    atomnum_N =  1;
    atomnum_CA = 2;
    atomnum_C =  3;
    atomnum_O =  4;
    atomnum_CB = 5;
  } else {//Eposition numbering
    atomnum_N =  1;
    atomnum_CA = 2;
    atomnum_C =  4;
    atomnum_O =  5;
    atomnum_CB = 3;
  }

	//debug option
	//
	bool const output_debug_pdbs=truefalseoption("output_debug_pdbs");

  //
  //file to output scores
  //
	std::string output_filename;
  stringafteroption( "output_file", "none", output_filename );
  if( output_filename == "none" ) {
		std::cout << "Need -output_file <filename> "
              << "on command-line " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }
  //
  //open the output file
  // ( leave it open since we write scores about 1/s or faster )
	utility::io::ozstream outfile;
  open_output_file( output_filename, outfile );


	//
	//put header in output file
	//******** Needs to be updated ************
	outfile << SS("scaff")
					<< SS("  loop")
					<< SS("   rng")
					<< SS("epi1_nat epi2_nat")
					<< SS(" epi1  epi2")
					<< SS(" sca1  sca2")
					<< SS("chainbreak_rms")
					<< SS("chbk_rms_nterm")
					<< SS("chbk_rms_cterm")
					<< SS(" scaff_filename") << std::endl;

  //
  //checkpoint file for running on clusters
  //
	std::string checkpoint_filename;
	std::string checkpoint_file_w_path;
  stringafteroption( "checkpoint", "none", checkpoint_filename );
  bool using_checkpoint = false;
  if ( checkpoint_filename == "none" ) {
		std::cout << "no checkpoint_filename, NOT using checkpointing" << std::endl;
  } else {
    using_checkpoint = true;
    checkpoint_file_w_path = files_paths::score_path + checkpoint_filename;
  }

	//
	//option to output pdbs oriented relative to nat_cmplx
	//
	bool output_pdbs = false;
	output_pdbs = truefalseoption("output_pdbs");

	//
	//parameters controlling matching
	//
	float ca_dist_cutoff = 6.0;
	realafteroption("rough_match_ca_dist_cutoff",ca_dist_cutoff,ca_dist_cutoff);

	int max_loopsize_difference = 10;
	intafteroption("max_loopsize_difference",max_loopsize_difference,max_loopsize_difference);


	//Read info on epitope_loops
	//
	int n_epi_loops_default = 1;
	int n_ranges_default = 1;
  FArray1D_int nres_epi_loops( n_epi_loops_default );//i=loop_num; j=1:nres_this_loop;
	FArray1D_int nranges_epi_loops( n_epi_loops_default );  //i=nranges_this_loop
	//
	//ranges_epi_loops gives low,high for consecutive residues within epitope loops
	//i=1,2: range_low, range_high for scaff in sequential numbering 1-nres_scaff
	//i=3,4: range_low, range_high for epitope in native epitope numbering
	//
	FArray3D_int ranges_epi_loops( 4, n_ranges_default, n_epi_loops_default );// j=1 range_low; j=2 range_high

  read_epi_loop_info( nres_epi_loops, nranges_epi_loops, ranges_epi_loops );
	int const n_epi_loops = nres_epi_loops.size();



	//
  //Read in list of Ab-scaffold complex pdbs...also reads information about
	//the loop to be grafted and the loop that is already been added to scaff ( by superpos or grafting )
  //

	int n_scaff = 0;
  FArray1D_string list_scaff_pdb;
	FArray1D_int list_graft_loop_id( n_scaff );
	FArray1D_int list_graft_range_id( n_scaff );
	FArray2D_int list_graft_resnums( 4, n_scaff );     //1,2 are graft_resnums in 1-x numbering, 3,4 are in native epitope numbering
	FArray2D_int list_scaff_gap_resnums( 2, n_scaff ); //1,2 are scaff_gap_resnums in 1-nres_scaff numbering

	FArray1D_int list_graft_loop_id_already( n_scaff );
  FArray2D_int list_graft_resnums_already( 4, n_scaff );     //1,2 are graft_resnums in 1-x numbering, 3,4 are in native epitope numbering
	FArray2D_int list_scaff_gap_resnums_already( 2, n_scaff ); //1,2 are scaff_gap_resnums in 1-nres_scaff numbering

	read_list_scaff_info_for_add_on_epi_graft_design( list_graft_loop_id,
																										list_graft_range_id,
																										list_graft_resnums,
																										list_scaff_gap_resnums,
																										list_graft_loop_id_already,
																										list_graft_resnums_already,
																										list_scaff_gap_resnums_already,
																										list_scaff_pdb );
	n_scaff = list_scaff_pdb.size();

  // set length of Ab
  // get from commandline
  int nres_Ab(0);
	intafteroption( "nres_Ab", 0, nres_Ab );
	if ( nres_Ab == 0 ) {
		std::cout << "must specify -nres_Ab <nres> on commandline" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

  // read in the native complex of Ab-epitope
	//
	//Assume that Ab comes first!!
  Pose nat_cmplx_pose;
	pose_from_pdb( nat_cmplx_pose, stringafteroption("native_complex"), fullatom, ideal_pose,
								 read_all_chains );

	nat_cmplx_pose.dump_pdb( "nat_cmplx_pose.pdb" );

  // flip symmetric sidechains
  {
    float const start_score( nat_cmplx_pose.score( weight_map ) );
    nat_cmplx_pose.refold_sidechains_from_chi();
		std::cout << "score-delta from sidechain flip: " <<
      nat_cmplx_pose.score( weight_map ) - start_score << std::endl;
  }

  int const nres_nat_cmplx( nat_cmplx_pose.total_residue() );

	//save nres for full-length peptide graft
	//
	int const nres_nat_pep = nres_nat_cmplx - nres_Ab;

	//make a pose that is Ab only and score it
	Pose nat_Ab_pose;
	nat_Ab_pose.simple_fold_tree( nres_Ab );
	nat_Ab_pose.set_fullatom_flag( true, false /*repack*/);
	copy_segment_to_new_pose( nat_cmplx_pose /*src_pose*/,
														1 /*begin_src*/,
														nres_Ab /*end_src*/,
														nat_Ab_pose /*pose_out*/);
	nat_Ab_pose.score( weight_map );
	//	nat_Ab_pose.dump_pdb("nat_Ab_pose.pdb");

	float const fa_rep_Ab ( nat_Ab_pose.get_0D_score( FA_REP ) );

  //make a pose that is epitope only and score it
  Pose nat_pep_pose;
  nat_pep_pose.simple_fold_tree( nres_nat_pep );
  nat_pep_pose.set_fullatom_flag( true, false /*repack*/);
  copy_segment_to_new_pose( nat_cmplx_pose, nres_Ab+1 /*begin_src*/, nres_nat_cmplx /*end_src*/, nat_pep_pose );
  nat_pep_pose.score( weight_map );
  float const fa_rep_nat_pep ( nat_pep_pose.get_0D_score( FA_REP ) );


  //make an all-gly version of the native epitope
  Pose nat_pep_gly_pose;
  nat_pep_gly_pose = nat_pep_pose;
  convert_resnum_range_to_gly( nat_pep_gly_pose, 1, nres_nat_pep );
	if (output_debug_pdbs) nat_pep_gly_pose.dump_pdb("nat_pep_gly_pose.pdb");

	//
	//Read info on natro_positions in epitope_loops
	//
  //Read in epitope positions at which native rotamers will be retained.
  // Notes: These are epitope positions that interact with Ab.
  //        These must be distinguished from epitope positions that point away
  //        from the Ab and can be repacked/redesigned.
  //        The list file must contain rosetta numbering in the first column.
  //
	//
	//For the "add_on" case, the input file must have a line for each epitope residue in the native_complex pdb
	//the loops must be ordered sequentially as in the native_complex
	//and the max range should be listed for each loop ( as in the native_complex )
	//
  int n_natro_in_epitope = 0;
  FArray1D_int list_natro_in_epitope( n_natro_in_epitope );
  FArray1D_bool natro_in_epitope( nres_nat_pep, false );
  get_natro_in_epitope( nres_nat_pep, n_natro_in_epitope,
                        list_natro_in_epitope,
                        natro_in_epitope );


	//
	//read in Ab-scaff complexes and test one at a time
	//

	//
	//must specify whether Ab or scaff comes first.
	//
	bool Ab_first =  false;
	bool scaff_first = false;
	scaff_first=truefalseoption("scaff_first");
	Ab_first=truefalseoption("Ab_first");
	if ( Ab_first && scaff_first ) {
		std::cout << "ERROR you cannot specify both Ab_first and scaff_first" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else {
		if ( !(Ab_first || scaff_first)) {
			std::cout << "ERROR must specify either Ab_first or scaff_first" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

  //
  //count # of sites scored, for checkpointing.
  //
  int n_scaff_scored = 0;

	for ( int i_scaff = 1; i_scaff <= n_scaff; i_scaff++ ) {

    reset_global_loop_info(); // required, otherwise there is confusion from prior iteration

		//
		//update count for checkpointing
		//
		++n_scaff_scored;
		if ( using_checkpoint ) {
			bool already_scored = check_conformation_already_scored( n_scaff_scored, checkpoint_file_w_path );
			if ( already_scored ) continue;
		}

		////////////////////////////////////////////////////////////////////
		//		load Ab-scaff complex into pose
		//

		std::string Ab_scaff_filename = files_paths::start_path + list_scaff_pdb( i_scaff );
		//files_paths::start_path + files_paths::start_file

		utility::io::izstream infile ( Ab_scaff_filename );
		if ( ! infile ) {
			std::cout	<< " cannot open file " << Ab_scaff_filename
								<< std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		//		std::cout << "successfully read Ab_scaff input complex" << SS( Ab_scaff_filename ) << std::endl;

		Pose Ab_scaff_pose;
		pose_from_pdb( Ab_scaff_pose, Ab_scaff_filename, fullatom, ideal_pose, read_all_chains );
		int const nres_Ab_scaff = Ab_scaff_pose.total_residue();
		int const nres_scaff = nres_Ab_scaff - nres_Ab;


		Pose scaff_pose;
		scaff_pose.simple_fold_tree( nres_scaff );
		scaff_pose.set_fullatom_flag( true, false );// set fullatom and do not repack

		Pose Ab_pose;
		Ab_pose.simple_fold_tree( nres_Ab );
		Ab_pose.set_fullatom_flag( true, false );// set fullatom and do not repack


		if ( scaff_first ) {

			copy_segment_to_new_pose( Ab_scaff_pose /*src_pose*/,
																1 /*begin_src*/,
																nres_scaff /*end_src*/,
																scaff_pose /*pose_out*/);

			copy_segment_to_new_pose( Ab_scaff_pose /*src_pose*/,
																nres_scaff+1 /*begin_src*/,
																nres_Ab_scaff /*end_src*/,
																Ab_pose /*pose_out*/);

			//
			//overwrite Ab_scaff_pose to make Ab first in Ab_scaff_pose
			//
			construct_pose_complex_from_p1_p2( Ab_pose, scaff_pose, Ab_scaff_pose );

		} else {
			if ( Ab_first ) {

			copy_segment_to_new_pose( Ab_scaff_pose /*src_pose*/,
																1 /*begin_src*/,
																nres_Ab /*end_src*/,
																Ab_pose /*pose_out*/);

			copy_segment_to_new_pose( Ab_scaff_pose /*src_pose*/,
																nres_Ab+1 /*begin_src*/,
																nres_Ab_scaff /*end_src*/,
																scaff_pose /*pose_out*/);

			} else {
				std::cout << "ERROR parsing input complex into scaff and Ab. " << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

    //
    //Get scores for native scaffold so we can compute ddg_stability for scaffold
    // *** We should probably just repack the scaff *** ( but only at positions that will design )

    scaff_pose.score( weight_map );
    float score_scaff_native   ( scaff_pose.get_0D_score( SCORE ) );
    float  fa_rep_scaff_native ( scaff_pose.get_0D_score( FA_REP ) );
    float  fa_atr_scaff_native ( scaff_pose.get_0D_score( FA_ATR ) );
    float  fa_sol_scaff_native ( scaff_pose.get_0D_score( FA_SOL ) );
    float  fa_dun_scaff_native ( scaff_pose.get_0D_score( FA_DUN ) );
    float  hbsc_scaff_native ( scaff_pose.get_0D_score( HB_SC ) );
    float  fa_prob_scaff_native ( scaff_pose.get_0D_score( FA_PROB ) );

		if ( output_debug_pdbs ) scaff_pose.dump_pdb("scaff_pose.pdb");

		//
		//make oriented pose
		//
		Pose Ab_scaff_oriented_pose;
		Ab_scaff_oriented_pose = Ab_scaff_pose;
		//use superimpose instead of 3 atoms to orient...this is more robust
		superimpose_using_ncaco(
														nat_cmplx_pose /*pose1*/,
														1 /*pose1_first*/,
														nres_Ab /*pose1_last*/,
														Ab_scaff_oriented_pose /*pose2*/, //pose2 will be changed, superimposed onto pose1
														1  /*pose2_first*/,
														nres_Ab /*pose2_last*/
														);

		//		Ab_scaff_oriented_pose.dump_pdb("Ab_scaff_oriented_pose.pdb");

		std::cout << "done superimpose" << std::endl;;

		//so we have
		//(1) nat_cmplx_pose -- it should be "Ab_epitope_nat_pose"
		//(2) Ab_scaff_oriented_pose
		//(3) which loop has been superposed onto this scaff?  loop_id_superposed_in_scaff( i_scaff )
		//(4) what range of residues contains the superposition?  list_range_superposed_in_scaff
		//list_range_superposed_in_scaff( 4, i_scaff )
		//i=1,2: range_low, range_high for scaff in sequential numbering 1-nres_scaff
		//i=3,4: range_low, range_high for epitope in native epitope numbering
		//

		//we must also read in an info_file telling about the loop to be grafted for each scaffold:
		//i_loop
		//graft_resnum_1 graft_resnum_2
		//graft_resnum_native_num_1 graft_resum_native_num_2 -- from this we can compute i_loop and i_range should agree with graft_resnum_1,_2 read in
		//scaff_gap_resnum_1, scaff_gap_resnum_2 -- below these are
		//
		int const loop_id_already = list_graft_loop_id_already( i_scaff );

		//			list_graft_loop_id,
		//				list_graft_range_id,
		//				list_graft_resnums,
		//				list_scaff_gap_resnums,
		//				list_graft_loop_id_already,
		//				list_graft_resnums_already,
		//				list_scaff_gap_resnums_already,
		//				list_scaff_pdb

		//
		//the loop_id we want to graft as second loop
		//
		int const i_loop = list_graft_loop_id( i_scaff );

		//
		//the range_id we want to graft
		//
		int const i_range = list_graft_range_id( i_scaff );

		//
		//dummy check: don't graft a loop if it is already in the scaffold
		//
		if ( i_loop == list_graft_loop_id_already( i_scaff ) ) {
			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

			continue;
		}
		//
		//dummy check: the graft_resnums input must agree with the loop subrange
		//
		if ( list_graft_resnums( 1, i_scaff ) != ranges_epi_loops( 1, i_range, i_loop ) ||
				 list_graft_resnums( 2, i_scaff ) != ranges_epi_loops( 2, i_range, i_loop ) ) {
			std::cout << "ERROR: input graft_resnums for i_scaff not agree with specified i_loop, i_range" << std::endl;
			std::cout << "i_scaff " << i_scaff << " list_graft_resnums( 1 and 2, i_scaff ) "
								<< list_graft_resnums( 1, i_scaff )
								<< "-" << list_graft_resnums( 2, i_scaff ) << std::endl;
			std::cout << "i_loop, i_range, ranges_epi_loops( 1 and 2, i_range, i_loop ) " << i_loop
								<< " " << i_range
								<< " " << ranges_epi_loops( 1, i_range, i_loop )
								<< "-" << ranges_epi_loops( 2, i_range, i_loop )
								<< std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		//
		//resnum_shift allows us to point to specific loop residues
		//within the nat_cmplx_pose, in which Ab is first, followed by loops 1,2,3, etc.
		//the array ranges_epi_loops specifies positions within each loop.
		//
		int resnum_shift_graft_in_native_complex = nres_Ab;
		for ( int i = 1; i <= i_loop - 1; i++ ) {
			resnum_shift_graft_in_native_complex += nres_epi_loops( i );
		}

		//graft_resnum_1 and graft_resnum_2 are first,last residues in graft numbered 1-nres_loop
		int const graft_resnum_1 = ranges_epi_loops( 1, i_range, i_loop );
		int const graft_resnum_2 = ranges_epi_loops( 2, i_range, i_loop );
		int const nres_graft = graft_resnum_2 - graft_resnum_1 + 1;

		//
		//the first,last seqpos for epitope as numbered in nat_cmplx with Ab first, followed by epitope loops
		//
		int const graft_resnum_1_in_native_complex = graft_resnum_1 + resnum_shift_graft_in_native_complex;
		int const graft_resnum_2_in_native_complex = graft_resnum_2 + resnum_shift_graft_in_native_complex;

		//
		//scaff_gap_resnum_1 and scaff_gap_resnum_2 in 1-nres_scaff numbering
		//note we will need to add nres_Ab to address these positions in Ab_scaff_pose.
		// we have ensured that Ab comes first in Ab_scaff_pose.
		//

		int const scaff_gap_resnum_1 = list_scaff_gap_resnums( 1, i_scaff );
		int const scaff_gap_resnum_2 = list_scaff_gap_resnums( 2, i_scaff );

		int const scaff_gap_resnum_1_already = list_scaff_gap_resnums_already( 1, i_scaff );
		int const scaff_gap_resnum_2_already = list_scaff_gap_resnums_already( 2, i_scaff );


		//
		//direction must be n2c
		//

		if ( scaff_gap_resnum_2 <= scaff_gap_resnum_1 ) {
					std::cout << "scaff_gap_resnum_2 <= scaff_gap_resnum_1...shouldn't happen...continue to next graft" << std::endl;;
				if ( using_checkpoint ) {
        update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
        }
				continue; //was break but should be continue
		}
		//
		//loopsize in scaff ( nres to be deleted from scaff )
		//skip it, if too big compared to size of graft itself
		//
		int const nres_range_scaff = scaff_gap_resnum_2 - scaff_gap_resnum_1 + 1;
		int const delta_loopsize = nres_range_scaff - nres_graft;
		if ( delta_loopsize > 0 && delta_loopsize > max_loopsize_difference ) {
			std::cout << "delta_loopsize > 0 && delta_loopsize > max_loopsize_difference...continue to next graft!" << std::endl;;
			if ( using_checkpoint ) {
        update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
      }
			continue; //was break, but should be continue

		}
		//
		//this graft site for loop2 cannot encompass the positions where loop1 is already superposed/grafted.
		//
		//Here scaff_gap_resnum_1 and scaff_gap_resnum_2 are already in 1-nres_scaff numbering
		//even though at this same point in the rough_match protocol they are not.
		//
		if ( scaff_gap_resnum_1 <= list_scaff_gap_resnums_already( 1, i_scaff ) &&
				 scaff_gap_resnum_2 >= list_scaff_gap_resnums_already( 2, i_scaff ) ) {
			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

			continue;
		}

		//
		//Now do some measure for chainbreak...
		//

		//
		//closure atoms
		//
		FArray1D_int closure_atom_cterm( 5 );
		closure_atom_cterm( 1 ) = atomnum_C;
		closure_atom_cterm( 2 ) = atomnum_N;
		closure_atom_cterm( 3 ) = atomnum_CA;
		closure_atom_cterm( 4 ) = atomnum_C;
		closure_atom_cterm( 5 ) = atomnum_O;

		FArray1D_int closure_atom_nterm( 5 );
		closure_atom_nterm( 1 ) = atomnum_N;
		closure_atom_nterm( 2 ) = atomnum_CA;
		closure_atom_nterm( 3 ) = atomnum_C;
		closure_atom_nterm( 4 ) = atomnum_O;
		closure_atom_nterm( 5 ) = atomnum_N;

		FArray1D_int closure_resnum_scaff_nterm( 5 );
		closure_resnum_scaff_nterm( 1 ) = scaff_gap_resnum_1;
		closure_resnum_scaff_nterm( 2 ) = scaff_gap_resnum_1;
		closure_resnum_scaff_nterm( 3 ) = scaff_gap_resnum_1;
		closure_resnum_scaff_nterm( 4 ) = scaff_gap_resnum_1;
		closure_resnum_scaff_nterm( 5 ) = scaff_gap_resnum_1 + 1;

		FArray1D_int closure_resnum_scaff_cterm( 5 );
		closure_resnum_scaff_cterm( 1 ) = scaff_gap_resnum_2 - 1;
		closure_resnum_scaff_cterm( 2 ) = scaff_gap_resnum_2;
		closure_resnum_scaff_cterm( 3 ) = scaff_gap_resnum_2;
		closure_resnum_scaff_cterm( 4 ) = scaff_gap_resnum_2;
		closure_resnum_scaff_cterm( 5 ) = scaff_gap_resnum_2;

		FArray1D_int closure_resnum_native_nterm( 5 );
		closure_resnum_native_nterm( 1 ) = graft_resnum_1_in_native_complex;
		closure_resnum_native_nterm( 2 ) = graft_resnum_1_in_native_complex;
		closure_resnum_native_nterm( 3 ) = graft_resnum_1_in_native_complex;
		closure_resnum_native_nterm( 4 ) = graft_resnum_1_in_native_complex;
		closure_resnum_native_nterm( 5 ) = graft_resnum_1_in_native_complex + 1;

		FArray1D_int closure_resnum_native_cterm( 5 );
		closure_resnum_native_cterm( 1 ) = graft_resnum_2_in_native_complex - 1;
		closure_resnum_native_cterm( 2 ) = graft_resnum_2_in_native_complex;
		closure_resnum_native_cterm( 3 ) = graft_resnum_2_in_native_complex;
		closure_resnum_native_cterm( 4 ) = graft_resnum_2_in_native_complex;
		closure_resnum_native_cterm( 5 ) = graft_resnum_2_in_native_complex;

		float dist2_nterm ( 0.0 ), dist2_cterm( 0.0 );
		int atom_count( 0 );
		for ( int i = 1; i <= 5; i++ ) {
			atom_count++;
			for ( int k = 1; k <= 3; k++ ) {
				dist2_nterm += square( Ab_scaff_oriented_pose.full_coord()(k,closure_atom_nterm(i),closure_resnum_scaff_nterm(i)+nres_Ab)
															 - nat_cmplx_pose.full_coord()(k,closure_atom_nterm(i),closure_resnum_native_nterm(i)) );
				dist2_cterm += square( Ab_scaff_oriented_pose.full_coord()(k,closure_atom_cterm(i),closure_resnum_scaff_cterm(i)+nres_Ab)
															 - nat_cmplx_pose.full_coord()(k,closure_atom_cterm(i),closure_resnum_native_cterm(i)) );
			}
		}
		float one_over_atom_count;

		//rms at nterm
		one_over_atom_count = 1.0 / atom_count;
		float chbk_rms_nterm = std::sqrt( dist2_nterm * one_over_atom_count );

		//rms at cterm
		float chbk_rms_cterm = std::sqrt( dist2_cterm * one_over_atom_count );

		//total rms
		float dist2_total = dist2_nterm + dist2_cterm;
		one_over_atom_count = 1.0 / ( 2.0 * atom_count );
		float chainbreak_rms = std::sqrt( dist2_total * one_over_atom_count );

		float max_chainbreak_rms_pre_closure = 3.0;
		realafteroption("max_chainbreak_rms_pre_closure",max_chainbreak_rms_pre_closure,max_chainbreak_rms_pre_closure);

		if ( chainbreak_rms > max_chainbreak_rms_pre_closure ) {
			std::cout << "pre-closure chainbreak_rms = " << chainbreak_rms << " > max_chainbreak_rms_pre_closure...skipping to next graft" << std::endl;

			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

			continue;
		}

		//
		//Do the loop-closure/scaff design using scaff_pose ( no Ab )
		//....the rb orientation of scaff and Ab won't change so it is easy to put Ab back in...
		//....it will just make the loop-refinement faster if we don't have the Ab around...
		//....but is it dangerous? may be introduce clashes with Ab? ( same as regular design, need to repack or trim design after )
		//

		//Hey, don't forget to to an intra-clash check right off the bat!
		//we may have to quit this one if there is too much intra clash with second loop
		//

		//so from here, we:
		//(1) make scaff_gly_w_gap_pose
		//(2) make graft_notermini_gly_pose and scaff_w_graft_gly_pose
		//(3) setup 2 loops to model ... any changes to how we communicate with pose_loops protocol? any difficulties with trim?
		//(4) close the loops
		//(5) measure intra clash after closure -- just score the scaff_w_graft_closed_gly and see if it has lots of clash
		//(6) transfer native rotamers from epitope to the new graft
		//(7) transfter native scaffold rotamers elsewhere ( this will include rotamers from first loop )
		//(8) identify intra design positions around second loop ( do we exclude positions near first loop?)
		//(9) do the intra design and report pre-trim-design mutations
		//(10) do a loop refinement
		//(11) measure ddg_bind and ddg_scaff and report pre-trim-design
		//(12) identify trimming design positions, do trim design, re-measure ddg_bind, ddg_scaff, report mutations due to trim


		//
		//extract scaff and Ab poses that have been oriented relative to native_complex
		//
		Pose scaff_oriented_pose;
		scaff_oriented_pose = scaff_pose;
    copy_segment_to_new_pose( Ab_scaff_oriented_pose /*src_pose*/,
                              nres_Ab+1              /*begin_src*/,
                              nres_Ab + nres_scaff  /*end_src*/,
                              scaff_oriented_pose   /*pose_out*/);
		Pose Ab_oriented_pose;
		Ab_oriented_pose = Ab_pose;
    copy_segment_to_new_pose( Ab_scaff_oriented_pose /*src_pose*/,
                              1                      /*begin_src*/,
                              nres_Ab               /*end_src*/,
                              Ab_oriented_pose     /*pose_out*/);
		if (output_debug_pdbs) Ab_oriented_pose.dump_pdb("Ab_oriented_pose.pdb");

		Pose scaff_gly_pose;
		scaff_gly_pose = scaff_oriented_pose;
		int gly_begin = 1;
		int gly_end = nres_scaff;
		convert_resnum_range_to_gly( scaff_gly_pose, gly_begin, gly_end );
		if (output_debug_pdbs) scaff_gly_pose.dump_pdb("scaff_gly_pose.pdb");

		//	construct gly-only version with graft-site removed -- scaff_gly_w_gap_pose
		//
		//  Note:  scaff_gap_resnum_1 =< gap =< scaff_gap_resnum_2
		//         gap DOES include the scaff_gap_resnum_x
		//
		Pose scaff_gly_w_gap_pose;
		int gap_begin = scaff_gap_resnum_1; //scaff_gap_resnum_1 + 1;
		int gap_end = scaff_gap_resnum_2;   //scaff_gap_resnum_2 - 1;
		make_pose_with_gap( scaff_gly_pose, gap_begin, gap_end, scaff_gly_w_gap_pose );
		//		scaff_gly_w_gap_pose.dump_pdb("scaff_gly_w_gap_pose.pdb");
		int const nres_scaff_w_gap = scaff_gly_w_gap_pose.total_residue();
		scaff_gly_w_gap_pose.score( weight_map );
		float fa_rep_scaff_gly_w_gap = scaff_gly_w_gap_pose.get_0D_score( FA_REP );

		if ( nres_scaff_w_gap != ( nres_scaff - ( gap_end - gap_begin + 1 ) ) ) {
			std::cout << "error on nres_scaff_w_gap " << nres_scaff_w_gap << " "
								<< nres_scaff << " " << gap_end << " " << gap_begin << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		if (output_debug_pdbs) scaff_gly_w_gap_pose.dump_pdb("scaff_gly_w_gap.pdb");

		//
		//make a pose that is the graft peptide only -- take from native_complex
		//
		Pose graft_pose;
		graft_pose.simple_fold_tree( nres_graft );
		graft_pose.set_fullatom_flag( true, false );// set fullatom and do not repack
		copy_segment_to_new_pose( nat_cmplx_pose /*src_pose*/,
															graft_resnum_1_in_native_complex      /*begin_src*/,
															graft_resnum_2_in_native_complex /*end_src*/,
															graft_pose   /*pose_out*/);

		//
		//make a graft pose that is all-gly
		//
		Pose graft_gly_pose;
		graft_gly_pose = graft_pose;
		gly_begin = 1;
		gly_end = nres_graft;
		convert_resnum_range_to_gly( graft_gly_pose, gly_begin, gly_end );


		//
		//Score the graft_gly_pose to get fa_rep
		//

		graft_gly_pose.score( weight_map );
		float const fa_rep_graft_gly ( graft_gly_pose.get_0D_score( FA_REP ) );

		if (output_debug_pdbs) graft_pose.dump_pdb("graft_pose.pdb");
		if (output_debug_pdbs) graft_gly_pose.dump_pdb("graft_gly_pose.pdb");



		//next we want to make a pose that has the graft + scaff_w_gap in all-gly
		//then score fa_rep for that pose and substract fa_rep for scaff_w_gap alone

		int const nres_scaff_w_graft = nres_graft + nres_scaff_w_gap;

		std::cout << "nres_scaff_w_graft, nres_graft, nres_scaff_w_gap = "
							<< nres_scaff_w_graft << " "
							<< nres_graft << " "
							<< nres_scaff_w_gap << std::endl;

		std::cout << "nres_scaff, scaff_gap_resnum_1, scaff_gap_resnum_2 "
							<< nres_scaff << " " << scaff_gap_resnum_1 << " "
							<< scaff_gap_resnum_2 << std::endl;

		Pose scaff_w_graft_gly_pose_old;
		make_scaff_w_graft_gly_pose( scaff_gly_pose,
																 graft_gly_pose,
																 scaff_gap_resnum_1,
																 scaff_gap_resnum_2,
																 nres_scaff_w_graft,
																 scaff_w_graft_gly_pose_old );//output

		std::string graft_align_system = "AO";
		std::string graft_dir = "---";

    Pose scaff_w_graft_gly_pose;
    new_make_scaff_w_graft_gly_pose( scaff_gly_pose,
                                     nat_pep_gly_pose,//includes all loops that are read in via native_complex
                                     graft_align_system,
                                     graft_dir,
                                     graft_resnum_1_in_native_complex-nres_Ab,//points to beginning of the loop from which graft is a sub-range
                                     graft_resnum_2_in_native_complex-nres_Ab,//point to the end of the loop from which graft is a sub-range
                                     scaff_gap_resnum_1,
                                     scaff_gap_resnum_2,
                                     nres_scaff_w_graft,
                                     scaff_w_graft_gly_pose );//output


		scaff_w_graft_gly_pose.score( weight_map );
		float const fa_rep_scaff_w_graft_gly ( scaff_w_graft_gly_pose.get_0D_score( FA_REP ) );
		if ( output_debug_pdbs ) {
      scaff_w_graft_gly_pose_old.dump_pdb("scaff_w_graft_gly_pose_old.pdb");
			std::string tmp_filename = "scaff_w_graft_gly_pose_" + string_of( i_scaff ) + ".pdb";
			scaff_w_graft_gly_pose.dump_pdb(tmp_filename);
		}

    ////////////////////////////////////////////////////////////////////
    //scaffold: now measure "intra" clash between graft and scaff backbones only
    ////////////////////////////////////////////////////////////////////

		float const fa_rep_intra_gly ( fa_rep_scaff_w_graft_gly - fa_rep_graft_gly - fa_rep_scaff_gly_w_gap );

		std::cout << "fa_rep_intra_gly, fa_rep_scaff_w_graft_gly, fa_rep_graft_gly, fa_rep_scaff_gly_w_gap "
							<< fa_rep_intra_gly << " " << fa_rep_scaff_w_graft_gly << " " << fa_rep_graft_gly << " "
							<< fa_rep_scaff_gly_w_gap << std::endl;


    ///////////////////////////////////////////////////////////////////////////////
    //Ask if this intra clash is below the loose threshold
    //Using a loose threshold initially prior to closing the chainbreak(s)
    //

    float max_fa_rep_intra_gly_before_close_default( 3000. ); //needs to be tested
    float max_fa_rep_intra_gly_before_close;
    realafteroption("max_fa_rep_intra_gly_before_close", max_fa_rep_intra_gly_before_close_default, max_fa_rep_intra_gly_before_close );

    //
    // if not satisfy loose_max, continue to next graft/scaff combo
    //
    if ( fa_rep_intra_gly > max_fa_rep_intra_gly_before_close ) {
			std::cout << "fa_rep_intra_gly too high, "
                << "going to next graft/scaff combo"
                << std::endl;
			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

      continue;
		}


    //
    //make a pose of Ab (all-atom) + scaff_w_graft (all-gly)
    //

    Pose Ab_scaff_w_graft_gly_pose;
		int const nres_Ab_scaff_w_graft = nres_Ab + nres_scaff_w_graft;
		//		Ab_scaff_w_graft_gly_pose.simple_fold_tree( nres_Ab_scaff_w_graft );
		//		Ab_scaff_w_graft_gly_pose.set_fullatom_flag( true, false/*repack*/ );
		construct_pose_complex_from_p1_p2( Ab_oriented_pose             /*p1*/,
																			 scaff_w_graft_gly_pose       /*p2*/,
																			 Ab_scaff_w_graft_gly_pose    /*complex*/);
		if ( Ab_scaff_w_graft_gly_pose.total_residue() != nres_Ab_scaff_w_graft ) {
			std::cout << "Error Ab_scaff_w_graft_gly_pose.total_residue() != nres_Ab_scaff_w_graft... "
								<< Ab_scaff_w_graft_gly_pose.total_residue() << " " << nres_Ab_scaff_w_graft << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

    //
    //Score the Ab_scaff_w_graft_gly_pose to get fa_rep
    //

    Ab_scaff_w_graft_gly_pose.score( weight_map );
    float const fa_rep_Ab_scaff_w_graft_gly ( Ab_scaff_w_graft_gly_pose.get_0D_score( FA_REP ) );
		std::cout << Ab_scaff_w_graft_gly_pose.show_scores() << std::endl;


    ////////////////////////////////////////////////////////////////////
    //measure "inter" clash between Ab all-atom and scaff_w_graft backbone
    ////////////////////////////////////////////////////////////////////

    float const fa_rep_inter_gly ( fa_rep_Ab_scaff_w_graft_gly - fa_rep_Ab - fa_rep_scaff_w_graft_gly );

		std::cout << "fa_rep_inter_gly, fa_rep_Ab_scaff_w_graft_gly, fa_rep_Ab, fa_rep_scaff_w_graft_gly "
              << fa_rep_inter_gly << " " << fa_rep_Ab_scaff_w_graft_gly << " "
              << fa_rep_Ab << " " << fa_rep_scaff_w_graft_gly << std::endl;


    ///////////////////////////////////////////////////////////////////////////////
    //Ask if this inter clash is below the loose threshold
    //Using a loose threshold initially prior to closing the chainbreak(s)
    //
    float const max_fa_rep_inter_gly_before_close_default( 3000. ); //needs to be tested
    float max_fa_rep_inter_gly_before_close;
    realafteroption("max_fa_rep_inter_gly_before_close",max_fa_rep_inter_gly_before_close_default,max_fa_rep_inter_gly_before_close);

    //NOTE NOTE NOTE: this is where we could enter into the rb_move procedure
    //use rb_moves for cases in which intra_clash is fine but inter_clash
    //needs to be improved.
    //to test we need another, slightly looser threshold for inter_clash
    //if max_fa_rep_inter_gly_before_close < fa_inter < max_fa_rep_inter_gly_before_close_II
    //then we try rb_moves to see if we can maintain low intra clash
    //but also get low inter clash
    //

    //
    // if not satisfy loose_max, continue to next graft/scaff combo
    //
    if ( fa_rep_inter_gly > max_fa_rep_inter_gly_before_close ) {
			std::cout << "fa_rep_inter_gly too high, "
                << "going to next graft/scaff combo"
                << std::endl;
			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

      continue;
    }

    //
    //Parameters controlling how many graft or scaff
    //residues can move to close the graft
    //
    int n_graft_res_allowed_bbmove_nterm = 0;
    int n_graft_res_allowed_bbmove_cterm = 0;
    int n_scaff_gap_res_allowed_bbmove_nterm = 3;
    int n_scaff_gap_res_allowed_bbmove_cterm = 3;
    intafteroption("n_graft_res_allowed_bbmove_nterm",0,n_graft_res_allowed_bbmove_nterm);
    intafteroption("n_graft_res_allowed_bbmove_cterm",0,n_graft_res_allowed_bbmove_cterm);
    intafteroption("n_scaff_gap_res_allowed_bbmove_nterm",3,n_scaff_gap_res_allowed_bbmove_nterm);
    intafteroption("n_scaff_gap_res_allowed_bbmove_cterm",3,n_scaff_gap_res_allowed_bbmove_cterm);
    if ( n_scaff_gap_res_allowed_bbmove_nterm < 1 ) {
			std::cout << "command-line input error: n_scaff_gap_res_allowed_bbmove_nterm MUST be >= 1" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }
    if ( n_scaff_gap_res_allowed_bbmove_cterm < 1 ) {
			std::cout << "command-line input error: n_scaff_gap_res_allowed_bbmove_cterm MUST be >= 1" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
    }



    //
    //Protocol I : close using all-gly, then design, then refine
    //

    int const n_loops = 2;

		//
		//Setup loop info
		//
    FArray1D_int loop_begin( n_loops );
    FArray1D_int loop_end( n_loops );
    FArray1D_int cut_point( n_loops );

    //
    //list_disallow_bbmove is used
    //to set allow_bbmove=false ( and allow_chimove=false !) at positions within graft
    //
    int nres_disallow_bbmove = 0;
    FArray1D_int list_disallow_bbmove( 0 );//initially dimension 0

    //if graft has diff num of residues than native scaffold gap,
    //then residue numbering on cterminal side of gap will change
    //
    int const resnum_shift_due_to_graft = nres_scaff_w_graft - nres_scaff;

		//
		//if we have used rb_moves, then we will have to close at least 4 chainbreaks!!
		//
		bool const rb_moves_used = false; //andrew's rot/trans moves will set this to true.

		//
		//NOTE: for add_on graft_design, loop closure/refinement will be done in presence of Ab
		//So numbering must account for the fact that Ab will be first, scaff second.
		//

		setup_loops_for_add_on( rb_moves_used,
														nres_Ab,
														scaff_gap_resnum_1,                                   //for loop to be grafted
														scaff_gap_resnum_2,                                   //for loop to be grafted
														scaff_gap_resnum_1_already,//for loop already in scaff
														scaff_gap_resnum_2_already,//for loop already in scaff
														n_graft_res_allowed_bbmove_nterm,
														n_graft_res_allowed_bbmove_cterm,
														n_scaff_gap_res_allowed_bbmove_nterm,
														n_scaff_gap_res_allowed_bbmove_cterm,
														resnum_shift_due_to_graft,
														n_loops,
														loop_begin,
														loop_end,
														cut_point,
														nres_disallow_bbmove,
														list_disallow_bbmove
														);
    ///////////////////////////////////////////////////////////////////////////////
    //close the loops
    ///////////////////////////////////////////////////////////////////////////////


    float chainbreak_score = 1000.;
    float max_chainbreak_score_after_close = 0.1;
    realafteroption("max_chainbreak_score_after_close",max_chainbreak_score_after_close,max_chainbreak_score_after_close);

    int max_closure_attempts = 3;
    intafteroption("max_closure_attempts",max_closure_attempts,max_closure_attempts);

    Pose Ab_scaff_w_closed_graft_gly_pose;
    int count_closure_attempts = 0;
    while ( chainbreak_score > max_chainbreak_score_after_close && count_closure_attempts < max_closure_attempts ) {

      Ab_scaff_w_closed_graft_gly_pose = Ab_scaff_w_graft_gly_pose;

      close_graft( n_loops,
                   loop_begin,
                   loop_end,
                   cut_point,
                   nres_disallow_bbmove,
                   list_disallow_bbmove,
                   Ab_scaff_w_closed_graft_gly_pose,
                   chainbreak_score );

      ++count_closure_attempts;
			std::cout << "Finished closure attempt # " << count_closure_attempts << " with chainbreak_score = " << chainbreak_score << std::endl;
      if ( chainbreak_score > max_chainbreak_score_after_close && count_closure_attempts < max_closure_attempts ) {
				std::cout << "chainbreak_score > max_chainbreak_score_after_close....try again " << std::endl;
      }
    }


    if ( chainbreak_score > max_chainbreak_score_after_close ) {
			std::cout << "After max_closure_attempts, chainbreak_score gt "
								<< " max_chainbreak_score_after_close....skip to next "
								<< std::endl;
			//
			//checkpoint for clusters
			//
			if ( using_checkpoint ) {
				update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			}

			continue;
    }  else {
			std::cout << "chainbreak score ok !" << std::endl;
		}

		if ( output_debug_pdbs ) Ab_scaff_w_closed_graft_gly_pose.dump_pdb("Ab_scaff_w_closed_graft_gly_pose.pdb");


    ///////////////////////////////////////////////////////////////////////////////
    //measure "inter" clash again, now that we have closed the graft.
    ///////////////////////////////////////////////////////////////////////////////


		///  SKIP THIS FOR NOW SINCE WE ARE CLOSING WITH ANTIBODY PRESENT ///



    ////////////////////////////////////////////////////////////////////////////////////////////////
		//
		//First transfer all rotamers from scaffold
		//
    ////////////////////////////////////////////////////////////////////////////////////////////////

		FArray1D_bool natro_scaff( nres_scaff, false );
		FArray1D_bool natro_scaff_w_graft( nres_scaff_w_graft, false );

		natro_scaff_w_graft = false; //initialize all positions to false
		for ( int ires = 1; ires <= scaff_gap_resnum_1 - 1; ires++ ) {
			natro_scaff( ires ) = true;
			natro_scaff_w_graft( ires ) = true;
		}
		for ( int ires = scaff_gap_resnum_2 + 1; ires <= nres_scaff; ires++ ) {
			natro_scaff( ires ) = true;
		}
		for ( int ires = scaff_gap_resnum_2 + 1 + resnum_shift_due_to_graft; ires <= nres_scaff_w_graft; ires++ ) {
			natro_scaff_w_graft( ires ) = true;
		}


    //
		//Now duplicate natro_scaff_w_graft but put in bool array that includes Ab first
		//( false at all Ab positions )
    FArray1D_bool natro_scaff_w_graft_in_complex_w_Ab( nres_Ab_scaff_w_graft, false );
    for ( int ires = nres_Ab+1; ires <= nres_Ab_scaff_w_graft; ires++ ) {
      natro_scaff_w_graft_in_complex_w_Ab( ires ) = natro_scaff_w_graft( ires - nres_Ab );
    }

		std::cout << "about to transfer rots to scaffold" << std::endl;

    Pose Ab_scaff_w_closed_graft_natro_pose;
    Ab_scaff_w_closed_graft_natro_pose = Ab_scaff_w_closed_graft_gly_pose;
    transfer_native_rotamers( scaff_oriented_pose, natro_scaff, natro_scaff_w_graft_in_complex_w_Ab, Ab_scaff_w_closed_graft_natro_pose );

		std::cout << "done transferring rots to scaffold" << std::endl;

    ////////////////////////////////////////////////////////////////////////////////////////////////
		//
		//Now transfer select rotamers within the graft
		//    Only transfer rotamers that interact with Ab.
		//    These positions have been specified by input file, are stored in bool array natro_in_epitope.
		//
		//Note: the graft termini WILL  have natro !!
    ////////////////////////////////////////////////////////////////////////////////////////////////

		//
    //What subset of natro_in_epitope will be used in graft?
    //

		//keep track of epitope natro positions within the graft alone
    int n_natro_in_graft = 0;
		FArray1D_bool natro_graft( nres_graft, false );

		//keep track of epitope natro positions within the scaff_w_graft
		FArray1D_bool natro_epitope_in_scaff_w_graft( nres_scaff_w_graft, false );
    FArray1D_bool natro_epitope_in_Ab_scaff_w_graft( nres_Ab_scaff_w_graft, false );

		//graft goes from
		//    graft_resnum_1          to  graft_resnum_2                                   in graft numbering
		//    scaff_gap_resnum_1      to  scaff_gap_resnum_2 + resnum_shift_due_to_graft   in scaff_w_graft numbering

		for ( int ires = 1; ires <= nres_graft; ires++ ) {
      //ires_epitope tells position within epitope including all loops listed in order as in native_complex
      //but ires_epitope does not include nres_Ab; so ires_epitope is 1 at nres_Ab+1 in native_complex
      int ires_epitope          = ires - 1 + graft_resnum_1_in_native_complex - nres_Ab;
      int ires_scaff_w_graft    = ires - 1 + scaff_gap_resnum_1;
//	  @author: menis
//	  The first residue of the graft must be augmented by Ab and first residue of graft in scaffold numbering
//    int ires_Ab_scaff_w_graft = ires_epitope + nres_Ab;
      int ires_Ab_scaff_w_graft = ires_scaff_w_graft + nres_Ab;

			std::cout << " ires, ires_epitope, ires_scaff_w_graft, natro_in_epitope(ires_epitope) [ " << ires << " ][ "
								<< ires_epitope << " ][ " << ires_scaff_w_graft << " ][ " << natro_in_epitope( ires_epitope ) << " ]" << std::endl;

			if ( natro_in_epitope( ires_epitope ) ) {
				++n_natro_in_graft;

				//logic check
				if ( ires_scaff_w_graft > scaff_gap_resnum_2 + resnum_shift_due_to_graft ) {
					std::cout << "Error setting up natro_epitope_in_scaff_w_graft " << std::endl;
					std::cout << "ires_scaff_w_graft > scaff_gap_resnum_2 + resnum_shift_due_to_graft" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}

				//keep track of epitope natro positions within the graft alone
				natro_graft( ires ) = true;

				//keep track of epitope natro positions within the scaff_w_graft
				natro_epitope_in_scaff_w_graft( ires_scaff_w_graft ) = true;
        		natro_epitope_in_Ab_scaff_w_graft( ires_Ab_scaff_w_graft ) = true;

			}

			//Note the positions that are not natro_in_epitope will just be GLY unless we make them something else.
			// else {
			//	++n_ala_in_graft;
			//	list_ala_in_graft.redimension( n_ala_in_graft );
			//	list_ala_in_graft( n_ala_in_graft ) = ires;
			//			}
		}

		std::cout << "about to transfer rots to graft" << std::endl;

    transfer_native_rotamers( graft_pose, natro_graft, natro_epitope_in_Ab_scaff_w_graft, Ab_scaff_w_closed_graft_natro_pose );

    if ( output_debug_pdbs ) Ab_scaff_w_closed_graft_natro_pose.dump_pdb("Ab_scaff_w_closed_graft_natro_pose.pdb");


    ////////////////////////////////////////////////////////////////////////////////////////////////
    //Identify design positions
    ////////////////////////////////////////////////////////////////////////////////////////////////

    // NOTE intra design positions are those near any part of the "loop" region that allows bbmove
		// or any part of the graft

    //
    //figure out maximal range of resnums over which backbone was allowed to move
    //************ NEED TO CHANGE THIS TO MAKE A LIST OF MOVE_BB RESIDUES AND PASS THE LIST TO PREPARE_GRAFTING_PROTOCOL..******
    int begin_move_bb = 999999;//large number
    int end_move_bb = 0; //small number
    for ( int i_loop = 1; i_loop <= n_loops; i_loop++ ) {
      if ( loop_begin( i_loop ) < begin_move_bb ) begin_move_bb = loop_begin( i_loop );
      if ( loop_end( i_loop ) > end_move_bb ) end_move_bb = loop_end( i_loop );
    }

		if ( begin_move_bb == 999999 || end_move_bb == 0 ) {
			std::cout << "ERROR setting begin_move_bb, end_move_bb.." << begin_move_bb << " " << end_move_bb << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		//
		//Add pre-existing epitope to the natro positions so they will not be designed!
		//
		for ( int i = scaff_gap_resnum_1_already; i <= scaff_gap_resnum_2_already; i++ ) {
			assert( i < scaff_gap_resnum_1 || i > scaff_gap_resnum_2 );
			if ( i < scaff_gap_resnum_1 ) natro_epitope_in_scaff_w_graft( i ) = true;
			if ( i > scaff_gap_resnum_2 ) natro_epitope_in_scaff_w_graft( i+resnum_shift_due_to_graft ) = true;
		}

    FArray2D_bool design_matrix_Ab_scaff( param::MAX_AA(), nres_Ab_scaff_w_graft, false );
    prepare_grafting_protocol_design_matrix( Ab_scaff_w_closed_graft_natro_pose,     //input
                                             nres_Ab,                   //input
                                             begin_move_bb-nres_Ab,      //beginning of move_bb region in 1-nres_scaff
                                             end_move_bb-nres_Ab,        //end       of move_bb region in 1-nres_scaff
                                             natro_epitope_in_scaff_w_graft,  //input  epitope_natro_start_in_scaff,
                                             design_matrix_Ab_scaff );        //output

    {
      bool found( false );
      for ( int i = 1; i <= nres_Ab_scaff_w_graft; i++ ) {
        for ( int aa = 1; aa <= param::MAX_AA(); aa++ ) {
          if ( design_matrix_Ab_scaff( aa, i ) ) {
						if (runlevel_ns::runlevel > runlevel_ns::quiet) {
							std::cout << "design_matrix_Ab_scaff is true at ires, aa " << i << " " << aa << " "
												<< SS(param_aa::param_aa_data::aa_name3(aa)) << std::endl;
						}
            found = true;
          }
        }
      }
      if ( !found ) std::cout << "nothing is true in design_matrix_Ab_scaff!!!" << std::endl;
    }

    //
    //how many design iterations at each position?
    //( default is 3 )
    int n_design_iterations = 3;
    intafteroption( "n_design_iterations", n_design_iterations, n_design_iterations );

    //
    //arrays to store score values for each design iteration
    //

    //Only keep the Ab_scaff pose and store total energy in an array
    //rank the total E at end
    //output best x by rank
    //for each output, extract scaff and Ab poses and
    //compute ddg_bind and ddg_stability on the fly
    //

    //total scores
    //
    FArray1D_float score_Ab_scaff_design( n_design_iterations, 0.0 );
    FArray1D_int rank_to_index_score_Ab_scaff_design( n_design_iterations, 0 );

    //vector or FArray of pointers to poses
    //
    //    std::vector< Pose* > pose_list;
    FArray1D< Pose* > pose_array_Ab_scaff_design( n_design_iterations );

    for ( int i_design = 1; i_design <= n_design_iterations; i_design++ ) {

      Pose Ab_scaff_design_pose;
      Ab_scaff_design_pose = Ab_scaff_w_closed_graft_natro_pose;
      design_using_design_matrix( Ab_scaff_design_pose, design_matrix_Ab_scaff );

      //save score for  this design
      //
      Ab_scaff_design_pose.score( weight_map );
      score_Ab_scaff_design( i_design ) = Ab_scaff_design_pose.get_0D_score( SCORE );
			std::cout << "i_design, score " << i_design << " " << score_Ab_scaff_design( i_design ) << std::endl;

      //dump design pdb
      if ( output_debug_pdbs ) Ab_scaff_design_pose.dump_pdb( "Ab_scaff_design_pose_" + string_of( i_scaff ) + "_" + string_of( i_design ) + ".pdb" );

      //allocate memory for new poses
      //
      Pose * pose_ptr_Ab_scaff_design( new Pose );

      //attach pointers to actual poses
      //either from misc
      //pose_from_misc( *pose_ptr, true, false, true );
      //or get it directly
      *pose_ptr_Ab_scaff_design = Ab_scaff_design_pose;

      //store the pose pointer in an array
      //
      pose_array_Ab_scaff_design( i_design ) = pose_ptr_Ab_scaff_design;

      //initialize the rank_to_index arrays
      //
      rank_to_index_score_Ab_scaff_design( i_design ) = i_design;

    }//design iterations

    //rank the ddg values
    //( this sets rank_to_index... in ascending order )
    rank( n_design_iterations, score_Ab_scaff_design, rank_to_index_score_Ab_scaff_design );

    //
    //output design pdbs and scores, mutations
    //

		std::string scaff_base_filename = list_scaff_pdb( i_scaff );
    if ( has_suffix( scaff_base_filename, ".pdb.gz" ) ) scaff_base_filename.erase( scaff_base_filename.length() - 7 ); // strip terminal .pdb.gz
    if ( has_suffix( scaff_base_filename, ".pdb" ) ) scaff_base_filename.erase( scaff_base_filename.length() - 4 ); // strip terminal .pdb

		std::string filename_base = scaff_base_filename + "_" + lead_zero_string_of( nres_scaff_w_graft, 4 ) + "_" +
      lead_zero_string_of( graft_resnum_1, 3 ) + "-" + lead_zero_string_of( graft_resnum_2, 3 ) +
      "_" + lead_zero_string_of( scaff_gap_resnum_1, 4 ) + "-" + lead_zero_string_of( scaff_gap_resnum_2, 4 );

    //
    //How many designs to output?
    //

    int n_design_output = 1;
    intafteroption("n_design_output",n_design_output,n_design_output);

    if ( n_design_output > n_design_iterations ) {
      n_design_output = n_design_iterations;
			std::cout << " requested n_design_output is > n_design_iterations, resetting n_design_output" << std::endl;
    }

    for ( int i_rank = 1; i_rank <= n_design_output; i_rank++ ) {

      //note index corresponds to i_design
      int const i_design = rank_to_index_score_Ab_scaff_design( i_rank );

      Pose & Ab_scaff_out_pose( *pose_array_Ab_scaff_design( i_design ) ); // 1->

      //
      //Output energies: total E, ddg-bind, and ddg-stability for scaffold
      //

      Ab_scaff_out_pose.score( weight_map );
      float score_out = Ab_scaff_out_pose.get_0D_score( SCORE );
      if ( std::abs( score_out - score_Ab_scaff_design( i_design ) ) > 0.01 ) {
				std::cout << "!!!!!!!!!!!! WARNING WARNING WARNING WARNING !!!!!!!!!!!!!!!" << std::endl;
				std::cout << "ERROR score disagreement between score_out " << score_out
                  << " and score_Ab_scaff_design( i_design ) " << score_Ab_scaff_design( i_design ) << std::endl;
				std::cout << "!!!!!!!!!!!! WARNING WARNING WARNING WARNING !!!!!!!!!!!!!!!" << std::endl;
      }
      float  fa_rep_Ab_scaff_design  ( Ab_scaff_out_pose.get_0D_score( FA_REP ) );
      float  fa_atr_Ab_scaff_design  ( Ab_scaff_out_pose.get_0D_score( FA_ATR ) );
      float  fa_sol_Ab_scaff_design  ( Ab_scaff_out_pose.get_0D_score( FA_SOL ) );
      float  fa_dun_Ab_scaff_design  ( Ab_scaff_out_pose.get_0D_score( FA_DUN ) );
      float  hbsc_Ab_scaff_design    ( Ab_scaff_out_pose.get_0D_score( HB_SC ) );
      float  fa_prob_Ab_scaff_design ( Ab_scaff_out_pose.get_0D_score( FA_PROB ) );

      //
      //Here extract scaff and Ab poses and compute ddb_bind and ddg_stability
      //

      Pose scaff_design_pose;
      scaff_design_pose.simple_fold_tree( nres_scaff_w_graft );
      scaff_design_pose.set_fullatom_flag( true, false /*repack*/ );
      copy_segment_to_new_pose( Ab_scaff_out_pose /*src_pose*/,
                                nres_Ab+1 /*begin_src*/,
                                nres_Ab_scaff_w_graft /*end_src*/,
                                scaff_design_pose /*pose_out*/);
      scaff_design_pose.score( weight_map );
      //      scaff_design_pose.dump_pdb("scaff_design_pose.pdb");
      Pose Ab_design_pose;
      Ab_design_pose.simple_fold_tree( nres_Ab );
      Ab_design_pose.set_fullatom_flag( true, false /*repack*/ );
      //          Ab_design_pose.copy_segment( nres_Ab, Ab_scaff_out_pose, 1, 1 /*begin_src*/ );
      copy_segment_to_new_pose( Ab_scaff_out_pose /*src_pose*/,
                                1                 /*begin_src*/,
                                nres_Ab           /*end_src*/,
                                Ab_design_pose    /*pose_out*/);
      Ab_design_pose.score( weight_map );
      //      Ab_design_pose.dump_pdb("Ab_design_pose.pdb");

      //extract scaff_design_pose and Ab_design_pose
      //score each

      scaff_design_pose.score( weight_map );
      float score_scaff_design   ( scaff_design_pose.get_0D_score( SCORE ) );
      float fa_rep_scaff_design  ( scaff_design_pose.get_0D_score( FA_REP ) );
      float fa_atr_scaff_design  ( scaff_design_pose.get_0D_score( FA_ATR ) );
      float fa_sol_scaff_design  ( scaff_design_pose.get_0D_score( FA_SOL ) );
      float fa_dun_scaff_design  ( scaff_design_pose.get_0D_score( FA_DUN ) );
      float hbsc_scaff_design    ( scaff_design_pose.get_0D_score( HB_SC ) );
      float fa_prob_scaff_design ( scaff_design_pose.get_0D_score( FA_PROB ) );

      Ab_design_pose.score( weight_map );
      float score_Ab_design   ( Ab_design_pose.get_0D_score( SCORE )  );
      float fa_rep_Ab_design  ( Ab_design_pose.get_0D_score( FA_REP ) );
      float fa_atr_Ab_design  ( Ab_design_pose.get_0D_score( FA_ATR ) );
      float fa_sol_Ab_design  ( Ab_design_pose.get_0D_score( FA_SOL ) );
      float fa_dun_Ab_design  ( Ab_design_pose.get_0D_score( FA_DUN ) );
      float hbsc_Ab_design    ( Ab_design_pose.get_0D_score( HB_SC ) );
      float fa_prob_Ab_design ( Ab_design_pose.get_0D_score( FA_PROB ) );

      //ddG-bind total score and components
      //
      float ddg_bind_Ab_scaff         = score_Ab_scaff_design( i_design ) - score_scaff_design - score_Ab_design;
      float ddg_bind_Ab_scaff_fa_rep  = fa_rep_Ab_scaff_design  - fa_rep_scaff_design  - fa_rep_Ab_design;
      float ddg_bind_Ab_scaff_fa_atr  = fa_atr_Ab_scaff_design  - fa_atr_scaff_design  - fa_atr_Ab_design;
      float ddg_bind_Ab_scaff_fa_sol  = fa_sol_Ab_scaff_design  - fa_sol_scaff_design  - fa_sol_Ab_design;
      float ddg_bind_Ab_scaff_fa_dun  = fa_dun_Ab_scaff_design  - fa_dun_scaff_design  - fa_dun_Ab_design;
      float ddg_bind_Ab_scaff_hbsc    = hbsc_Ab_scaff_design    - hbsc_scaff_design    - hbsc_Ab_design;
      float ddg_bind_Ab_scaff_fa_prob = fa_prob_Ab_scaff_design - fa_prob_scaff_design - fa_prob_Ab_design;

      //ddG-stability total score and components
      //
      float ddg_scaff                 = score_scaff_design   - score_scaff_native;
      float ddg_scaff_fa_rep          = fa_rep_scaff_design  - fa_rep_scaff_native;
      float ddg_scaff_fa_atr          = fa_atr_scaff_design  - fa_atr_scaff_native;
      float ddg_scaff_fa_sol          = fa_sol_scaff_design  - fa_sol_scaff_native;
      float ddg_scaff_fa_dun          = fa_dun_scaff_design  - fa_dun_scaff_native;
      float ddg_scaff_hbsc            = hbsc_scaff_design    - hbsc_scaff_native;
      float ddg_scaff_fa_prob         = fa_prob_scaff_design - fa_prob_scaff_native;
			std::string filename;
			std::string design_id_str = right_string_of( i_design, 3, '0' );
      int const i_refine = 0;
			std::string refine_id_str = right_string_of( i_refine, 3, '0' );
			std::string filename_base2 = filename_base + "_" + design_id_str + "_" + refine_id_str;
      get_filename_with_energies( filename_base2, score_Ab_scaff_design( i_design ), ddg_bind_Ab_scaff, filename );

      if ( output_pdbs ) {
				Ab_scaff_out_pose.pdb_info().set_use_pdb_numbering( true );
        Ab_scaff_out_pose.dump_pdb( files_paths::pdb_out_path + filename );
      }

      outfile   << SS( "ddg_bind_Ab_scaff: " )
                << SS( "ides" )
                << SS( "     ddg" )
                << SS( "     atr" )
                << SS( "     rep" )
                << SS( "     sol" )
                << SS( "     dun" )
                << SS( "    hbsc" )
                << SS( "    prob" )
                << SS( "   scaff" )
                << std::endl;

      outfile   << SS( "ddg_bind_Ab_scaff: " )
                << I( 4, i_design ) << " "
                << F( 8, 2, ddg_bind_Ab_scaff        ) << " "
                << F( 8, 2, ddg_bind_Ab_scaff_fa_atr ) << " "
                << F( 8, 2, ddg_bind_Ab_scaff_fa_rep ) << " "
                << F( 8, 2, ddg_bind_Ab_scaff_fa_sol ) << " "
                << F( 8, 2, ddg_bind_Ab_scaff_fa_dun ) << " "
                << F( 8, 2, ddg_bind_Ab_scaff_hbsc   ) << " "
                << F( 8, 2, ddg_bind_Ab_scaff_fa_prob) << " "
                << filename
                << std::endl;

      outfile   << SS( "ddg_scaff        : " )
                << SS( "ides" )
                << SS( "     ddg" )
                << SS( "     atr" )
                << SS( "     rep" )
                << SS( "     sol" )
                << SS( "     dun" )
                << SS( "    hbsc" )
                << SS( "    prob" )
                << SS( "   scaff" )
                << std::endl;

      outfile   << SS( "ddg_scaff        : " )
                << I( 4, i_design ) << " "
                << F( 8, 2, ddg_scaff        ) << " "
                << F( 8, 2, ddg_scaff_fa_atr ) << " "
                << F( 8, 2, ddg_scaff_fa_rep ) << " "
                << F( 8, 2, ddg_scaff_fa_sol ) << " "
                << F( 8, 2, ddg_scaff_fa_dun ) << " "
                << F( 8, 2, ddg_scaff_hbsc   ) << " "
                << F( 8, 2, ddg_scaff_fa_prob) << " "
                << filename
                << std::endl;



			//
			//output list of mutations
			//

			//regions in scaff to specify mutations:
			//1 -> scaff_gap_resum_1-1                                                  nterm of graft
			//scaff_gap_resnum_1 -> scaff_gap_resum_2+resnum_shift_due_to_graft         within graft
			//scaff_gap_resum_2+1+resnum_shift_due_to_graft -> nres_scaff_w_graft       cterm of graft

			for ( int ires = 1; ires < scaff_gap_resnum_1; ires++ ) {
				int ires_scaff_in_cmplx = ires + nres_Ab; //scaff positions in cmplx of Ab+scaff, Ab is first
				int const aa_wt = scaff_pose.res( ires );
				int const aa = Ab_scaff_out_pose.res( ires_scaff_in_cmplx );
				if ( aa != aa_wt ) {
					std::string const aa3_wt = param_aa::param_aa_data::aa_name3( aa_wt );
					std::string const aa3 = param_aa::param_aa_data::aa_name3( aa );
					outfile   << "Mutation:" << I( 4, ires ) << " " << SS( aa3_wt ) << " --> " << I( 4, ires_scaff_in_cmplx ) << SS( aa3 ) << " "
										<< filename << " nterm_to_graft" << std::endl;
				}
			}

			//
			//*For now just consider all positions within the graft as a mutation*
			//
			//But we could make this better:
			//For outputting mutations we should in principle consider whether there is
			//an insertion or a deletion due to the graft
			//and whether the insertion/deletion is on the n or c side of the graft
			//then we can consider other positions as point mutations and it is possible that
			//some positions will remain WT
			//

			for ( int ires = scaff_gap_resnum_1; ires <= scaff_gap_resnum_2+resnum_shift_due_to_graft; ires++ ) {

				int ires_scaff_in_cmplx = ires + nres_Ab; //scaff positions in cmplx of Ab+scaff, Ab is first
				int const aa = Ab_scaff_out_pose.res( ires_scaff_in_cmplx );
					std::string const aa3 = param_aa::param_aa_data::aa_name3( aa );
					outfile   << "Mutation:" << SS( "XXXX" ) << " " << SS( "XXX" ) << " --> " << I( 4, ires_scaff_in_cmplx ) << SS( aa3 ) << " "
										<< filename << " within_graft" << std::endl;
			}

			for ( int ires = scaff_gap_resnum_2+1+resnum_shift_due_to_graft; ires <= nres_scaff_w_graft; ires++ ) {

				//ires_scaff is scaff positions in cmplx of Ab+scaff, Ab is first
				int ires_scaff_in_cmplx = ires + nres_Ab;
				int ires_scaff_no_graft = ires - resnum_shift_due_to_graft;//ires in wt scaffold

				int const aa_wt = scaff_pose.res( ires_scaff_no_graft );
				int const aa = Ab_scaff_out_pose.res( ires_scaff_in_cmplx );
				if ( aa != aa_wt ) {
					std::string const aa3_wt = param_aa::param_aa_data::aa_name3( aa_wt );
					std::string const aa3 = param_aa::param_aa_data::aa_name3( aa );
					outfile   << "Mutation:" << I( 4, ires ) << " " << SS( aa3_wt ) << " --> " << I( 4, ires_scaff_in_cmplx ) << SS( aa3 ) << " "
										<< filename << " graft_to_cterm" << std::endl;
				}
			}
		}

    //
    //loop refinement step
    //
    //---------- is it necessary?
    //---------- should we minimize the surrounding scaffold backbone ?
    //---------- will we need to relax/minimize the entire scaffold backbone ?


    //
    //make list of the unique designs
    //
    //Note this is stupid, we should avoid saving duplicate designs
    //in the pose_array above

    int n_unique_designs = 0;
    FArray1D_int list_unique_designs_by_rank( n_design_iterations, 0 );
    float Eprev = 99999.;
    for ( int i_rank = 1; i_rank <= n_design_iterations; i_rank++ ) {
      float Edes = score_Ab_scaff_design( i_rank );
      if ( Edes != Eprev ) {
        list_unique_designs_by_rank( ++n_unique_designs ) = i_rank;
      }
    }
    list_unique_designs_by_rank.redimension( n_unique_designs );
    //
    //How many designs to refine?
    //

    int n_designs_to_refine = 1;
    intafteroption( "n_designs_to_refine", n_designs_to_refine, n_designs_to_refine );

    if ( n_designs_to_refine > n_unique_designs ) {
			std::cout << "WARNING: n_designs_to_refine > n_unique_designs ... resetting n_designs_to_refine to n_unique_designs = " << n_unique_designs << std::endl;
      n_designs_to_refine = n_unique_designs;
    }

    for ( int i_design_to_refine = 1; i_design_to_refine <= n_designs_to_refine; i_design_to_refine++ ) {

      int i_rank = list_unique_designs_by_rank( i_design_to_refine );

      //index to the array of designed poses
      int const i_design = rank_to_index_score_Ab_scaff_design( i_rank );

      Pose & Ab_scaff_design_pose( *pose_array_Ab_scaff_design( i_design ) ); // 1->

      //
      //Output energies: total E, ddg-bind, and ddg-stability for scaffold
      //

      Ab_scaff_design_pose.score( weight_map );
      float score_out = Ab_scaff_design_pose.get_0D_score( SCORE );
      if ( std::abs( score_out - score_Ab_scaff_design( i_design ) ) > 0.01 ) {
				std::cout << "!!!!!!!!!!!! WARNING WARNING WARNING WARNING !!!!!!!!!!!!!!!" << std::endl;
				std::cout << "ERROR score disagreement between score_out " << score_out
                  << " and score_Ab_scaff_design( i_design ) " << score_Ab_scaff_design( i_design ) << std::endl;
				std::cout << "!!!!!!!!!!!! WARNING WARNING WARNING WARNING !!!!!!!!!!!!!!!" << std::endl;
      }
      float  fa_rep_Ab_scaff_design  ( Ab_scaff_design_pose.get_0D_score( FA_REP ) );
      float  fa_atr_Ab_scaff_design  ( Ab_scaff_design_pose.get_0D_score( FA_ATR ) );
      float  fa_sol_Ab_scaff_design  ( Ab_scaff_design_pose.get_0D_score( FA_SOL ) );
      float  fa_dun_Ab_scaff_design  ( Ab_scaff_design_pose.get_0D_score( FA_DUN ) );
      float  hbsc_Ab_scaff_design    ( Ab_scaff_design_pose.get_0D_score( HB_SC ) );
      float  fa_prob_Ab_scaff_design ( Ab_scaff_design_pose.get_0D_score( FA_PROB ) );

      //
      //Here extract scaff and Ab poses and compute ddb_bind and ddg_stability
      //

      Pose scaff_design_pose;
      scaff_design_pose.simple_fold_tree( nres_scaff_w_graft );
      scaff_design_pose.set_fullatom_flag( true, false /*repack*/ );
      //      scaff_design_pose.copy_segment( nres_scaff_w_graft, Ab_scaff_design_pose, 1, nres_Ab+1 /*begin_src*/ );
      copy_segment_to_new_pose( Ab_scaff_design_pose /*src_pose*/,
                                nres_Ab+1 /*begin_src*/,
                                nres_Ab_scaff_w_graft /*end_src*/,
                                scaff_design_pose /*pose_out*/);
      scaff_design_pose.score( weight_map );
      if ( output_debug_pdbs ) scaff_design_pose.dump_pdb("scaff_design_pose_before_refine_" + string_of( i_scaff ) + "_" + string_of( i_design ) + ".pdb");
      if ( output_debug_pdbs ) Ab_scaff_design_pose.dump_pdb("Ab_scaff_design_pose_before_refine_" + string_of( i_scaff ) + "_" + string_of( i_design ) + ".pdb");

      Pose Ab_design_pose;
      Ab_design_pose.simple_fold_tree( nres_Ab );
      Ab_design_pose.set_fullatom_flag( true, false /*repack*/ );
      //          Ab_design_pose.copy_segment( nres_Ab, Ab_scaff_design_pose, 1, 1 /*begin_src*/ );
      copy_segment_to_new_pose( Ab_scaff_design_pose /*src_pose*/,
                                1                 /*begin_src*/,
                                nres_Ab           /*end_src*/,
                                Ab_design_pose    /*pose_out*/);
      Ab_design_pose.score( weight_map );
      //      Ab_design_pose.dump_pdb("Ab_design_pose.pdb");

      //
      //Now do the loop refinement or any other desired refinement
      //

      int n_refine_iterations = 1;
      intafteroption( "n_refine_iterations", n_refine_iterations, n_refine_iterations );

      int const max_refine_iterations = 50;

      if ( n_refine_iterations > max_refine_iterations ) {
				std::cout << "WARNING: requested n_refine_iterations > max_refine_iterations, resetting to max = " << max_refine_iterations << std::endl;
        n_refine_iterations = max_refine_iterations;
      }
      for ( int i_refine = 1; i_refine <= n_refine_iterations; i_refine++ ) {

        Pose Ab_scaff_refine_pose;
        Ab_scaff_refine_pose = Ab_scaff_design_pose;

        float chainbreak_score_refine = 1000.;
        close_graft( n_loops,
                     loop_begin,
                     loop_end,
                     cut_point,
                     nres_disallow_bbmove,
                     list_disallow_bbmove,
                     Ab_scaff_refine_pose,
                     chainbreak_score_refine );

        if ( chainbreak_score_refine > max_chainbreak_score_after_close ) {
					std::cout << "!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!" << std::endl;
					std::cout << "!!WARNING!! After refine: chainbreak_score_refine = " << chainbreak_score_refine << " > max_chainbreak_score_after_close" << std::endl;
					std::cout << "!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!!!WARNING!!" << std::endl;
        }

        if ( output_debug_pdbs ) Ab_scaff_refine_pose.dump_pdb( "Ab_scaff_refine_pose_" + string_of( i_design ) + "_" + string_of( i_refine ) +"_.pdb");

				//
				//make Ab_refine and scaff_refine poses
				//
				Pose Ab_refine_pose;
				Ab_refine_pose.simple_fold_tree( nres_Ab );
				Ab_refine_pose.set_fullatom_flag( true, false );
				copy_segment_to_new_pose( Ab_scaff_refine_pose /*src_pose*/,
																	1                 /*begin_src*/,
																	nres_Ab           /*end_src*/,
																	Ab_refine_pose    /*pose_out*/);
				Pose scaff_refine_pose;
				scaff_refine_pose.simple_fold_tree( nres_scaff_w_graft );
				scaff_refine_pose.set_fullatom_flag( true, false );
				copy_segment_to_new_pose( Ab_scaff_refine_pose /*src_pose*/,
																	nres_Ab+1             /*begin_src*/,
																	nres_Ab_scaff_w_graft /*end_src*/,
																	scaff_refine_pose     /*pose_out*/);


        //
        //Output energies: total E, ddg-bind, and ddg-stability for scaffold
        //

        Ab_scaff_refine_pose.score( weight_map );
        float  score_Ab_scaff_refine    ( Ab_scaff_refine_pose.get_0D_score( SCORE ) );
        float  fa_rep_Ab_scaff_refine  ( Ab_scaff_refine_pose.get_0D_score( FA_REP ) );
        float  fa_atr_Ab_scaff_refine  ( Ab_scaff_refine_pose.get_0D_score( FA_ATR ) );
        float  fa_sol_Ab_scaff_refine  ( Ab_scaff_refine_pose.get_0D_score( FA_SOL ) );
        float  fa_dun_Ab_scaff_refine  ( Ab_scaff_refine_pose.get_0D_score( FA_DUN ) );
        float  hbsc_Ab_scaff_refine    ( Ab_scaff_refine_pose.get_0D_score( HB_SC ) );
        float  fa_prob_Ab_scaff_refine ( Ab_scaff_refine_pose.get_0D_score( FA_PROB ) );

        scaff_refine_pose.score( weight_map );
        float score_scaff_refine   ( scaff_refine_pose.get_0D_score( SCORE ) );
        float fa_rep_scaff_refine  ( scaff_refine_pose.get_0D_score( FA_REP ) );
        float fa_atr_scaff_refine  ( scaff_refine_pose.get_0D_score( FA_ATR ) );
        float fa_sol_scaff_refine  ( scaff_refine_pose.get_0D_score( FA_SOL ) );
        float fa_dun_scaff_refine  ( scaff_refine_pose.get_0D_score( FA_DUN ) );
        float hbsc_scaff_refine    ( scaff_refine_pose.get_0D_score( HB_SC ) );
        float fa_prob_scaff_refine ( scaff_refine_pose.get_0D_score( FA_PROB ) );

        Ab_refine_pose.score( weight_map );
        float score_Ab_refine   ( Ab_refine_pose.get_0D_score( SCORE )  );
        float fa_rep_Ab_refine  ( Ab_refine_pose.get_0D_score( FA_REP ) );
        float fa_atr_Ab_refine  ( Ab_refine_pose.get_0D_score( FA_ATR ) );
        float fa_sol_Ab_refine  ( Ab_refine_pose.get_0D_score( FA_SOL ) );
        float fa_dun_Ab_refine  ( Ab_refine_pose.get_0D_score( FA_DUN ) );
        float hbsc_Ab_refine    ( Ab_refine_pose.get_0D_score( HB_SC ) );
        float fa_prob_Ab_refine ( Ab_refine_pose.get_0D_score( FA_PROB ) );

        //ddG-bind total score and components
        //
        float ddg_bind_Ab_scaff_refine         = score_Ab_scaff_refine   - score_scaff_refine - score_Ab_refine;
        float ddg_bind_Ab_scaff_fa_rep_refine  = fa_rep_Ab_scaff_refine  - fa_rep_scaff_refine  - fa_rep_Ab_refine;
        float ddg_bind_Ab_scaff_fa_atr_refine  = fa_atr_Ab_scaff_refine  - fa_atr_scaff_refine  - fa_atr_Ab_refine;
        float ddg_bind_Ab_scaff_fa_sol_refine  = fa_sol_Ab_scaff_refine  - fa_sol_scaff_refine  - fa_sol_Ab_refine;
        float ddg_bind_Ab_scaff_fa_dun_refine  = fa_dun_Ab_scaff_refine  - fa_dun_scaff_refine  - fa_dun_Ab_refine;
        float ddg_bind_Ab_scaff_hbsc_refine    = hbsc_Ab_scaff_refine    - hbsc_scaff_refine    - hbsc_Ab_refine;
        float ddg_bind_Ab_scaff_fa_prob_refine = fa_prob_Ab_scaff_refine - fa_prob_scaff_refine - fa_prob_Ab_refine;

        //ddG-stability total score and components
        //
        float ddg_scaff_refine                 = score_scaff_refine   - score_scaff_native;
        float ddg_scaff_fa_rep_refine          = fa_rep_scaff_refine  - fa_rep_scaff_native;
        float ddg_scaff_fa_atr_refine          = fa_atr_scaff_refine  - fa_atr_scaff_native;
        float ddg_scaff_fa_sol_refine          = fa_sol_scaff_refine  - fa_sol_scaff_native;
        float ddg_scaff_fa_dun_refine          = fa_dun_scaff_refine  - fa_dun_scaff_native;
        float ddg_scaff_hbsc_refine            = hbsc_scaff_refine    - hbsc_scaff_native;
        float ddg_scaff_fa_prob_refine         = fa_prob_scaff_refine - fa_prob_scaff_native;


				std::string filename;
				std::string design_id_str = right_string_of( i_design, 3, '0' );
				std::string refine_id_str = right_string_of( i_refine, 3, '0' );
				std::string filename_base2 = filename_base + "_" + design_id_str + "_" + refine_id_str;

        get_filename_with_energies( filename_base2, score_Ab_scaff_refine, ddg_bind_Ab_scaff_refine, filename );

        if ( output_pdbs ) {
					Ab_scaff_refine_pose.pdb_info().set_use_pdb_numbering( true );
          Ab_scaff_refine_pose.dump_pdb( files_paths::pdb_out_path + filename );
        }

				outfile   << SS( "ddg_bind_Ab_scaff_refine:" )
									<< SS( "ides" )
									<< SS( "iref" )
									<< SS( "     ddg" )
									<< SS( "     atr" )
									<< SS( "     rep" )
									<< SS( "     sol" )
									<< SS( "     dun" )
									<< SS( "    hbsc" )
									<< SS( "    prob" )
									<< SS( "   scaff" )
									<< std::endl;

				outfile   << SS( "ddg_bind_Ab_scaff_refine: " )
									<< I( 4, i_design ) << " "
									<< I( 4, i_refine ) << " "
									<< F( 8, 2, ddg_bind_Ab_scaff_refine        ) << " "
									<< F( 8, 2, ddg_bind_Ab_scaff_fa_atr_refine ) << " "
									<< F( 8, 2, ddg_bind_Ab_scaff_fa_rep_refine ) << " "
									<< F( 8, 2, ddg_bind_Ab_scaff_fa_sol_refine ) << " "
									<< F( 8, 2, ddg_bind_Ab_scaff_fa_dun_refine ) << " "
									<< F( 8, 2, ddg_bind_Ab_scaff_hbsc_refine   ) << " "
									<< F( 8, 2, ddg_bind_Ab_scaff_fa_prob_refine) << " "
									<< filename
									<< std::endl;

				outfile   << SS( "ddg_scaff_refine        :" )
									<< SS( "ides" )
									<< SS( "iref" )
									<< SS( "     ddg" )
									<< SS( "     atr" )
									<< SS( "     rep" )
									<< SS( "     sol" )
									<< SS( "     dun" )
									<< SS( "    hbsc" )
									<< SS( "    prob" )
									<< SS( "   chbk" )
					//                << SS( "   chbk_ov" )
									<< SS( "   scaff" )
									<< std::endl;

				outfile   << SS( "ddg_scaff_refine        : " )
									<< I( 4, i_design ) << " "
									<< I( 4, i_refine ) << " "
									<< F( 8, 2, ddg_scaff_refine        ) << " "
									<< F( 8, 2, ddg_scaff_fa_atr_refine ) << " "
									<< F( 8, 2, ddg_scaff_fa_rep_refine ) << " "
									<< F( 8, 2, ddg_scaff_fa_sol_refine ) << " "
									<< F( 8, 2, ddg_scaff_fa_dun_refine ) << " "
									<< F( 8, 2, ddg_scaff_hbsc_refine   ) << " "
									<< F( 8, 2, ddg_scaff_fa_prob_refine) << " "
									<< F( 8, 4, chainbreak_score_refine ) << " "
					//                << F( 8, 2, chainbreak_overlap_scaff_refine ) << " "
									<< filename
									<< std::endl;


      }//refine iterations

    }//i_design_to_refine

    //
    //Delete the array of pointers
    //
    for ( int i_design = 1; i_design <= n_design_iterations; ++i_design ) {
      //      delete pose_list[k];
      delete pose_array_Ab_scaff_design( i_design );
    }

    //
    //checkpoint for clusters
    //
    if ( using_checkpoint ) {
      update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
    }

  }//i_scaff

  //
  //Close the output file
  //
  outfile.close();
  outfile.clear();
}
//////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////
void
get_closure_atom_resnum(
												bool const & fullatom,
												std::string const & graft_dir,
												int const & graft_resnum_1,
												int const & graft_resnum_2,
												int const & jres_scaff,
												FArray1D_int & closure_atom,
												FArray1D_int & closure_resnum_scaff,
												FArray1D_int & closure_resnum_graft
												)
{

	//
	//atomnums in full_coord or Eposition
	//
	int atomnum_N,atomnum_CA,atomnum_C,atomnum_O,atomnum_CB;
	if ( fullatom ) {
		atomnum_N =  1;
		atomnum_CA = 2;
		atomnum_C =  3;
		atomnum_O = 4;
		atomnum_CB = 5;
	} else {
		atomnum_N =  1;
		atomnum_CA = 2;
		atomnum_C =  4;
		atomnum_O = 5;
		atomnum_CB = 3;
	}

	if ( graft_dir == "n2c" ) {

		closure_atom(   1 ) = atomnum_C;
		closure_atom(   2 ) = atomnum_N;
		closure_atom(   3 ) = atomnum_CA;
		closure_atom(   4 ) = atomnum_C;
		closure_atom(   5 ) = atomnum_O;

		//
		//peptide closure atoms at graft_resnum_2
		//
		closure_resnum_graft( 1 ) = graft_resnum_2 - 1;
		closure_resnum_graft( 2 ) = graft_resnum_2;
		closure_resnum_graft( 3 ) = graft_resnum_2;
		closure_resnum_graft( 4 ) = graft_resnum_2;
		closure_resnum_graft( 5 ) = graft_resnum_2;

		//
		//scaff closure resnum:
		//
		closure_resnum_scaff( 1 ) = jres_scaff - 1;
		closure_resnum_scaff( 2 ) = jres_scaff;
		closure_resnum_scaff( 3 ) = jres_scaff;
		closure_resnum_scaff( 4 ) = jres_scaff;
		closure_resnum_scaff( 5 ) = jres_scaff;

	} else { // graft_dir == c2n

		closure_atom(   1 ) = atomnum_N;
		closure_atom(   2 ) = atomnum_CA;
		closure_atom(   3 ) = atomnum_C;
		closure_atom(   4 ) = atomnum_O;
		closure_atom(   5 ) = atomnum_N;

		//
		//peptide closure atoms at graft_resnum_1
		//
		closure_resnum_graft( 1 ) = graft_resnum_1;
		closure_resnum_graft( 2 ) = graft_resnum_1;
		closure_resnum_graft( 3 ) = graft_resnum_1;
		closure_resnum_graft( 4 ) = graft_resnum_1;
		closure_resnum_graft( 5 ) = graft_resnum_1 + 1;

		//
		//scaff closure atoms, seqpos:
		//
		closure_resnum_scaff( 1 ) = jres_scaff;
		closure_resnum_scaff( 2 ) = jres_scaff;
		closure_resnum_scaff( 3 ) = jres_scaff;
		closure_resnum_scaff( 4 ) = jres_scaff;
		closure_resnum_scaff( 5 ) = jres_scaff + 1;
	}
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
prepare_grafting_protocol_design_matrix(
																				pose_ns::Pose & pose,
																				int const & nres_Ab,
																				int const & begin_move_bb,
																				int const & end_move_bb,
																				FArray1D_bool const & epitope_natro_start_in_scaff,
																				FArray2D_bool & design_matrix
																				)
{

	using namespace pose_ns;//for Score_weight_map
  Score_weight_map weight_map( score12 );

	//Note: assumes that the pose contains Ab first, scaff second.
	//

	int const nres_Ab_scaff = pose.total_residue();
	int const nres_scaff = nres_Ab_scaff - nres_Ab;


	///////////////////////////////////////////////////////////////////////////////////////////////////
	//parameters for what can repack during design
	///////////////////////////////////////////////////////////////////////////////////////////////////

	bool allow_repack_epitope = false;
	if ( truefalseoption( "allow_repack_epitope" ) ) allow_repack_epitope = true;

	bool allow_repack_Ab = false;
	if ( truefalseoption( "allow_repack_Ab" ) ) allow_repack_Ab = true;

	bool allow_ALLAA_at_inter_despos = false;
	if (  truefalseoption( "allow_ALLAA_at_inter_despos" ) ) allow_ALLAA_at_inter_despos = true;

	///////////////////////////////////////////////////////////////////////////////////////////////////
	// inter,intra design positions ( within scaffold only )
	//
	//Note scaff positions within complex are always i+nres_Ab
	//
	FArray1D_bool is_inter_despos( nres_Ab_scaff, false );
	FArray1D_bool is_intra_despos( nres_Ab_scaff, false );


	///////////////////////////////////////////////////////////////////////////////////////////////////
	//Ab residues at the interface
	//
	FArray1D_bool is_inter_seqpos_Ab( nres_Ab, false );

	///////////////////////////////////////////////////////////////////////////////////////////////////
	//distance cutoffs for heavy-heavy distances
	//
	//default cutoffs are 4.0 for immunogen design
	//but for inhibitor design cutoff_inter should be expanded to 6.0
	//
	float const default_cutoff_inter = 4.0;
	float const cutoff_inter = realafteroption("cutoff_inter",default_cutoff_inter);
	float const default_cutoff_intra = 4.0;
	float const cutoff_intra = realafteroption("cutoff_intra",default_cutoff_intra);

	///////////////////////////////////////////////////////////////////////////////////////////////////
	//cutoff_Ab_epitope used when want to repack Ab positions not in direct contact with epitope
	//
	float const default_cutoff_Ab_epitope = 4.5;
	float const cutoff_Ab_epitope = realafteroption("cutoff_Ab_epitope",default_cutoff_Ab_epitope);

	std::cout << "begin_move_bb, end_move_bb " << begin_move_bb << " " << end_move_bb << std::endl;

	//make a list of epitope positions within the scaff
	//
	FArray1D_int list_epitope_natro_start_in_scaff( 0 );
	int n_epitope_natro_start = 0;
	for ( int i = 1; i <= nres_scaff; i++ ) {
		if ( epitope_natro_start_in_scaff( i ) ) {
			++n_epitope_natro_start;
			list_epitope_natro_start_in_scaff.redimension( n_epitope_natro_start );
			list_epitope_natro_start_in_scaff( n_epitope_natro_start ) = i;
		}
	}


	for ( int i = 1; i <= nres_scaff; i++ ) {

		//exclude epitope natro positions from inter/intra design
		//unless allow_repack_epitope is true ( that is covered below )
		//
		if ( epitope_natro_start_in_scaff( i ) ) {
			if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
				std::cout << "prepare_design_matrix: exclude epitope_natro from design at i = " << i << std::endl;
		}
			continue;
		}

		//ires_scaff is resnum in Ab-scaff cmplx...Ab is first, scaff second
		int const ires_scaff = i+nres_Ab;

		std::cout << "i " << i << " ires_scaff " << ires_scaff << std::endl;

		int start_atm1, end_atm1;
		int const aa1 = pose.res( ires_scaff );
		int const aav1 = pose.res_variant( ires_scaff );
		if ( aa1 == param_aa::aa_gly ) {
			start_atm1 = 2;
			end_atm1 = 2;
		} else {
			start_atm1 = 5;
			end_atm1 = aaproperties_pack::nheavyatoms( aa1, aav1 );
		}

		/////////////////////////////////////////////////////////////////////////////////////////////////
		//Identify "inter" design positions: (non-epitope) scaff positions near Ab...these will shrink to AGST
		/////////////////////////////////////////////////////////////////////////////////////////////////

		for ( int j=1; j<=nres_Ab; ++j ) {
			int const jres_Ab = j;           //Ab is first in the cmplx
			int start_atm2, end_atm2;
			int aa2 = pose.res( jres_Ab );
			int aav2 = pose.res_variant( jres_Ab );
			if ( aa2 == param_aa::aa_gly ) {
				start_atm2 = 2;
				end_atm2 = 2;
			} else {
				start_atm2 = 5;
				end_atm2 = aaproperties_pack::nheavyatoms( aa2, aav2 );
			}

			float dis;
			for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
				for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
					dis = distance( numeric::xyzVector_float( &( pose.full_coord()(1,atm1,ires_scaff))),
													numeric::xyzVector_float( &( pose.full_coord()(1,atm2,jres_Ab))));
					if ( dis < cutoff_inter ) {
						is_inter_despos( ires_scaff ) = true;
						is_inter_seqpos_Ab( jres_Ab ) = true;
						if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
							std::cout << "prepare_des_matrix: inter_despos on Ab, scaff " << jres_Ab << " --- "
												<< ires_scaff << std::endl;
						}
					}
				} // atm2 -- in Ab
			} // atm1 -- in scaff
		}//j -- residues in Ab

		/////////////////////////////////////////////////////////////////////////////////////////////////
		//Identify "intra" design positions: scaff positions near epitope (below we exclude "inter" positions)
		/////////////////////////////////////////////////////////////////////////////////////////////////

		//
		//Includes scaff positions near the graft or near flanking positions that were allowed to move during loop modeling
		//

		for ( int j = begin_move_bb; j <= end_move_bb; j++ ) {
			int jres_epitope = j + nres_Ab;

			std::cout << "j, jres_epitope " << j << " " << jres_epitope << std::endl;

			int start_atm2, end_atm2;
			int aa2 = pose.res( jres_epitope );
			int aav2 = pose.res_variant( jres_epitope );
			if ( aa2 == param_aa::aa_gly ) {
				start_atm2 = 2;
				end_atm2 = 2;
			} else {
				start_atm2 = 5;
				end_atm2 = aaproperties_pack::nheavyatoms( aa2, aav2 );
			}

			float dis;
			for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
				for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
					dis = distance( numeric::xyzVector_float( &( pose.full_coord()(1,atm1,ires_scaff))),
													numeric::xyzVector_float( &( pose.full_coord()(1,atm2,jres_epitope))));
					//	std::string name1,name2;
					//	atom_name_from_atom_num( atm1, aa1, aav1, name1 );
					//atom_name_from_atom_num( atm2, aa2, aav2, name2 );
					//std::cout << "intra dis: ires_scaff,atm1 " << ires_scaff << " " << name1 << " jres_epitope,atm2 "
					//					<< jres_epitope << " " << name2 << " " << dis << " " << disbk << std::endl;
					if ( dis < cutoff_intra ) {
						is_intra_despos( ires_scaff ) = true;
						if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
							std::cout << "prepare_des_matrix: intra_despos on scaff " << ires_scaff << std::endl;
						}
					}
				} // atm2 -- in epitope
			} // atm1 -- in scaff
		}//j -- residues in epitope
	}//i -- residues in scaff

	///////////////////////////////////////////////////////////////////////////////////////////////////
	//make short lists of residue numbers from bools identified in loops above
	///////////////////////////////////////////////////////////////////////////////////////////////////

	//make lists for intra, inter despos
	//
	FArray1D_int list_inter_despos( 0 );
	int n_inter_despos = 0;
	FArray1D_int list_intra_despos( 0 );
	int n_intra_despos = 0;
	for ( int i = 1; i <= nres_scaff; i++ ) {

		//ires_scaff is resnum in Ab-scaff cmplx...Ab is first, scaff second
		int const ires_scaff = i+nres_Ab;

		if ( is_inter_despos ( ires_scaff ) ) {
			n_inter_despos++;
			list_inter_despos.redimension( n_inter_despos );
			list_inter_despos( n_inter_despos ) = ires_scaff;
		}

		if ( is_intra_despos ( ires_scaff ) ) {
			n_intra_despos++;
			list_intra_despos.redimension( n_intra_despos );
			list_intra_despos( n_intra_despos ) = ires_scaff;
		}
	}

	//make list for Ab residues at interface
	//
	FArray1D_int list_inter_seqpos_Ab( 0 );
	int n_inter_seqpos_Ab = 0;
	for ( int i = 1; i <= nres_Ab; i++ ) {
		if ( is_inter_seqpos_Ab( i ) ) {
			n_inter_seqpos_Ab++;
			list_inter_seqpos_Ab.redimension( n_inter_seqpos_Ab );
			list_inter_seqpos_Ab( n_inter_seqpos_Ab ) = i;
		}
	}

	///////////////////////////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////////////////////////////////////
	//identify Ab positions at interface that can repack
	///////////////////////////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////////////////////////////////////

	//if allow_repack_Ab, here loop over Ab residues at interface...
	//if allow_repack_epitope is also true, then all Ab interface residues can repack
	//if allow_repack_epitope is NOT true, then Ab interface residues can repack
	//only if the Ab position is not too close to epitope
	//
	FArray1D_bool is_Ab_repack_seqpos( nres_Ab );
	FArray1D_int list_Ab_repack_seqpos( 0 );
	int n_Ab_repack_seqpos = 0;

	if ( allow_repack_Ab ) {
		for ( int i = 1; i <= n_inter_seqpos_Ab; i++ ) {
			int ires_Ab = list_inter_seqpos_Ab( i );

			int start_atm1, end_atm1;
			int const aa1 = pose.res( ires_Ab );
			int const aav1 = pose.res_variant( ires_Ab );
			if ( aa1 == param_aa::aa_gly ) {
				start_atm1 = 2;
				end_atm1 = 2;
			} else {
				start_atm1 = 5;
				end_atm1 = aaproperties_pack::nheavyatoms( aa1, aav1 );
			}

			for ( int j = 1; j <= n_epitope_natro_start; j++ ) {
				int jres_epitope = list_epitope_natro_start_in_scaff( j ) + nres_Ab;
				int start_atm2, end_atm2;
				int aa2 = pose.res( jres_epitope );
				int aav2 = pose.res_variant( jres_epitope );
				if ( aa2 == param_aa::aa_gly ) {
					start_atm2 = 2;
					end_atm2 = 2;
				} else {
					start_atm2 = 5;
					end_atm2 = aaproperties_pack::nheavyatoms( aa2, aav2 );
				}

				float dis;
				for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
					for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
						//distance_bk( pose.full_coord()(1,atm1,i), pose.full_coord()(1,atm2,j), dis );
						dis = distance( numeric::xyzVector_float( &( pose.full_coord()(1,atm1,ires_Ab))),
														numeric::xyzVector_float( &( pose.full_coord()(1,atm2,jres_epitope))));
						if ( allow_repack_epitope || dis > cutoff_Ab_epitope ) {
							is_Ab_repack_seqpos( ires_Ab ) = true;
						}
					} // atm2 -- in Ab
				} // atm1 -- in scaff
			}// j epitope_natro_start
		} //i inter_seqpos_Ab

		//compile list for Ab repack positions
		//
		for ( int i = 1; i <= nres_Ab; i++ ) {
			if ( is_Ab_repack_seqpos( i ) ) {
				n_Ab_repack_seqpos++;
				list_Ab_repack_seqpos.redimension( n_Ab_repack_seqpos );
				list_Ab_repack_seqpos( n_Ab_repack_seqpos ) = i;
			}
		}
	} // if allow_repack_Ab


	///////////////////////////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////////////////////////////////////
	//Now setup the design_matrix
	///////////////////////////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////////////////////////////////////

	//
	//Identify disulfides to ensure that design does not kill a disulfide
	//
	disulfides::options::find_disulf = true;
	pose.score( weight_map );
	std::cout << "n_disulf = " << disulfides::BOUNDARY::get_n_disulf_centroid() << std::endl;


	//which aa are allowed for design at "inter" design positions?
	//
	int const n_aa_allowed_at_inter_despos = 4;
	FArray1D_int aa_allowed_at_inter_despos( n_aa_allowed_at_inter_despos );
	{
		int i = 0;
		aa_allowed_at_inter_despos( ++i ) = param_aa::aa_ala;
		aa_allowed_at_inter_despos( ++i ) = param_aa::aa_gly;
		aa_allowed_at_inter_despos( ++i ) = param_aa::aa_ser;
		aa_allowed_at_inter_despos( ++i ) = param_aa::aa_thr;
	}

	//inter despos
	//
	for ( int i = 1; i <= n_inter_despos; i++ ) {
		int ires_scaff = list_inter_despos( i );
		assert( ires_scaff > nres_Ab );
		std::cout << "inter_despos: " << i << " " << ires_scaff << std::endl;
		for ( int i_aa = 1; i_aa <= n_aa_allowed_at_inter_despos; i_aa++ ) {
			int const aa = aa_allowed_at_inter_despos( i_aa );
			design_matrix( aa, ires_scaff ) = true;
			if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
				std::cout << "prepare_design_matrix...matrix should be true for inter at ires, aa " << ires_scaff
									<< " " << aa << " " << param_aa::param_aa_data::aa_name3(aa) << std::endl;
			}
		}
	}

	//intra despos
	//
	//default is to exclude "inter" despos so that inter despos are required to shrink
	//
	//Allow ALLAA (not cysteine) at intra design positions
	//except if this position is disulf-bonded cys, keep the cys natro.
	//
	for ( int i = 1; i <= n_intra_despos; i++ ) {
		int ires_scaff = list_intra_despos( i );
		assert( ires_scaff > nres_Ab );
		if ( is_inter_despos( ires_scaff ) ) continue; //<---exclude "inter"
		if ( disulfides::BOUNDARY::cys_res_in_disulf( ires_scaff ) ) {
			std::cout << "skipping intra_despos cys in disulf " << ires_scaff << std::endl;
			continue;
		}
		std::cout << "intra_despos: " << i << " " << ires_scaff << std::endl;
		for ( int aa = 1; aa < param_aa::aa_cys; aa++ ) {
			design_matrix( aa, ires_scaff ) = true;
			if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
				std::cout << "prepare_design_matrix...matrix should be true for intra at ires, aa " << ires_scaff
									<< " " << aa << " " << param_aa::param_aa_data::aa_name3(aa) << std::endl;
			}
		}
		for ( int aa = param_aa::aa_cys+1; aa <= 20; aa++ ) {
			design_matrix( aa, ires_scaff ) = true;
			if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
				std::cout << "prepare_design_matrix...matrix should be true for intra at ires, aa " << ires_scaff
									<< " " << aa << " " << param_aa::param_aa_data::aa_name3(aa) << std::endl;
			}
		}
	}

	//Ab repack positions
	//
	if ( allow_repack_Ab ) {
		for ( int i = 1; i <= n_Ab_repack_seqpos; i++ ) {
			int ires_Ab = list_Ab_repack_seqpos( i );
			int const aa = pose.res( ires_Ab );
			design_matrix( aa, ires_Ab ) = true;
		}
	}

	//epitope repack positions
	//
	if ( allow_repack_epitope ) {
		for ( int i = 1; i <= n_epitope_natro_start; i++ ) {
			int const ires = list_epitope_natro_start_in_scaff( i ) + nres_Ab;
			int const aa = pose.res( ires );
			design_matrix( aa, ires ) = true;
		}
	}
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
construct_pose_complex_from_p1_p2(
																	pose_ns::Pose const & pose1,
																	pose_ns::Pose const & pose2,
																	pose_ns::Pose & pose3
																	)
{
	int const nres1 = pose1.total_residue();
	int const nres2 = pose2.total_residue();
	int const nres3 = nres1 + nres2;

	pose3.simple_fold_tree( nres3 );
	pose3.set_fullatom_flag( true, false );// set fullatom and do not repack

	FArray3D_float fcoord3( 3, param::MAX_ATOM(), nres3 );

	int ires3 = 0;
	for ( int ires = 1; ires <= nres1; ires++ ) {
		++ires3;
		pose3.set_res        ( ires3, pose1.res( ires ) );
    pose3.set_res_variant( ires3, pose1.res_variant( ires ));
    pose3.set_phi        ( ires3, pose1.phi( ires ) );
    pose3.set_psi        ( ires3, pose1.psi( ires ) );
    pose3.set_omega      ( ires3, pose1.omega( ires ) );
    pose3.set_secstruct  ( ires3, pose1.secstruct( ires ) );
    pose3.set_name       ( ires3, pose1.name( ires ) );
		int const aa( pose1.res( ires ) );
		int const aav( pose1.res_variant( ires ));
		for ( int i_atom = 1; i_atom <= aaproperties_pack::natoms( aa, aav ); ++i_atom ) {
			for ( int k = 1; k <= 3; k++ ) {
				fcoord3( k, i_atom, ires3 ) = pose1.full_coord()( k, i_atom, ires );
			}
		}

		// chain id
		pose3.pdb_info().set_pdb_chain( ires3, pose1.pdb_info().res_chain( ires ) );

		// pdb residue numbering
		pose3.pdb_info().set_pdb_res( ires3, ires );

		// insertion code (none)
		pose3.pdb_info().set_pdb_insert_let( ires3, ' ' );
	}

	for ( int ires = 1; ires <= nres2; ires++ ) {
		++ires3;
		pose3.set_res        ( ires3, pose2.res( ires ) );
    pose3.set_res_variant( ires3, pose2.res_variant( ires ));
    pose3.set_phi        ( ires3, pose2.phi( ires ) );
    pose3.set_psi        ( ires3, pose2.psi( ires ) );
    pose3.set_omega      ( ires3, pose2.omega( ires ) );
    pose3.set_secstruct  ( ires3, pose2.secstruct( ires ) );
    pose3.set_name       ( ires3, pose2.name( ires ) );
		int const aa( pose2.res( ires ) );
		int const aav( pose2.res_variant( ires ));
		for ( int i_atom = 1; i_atom <= aaproperties_pack::natoms( aa, aav ); ++i_atom ) {
			for ( int k = 1; k <= 3; k++ ) {
				fcoord3( k, i_atom, ires3 ) = pose2.full_coord()( k, i_atom, ires );
				}
			}

		// chain id
		pose3.pdb_info().set_pdb_chain( ires3, pose2.pdb_info().res_chain( ires ) );

		// pdb residue numbering
		pose3.pdb_info().set_pdb_res( ires3, ires );

		// insertion code (none)
		pose3.pdb_info().set_pdb_insert_let( ires3, ' ' );
		}

	//Error check
	//
	if ( ! ( ires3 == nres3 ) ) {
		std::cout << "error setting up cmplx pose from p1 p2" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	//
	//Setup Epos for the new pose
	//
	FArray3D_float Epos3( 3, param::MAX_POS, nres3 );//MAX_POS = 5

	for ( int ires = 1; ires <= nres3; ires++ ) {
		for ( int k = 1; k <= 3; k++ ) {
			Epos3(k,1,ires) = fcoord3(k,1,ires);
			Epos3(k,2,ires) = fcoord3(k,2,ires);
			Epos3(k,4,ires) = fcoord3(k,3,ires);
			Epos3(k,5,ires) = fcoord3(k,4,ires);
			Epos3(k,3,ires) = fcoord3(k,5,ires);
			}
		}
	//finish by setting coords
	//
	bool const ideal_pose ( false );
	bool const check_missing ( false );
	pose3.set_coords( ideal_pose, Epos3, fcoord3, check_missing );
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
//transfer rotamers from pose1 to pose2 according to boolean arrays
//natro1 maps to natro2
void
transfer_native_rotamers(
												 pose_ns::Pose const & pose1,
												 FArray1D_bool const & natro1, //bool at each position
												 FArray1D_bool const & natro2, //bool at each position
												 pose_ns::Pose & pose2
												 )
{

	using namespace pose_ns;

	int const nres1 = pose1.total_residue();
	int const nres2 = pose2.total_residue();

 	std::cout << "in transfer_native_rotamers...nres1, nres2 = " << nres1 << " " << nres2 << std::endl;

	//first make sure the bool arrays have the same number of true elements
	//these are the positions to transfer from 1 -> 2.
	//
	int n_natro1 = 0;
	FArray1D_int list_natro1( 0 );
	for ( int ires1 = 1; ires1 <= nres1; ires1++ ) {
		if ( natro1( ires1 ) ) {
			++n_natro1;
			list_natro1.redimension( n_natro1 );
			list_natro1( n_natro1 ) = ires1;
		}
		//		std::cout << "ires1, natro1, n_natro1 " << ires1 << " " << natro1(ires1)  << " " << n_natro1 << std::endl;
	}
	int n_natro2 = 0;
	FArray1D_int list_natro2( 0 );
	for ( int ires2 = 1; ires2 <= nres2; ires2++ ) {
		if ( natro2( ires2 ) ) {
			++n_natro2;
			list_natro2.redimension( n_natro2 );
			list_natro2( n_natro2 ) = ires2;
		}
		//		std::cout << "ires2, natro2, n_natro2 " << ires2 << " " << natro2(ires2) << " " << n_natro2 << std::endl;
	}

	if ( ! ( n_natro1 == n_natro2 ) ) {
		std::cout << "error in transfer_native_rotamers, n_natro not same for pose1 and pose2 " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::cout << "transfer_native_rotamers...will transfer " << n_natro1 << " rotamers" << std::endl;

	//Now use design to put the correct aa from pose1 at the correct position in pose2
	// this is not yet transferring rotamers
	//
	FArray2D_bool design_matrix_local( param::MAX_AA(), nres2, false );
	for ( int i_natro = 1; i_natro <= n_natro1; i_natro++ ) {
		int const aa = pose1.res( list_natro1( i_natro ) );
		int const ires2 = list_natro2( i_natro );
		design_matrix_local( aa, ires2 ) = true;
	}

	design_using_design_matrix( pose2, design_matrix_local );

	std::cout << "transfer_native_rotamers...done with design" << std::endl;

	if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
		for ( int i=1; i<= pose2.total_residue(); i++ ) {
			std::cout << "pose2 new aa: " << I(4,i) << " " << pose2.res( i ) << " "
								<< SS( param_aa::param_aa_data::aa_name3( pose2.res( i ) ) ) << std::endl;
		}
	}
	//debug
	//	pose2.dump_pdb("pose2_with_new_aa_" + string_of( n_natro1 ) + ".pdb" );

	//Now we have the correct aa in place, we can transfer rotamer coords
	//
	//copy of pose1 we can change
	Pose pose1_tmp;
	pose1_tmp = pose1;
	//copy of pose2 coords we can change
	//	FArray3D_float fcoord_pose2( 3, param::MAX_ATOM(), nres2 );
	FArray3D_float fcoord_pose2( pose2.full_coord() );

	//stuff used for orienting pose1
	//
	FArray2D_float coords_3_atoms( 3, 3 );
	FArray2D_float new_coords_3_atoms( 3, 3 );
	int const atom_num_N = 1;
	int const atom_num_CA = 2;
	int const atom_num_CB = 3;
	int const atom_num_C = 4;
	int const atom_num_O = 5;

	for ( int i_natro = 1; i_natro <= n_natro1; i_natro++ ) {

		int const ires1 = list_natro1( i_natro );
		int const ires2 = list_natro2( i_natro );
		int const aa1 = pose1.res( ires1 );
		int const aa2 = pose2.res( ires2 );
		int const aav1 = pose1.res_variant( ires1 );
		int const aav2 = pose2.res_variant( ires2 );

		std::cout << "transferring...i_natro, ires1, ires2 " << i_natro << " " << ires1 << " ---> " << ires2 << std::endl;

		if ( ! ( aa1 == aa2 ) ) {
			std::cout << "Error in transfer rotamer, aa1 != aa2" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if ( ! ( aav1 == aav2 ) ) {
			std::cout << "Error in transfer rotamer, aav1 != aav2" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		//		if ( pose1.res( ires1 ) == param_aa::aa_gly ) continue; //pose2 already gly not need to xfer gly.
		for ( int k = 1; k <= 3 ; k++ ) {
			//
			//align using backbone atoms and not CB b/c CB from design will be in ideal position
			//// ( also if want GLY why wouldn't CB cause problem ? )
			//
			coords_3_atoms( k, 1 ) = pose1.full_coord()( k, atom_num_N, ires1 );  //bills was CA
			coords_3_atoms( k, 2 ) = pose1.full_coord()( k, atom_num_CA, ires1 ); //bills was CB
			coords_3_atoms( k, 3 ) = pose1.full_coord()( k, atom_num_C, ires1 );  //bills was C
			new_coords_3_atoms( k, 1 ) = pose2.full_coord()( k, atom_num_N, ires2 );  //bills was CA
			new_coords_3_atoms( k, 2 ) = pose2.full_coord()( k, atom_num_CA, ires2 ); //bills was CB
			new_coords_3_atoms( k, 3 ) = pose2.full_coord()( k, atom_num_C, ires2 );  //bills was C
		}
		//orient pose1_tmp so that pose1 ires1 is aligned with pose2 ires2
		// ( I should change this to orient_coords and just pass fcoord_pose1 ) !!!
		orient_pose( coords_3_atoms, new_coords_3_atoms, pose1_tmp );
		//copy coords from pose1 to pose2 for this rotamer
		//
		for ( int iatom = aaproperties_pack::first_scatom( aa1, aav1 ); iatom <= aaproperties_pack::natoms( aa1, aav1 ); iatom++ ) {
			for ( int k = 1; k <= 3; k++ ) {
				fcoord_pose2( k, iatom, ires2 ) = pose1_tmp.full_coord()( k, iatom, ires1 );
			}
		}
		//reset coords in pose1_tmp;
		pose1_tmp = pose1;
	}//i_natro

	std::cout << "done rot xfer now get Epos2 " << std::endl;

	//
	//at this point all intended rotamers have been
	//transferred from pose1 to the array fcoord_pose2
	//

	//compute new Eposition array for pose2 with new rotamers
	//

	FArray3D_float Epos2( 3, param::MAX_POS, nres2 );//MAX_POS = 5
	for ( int i = 1; i <= nres2; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			Epos2(k,1,i) = fcoord_pose2(k,1,i);
			Epos2(k,2,i) = fcoord_pose2(k,2,i);
			Epos2(k,4,i) = fcoord_pose2(k,3,i);
			Epos2(k,5,i) = fcoord_pose2(k,4,i);
			Epos2(k,3,i) = fcoord_pose2(k,5,i);
		}
	}

	//Put new coords in the pose2
	//
	bool const ideal_pose = false;
	bool const check_missing = false;
	pose2.set_coords( ideal_pose, Epos2, fcoord_pose2, check_missing );

	//	pose2.dump_pdb("pose2_with_" + string_of( n_natro1 ) + "_native_rotamers.pdb" );
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
orient_pose(
						FArray2DB_float const & coords_3_atoms,
						FArray2Da_float const & new_coords_3_atoms,
						pose_ns::Pose & pose
						)
{

	//////////////////////////////////////////////////////////////////////////////
	//get GL_matrix that transforms coords -> new_coords
	//

	FArray2D_float Mgl( 4, 4 );

	get_GL_matrix( coords_3_atoms( 1, 1 ),
								 coords_3_atoms( 1, 2 ),
								 coords_3_atoms( 1, 3 ),
								 new_coords_3_atoms( 1, 1 ),
								 new_coords_3_atoms( 1, 2 ),
								 new_coords_3_atoms( 1, 3 ),
								 Mgl );

	//local copy of pose coordinates
	//we will re-orient these coords in space
	//then put the new coords back into the pose.
	//
	FArray3D_float fcoord ( pose.full_coord() );

	// apply the transformation to the fcoord
	//

	int const nres = pose.total_residue();
	for ( int seqpos = 1; seqpos <= nres; seqpos++ ) {
		int const aa( pose.res( seqpos ) );
		int const aav( pose.res_variant( seqpos ));
		for ( int i_atom = 1; i_atom <= aaproperties_pack::natoms( aa, aav ); ++i_atom ) {
			GL_rot_in_place( Mgl, fcoord( 1, i_atom, seqpos ) );
		}
	}


	//make an Eposition array that is consistent with
	//transformed fcoord array.
	//

	FArray3D_float Epos( 3, param::MAX_POS, nres );//MAX_POS = 5

	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			Epos(k,1,i) = fcoord(k,1,i);
			Epos(k,2,i) = fcoord(k,2,i);
			Epos(k,4,i) = fcoord(k,3,i);
			Epos(k,5,i) = fcoord(k,4,i);
			Epos(k,3,i) = fcoord(k,5,i);
		}
	}

	//Put oriented coords in the pose
	//

	bool const ideal_pose = false;
	bool const check_missing = false;
	pose.set_coords( ideal_pose, Epos, fcoord, check_missing );

}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
double
superimpose_using_ncaco(
														 pose_ns::Pose & pose1,
														 int const & pose1_first,
														 int const & pose1_last,
														 pose_ns::Pose & pose2, //pose2 will be changed, superimposed onto pose1
														 int const & pose2_first,
														 int const & pose2_last
														 )
{

	//superimpose pose2 onto pose1 over the range using backbone atoms N,CA,C,0

	int const nres1 = pose1.total_residue();
	int const nres2 = pose2.total_residue();

	if ( nres1 == 0 || nres2 == 0 ) {
		std::cout << "ERROR superimpose_pose: nres1 or nres2 == 0 " << nres1 << " " << nres2 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( pose1_first < 0 || pose1_first > nres1 || pose1_last < 0 || pose1_last > nres1 || pose1_last <= pose1_first ) {
		std::cout << "ERROR superimpose_pose: problem with pose1_first, pose1_last, nres1: "
							<< pose1_first << ' ' << pose1_last << ' ' << nres1 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( pose2_first < 0 || pose2_first > nres2 || pose2_last < 0 || pose2_last > nres2 || pose2_last <= pose2_first ) {
		std::cout << "ERROR superimpose_pose: problem with pose2_first, pose2_last, nres2: "
							<< pose2_first << ' ' << pose2_last << ' ' << nres2 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	int const nres1_sup = pose1_last - pose1_first + 1;
	int const nres2_sup = pose2_last - pose2_first + 1;
	if ( nres1_sup != nres2_sup ) {
		std::cout << "ERROR superimpose_pose: nres1_sup != nres2_sup: " << nres1_sup << ' ' << nres2_sup << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}


	//local copy of pose coordinates
	//we will re-orient these coords in space
	//then put the new coords back into the pose.
	//
	FArray3D_float fcoord1 ( pose1.full_coord() );
	FArray3D_float fcoord2 ( pose2.full_coord() );

	//count natoms total for each pose.
	//these are the total number of atoms that will be moved.
	//so these can be larger than the natoms_sup that will be used to compute the superposition
	//natoms1 and natoms2 do not have to be equal.
	//
	int natoms1 = 0;
	for ( int ires = 1; ires <= nres1; ires++ ) {
    int const aa( pose1.res( ires ) );
    int const aav( pose1.res_variant( ires ));
    for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
			++natoms1;
		}
	}

	int natoms2 = 0;
	for ( int ires = 1; ires <= nres2; ires++ ) {
    int const aa( pose2.res( ires ) );
    int const aav( pose2.res_variant( ires ));
    for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
			++natoms2;
		}
	}

	//setup one-dimensional coordinate arrays
	//
	FArray2D_double x1( 3, natoms1 );
	FArray2D_double x2( 3, natoms2 );
	FArray2D_double x1_oriented( 3, natoms1 );
	FArray2D_double x2_oriented( 3, natoms2 );//should be same as x2

	//count natoms total for each pose.
	//these are the total number of atoms that will be moved.
	//so these can be larger than the natoms_sup that will be used to compute the superposition
	//natoms1 and natoms2 do not have to be equal.
	//
	int atom_count = 0;
	for ( int ires = 1; ires <= nres1; ires++ ) {
    int const aa( pose1.res( ires ) );
    int const aav( pose1.res_variant( ires ));
    for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
			++atom_count;
			for ( int k = 1; k <= 3; k++ ) {
				x1( k, atom_count ) = pose1.full_coord()( k, iatom, ires );
				x1_oriented( k, atom_count ) = x1( k, atom_count );
			}
		}
	}

	atom_count = 0;
	for ( int ires = 1; ires <= nres2; ires++ ) {
    int const aa( pose2.res( ires ) );
    int const aav( pose2.res_variant( ires ));
    for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
			++atom_count;
			for ( int k = 1; k <= 3; k++ ) {
				x2( k, atom_count ) = pose2.full_coord()( k, iatom, ires );
				x2_oriented( k, atom_count ) = x2( k, atom_count );
			}
		}
	}


	int const n_bb_atoms_to_superimpose_each_rsd = 4;

	//nres to superimpose via N,CA,C,O
	//
	int const nres_sup = nres1_sup;

	//natoms to superimpose
	//
	int const natoms_sup = nres_sup * n_bb_atoms_to_superimpose_each_rsd; //only superimposing N,CA,C,O

	//if passing in fullatom coords to get fullcoord oriented, then we need to pass in the correct index
	//to each pose.
	//the indices can be different if the sequences are different
	//we want to loop over the pairs of aligned residues

	FArray1D_int list_atoms_to_superimpose_1( natoms_sup, 0 );
	FArray1D_int list_atoms_to_superimpose_2( natoms_sup, 0 );

	{
		int natoms_tot = 0;
		int natoms_sup1 = 0;
		for ( int ires = 1; ires <= nres1; ires++ ) {
			int const aa( pose1.res( ires ) );
			int const aav( pose1.res_variant( ires ));
			for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
				++natoms_tot;
				if ( ires >= pose1_first && ires <= pose1_last && iatom <= n_bb_atoms_to_superimpose_each_rsd ) {
					list_atoms_to_superimpose_1( ++natoms_sup1 ) = natoms_tot;
					/*
					{//debug
					std::string name;
					atom_name_from_atom_num( iatom, aa, aav, name );
					std::cout << "atoms1_sup: i, ires, atom_name " << natoms_sup1 << ' ' << ires << ' ' << SS(name) << std::endl;
					}
					*/
				}
			}
		}
	}
	{
		int natoms_tot = 0;
		int natoms_sup2 = 0;
		for ( int ires = 1; ires <= nres2; ires++ ) {
			int const aa( pose2.res( ires ) );
			int const aav( pose2.res_variant( ires ));
			for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
				++natoms_tot;
				if ( ires >= pose2_first && ires <= pose2_last && iatom <= n_bb_atoms_to_superimpose_each_rsd ) {
					list_atoms_to_superimpose_2( ++natoms_sup2 ) = natoms_tot;
					/*
					{//debug
					std::string name;
					atom_name_from_atom_num( iatom, aa, aav, name );
					std::cout << "atoms2_sup: i, ires, atom_name " << natoms_sup2 << ' ' << ires << ' ' << SS(name) << std::endl;
					}
					*/
				}
			}
		}
	}

	//addressing charlie's globals
	//
	clear_rms();

	//set globals for this window
	//add_rms_asym for non-symmetric positions in pep and prot


	for ( int i = 1; i <= natoms_sup; i++ ) {
		int index1 = list_atoms_to_superimpose_1( i );
		int index2 = list_atoms_to_superimpose_2( i );
		add_rms_asym( index1, index2, x1, x2 );
	}


	/*
	//ipro is position in protein within the peptide window (resnum in protein)
	//ipep is position in peptide within the protein window (resnum in peptide)
	for ( int ipep = begin_sup_pep; ipep <= end_sup_pep; ++ipep ) {
		int ipro = begin_sup_pro + ipep - begin_sup_pep;
		//push rms for all 4 bb atoms at ipep,ipro
		for ( int index = 1; index <= 4; ++index ) {
			int pos_pep = (ipep-1)*4+index;  //pos'n in the 1D array of N,CA,C,O coords for all peptide residues
			int pos_pro = (ipro-1)*4+index;  //pos'n in the 1D array of N,CA,C,O coords for all protein residues
			add_rms_asym(pos_pro,pos_pep,xpro0,xpep0);
		}//index
	} //ipep

	// find alignment between peptide and protein window
	// rmsfitca3 rotates all of the residues but the rotation only aligns the
	// residues pushed into add_rms.

	double rms = 0.0;
	rmsfitca3_asym(natm_xpro,natm_xpep,xpro0,xpro,xpep0,xpep,rms);

	*/

	// find alignment between peptide and protein window
	// rmsfitca3 rotates all of the residues but the rotation only aligns the
	// residues pushed into add_rms.
	//with retain_com = true the x2_oriented will be superimposed on x1
	//otherwise it will come out with c.o.m at the origin.

	double rms = 0.0;
	bool retain_com = true;
	rmsfitca3_asym( natoms1, natoms2, x1, x1_oriented, x2, x2_oriented, rms, retain_com );


	std::cout << "x2-x1 rms = " << rms << std::endl;

	//Now translate x1,x2 back into FArrays....
	//

	FArray3D_float fcoord1_oriented( pose1.full_coord() );
	FArray3D_float fcoord2_oriented( pose2.full_coord() );
	fcoord1_oriented = 0.0;
	fcoord2_oriented = 0.0;

	//fcoord1_out should be same as fcoord1
	//
	{
		int atom_count = 0;
		for ( int ires = 1; ires <= nres1; ires++ ) {
			int const aa( pose1.res( ires ) );
			int const aav( pose1.res_variant( ires ));
			for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
				++atom_count;
				for ( int k = 1; k <= 3; k++ ) {
					fcoord1_oriented( k, iatom, ires ) = x1_oriented( k, atom_count );
				}
			}
		}
	}

	//fcoord2_out are the re-oriented coordinates for pose2
	//
	{
		int atom_count = 0;
		for ( int ires = 1; ires <= nres2; ires++ ) {
			int const aa( pose2.res( ires ) );
			int const aav( pose2.res_variant( ires ));
			for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
				++atom_count;
				for ( int k = 1; k <= 3; k++ ) {
					fcoord2_oriented( k, iatom, ires ) = x2_oriented( k, atom_count );
				}
			}
		}
	}

	//make an Eposition array that is consistent with
	//transformed fcoord array.
	//

	FArray3D_float Epos( 3, param::MAX_POS, nres2 );//MAX_POS = 5

	for ( int i = 1; i <= nres2; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			Epos(k,1,i) = fcoord2_oriented(k,1,i);
			Epos(k,2,i) = fcoord2_oriented(k,2,i);
			Epos(k,4,i) = fcoord2_oriented(k,3,i);
			Epos(k,5,i) = fcoord2_oriented(k,4,i);
			Epos(k,3,i) = fcoord2_oriented(k,5,i);
		}
	}

	//Put oriented coords in the pose
	//

	bool const ideal_pose = false;
	bool const check_missing = false;
	pose2.set_coords( ideal_pose, Epos, fcoord2_oriented, check_missing );

	//	pose1.dump_pdb("pose1.pdb");
	//	pose2.dump_pdb("pose2_oriented.pdb");

	return rms;
}


void
design_using_design_matrix(
													 pose_ns::Pose & pose_in,
													 FArray2D_bool const & design_matrix_local
													 )
{
	//
	//fill misc arrays from this pose
	//
	//  Note: must do this (b/c sets total_residue) before setting design_matrix
	//
	pose_to_misc( pose_in );

	//
	//Set design globals for design_matrix to control which aa allowed at which positions.
	//

	design::use_design_matrix = true;
	design::design_matrix = false; //design_matrix( MAX_AA(), MAX_RES() );
	design::design_commands::try_both_his_tautomers = true;
  //trigger jk's additional call to optimizeH to build "similar" rotamers ( see RotamerSet.cc )
  //this not set in options.cc b/c input_fa not set in pose1 mode.
  //this adds just a handful of additional rotamers in each nglyco site design.
	design::active_rotamer_options.use_input_sc = true;
  if ( truefalseoption( "large_rotamer_set" ) ) select_rotamer_set( "large" );
  if ( truefalseoption( "favor_native_residue" ) ) {
    set_favor_native_residue( true );
    float native_bonus_tmp;
    realafteroption("favor_native_residue",-1.5, native_bonus_tmp);
    set_native_bonus( native_bonus_tmp );
  }

	int const nres = pose_in.total_residue();

	for ( int ires = 1 ; ires <= nres ; ++ires ) {
		for ( int aa = 1; aa <= param::MAX_AA(); aa++ ) {
			design::design_matrix( aa, ires ) = design_matrix_local( aa, ires );
		}
	}

	//
	//variables to set for the call to pack_rotamers
	//
	std::string pack_mode( "design" );
	bool make_output_file( false );
	FArray1D_bool allow_repack_local( param::MAX_RES()(), false );
	bool include_current( true );
	bool include_extra ( false );
	FArray2D_int extra_rot( param::MAX_CHI, param::MAX_RES()() ); // dummy variable in this instance
	FArray2D_float extra_chi( param::MAX_CHI, param::MAX_RES()() ); // dummy variable in this instance

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task(pose_in);
	Task.set_task( pack_mode, make_output_file, allow_repack_local,
								 include_current, include_extra, extra_rot, extra_chi);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();
	//
	//design using misc arrays
	//
	pack_rotamers( pose_in, Task );

	//
	//retrieve design from misc.
	//
	// yab: merge into mainline; below isn't necessary anymore
//	bool const fullatom( true );
//	bool const ideal_pose( false );
//	bool const coords_init( true );
//	pose_from_misc( pose_out, fullatom, ideal_pose, coords_init );
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

void
close_graft(
						int const & n_loops,
						FArray1DB_int const & loop_begin,
						FArray1DB_int const & loop_end,
						FArray1DB_int const & cut_point,
						int const & nres_disallow_bbmove,
						FArray1DB_int const & list_disallow_bbmove,
						pose_ns::Pose & pose,
						float & chainbreak_score
						)
{
	//////////////////////////////////////////////////////////////////////
	//close the graft using pose_loops protocol
	//Note: to close a graft there will be either one loop or two loops
	//////////////////////////////////////////////////////////////////////

	// first save pdb information, the pose_loops protocol ignores this information -- upon recall (e.g. pose_from_misc after closure) the chain id etc. in the region of the loop is lost
	FArray1D_char pdb_chain( pose.pdb_info().res_chain() );
	FArray1D_int  pdb_residue_num( pose.pdb_info().pdb_res_num() );
	FArray1D_char pdb_insertion_code( pose.pdb_info().pdb_insert_let() );

	//
	//put loop information into pose_loops_input_ns
	//
	pose_loops_input_ns::start_from_extend = false;
	pose_loops_input_ns::n_loops = n_loops;
	pose_loops_input_ns::loop_begin.redimension( n_loops );
	pose_loops_input_ns::loop_end.redimension( n_loops );
	pose_loops_input_ns::cut_point.redimension( n_loops );
	pose_loops_input_ns::chainbreak_score = 1000.;


	for( int i=1; i<=n_loops; ++i ) {
		pose_loops_input_ns::loop_begin(i) = loop_begin(i);
		pose_loops_input_ns::loop_end(i) = loop_end(i);
		pose_loops_input_ns::cut_point(i) = cut_point(i);
		if (runlevel_ns::runlevel > runlevel_ns::quiet) {
			std::cout << "put loop info into pose_loops_input_ns:: "
								<< pose_loops_input_ns::loop_begin(i) << " "
								<< pose_loops_input_ns::loop_end(i) << " "
								<< pose_loops_input_ns::cut_point(i) << " "
								<< std::endl;
		}
	}
	pose_loops_input_ns::nres_disallow_bbmove = nres_disallow_bbmove;
	pose_loops_input_ns::list_disallow_bbmove.redimension( nres_disallow_bbmove );
	if ( nres_disallow_bbmove > 0 ) {
		for ( int i = 1; i <= nres_disallow_bbmove; i++ ) {
			pose_loops_input_ns::list_disallow_bbmove( i ) =
				list_disallow_bbmove( i );
		}
	}

	//Set refine_only in pose_loops_input_ns
	//
	//refine only means no fragment insertions in loop
	//only small,shear,repack + minization, mc accept
	//
	pose_loops_input_ns::refine_only = true;

	//Set small/shear move max size
	//default values are strand: 5.0, loop: 6.0, helix: 0.0
	float helix  = { 0.0 };
	float strand = { 5.0 };
	float other   = { 6.0 };

	set_smallmove_size(helix,strand,other);

	//Must set input_fa to true..pose_loops reads that global
	//and sets pose.fullatom accordingly

	files_paths::input_fa = true;

	//Set trim to true... loop refinement is much faster
	//
	loops_ns::loop_bool::trim = truefalseoption("trim");// Disambiguate from ObjexxFCL function trim

	//other useful loop options that pose_loops utilizes
	//
	loops_ns::loop_bool::fast = truefalseoption("fast");
	loops_ns::loop_bool::fix_natsc = truefalseoption("fix_natsc");

	//Do we need to set this before using pose_loops?
	//	set_fa_lj_rep_slope("highres");

	//put pose in misc for pose_loops to access
	//
	pose_to_misc( pose );

	//run pose_loops protocol
	//
	pose_loops_main();

	//
	//set pose flag to true....it is set to false at the end of pose_loops_main.
	//
	pose_flag_ns::pose_flag_setting = true;

  //
  //set pose_loops_flag to false....it is set to true at the top of pose_loops_main
  //
	loops_ns::loop_flag_bool::pose_loop_flag = false;

	//retrieve pose with closed graft
	//
	bool const fullatom( true );
	bool const ideal_pose( false );
	bool const coords_init( true );
	pose_from_misc( pose, fullatom, ideal_pose, coords_init );

	// reinstate pdb information
	for ( int r = 1, re = pose.total_residue(); r <= re; ++r ) {
		pose.pdb_info().set_pdb_chain( r, pdb_chain( r ) );
		pose.pdb_info().set_pdb_res( r, pdb_residue_num( r ) );
		pose.pdb_info().set_pdb_insert_let( r, pdb_insertion_code( r ) );
	}

	//retrieve chainbreak_score
	//
	chainbreak_score = pose_loops_input_ns::chainbreak_score;

	//reset loop globals in rosetta, else scoring of other structures will invoke loop_get_refold_dir, etc.
	reset_global_loop_info();

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
setup_loops_for_add_on(
											 bool const & rb_moves_used,
											 int const & nres_Ab,
											 int const & scaff_gap_resnum_1,                 //for loop to be grafted
											 int const & scaff_gap_resnum_2,                 //for loop to be grafted
											 int const & scaff_gap_resnum_1_already,//for loop already in scaff
											 int const & scaff_gap_resnum_2_already,//for loop already in scaff
											 int const & n_graft_res_allowed_bbmove_nterm,
											 int const & n_graft_res_allowed_bbmove_cterm,
											 int const & n_scaff_gap_res_allowed_bbmove_nterm,
											 int const & n_scaff_gap_res_allowed_bbmove_cterm,
											 int const & resnum_shift_due_to_graft,
											 int const & n_loops,
											 FArray1DB_int & loop_begin,
											 FArray1DB_int & loop_end,
											 FArray1DB_int & cut_point,
											 int & nres_disallow_bbmove,
											 FArray1D_int & list_disallow_bbmove
											 )
{

	if ( (n_loops!=2 && !rb_moves_used ) || ( n_loops!=4 && rb_moves_used ) ) {
		std::cout << "setup_loops_for_add_on_graft input error: n_loops must be 2, unless rb_moves_used, in which case n_loops must be 4" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( n_scaff_gap_res_allowed_bbmove_nterm < 1 ) {
		std::cout << "setup_loops input error: n_scaff_gap_res_allowed_bbmove_nterm must be >=1" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( n_scaff_gap_res_allowed_bbmove_cterm < 1 ) {
		std::cout << "setup_loops input error: n_scaff_gap_res_allowed_bbmove_cterm must be >=1" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//
	//there are at least two loops, must close both -- think of both as running n2c
	//

	//closure at gap nterm
	loop_begin( 1 ) =  scaff_gap_resnum_1 - n_scaff_gap_res_allowed_bbmove_nterm;
	loop_end( 1 )  =   scaff_gap_resnum_1 - 1 + n_graft_res_allowed_bbmove_nterm;
	cut_point( 1 ) =   scaff_gap_resnum_1 - 1;
	//closure at gap cterm
	loop_begin( 2 )  = scaff_gap_resnum_2 + 1 - n_graft_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;
	loop_end( 2 ) =    scaff_gap_resnum_2 + n_scaff_gap_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;
	cut_point( 2 ) =   scaff_gap_resnum_2 + resnum_shift_due_to_graft;

	std::cout << "loops for add_on graft:" << "1" << ' ' << loop_begin( 1 ) << ' ' << loop_end( 1 ) << ' ' << cut_point( 1 ) << std::endl;
	std::cout << "loops for add_on graft:" << "2" << ' '<< loop_begin( 2 ) << ' ' << loop_end( 2 ) << ' ' << cut_point( 2 ) << std::endl;

	if ( loop_end( 1 ) >= loop_begin( 2 ) ) {
		std::cout << "ERROR in setup_loops_for_add_on: loop_end( 1 ) >= loop_begin( 2 )" << std::endl;
		std::cout << "You must have at least one position within graft that cannot bbmove!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}


	if ( rb_moves_used ) {
		//rb_move will have broken the chain on both sides of the pre-existing graft too!
		//there are two additional loops, must close all!

		//Note: resnum_shift_due_to_graft should not apply here.
		//Assume that when we read in this scaff, it has been renumbered accordingly to include the pre-existing graft
		//
		//closure at gap nterm
		loop_begin( 3 ) =  scaff_gap_resnum_1_already - n_scaff_gap_res_allowed_bbmove_nterm;
		loop_end( 3 )  =   scaff_gap_resnum_1_already - 1 + n_graft_res_allowed_bbmove_nterm;
		cut_point( 3 ) =   scaff_gap_resnum_1_already - 1;
		//closure at gap cterm
		loop_begin( 4 )  = scaff_gap_resnum_2_already + 1 - n_graft_res_allowed_bbmove_cterm;
		loop_end( 4 ) =    scaff_gap_resnum_2_already + n_scaff_gap_res_allowed_bbmove_cterm;
		cut_point( 4 ) =   scaff_gap_resnum_2_already;

	std::cout << "loops for add_on_graft: " << "3" << ' ' << loop_begin( 3 ) << ' ' << loop_end( 3 ) << ' ' << cut_point( 3 ) << std::endl;
	std::cout << "loops for add_on_graft: " << "4" << ' '<< loop_begin( 4 ) << ' ' << loop_end( 4 ) << ' ' << cut_point( 4 ) << std::endl;

	if ( loop_end( 3 ) >= loop_begin( 4 ) ) {
		std::cout << "ERROR in setup_loops_for_add_on: loop_end( 3 ) >= loop_begin( 4 )" << std::endl;
		std::cout << "You must have at least one position within graft that cannot bbmove!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	}

	//
	//identify positions within graft to disallow bbmoves (or repacking!)
	//

	//disallow for the current graft:
	nres_disallow_bbmove = 0;
	int first_disallow = scaff_gap_resnum_1 + n_graft_res_allowed_bbmove_nterm;
	int last_disallow = scaff_gap_resnum_2 - n_graft_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;
	for ( int ires = first_disallow; ires <= last_disallow; ires++ ) {
		++nres_disallow_bbmove;
		list_disallow_bbmove.redimension( nres_disallow_bbmove );
		list_disallow_bbmove( nres_disallow_bbmove ) = ires;
	}

	//*** WILL THIS WORK? THESE PRE-EXISTING POSITIONS ARE NOT WITHIN CURRENT GRAFT. WLLL BE CONTROLLED?

	//disallow for the pre-existing graft:
	//Note: as above, resnum_shift_due_to_graft should not apply here.
	if ( rb_moves_used ) {
		first_disallow = scaff_gap_resnum_1_already + n_graft_res_allowed_bbmove_nterm;
		last_disallow = scaff_gap_resnum_2_already - n_graft_res_allowed_bbmove_cterm;
	} else {
		// no rb_moves, then disallow anything that moved when closing that initial graft
		first_disallow = scaff_gap_resnum_1_already - n_scaff_gap_res_allowed_bbmove_nterm;
		last_disallow = scaff_gap_resnum_2_already + n_scaff_gap_res_allowed_bbmove_cterm;
	}
	for ( int ires = first_disallow; ires <= last_disallow; ires++ ) {
		++nres_disallow_bbmove;
		list_disallow_bbmove.redimension( nres_disallow_bbmove );
		list_disallow_bbmove( nres_disallow_bbmove ) = ires;
	}

	//
	//Here at the end, add nres_Ab to everything since the loop closure will be done in presence of antibody
	//and since antibody will be listed first in the complex
	//
	loop_begin += nres_Ab;
	loop_end += nres_Ab;
	cut_point += nres_Ab;
	if ( nres_disallow_bbmove > 0 ) list_disallow_bbmove += nres_Ab;


}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
setup_loops(
						bool const & rb_moves_used,
						bool const & use_lever_moves,
						std::string const & graft_dir,
						std::string const & graft_align_system,
						int const & scaff_gap_resnum_1,
						int const & scaff_gap_resnum_2,
						int const & n_graft_res_allowed_bbmove_nterm,
						int const & n_graft_res_allowed_bbmove_cterm,
						int const & n_scaff_gap_res_allowed_bbmove_nterm,
						int const & n_scaff_gap_res_allowed_bbmove_cterm,
						int const & resnum_shift_due_to_graft,
						int const & n_loops,
						FArray1DB_int & loop_begin,
						FArray1DB_int & loop_end,
						FArray1DB_int & cut_point,
						int & nres_disallow_bbmove,
						FArray1D_int & list_disallow_bbmove
						)
{

	if ( n_loops < 1 || n_loops > 2 ) {
		std::cout << "setup_loops input error: n_loops must be 1 or 2 " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( rb_moves_used && use_lever_moves ) {
		std::cout << "setup_loops input error: cannot use lever_moves if already rb_moves_used" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( graft_align_system == "S" && use_lever_moves ) {
		std::cout << "setup_loops input error: cannot use_lever_moves with graft_align_system=S for grafting by superposition " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( graft_align_system == "E" && use_lever_moves ) {
		std::cout << "setup_loops input error: cannot use_lever_moves with graft_align_system=E for grafting by endpoint superposition " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( n_scaff_gap_res_allowed_bbmove_nterm < 1 ) {
		std::cout << "setup_loops input error: n_scaff_gap_res_allowed_bbmove_nterm must be >=1" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( n_scaff_gap_res_allowed_bbmove_cterm < 1 ) {
		std::cout << "setup_loops input error: n_scaff_gap_res_allowed_bbmove_cterm must be >=1" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//least aggressive protocol: no rb_moves and no lever moves,  bbmove on one side to close that side     ( nloops = 1 )
	//more aggressive  protocol: no rb_moves but yes lever moves, bbmove on both sides to close one side    ( nloops = 1 )
	//most aggressive protocol:  yes rb moves                     bbmove on boths sides to close both sides ( nloops = 2 )
	//most aggressive protocol:  graft_align_system=S             bbmove on boths sides to close both sides ( nloops = 2 )

	if ( rb_moves_used || graft_align_system == "S" || graft_align_system == "E" ) {

		//there are two loops, must close both
		//closure at gap nterm
		loop_begin( 1 ) =  scaff_gap_resnum_1 - n_scaff_gap_res_allowed_bbmove_nterm;
		loop_end( 1 )  =   scaff_gap_resnum_1 - 1 + n_graft_res_allowed_bbmove_nterm;
		cut_point( 1 ) =   scaff_gap_resnum_1 - 1;
		//closure at gap cterm
		loop_begin( 2 )  = scaff_gap_resnum_2 + 1 - n_graft_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;
		loop_end( 2 ) =    scaff_gap_resnum_2 + n_scaff_gap_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;
		cut_point( 2 ) =   scaff_gap_resnum_2 + resnum_shift_due_to_graft;

		if ( runlevel_ns::runlevel > runlevel_ns::silent ) {
			std::cout << "rb_moves or S graft: loops " << "1" << ' ' << loop_begin( 1 ) << ' ' << loop_end( 1 ) << ' ' << cut_point( 1 ) << std::endl;
			std::cout << "rb_moves or S graft: loops " << "2" << ' '<< loop_begin( 2 ) << ' ' << loop_end( 2 ) << ' ' << cut_point( 2 ) << std::endl;
		}
		if ( loop_end( 1 ) >= loop_begin( 2 ) ) {
			std::cout << "ERROR in setup_loops: loop_end( 1 ) >= loop_begin( 2 )" << std::endl;
			std::cout << "You must have at least one position within graft that cannot bbmove!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}


	} else {
		if ( use_lever_moves ) { //yes lever moves...one chainbreak, bbmove on both sides to close it.

			//with lever moves,
			//there is 1 loop, it spans across graft
			//the loop begin,end are same for graft_dir = n2c or c2n
			//
			loop_begin( 1 )  = scaff_gap_resnum_1 - n_scaff_gap_res_allowed_bbmove_nterm;
			loop_end( 1 ) =    scaff_gap_resnum_2 + n_scaff_gap_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;

			//with lever moves, bbmoves are allowed at nterm and cterm of gap ( and possibly at termini of graft )
			//
			//identify positions within graft to disallow bbmoves
			//
			nres_disallow_bbmove = 0;
			int const first_disallow = scaff_gap_resnum_1 + n_graft_res_allowed_bbmove_nterm;
			int const last_disallow = scaff_gap_resnum_2 - n_graft_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;
			for ( int ires = first_disallow; ires <= last_disallow; ires++ ) {
				++nres_disallow_bbmove;
				list_disallow_bbmove.redimension( nres_disallow_bbmove );
				list_disallow_bbmove( nres_disallow_bbmove ) = ires;
			}

			if ( runlevel_ns::runlevel > runlevel_ns::silent ) {
				std::cout << "first_disallow, last_disallow = " << first_disallow << " " << last_disallow << std::endl;
				std::cout << "loop_begin( 1 ), loop_end( 1 ) = " << loop_begin( 1 ) << " " << loop_end( 1 ) << std::endl;
			}
			//set the cut_points, these differ for n2c or c2n
			//
			//note that we retain coords of scaff_gap_resnum_1 and scaff_gap_resnum_2
			//so with n2c the chainbreak is between scaff_gap_resnum_2 -1 and scaff_gap_resnum_2

			if ( graft_dir == "n2c" ) {
				//closure at gap cterm
				cut_point( 1 ) =   scaff_gap_resnum_2 + resnum_shift_due_to_graft;
			} else if ( graft_dir == "c2n" ) {
				//closure at gap nterm
				cut_point( 1 ) =   scaff_gap_resnum_1 - 1;
			}
		} else {//simplest case, 1 side to close, no lever moves

			if ( graft_dir == "n2c" ) {
				//closure at gap cterm
				loop_begin( 1 )  = scaff_gap_resnum_2 + 1 - n_graft_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;
				loop_end( 1 ) =    scaff_gap_resnum_2 + n_scaff_gap_res_allowed_bbmove_cterm + resnum_shift_due_to_graft;
				cut_point( 1 ) =   scaff_gap_resnum_2 + resnum_shift_due_to_graft;     //cut is between cut_point.C and cut_point+1.N
			} else if ( graft_dir == "c2n" ) {
				//closure at gap nterm
				loop_begin( 1 ) =  scaff_gap_resnum_1 - n_scaff_gap_res_allowed_bbmove_nterm;
				loop_end( 1 )  =   scaff_gap_resnum_1 - 1 + n_graft_res_allowed_bbmove_nterm;
				cut_point( 1 ) =   scaff_gap_resnum_1 - 1;                             //cut is between cut_point.C and cut_point+1.N
			}
		}
	}
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//creates a gap in the pose that include gap_begin and gap_end
void
make_pose_with_gap(
								 pose_ns::Pose & pose,
								 int const & gap_begin,
								 int const & gap_end,
								 pose_ns::Pose & pose_w_gap
								 )
{

	if ( ! ( gap_begin < gap_end ) ) {
		std::cout << "error in make_pose_with_gap:: gap_end not gt gap_begin" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int const nres = pose.total_residue();
	int const nres_w_gap = nres - ( gap_end - gap_begin + 1 );


		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "Removing residues " << gap_begin << " to " << gap_end << " from pose " << std::endl;
		}

	//
	//array "keep" tells which residues to keep in trimmed pose
	//
	FArray1D_bool keep( nres, true );

	for ( int ires = gap_begin; ires <= gap_end; ires++ ) {
		keep( ires ) =  false;
	}

	// make a simple fold_tree
	//
	pose_w_gap.simple_fold_tree( nres_w_gap );

	//Now do Chu's copying stuff.
	//
	FArray1D_int trim_map( nres, 0 );
	FArray1D_int trim_map_reverse( nres, 0 );
	copy_info_to_trim_pose(
												 pose,
												 keep,
												 pose_w_gap,
												 trim_map,
                         trim_map_reverse
												 );
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
//**before calling this routine**
//you must setup the fold tree for trim pose
//
//this routine is blocks copied out of chu's pose_loops_trim_template
//this is overloaded version of below.

void
copy_info_to_trim_pose(
                       pose_ns::Pose & input_pose,
                       FArray1DB_bool const & trim_res,
                       pose_ns::Pose & trim_pose        //output
                       )
{

  // book keeping between input_pose and trim_pose
  int const nres( input_pose.total_residue() );
	FArray1D_int trim_map( nres, 0 ), trim_map_reverse( nres, 0 );

	copy_info_to_trim_pose( input_pose, trim_res, trim_pose, trim_map, trim_map_reverse );
}

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
//**before calling this routine**
//you must setup the fold tree for trim pose
//
//this routine is blocks copied out of chu's pose_loops_trim_template

void
copy_info_to_trim_pose(
                       pose_ns::Pose & input_pose,
                       FArray1DB_bool const & trim_res,
                       pose_ns::Pose & trim_pose,        //output
                       FArray1DB_int & trim_map,          //output
                       FArray1DB_int & trim_map_reverse   //output
                       )
{

  // book keeping between input_pose and trim_pose
  int const nres( input_pose.total_residue() );
  //  FArray1D_int trim_map( nres, 0 ), trim_map_reverse( nres, 0 );
  int j = 0;
  for( int i=1; i<=nres; ++i ) {
    if( trim_res(i) ) {
      trim_map(i) = ++j;
      trim_map_reverse(j) = i;
    }
  }
  int const trim_total_residue = j;

  // do the dirty copying work
  int i;
  FArray3D_float trim_Epos(3, param::MAX_POS, trim_total_residue);
  FArray3D_float trim_fullcoord(3, param::MAX_ATOM(), trim_total_residue);
  FArray3D_float const & input_Epos( input_pose.Eposition() );
  FArray3D_float const & input_fullcoord( input_pose.full_coord() );

  for( int j=1; j<=trim_total_residue; ++j ) {
    i = trim_map_reverse(j);
    trim_pose.set_phi        ( j, input_pose.phi(i) );
    trim_pose.set_psi        ( j, input_pose.psi(i) );
    trim_pose.set_omega      ( j, input_pose.omega(i) );
    trim_pose.set_secstruct  ( j, input_pose.secstruct(i) );
    trim_pose.set_name       ( j, input_pose.name(i) );
    trim_pose.set_res        ( j, input_pose.res(i) );
    trim_pose.set_res_variant( j, input_pose.res_variant(i) );
    for( int k=1; k<=param::MAX_POS; ++k ) {
      for( int l=1; l<=3; ++l ) {
        trim_Epos(l,k,j) = input_Epos(l,k,i);
      }
    }

		// chain id
		trim_pose.pdb_info().set_pdb_chain( j, input_pose.pdb_info().res_chain( i ) );
  }
  if( input_pose.fullatom() ) {
    trim_pose.set_fullatom_flag(true,false);// set fullatom and do not repack
    for( int j=1; j<=trim_total_residue; ++j ) {
      i = trim_map_reverse(j);
      for( int k=1; k<=param::MAX_ATOM()(); ++k ) {
        for( int l=1; l<=3; ++l ) {
          trim_fullcoord(l,k,j) = input_fullcoord(l,k,i);
        }
      }
    }
  }

  bool const ideal_pos( false ), check_missing( false );
  // this will update jumps.
  trim_pose.set_coords( ideal_pos, trim_Epos, trim_fullcoord, check_missing );
  // Important to call pose_to_misc; see below for explanation.
  pose_to_misc( trim_pose );
}


////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

void
evaluate_peptide_prot_interaction_vdwE(
													 int const nres_pep,            //input
													 FArray1D_int const aa_pep,          //input
													 FArray1D_int const aav_pep,         //input
													 FArray3D_float const & coords_pep,   //input
													 pose_ns::Pose & prot_pose,        //input protein pose
													 float & atrE,               //output
													 float & repE               //output
													 )
{

  using namespace aaproperties_pack;
	//  using namespace param_pack; //for the weights

  bool debug ( false );

  //
  //Evaluate interaction between a protein and a peptide
  //

  //weight normally used so count_pair can give diff wts to diff interactions
  //
  float const cp_weight = 1.0;

  int const nres_prot = prot_pose.total_residue();
  FArray3D_float coords_prot ( prot_pose.full_coord() );

	atrE = 0.0;
	repE = 0.0;
	float solE = 0.0;//not using this but it is calculated.

	//loop over residues in the peptide
	//
	for ( int res1 = 1; res1 <= nres_pep; res1++ ) {
		int const aa1 = aa_pep( res1 );
		int const aav1 = aav_pep( res1 );
		int const natoms1 = natoms( aa1, aav1 );
		//coord1,coord2 must be (3,MAX_ATOM()), b/c get_hbE has dimension stmt
		FArray2D_float coord1( 3, param::MAX_ATOM(), 0.0f );
		for ( int iatom = 1; iatom <= natoms1; iatom++ ) {
			for ( int k = 1; k <= 3; k++ ) {
				coord1( k, iatom ) = coords_pep( k, iatom, res1 );
			}
		}

		//loop over residues in the protein
		//
		for ( int res2 = 1; res2 <= nres_prot; res2++ ) {
			int const aa2 ( prot_pose.res( res2 ) );
			int const aav2 ( prot_pose.res_variant( res2 ) );
			int const natoms2 = natoms( aa2, aav2 );
			//coord1,coord2 must be (3,MAX_ATOM()), b/c get_hbE has dimension stmt
			FArray2D_float coord2( 3, param::MAX_ATOM(), 0.0f );
			for ( int iatom = 1; iatom <= natoms2; iatom++ ) {
				for ( int k = 1; k <= 3; k++ ) {
					coord2( k, iatom ) = coords_prot( k, iatom, res2 );
				}
			}

			//this next double loop modified from get_sc_scE
			//but first_scatom was replaced by 1 for atom1 and atom2
			//
			for ( int atom1 = 1, atom1e = nheavyatoms(aa1,aav1),
							atom2s = 1, atom2e = nheavyatoms(aa2,aav2);
						atom1 <= atom1e; ++atom1 ) {
				for ( int atom2 = atom2s; atom2 <= atom2e; ++atom2 ) {

					//the code within this loop mimics a call to fast_pairenergy_with_hydrogens
					//except we have eliminated any reliance on count_pair
					//    fast_pairenergy_with_hydrogens(res1,res2,atom1,atom2,aa1,aa2,aav1,aav2,
					//                                             coord1,coord2,solvE,atrE,repE);

					float d2 ( 0.0 );
					int const atype1 = fullatom_type(atom1,aa1,aav1);
					int const atype2 = fullatom_type(atom2,aa2,aav2);

					fast_pairenergy(coord1(1,atom1),coord2(1,atom2),atype1,atype2,
													solE,atrE,repE,d2,cp_weight);
					//        std::cout << A( 20, "fast_pair: " ) << I( 3, atype1 ) << " " << I( 3, atype2 )
					//                  << " " << F( 10, 5, atrE ) << " " << F( 10, 5, repE )
					//                  << " " << F( 10, 5, solE ) << std::endl;

					//bk   hydrogens are done separately so that a very short cutoff can be
					//bk   used before entering the hbond energy calculation - this can be
					//bk   done because hydrogens are only being used for repulsions

					if ( d2 < pdbstatistics_pack::hydrogen_interaction_cutoff ) {
						fast_pairenergy_attached_h_no_count_pair(atom1,atom2,aa1,aa2,aav1,aav2,
																										 coord1,coord2,repE);
						//          std::cout << A( 20, "fast_pair_h: " ) << I( 3, atype1 ) << " " << I( 3, atype2 )
						//                    << " " << F( 10, 5, atrE ) << " " << F( 10, 5, repE )
						//                    << " " << F( 10, 5, solE ) << std::endl;
						//    fast_pairenergy_attached_h(res1,res2,atom1,atom2,aa1,aa2,aav1,aav2,
						//     coord1,coord2,repE);
					}
				}
			}
		}//res2
	}//res1
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
//This file is here b/c it is inlined in its usual place..I doubt it is necessary.
/// @begin fast_pairenergy_hydrogens_copy
///
/// @brief
///
/// @detailed
///bk    uses lookup table to find the pairwise energies between two atoms,
///bk    this function should only be used in the case that one of the partners
///bk    is a hydrogen.
///bk    currently only the repulsive term is being evaluated with hydrogens
///
///ctsa   -- this version of pairenergy is used for all calls made from
///ctsa   functions in pack.cc (fullatom_energy.cc still calls normal pairenergy)
///ctsa   -- modified so that energies are summed within this function instead
///ctsa    of stored and passed back out to be summed in the calling loop,
///
/// @param  atom1 - [in/out]? -
/// @param  atom2 - [in/out]? -
/// @param  attype1 - [in/out]? -
/// @param  attype2 - [in/out]? -
/// @param  repE - [in/out]? -
///
/// @global_read
///
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
fast_pairenergy_hydrogens_copy(
															 FArray1Da_float atom1,
															 FArray1Da_float atom2,
															 int attype1,
															 int attype2,
															 float & repE,
															 float const cp_weight
															 )
{
	using namespace pdbstatistics_pack;

	float const a_1 = atom1[ 0 ] - atom2[ 0 ]; // atom1(1) - atom2(1)
	float const a_2 = atom1[ 1 ] - atom2[ 1 ]; // atom1(2) - atom2(2)
	float const a_3 = atom1[ 2 ] - atom2[ 2 ]; // atom1(3) - atom2(3)
	float const d2 = ( ( a_1 * a_1 ) + ( a_2 * a_2 ) + ( a_3 * a_3 ) );

	if ( d2 >= safe_max_dis2 || d2 == 0 ) return;

	//ctsa
	//ctsa  get bins and interpolation fraction
	//ctsa
	float const d2_bin = fa_bins_per_A2 * d2;
	int const disbin1 = static_cast< int >( d2_bin ) + 1;
	//  int const disbin2 = disbin1 + 1;
	float const frac = d2_bin - ( disbin1 - 1 );

	//ctsa
	//ctsa   tables have been hacked so that if disbin2 = lastbin, all values = 0.
	//ctsa
	int l = pdbstatistics::pCurrentEtable->ljrep.index(disbin1,attype2,attype1);
	float const repulse1 = pdbstatistics::pCurrentEtable->ljrep[  l ]; // ljrep(disbin1,attype2,attype1);
	float const repulse2 = pdbstatistics::pCurrentEtable->ljrep[ ++l ]; // ljrep(disbin2,attype2,attype1);
	repE += ( repulse1 + frac * ( repulse2 - repulse1 ) ) * cp_weight;

}


////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin fast_pairenergy_attached_h_no_count_pair
///
/// @brief
///bills this is a copy of the routine "fast_pairenergy_attached_h" in pack.cc
///bills I have commented out the calls to count_pair
///bills this routine is called by fast_pairenergy_with_hydrogens_no_count_pair
///
///bk  retrieve energies involving hydrogens attached to atom1 and atom2.
///bk  currently only repulsive is evaluated with hydrogens
///
/// @detailed
///
/// @param  atom1 - [in/out]? -
/// @param  atom2 - [in/out]? -
/// @param  aa1 - [in/out]? -
/// @param  aa2 - [in/out]? -
/// @param  aav1 - [in/out]? -
/// @param  aav2 - [in/out]? -
/// @param  coord1 - [in/out]? -
/// @param  coord2 - [in/out]? -
/// @param  repE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fast_pairenergy_attached_h_no_count_pair(
																				 int const atom1,
																				 int const atom2,
																				 int const aa1,
																				 int const aa2,
																				 int const aav1,
																				 int const aav2,
																				 FArray2Da_float coord1,
																				 FArray2Da_float coord2,
																				 float & repE
																				 )
{
  using namespace aaproperties_pack;
  using namespace param;
	//  using namespace dna_variables;

  coord1.dimension( 3, MAX_ATOM() );
  coord2.dimension( 3, MAX_ATOM() );


  float cp_weight = 1.0;
  int const n2e = nhydrogens_on_atm(atom2,aa2,aav2);
  int const l2s = hydrogens_on_atm.index(1,atom2,aa2,aav2);
  float & coord2_a2( coord2(1,atom2) );

  int const fullatom_type_a2 = fullatom_type(atom2,aa2,aav2);
  int const lf10 = fullatom_type.index(1,aa1,aav1) - 1;
  int const lf20 = fullatom_type.index(1,aa2,aav2) - 1;
  for ( int n1 = 1, l1 = hydrogens_on_atm.index(n1,atom1,aa1,aav1),
					n1e = nhydrogens_on_atm(atom1,aa1,aav1); n1 <= n1e; ++n1, ++l1 ) {
    int const hatom1 = hydrogens_on_atm[ l1 ];
		// hydrogens_on_atm(n1,atom1,aa1,aav1)
    float & coord1_h1( coord1(1,hatom1) );
    int const fullatom_type_h1 = fullatom_type[ lf10 + hatom1 ];
		// fullatom_type(hatom1,aa1,aav1)

    //    if ( count_pair(res1,hatom1,aa1,aav1,res2,atom2,aa2,aav2,true,cp_weight) )
    fast_pairenergy_hydrogens_copy(coord1_h1,coord2_a2,
																	 fullatom_type_h1,fullatom_type_a2,repE,cp_weight);

    for ( int n2 = 1, l2 = l2s; n2 <= n2e; ++n2, ++l2 ) {
      int const hatom2 = hydrogens_on_atm[ l2 ];
			// hydrogens_on_atm(n2,atom2,aa2,aav2)
      //      if ( count_pair(res1,hatom1,aa1,aav1,res2,hatom2,aa2,aav2,true,cp_weight) )
      fast_pairenergy_hydrogens_copy(coord1_h1,coord2(1,hatom2),
																		 fullatom_type_h1,fullatom_type[ lf20 + hatom2 ],repE,cp_weight);
			// fullatom_type(hatom2,aa2,aav2)
    }
  }

  float & coord1_a1( coord1(1,atom1) );

  int const fullatom_type_a1 = fullatom_type(atom1,aa1,aav1);
  for ( int n2 = 1, l2 = l2s; n2 <= n2e; ++n2, ++l2 ) {
    int const hatom2 = hydrogens_on_atm[ l2 ];
		// hydrogens_on_atm(n2,atom2,aa2,aav2)
    //    if ( count_pair(res1,atom1,aa1,aav1,res2,hatom2,aa2,aav2,true,cp_weight) )
    fast_pairenergy_hydrogens_copy(coord1_a1,coord2(1,hatom2),
																	 fullatom_type_a1,fullatom_type[ lf20 + hatom2 ],repE,cp_weight);
		// fullatom_type(hatom2,aa2,aav2)
  }
}

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

void
convert_resnum_range_to_gly(
														pose_ns::Pose & pose_in,
														int const & resnum1,
														int const & resnum2
														)
{

	//
	//error check
	//
	int const nres_pose_in ( pose_in.total_residue() );
	if ( ( resnum1 > resnum2 ) || ( resnum1 < 1 ) || ( resnum2 > nres_pose_in ) ) {
		std::cout << "error convert_resnum_range_to_gly "
							<< resnum1 << " " << resnum2 << " " << nres_pose_in << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//
	//fill misc arrays from this pose
	//
	//  Note: must do this (b/c sets total_residue) before setting design_matrix
	//
	pose_to_misc( pose_in );

	//
	//Set design globals for design_matrix to control which aa allowed at which positions.
	//

	design::use_design_matrix = true;
	design::design_matrix = false; //design_matrix( MAX_AA(), MAX_RES() );

	//
	//only allow gly in the range resnum1 - resnum2
	//
	for ( int ires = resnum1 ; ires <= resnum2 ; ++ires ) {
		design::design_matrix( param_aa::aa_gly, ires ) = true;
	}

	//
	//variables to set for the call to pack_rotamers
	//
	std::string pack_mode( "design" );
	bool make_output_file( false );
	FArray1D_bool allow_repack_local( param::MAX_RES()(), false );
	bool include_current( true );
	bool include_extra ( false );
	FArray2D_int extra_rot( param::MAX_CHI, param::MAX_RES()() ); // dummy variable in this instance
	FArray2D_float extra_chi( param::MAX_CHI, param::MAX_RES()() ); // dummy variable in this instance

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose_in );
	Task.set_task( pack_mode, make_output_file, allow_repack_local,
								 include_current, include_extra, extra_rot, extra_chi);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();
	//
	//design using misc arrays
	//
	pack_rotamers( pose_in, Task);

	//
	//retrieve design from misc.
	//
	// yab: merge into mainline; below not necessary
//	bool const fullatom( true );
//	bool const ideal_pose( false );
//	bool const coords_init( true );
//	pose_from_misc( pose_out, fullatom, ideal_pose, coords_init );
}

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

void
get_alignment_coords(
										 pose_ns::Pose const & nat_pep_gly_pose,
										 pose_ns::Pose const & scaff_gly_pose,
										 std::string const & graft_align_system,
										 std::string const & graft_dir,
										 int const & graft_resnum_1,
										 int const & graft_resnum_2,
										 int const & scaff_gap_resnum_1,
										 int const & scaff_gap_resnum_2,
										 FArray1D_float & coords_pep_cent,
										 FArray1D_float & coords_pep_forw,
										 FArray1D_float & coords_pep_back,
										 FArray1D_float & coords_scaff_cent,
										 FArray1D_float & coords_scaff_forw,
										 FArray1D_float & coords_scaff_back
										 )
{
	//
	//atomnums in Eposition
	//
	int const atomnum_N =  1;  //1 in fcoord
	int const atomnum_CA = 2;  //2 in fcoord
	int const atomnum_C =  4;  //3 in fcoord
	int const atomnum_O = 5;  //4 in fcoord

	int atomnum_cent( 0 ), atomnum_forw( 0 ), atomnum_back( 0 );
	int pep_resnum_cent( 0 ), pep_resnum_forw( 0 ), pep_resnum_back( 0 );
	int scaff_resnum_cent( 0 ), scaff_resnum_forw( 0 ), scaff_resnum_back( 0 );

	if ( graft_align_system == "N" ) {
		atomnum_cent = atomnum_N;
		atomnum_forw = atomnum_CA;
		atomnum_back = atomnum_C;
		if ( graft_dir == "n2c" ) {
			pep_resnum_cent = graft_resnum_1 + 1;
			pep_resnum_forw = graft_resnum_1 + 1;
			pep_resnum_back = graft_resnum_1;
			scaff_resnum_cent = scaff_gap_resnum_1 + 1; // == scaff_stem_resnum_1
			scaff_resnum_forw = scaff_gap_resnum_1 + 1;
			scaff_resnum_back = scaff_gap_resnum_1;
		} else {
			pep_resnum_cent = graft_resnum_2;
			pep_resnum_forw = graft_resnum_2;
			pep_resnum_back = graft_resnum_2 - 1;
			scaff_resnum_cent = scaff_gap_resnum_2;
			scaff_resnum_forw = scaff_gap_resnum_2;
			scaff_resnum_back = scaff_gap_resnum_2 - 1; // == scaff_stem_resnum_2
		}
	} else if ( graft_align_system == "CA" ) {
		atomnum_cent = atomnum_CA;
		atomnum_forw = atomnum_C;
		atomnum_back = atomnum_N;
		if ( graft_dir == "n2c" ) {
			pep_resnum_cent = graft_resnum_1;
			pep_resnum_forw = graft_resnum_1;
			pep_resnum_back = graft_resnum_1;
			scaff_resnum_cent = scaff_gap_resnum_1;
			scaff_resnum_forw = scaff_gap_resnum_1;
			scaff_resnum_back = scaff_gap_resnum_1;
		} else {
			pep_resnum_cent = graft_resnum_2;
			pep_resnum_forw = graft_resnum_2;
			pep_resnum_back = graft_resnum_2;
			scaff_resnum_cent = scaff_gap_resnum_2;
			scaff_resnum_forw = scaff_gap_resnum_2;
			scaff_resnum_back = scaff_gap_resnum_2;
		}
	} else if ( graft_align_system == "C" ) {
		atomnum_cent = atomnum_C;
		atomnum_forw = atomnum_N;
		atomnum_back = atomnum_CA;
		if ( graft_dir == "n2c" ) {
			pep_resnum_cent = graft_resnum_1;     //C
			pep_resnum_forw = graft_resnum_1 + 1; //N
			pep_resnum_back = graft_resnum_1;     //CA
			scaff_resnum_cent = scaff_gap_resnum_1;
			scaff_resnum_forw = scaff_gap_resnum_1 + 1;
			scaff_resnum_back = scaff_gap_resnum_1;
		} else {
			pep_resnum_cent = graft_resnum_2 - 1;        //C
			pep_resnum_forw = graft_resnum_2;            //N
			pep_resnum_back = graft_resnum_2 - 1;        //CA
			scaff_resnum_cent = scaff_gap_resnum_2 - 1; //C
			scaff_resnum_forw = scaff_gap_resnum_2;     //N
			scaff_resnum_back = scaff_gap_resnum_2 - 1; //CA
		}
	}

	for ( int k = 1; k <= 3; k++ ) {
		coords_pep_cent( k ) = nat_pep_gly_pose.Eposition()( k, atomnum_cent, pep_resnum_cent );
		coords_pep_forw( k ) = nat_pep_gly_pose.Eposition()( k, atomnum_forw, pep_resnum_forw );
		coords_pep_back( k ) = nat_pep_gly_pose.Eposition()( k, atomnum_back, pep_resnum_back );
		coords_scaff_cent( k ) = scaff_gly_pose.Eposition()( k, atomnum_cent, scaff_resnum_cent );
		coords_scaff_forw( k ) = scaff_gly_pose.Eposition()( k, atomnum_forw, scaff_resnum_forw );
		coords_scaff_back( k ) = scaff_gly_pose.Eposition()( k, atomnum_back, scaff_resnum_back );
	}
}

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

void
make_scaff_w_graft_gly_pose(
														pose_ns::Pose const & scaff_gly_pose,
														pose_ns::Pose const & graft_gly_pose,
														int const & scaff_gap_resnum_1,
														int const & scaff_gap_resnum_2,
														int const & nres_scaff_w_graft,
														pose_ns::Pose & scaff_w_graft_gly_pose  //output
														)
{

	scaff_w_graft_gly_pose.simple_fold_tree( nres_scaff_w_graft );
	scaff_w_graft_gly_pose.set_fullatom_flag( true, false );// set fullatom and do not repack

	FArray3D_float coords_scaff_w_graft_gly( 3, param::MAX_ATOM(), nres_scaff_w_graft );

	int ires_scaff_w_graft = 0;
	for ( int ires = 1; ires < scaff_gap_resnum_1; ires++ ) {
		++ires_scaff_w_graft;
		scaff_w_graft_gly_pose.set_res        ( ires_scaff_w_graft, scaff_gly_pose.res( ires ) );
		scaff_w_graft_gly_pose.set_res_variant( ires_scaff_w_graft, scaff_gly_pose.res_variant( ires ));
		scaff_w_graft_gly_pose.set_phi        ( ires_scaff_w_graft, scaff_gly_pose.phi( ires ) );
		scaff_w_graft_gly_pose.set_psi        ( ires_scaff_w_graft, scaff_gly_pose.psi( ires ) );
		scaff_w_graft_gly_pose.set_omega      ( ires_scaff_w_graft, scaff_gly_pose.omega( ires ) );
		scaff_w_graft_gly_pose.set_secstruct  ( ires_scaff_w_graft, scaff_gly_pose.secstruct( ires ) );
		scaff_w_graft_gly_pose.set_name       ( ires_scaff_w_graft, scaff_gly_pose.name( ires ) );
		int  const aa( scaff_gly_pose.res( ires ) );
		int  const aav( scaff_gly_pose.res_variant( ires ) );
		int const natoms( aaproperties_pack::natoms( aa, aav ) );
		//res and res_variant are needed to be set before set_coords below

		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "ires_scaff_w_graft,aa,aav:: " << ires_scaff_w_graft << " "
								<< aa << " " << aav << " scaff ires = " << ires << std::endl;
		}
		for ( int iatom = 1; iatom <= natoms; iatom++ ) {
			for ( int k = 1; k <= 3; k++ ) {
				coords_scaff_w_graft_gly( k, iatom, ires ) =
					scaff_gly_pose.full_coord()( k, iatom, ires );
			}
		}
	}
	//
	//loop over graft
	//
	int const nres_graft ( graft_gly_pose.total_residue() );
	for ( int ires = 1; ires <= nres_graft; ires++ ) {
		++ires_scaff_w_graft;
		scaff_w_graft_gly_pose.set_res        ( ires_scaff_w_graft, graft_gly_pose.res( ires ) );
		scaff_w_graft_gly_pose.set_res_variant( ires_scaff_w_graft, graft_gly_pose.res_variant( ires ));
		scaff_w_graft_gly_pose.set_phi        ( ires_scaff_w_graft, graft_gly_pose.phi( ires ) );
		scaff_w_graft_gly_pose.set_psi        ( ires_scaff_w_graft, graft_gly_pose.psi( ires ) );
		scaff_w_graft_gly_pose.set_omega      ( ires_scaff_w_graft, graft_gly_pose.omega( ires ) );
		scaff_w_graft_gly_pose.set_secstruct  ( ires_scaff_w_graft, graft_gly_pose.secstruct( ires ) );
		scaff_w_graft_gly_pose.set_name       ( ires_scaff_w_graft, graft_gly_pose.name( ires ) );
		int const aa( graft_gly_pose.res( ires ) );
		int const aav( graft_gly_pose.res_variant( ires ) );
		int const natoms( aaproperties_pack::natoms( aa, aav ) );
		//res and res_variant are needed to be set before set_coords below
		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "ires_scaff_w_graft,aa,aav:: " << ires_scaff_w_graft << " "
								<< aa << " " << aav << " graft ires = " << ires << std::endl;
		}
		for ( int iatom = 1; iatom <= natoms; iatom++ ) {
			for ( int k = 1; k <= 3; k++ ) {
				coords_scaff_w_graft_gly( k, iatom, ires_scaff_w_graft ) =
					graft_gly_pose.full_coord()( k, iatom, ires );
			}
		}
	}
	int const nres_scaff ( scaff_gly_pose.total_residue() );
	for ( int ires = scaff_gap_resnum_2 + 1; ires <= nres_scaff; ires++ ) {
		++ires_scaff_w_graft;
		scaff_w_graft_gly_pose.set_res        ( ires_scaff_w_graft, scaff_gly_pose.res( ires ) );
		scaff_w_graft_gly_pose.set_res_variant( ires_scaff_w_graft, scaff_gly_pose.res_variant( ires ));
		scaff_w_graft_gly_pose.set_phi        ( ires_scaff_w_graft, scaff_gly_pose.phi( ires ) );
		scaff_w_graft_gly_pose.set_psi        ( ires_scaff_w_graft, scaff_gly_pose.psi( ires ) );
		scaff_w_graft_gly_pose.set_omega      ( ires_scaff_w_graft, scaff_gly_pose.omega( ires ) );
		scaff_w_graft_gly_pose.set_secstruct  ( ires_scaff_w_graft, scaff_gly_pose.secstruct( ires ) );
		scaff_w_graft_gly_pose.set_name       ( ires_scaff_w_graft, scaff_gly_pose.name( ires ) );
		int  const aa( scaff_gly_pose.res( ires ) );
		int  const aav( scaff_gly_pose.res_variant( ires ) );
		int const natoms( aaproperties_pack::natoms( aa, aav ) );
		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "ires_scaff_w_graft,aa,aav:: " << ires_scaff_w_graft << " "
								<< aa << " " << aav << " scaff ires = " << ires << std::endl;
		}
		for ( int iatom = 1; iatom <= natoms; iatom++ ) {
			for ( int k = 1; k <= 3; k++ ) {
				coords_scaff_w_graft_gly( k, iatom, ires_scaff_w_graft ) =
					scaff_gly_pose.full_coord()( k, iatom, ires );
			}
		}
	}

	FArray3D_float Epos_scaff_w_graft_gly( 3, param::MAX_POS, nres_scaff_w_graft );//MAX_POS = 5

	for ( int i = 1; i <= nres_scaff_w_graft; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			Epos_scaff_w_graft_gly(k,1,i) = coords_scaff_w_graft_gly(k,1,i);
			Epos_scaff_w_graft_gly(k,2,i) = coords_scaff_w_graft_gly(k,2,i);
			Epos_scaff_w_graft_gly(k,4,i) = coords_scaff_w_graft_gly(k,3,i);
			Epos_scaff_w_graft_gly(k,5,i) = coords_scaff_w_graft_gly(k,4,i);
			Epos_scaff_w_graft_gly(k,3,i) = coords_scaff_w_graft_gly(k,5,i);
		}
	}
	//
	//for safety these should be inputs to this routine.
	//
	bool const ideal_pose ( false );
	bool const check_missing ( false );
	scaff_w_graft_gly_pose.set_coords( ideal_pose, Epos_scaff_w_graft_gly, coords_scaff_w_graft_gly, check_missing );

	// now save chain_id and set additional pdb info
	char chain_id = scaff_gly_pose.pdb_info().res_chain( 1 ); // assumes only one chain present
	for (int r = 1, re = scaff_w_graft_gly_pose.total_residue(); r <= re; ++r ) {
		// chain id
		scaff_w_graft_gly_pose.pdb_info().set_pdb_chain( r, chain_id );

		// pdb residue numbering
		scaff_w_graft_gly_pose.pdb_info().set_pdb_res( r, r );

		// insertion code (none)
		scaff_w_graft_gly_pose.pdb_info().set_pdb_insert_let( r, ' ' );
	}
}




////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////


// This routine tries to retain both atoms across a peptide bond C-N at
// the "takeoff" side of a graft.
void
new_make_scaff_w_graft_gly_pose(
                                pose_ns::Pose const & scaff_gly_pose,
                                pose_ns::Pose const & pep_gly_oriented_pose,
                                std::string const & graft_align_system,
                                std::string const & graft_dir,
                                int const & graft_resnum_1,
                                int const & graft_resnum_2,
                                int const & scaff_gap_resnum_1,
                                int const & scaff_gap_resnum_2,
                                int const & nres_scaff_w_graft,
                                pose_ns::Pose & scaff_w_graft_gly_pose  //output
                                )
{

  //
  //atomnums in fullcoord
  //
  int const atomnum_N =  1;  //1 in fcoord
  int const atomnum_CA = 2;  //2 in fcoord
  int const atomnum_C =  3;  //3 in fcoord
  int const atomnum_O = 4;  //4 in fcoord
  int const atomnum_gly_H = 5;  //in fcoord, atom number for H on bb nitrogen of GLY
  int const atomnum_gly_1HA = 6; //in fcoord, atom number for 1HA on CA of GLY
  int const atomnum_gly_2HA = 7; //in fcoord, atom number for 2HA on CA of GLY

	int const resnum_shift_due_to_graft = nres_scaff_w_graft - scaff_gly_pose.total_residue();
	int const scaff_gap_resnum_2_shifted = scaff_gap_resnum_2 + resnum_shift_due_to_graft;

	if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
		std::cout << "resnum_shift_due_to_graft, scaff_gap_resnum_2_shifted = " << resnum_shift_due_to_graft << " " << scaff_gap_resnum_2_shifted << std::endl;
		std::cout << "graft_resnum_2 = " << graft_resnum_2 << std::endl;
	}

  scaff_w_graft_gly_pose.simple_fold_tree( nres_scaff_w_graft );
  scaff_w_graft_gly_pose.set_fullatom_flag( true, false );// set fullatom and do not repack

  FArray3D_float coords_scaff_w_graft_gly( 3, param::MAX_ATOM(), nres_scaff_w_graft );

  int ires_scaff_w_graft = 0;
  for ( int ires = 1; ires <= scaff_gap_resnum_1 - 1; ires++ ) {
    ++ires_scaff_w_graft;

		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "1-scaff_gap_resnum-1: " << I(4,ires) << " ---> " << I(4,ires_scaff_w_graft) << std::endl;
		}
    scaff_w_graft_gly_pose.set_res        ( ires_scaff_w_graft, scaff_gly_pose.res( ires ) );
    scaff_w_graft_gly_pose.set_res_variant( ires_scaff_w_graft, scaff_gly_pose.res_variant( ires ));
    scaff_w_graft_gly_pose.set_phi        ( ires_scaff_w_graft, scaff_gly_pose.phi( ires ) );
    scaff_w_graft_gly_pose.set_psi        ( ires_scaff_w_graft, scaff_gly_pose.psi( ires ) );
    scaff_w_graft_gly_pose.set_omega      ( ires_scaff_w_graft, scaff_gly_pose.omega( ires ) );
    scaff_w_graft_gly_pose.set_secstruct  ( ires_scaff_w_graft, scaff_gly_pose.secstruct( ires ) );
    scaff_w_graft_gly_pose.set_name       ( ires_scaff_w_graft, scaff_gly_pose.name( ires ) );
    int  const aa( scaff_gly_pose.res( ires ) );
    int  const aav( scaff_gly_pose.res_variant( ires ) );
    int const natoms( aaproperties_pack::natoms( aa, aav ) );
    //res and res_variant are needed to be set before set_coords below

    if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "ires_scaff_w_graft,aa,aav:: " << ires_scaff_w_graft << " "
                << aa << " " << aav << " scaff ires = " << ires << std::endl;
    }
    for ( int iatom = 1; iatom <= natoms; iatom++ ) {
      for ( int k = 1; k <= 3; k++ ) {
        coords_scaff_w_graft_gly( k, iatom, ires_scaff_w_graft ) =
          scaff_gly_pose.full_coord()( k, iatom, ires );
      }
    }
  }

  //
  //loop over graft residues
	//
	//here put in graft coords from:  graft_resnum_1      to  graft_resnum_2
	//                         from:  scaff_gap_resnum_1  to  scaff_gap_resnum_2
	//
	//below we will make minor corrections to this at the edges
	//
		for ( int ires = graft_resnum_1; ires <= graft_resnum_2; ires++ ) {
    ++ires_scaff_w_graft;
		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "graft_resnum_1-graft_resnum_2: " << I(4,ires) << " ---> " << I(4,ires_scaff_w_graft) << std::endl;
		}
    scaff_w_graft_gly_pose.set_res        ( ires_scaff_w_graft, pep_gly_oriented_pose.res( ires ) );
    scaff_w_graft_gly_pose.set_res_variant( ires_scaff_w_graft, pep_gly_oriented_pose.res_variant( ires ));
    scaff_w_graft_gly_pose.set_phi        ( ires_scaff_w_graft, pep_gly_oriented_pose.phi( ires ) );
    scaff_w_graft_gly_pose.set_psi        ( ires_scaff_w_graft, pep_gly_oriented_pose.psi( ires ) );
    scaff_w_graft_gly_pose.set_omega      ( ires_scaff_w_graft, pep_gly_oriented_pose.omega( ires ) );
    scaff_w_graft_gly_pose.set_secstruct  ( ires_scaff_w_graft, pep_gly_oriented_pose.secstruct( ires ) );
    scaff_w_graft_gly_pose.set_name       ( ires_scaff_w_graft, pep_gly_oriented_pose.name( ires ) );
    int const aa( pep_gly_oriented_pose.res( ires ) );
    int const aav( pep_gly_oriented_pose.res_variant( ires ) );
    int const natoms( aaproperties_pack::natoms( aa, aav ) );
    //res and res_variant are needed to be set before set_coords below
    if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "ires_scaff_w_graft,aa,aav:: " << ires_scaff_w_graft << " "
                << aa << " " << aav << " graft ires = " << ires << std::endl;
    }
    for ( int iatom = 1; iatom <= natoms; iatom++ ) {
      for ( int k = 1; k <= 3; k++ ) {
        coords_scaff_w_graft_gly( k, iatom, ires_scaff_w_graft ) =
          pep_gly_oriented_pose.full_coord()( k, iatom, ires );
      }
    }
  }
  int const nres_scaff ( scaff_gly_pose.total_residue() );
  for ( int ires = scaff_gap_resnum_2 + 1; ires <= nres_scaff; ires++ ) {
    ++ires_scaff_w_graft;
		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "scaff_gap_resnum_2+1->: " << I(4,ires) << " ---> " << I(4,ires_scaff_w_graft) << std::endl;
		}
    scaff_w_graft_gly_pose.set_res        ( ires_scaff_w_graft, scaff_gly_pose.res( ires ) );
    scaff_w_graft_gly_pose.set_res_variant( ires_scaff_w_graft, scaff_gly_pose.res_variant( ires ));
    scaff_w_graft_gly_pose.set_phi        ( ires_scaff_w_graft, scaff_gly_pose.phi( ires ) );
    scaff_w_graft_gly_pose.set_psi        ( ires_scaff_w_graft, scaff_gly_pose.psi( ires ) );
    scaff_w_graft_gly_pose.set_omega      ( ires_scaff_w_graft, scaff_gly_pose.omega( ires ) );
    scaff_w_graft_gly_pose.set_secstruct  ( ires_scaff_w_graft, scaff_gly_pose.secstruct( ires ) );
    scaff_w_graft_gly_pose.set_name       ( ires_scaff_w_graft, scaff_gly_pose.name( ires ) );
    int  const aa( scaff_gly_pose.res( ires ) );
    int  const aav( scaff_gly_pose.res_variant( ires ) );
    int const natoms( aaproperties_pack::natoms( aa, aav ) );
    if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "ires_scaff_w_graft,aa,aav:: " << ires_scaff_w_graft << " "
                << aa << " " << aav << " scaff ires = " << ires << std::endl;
    }
    for ( int iatom = 1; iatom <= natoms; iatom++ ) {
      for ( int k = 1; k <= 3; k++ ) {
        coords_scaff_w_graft_gly( k, iatom, ires_scaff_w_graft ) =
          scaff_gly_pose.full_coord()( k, iatom, ires );
      }
    }
  }
  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  //
  //make corrections to retain peptide-bond ( C-N ) from either scaff or graft...do not split the peptide bond
  //above we have taken all of scaff_gap_resnum_1 from graft, but we need to use particular atoms from the scaffold to maintain
  //proper peptide-bond geometry
  //
  //
  //Note: also make corresponding corrections to phi,psi,omega...so far these have been set to scaff values at scaff_gap_resnum_1
  //
  //

	if ( graft_align_system != "AO" && graft_align_system != "S" ) {
		if ( graft_dir == "n2c" ) {
			if ( graft_align_system == "N" ) {
				// for N n2c graft, the graft coords start at C and O at scaff_gap_resnum_1
				//we have already put in graft coords for this residue so we need to replace the N,CA using scaff coords
				for ( int k = 1; k <= 3; k++ ) {
					coords_scaff_w_graft_gly( k, atomnum_N,       scaff_gap_resnum_1 ) = scaff_gly_pose.full_coord()( k, atomnum_N,       scaff_gap_resnum_1 );
					coords_scaff_w_graft_gly( k, atomnum_gly_H,   scaff_gap_resnum_1 ) = scaff_gly_pose.full_coord()( k, atomnum_gly_H,   scaff_gap_resnum_1 );
					coords_scaff_w_graft_gly( k, atomnum_CA,      scaff_gap_resnum_1 ) = scaff_gly_pose.full_coord()( k, atomnum_CA,      scaff_gap_resnum_1 );
					coords_scaff_w_graft_gly( k, atomnum_gly_1HA, scaff_gap_resnum_1 ) = scaff_gly_pose.full_coord()( k, atomnum_gly_1HA, scaff_gap_resnum_1 );
					coords_scaff_w_graft_gly( k, atomnum_gly_2HA, scaff_gap_resnum_1 ) = scaff_gly_pose.full_coord()( k, atomnum_gly_2HA, scaff_gap_resnum_1 );
				}
			} else {
				if ( graft_align_system == "CA" || graft_align_system == "C" ) {
					//for CA n2c graft, the graft coords include CA, C, O at scaff_gap_resnum_1
					//same for C n2c graft
					//we have already put in graft coords for all of this residue, so we need to replace the N with scaff coords
					for ( int k = 1; k <= 3; k++ ) {
						coords_scaff_w_graft_gly( k, atomnum_N,       scaff_gap_resnum_1 ) = scaff_gly_pose.full_coord()( k, atomnum_N,       scaff_gap_resnum_1 );
						coords_scaff_w_graft_gly( k, atomnum_gly_H,   scaff_gap_resnum_1 ) = scaff_gly_pose.full_coord()( k, atomnum_gly_H,   scaff_gap_resnum_1 );
					}
				}//CA or C
			}//N or not N
			//
		}

		//
		//make corrections to retain peptide-bond ( C-N ) from either scaff or graft...do not split the peptide bond
		//above we have taken all of scaff_gap_resnum_2 from graft, but we need to use particular atoms from the scaffold to maintain
		//proper peptide-bond geometry
		//
		//Note: also make corresponding corrections to phi,psi,omega...so far these have been set to scaff values at scaff_gap_resnum_2
		//

		if ( graft_dir == "c2n" ) {
			if ( graft_align_system == "N" || graft_align_system == "CA" ) {
				// for N c2n graft, the graft coords include N,CA at  scaff_gap_resnum_2
				// same for CA c2n graft
				//we have already put in graft coords for this entire residue so we need to replace the C=O with scaff coords
				if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
					std::cout << "coords_scaff_w_graft_gly before (N)" << std::endl;
					print_atom_coords(std::cout, coords_scaff_w_graft_gly( 1, atomnum_N, scaff_gap_resnum_2_shifted ));
					std::cout << "coords_scaff_w_graft_gly before (CA)" << std::endl;
					print_atom_coords(std::cout, coords_scaff_w_graft_gly( 1, atomnum_CA, scaff_gap_resnum_2_shifted ));
				}
				for ( int k = 1; k <= 3; k++ ) {
					coords_scaff_w_graft_gly( k, atomnum_C,         scaff_gap_resnum_2_shifted ) = scaff_gly_pose.full_coord()( k, atomnum_C, scaff_gap_resnum_2 );
					coords_scaff_w_graft_gly( k, atomnum_O,         scaff_gap_resnum_2_shifted ) = scaff_gly_pose.full_coord()( k, atomnum_O, scaff_gap_resnum_2 );
				}
				if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
					std::cout << "coords_scaff_w_graft_gly after (N)" << std::endl;
					print_atom_coords(std::cout, coords_scaff_w_graft_gly( 1, atomnum_N, scaff_gap_resnum_2_shifted ));
					std::cout << "coords_scaff_w_graft_gly after (CA)" << std::endl;
					print_atom_coords(std::cout, coords_scaff_w_graft_gly( 1, atomnum_CA, scaff_gap_resnum_2_shifted ));
				}
			} else {
				if ( graft_align_system == "C" ) {
					//for C n2c graft, the graft coords include N at scaff_gap_resnum_2
					//same for C n2c graft
					//we have already put in graft coords for this entire residue so we need to replace the CA and C=O with scaff coords
					for ( int k = 1; k <= 3; k++ ) {
						coords_scaff_w_graft_gly( k, atomnum_CA,      scaff_gap_resnum_2_shifted ) = scaff_gly_pose.full_coord()( k, atomnum_CA,      scaff_gap_resnum_2 );
						coords_scaff_w_graft_gly( k, atomnum_gly_1HA, scaff_gap_resnum_2_shifted ) = scaff_gly_pose.full_coord()( k, atomnum_gly_1HA, scaff_gap_resnum_2 );
						coords_scaff_w_graft_gly( k, atomnum_gly_2HA, scaff_gap_resnum_2_shifted ) = scaff_gly_pose.full_coord()( k, atomnum_gly_2HA, scaff_gap_resnum_2 );
						coords_scaff_w_graft_gly( k, atomnum_C,       scaff_gap_resnum_2_shifted ) = scaff_gly_pose.full_coord()( k, atomnum_C,       scaff_gap_resnum_2 );
						coords_scaff_w_graft_gly( k, atomnum_O,       scaff_gap_resnum_2_shifted ) = scaff_gly_pose.full_coord()( k, atomnum_O,       scaff_gap_resnum_2 );
					}
				}//C
			}
		}//if c2n
	}//not AO and not S


	//
	// calculate psi,phi from coords at edges
	//
	float const coord_phi_1
		( periodic_range( dihedral( coords_scaff_w_graft_gly( 1, atomnum_gly_H, scaff_gap_resnum_1 ), // H
																coords_scaff_w_graft_gly( 1, atomnum_N,     scaff_gap_resnum_1 ), // N
																coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_1 ), // CA
																coords_scaff_w_graft_gly( 1, atomnum_C,     scaff_gap_resnum_1 )) // C
											+ 180.0, 360.0 ) );
	float const coord_psi_1
		( periodic_range( dihedral( coords_scaff_w_graft_gly( 1, atomnum_N,     scaff_gap_resnum_1 ), // N
																coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_1 ), // CA
																coords_scaff_w_graft_gly( 1, atomnum_C,     scaff_gap_resnum_1 ), // C
																coords_scaff_w_graft_gly( 1, atomnum_O,     scaff_gap_resnum_1 )) // O
											+ 180.0, 360.0 ) );
	float const coord_omega_1
		( periodic_range( dihedral( coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_1 ),   // CA
																coords_scaff_w_graft_gly( 1, atomnum_C,     scaff_gap_resnum_1 ),   // C
																coords_scaff_w_graft_gly( 1, atomnum_N,     scaff_gap_resnum_1+1 ), // N
																coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_1+1 )) // CA
											, 360.0 ) );
	//
	//omega at scaff_gap_resnum_1-1...in some cases this needs to be re-computed
	//
	float const coord_omega_minus1
		( periodic_range( dihedral( coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_1-1 ),   // CA
																coords_scaff_w_graft_gly( 1, atomnum_C,     scaff_gap_resnum_1-1 ),   // C
																coords_scaff_w_graft_gly( 1, atomnum_N,     scaff_gap_resnum_1 ), // N
																coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_1 )) // CA
											, 360.0 ) );

	//
	//ensure torsion angles are consistent with adjusted coords
	//( all of these are not necessary in all cases of n2c/c2n//N,CA,C but it does not hurt to do it )
	//
	scaff_w_graft_gly_pose.set_phi(   scaff_gap_resnum_1, coord_phi_1   );
	scaff_w_graft_gly_pose.set_psi(   scaff_gap_resnum_1, coord_psi_1   );
	scaff_w_graft_gly_pose.set_omega( scaff_gap_resnum_1, coord_omega_1 );
	scaff_w_graft_gly_pose.set_omega( scaff_gap_resnum_1-1, coord_omega_minus1 );

	if ( runlevel_ns::runlevel > runlevel_ns::silent ) {
		std::cout << "phi_1: " << coord_phi_1 << " psi_1: " << coord_psi_1 << " omega_1: " << coord_omega_1 << " omega_minus1: " << coord_omega_minus1 << std::endl;
	}

	//
	// calculate psi,phi from coords
	//
	float const coord_phi_2
		( periodic_range( dihedral( coords_scaff_w_graft_gly( 1, atomnum_gly_H, scaff_gap_resnum_2_shifted ), // H
																coords_scaff_w_graft_gly( 1, atomnum_N,     scaff_gap_resnum_2_shifted ), // N
																coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_2_shifted ), // CA
																coords_scaff_w_graft_gly( 1, atomnum_C,     scaff_gap_resnum_2_shifted )) // C
											+ 180.0, 360.0 ) );
	float const coord_psi_2
		( periodic_range( dihedral( coords_scaff_w_graft_gly( 1, atomnum_N,     scaff_gap_resnum_2_shifted ), // N
																coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_2_shifted ), // CA
																coords_scaff_w_graft_gly( 1, atomnum_C,     scaff_gap_resnum_2_shifted ), // C
																coords_scaff_w_graft_gly( 1, atomnum_O,     scaff_gap_resnum_2_shifted )) // O
											+ 180.0, 360.0 ) );
	float const coord_omega_2
		( periodic_range( dihedral( coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_2_shifted ),   // CA
																coords_scaff_w_graft_gly( 1, atomnum_C,     scaff_gap_resnum_2_shifted ),   // C
																coords_scaff_w_graft_gly( 1, atomnum_N,     scaff_gap_resnum_2_shifted+1 ), // N
																coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_2_shifted+1 )) // CA
											, 360.0 ) );

	//
	//omega at scaff_gap_resnum_2-1...in some cases this needs to be re-computed
	//
	float const coord_omega_minus2
		( periodic_range( dihedral( coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_2_shifted-1 ),   // CA
																coords_scaff_w_graft_gly( 1, atomnum_C,     scaff_gap_resnum_2_shifted-1 ),   // C
																coords_scaff_w_graft_gly( 1, atomnum_N,     scaff_gap_resnum_2_shifted ), // N
																coords_scaff_w_graft_gly( 1, atomnum_CA,    scaff_gap_resnum_2_shifted )) // CA
											, 360.0 ) );

	//
	//ensure torsion angles are consistent with adjusted coords
	//( all of these are not necessary in all cases of n2c/c2n//N,CA,C but it does not hurt to do it )
	//
	scaff_w_graft_gly_pose.set_phi(   scaff_gap_resnum_2_shifted, coord_phi_2   );
	scaff_w_graft_gly_pose.set_psi(   scaff_gap_resnum_2_shifted, coord_psi_2   );
	scaff_w_graft_gly_pose.set_omega( scaff_gap_resnum_2_shifted, coord_omega_2 );
	scaff_w_graft_gly_pose.set_omega( scaff_gap_resnum_2_shifted-1, coord_omega_minus2 );


	if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
		std::cout << " scaff_gap_resnum_2_shifted " << scaff_gap_resnum_2_shifted << std::endl;
		std::cout << "phi, psi, omega, omega_minus2: " << coord_phi_2 << " " << coord_psi_2 << " " << coord_omega_2 << " " << coord_omega_minus2 << std::endl;
	}

  FArray3D_float Epos_scaff_w_graft_gly( 3, param::MAX_POS, nres_scaff_w_graft );//MAX_POS = 5

  for ( int i = 1; i <= nres_scaff_w_graft; ++i ) {
    for ( int k = 1; k <= 3; ++k ) {
      Epos_scaff_w_graft_gly(k,1,i) = coords_scaff_w_graft_gly(k,1,i);
      Epos_scaff_w_graft_gly(k,2,i) = coords_scaff_w_graft_gly(k,2,i);
      Epos_scaff_w_graft_gly(k,4,i) = coords_scaff_w_graft_gly(k,3,i);
      Epos_scaff_w_graft_gly(k,5,i) = coords_scaff_w_graft_gly(k,4,i);
      Epos_scaff_w_graft_gly(k,3,i) = coords_scaff_w_graft_gly(k,5,i);
    }
  }
  //
  //for safety these should be inputs to this routine.
  //
  bool const ideal_pose ( false );
  bool const check_missing ( false );
  scaff_w_graft_gly_pose.set_coords( ideal_pose, Epos_scaff_w_graft_gly, coords_scaff_w_graft_gly, check_missing );

	// now save chain_id
	char chain_id = scaff_gly_pose.pdb_info().res_chain( 1 ); // assumes only one chain present
	for (int r = 1, re = scaff_w_graft_gly_pose.total_residue(); r <= re; ++r ) {
		// chain id
		scaff_w_graft_gly_pose.pdb_info().set_pdb_chain( r, chain_id );

		// pdb residue numbering
		scaff_w_graft_gly_pose.pdb_info().set_pdb_res( r, r );

		// insertion code (none)
		scaff_w_graft_gly_pose.pdb_info().set_pdb_insert_let( r, ' ' );
	}
}







////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////

void
make_Ab_scaff_w_graft_gly_pose(
															 pose_ns::Pose const & Ab_oriented_pose,
															 pose_ns::Pose const & scaff_w_graft_gly_pose,
															 pose_ns::Pose & Ab_scaff_w_graft_gly_pose
															 )
{
	int const nres_Ab = Ab_oriented_pose.total_residue();
	FArray1D_int const & aa_Ab = Ab_oriented_pose.res();
	FArray1D_int const & aav_Ab = Ab_oriented_pose.res_variant();
	FArray3D_float const & coords_Ab_oriented = Ab_oriented_pose.full_coord();

	int const nres_scaff_w_graft ( scaff_w_graft_gly_pose.total_residue() );
	int const nres_Ab_scaff_w_graft = nres_Ab + nres_scaff_w_graft;

		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "nres_Ab_scaff_w_graft, nres_Ab, nres_scaff_w_graft "
								<< nres_Ab_scaff_w_graft << " " << nres_Ab << " "
								<< nres_scaff_w_graft << std::endl;
		}

	Ab_scaff_w_graft_gly_pose.simple_fold_tree( nres_Ab_scaff_w_graft );
  Ab_scaff_w_graft_gly_pose.set_fullatom_flag( true, false );// set fullatom and do not repack

	FArray3D_float coords_Ab_scaff_w_graft_gly( 3, param::MAX_ATOM(), nres_Ab_scaff_w_graft );

	//
	//put Ab first
	//
	int ires_Ab_scaff_w_graft( 0 );
	for ( int ires = 1; ires <= nres_Ab; ires++ ) {
		++ires_Ab_scaff_w_graft;

		//we cannot do this part unless we actually read in the Ab_pose.
		/*
    Ab_scaff_w_graft_gly_pose.set_res        ( ires_Ab_scaff_w_graft, Ab_pose.res( ires ) );
    Ab_scaff_w_graft_gly_pose.set_res_variant( ires_Ab_scaff_w_graft, Ab_pose.res_variant( ires ));
    Ab_scaff_w_graft_gly_pose.set_phi        ( ires_Ab_scaff_w_graft, Ab_pose.phi( ires ) );
    Ab_scaff_w_graft_gly_pose.set_psi        ( ires_Ab_scaff_w_graft, Ab_pose.psi( ires ) );
    Ab_scaff_w_graft_gly_pose.set_omega      ( ires_Ab_scaff_w_graft, Ab_pose.omega( ires ) );
    Ab_scaff_w_graft_gly_pose.set_secstruct  ( ires_Ab_scaff_w_graft, Ab_pose.secstruct( ires ) );
    Ab_scaff_w_graft_gly_pose.set_name       ( ires_Ab_scaff_w_graft, Ab_pose.name( ires ) );
		*/

		int const aa( aa_Ab( ires ) );
		int const aav( aav_Ab( ires ));
		Ab_scaff_w_graft_gly_pose.set_res( ires, aa );
		Ab_scaff_w_graft_gly_pose.set_res_variant( ires, aav );
		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "Ab_scaff... " << ires_Ab_scaff_w_graft << " "
								<< aa << " " << aav << " Ab " << std::endl;
		}
		int const natoms( aaproperties_pack::natoms( aa, aav ) );
		for ( int iatom = 1; iatom <= natoms; ++iatom ) {
			for ( int k = 1; k <= 3; k++ ) {
				coords_Ab_scaff_w_graft_gly( k, iatom, ires_Ab_scaff_w_graft ) =
					coords_Ab_oriented( k, iatom, ires );
			}
		}

		// chain id
		Ab_scaff_w_graft_gly_pose.pdb_info().set_pdb_chain( ires_Ab_scaff_w_graft, Ab_oriented_pose.pdb_info().res_chain( ires ) );

		// pdb residue numbering
		Ab_scaff_w_graft_gly_pose.pdb_info().set_pdb_res( ires_Ab_scaff_w_graft, ires );

		// insertion code (none)
		Ab_scaff_w_graft_gly_pose.pdb_info().set_pdb_insert_let( ires_Ab_scaff_w_graft, ' ' );
	}
	//
	//Add scaff_w_graft
	//
	for ( int ires = 1; ires <= nres_scaff_w_graft; ires++ ) {
		++ires_Ab_scaff_w_graft;
    Ab_scaff_w_graft_gly_pose.set_res        ( ires_Ab_scaff_w_graft, scaff_w_graft_gly_pose.res( ires ) );
    Ab_scaff_w_graft_gly_pose.set_res_variant( ires_Ab_scaff_w_graft, scaff_w_graft_gly_pose.res_variant( ires ));
    Ab_scaff_w_graft_gly_pose.set_phi        ( ires_Ab_scaff_w_graft, scaff_w_graft_gly_pose.phi( ires ) );
    Ab_scaff_w_graft_gly_pose.set_psi        ( ires_Ab_scaff_w_graft, scaff_w_graft_gly_pose.psi( ires ) );
    Ab_scaff_w_graft_gly_pose.set_omega      ( ires_Ab_scaff_w_graft, scaff_w_graft_gly_pose.omega( ires ) );
    Ab_scaff_w_graft_gly_pose.set_secstruct  ( ires_Ab_scaff_w_graft, scaff_w_graft_gly_pose.secstruct( ires ) );
    Ab_scaff_w_graft_gly_pose.set_name       ( ires_Ab_scaff_w_graft, scaff_w_graft_gly_pose.name( ires ) );
		int const aa( scaff_w_graft_gly_pose.res( ires ) );
		int const aav( scaff_w_graft_gly_pose.res_variant( ires ));
		int const natoms( aaproperties_pack::natoms( aa, aav ) );
		if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
			std::cout << "Ab_scaff... " << ires_Ab_scaff_w_graft << " "
								<< aa << " " << aav << " scaff_w_graft " << std::endl;
		}
		for ( int iatom = 1; iatom <= natoms; ++iatom ) {
			for ( int k = 1; k <= 3; k++ ) {
				coords_Ab_scaff_w_graft_gly( k, iatom, ires_Ab_scaff_w_graft ) =
					scaff_w_graft_gly_pose.full_coord()( k, iatom, ires );
			}
		}

		// chain id
		Ab_scaff_w_graft_gly_pose.pdb_info().set_pdb_chain( ires_Ab_scaff_w_graft, scaff_w_graft_gly_pose.pdb_info().res_chain( ires ) );

		// pdb residue numbering
		Ab_scaff_w_graft_gly_pose.pdb_info().set_pdb_res( ires_Ab_scaff_w_graft, ires );

		// insertion code (none)
		Ab_scaff_w_graft_gly_pose.pdb_info().set_pdb_insert_let( ires_Ab_scaff_w_graft, ' ' );
	}
	//
	//Error check
	//
	if ( ! ( ires_Ab_scaff_w_graft == nres_Ab_scaff_w_graft ) ) {
		std::cout << "error setting up Ab_scaff_w_graft_pose" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	//
	//Setup Epos for the new pose
	//
	FArray3D_float Epos_Ab_scaff_w_graft_gly( 3, param::MAX_POS, nres_Ab_scaff_w_graft );//MAX_POS = 5
	for ( int ires = 1; ires <= nres_Ab_scaff_w_graft; ires++ ) {
		for ( int k = 1; k <= 3; k++ ) {
			Epos_Ab_scaff_w_graft_gly(k,1,ires) = coords_Ab_scaff_w_graft_gly(k,1,ires);
			Epos_Ab_scaff_w_graft_gly(k,2,ires) = coords_Ab_scaff_w_graft_gly(k,2,ires);
			Epos_Ab_scaff_w_graft_gly(k,4,ires) = coords_Ab_scaff_w_graft_gly(k,3,ires);
			Epos_Ab_scaff_w_graft_gly(k,5,ires) = coords_Ab_scaff_w_graft_gly(k,4,ires);
			Epos_Ab_scaff_w_graft_gly(k,3,ires) = coords_Ab_scaff_w_graft_gly(k,5,ires);
		}
	}
	bool const ideal_pose ( false );
	bool const check_missing ( false );
	Ab_scaff_w_graft_gly_pose.set_coords( ideal_pose, Epos_Ab_scaff_w_graft_gly, coords_Ab_scaff_w_graft_gly, check_missing );
}

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
read_list_graft_ranges(
											 int & n_graft_ranges,
											 FArray2D_int & list_graft_ranges
											 )
{

	std::string filename;
	filename = stringafteroption( "list_graft_ranges");

	utility::io::izstream infile ( filename );
	if ( ! infile ) {
		std::cout << "read_list_graft_ranges:"
							<< " cannot open file " << filename
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	n_graft_ranges = 0;
	std::string line;
	while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
		if ( line == blank ) continue;
		std::istringstream line_stream( line );
		++n_graft_ranges;
		list_graft_ranges.redimension( 4, n_graft_ranges );
		line_stream >> list_graft_ranges( 1, n_graft_ranges )
								>> list_graft_ranges( 2, n_graft_ranges )
								>> list_graft_ranges( 3, n_graft_ranges )
								>> list_graft_ranges( 4, n_graft_ranges );
		std::cout << "graft_range " << I( 5, n_graft_ranges );
		for ( int i = 1; i <= 4; i++ ) {
			std::cout << " " << SS( list_graft_ranges( i, n_graft_ranges ));
		}
		std::cout << std::endl;
		//Error check for missing values
		for ( int i = 1; i <= 4; i++ ) {
			if ( list_graft_ranges( i, n_graft_ranges ) == 0 ) {
				std::cout << "ERROR in read_list_graft_ranges, illegal 0 at n_graft_ranges, i = "
									<< n_graft_ranges << ' ' << i << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
	}

	if ( n_graft_ranges == 0 ) {
		std::cout << "No graft ranges read from file " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
read_list_scaff_pdb(
										int & n_scaff,
										FArray1D_string & list_scaff_pdb
										)
{

	std::string filename;
	filename = stringafteroption( "list_scaff_pdb");

	utility::io::izstream infile ( filename );
	if ( ! infile ) {
		std::cout << "read_list_scaff_pdb:"
							<< " cannot open file " << filename
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	n_scaff = 0;
	std::string line;
	while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
		if ( line == blank ) continue;
		std::istringstream line_stream( line );
		++n_scaff;
		list_scaff_pdb.redimension( n_scaff );
		line_stream >> list_scaff_pdb( n_scaff );
		std::cout << "scaff_pdb " << I( 5, n_scaff ) << " "
							<< SS( list_scaff_pdb( n_scaff ))  << std::endl;
	}

	if ( n_scaff == 0 ) {
		std::cout << "No scaffold pdbs read from file " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
//
//Note for the case of the add_on graft, the list of natro_in_epitope
//must cover all loops in sequential order, and it must include the
//maximum range of each loop
//


void
get_natro_in_epitope(
										 int const nres_epitope,
										 int & n_natro_in_epitope,
										 FArray1D_int & list_natro_in_epitope,
										 FArray1D_bool & natro_in_epitope
										 )
{
	std::string filename;
	filename = stringafteroption( "list_natro_in_epitope");

	utility::io::izstream infile ( filename );
	if ( ! infile ) {
		std::cout << "read_natro_in_epitope:"
							<< " cannot open file " << filename
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	n_natro_in_epitope = 0;
	std::string line;
	while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
		if ( line == blank ) continue;
		std::istringstream line_stream( line );
		++n_natro_in_epitope;
		list_natro_in_epitope.redimension( n_natro_in_epitope );
		line_stream >> list_natro_in_epitope( n_natro_in_epitope );
		std::cout << "natro_in_epitope " << I( 5, n_natro_in_epitope ) << " "
							<< SS( list_natro_in_epitope( n_natro_in_epitope ))  << std::endl;
		}

	if ( n_natro_in_epitope == 0 ) {
		std::cout << "n_natro_in_epitope = 0 " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	//
	//Convert the list of natro_start positions into a bool for all nres
	//

	for ( int ires = 1; ires <= nres_epitope; ires++ ) {
		natro_in_epitope( ires ) =  false;
		for ( int i_req = 1; i_req <= n_natro_in_epitope; i_req++ ) {
			if ( ires == list_natro_in_epitope( i_req ) ) {
				natro_in_epitope( ires ) =  true;
				break;
				}
			}
		}
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
read_epi_loop_info(
									 FArray1D_int & nres_epi_loops,
									 FArray1D_int & nranges_epi_loops,
									 FArray3D_int & ranges_epi_loops
									 )
{
	std::string filename;
	filename = stringafteroption( "epi_loop_info");

	utility::io::izstream infile ( filename );
	if ( ! infile ) {
		std::cout << "read_epi_loop_info:"
							<< " cannot open file " << filename
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int max_nranges = 0;

	std::string dummy;
	int n_loop = 0;
	std::string line;
	while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
		if ( line == blank ) continue;
		std::cout << "reading: " << line << std::endl;
		std::istringstream line_stream( line );

		//
		//"loop:" statement
		//
		if ( line.substr(0,5)=="loop:" ) {
			++n_loop;

			//
			//"nres:" statement
			//
			std::string line2;
			getline( infile, line2 );
			std::cout << "reading: " << line2 << std::endl;
			if (line2.substr(0,5)!="nres:") {
				std::cout << "read_epi_loop_info, expect nres:  but got " << line2 << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			std::istringstream line_stream2( line2 );
			nres_epi_loops.redimension( n_loop );
			line_stream2 >> dummy >> nres_epi_loops( n_loop );

			//
			//"nranges:" statement
			//
			std::string line3;
			getline( infile, line3 );
			std::cout << "reading: " << line3 << std::endl;
			if (line3.substr(0,8)!="nranges:") {
				std::cout << "read_epi_loop_info, expect nranges:  but got " << line3 << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			std::istringstream line_stream3( line3 );
			nranges_epi_loops.redimension( n_loop  );
			line_stream3 >> dummy >>  nranges_epi_loops( n_loop );

			//necessary to maintain the largest dimension
			if ( nranges_epi_loops( n_loop ) > max_nranges ) max_nranges = nranges_epi_loops( n_loop );

			//
			//read all "range:" statements for this loop
			//

			ranges_epi_loops.redimension( 4, max_nranges, n_loop  );
			for ( int i_range = 1; i_range <= nranges_epi_loops( n_loop ); i_range++ ) {
				std::string line4;
				getline( infile, line4 );
				std::cout << "reading: " << line4 << std::endl;
				if (line4.substr(0,6)!="range:") {
					std::cout << "read_epi_loop_info, expect range:  but got " << line4 << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				std::istringstream line_stream4( line4 );

				line_stream4 >> dummy
										 >> ranges_epi_loops( 1, i_range, n_loop )    //range_low  in sequential numbering from 1-x
										 >> ranges_epi_loops( 2, i_range, n_loop )    //range_high in sequential numbering from 1-x
										 >> ranges_epi_loops( 3, i_range, n_loop )    //range_low  in native epitope sequence numbering
										 >> ranges_epi_loops( 4, i_range, n_loop );   //range_high in native epitope sequence numbering
			}//i_range
		}//loop: statement read from file
	}//while getline

	if ( n_loop == 0 ) {
		std::cout << "n_loop = 0 " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//report what you read in
	std::cout << "read in n_loops : " << n_loop << std::endl;
	for ( int iloop = 1; iloop <= n_loop; iloop++ ) {
		std::cout << "loop: " << iloop << " nranges: " << nranges_epi_loops( iloop ) << std::endl;
		for ( int irange = 1; irange <= nranges_epi_loops( iloop ); irange++ ) {
			std::cout << "loop: " << iloop << " range: " << irange << " "
								<< ranges_epi_loops( 1, irange, iloop ) << "-"
								<< ranges_epi_loops( 2, irange, iloop ) << " "
								<< ranges_epi_loops( 3, irange, iloop ) << "-"
								<< ranges_epi_loops( 4, irange, iloop ) << std::endl;
		}
	}
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
read_list_superpos_scaff_info(
															FArray1D_string & list_scaff_pdb,
															FArray2D_int & list_range_superposed_in_scaff
															)
{

	std::string filename;
	filename = stringafteroption( "list_superpos_scaff_info");

	utility::io::izstream infile ( filename );
	if ( ! infile ) {
		std::cout << "read_list_superpos_scaff_info:"
							<< " cannot open file " << filename
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int n_scaff = 0;
	std::string line;
	while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
		if ( line == blank ) continue;
		std::istringstream line_stream( line );
		++n_scaff;
		list_scaff_pdb.redimension( n_scaff );
		list_range_superposed_in_scaff.redimension( 4, n_scaff );
		line_stream >> list_scaff_pdb( n_scaff )
								>> list_range_superposed_in_scaff( 1, n_scaff )    //start of resnum range in sequential numbering 1-x
								>> list_range_superposed_in_scaff( 2, n_scaff )    // last of resnum range in sequential numbering 1-x
								>> list_range_superposed_in_scaff( 3, n_scaff )    //first of resnum range in native epitope numbering
								>> list_range_superposed_in_scaff( 4, n_scaff );   //last  of resnum range in native epitope numbering

		std::cout << "scaff_pdb " << I( 5, n_scaff ) << " "
							<< SS( list_scaff_pdb( n_scaff ))  << " "
							<< list_range_superposed_in_scaff( 1, n_scaff ) << " "
							<< list_range_superposed_in_scaff( 2, n_scaff ) << " "
							<< list_range_superposed_in_scaff( 3, n_scaff ) << " "
							<< list_range_superposed_in_scaff( 4, n_scaff ) << std::endl;
	}

	if ( n_scaff == 0 ) {
		std::cout << "No scaffold pdbs read from file " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void
read_list_scaff_info_for_epi_graft_design(
																					FArray1D_string & list_graft_dir,
																					FArray1D_string & list_graft_align_system,
																					FArray2D_int & list_graft_resnums,
																					FArray2D_int & list_scaff_gap_resnums,
																					FArray1D_string & list_scaff_pdb
																					)
{

	std::string filename;
	filename = stringafteroption( "list_scaff_info_for_epi_graft_design");

	utility::io::izstream infile ( filename );
	if ( ! infile ) {
		std::cout << "read_list_scaff_info_for_epi_graft_design:"
							<< " cannot open file " << filename
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int n_scaff = 0;
	std::string line;
	while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
		if ( line == blank ) continue;
		std::istringstream line_stream( line );
		++n_scaff;
		list_graft_dir.redimension( n_scaff );
		list_graft_align_system.redimension( n_scaff );
		list_graft_resnums.redimension( 4, n_scaff );
		list_scaff_gap_resnums.redimension( 2, n_scaff );
		list_scaff_pdb.redimension( n_scaff );
		line_stream >> list_graft_dir( n_scaff )
								>> list_graft_align_system( n_scaff )
								>> list_graft_resnums( 3, n_scaff )                   //start of resnum range in native epitope numbering
								>> list_graft_resnums( 4, n_scaff )                   // last of resnum range in native epitope numbering
								>> list_graft_resnums( 1, n_scaff )                   //start of resnum range in sequential numbering 1-x
								>> list_graft_resnums( 2, n_scaff )                   // last of resnum range in sequential numbering 1-x
								>> list_scaff_gap_resnums( 1, n_scaff )
								>> list_scaff_gap_resnums( 2, n_scaff )
								>> list_scaff_pdb( n_scaff );

		std::cout << "scaff_pdb " << I( 5, n_scaff ) << " "
							<< SS( list_graft_dir( n_scaff ) ) << " "
							<< SS( list_graft_align_system( n_scaff ) ) << " "
							<< list_graft_resnums( 3, n_scaff ) << "-" << list_graft_resnums( 4, n_scaff ) << " "
							<< list_graft_resnums( 1, n_scaff ) << "-" << list_graft_resnums( 2, n_scaff ) << " "
							<< list_scaff_gap_resnums( 1, n_scaff ) << "-" << list_scaff_gap_resnums( 2, n_scaff ) << " "
							<< SS( list_scaff_pdb( n_scaff )) << std::endl;


		////////////////////////////////////////////////////////////////////
		//Error check the inputs
		//
		//		if ( ! ( graft_resnum_1 < graft_resnum_2 ) ) {
		if ( list_graft_resnums( 1, n_scaff ) >= list_graft_resnums( 2, n_scaff ) ) {
			std::cout << "error with graft definition " << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		//		if ( ! ( graft_resnum_1 > 1 ) ) {
		if ( list_graft_resnums( 1, n_scaff ) < 1 ) {
			std::cout << "error with graft definition:: graft_resnum_1 must be >= 1 " << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		//		if ( ! ( scaff_gap_resnum_1 < scaff_gap_resnum_2 ) ) {
		if ( list_scaff_gap_resnums( 1, n_scaff ) >= list_scaff_gap_resnums( 2, n_scaff ) ) {
			std::cout << "error with gap definition " << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		//		if ( ! ( scaff_gap_resnum_1 > 1 ) ) {
		if ( list_scaff_gap_resnums( 1, n_scaff ) < 1 ) {
			std::cout << "error with gap definition:: scaff_gap_resnum_1 must be >= 1 " << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		//		if ( ! ( graft_dir == "n2c" || graft_dir == "c2n" ) ) {
		if ( ! ( list_graft_dir( n_scaff ) == "n2c" || list_graft_dir( n_scaff ) == "c2n" ) ) {
			std::cout << "error with graft_dir [" << list_graft_dir( n_scaff ) << "]" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if ( ! ( list_graft_align_system( n_scaff ) == "N" ||
						 list_graft_align_system( n_scaff ) == "CA" ||
						 list_graft_align_system( n_scaff ) == "C"  ||
						 list_graft_align_system( n_scaff ) == "S"  ||
						 list_graft_align_system( n_scaff ) == "E" ) ) {
			std::cout << "error with graft_align_system [" << list_graft_align_system( n_scaff ) << "]" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

	}

	if ( n_scaff == 0 ) {
		std::cout << "No scaffold pdbs read from file " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}


}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void
read_list_scaff_info_for_add_on_epi_graft_design(
																								 FArray1D_int & list_graft_loop_id,
																								 FArray1D_int & list_graft_range_id,
																								 FArray2D_int & list_graft_resnums,
																								 FArray2D_int & list_scaff_gap_resnums,
																								 FArray1D_int & list_graft_loop_id_already,
																								 FArray2D_int & list_graft_resnums_already,
																								 FArray2D_int & list_scaff_gap_resnums_already,
																								 FArray1D_string & list_scaff_pdb
																								 )
{

	//Here are the outputs from the rough_match protocol and on the right I show what variables we read them into.
	/*
 << SS("scaff")
 << SS("  loop")                           //list_loop_id_to_be_grafted( n_scaff );
 << SS("   rng")                           //list_range_id_to_be_grafted( n_scaff );
 << SS("epi1_nat epi2_nat")                //list_specific_graft_resnums( 4, n_scaff ); i=3,4 only
 << SS(" epi1  epi2")                      //list_specific_graft_resnums( 4, n_scaff ); i=1,2 only
 << SS(" sca1  sca2")                      //list_specific_scaff_gap_resnums( 2, n_scaff );
 << SS("chainbreak_rms")                   //not need
 << SS("chbk_rms_nterm")                   //not need
 << SS("chbk_rms_cterm")                   //not need
 << SS("loop1_id")                         //list_loop_id_already( n_scaff );
 << SS("loop1_epi1_nat loop1_epi2_nat")    //list_graft_resnums_already( 4, n_scaff ); i=3,4 only
 << SS("loop1_epi1 loop1_epi2")            //list_graft_resnums_already( 4, n_scaff ); i=1,2 only
 << SS(" scaff_filename") << std::endl;
	*/


	std::string filename;
	filename = stringafteroption( "list_scaff_info_for_add_on_epi_graft_design");

	utility::io::izstream infile ( filename );
	if ( ! infile ) {
		std::cout << "read_list_scaff_info_for_add_on_epi_graft_design:"
							<< " cannot open file " << filename
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int n_scaff = 0;
	std::string line;
	while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
		if ( line == blank ) continue;
		std::istringstream line_stream( line );
		++n_scaff;
		list_graft_loop_id.redimension( n_scaff );
		list_graft_range_id.redimension( n_scaff );
		list_graft_resnums.redimension( 4, n_scaff );
		list_scaff_gap_resnums.redimension( 2, n_scaff );
		list_graft_loop_id_already.redimension( n_scaff );
		list_graft_resnums_already.redimension( 2, n_scaff );
		list_scaff_gap_resnums_already.redimension( 2, n_scaff );
		list_scaff_pdb.redimension( n_scaff );
		line_stream >> list_graft_loop_id( n_scaff )
								>> list_graft_range_id( n_scaff )
								>> list_graft_resnums( 3, n_scaff )                   //start of resnum range in native epitope numbering
								>> list_graft_resnums( 4, n_scaff )                   // last of resnum range in native epitope numbering
								>> list_graft_resnums( 1, n_scaff )                   //start of resnum range in sequential numbering 1-x
								>> list_graft_resnums( 2, n_scaff )                   // last of resnum range in sequential numbering 1-x
								>> list_scaff_gap_resnums( 1, n_scaff )
								>> list_scaff_gap_resnums( 2, n_scaff )
								>> list_graft_loop_id_already( n_scaff )
								>> list_graft_resnums_already( 1, n_scaff )
								>> list_graft_resnums_already( 2, n_scaff )
								>> list_scaff_gap_resnums_already( 1, n_scaff )
								>> list_scaff_gap_resnums_already( 2, n_scaff )
								>> list_scaff_pdb( n_scaff );

		std::cout << "scaff_pdb " << I( 5, n_scaff ) << " "
							<< SS( list_graft_loop_id( n_scaff ) ) << " "
							<< SS( list_graft_range_id( n_scaff ) ) << " "
							<< list_graft_resnums( 3, n_scaff ) << "-" << list_graft_resnums( 4, n_scaff ) << " "
							<< list_graft_resnums( 1, n_scaff ) << "-" << list_graft_resnums( 2, n_scaff ) << " "
							<< list_scaff_gap_resnums( 1, n_scaff ) << "-" << list_scaff_gap_resnums( 2, n_scaff ) << " "
							<< list_graft_loop_id_already( n_scaff ) << " "
							<< list_graft_resnums_already( 1, n_scaff ) << "-" << list_graft_resnums_already( 2, n_scaff ) << " "
							<< list_scaff_gap_resnums_already( 1, n_scaff ) << "-" <<list_scaff_gap_resnums_already( 2, n_scaff ) << " "
							<< SS( list_scaff_pdb( n_scaff )) << std::endl;
	}

	if ( n_scaff == 0 ) {
		std::cout << "No scaffold pdbs read from file " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void
get_loop_id_superposed_in_scaff(
																int const & n_scaff,
																FArray2D_int const & list_range_superposed_in_scaff,
																int const & n_epi_loops,
																FArray1D_int const & nranges_epi_loops,
																FArray3D_int const & ranges_epi_loops,
																FArray1D_int & loop_id_superposed_in_scaff
																)
{
	std::cout << " in get_loop_id " << std::endl;
	if ( n_epi_loops == 0 ) {
		std::cout << "ERROR get_loop_id_superposed_in_scaff: input n_epi_loops = 0 " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( n_scaff == 0 ) {
		std::cout << "ERROR get_loop_id_superposed_in_scaff: input n_scaff = 0 " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i_scaff = 1; i_scaff <= n_scaff; i_scaff++ ) {
		//begin, end of superpos region in scaffold...but given in corresponding native epitope numbering
		//
		int const begin_range_superposed_in_scaff_nat_numbering = list_range_superposed_in_scaff( 3, i_scaff );
		int const end_range_superposed_in_scaff_nat_numbering = list_range_superposed_in_scaff( 4, i_scaff );
		int loop_id = 0;
		bool found = false;
		for ( int i_loop = 1; i_loop <= n_epi_loops; i_loop++ ) {
			int const nranges = nranges_epi_loops( i_loop );
			for ( int i_range = 1; i_range <= nranges; i_range++ ) {
				int const begin_nat_loop = ranges_epi_loops( 3, i_range, i_loop );
				int const end_nat_loop = ranges_epi_loops( 4, i_range, i_loop );
				if ( begin_range_superposed_in_scaff_nat_numbering == begin_nat_loop &&
						 end_range_superposed_in_scaff_nat_numbering == end_nat_loop ) {
					loop_id = i_loop;
					found = true;
					break;//break out of loop over i_range
				}
			}
			if ( found ) break;//break out of loop over i_loop
		}
		if ( loop_id ==0 || !found ) {
			std::cout << "ERROR in get_loop_id_superposed_in_scaff, loop_id, found = " << loop_id << " " << found << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		loop_id_superposed_in_scaff( i_scaff ) = loop_id;
		std::cout << "loop_id_superposed_in_scaff: " << i_scaff << " "
							<< list_range_superposed_in_scaff( 3, i_scaff ) << "-"
							<< list_range_superposed_in_scaff( 4, i_scaff ) << " "
							<< loop_id << std::endl;
	}
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
copy_segment_to_new_pose(
												 pose_ns::Pose & pose_in,
												 int const & pos1,
												 int const & pos2,
												 pose_ns::Pose & pose_out
												 )
{
	int const nres_in = pose_in.total_residue();
	int const nres_out = pose_out.total_residue();
	//
	//Error check
	//
	if ( pos1 < 0 || pos1 > nres_in ) {
		std::cout << "ERROR in copy_segment_to_new_pose: pos1, nres_in "
							<< pos1 << ' ' << nres_in << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( pos2 < 0 || pos2 > nres_in ) {
		std::cout << "ERROR in copy_segment_to_new_pose: pos2, nres_in "
							<< pos2 << ' ' << nres_in << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( pos1 >= pos2 ) {
		std::cout << "ERROR in copy_segment_to_new_pose: pos1 >= pos2 "
							<< pos1 << ' ' << pos2 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( pos2-pos1+1 > nres_out ) {
		std::cout << "ERROR in copy_segment_to_new_pose: length_segment > nres_out. length: "
							<< pos2-pos1+1 << " nres_out: " << nres_out << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	FArray1D_bool trim( nres_in, false );
	for ( int i = pos1 ; i <= pos2; i++ ) trim( i ) = true;
	copy_info_to_trim_pose( pose_in, trim, pose_out );
}
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
void
open_output_file(
								 std::string const & output_filename,
								 utility::io::ozstream & outfile
								 )
{
	std::string filename = output_filename;

	// want scorefile to live in score_path
	filename = files_paths::score_path + filename;

	// check if file exists
	utility::io::izstream infile ( filename );
	if ( infile ) {
		outfile.open_append( filename );
	} else {
		outfile.open( filename );

		if ( ! outfile.good() ) {
			std::cout << "cant open output_file for writing: "
								<< filename << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		//
		//only write the header the first time you open the file
		//
		//you can do that here if you want...
	}
	if ( ! outfile.good() ) {
		std::cout << "cant open output_file for writing: "
							<< filename << std::endl;
	}
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
//check-pointing for running on clusters
void
update_conformation_already_scored(
                                   int const & n,                 //number for conformation just completed
                                   std::string const & fullname  //path + filename for checkpoint file
                                   )
{

	//  std::string fullname  = files_paths::score_path + filename;
	utility::io::ozstream my_ozstream( fullname );
	my_ozstream.open( fullname );
	if ( !my_ozstream ) {
    	my_ozstream.close();
    	my_ozstream.clear();
    	return;
  	}
	my_ozstream << SS( n ) << '\n'; //std::endl;
	my_ozstream.close();
	my_ozstream.clear();
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
//has conformation number n already been completed?
bool
check_conformation_already_scored(
                                  int const & no,         //input: number of the current conformation
                                  std::string const & fullname  //input: path + filename to checkpoint file.
                                  )
{

  int last_output;//last conformation completed, according to checkpoint file

  bool already_scored = false;

  //  std::string fullname = files_paths::score_path + filename;
	utility::io::izstream my_izstream( fullname );
  if ( !my_izstream ) { // checkpoint file doesn't exist yet
    last_output = 0;
  } else {
    my_izstream >> last_output; // >> skip; //for some reason skip does not compile.
    int loop_counter(0);
    while ( my_izstream.fail() ) {
      ++loop_counter;
      if ( loop_counter > 120 ) {
        last_output = no-1;
        break;
      }
			std::cout << "WARNING: check_conformation_already_scored: can not read from " <<
        fullname.c_str() << std::endl;
      my_izstream.clear();
      my_izstream.close();
			utility::sys_sleep(1);
      my_izstream.open( fullname.c_str() );
      //      my_izstream >> last_start >> skip;
      my_izstream >> last_output; //  >> skip; //for some reason skip does not compile.
			std::cout << "WARNING: no last_output " << no << ' ' << last_output<< std::endl;
    }
  }
  my_izstream.close();
  my_izstream.clear();

  if ( no <= last_output ) {
    already_scored = true;
  } else if ( no == last_output + 1 ) {
    already_scored = false;
  } else if ( no > last_output + 1 ) {
		std::cout << "ERROR: check_conformation_already_scored: no > last_output+1 "
              << no << ' ' << last_output+1 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }
  return already_scored;
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
void
get_Ab_scaff_refine_pose(
												 pose_ns::Pose & Ab_scaff_design_pose,
												 pose_ns::Pose & Ab_design_pose,
												 //												 pose_ns::Pose & scaff_design_pose,
												 pose_ns::Pose & scaff_refine_pose,
												 int const & resnum_first_disallow_bbmove,
												 pose_ns::Pose & Ab_refine_pose,
												 pose_ns::Pose & Ab_scaff_refine_pose
												 )
{
	//
	//atomnums in full_coord
	//
	int atomnum_N,atomnum_CA,atomnum_C;
	atomnum_N =  1;
	atomnum_CA = 2;
	atomnum_C =  3;

	//
	//Alignment coords to map Ab rigid-body orientation before->after refine.
	//

	//
	//first residue in the graft that moves as rigid-body relative to the scaffold
	//
	int resnum_in_graft_for_align = resnum_first_disallow_bbmove;
	std::cout << "get_Ab_scaff_refine_pose: resnum_in_graft_for_align = " << resnum_in_graft_for_align << std::endl;

	//
	//atom numbers used for coord alignment
	//
	FArray1D_int atomnum_align( 3 );
	atomnum_align( 1 ) = atomnum_N; //back
	atomnum_align( 2 ) = atomnum_CA; //cent
	atomnum_align( 3 ) = atomnum_C; //forw

	int const nres_Ab ( Ab_design_pose.total_residue() );

	//before refine ( use scaff_design_pose ):
	//after refine ( use scaff_refine_pose ):
	//
	FArray2D_float graft_coords_before_refine( 3, 3 ); //second index is 1=back,2=cent,3=forw
	FArray2D_float graft_coords_after_refine( 3, 3 ); //second index is 1=back,2=cent,3=forw

	for ( int i_atom = 1; i_atom <= 3; i_atom++ ) {
		for ( int k = 1; k <= 3; k++ ) {
			graft_coords_before_refine( k, i_atom ) = Ab_scaff_design_pose.full_coord()( k, atomnum_align( i_atom ), resnum_in_graft_for_align+nres_Ab );//Ab is first
			graft_coords_after_refine(  k, i_atom ) =    scaff_refine_pose.full_coord()( k, atomnum_align( i_atom ), resnum_in_graft_for_align );
		}
	}

	//
	//Mgl_refine is rot/trans matrix describing rigid-body movement of graft during refinement
	//

	FArray2D_float Mgl_refine( 4, 4 );
	get_GL_matrix( graft_coords_before_refine( 1, 1 ),
								 graft_coords_before_refine( 1, 2 ),
								 graft_coords_before_refine( 1, 3 ),
								 graft_coords_after_refine(  1, 1 ),
								 graft_coords_after_refine(  1, 2 ),
								 graft_coords_after_refine(  1, 3 ),
								 Mgl_refine );

	//
	//New Ab coords
	//
	FArray3D_float coords_Ab_after_refine( 3, param::MAX_ATOM(), nres_Ab );
	for ( int i_res = 1; i_res <= nres_Ab; i_res++ ) {
		int const aa( Ab_design_pose.res( i_res ) );
		int const aav( Ab_design_pose.res_variant( i_res ));
		for ( int i_atom = 1; i_atom <= aaproperties_pack::natoms( aa, aav ); ++i_atom ) {
			for ( int k = 1; k <= 3; k++ ) {
				//initialize as coords before refine
				coords_Ab_after_refine( k, i_atom, i_res ) = Ab_scaff_design_pose.full_coord()( k, i_atom, i_res );
			}
			//transform to coords after refine
			GL_rot_in_place( Mgl_refine, coords_Ab_after_refine( 1, i_atom, i_res ) );
		}
	}

	//	Pose Ab_after_refine_pose;
	Ab_refine_pose = Ab_design_pose;
	FArray3D_float Epos_Ab_after_refine( 3, param::MAX_POS, nres_Ab );//MAX_POS = 5

	for ( int i_res = 1; i_res <= nres_Ab; i_res++ ) {
		for ( int k = 1; k <= 3; ++k ) {
			Epos_Ab_after_refine(k,1,i_res) = coords_Ab_after_refine(k,1,i_res);
			Epos_Ab_after_refine(k,2,i_res) = coords_Ab_after_refine(k,2,i_res);
			Epos_Ab_after_refine(k,4,i_res) = coords_Ab_after_refine(k,3,i_res);
			Epos_Ab_after_refine(k,5,i_res) = coords_Ab_after_refine(k,4,i_res);
			Epos_Ab_after_refine(k,3,i_res) = coords_Ab_after_refine(k,5,i_res);
		}
	}

	bool ideal_pose ( false );
	bool check_missing ( false );
	Ab_refine_pose.set_coords( ideal_pose, Epos_Ab_after_refine, coords_Ab_after_refine, check_missing );

	//Pose Ab_scaff_refine_pose;
	construct_pose_complex_from_p1_p2( Ab_refine_pose, scaff_refine_pose, Ab_scaff_refine_pose );

}
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
void
get_filename_with_energies(
													 std::string const & filename_base,
													 float const & energy1,       //Etot
													 float const & energy2,        //ddg
													 std::string & filename_out    //output filename
													 )
{

	//
	//return filename =  filename_base + formatted energy1 + formatted energy2 .pdb
	//

	//Put energies in the pdb filename, but ensure no overflow
	//
	float const max_val_tmp = 99999.99; // big (+) number 8 digit
	float const min_val_tmp = -9999.99; // big (-) number 7 digit
	std::string Etot_str;
	if ( energy1 > max_val_tmp ) {
		Etot_str = fixed_string_of( max_val_tmp, 8, 2 );
		std::cout << "Etot > " << max_val_tmp << std::endl;
	} else {
		if ( energy1 < min_val_tmp ) {
			Etot_str = fixed_string_of( min_val_tmp, 8, 2 );
			std::cout << "Etot < " << min_val_tmp << std::endl;
		} else {
			//
			//Etot reported here
			//
			Etot_str = fixed_string_of( energy1, 8, 2 );
		}
	}
	std::string ddg_bind_str;
	if ( energy2 > max_val_tmp ) {
		ddg_bind_str = fixed_string_of( max_val_tmp, 8, 2 );
	} else {
		if ( energy2 < min_val_tmp ) {
			ddg_bind_str = fixed_string_of( min_val_tmp, 8, 2 );
		} else {
			ddg_bind_str = fixed_string_of( energy2, 8, 2 );
		}
	}

	std::string Etot_str_formatted = right_string_of( trim( left_justify( Etot_str ) ), 8, '_' );
	std::string ddg_bind_str_formatted = right_string_of( trim( left_justify( ddg_bind_str ) ), 8, '_' );
	filename_out = filename_base + "_" + Etot_str_formatted + "_" + ddg_bind_str_formatted + "_.pdb";
	std::cout << "filename: " << SS( filename_out ) << std::endl;

	/*
		std::cout << "max_val_tmp, min_val_tmp " << max_val_tmp << " " << min_val_tmp << std::endl;
		std::cout << "Etot " << score_Ab_scaff_design( i_design ) << std::endl;
		std::cout << "Etot_str, Etot_str_formatted " << SS(Etot_str) << " " << SS(Etot_str_formatted) << std::endl;
	*/
}


///////////////////////////
/* yab: temporary cheats */
///////////////////////////

using namespace rootstock;


/////////////////////////////////////////////////////
/* yab: below is additional code for all protocols */
/////////////////////////////////////////////////////


/// @brief cheat here and call copy_segment_to_new_pose using ClosedInterval for clarity, merge and clean later
/// @note retains Bill's 'in' -> 'variables' -> 'out' syntax
void
copy_segment_to_new_pose(
	pose_ns::Pose & in,
	epigraft::ResidueRange const & rr,
	pose_ns::Pose & out
)
{
	copy_segment_to_new_pose( in, rr.left(), rr.right(), out );
}

/// @brief cheat for clarity on merging scaffold and epitope
void
insert_epitope(
	pose_ns::Pose & scaffold,
	pose_ns::Pose & epitope,
	epigraft::ResidueRange const & match_range,
	pose_ns::Pose & result
)
{
	// new total residue count computed and fed as follows:
	// take scaffold, remove all in match_range including the endpoints, then add epitope
	make_scaff_w_graft_gly_pose( scaffold, epitope, match_range.left(), match_range.right(),
	                             scaffold.total_residue() - match_range.length() + epitope.total_residue(),
	                             result );
}


////////////////////////////////////////////////////////
/* yab: below is additional code for epi_graft_design */
////////////////////////////////////////////////////////

/// @brief resets global loop info
void
reset_global_loop_info()
{
	loops_ns::loop_int::num_loop = 0;
	loops_ns::loop_int::loop_begin = 0;
	loops_ns::loop_int::loop_end = 0;
	loops_ns::loop_int::loop_map = 0;
}

