// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   epigraft_functions.hh
/// @brief  Various functions for epigraft protocols.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)
/// @author Sergey Menis (menis@u.washington.edu)


#ifndef INCLUDED_epigraft_epigraft_functions_HH_
#define INCLUDED_epigraft_epigraft_functions_HH_

// epigraft headers
#include <epigraft/epigraft_types.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/epi_graft.h>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// Rosetta headers
#include <aaproperties_pack.h>
#include <param.h>
#include <param_pack.h>
#include <pdbstatistics_pack.h>
#include <pose.h>
#include <refold.h>
#include <rms.h>

// numeric headers
#include <numeric/conversions.hh>
#include <numeric/xyzVector.hh>

// C++ headers
#include <set>


namespace epigraft {


using namespace rootstock; // temporary

/// @brief compute numbering offset in native (peptide) structure for this loop_id
/// @note  We should re-standardize the input format -- the following routine will not work correctly
/// @note  if there is no single longest subrange!
/// @note  Consider wrapping this into a LoopsToScan class that subclasses utility::vector1.
// TODO: find a better mechanism for this, such as a LoopsToScan class as indicated above
inline
Integer
compute_loop_offset(
	utility::vector1< LoopInfo > const & loops_to_scan,
	Integer loop_id
)
{
	Integer offset = 0;

	LoopInfo const & current_loop = loops_to_scan[ loop_id ];

	for ( Integer k = 1, ke = loops_to_scan.size(); k <= ke; ++k ) {
		if ( k != loop_id ) {

			LoopInfo const & other_loop = loops_to_scan[ k ];

			if ( current_loop.full_native_range().overlaps( other_loop.full_native_range() ) ) {
				if ( current_loop.full_native_range().begin() >= other_loop.full_native_range().begin() ) {
					// here the two ranges overlap and the left endpoint of the current loop lies to the right
					// of the left endpoint of the other loop
					offset += current_loop.full_native_range().begin() - other_loop.full_native_range().begin();
				}

			} else if ( current_loop.full_native_range().begin() > other_loop.full_native_range().begin() ) {
				// here the two ranges are disjoint and the current loop lies completely on the right side
				// of the other loop
				offset += other_loop.full_native_range().length();
			}
		}
	}

	return offset;
}


/// @brief inner product between two vectors a->b defined by two pairs of points
/// @note  possible to do orientation check between a pair of vectors (two pairs of points) by their inner product,
/// @note  if negative or zero then return false
/// @note  this is equivalent to saying that the angle between them is >= 90 degrees, or that
/// @note  they are on opposite sides of the sphere
/// @note  Meant as a filter for loop closure.
template< typename T >
inline
Real
inner_product(
	T const * pair1_a,
	T const * pair1_b,
	T const * pair2_a,
	T const * pair2_b
)
{
	numeric::xyzVector< T > pair1( pair1_b );
	pair1 = pair1 - numeric::xyzVector< T >( pair1_a );

	numeric::xyzVector< T > pair2( pair2_b );
	pair2 = pair2 - numeric::xyzVector< T >( pair2_a );

	return numeric::inner_product( pair1, pair2 );
}


/// @brief angle between two vectors a->b defined by two pairs of points in radians
/// @note  Meant as a filter for loop closure.
template< typename T >
inline
T
angle_of_radians(
	T const * pair1_a,
	T const * pair1_b,
	T const * pair2_a,
	T const * pair2_b
)
{
	using namespace numeric;

	xyzVector< T > pair1( pair1_b );
	pair1 = pair1 - xyzVector< T >( pair1_a );

	xyzVector< T > pair2( pair2_b );
	pair2 = pair2 - xyzVector< T >( pair2_a );

	return angle_of( pair1, pair2 );
}


/// @brief angle between two vectors a->b defined by two pairs of points in degrees
/// @note  Meant as a filter for loop closure.
/// @note  the full function exists instead of calling 'closure_angle_radians'
/// @note  to eliminate the extra expense of creating an additional FArray argument array
template< typename T >
inline
T
angle_of_degrees(
	T const * pair1_a,
	T const * pair1_b,
	T const * pair2_a,
	T const * pair2_b
)
{
	using namespace numeric;

	xyzVector< T > pair1( pair1_b );
	pair1 = pair1 - xyzVector< T >( pair1_a );

	xyzVector< T > pair2( pair2_b );
	pair2 = pair2 - xyzVector< T >( pair2_a );

	return numeric::conversions::degrees( angle_of( pair1, pair2 ) );
}


/// @brief pose preparation, sets simple fold tree and fullatom flag to true
inline
void
prepare_pose(
	Pose & p,
	int const & nres
)
{
	p.simple_fold_tree( nres );
	p.set_fullatom_flag( true, false ); // force booleans: fullatom, repack
}


/// @brief transfer only partial identity information (res, res_variant, secstruct, name) between two Poses
/// @note  this function exists without transferring dihedrals and coordinates because Pose::set_coords
/// @note  does internal recalculations to dihedrals and requires all coordinates to exist
inline
void
transfer_identity_info_between_Poses(
	Pose const & p1,
	ResidueRange const & p1_range,
	Pose & p2,
	int const & p2_begin,
	bool const & keep_pdb_info = false
)
{
	for ( Integer residue = p1_range.begin(), last_residue = p1_range.end(), p2_count = p2_begin; residue <= last_residue; ++residue, ++p2_count ) {
		p2.set_res        ( p2_count, p1.res( residue ) );
		p2.set_res_variant( p2_count, p1.res_variant( residue ));
		p2.set_secstruct  ( p2_count, p1.secstruct( residue ) );
		p2.set_name       ( p2_count, p1.name( residue ) );
	}

	if ( keep_pdb_info ) { // outside prior loop for speed
		for ( Integer residue = p1_range.begin(), last_residue = p1_range.end(), p2_count = p2_begin; residue <= last_residue; ++residue, ++p2_count ) {
			p2.pdb_info().set_pdb_chain( p2_count, p1.pdb_info().res_chain( residue ) );
			p2.pdb_info().set_pdb_res( p2_count, p1.pdb_info().pdb_res_num( residue ) );
			p2.pdb_info().set_pdb_insert_let( p2_count, p1.pdb_info().pdb_insert_let( residue ) );
		}
	}
}


/// @brief transfer only partial identity information (res, res_variant, secstruct, name) between two Poses
/// @note  this function exists without transferring dihedrals and coordinates because Pose::set_coords
/// @note  does internal recalculations to dihedrals and requires all coordinates to exist
inline
void
transfer_identity_info_between_Poses(
	Pose const & p1,
	utility::vector1< bool > const & keep,
	Pose & p2,
	int const & p2_begin,
	bool const & keep_pdb_info = false
)
{
	for ( Integer residue = 1, last_residue = p1.total_residue(), p2_count = p2_begin; residue <= last_residue; ++residue ) {
		if ( keep[ residue ] ) {

			p2.set_res        ( p2_count, p1.res( residue ) );
			p2.set_res_variant( p2_count, p1.res_variant( residue ));
			p2.set_secstruct  ( p2_count, p1.secstruct( residue ) );
			p2.set_name       ( p2_count, p1.name( residue ) );

			++p2_count;
		}
	}

	if ( keep_pdb_info ) { // outside prior loop for speed
		for ( Integer residue = 1, last_residue = p1.total_residue(), p2_count = p2_begin; residue <= last_residue; ++residue ) {
			if ( keep[ residue ] ) {
				p2.pdb_info().set_pdb_chain( p2_count, p1.pdb_info().res_chain( residue ) );
				p2.pdb_info().set_pdb_res( p2_count, p1.pdb_info().pdb_res_num( residue ) );
				p2.pdb_info().set_pdb_insert_let( p2_count, p1.pdb_info().pdb_insert_let( residue ) );

				++p2_count;
			}
		}
	}
}


/// @brief fill full_coord from Pose, 'index' specifies the first residue in the full_coord
/// @brief to be filled and is required!  index is modified so that it ends up at the _next_
/// @brief residue to be filled
inline
void
fill_full_coord_from_Pose(
	Pose const & p,
	ResidueRange const & range,
	FArray3D_float & full_coord,
	int & index
)
{
	FArray3D_float const & p_full_coord = p.full_coord();

	for ( Integer residue = range.begin(), last_residue = range.end(); residue <= last_residue; ++residue, ++index ) {
	    int const aa( p.res( residue ) );
		int const aav( p.res_variant( residue ) );
		for ( Integer atom = 1, last_atom = aaproperties_pack::natoms( aa, aav ); atom <= last_atom; ++atom ) {
			for ( Integer i = 1; i <= 3; ++i ) {
				full_coord( i, atom, index ) = p_full_coord( i, atom, residue );
			}
		}
	}
}


/// @brief fill full_coord from Pose, 'index' specifies the first residue in the full_coord
/// @brief to be filled and is required!  index is modified so that it ends up at the _next_
/// @brief residue to be filled
inline
void
fill_full_coord_from_Pose(
	Pose const & p,
	utility::vector1< bool > const & keep,
	FArray3D_float & full_coord,
	int & index
)
{
	FArray3D_float const & p_full_coord = p.full_coord();

	for ( Integer residue = 1, last_residue = p.total_residue(); residue <= last_residue; ++residue ) {
		if ( keep[ residue ] ) {
		    int const aa( p.res( residue ) );
			int const aav( p.res_variant( residue ) );
			for ( Integer atom = 1, last_atom = aaproperties_pack::natoms( aa, aav ); atom <= last_atom; ++atom ) {
				for ( Integer i = 1; i <= 3; ++i ) {
					full_coord( i, atom, index ) = p_full_coord( i, atom, residue );
				}
			}

			++index;
		}
	}
}


/// @brief make Eposition from full_coord
/// @note  in Rosetta full_coord is sometimes dimensioned with MAX_RES... which means we
/// @note  have to pass "nres" and can't use FArray's internal size function because
/// @note  the two might not be in sync
inline
void
full_coord_to_Eposition(
	int const & nres,
	FArray3D_float & full_coord,
	FArray3D_float & Eposition
)
{
	Eposition.dimension( 3, param::MAX_POS, nres );

	for ( Integer residue = 1; residue <= nres; ++residue ) {
		for ( Integer i = 1; i <= 3; ++ i ) {
				Eposition(i, 1, residue) = full_coord(i, 1, residue); // N
				Eposition(i, 2, residue) = full_coord(i, 2, residue); // CA
				Eposition(i, 4, residue) = full_coord(i, 3, residue); // C
				Eposition(i, 5, residue) = full_coord(i, 4, residue); // O
				Eposition(i, 3, residue) = full_coord(i, 5, residue); // CB
		}
	}
}


/// @brief replace a segment in a Pose with an external segment
inline
void
replace_segment_in_Pose(
	Pose const & protein,
	ResidueRange const & replacement_range,
	Pose const & segment,
	Pose & altered_protein
)
{
	// new total residue = p - removed + segment
	Integer total_residue = protein.total_residue() - replacement_range.length() + segment.total_residue();

	prepare_pose( altered_protein, total_residue );

	FArray3D_float full_coord( 3, param::MAX_ATOM(), total_residue );

	// keep a counter, tracks where the next residue will be placed (equivalent to number of residues processed + 1)
	Integer counter = 1;

	// first, transfer up to residues before the break
	ResidueRange range_before_break = ResidueRange( 1, replacement_range.begin() - 1 );
	transfer_identity_info_between_Poses( protein, range_before_break, altered_protein, counter );
	fill_full_coord_from_Pose( protein, range_before_break, full_coord, counter );

	assert( counter == replacement_range.begin() ); // check counter

	// second, transfer the entire segment (segment endpoints replace residue before and after the break)
	ResidueRange segment_range = ResidueRange( 1, segment.total_residue() );
	transfer_identity_info_between_Poses( segment, segment_range, altered_protein, counter );
	fill_full_coord_from_Pose( segment, segment_range, full_coord, counter );

	assert( counter == replacement_range.begin() + segment.total_residue() ); // check counter

	// third, transfer everything else after the break
	ResidueRange range_after_break = ResidueRange( replacement_range.end() + 1, protein.total_residue() );
	transfer_identity_info_between_Poses( protein, range_after_break, altered_protein, counter );
	fill_full_coord_from_Pose( protein, range_after_break, full_coord, counter );

	assert( counter == total_residue + 1 ); // check counter

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, total_residue );
	full_coord_to_Eposition( total_residue, full_coord, Eposition );

	altered_protein.set_coords( false, Eposition, full_coord, false ); // internally recomputes all angles

	// save pdb information
	char chain_id = protein.pdb_info().res_chain( 1 ); // assumes only one chain present
	for (int residue = 1, last_residue = altered_protein.total_residue(); residue <= last_residue; ++residue ) {
		// chain id
		altered_protein.pdb_info().set_pdb_chain( residue, chain_id );

		// pdb residue numbering
		altered_protein.pdb_info().set_pdb_res( residue, residue );

		// insertion code (none)
		altered_protein.pdb_info().set_pdb_insert_let( residue, ' ' );
	}
}


/// @brief insert a segment into a Pose
inline
void
insert_segment_into_Pose(
	Pose const & protein,
	Integer const & residue_before_break,
	Pose const & segment,
	Pose & altered_protein
)
{
	// new total residue = p + segment
	Integer total_residue = protein.total_residue() + segment.total_residue();

	prepare_pose( altered_protein, total_residue );

	FArray3D_float full_coord( 3, param::MAX_ATOM(), total_residue );

	// keep a counter, tracks where the next residue will be placed (equivalent to number of residues processed + 1)
	Integer counter = 1;

	// first, transfer up to residues before the break
	ResidueRange range_before_break = ResidueRange( 1, residue_before_break );
	transfer_identity_info_between_Poses( protein, range_before_break, altered_protein, counter );
	fill_full_coord_from_Pose( protein, range_before_break, full_coord, counter );

	assert( counter == residue_before_break + 1 ); // check counter

	// second, transfer the entire segment (segment endpoints replace residue before and after the break)
	ResidueRange segment_range = ResidueRange( 1, segment.total_residue() );
	transfer_identity_info_between_Poses( segment, segment_range, altered_protein, counter );
	fill_full_coord_from_Pose( segment, segment_range, full_coord, counter );

	assert( counter == residue_before_break + segment.total_residue() + 1 ); // check counter

	// third, transfer everything else after the break
	ResidueRange range_after_break = ResidueRange( residue_before_break + 1, protein.total_residue() );
	transfer_identity_info_between_Poses( protein, range_after_break, altered_protein, counter );
	fill_full_coord_from_Pose( protein, range_after_break, full_coord, counter );

	assert( counter == total_residue + 1 ); // check counter

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, total_residue );
	full_coord_to_Eposition( total_residue, full_coord, Eposition );

	altered_protein.set_coords( false, Eposition, full_coord, false );

	// save pdb information
	char chain_id = protein.pdb_info().res_chain( 1 ); // assumes only one chain present
	for (int residue = 1, last_residue = altered_protein.total_residue(); residue <= last_residue; ++residue ) {
		// chain id
		altered_protein.pdb_info().set_pdb_chain( residue, chain_id );

		// pdb residue numbering
		altered_protein.pdb_info().set_pdb_res( residue, residue );

		// insertion code (none)
		altered_protein.pdb_info().set_pdb_insert_let( residue, ' ' );
	}
}


/// @brief remove a segment from Pose -- gaps the original Pose, including removal of endpoint residues
// TODO: optional, copy as much as possible using 0-based indexing to speed up construction
//       through skipping embedded 'if' statements and maintaining contiguous memory
//       access; this might screw up if we're trying to handle symmetry routines though...
inline
void
remove_segment_from_Pose(
	Pose & p,
	ResidueRange const & rr,
	Pose & gapped,
	bool const & keep_pdb_info = false
)
{
	prepare_pose( gapped, p.total_residue() - rr.length() );

	Integer counter = 1; // new residue counter

	FArray3D_float full_coord( 3, param::MAX_ATOM(), gapped.total_residue() );

	// do first range
	if ( rr.begin() > 1 ) {
		ResidueRange first_range = ResidueRange( 1, rr.begin() - 1 );

		// transfer non-coordinate info
		transfer_identity_info_between_Poses( p, first_range, gapped, counter, keep_pdb_info );

		// make full_coord
		fill_full_coord_from_Pose( p, first_range, full_coord, counter ); // counter updated here
	}

	// do second range
	if ( rr.end() < p.total_residue() ) {
		ResidueRange second_range = ResidueRange( rr.end() + 1, p.total_residue() );

		// transfer non-coordinate info
		transfer_identity_info_between_Poses( p, second_range, gapped, counter, keep_pdb_info );

		// make full_coord
		fill_full_coord_from_Pose( p, second_range, full_coord, counter ); // counter updated here
	}

	assert( counter - 1 == gapped.total_residue() ) ;

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, gapped.total_residue() );
	full_coord_to_Eposition( gapped.total_residue(), full_coord, Eposition );

	gapped.set_coords( false, Eposition, full_coord, false ); // internally recomputes all angles
}


/// @brief extract a segment from Pose -- trims the original Pose, includes endpoint residues
// TODO: optional, copy as much as possible using 0-based indexing to speed up construction
//       through skipping embedded 'if' statements and maintaining contiguous memory
//       access; this might screw up if we're trying to handle symmetry routines though...
inline
void
extract_segment_from_Pose(
	Pose const & p,
	ResidueRange const & rr,
	Pose & extracted,
	bool const & keep_pdb_info = false
)
{
	prepare_pose( extracted, rr.length() );

	// transfer non-coordinate info
	transfer_identity_info_between_Poses( p, rr, extracted, 1, keep_pdb_info );

	// make full_coord
	int counter = 1;
	FArray3D_float full_coord( 3, param::MAX_ATOM(), extracted.total_residue() );
	fill_full_coord_from_Pose( p, rr, full_coord, counter );

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, extracted.total_residue() );
	full_coord_to_Eposition( extracted.total_residue(), full_coord, Eposition );

	extracted.set_coords( false, Eposition, full_coord, false ); // internally recomputes all angles
}


/// @brief extract a set of residues from a Pose
/// @note  all residues set to 'true' in keep are kept, while all others are removed
inline
void
fragment_Pose(
	Pose const & p,
	utility::vector1< bool > const & keep,
	Pose & fragmented,
	bool const & keep_pdb_info = false
)
{
	// count number of residue
	Integer total_residue = 0;
	for ( Integer residue = 1, last_residue = keep.size(); residue <= last_residue; ++residue ) {
		if ( keep[ residue ] ) {
			total_residue++;
		}
	}

	prepare_pose( fragmented, total_residue );

	// transfer non-coordinate info
	transfer_identity_info_between_Poses( p, keep, fragmented, 1, keep_pdb_info );

	// make full_coord
	int counter = 1;
	FArray3D_float full_coord( 3, param::MAX_ATOM(), fragmented.total_residue() );
	fill_full_coord_from_Pose( p, keep, full_coord, counter );

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, fragmented.total_residue() );
	full_coord_to_Eposition( fragmented.total_residue(), full_coord, Eposition );

	fragmented.set_coords( false, Eposition, full_coord, false ); // internally recomputes all angles
}


/// @brief extract backbone coordinates for one residues from full_coord
/// @note  in Pose full_coord, backbone atoms N, CA, C, O are indices 1 to 4 in order
/// @note  output FArray will be dimensioned to 3x4!
template< typename T >
inline
void
extract_1bb_from_full_coord(
	FArray3D< T > const & p_full_coord,
	Integer const & residue,
	FArray2D< T > & crd
)
{
	crd.dimension( 3, 4 ); // 1st dimension: x,y,z | 2nd dimension: N, CA, C, O

	// 0-based linear indexing for speed
	Integer crd_idx = 0;

	// grab first residue N, CA, C, O
	Integer fc_idx = p_full_coord.index( 1, 1, residue );
	for ( Integer j = 1; j <= 4; ++j ) {
		for ( Integer i = 1; i <= 3; ++i, ++crd_idx, ++fc_idx ) {
			crd[ crd_idx ] = p_full_coord[ fc_idx ];
		}
	}

	assert( crd_idx == 12 ); // 12 array elements {0 .. 11} should have been filled
}


/// @brief extract backbone coordinates for one residues from Pose
/// @note  in Pose full_coord, backbone atoms N, CA, C, O are indices 1 to 4 in order
/// @note  output FArray will be dimensioned to 3x4!
template< typename T >
inline
void
extract_1bb_from_Pose(
	Pose const & p,
	Integer const & residue,
	FArray2D< T > & crd
)
{
	extract_1bb_from_full_coord( p.full_coord(), residue, crd );
}


/// @brief extract backbone coordinates for two residues from full_coord
/// @note  in Pose full_coord, backbone atoms N, CA, C, O are indices 1 to 4 in order
/// @note  output FArray will be dimensioned to 3x8!
template< typename T >
inline
void
extract_2bb_from_full_coord(
	FArray3D_float const & p_full_coord,
	ResidueRange const & rr,
	FArray2D< T > & crd
)
{
	crd.dimension( 3, 8 ); // 1st dimension: x,y,z | 2nd dimension: N, CA, C, O

	// 0-based linear indexing for speed
	Integer crd_idx = 0;

	// grab first residue N, CA, C, O
	Integer fc_idx = p_full_coord.index( 1, 1, rr.begin() );
	for ( Integer j = 1; j <= 4; ++j ) {
		for ( Integer i = 1; i <= 3; ++i, ++crd_idx, ++fc_idx ) {
			crd[ crd_idx ] = p_full_coord[ fc_idx ];
		}
	}

	// grab second residue N, CA, C, O
	fc_idx = p_full_coord.index( 1, 1, rr.end() ); // 0-based linear indexing for speed
	for ( Integer j = 1; j <= 4; ++j ) {
		for ( Integer i = 1; i <= 3; ++i, ++crd_idx, ++fc_idx ) {
			crd[ crd_idx ] = p_full_coord[ fc_idx ];
		}
	}

	assert( crd_idx == 24 ); // 24 array elements {0 .. 23} should have been filled
}


/// @brief extract backbone coordinates for two residues from Pose
/// @note  in Pose full_coord, backbone atoms N, CA, C, O are indices 1 to 4 in order
/// @note  output FArray will be dimensioned to 3x8!
template< typename T >
inline
void
extract_2bb_from_Pose(
	Pose const & p,
	ResidueRange const & rr,
	FArray2D< T > & crd
)
{
	extract_2bb_from_full_coord( p.full_coord(), rr, crd );
}


/// @brief extract backbone coordinates for residues in given range from Pose, meant for use with
/// @brief superimpose()
/// @note  in Pose full_coord, backbone atoms N, CA, C, O are indices 1 to 4 in order
/// @note  output FArray will be dimensioned to 3x(4*rr.length()) !
template< typename T >
inline
void
extract_bb_from_Pose(
	Pose const & p,
	ResidueRange const & rr,
	FArray2D< T > & crd
)
{
	crd.dimension( 3, 4 * rr.length() ); // 1st dimension: x,y,z | 2nd dimension: N, CA, C, O

	FArray3D_float const & p_full_coord = p.full_coord();

	// 0-based linear indexing for speed
	Integer crd_idx = 0;

	// grab N, CA, C, O from residues
	for ( Integer residue = rr.begin(), last_residue = rr.end(); residue <= last_residue; ++residue ) {
		Integer fc_idx = p_full_coord.index( 1, 1, residue );
		for ( Integer j = 1; j <= 4; ++j ) {
			for ( Integer i = 1; i <= 3; ++i, ++crd_idx, ++fc_idx ) {
				crd[ crd_idx ] = p_full_coord[ fc_idx ];
			}
		}
	}

	assert( crd_idx == 3 * 4 * rr.length() ); // number of array elements that should have been filled
}


/// @brief extract backbone coordinates for specified residues from Pose, meant for use with
/// @brief superimpose()
/// @note  in Pose full_coord, backbone atoms N, CA, C, O are indices 1 to 4 in order
/// @note  output FArray will be dimensioned to 3x(4*container.size()) !
/// @note  backbone coordinates will be loaded in the order given by the container's iterator!
template< typename Container, typename T >
inline
void
extract_bb_from_Pose(
	Pose const & p,
	Container const & container,
	FArray2D< T > & crd
)
{
	crd.dimension( 3, 4 * container.size() ); // 1st dimension: x,y,z | 2nd dimension: N, CA, C, O

	FArray3D_float const & p_full_coord = p.full_coord();

	// 0-based linear indexing for speed
	Integer crd_idx = 0;

	// grab N, CA, C, O from residues
	for ( typename Container::const_iterator ri = container.begin(), rie = container.end(); ri !=rie; ++ri ) {
		Integer fc_idx = p_full_coord.index( 1, 1, *ri );
		for ( Integer j = 1; j <= 4; ++j ) {
			for ( Integer i = 1; i <= 3; ++i, ++crd_idx, ++fc_idx ) {
				crd[ crd_idx ] = p_full_coord[ fc_idx ];
			}
		}
	}

	assert( (Size)crd_idx == 3 * 4 * container.size() ); // number of array elements that should have been filled
}


/// @brief cheat for clarity on conversion to gly
inline
void
convert_to_gly(
	Pose & p,
	ResidueRange const & rr
)
{
	convert_resnum_range_to_gly( p, rr.left(), rr.right() );
}


/// @brief cheat for clarity on conversion to gly
inline
void
convert_to_gly(
	Pose & p
)
{
	convert_resnum_range_to_gly( p, 1, p.total_residue() );
}


/// @brief superposition of two sets of coordinates, meant for backbone (N, CA, C, O) superposition
/// @note  x2 moves onto x1
inline
void
superimpose(
	FArray2D_double const & x1,
	FArray2D_double const & x2,
	FArray2D_double & x1_oriented,
	FArray2D_double & x2_oriented,
	double & rms_error
)
{
	assert( x1.size2() == x2.size2() ); // check second dimension of arrays, must be same size

	// dimension output arrays
	x1_oriented.dimension( x1.size1(), x1.size2() );
	x2_oriented.dimension( x2.size1(), x2.size2() );

	clear_rms(); // clear globals in rms.h/.cc routines

	// add "cross-moments" for each pair of coordinates (see rms.h/.cc)
	for ( int i = 1, ie = x1.size2(); i <= ie; ++i ) {
		add_rms_asym( i, i, x1, x2 );
	}

	// do alignment
	rmsfitca3_asym( x1.size2(), x2.size2(), x1, x1_oriented, x2, x2_oriented, rms_error, true ); // boolean: retain center of mass (at x1)
}


/// @brief orient full_coord array by transformation matrix via "GL routines" (see rosetta++/refold.cc)
/// @note  no dimensioning call for speed -- make sure out_full_coord is dimensioned properly!
inline
void
transform_full_coord_via_GL(
	FArray2D_float const & RT,
	Pose const & p,
	FArray3D_float & out_full_coord
)
{
	FArray3D_float const & p_full_coord = p.full_coord();

	for ( Integer residue = 1, last_residue = p.total_residue(); residue <= last_residue; ++residue ) {
		Integer const aa = p.res( residue );
		Integer const aav = p.res_variant( residue );
		for ( Integer atom = 1, last_atom = aaproperties_pack::natoms( aa, aav ); atom <= last_atom; ++atom ) {
			GL_rot( RT, p_full_coord( 1, atom, residue ), out_full_coord( 1, atom, residue ) );
		}
	}
}


/// @brief orient coordinates in place by transformation matrix via "GL routines" (see rosetta++/refold.cc)
inline
void
transform_crd_via_GL(
	FArray2D_float const & RT,
	FArray2D_float & crd
)
{
	assert( crd.size1() == 3 ); // check that we're dealing with x,y,z coordinates

	for ( Integer j = 1, je = crd.size2(); j <= je; ++j ) {
		GL_rot_in_place( RT, crd( 1, j ) );
	}
}


/// @brief orient coordinates by transformation matrix via "GL routines" (see rosetta++/refold.cc)
inline
void
transform_crd_via_GL(
	FArray2D_float const & RT,
	FArray2D_float const & in_crd,
	FArray2D_float & out_crd
)
{
	assert( in_crd.size1() == 3 ); // check that we're dealing with x,y,z coordinates

	for ( Integer j = 1, je = in_crd.size2(); j <= je; ++j ) {
		GL_rot( RT, in_crd( 1, j ), out_crd( 1, j ) );
	}
}


/// @brief compose two transformation ("GL") matrices, matrices must all be 4x4!
/// @note  resulting matrix is the same as if 'f' is applied first and 's' afterwards
template< typename T >
inline
void
compose_GL(
	FArray2D< T > const & s,
	FArray2D< T > const & f,
	FArray2D< T > & result
)
{
	// matrix multiply 's' * 'f' to get result
	// remember that FArrays are column-major:
	// | 0 4  8 12 |
	// | 1 5  9 13 |
	// | 2 6 10 14 |
	// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 | so simplifications are performed below
	result[ 0 ]  = s[ 0 ]*f[ 0 ]  + s[ 4 ]*f[ 1 ]  + s[ 8 ] *f[ 2 ]; //  + s[ 12 ]*f[ 3 ];
	result[ 1 ]  = s[ 1 ]*f[ 0 ]  + s[ 5 ]*f[ 1 ]  + s[ 9 ] *f[ 2 ]; //  + s[ 13 ]*f[ 3 ];
	result[ 2 ]  = s[ 2 ]*f[ 0 ]  + s[ 6 ]*f[ 1 ]  + s[ 10 ]*f[ 2 ]; //  + s[ 14 ]*f[ 3 ];
//	result[ 3 ]  = s[ 3 ]*f[ 0 ]  + s[ 7 ]*f[ 1 ]  + s[ 11 ]*f[ 2 ]  + s[ 15 ]*f[ 3 ]; // always 0
	result[ 3 ]  = 0.0;
	result[ 4 ]  = s[ 0 ]*f[ 4 ]  + s[ 4 ]*f[ 5 ]  + s[ 8 ] *f[ 6 ]; //  + s[ 12 ]*f[ 7 ];
	result[ 5 ]  = s[ 1 ]*f[ 4 ]  + s[ 5 ]*f[ 5 ]  + s[ 9 ] *f[ 6 ]; //  + s[ 13 ]*f[ 7 ];
	result[ 6 ]  = s[ 2 ]*f[ 4 ]  + s[ 6 ]*f[ 5 ]  + s[ 10 ]*f[ 6 ]; //  + s[ 14 ]*f[ 7 ];
//	result[ 7 ]  = s[ 3 ]*f[ 4 ]  + s[ 7 ]*f[ 5 ]  + s[ 11 ]*f[ 6 ]  + s[ 15 ]*f[ 7 ]; // always 0
	result[ 7 ]  = 0.0;
	result[ 8 ]  = s[ 0 ]*f[ 8 ]  + s[ 4 ]*f[ 9 ]  + s[ 8 ] *f[ 10 ]; // + s[ 12 ]*f[ 11 ];
	result[ 9 ]  = s[ 1 ]*f[ 8 ]  + s[ 5 ]*f[ 9 ]  + s[ 9 ] *f[ 10 ]; // + s[ 13 ]*f[ 11 ];
	result[ 10 ] = s[ 2 ]*f[ 8 ]  + s[ 6 ]*f[ 9 ]  + s[ 10 ]*f[ 10 ]; // + s[ 14 ]*f[ 11 ];
//	result[ 11 ] = s[ 3 ]*f[ 8 ]  + s[ 7 ]*f[ 9 ]  + s[ 11 ]*f[ 10 ] + s[ 15 ]*f[ 11 ]; // always 0
	result[ 11 ]  = 0.0;
	result[ 12 ] = s[ 0 ]*f[ 12 ] + s[ 4 ]*f[ 13 ] + s[ 8 ] *f[ 14 ] + s[ 12 ]; // *f[ 15 ];
	result[ 13 ] = s[ 1 ]*f[ 12 ] + s[ 5 ]*f[ 13 ] + s[ 9 ] *f[ 14 ] + s[ 13 ]; // *f[ 15 ];
	result[ 14 ] = s[ 2 ]*f[ 12 ] + s[ 6 ]*f[ 13 ] + s[ 10 ]*f[ 14 ] + s[ 14 ]; // *f[ 15 ];
//	result[ 15 ] = s[ 3 ]*f[ 12 ] + s[ 7 ]*f[ 13 ] + s[ 11 ]*f[ 14 ] + s[ 15 ]*f[ 15 ]; // always 1
	result[ 15 ] = 1.0;

}


/// @brief get GL matrix from axis-angle pair, angle must be in degrees, and FArray2D must be 4x4!
/// @note  Rodrigues' formula
template< typename T >
inline
void
GL_from_axis_angle_degrees(
	numeric::xyzVector< T > const & axis,
	T const & angle,
	FArray2D< T > & RT
)
{
	using std::cos;
	using std::sin;

	assert( RT.size1() == 4 && RT.size2() == 4 );

	numeric::xyzVector< T > v( axis );
	v.normalize(); // force this, but if you know the input vectors will be normalized, then turn it off

	T const theta( numeric::conversions::radians( angle ) );

	// remember that FArrays are column-major:
	// | 0 4  8 12 |
	// | 1 5  9 13 |
	// | 2 6 10 14 |
	// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 |
	RT[ 0 ]  = cos( theta ) + v.x() * v.x() * ( 1 - cos( theta ) );
	RT[ 1 ]  = v.z() * sin( theta ) + v.x() * v.y() * ( 1 - cos( theta ) );
	RT[ 2 ]  = -v.y() * sin( theta ) + v.x() * v.z() * ( 1 - cos( theta ) );
	RT[ 3 ]  = 0.0;
	RT[ 4 ]  = v.x() * v.y() * ( 1 - cos( theta ) ) - v.z() * sin( theta );
	RT[ 5 ]  = cos( theta ) + v.y() * v.y() * ( 1 - cos( theta ) );
	RT[ 6 ]  = v.x() * sin( theta ) + v.y() * v.z() * ( 1 - cos( theta ) );
	RT[ 7 ]  = 0.0;
	RT[ 8 ]  = v.y() * sin( theta ) + v.x() * v.z() * ( 1 - cos( theta ) );
	RT[ 9 ]  = -v.x() * sin( theta ) + v.y() * v.z() * ( 1 - cos( theta ) );
	RT[ 10 ] = cos( theta ) + v.z() * v.z() * ( 1 - cos( theta ) );
	RT[ 11 ] = 0.0;
	RT[ 12 ] = 0.0;
	RT[ 13 ] = 0.0;
	RT[ 14 ] = 0.0;
	RT[ 15 ] = 1.0;
}


/// @brief get GL matrix from axis-angle pair, angle must be in radians, and FArray2D must be 4x4!
/// @note  Rodrigues' formula
template< typename T >
inline
void
GL_from_axis_angle_radians(
	numeric::xyzVector< T > const & axis,
	T const & angle,
	FArray2D< T > & RT
)
{
	using std::cos;
	using std::sin;

	assert( RT.size1() == 4 && RT.size2() == 4 );

	numeric::xyzVector< T > v( axis );
	v.normalize(); // force this, but if you know the input vectors will be normalized, then turn it off

	// remember that FArrays are column-major:
	// | 0 4  8 12 |
	// | 1 5  9 13 |
	// | 2 6 10 14 |
	// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 |
	RT[ 0 ]  = cos( angle ) + v.x() * v.x() * ( 1 - cos( angle ) );
	RT[ 1 ]  = v.z() * sin( angle ) + v.x() * v.y() * ( 1 - cos( angle ) );
	RT[ 2 ]  = -v.y() * sin( angle ) + v.x() * v.z() * ( 1 - cos( angle ) );
	RT[ 3 ]  = 0.0;
	RT[ 4 ]  = v.x() * v.y() * ( 1 - cos( angle ) ) - v.z() * sin( angle );
	RT[ 5 ]  = cos( angle ) + v.y() * v.y() * ( 1 - cos( angle ) );
	RT[ 6 ]  = v.x() * sin( angle ) + v.y() * v.z() * ( 1 - cos( angle ) );
	RT[ 7 ]  = 0.0;
	RT[ 8 ]  = v.y() * sin( angle ) + v.x() * v.z() * ( 1 - cos( angle ) );
	RT[ 9 ]  = -v.x() * sin( angle ) + v.y() * v.z() * ( 1 - cos( angle ) );
	RT[ 10 ] = cos( angle ) + v.z() * v.z() * ( 1 - cos( angle ) );
	RT[ 11 ] = 0.0;
	RT[ 12 ] = 0.0;
	RT[ 13 ] = 0.0;
	RT[ 14 ] = 0.0;
	RT[ 15 ] = 1.0;
}


/// @brief 5-point chainbreak rms, FArray2D is 3x5 typically representing 4 bb atoms for a residue and an additional 5th atom
template< typename T >
inline
T
five_point_chainbreak(
	FArray2D< T > const & a,
	FArray2D< T > const & b
)
{
	T rms = 0.0;

	for ( Integer i = 0; i < 15; ++i ) {
		rms += square( a[ i ] - b[ i ] );
	}

	rms = std::sqrt( rms / 5 );

	return rms;
}


/// @brief constructs bounding box around a Pose given range
inline
BoundingBox
bounding_box(
	Pose const & p,
	ResidueRange const & range
)
{
	FArray3D_float const & fc = p.full_coord();

	BoundingBox bb( PointPosition( fc( 1, 1, range.begin() ), fc( 2, 1, range.begin() ), fc( 3, 1, range.begin() ) ) );

	for ( int res = range.begin(), nres = range.end(); res <= nres; ++res ) {
		int const aa( p.res( res ) );
		int const aav( p.res_variant( res ) );
		for ( int atom = 1; atom <= aaproperties_pack::natoms( aa, aav ); ++atom ) {
			bb.add( PointPosition( fc( 1, atom, res ), fc( 2, atom, res ), fc( 3, atom, res ) ) );
		}
	}

	return bb;
}


/// @brief constructs bounding box around a Pose for the indicated residues
/// @note template BooleanType requires operator []
template< typename BooleanType >
inline
BoundingBox
bounding_box(
	Pose const & p,
	BooleanType const & check_residue
)
{
	FArray3D_float const & fc = p.full_coord();

	// find first residue to be put into bounding box
	Integer first_residue = 0;
	for ( Size i = 1, ie = p.total_residue(); i <= ie; ++i ) {
		if ( check_residue[ i ] ) {
			first_residue = i;
			break;
		}
	}

	assert( first_residue > 0 );

	BoundingBox bb( PointPosition( fc( 1, 1, first_residue ), fc( 2, 1, first_residue ), fc( 3, 1, first_residue ) ) );

	for ( int res = 1, nres = p.total_residue(); res <= nres; ++res ) {
		if ( check_residue[ res ] ) {
			int const aa( p.res( res ) );
			int const aav( p.res_variant( res ) );
			for ( int atom = 1; atom <= aaproperties_pack::natoms( aa, aav ); ++atom ) {
				bb.add( PointPosition( fc( 1, atom, res ), fc( 2, atom, res ), fc( 3, atom, res ) ) );
			}
		}
	}

	return bb;
}


/// @brief constructs bounding box around a Pose given range
inline
BoundingBox
bounding_box(
	Pose const & p
)
{
	return bounding_box( p, ResidueRange( 1, p.total_residue() ) ) ;
}


/// @brief fills in octree with atoms from a Pose
inline
void
fill_octree(
	Octree< AtomPoint > & oc,
	Pose const & p,
	ResidueRange const & range
)
{
	FArray3D_float const & fc = p.full_coord();

	for ( Integer res = range.begin(), nres = range.end(); res <= nres; ++res ) {
		Integer const aa( p.res( res ) );
		Integer const aav( p.res_variant( res ) );
		for ( Integer atom = 1; atom <= aaproperties_pack::natoms( aa, aav ); ++atom ) {
			oc.add( AtomPoint( res, atom, fc( 1, atom, res ), fc( 2, atom, res ), fc( 3, atom, res ) ) );
		}
	}
}

/// @brief fills in octree with atoms from a Pose for the indicated residues
/// @note template BooleanType requires operator []
template< typename BooleanType >
inline
void
fill_octree(
	Octree< AtomPoint > & oc,
	Pose const & p,
	BooleanType const & check_residue
)
{
	FArray3D_float const & fc = p.full_coord();

	for ( Integer res = 1, nres = p.total_residue(); res <= nres; ++res ) {
		if ( check_residue[ res ] ) {
			Integer const aa( p.res( res ) );
			Integer const aav( p.res_variant( res ) );
			for ( Integer atom = 1; atom <= aaproperties_pack::natoms( aa, aav ); ++atom ) {
				oc.add( AtomPoint( res, atom, fc( 1, atom, res ), fc( 2, atom, res ), fc( 3, atom, res ) ) );
			}
		}
	}
}



/// @brief fills in octree with atoms from a Pose given a range
inline
void
fill_octree(
	Octree< AtomPoint > & oc,
	Pose const & p
)
{
	fill_octree( oc, p, ResidueRange( 1, p.total_residue() ) );
}


/// @brief fills in octree with CA atoms from a Pose
/// @note  there are too many type dependencies in Rosetta atom types, so for now we split out
/// @note  this routine rather than e.g. use a map to have a more generalized routine
inline
void
fill_octree_CA(
	Octree< AtomPoint > & oc,
	Pose const & p
)
{
	FArray3D_float const & fc = p.full_coord();

	for ( Integer res = 1, nres = p.total_residue(); res <= nres; ++res ) {
		oc.add( AtomPoint( res, 2, fc( 1, 2, res ), fc( 2, 2, res ), fc( 3, 2, res ) ) ); // 'CA'
	}
}


/// @brief fills in octree with CB atoms from a Pose
/// @note  there are too many type dependencies in Rosetta atom types, so for now we split out
/// @note  this routine rather than e.g. use a map to have a more generalized routine
/// @warning Does not check for gly!  This function is meant to be used with all-ala scaffolds.
inline
void
fill_octree_CB(
	Octree< AtomPoint > & oc,
	Pose const & p
)
{
	FArray3D_float const & fc = p.full_coord();

	for ( Integer res = 1, nres = p.total_residue(); res <= nres; ++res ) {
		oc.add( AtomPoint( res, 2, fc( 1, 5, res ), fc( 2, 5, res ), fc( 3, 5, res ) ) ); // 'CB'
	}
}


/// @brief queries octree using a Pose and counts number of bumps within the given distance
inline
Integer
count_bumps(
	Octree< AtomPoint > const & oc,
	Pose const & p,
	Distance const cutoff
)
{
	std::set< AtomPoint > bumping;

	FArray3D_float const & fc = p.full_coord();

	for ( int res = 1, nres = p.total_residue(); res <= nres; ++res ) {
		int const aa( p.res( res ) );
		int const aav( p.res_variant( res ) );
		for ( int atom = 1; atom <= aaproperties_pack::natoms( aa, aav ); ++atom ) {

			std::vector< AtomPoint > nn = oc.near_neighbors( cutoff, (Real)fc( 1, atom, res ), (Real)fc( 2, atom, res ), (Real)fc( 3, atom, res ) );
			bumping.insert( nn.begin(), nn.end() );

		}
	}

	return bumping.size();
}


/// @brief calculate lj-repulsive between two atoms
/// @note  code clipped from pairenergy.h
inline
Real
lj_rep(
	Integer const & atom_type1,
	Integer const & atom_type2,
	Real const & d2
)
{
	if ( d2 >= pdbstatistics_pack::safe_max_dis2 || d2 == 0 ) { // no explosions ;)
		return 0.0;
	}

	// bins and interpolation fraction
	Real const d2_bin = d2 * pdbstatistics_pack::fa_bins_per_A2;
	Integer const disbin1 = static_cast< int >( d2_bin ) + 1;
	Real const frac = d2_bin - ( disbin1 - 1 );

	Integer const l1 = pdbstatistics_pack::pCurrentEtable->ljatr.index( disbin1, atom_type1, atom_type2 );
	Integer const l2 = l1 + 1;
	Real const e1 = pdbstatistics_pack::pCurrentEtable->ljrep[ l1 ];
	return e1 + frac * ( pdbstatistics_pack::pCurrentEtable->ljrep[ l2 ] - e1 ); // repulsive
}


/// @brief fast fullatom lj-repulsive using octree, takes in full_coord array as input --
/// @brief appropriate for when coordinates are changing quickly and Pose should not be
/// @brief reconstructed each time
/// @note  If exclude_terminal_bb is true, all N and C terminus backbone atoms are
/// @note  excluded from the calculation (but sidechain is kept).
inline
Real
octree_inter_rep(
	Octree< AtomPoint > const & oc,
	Pose const & model,
	Pose const & query,
	FArray3D_float const & query_full_coord,
	bool const & exclude_terminal_bb = false
)
{
	Real repulsive = 0.0f;

	for ( Integer q_res = 1, q_nres = query.total_residue(); q_res <= q_nres; ++q_res ) {
		Integer const q_aa( query.res( q_res ) );
		Integer const q_aav( query.res_variant( q_res ) );
		for ( Integer q_atom = 1, q_atom_e = aaproperties_pack::natoms( q_aa, q_aav ); q_atom <= q_atom_e; ++q_atom ) {
			Integer query_atom_type = aaproperties_pack::fullatom_type( q_atom, q_aa, q_aav );

			Integer idx = query_full_coord.index( 1, q_atom, q_res );

			std::vector< AtomPoint > nn = oc.near_neighbors( 5.5, (Real)query_full_coord[ idx ], (Real)query_full_coord[ idx + 1 ], (Real)query_full_coord[ idx + 2 ] );

			for ( std::vector< AtomPoint >::const_iterator ia = nn.begin(), iae = nn.end(); ia != iae; ++ia ) {

				Integer const m_res = (*ia).residue_id();
				Integer const m_aa = model.res( m_res );
				Integer const m_aav = model.res_variant( m_res );
				Integer const m_atom = (*ia).atom_id();

				Integer model_atom_type = aaproperties_pack::fullatom_type( m_atom, m_aa, m_aav );

				Real rep = lj_rep( model_atom_type, query_atom_type, distance_squared( (*ia), numeric::xyzTriple< float >( &query_full_coord[ idx ] ) ) );

				repulsive += param_pack::pack_wts.Wrep() * rep;
			}
		}
	}

	// if excluding terminal backbone atoms, we subtract the repulsion from the backbone
	// heavy atoms
	if ( exclude_terminal_bb ) {
		static Integer r_idx[ 2 ] = { 1, 0 }; // n/c terminal residue id -- c filled in below
		static Integer bb_idx[ 6 ] = { 1, 2, 3, 4, 0, 0 }; // N,CA,C,O,HN,HA -- last two will be filled in below
		static Integer q_res = 0, q_atom = 0; // scratch

		// set c-terminal residue ids
		r_idx[ 1 ] = query.total_residue();

		for ( Integer r = 0; r < 2; ++r ) {
			// set terminal residue and fill in HN,HA indices
			q_res = r_idx[ r ];
			Integer const q_aa( query.res( q_res ) );
			Integer const q_aav( query.res_variant( q_res ) );
			bb_idx[ 4 ] = aaproperties_pack::HNpos( q_aa, q_aav );
			bb_idx[ 5 ] = aaproperties_pack::HApos( q_aa, q_aav );

			// add back terminus backbone repulsive
			for ( Integer i = 0; i < 6; ++i ) {
				q_atom = bb_idx[ i ];
				Integer query_atom_type = aaproperties_pack::fullatom_type( q_atom, q_aa, q_aav );

				Integer idx = query_full_coord.index( 1, q_atom, q_res );

				std::vector< AtomPoint > nn = oc.near_neighbors( 5.5, (Real)query_full_coord[ idx ], (Real)query_full_coord[ idx + 1 ], (Real)query_full_coord[ idx + 2 ] );

				for ( std::vector< AtomPoint >::const_iterator ia = nn.begin(), iae = nn.end(); ia != iae; ++ia ) {

					Integer const m_res = (*ia).residue_id();
					Integer const m_aa = model.res( m_res );
					Integer const m_aav = model.res_variant( m_res );
					Integer const m_atom = (*ia).atom_id();

					Integer model_atom_type = aaproperties_pack::fullatom_type( m_atom, m_aa, m_aav );

					Real rep = lj_rep( model_atom_type, query_atom_type, distance_squared( (*ia), numeric::xyzTriple< float >( &query_full_coord[ idx ] ) ) );

					repulsive -= param_pack::pack_wts.Wrep() * rep;
				}
			} // done adding back terminus backbone repulsive
		}
	}

	return repulsive;
}



/// @brief fast fullatom lj-repulsive using octree
/// @note  If exclude_terminal_atoms is true, then the the N at the
/// @note  N-terminus and the C at the C-terminus are not included
/// @note  in the calculation.  This scenario is appropriate for
/// @note  intra-clash check with no overlapping residues.
inline
Real
octree_inter_rep(
	Octree< AtomPoint > const & oc,
	Pose const & model,
	Pose const & query,
	bool const & exclude_terminal_atoms = false
)
{
	return octree_inter_rep( oc, model, query, query.full_coord(), exclude_terminal_atoms );
}


/// @brief fast fullatom lj-repulsive using octree, takes in full_coord array as input --
/// @brief appropriate for when coordinates are changing quickly and Pose should not be
/// @brief reconstructed each time
/// @note template BooleanType requires operator []
inline
Real
octree_inter_rep(
	Octree< AtomPoint > const & oc,
	Pose const & model,
	Pose const & query,
	FArray3D_float const & query_full_coord,
	ResidueRange const & query_residue_range
)
{
	Real repulsive = 0.0f;

	Integer q_res = query_residue_range.begin();
	Integer q_nres = query_residue_range.end();

	for ( ; q_res <= q_nres; ++q_res ) {
		Integer const q_aa( query.res( q_res ) );
		Integer const q_aav( query.res_variant( q_res ) );
		for ( Integer q_atom = 1, q_atom_e = aaproperties_pack::natoms( q_aa, q_aav ); q_atom <= q_atom_e; ++q_atom ) {
			Integer query_atom_type = aaproperties_pack::fullatom_type( q_atom, q_aa, q_aav );

			Integer idx = query_full_coord.index( 1, q_atom, q_res );

			std::vector< AtomPoint > nn = oc.near_neighbors( 5.5, (Real)query_full_coord[ idx ], (Real)query_full_coord[ idx + 1 ], (Real)query_full_coord[ idx + 2 ] );

			for ( std::vector< AtomPoint >::const_iterator ia = nn.begin(), iae = nn.end(); ia != iae; ++ia ) {

				Integer const m_res = (*ia).residue_id();
				Integer const m_aa = model.res( m_res );
				Integer const m_aav = model.res_variant( m_res );
				Integer const m_atom = (*ia).atom_id();

				Integer model_atom_type = aaproperties_pack::fullatom_type( m_atom, m_aa, m_aav );

				Real rep = lj_rep( model_atom_type, query_atom_type, distance_squared( (*ia), numeric::xyzTriple< float >( &query_full_coord[ idx ] ) ) );

				repulsive += param_pack::pack_wts.Wrep() * rep;
			}

		} // foreach atom
	}

	return repulsive;
}


/// @brief fast fullatom lj-repulsive using octree
/// @note template BooleanType requires operator []
inline
Real
octree_inter_rep(
	Octree< AtomPoint > const & oc,
	Pose const & model,
	Pose const & query,
	ResidueRange const & query_residue_range
)
{
	return octree_inter_rep( oc, model, query, query.full_coord(), query_residue_range );
}


} // namespace epigraft


#endif /*INCLUDED_epigraft_epigraft_functions_HH_*/
