// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   epigraft_io.hh
/// @brief  Common I/O routines for epigraft.
/// @author Bill Schief (schief@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_epigraft_io_HH_
#define INCLUDED_epigraft_epigraft_io_HH_


// package headers
#include <epigraft/epigraft_types.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/match/MatchResult.hh>

// ObjexxFCL headers
#include <ObjexxFCL/string.functions.hh>

// utility headers
#include <utility/io/ozstream.fwd.hh>
#include <utility/vector1.fwd.hh>

// C++ headers
#include <fstream>
#include <map>
#include <set>
#include <string>
#include <vector>


namespace epigraft {


/// @brief split given string to a container of strings, T must have method .push_back()
/// @param[in] start_from  start from column index and skip all other entries (counting from 1)
template< typename T >
void
split_string(
	std::string s,
	T & v,
	Size const & start_from = 1
)
{
	ObjexxFCL::strip_whitespace( s ); // this is required, otherwise an infinite loop can occur because the current_pos can go to -1 and comparison with end_pos is invalid
	std::istringstream in( s );

	std::streampos current_pos( in.tellg() );
	in.seekg( 0, std::ios::end );
	std::streampos end_pos( in.tellg() );
	in.seekg( 0, std::ios::beg );

	std::string entry;

	Size index = 1; // starting column entry
	while ( index < start_from && current_pos != end_pos ) {
		in >> entry;
		current_pos = in.tellg();
		++index;
	}

	while ( current_pos != end_pos ) {
		in >> entry;
		v.push_back( entry );
		current_pos = in.tellg();
	}
}


/// @brief get base pdb filename sans .pdb or .pdb.gz suffix
std::string
base_pdb_filename(
	std::string const & filename
);


/// @brief recursively create directory structure
/// @note will check if a subdirectory of given path exists before making
/// @warning Doesn't do path separator conversion!  Make sure you do this before calling the function.
void
recursively_create_directory(
	std::string const & path
);


/// @brief recursively create directory structure for filename
/// @details filename needs to have full path
void
recursively_create_directory_for_filename(
	std::string const & filename_with_path
);


/// @brief return filenames in a directory
/// @param[in] dirpath the name of the directory
/// @param[in] include_hidden whether to include hidden files/dirs, e.g.
///  in unix these are files whose names begin with '.'; default false
std::vector< std::string >
dir(
	std::string const & dirpath,
	bool const include_hidden = false
);


/// @brief count number of files in directory
/// @param[in] dirpath the name of the directory
/// @param[in] include_hidden whether to include hidden files/dirs, e.g.
///  in unix these are files whose names begin with '.'; default false
std::size_t
count_n_files(
	std::string const & dirpath,
	bool const include_hidden = false
);


/// @brief load scaffold filenames
void
load_scaffold_filenames(
	std::string const & filename,
	utility::vector1< std::string > & v
);


/// @brief load epitope ranges
void
load_epitope_subranges(
	std::string const & filename,
	utility::vector1< LoopInfo > & v,
	std::set< Integer > & user_specified_superposition_residues,
	bool use_SS_align = false
);


/// @brief make sure epitope subranges and pdb file match
/// @note looks up pdb information via pdb info in pose
bool
epitope_and_epitope_subranges_one_to_one(
	utility::vector1< LoopInfo > const & loops,
	Pose & pose
);


/// @brief open output file
void
open_output_file(
	std::string const & filename,
	utility::io::ozstream & outfile
);


/// @brief load old format match results as input
/// @note  file format:
/// @note  direc  initial_align_sys  native_loop_begin  native_loop_end  loop_subrange_begin  loop_subrange_end  scaffold_range_begin  scaffold_range_end  filename
void
load_old_format_match_results(
	std::string const & filename,
	std::map< std::string, utility::vector1< epigraft::match::MatchResult > > & results
);


} // namespace epigraft

#endif /*INCLUDED_epigraft_epigraft_io_HH_*/
