// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 12560 $
//  $Date: 2007-02-07 17:18:53 -0800 (Wed, 07 Feb 2007) $
//  $Author: yab $


// Rosetta Headers
#include "ligand.h"
#include "aaproperties_pack.h"
#include "add_pser.h"
#include "after_opts.h"
#include "are_they_neighbors.h"
#include "atom_is_backbone.h"
#include "BondSmallMolecule.h"
#include "cenlist.h"
#include "design.h"
#include "docking.h"
#include "docking_ns.h"
#include "docking_movement.h"
#include "docking_minimize.h"
#include "docking_minimize_ns.h"
#include "docking_db.h"
#include "enzyme.h"
#include "etable.h"
#include "FArray_xyz_functions.h"
#include "fast_pairenergy.h"
#include "favor_residue_ns.h"
#include "files_paths.h"
#include "fullatom.h"
#include "fullatom_setup.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "interface.h"
#include "ligand_ns.h"
#include "maps_ns.h"
#include "misc.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "pairenergy.h"
#include "param.h"
#include "param_pack.h"
#include "pdbstatistics_pack.h"
#include "read_aaproperties.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "score_ns.h"
#include "template_pack.h"
#include "water.h"
#include "water_ns.h"
#include "design.h"
#include "enzyme.h"
#include "enzyme_ns.h"
#include "gb_elec.h"
#include "weights_manager.h"
//KMa surface 2006-02
#include "SurfaceMode.h"
#include "surface.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>

// Triplet Headers
#include "triplet/triplet.h"
#include "triplet/xyzVector.h"
#include "triplet/xyzVector_io.h"

// Utility Headers
#include "utility/basic_sys_util.h"
#include "utility/irstream.h"
#include "utility/orstream.h"
#include "utility/izstream.h"
#include "utility/PeriodicSplineReader.h"

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <vector>
#include <list>

using namespace aaproperties_pack;
using namespace etable;

// =============================================================================
//
//     ligand.cc
//
//     Jens Meiler April 2003
//	   Kristian Kaufmann October 2005
// =============================================================================

// =============================================================================
//
// These are the first changes in order to allow ROSETTA to handle ligands in the
// fullatom energy evaluation, docking, and design parts. For this a new flag
// " -ligand " was added. If this flag is included, ROSETTA will read all HETATM
// lines as hetero atoms and try to map those on ROSETTA atom types by their name.
// So currently the names need to be consistent with one of the 26 atom names that
// ROSETTA has definined. ( In future I plane to add more atom types and some
// automated atom type recognition. ) For design and full atom energy evaluation it
// will include the interactions between protein and ligand into the comuptation.
// For docking the ligand is actually used in stead of the second protein and a
// protein - ligand docking is performed. Note that there is no global search setup
// but only a local fit of a ligand into a given pocket is performed. command line
// would be " rosetta xx xxxx 1 -dock -ligand -dock_mcm -dock_pert 5 5 95 -ex1
// -ex2aro_only -find_disulf -norepack_disulf -s xxxx.pdb ". ROSETTA will try to find
// an optimal position for the ligand in a 10x10x10A cube around its start position
// with full reorientation. For design the command line might be " rosetta -design
// -fixbb -use_input_sc -ex1 -ex2 -ligand -s xxxx.pdb -resfile resfile.mutations
// -ndruns 20 ". ROSETTA will attempt to redisign the protein around the given
// ligand. In any case if the " -ligand " is not set, nothing should change to any
// ROSETTA run.
//
// The ligand specific data containers are defined in " ligand.h " the functoins
// in " ligand.cc ". Further, the docking, design, pack, rotamer_trials, and full-
// atom energy evaluation were changed to call the ligadn energy functions if the
// ligand flag is true. The docking protocoil is at some points different from the
// protein-protein docking ... (1) there is a local search for a fitting orientation
// and position of the ligand with only VDW repulsive energies eveluated instead of
// the sliding into a contact. Also the ligand is brought into a random orientation
// instead the usual slight perturbation done while docking.
//
// For the ligand protein interaction vdw atraction, vdw repulsion, solvation,
// coulomb interaction, and hydrogen bonds are evaluated in full atom mode. Only
// amino acids that are detected to belong to the interface are considered in
// this evaluation. Coloumb interactions are evaluated with a distance dependent
// distant dependent dielectric constant.
//
// " vdw.cc " had already a function " vdw_hetero " that is used to prevent
// clashes in loop building. I did not change this function and I also kept the
// simple atom type mapping used for this function. However, it does now refer to
// data stored in " ligand.h ".
//
// =============================================================================

// =============================================================================
//     bool function to check on ligand_flag
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_ligand_flag
///
/// @brief
///     bool function to check on ligand_flag
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_ligand_flag()
{
	using namespace ligand;
	using namespace favor_residue;
	static bool init = { false };

  if ( !init ) {
    if (ligand_one==NULL ){
      if(ligand_ptr_vector.empty()) {ligand_ptr_vector.push_back(new Ligand());}
      ligand_one=ligand_ptr_vector[0];
    }
    ligand::ligand_flag = truefalseoption("ligand");
    init = true;

    if ( ligand::ligand_flag ) {
      InitializePackerWeightsDB();
      //mj apply ligand packer weights
      get_ligand_packer_weights();
      ligand_one->fix_ligand_HO = truefalseoption("fix_ligand_HO");
      ligand::ligand_flexible= truefalseoption("flexible_ligand");
      ligand::ligand_mdlfile = truefalseoption("ligand_mdlfile");
    }

    get_enzyme_flag();
  }
  return ligand::ligand_flag;
}

Ligand &
get_ligand_one()
{
  return *ligand::ligand_one;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_ligand_flag
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_ligand_flag( bool setting )
{
	using namespace ligand;

	ligand::ligand_flag = setting;
}

// =============================================================================
//     function to set packer weights for ligand mode
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_ligand_packer_weights
///
/// @brief
///     function to set packer weights for ligand mode
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_ligand_packer_weights()
{
//mj these are optimized weigths for sequence recovery inligand interfaces
//mj 01/14/2004

	using namespace ligand;
	using namespace param_pack;

	if ( gen_born && soft_rep )
	{
		RetrieveWeightsToCurrent( PW_LIGAND_SOFT_REP_GEN_BORN );
		std::cout << "!!! ligand weights for packer applied with -gen_born and -soft_rep falgs in use !!!" << std::endl;
		return;
	}

	if ( gen_born )
	{
		RetrieveWeightsToCurrent( PW_LIGAND_GEN_BORN );
		std::cout << "!!! ligand weights for packer applied with -gen_born falg in use !!!" << std::endl;
		return;
	}

	if ( soft_rep )
	{
		RetrieveWeightsToCurrent( PW_LIGAND_SOFT_REP );
		std::cout << "!!! ligand weights for packer applied with -soft_rep falgs in use !!!" << std::endl;
		return;
	}

	//mj standard
	RetrieveWeightsToCurrent( PW_LIGAND );
	std::cout << "!!! ligand weights for packer applied !!!" << std::endl;
}

// =============================================================================
//     void to reset hetero atoms
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin reset_hetero
///
/// @brief
///     void to reset hetero atoms
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
reset_hetero()
{
  for(size_t i=0; i < ligand::ligand_ptr_vector.size(); i++){
    delete ligand::ligand_ptr_vector[i];
  }
  ligand::ligand_ptr_vector.clear();
  ligand::ligand_ptr_vector.push_back(new Ligand());
  ligand::ligand_one=ligand::ligand_ptr_vector[0];
  return;
}

// =============================================================================
//     bool function to check on recompute_protein_ligand_interface
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_recompute_ligand_interface
///
/// @brief
///     bool function to check on recompute_protein_ligand_interface
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_recompute_ligand_interface(Ligand & molecule)
{
	return molecule.recompute_ligand_interface;
}

// =============================================================================
//     function to translate atom name into type
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin atom_num_from_atom_char
///
/// @brief
///     function to translate atom name into type
///
/// @detailed
///
/// @param  ac - [in/out]? -
/// @param  at - [in/out]? -
/// @param  hyb - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
atom_num_from_atom_char(
	std::string const & ac,
	int & at,
	int & act
)
{
	using namespace aaproperties_pack;
	using namespace param;

//mj find fullatomtype by name
	at = -1;
	act = 0;
	for ( int i = 1, e = MAX_ATOMTYPES(); i <= e; ++i ) {
		if ( ac == atom_type_name(i) ) {
			at = i;
			act = atom_type_cent(i);
		}
	}

//mj stop if atom type is undefined
	if ( at == -1 ) {
		if( get_enable_ligaa_flag() ) {
			//lin continue if get_enzyme_flag
			std::cout<<"unknown atom type"<< SS( ac ) <<", set to VHPO"<<std::endl;
			at = 66; act = atom_type_cent( 66 );
		} else {
			std::cout << "unknown atom_type " << SS( ac ) << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
}


// =============================================================================
//     PDB reader for HETATM's
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin read_pdb_hetero
///
/// @brief
///     PDB reader for HETATM's
///
/// @detailed
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified October 2005 Kristian Kaufmann
/////////////////////////////////////////////////////////////////////////////////
void
read_pdb_hetero(
  utility::io::irstream & iunit,
  Ligand & molecule
)
{
	using namespace ligand;
	using namespace param;
	using namespace runlevel_ns;
	using namespace misc;

//mj local variables
	std::string pdb_card; // first word of each line in a pdb file
	std::string atom_name; // atom name such as CA
	std::string res_name; // amino acid in three letter code
	char chain_id; // key to id chain
	char conf_id; // key to id multiple atom conformations
	int pdb_res; // residue number in pdb file
	int pdb_number; // HETATM number in pdb file
	char insert_id; // key to id insertions
	double xpos,ypos,zpos; // coordinates
	std::string chem_symbol; // used for hetatm
	float occ,temp; //lin read the occupancy and bvalue
	float total_charge; // total charge
	size_t h_atom_count;
	int atom_num; // atom# as defined in global.h
	int atom_cent_tmp;
	int atom_num_tmp; //kwk facilatates changes in storage process for atom type
	int const hetero_atom_limit = HETERO_ATOM_MAX()() - 1;
	triplet::xyzVector<double> new_center_of_protein,TRA_surface ; //KMa surface
	//???????????????????????????????????????????????????????????????????????????????


//mj function body
	iunit.clear();
	iunit.seek_beg();
	for (size_t i=0;
		i<molecule.atom_vector.size();
		i++){
		delete molecule.atom_vector[i];
	}
	molecule.atom_vector.clear(); // initialize, all old hetero atoms gone
	molecule.virtual_atom_count = 0;
	total_charge = 0.0;

L450:

	iunit >> bite( 6, pdb_card ) >> bite( 6, pdb_number ) >>
	 bite( 4, atom_name ) >> bite( conf_id ) >> bite( 3, res_name ) >>
	 skip( 1 ) >> bite( chain_id ) >> bite( 4, pdb_res ) >>
	 bite( insert_id ) >> skip( 3 ) >>
	 bite( 8, xpos ) >> bite( 8, ypos ) >> bite( 8, zpos ) >>
	 bite( 6, occ ) >> bite( 6, temp ) >> skip( 10 ) >> bite( 2, chem_symbol ) >> skip;
	if ( iunit.eof() ) {
		goto L350;
	} else if ( iunit.fail() ) {
		iunit.clear();
		iunit >> skip;
		goto L450;
	}

	if ( pdb_card != "HETATM" ) goto L450; // header information
	if ( res_name == "DOC" ) goto L450; // ignore docking coordinate system

//cems in this mark I edition of this program I will assume that HETAM are one
//cems contiguous block

	while ( int(molecule.atom_vector.size()) < hetero_atom_limit ) {

		if ( pdb_card == "END   " ) goto L350;
		if ( pdb_card == "TER   " ) goto L350;
		if ( pdb_card == ".     " ) goto L350;
		if ( pdb_card == " %" )    goto L350; // rosetta output
// kma vc++8 error: change to  ( pdb_card[1] == '%' )
		if ( pdb_card == "ENDMDL" ) goto L350;
		if ( pdb_card == "CHARGE" ) {
			total_charge = xpos;
			goto L475;
		}

//KMa surface
		if( surface::mode.flag.is_enabled() ) {
			if ( pdb_card == "SURFA0" )	{
			 surface::mode.PDB3atoms_planeCell[0]=pdb_number;
					surface::mode.surface3pointA0.x() = xpos;
					surface::mode.surface3pointA0.y() = ypos;
					surface::mode.surface3pointA0.z() = zpos;
			}
			if ( pdb_card == "SURFA1" ){
			 surface::mode.PDB3atoms_planeCell[1]=pdb_number;
					surface::mode.surface3pointB0.x() = xpos;
					surface::mode.surface3pointB0.y() = ypos;
					surface::mode.surface3pointB0.z() = zpos;

			}
			if ( pdb_card == "SURFA2" )	{
			 surface::mode.PDB3atoms_planeCell[2]=pdb_number;
					surface::mode.surface3pointC0.x() = xpos;
					surface::mode.surface3pointC0.y() = ypos;
					surface::mode.surface3pointC0.z() = zpos;

				std::cout <<  " SURFA0 pdbnumber = "
					<<	 surface::mode.PDB3atoms_planeCell[0] << std::endl;
				std::cout <<  " SURFA1 pdbnumber = "
					<<	 surface::mode.PDB3atoms_planeCell[1] << std::endl;
				std::cout <<  " SURFA2 pdbnumber = "
					<<	 surface::mode.PDB3atoms_planeCell[2] << std::endl;

			}

		}//surface::mode.flag.is_enabled()




		if ( pdb_card != "HETATM" ) goto L475; // read next line & restart loop
		if ( res_name == "HOH" )    goto L475; // ignore water
		if ( res_name == "DOC" )    goto L475; // ignore docking data

		get_hetatom_num( chem_symbol, atom_num );

		if ( !get_ligand_flag() && atom_num == 0 ) {
			if ( runlevel >= inform ) std::cout << "unknown hetero atom type " <<
			 chem_symbol << " will treat as carbon" << std::endl;
			atom_num = 1;
		}

//mj we need the hydrogens now for packing and docking
//mj		if ( atom_num == -1 ) goto L475; // skip hydrogens
//mj save information for hetero atoms
    molecule.atom_vector.push_back(new Atom());
    h_atom_count=molecule.atom_vector.size()-1;
    molecule.atom_vector[h_atom_count]->set_coordinates(xpos,ypos,zpos);
    molecule.atom_vector[h_atom_count]->set_ros_atom_cent(atom_num);
    molecule.atom_vector[h_atom_count]->set_pdb_number( pdb_number);


    molecule.atom_vector[h_atom_count]->set_atom_name(atom_name);
    molecule.hetero_atom_resid(molecule.atom_vector.size()) = res_name; // save residue name
    molecule.atom_vector[h_atom_count]->set_element(
            atom_chem::get_element_type_from_atom_type(atom_name));
    if (molecule.atom_vector[h_atom_count]->get_element_type()==atom_chem::UNDEFINED_ELEMENT_TYPE){
      molecule.atom_vector[h_atom_count]->set_element(atom_chem::get_element_type_from_symbol(chem_symbol));
    }
    molecule.virtual_atom_map(molecule.atom_vector.size()) = -1;
    molecule.virtual_atom_bnd(molecule.atom_vector.size()) = -1;

		//lin occ and temp
		molecule.atom_vector[h_atom_count]->set_occupancy( occ );
		molecule.atom_vector[h_atom_count]->set_bvalue( temp );


    if ( get_ligand_flag() || get_enable_ligaa_flag() ) {                  // obtain full atom type
      atom_num_from_atom_char( atom_name,          // if ligand is used for
                               atom_num_tmp,atom_cent_tmp); // packing or docking

      molecule.atom_vector[h_atom_count]->set_ros_atom_type(atom_num_tmp);
      // if full atom then more of the centroid types are difined. see aaproperties_pack and atom_num_from_atom_char
      molecule.atom_vector[h_atom_count]->set_ros_atom_cent(atom_cent_tmp);
//KMa
      if ( runlevel_ns::runlevel >= runlevel_ns::inform ){
      std::cout << "atom " << atom_name << ' ' << res_name <<
       " was set to type "
        << molecule.atom_vector[h_atom_count]->get_ros_atom_type()
        << " and centroid type " << atom_cent_tmp << std::endl;
      }


      if ( molecule.atom_vector[h_atom_count]->Is_Virtual() ){
        molecule.atom_vector[h_atom_count]->set_virtual_true();
         ++molecule.virtual_atom_count;
       }
      if ( molecule.atom_vector[h_atom_count]->get_ros_atom_cent() == 0 ) {
        if ( atom_cent_tmp == 0) {
          atom_cent_tmp = 1;
          if ( runlevel >= inform ) std::cout << "unknown hetero atom type " <<
                                      chem_symbol << " will treat as carbon" << std::endl;
        }
        molecule.atom_vector[h_atom_count]->set_ros_atom_cent(atom_cent_tmp);
        if ( runlevel >= inform ) std::cout <<
         "unknown hetero atom type will set to " <<
         SS(molecule.atom_vector[h_atom_count]->get_ros_atom_cent()  ) << std::endl;
      }
    }

//mj end saving information

L475:;
    iunit >> bite( 6, pdb_card ) >> bite( 6, pdb_number ) >>
     bite( 4, atom_name ) >> bite( conf_id ) >> bite( 3, res_name ) >>
     skip( 1 ) >> bite( chain_id ) >> bite( 4, pdb_res ) >>
     bite( insert_id ) >> skip( 3 ) >>
     bite( 8, xpos ) >> bite( 8, ypos ) >> bite( 8, zpos ) >>
     bite( 6, occ ) >> bite( 6, temp ) >> skip( 10 ) >> bite( 2, chem_symbol ) >> skip;
    if ( iunit.eof() ) {
      goto L350;
    } else if ( iunit.fail() ) {
      iunit.clear();
      iunit >> skip;
      goto L475;
    }

  }
L350:

//mj test for existence of ligand if ligand flag is on
  if ( ( get_ligand_flag() ||  get_enable_ligaa_flag() ) && molecule.atom_vector.size() == 0 ) {
    std::cout << "ERROR: ROSETTA was called with the -ligand" <<
     " option was used but no HETATM was found in PDB file" << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }


  if( surface::mode.flag.is_enabled() ) { //KMa surface


//  distances of the 2 basis vectors of the
//  surface crystal system that are the unit cell dimensions
     surface::mode.ABdistance =
      distance ( surface::mode.surface3pointA0, surface::mode.surface3pointB0 );
     surface::mode.ACdistance =
      distance ( surface::mode.surface3pointA0, surface::mode.surface3pointC0 );
     std::cout << "ABdistance= " << surface::mode.ABdistance << std::endl;
     std::cout << "ACdistance= " << surface::mode.ACdistance << std::endl;

// vectors of the 2 basis vectors of the surface crystal
// system that are the unit cell dimensions
     surface::mode.vAB0 =
      surface::mode.surface3pointB0 - surface::mode.surface3pointA0;
     surface::mode.vAC0 =
      surface::mode.surface3pointC0 - surface::mode.surface3pointA0;

//translate if the case
    TRA_surface = surface::mode.protein_surfacemove_symmetry();
  std::cout << "LIGAND new_center_of_surface x= "
    <<  surface::mode.centerG_surface.x()
    << " y= " <<   surface::mode.centerG_surface.y()
    << " z=" <<  surface::mode.centerG_surface.z() << std::endl;

  std::cout << "SURFACE translation x= " <<  TRA_surface.x()
  << " y= " << TRA_surface.y() << " z=" <<  TRA_surface.z() << std::endl;
  }



//mj additional setup for fullatom treatment of ligands
  if ( get_ligand_flag() ) {
    initialize_fullatom();
    initialize_fullcoord_array( Eposition, full_coord, total_residue, res, res_variant );
//mj compute h-bond acceptors and donors
    detect_hetero_atom_hbond(molecule);

//kwk flexible ligand setup
  if(ligand::ligand_flexible){
    molecule.setup_bond_graph();
    molecule.calculate_internal_coordinates(true);
    molecule.set_ligand_atom_types();
    molecule.get_ligand_internal_energy(true);
  }

//mj compute charges
    update_hetero_atom_charge( total_charge,molecule );

//mj set recompute interface to true
    molecule.recompute_ligand_interface = true;

//mj set covalent bonds
    detect_virtual_atom_covalent_bond(molecule);

//mj detect interface
    detect_ligand_interface(molecule, false);

  } else if ( get_enable_ligaa_flag() ) {
    update_hetero_atom_charge( total_charge,molecule );
	}
}
// =============================================================================
//     function for reading MDL format files for ligands
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin read_mdl
///
/// @brief
///     function for reading MDL (mol) format files into a ligand object
///
/// @detailed
///      The first three characters on the fourth line must be an integer of the
///  number of atoms the next three characters must be the number of bonds. If
/// either of these numbers are incorrect the function will not fail but will
/// give meaningless results. The fifth line through the 4+number_of_atoms line
/// will be read as follows.
///   xxxxxxxxxxyyyyyyyyyyzzzzzzzzzz_AAAAOOOOOOTTTTTT
/// The first ten characters are read as the x
/// coordinate, the next ten as the y coordinate, the following 10 as the z
/// coordinate. The 4 characters after the z coordinate will be read as the atom
/// type name. If this is not one of the assign names in atom_chem.cc then the
/// function will fail. When reading the bond records the information is stored
/// in vector of triplet xyzVector<size_t> which is then passed to a Ligand
/// object function to setup the bond network. The first six characters in
/// blocks of three characters indicate the line numbers in the atom block which
/// are bonded e.g. XX1X15 atom in line 1 of the atom block bonded to atom in
/// line 15 of the atom block. Characters from 7-9 indicate the bond_type
/// 0=NO_BOND,1=SINGLE, 2=DOUBLE, 3=TRIPLE, 4=AROMATIC, 5=CONJUGATED,
/// 6=HYDROGEN_BOND, 7=METALLIC_BOND, 8=UNDEFINED_BOND_TYPE. The last currently
/// is configured only to accept charged information. If a total charge is not
/// given using an "M CHG" in the first five characters then the partial charges
/// will be adjusted using the update_hetero_atom_charge function with an
/// assumed total charge of 0.000. If "MACHG" records are given for each
//  of the atoms then given charges will be assigned to the respective atom
/// according to the number in the next three characters following the "MACHG".
/// if any of the atoms is left with a default partialcharge_ of 9999.9 then
/// sum of partialcharges not equal to 9999.9 will be supplied as the total
/// charge to the update_hetero_atom_function. Any "MACHG" flags will override
/// any "M CHG" given. The charges will be extracted from the columns X in the
/// following template "M CHGAAAXXXX.XXXX".
///
/// @param  mdlfile - [in] - string of the mdlfile
/// @param  molecule - [out] - Ligand object into which the data is place.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///     Kristian Kaufmann Vanderbilt University
/// @last_modified
///     2-9-2006
/////////////////////////////////////////////////////////////////////////////////
int
read_mdl(
 utility::io::izstream & iunit,
 Ligand * molecule
){
  bool update_partial_charges=true;
	if( iunit.fail() || (molecule->atom_vector.size()!=0)){
    std::cout << "Either the mdlfile file failed to open or the Ligand object";
    std::cout << std::endl <<"    passed was not empty." << std::endl;
    return -1;
  }else{
    bool done_reading=false; //file parsing control
    size_t number_of_atoms;  //file parsing control
    size_t number_of_bonds;  //file parsing control
    int atom_num, atom_num_tmp, atom_cent_tmp; //atom type variables
    size_t atom_1, atom_2, bond_type;// bond parsing variables
    std::list< triplet::xyzVector<size_t> > bond_map;// bond_graph_variable
    float xpos, ypos, zpos;// atom parsing variable
    std::string pdb_atomname; // atom parsing variable
    std::string chem_symbol; // atom parsing variable
    std::string charge_flag; // charge parsing variable
		float occ, temp; //occupancy and bvalue
    int root_atom = -10; // variable for setting root atom if desired
    size_t n_assign_charges=0;
    double p_charge;
    float  total_charge=0.0;


    size_t i=1;
    while(!done_reading){
      if(i<4){
        iunit >> skip;// skips first three lines of mdl file
        i++;
      }else if(i==4){ // obtains atom number and bond number information
        iunit >> bite(3, number_of_atoms) >> bite(3, number_of_bonds) >> skip;
        i++;
      }else if(i<=4+number_of_atoms && i>4){// read atom block and stores in
        iunit >> bite(10, xpos) >> bite(10, ypos) >> bite(10,zpos) >>
          skip(1) >> bite(4, pdb_atomname) >> bite( 6, occ ) >>
					bite( 6, temp ) >> skip;      // molecule
				molecule->atom_vector.push_back(new Atom());
				molecule->atom_vector.back()->set_coordinates(xpos,ypos,zpos);
				molecule->atom_vector.back()->set_pdb_number(i-4);
				molecule->atom_vector.back()->set_atom_name(pdb_atomname);
				molecule->hetero_atom_resid(molecule->atom_vector.size())="XXX";
				molecule->atom_vector.back()->set_element(
            atom_chem::get_element_type_from_atom_type(pdb_atomname)
          );
        chem_symbol=atom_chem::element_type_symbol[int(
					molecule->atom_vector.back()->get_element_type())];
				molecule->virtual_atom_map(molecule->atom_vector.size())=-1;
				molecule->virtual_atom_bnd(molecule->atom_vector.size())=-1;
        get_hetatom_num( chem_symbol, atom_num );
				molecule->atom_vector.back()->set_ros_atom_cent(atom_num);
				molecule->atom_vector.back()->set_occupancy(occ);
				molecule->atom_vector.back()->set_bvalue(temp);
        if (get_ligand_flag()){
          atom_num_from_atom_char(pdb_atomname, atom_num_tmp, atom_cent_tmp);
					molecule->atom_vector.back()->set_ros_atom_type(atom_num_tmp);
					molecule->atom_vector.back()->set_ros_atom_cent(atom_cent_tmp);
          if( runlevel_ns::runlevel >= runlevel_ns::inform){
            std::cout << "atom " << pdb_atomname << " was set to type";
            std::cout << atom_num_tmp << " and centroid " << atom_cent_tmp;
            std::cout << std::endl;
          }
          if( atom_num_tmp > param::MAX_REALTYPES ){
						molecule->atom_vector.back()->set_virtual_true();
						++molecule->virtual_atom_count;
          }
          if( atom_cent_tmp == 0){
            if(runlevel_ns::runlevel >= runlevel_ns::inform){
              std::cout << "Unknown atom type will treat as Carbon" << std::endl;
              std::cout << "Setting atom " << number_of_atoms-4 << " to carbon";
              std::cout << std::endl;
            }
						molecule->atom_vector.back()->set_ros_atom_cent(1);
          }
        }
        i++;
      }else if(i<=4+number_of_atoms+number_of_bonds && i>4+number_of_atoms){
        iunit >> bite(3, atom_1) >> bite(3, atom_2) >> bite(3,bond_type)
              >> skip;
//        pushing back atomline-1 converts to a zero basis which is what
// atom_vector is on
        triplet::xyzVector< size_t > bond_info(atom_1-1,
                             atom_2-1,bond_type);
        bond_map.push_back(bond_info);
        if(runlevel_ns::runlevel >= runlevel_ns::inform){
          std::cout << "Storing " << bond_info.x()+1 << " " << bond_info.y()+1 << " " <<bond_info.z();
          std::cout << " for bond " << i-4-number_of_atoms << std::endl;
        }
        i++;
      }else{
        iunit >> bite( 5, charge_flag );
        if( charge_flag=="M END"){
          done_reading=true;
        }else if( charge_flag=="MACHG"){
          n_assign_charges++;
          iunit >> bite(3, atom_1) >> bite(9, p_charge) >> skip;
					molecule->atom_vector[atom_1]->set_partial_charge(p_charge);
          update_partial_charges=false;
        }else if( charge_flag=="M CHG"){
          iunit >> skip(3) >> bite(9, total_charge) >> skip;
        }else if( charge_flag=="MROOT"){
          iunit >> bite(3, root_atom) >> skip;
		}else if ( charge_flag=="$$$$ "){
					iunit >> skip;
					ligand::ligand_ptr_vector.push_back(new Ligand());
					read_mdl( iunit, (ligand::ligand_ptr_vector.back()));
        }else if( charge_flag == "MPROT" ){
          std::string protocol_flag;
          iunit >> skip(3) >> bite(16,protocol_flag);
          if( protocol_flag == "fix_ligand_OH" ){
            molecule->fix_ligand_HO=true;
          }else if( protocol_flag == "use_input_ligand" ){
            molecule->use_input_ligand=true;
          }
        }else{
          iunit >> skip;
        }
        i++;
      }
      if ( iunit.eof() ) {
        done_reading=true;
      }else if ( iunit.fail() ) {
        std::cout << "Stream reading mdlfile entered a fail state" << std::endl;
        iunit.clear();
        iunit >> skip;
      }
    }//while
		if(molecule->atom_vector.size()==0){
          std::cout << "mdlfile not contain atom information.";
          std::cout << std::endl;
          return -1;
    }
		if(get_ligand_flag() && molecule->atom_vector.size()==0){
      std::cout << "ERROR -ligand -ligand_mdlfile was called but mdl read";
      std::cout << std::endl << "  function wasn't able to read any atoms";
      std::cout << std::endl;
    }
    if(get_ligand_flag() || get_enable_ligaa_flag()){ //full atom setup functions

      //fullatom initialization
      initialize_fullatom();
      initialize_fullcoord_array( misc::Eposition, misc::full_coord,
        misc::total_residue, misc::res, misc::res_variant );

      //setting charge in atoms
      if(!update_partial_charges){
        total_charge=0.0;
				for(ligand::Atom_Itr current_atom=molecule->atom_vector.begin(); current_atom!=
					molecule->atom_vector.end(); current_atom++ ){
            if((*current_atom)->get_partial_charge()>=9999.9-0.0001 &&
               (*current_atom)->get_partial_charge()<=9999.9+0.0001){
               update_partial_charges=true;
            }else{
              total_charge += (*current_atom)->get_partial_charge();
            }
        }
        if(update_partial_charges){
					update_hetero_atom_charge( total_charge,*molecule );
        }

      }else{
				update_hetero_atom_charge( total_charge,*molecule );
      }
      //computing h-bond acceptors
			detect_hetero_atom_hbond(*molecule);

      //initializing ligand_flexibility
      if(ligand::ligand_flexible){
				molecule->setup_bond_graph(bond_map, root_atom);
				molecule->calculate_internal_coordinates(true);
				molecule->set_ligand_atom_types();
				molecule->get_ligand_internal_energy(true);
	      molecule->optimize_H();
				molecule->compute_ligand_conformations();
      } else {
        molecule->setup_bond_graph(bond_map, root_atom);
				molecule->calculate_internal_coordinates(true);
      }

      //set recompute_ligand_interface
			molecule->recompute_ligand_interface=true;
      //set covalent bonds
			detect_virtual_atom_covalent_bond(*molecule);
      //detect interface
			detect_ligand_interface(*molecule,false);

    }//fullatom setup
  }//else
  return 1;
}

// =============================================================================
//     function for mapping any heteroa atoms on one of the five AA bb atoms
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_hetatom_num
///
/// @brief
///     function for mapping any heteroa atoms on one of the five AA bb atoms
///
/// @detailed
///
/// @param  sname - [in/out]? - cems short name
/// @param  number - [in/out]? - atom number
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_hetatom_num(
  std::string const & sname, //cems short name
  int & number // atom number
)
{
// the numerical assignment here intended to mimic
// similar atoms we have tabulated and recycle the atom_pair distance values
// from rasmol:
// C 1.8A  similar to Cbeta carbon
// N 1.65  similar to backbone N
// O 1.4   similar to backbone oxygen
// S 1.8   similar to Cbeta
// P 1.85  similar to Cbeta
// virtual atom type is ignored by treating as hydrogen

  if ( sname == " C" ) {
    number = 3;
  } else if ( sname == " N" ) {
    number = 1;
  } else if ( sname == " O" ) {
    number = 5;
  } else if ( sname == " S" ) {
    number = 3;
  } else if ( sname == " P" ) {
    number = 3;
  } else if (sname ==" H")  {// kma vc++8 error: else if ( sname[1] == 'H' )
    number = -1;
  } else if ( sname == " V" ) {
    number = -1;
  } else {
    number = 0;
  }

//  std::cout << "hetero atom type: " << sname << SS( number ) << std::endl;

}

// =============================================================================
//     PDB writer for HETATM's
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin make_pdb_hetero
///
/// @brief
///     PDB writer for HETATM's
///
/// @detailed
///
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_pdb_hetero(
  utility::io::orstream & iunit,
  Ligand & molecule
)
{
  using namespace ligand;

//mj local variables
  float const b1 = { 1.0 };
  float const b2 = { 0.0 };

//mj function body
  for ( size_t i = 0; i < molecule.atom_vector.size(); ++i ) {
    std::string atom_name, chem_symbol;
    atom_name=molecule.atom_vector[i]->get_atom_name();
    chem_symbol=atom_chem::element_type_symbol[int(
        molecule.atom_vector[i]->get_element_type())
      ];
    triplet::xyzVector_float coordinates;
    coordinates=molecule.atom_vector[i]->get_coordinates();
    iunit << "HETATM" <<
    I( 5, molecule.atom_vector[i]->get_pdb_number() ) << ' '
    << atom_name << ' ' << molecule.hetero_atom_resid(i+1) << "  " <<
    I( 4, 900 ) << "    " <<
    F( 8, 3, coordinates.x() ) <<
    F( 8, 3, coordinates.y() ) <<
    F( 8, 3, coordinates.z() ) <<
    F( 6, 2, b1 ) << F( 6, 2, b2 ) << space( 10 ) <<
    chem_symbol << std::endl;
  }
  if ( get_ligand_flag() ) iunit << "CHARGE" << space( 24 ) <<
   F( 8, 3, molecule.hetero_atom_total_charge ) << std::endl;
  if( get_ligand_flag() && ligand::ligand_flexible ) {
    ligand::Ligand_AtomOP parent_atom=NULL;
    ligand::Ligand_AtomOP child_atom=NULL;
    molecule.calculate_internal_coordinates(false);
		iunit << "REMARK HETATM CONECT statements pdbnumber pdbnumber bondtype child_atom index length angle dihedral " << std::endl;
    for ( ligand::Bond_Itr current_bond=
      molecule.bond_vector.begin();
      current_bond!=molecule.bond_vector.end();current_bond++){

      (*current_bond)->bonded_atoms(parent_atom,child_atom);
    iunit << "REMARK " << parent_atom->get_pdb_number() << " " <<
        child_atom->get_pdb_number() << " " <<
        int((*current_bond)->get_bondtype()) <<
        " " << child_atom->get_index() << " " <<
        F( 6 ,3,(*current_bond)->get_length()) << " " <<
        F( 6 ,3,(*current_bond)->get_angle()) << " " <<
        F( 6 ,3,(*current_bond)->get_dihedral()) << std::endl;
    }

  }
  iunit << A( 6, "TER" ) << std::endl;
//mj end revise

//KMa print final surface center and 3d distance for the center of protein

  if(surface::mode.flag.is_enabled()){
  std::cout << "centerG_surface FINAL x= " << docking::part_centroid(1,2) << " y= " <<
  docking::part_centroid(2,2)  << " z=" << docking::part_centroid(3,2)  << std::endl;
  std::cout << "center PLANE 3d distance of the protein FINAL ="
  << sqrt((docking::part_centroid(1,2)-docking::part_centroid(1,1))*
      (docking::part_centroid(1,2)-docking::part_centroid(1,1))+
      (docking::part_centroid(2,2)-docking::part_centroid(2,1))*
      (docking::part_centroid(2,2)-docking::part_centroid(2,1))+
      (docking::part_centroid(3,2)-docking::part_centroid(3,1))*
      (docking::part_centroid(3,2)-docking::part_centroid(3,1)) )
   << std::endl;
  }

}

// =============================================================================
//     Detection of possible Hbond donors and aceptors
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin detect_hetero_atom_hbond
///
/// @brief
///     Detection of possible Hbond donors and aceptors
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
detect_hetero_atom_hbond(
  Ligand & molecule
)
{
  using namespace ligand;

//mj local variables
  size_t atom_1, atom_2;
  int closest, O_base,lid;
  float min_dist, dist;
  triplet::xyzVector_float atom_1_coordinates, atom_2_coordinates,
    closest_coordinates;
  std::string closest_atom_name, atom_1_name;

//mj function body
  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;

//mj find h-bond donors
  molecule.hetero_atom_hbond_don_count = 0; // erase all existing h-bond donors
  for ( atom_1 = 0; atom_1 < molecule.atom_vector.size(); ++atom_1 ) {
   // start loop over all hetero atoms to find polar H's
//KMa find hbond H
    lid=molecule.atom_vector[atom_1]->get_ros_atom_type();
    if ( fa_hbondH( lid ) == true )
     {
      atom_1_coordinates=
        molecule.atom_vector[atom_1]->get_coordinates();
//mj find closest non-hydrogen atom
      closest = -1;
      min_dist = 0.0;

      for ( atom_2 = 0; atom_2 < molecule.atom_vector.size(); ++atom_2 ) {
       // start loop over all hetero atoms to find closest heavy atom
//KMa all hydrogen atoms have atom_type_cent -1
        lid=molecule.atom_vector[atom_2]->get_ros_atom_type();
        if ( atom_type_cent( lid ) != -1 )
         {
         // start allow only hetero atoms
          atom_2_coordinates=
            molecule.atom_vector[atom_2]->get_coordinates();
          dist=distance( atom_1_coordinates,
            atom_2_coordinates);
//          distance_bk(hetero_atom_coord(1,atom_1),
//           hetero_atom_coord(1,atom_2),dist);
          if ( closest == -1 || dist <= min_dist ) {
            min_dist = dist;
            closest = atom_2;
          }
        }    // end allow only hetero atoms
      }       // end loop over all hetero atoms to find closest heavy atom

//mj save information in donor field
      if ( closest != -1 )
      {

//mj in case of B-O-H find B
        O_base = -1;
        min_dist = 0.0;
//KMa possible hydroxyl oxygen list //needs update if someone add another one!
        lid=molecule.atom_vector[closest]->get_ros_atom_type();
        if ( atom_type_name( lid ) == "OH  " || atom_type_name( lid ) == "OHha"
          || atom_type_name( lid ) == "Oice")
        {
          closest_coordinates=
            molecule.atom_vector[closest]->get_coordinates();
          for ( atom_2 = 0; atom_2 < molecule.atom_vector.size(); ++atom_2 ) {
            // start loop over all hetero atoms to find closest heavy atom
//KMa all hydrogen atoms have atom_type_cent -1
            lid=molecule.atom_vector[atom_2]->get_ros_atom_type();
            if (atom_type_cent( lid ) != -1 && int (atom_2) != closest)
             {
              atom_2_coordinates=
                molecule.atom_vector[atom_2]->get_coordinates();
              dist=distance( closest_coordinates,
                atom_2_coordinates);

              // start allow only hetero atoms
//              distance_bk(hetero_atom_coord(1,closest),
//                          hetero_atom_coord(1,atom_2),dist);
              if ( O_base == -1 || dist <= min_dist ) {
                min_dist = dist;
                O_base = atom_2;
              }
            }    // end allow only hetero atoms
          }
        }

//mj store data
        ++molecule.hetero_atom_hbond_don_count;
        molecule.hetero_atom_hbond_don(1,molecule.hetero_atom_hbond_don_count) = atom_1;
        molecule.hetero_atom_hbond_don(2,molecule.hetero_atom_hbond_don_count) = closest;
        molecule.hetero_atom_hbond_don(3,molecule.hetero_atom_hbond_don_count) = 0;
        molecule.hetero_atom_hbond_don(4,molecule.hetero_atom_hbond_don_count) = O_base;

//mj print information

        closest_atom_name=molecule.atom_vector[closest]->get_atom_name();
        atom_1_name=molecule.atom_vector[atom_1]->get_atom_name();
//KMa runlevel for printing hbond
        if ( runlevel_ns::runlevel >= runlevel_ns::inform ){
        std::cout << "Detected H-Bond donor   : " <<
          closest_atom_name << " in " <<
          molecule.hetero_atom_resid(closest+1) << " and " <<
          atom_1_name << " in " <<
          molecule.hetero_atom_resid(atom_1+1) << " !" << std::endl;
        }

//mj print warning if atom closest is Carbon
        if ( molecule.atom_vector[closest]->get_ros_atom_type() <= 6 ||
         molecule.atom_vector[closest]->get_ros_atom_type() == 18 ||
         molecule.atom_vector[closest]->get_ros_atom_type() == 19 ) {
          std::cout << "WARNING: Closest atom to polar H is C " <<
           closest_atom_name << " in " <<
           molecule.hetero_atom_resid(closest+1) << " !" << std::endl;
        }
      } else {

//mj print error if for some reason no closest atom was found
        std::cout << "ERROR: No bond non-hydrogen atom found for " <<
         atom_1_name << " in " <<
         molecule.hetero_atom_resid(atom_1+1) << " !" << std::endl;
      }

    }                     // end find polar hydrogens
  }                        // end loop over all hetero atoms to find polar H's



//mj find h-bond acceptors
  molecule.hetero_atom_hbond_acc_count = 0; // erase all existing h-bond acceptors
  for ( atom_1 = 0; atom_1 < molecule.atom_vector.size(); ++atom_1 ) {
   // start loop over all hetero atoms to find O's and N's
//KMa all acceptors search, boolean check from atom HBond properties
//KMa atom 9='NH2O' is not acceptor but added for ligand benchmarks:aa1dg5//personal comm with KKaufmann

    lid=molecule.atom_vector[atom_1]->get_ros_atom_type();
    if (fa_acceptor( lid ) == true || lid == 9 ) // magic numbers are evil
    {

//mj find closest non-hydrogen atom
      atom_1_coordinates=
        molecule.atom_vector[atom_1]->get_coordinates();
      closest = -1;
      min_dist = 0.0;
      for ( atom_2 = 0; atom_2 < molecule.atom_vector.size(); ++atom_2 ) {
       // start loop over all hetero atoms to find closest heavy atom
//KMa all hydrogen atoms have atom_type_cent -1
        lid=molecule.atom_vector[atom_2]->get_ros_atom_type();
        if (atom_type_cent( lid ) != -1 && atom_2 != atom_1)
        {
          atom_2_coordinates=molecule.atom_vector[atom_2]->get_coordinates();
          dist=distance( atom_1_coordinates,
            atom_2_coordinates);
//            distance_bk(hetero_atom_coord(1,atom_1),
//           hetero_atom_coord(1,atom_2),dist);
          if ( closest == -1 || dist <= min_dist ) {
            min_dist = dist;
            closest = atom_2;
          }
        }        // end allow only hetero atoms
      }           // end loop over all hetero atoms to find closest heavy atom

//mj save information in acceptor field
      if ( closest != -1 ) {
        ++molecule.hetero_atom_hbond_acc_count;
        molecule.hetero_atom_hbond_acc(1,molecule.hetero_atom_hbond_acc_count) = atom_1;
        molecule.hetero_atom_hbond_acc(2,molecule.hetero_atom_hbond_acc_count) = closest;
        molecule.hetero_atom_hbond_acc(3,molecule.hetero_atom_hbond_acc_count) = 0;

//mj print infomration
        closest_atom_name=molecule.atom_vector[closest]->get_atom_name();
        atom_1_name=molecule.atom_vector[atom_1]->get_atom_name();

//KMa runlevel for printing hbond
        if ( runlevel_ns::runlevel >= runlevel_ns::inform ){
        std::cout << "Detected H-Bond acceptor: " <<
        closest_atom_name << " in " <<
        molecule.hetero_atom_resid(closest+1) << " and " <<
        atom_1_name << " in " <<
        molecule.hetero_atom_resid(atom_1+1) << " !" << std::endl;
        }
      } else {

//mj print error if for some reason no closest atom was found
        std::cout << "ERROR: No bond non-hydrogen atom found for " <<
         atom_1_name << " in " << molecule.hetero_atom_resid(atom_1+1) <<
         " !" << std::endl;
      }

    }              // end find O's and N's
  }                 // end loop over all hetero atoms to find O's and N's
}

// =============================================================================
//     Reset count of satisfied hbond donors and acceptors
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_satisfied_hetero_atom_hbond
///
/// @brief
///     Reset count of satisfied hbond donors and acceptors
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_satisfied_ha_hbond(
  Ligand & molecule
)
{
  using namespace ligand;

//mj function body
  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;

//mj reset donors
  for ( int atom = 1; atom <= molecule.hetero_atom_hbond_don_count; ++atom ) {
    molecule.hetero_atom_hbond_don(3,atom) = 0;
  }

//mj reset acceptors
  for ( int atom = 1; atom <= molecule.hetero_atom_hbond_acc_count; ++atom ) {
    molecule.hetero_atom_hbond_acc(3,atom) = 0;
  }

}

// =============================================================================
//     Reset count of satisfied hbond donors and acceptors
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_satisfied_hetero_atom_hbond
///
/// @brief
///     Reset count of satisfied hbond donors and acceptors
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
count_satisfied_ha_hbond(
  int & don_sat,
  int & don_non,
  int & acc_sat,
  int & acc_non,
  Ligand & molecule
)
{
  using namespace ligand;

//mj function body
  don_sat = 0;
  don_non = 0;
  acc_sat = 0;
  acc_non = 0;
  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;

//mj reset donors
  for ( int atom = 1; atom <= molecule.hetero_atom_hbond_don_count; ++atom ) {
    if ( molecule.hetero_atom_hbond_don(3,atom) > 0 ) {
      ++don_sat;
    } else {
      ++don_non;
    }
  }

//mj reset acceptors
  for ( int atom = 1; atom <= molecule.hetero_atom_hbond_acc_count; ++atom ) {
    if ( molecule.hetero_atom_hbond_acc(3,atom) > 0 ) {
      ++acc_sat;
    } else {
      ++acc_non;
    }
  }

}

// =============================================================================
//     recompute charges
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin update_hetero_atom_charge
///
/// @brief recompute charges
///
/// @detailed
///
/// @param  total_charge - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_hetero_atom_charge(
  float const total_charge,
  Ligand & molecule
)
{
  using namespace aaproperties_pack;
  using namespace ligand;
  using namespace param;

//mj sum simple charges
  float added_charge = 0.0;
  int hetatom_count = 0;
  for ( size_t hetatom_id = 0; hetatom_id < molecule.atom_vector.size(); ++hetatom_id ) {
   // start loop over all hetero atoms
    added_charge += charge(molecule.atom_vector[hetatom_id]->get_ros_atom_type());
    if ( !molecule.atom_vector[hetatom_id]->Is_Virtual() )
     ++hetatom_count; // don't use virtual atoms
  }

//mj compute correction per atom
  float correction = 0.0;
  if ( hetatom_count > 0 ) correction =
   ( total_charge - added_charge ) / hetatom_count;

//mj apply correction
  for ( size_t hetatom_id = 0; hetatom_id < molecule.atom_vector.size(); ++hetatom_id ) {
   // start loop over all hetero atoms
    if ( !molecule.atom_vector[hetatom_id]->Is_Virtual() ) // don't use virtual atoms
     molecule.atom_vector[hetatom_id]->set_partial_charge(
     correction + charge(
       molecule.atom_vector[hetatom_id]->get_ros_atom_type()));
  }
  molecule.hetero_atom_total_charge = total_charge;

//mj report result
  std::cout << "update_hetero_atom_charge:\n" << "    total charge       = "
            << SS( total_charge ) << "    sum simple charges = " << SS( added_charge )
            << "    correction         = " << SS( correction ) << std::endl;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin ligand_use_atom_initializer
///
/// @brief
///
/// @detailed
///
/// @param  use_atom - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
ligand_use_atom_initializer( FArray1D_int & use_atom )
{
  int i = 0;
  use_atom( ++i ) = 5; // A,C,D,E,F
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 2; // G,H,I,K,L
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5; // M,N,P,Q,R
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5; // S,T,V,W,Y
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
  use_atom( ++i ) = 5;
	if ( add_pser() ) {
		//KMa phospho_ser
		use_atom( ++i ) = 5;
	}
	if ( design::dna_interface ) {
		use_atom( ++i ) = 5;  // KMa phospho_ser
		use_atom( ++i ) = 11; // G,A,C,T
		use_atom( ++i ) = 11;
		use_atom( ++i ) = 11;
		use_atom( ++i ) = 11;
		use_atom( ++i ) = 12; // rG,rA,rC,rT
		use_atom( ++i ) = 12;
		use_atom( ++i ) = 12;
		use_atom( ++i ) = 12;
	}
	if( get_enable_ligaa_flag() ){
		use_atom( ++i ) = 1; // ligand
		use_atom( ++i ) = 1;
	}
}


// =============================================================================
// Compute distance between ligand atom and amino acid and sets neighbor flag
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin hetero_atom_amino_acid_distance
///
/// @brief
/// Compute distance between ligand atom and amino acid and sets neighbor flag
///
/// @detailed
///
/// @param  aa - [in/out]? -
/// @param  coord - [in/out]? -
/// @param  het_coord - [in/out]? -
/// @param  dis2 - [in/out]? -
/// @param  neighbor - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
hetero_atom_amino_acid_distance(
  int const aa,
  FArray2Da_float coord,
  FArray1Da_float het_coord,
  float & dis2,
  bool & neighbor
)
{
  using namespace aaproperties_pack;
  using namespace param;

  coord.dimension( 3, MAX_ATOM() );
  het_coord.dimension( 3 );

//mj local
  static FArray1D_int const use_atom( MAX_AA(), ligand_use_atom_initializer );
//mj function body
  distance2_bk(coord(1,use_atom(aa)),het_coord(1),dis2);
  float const pc1 = paircutoff(aa,6)+1.0;
  neighbor = ( dis2 <= pc1 * pc1 );
}


// =============================================================================
// Computes VDW and SOLVATION energies between AA and LIGAND for repacking (pack.cc)
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_sc_haE
///
/// @brief
/// Computes VDW and SOLVATION energies between AA and LIGAND for repacking (pack.cc)
///
/// @detailed
///
/// @param  aaid - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  coord - [in/out]? -
/// @param  solvE - [in/out]? -
/// @param  atrE - [in/out]? -
/// @param  repE - [in/out]? -
/// @param  eval_bb - [in/out]? -
/// @param  eval_sc - [in/out]? -
/// @param  update_deriv - [in/out]? -
/// @param  f1 - [in/out]? -
/// @param  f2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_sc_haE(
  int const aaid,
  int const aa,
  int const aav,
  FArray2Da_float coord,
  float & solvE,
  float & atrE,
  float & repE,
  bool eval_bb,
  bool eval_sc,
  Ligand & molecule
)
{
  using namespace aaproperties_pack;
  using namespace interface;
  using namespace ligand;
  using namespace enzyme;
  using namespace param;

  coord.dimension( 3, MAX_ATOM() );

//mj local variables
  size_t hetatom_id,atom1,start_id, end_id;
  float d2;
  bool neighbor;
  float const cp_weight = { 1.0 };
  triplet::xyzVector_float hetatom_id_coordinate;
  FArray1D_float het_coord(3);
//mj set variables to 0
  solvE = 0.0;
  atrE = 0.0;
  repE = 0.0;

//mj function body
  if ( !get_ligand_flag() ||
        molecule.atom_vector.size() == 0 ) return;
  if ( !int_res(aaid) && !enzyme_vrot ) return;

  bool dont_eval_exclude = ! get_enzyme_flag(); // skip the exclude_evaluation call unless in enzyme mode

// loop over all hetero atoms
  for ( hetatom_id = 0; hetatom_id < molecule.atom_vector.size(); ++hetatom_id ) {
    if ( !molecule.atom_vector[hetatom_id]->Is_Virtual() ) {
    copy_to_FArray(molecule.atom_vector[hetatom_id]->get_coordinates(),het_coord);

    //mj compute distance^2 of hetero atom to amino acid
    hetero_atom_amino_acid_distance(aa,coord(1,1),
      het_coord(1),d2,neighbor);

    //mj if distance smaller aa - GLYCINE cutoff
    if ( neighbor ) {           // start check atom aa distance

      //mj compute first and last atom to use depending on eval_bb and eval_sc
      start_id = eval_bb ? 1 : 5;
      end_id   = eval_sc ? natoms(aa,aav): 4;

      //mj loop over all protein atoms
      for ( atom1 = start_id; atom1 <= end_id; ++atom1 )
        if ( dont_eval_exclude || !exclude_interaction( (hetatom_id+1), atom1, aa, aav ) )
          fast_pairenergy(coord(1,atom1),het_coord(1),
                        fullatom_type(atom1,aa,aav),molecule.atom_vector[hetatom_id]->get_ros_atom_type(),solvE,atrE,
                        repE,d2,cp_weight);
    } //if neighbor loop
  }}//if virtual block statement and for hetatom_id loop
}

// =============================================================================
//     Computes VDW and SOLVATION energiesderivatives for one particular chi
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_sc_haD_chi
///
/// @brief
///     Computes VDW and SOLVATION energiesderivatives for one particular chi
///
/// @detailed
///
/// @param  aaid - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  coord - [in/out]? -
/// @param  chi - [in/out]? -
/// @param  f1 - [in/out]? -
/// @param  f2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_sc_haD_chi(
  int const aaid,
  int const aa,
  int const aav,
  FArray2Da_float coord,
  int & chi,
  FArray1Da_float f1,
  FArray1Da_float f2,
  Ligand & molecule
)
{
  using namespace aaproperties_pack;
  using namespace interface;
  using namespace ligand;
  using namespace param;
  using namespace param_pack;

//mj return if not funny
  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;
  if ( !int_res(aaid) ) return;

  coord.dimension( 3, MAX_ATOM() );
  f1.dimension( 3 );
  f2.dimension( 3 );

//mj local variables
  float dE_dr,datr,drep,dsolv,dlj;
  triplet::xyzVector_float hetatom_id_coordinate;
  FArray1D_float het_coord(3);
//mj function body
  f1 = 0.0;
  f2 = 0.0;

  bool dont_eval_exclude = ! get_enzyme_flag(); // skip the exclude_evaluation call unless in enzyme mode

//mj loop obver all hetero atoms
  for ( size_t hetatom_id = 0; hetatom_id < molecule.atom_vector.size(); ++hetatom_id )
    if ( !molecule.atom_vector[hetatom_id]->Is_Virtual() ) {

      copy_to_FArray(molecule.atom_vector[hetatom_id]->get_coordinates(),het_coord);
    //mj loop over all aminoacid atoms
    for ( int i = 1, ie = length_chi_rigid_atom_list(chi,aa,aav); i <= ie; ++i ) {
      int atom1 = chi_rigid_atom_list(i,chi,aa,aav);

      //mj check for covalent bond
      if (  dont_eval_exclude || !exclude_interaction( (hetatom_id+1), atom1, aa, aav ) ) {

        //mj call pairderiv
        pairderiv(coord(1,atom1),het_coord(1),
                  fullatom_type(atom1,aa,aav),molecule.atom_vector[hetatom_id]->get_ros_atom_type(),dlj,datr,drep,
                  dsolv);

        //mj weight and sum
        dE_dr = datr*pack_wts.Wlig_atr() + drep*pack_wts.Wlig_rep() + dsolv*pack_wts.Wlig_sol();

        //mj accumulate change in f1, f2
        do_f1_f2_formula(coord(1,atom1),het_coord(1),dE_dr,
                         f1,f2);
      }
    }
  }
}

// =============================================================================
//     Computes VDW energy between AA h2o's and LIGAND for repacking (pack.cc)
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_sc_ha_h2oE
///
/// @brief
///     Computes VDW energy between AA h2o's and LIGAND for repacking (pack.cc)
///
/// @detailed
///
/// @param  aaid - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  coord - [in/out]? -
/// @param  h2oE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_sc_ha_h2oE(
  int const aaid,
  int const aa,
  int const aav,
  FArray2Da_float coord,
  float & h2oE,
  Ligand & molecule
)
{
  using namespace aaproperties_pack;
  using namespace interface;
  using namespace ligand;
  using namespace param;

  coord.dimension( 3, MAX_ATOM() );

//mj local variables
  float d2;
  bool neighbor;
  triplet::xyzVector_float hetatom_id_coordinate;
  FArray1D_float het_coord(3);

//mj set variables to 0
  h2oE = 0.0;

//mj function body
  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;
  if ( !int_res(aaid) ) return;

//mj  loop over all hetero atoms
  for ( int hetatom_id = 0, ne = nh2o( aa, aav ); hetatom_id < int(molecule.atom_vector.size()); ++hetatom_id )  {
    if ( !molecule.atom_vector[hetatom_id]->Is_Virtual() ) {
      copy_to_FArray(molecule.atom_vector[hetatom_id]->get_coordinates(),het_coord);
      //mj compute distance^2 of hetero atom to amino acid
      hetero_atom_amino_acid_distance(aa,coord(1,1),
        het_coord(1),d2,neighbor);

      //mj if distance smaller aa - GLYCINE cutoff
      if ( neighbor ) {
        for ( int n = 1; n <=  ne; ++n ) {
          fast_pairenergy_h2o( fullatom_type(h2opos(n,aa,aav),aa,aav),
           coord(1,h2opos(n,aa,aav)), molecule.atom_vector[hetatom_id]->get_ros_atom_type(),
           het_coord(1), h2oE );
        }
      }
    }
  }
}

// =============================================================================
//     Computes H-BOND energies between AA and LIGAND for repacking (pack.cc)
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_sc_ha_hbondE
///
/// @brief
///     Computes H-BOND energies between AA and LIGAND for repacking (pack.cc)
///
/// @detailed
///
/// @param  aaid - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  coord - [in/out]? -
/// @param  hbondE - [in/out]? -
/// @param  eval_bb - [in/out]? -
/// @param  eval_sc - [in/out]? -
/// @param  update_deriv - [in/out]? -
/// @param  f1 - [in/out]? -
/// @param  f2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_sc_ha_hbondE(
  int const aaid,
  int const aa,
  int const aav,
  FArray2Da_float coord,
  float & hbondE,
  bool eval_bb,
  bool eval_sc,
  bool update_deriv,
  FArray2DB_float & f1,
  FArray2DB_float & f2,
  Ligand & molecule
)
{
  using namespace aaproperties_pack;
  using namespace interface;
  using namespace hbonds::hbond_db;
  using namespace ligand;
  using namespace param;
  using namespace param_pack;

  coord.dimension( 3, MAX_ATOM() );

  //mj local
  int hybrid, aatm, datm, hatm, batm, Oatm; // batm - aatm -> hatm - datm
  FArray2D_float deriv( 3, 2 );
  float dis2, energy;
  bool neighbor;
  triplet::xyzVector_double aatm_coordinate, batm_coordinate;
  triplet::xyzVector_double datm_coordinate, hatm_coordinate;
  FArray1D_float het_coord(3), het_b_coord(3), het_d_coord(3), het_h_coord(3);

  // initialize derivative
  if ( update_deriv ) {
    f1 = 0.0;
    f2 = 0.0;
  }

  //mj set variables zero
  hbondE = 0.0;

  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;
  if ( !int_res(aaid) ) return;

  //mj AA donor / LIGAND acceptor
  for ( int anum = 1, hnume = nH_polar(aa,aav); anum <= molecule.hetero_atom_hbond_acc_count; ++anum ) {
   // loop over all ligand aceptors

    //mj get acceptor atom and base
    aatm = molecule.hetero_atom_hbond_acc(1,anum);
    batm = molecule.hetero_atom_hbond_acc(2,anum);
    hybrid = hybridization(molecule.atom_vector[aatm]->get_ros_atom_type());

    //mj compute distance hetero atom - AA
    copy_to_FArray(molecule.atom_vector[aatm]->get_coordinates(),het_coord);
    hetero_atom_amino_acid_distance(aa,coord,het_coord(1),dis2,
     neighbor);

    //mj go on, if close enough
    if ( neighbor ) {            // compute energy only if close in space
      for ( int hnum = 1; hnum <= hnume; ++hnum ) {
       // loop over all polar H's in a AA

        //mj get hydrogen atom and donor
        hatm = Hpos_polar(hnum,aa,aav);
        datm = Hbase(hatm,aa,aav);

        //mj do only compute eval_sc or eval_bb says
        if ( ( !atom_is_backbone(datm,aa,aav) && eval_sc ) ||
         ( atom_is_backbone(datm,aa,aav) && eval_bb ) ) {
          copy_to_FArray(molecule.atom_vector[batm]->get_coordinates(),het_b_coord);
          //mj compute energy of hydrogen bond
          hb_energy_deriv(0,0,coord(1,datm),coord(1,hatm),
           het_coord(1),het_b_coord(1),
           het_b_coord(1),
           /* dont care - second base for ring nitrogens */ hybrid,true,
           energy,update_deriv,deriv,
           /*pb right now no specific treatment for charged acceptor oxygen in ligands*/
           0,0,false,false);

          if ( energy < MAX_HB_ENERGY ) {          // only count if h bond
            ++molecule.hetero_atom_hbond_acc(3,anum);
            hbondE += energy;
            if ( update_deriv ) {
              for ( int k = 1; k <= 3; ++k ) { // accumulate f1 & f2
                f1(k,hatm) += pack_wts.Wlig_hb() * deriv(k,1);
                f2(k,hatm) += pack_wts.Wlig_hb() * deriv(k,2);
              }
            }
          }                   // end only count if h bond
        }                      // end compute sc h bonds only if fullatom
      }                         // end loop over all polar H's in a AA
    }                            // end compute energy only if close in space
  }                               // end loop over all ligand aceptors

  //mj AA acceptor / LIGAND donor
  for ( int hnum = 1; hnum <= molecule.hetero_atom_hbond_don_count; ++hnum ) {
   // loop over all ligand donors

    //mj get acceptor atom and base
    hatm = molecule.hetero_atom_hbond_don(1,hnum);
    datm = molecule.hetero_atom_hbond_don(2,hnum);
    Oatm = molecule.hetero_atom_hbond_don(4,hnum);

    //mj compute dstance hetero atom - AA
      copy_to_FArray(molecule.atom_vector[datm]->get_coordinates(),het_d_coord);

    hetero_atom_amino_acid_distance(aa,coord,het_d_coord(1),dis2,
                    neighbor);

    //mj go on, if close enough
    if ( neighbor ) {                 // compute energy only if close in space
      for ( int anum = 1, anume = nacceptors(aa,aav); anum <= anume; ++anum ) {
       // loop over all aceptors in a AA

        //mj get hydrogen atom and donor
        aatm = accpt_pos(anum,aa,aav);
        batm = abase(aatm,aa,aav);
        hybrid = hybridization(fullatom_type(aatm,aa,aav));

        //mj do only compute eval_sc or eval_bb says
        if ( ( !atom_is_backbone(aatm,aa,aav) && eval_sc ) ||
         ( atom_is_backbone(aatm,aa,aav) && eval_bb ) ) {
         // compute sc h bonds only if fullatom
          copy_to_FArray(molecule.atom_vector[hatm]->get_coordinates(),het_h_coord);

          // if not -O-H
          if ( Oatm == -1 || molecule.fix_ligand_HO ) {
            //mj compute energy of hydrogen bond
            hb_energy_deriv(0,0,het_d_coord(1),
                            het_h_coord(1),coord(1,aatm),coord(1,batm),
                            coord(1,batm), /* dont care - second base for ring nitrogens */
                            hybrid,true,energy,update_deriv,deriv,
                            /*pb right now no specific treatment for charged acceptor oxygen in ligands*/
                            0,0,false,false);
          }

          // in case of -O-H optimize H position in ligand first
          else {

            ////////////////////////////////////////////////
            //
            //                            X
            //                   o--H
            //                   | /
            //                   |/ 0.98A
            //                   O 109.5grad
            //                   |
            //                   |
            //                   B
            //
            ////////////////////////////////////////////////

            // find optimized H positon
            double const sin109 = { 0.9426 * 0.98 }; // sin( 70.5) * bondlength( O-H)
            double const cos109 = { 0.3338 * 0.98 }; // cos( 70.5) * bondlength( O-H)
             static FArray1D_float opt_Hpos( 3 );
            triplet::xyzVector_double opt_Hpos_vect;
//kwk reset as xyzVectors    static FArray1D_float B_O_vect( 3 );
            triplet::xyzVector_double B_O_vect;
//kwk            static FArray1D_float O_X_vect( 3 );
            triplet::xyzVector_double O_X_vect;
//kwk            static FArray1D_float ort_vect( 3 );
            triplet::xyzVector_double ort_vect;
            triplet::xyzVector_double aatm_vect;
            aatm_vect.assign(coord(1,aatm), coord(2,aatm), coord(3,aatm));
            // compute B-O and O-X vector
            B_O_vect= molecule.atom_vector[datm]->get_coordinates()
                   - molecule.atom_vector[Oatm]->get_coordinates();
            O_X_vect= aatm_vect - molecule.atom_vector[datm]->get_coordinates();
            // normalize B-O and O-X vector
            B_O_vect.normalize();
            O_X_vect.normalize();
            // compute cos angle between B-O and O-X vector and orthogonal vector to B-O in plane B-O-X
            double cosangle = inner_product( B_O_vect, O_X_vect );
            cosangle = std::min( 1.0, std::max( -1.0, cosangle ) );
            ort_vect = O_X_vect - cosangle * B_O_vect;
            ort_vect.normalize();
            // compute H position in B-O-X plane
            opt_Hpos_vect = molecule.atom_vector[datm]->get_coordinates()
                      + B_O_vect *  cos109
                      + ort_vect *  sin109;
            opt_Hpos(1)=opt_Hpos_vect.x();
            opt_Hpos(2)=opt_Hpos_vect.y();
            opt_Hpos(3)=opt_Hpos_vect.z();
            //mj debug
            //std::cout << "ATOM      1  H   XXX X   1    " << F( 8, 3, opt_Hpos( 1 )) << F( 8, 3, opt_Hpos( 2 )) << F( 8, 3, opt_Hpos( 3 )) << std::endl;
            //std::cout << "ATOM      2  O   XXX X   1    " << F( 8, 3, hetero_atom_coord(1,datm)) << F( 8, 3, hetero_atom_coord(2,datm)) << F( 8, 3, hetero_atom_coord(3,datm)) << std::endl;
            //std::cout << "ATOM      3  C   XXX X   1    " << F( 8, 3, hetero_atom_coord(1,Oatm)) << F( 8, 3, hetero_atom_coord(2,Oatm)) << F( 8, 3, hetero_atom_coord(3,Oatm)) << std::endl;
            //std::cout << "ATOM      4  O   XXX X   1    " << F( 8, 3, coord(1,aatm)) << F( 8, 3, coord(2,aatm)) << F( 8, 3, coord(3,aatm)) << std::endl;

            //mj compute energy of hydrogen bond
            copy_to_FArray(molecule.atom_vector[datm]->get_coordinates(), het_d_coord);
            hb_energy_deriv(0,0,het_d_coord(1), opt_Hpos(1),
              coord(1,aatm), coord(1,batm), coord(1,batm), /* dont care - second base for ring nitrogens */
              hybrid,true,energy,update_deriv,deriv,
              /*pb right now no specific treatment for charged acceptor oxygen in ligands*/
              0,0,false,false);
          }

          if ( energy < MAX_HB_ENERGY ) {         // only count if h bond
            ++molecule.hetero_atom_hbond_don(3,hnum);
            hbondE += energy;
            if ( update_deriv ) {
              for ( int k = 1; k <= 3; ++k ) { // accumulate f1 & f2
                f1(k,aatm) -= pack_wts.Wlig_hb() * deriv(k,1);
                f2(k,aatm) -= pack_wts.Wlig_hb() * deriv(k,2);
              }
            }
          }                   // end only count if h bond
        }                      // end compute sc h bonds only if fullatom
      }                         // end loop over all acceptors in a AA
    }                            // end compute energy only if close in space
  }                               // end loop over all ligand donors
}

// =============================================================================
//     Computes ELECTROSTATICS between AA and LIGAND for repacking
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_sc_ha_coulombE
///
/// @brief
///     Computes ELECTROSTATICS between AA and LIGAND for repacking
///
/// @detailed
///
/// @param  aaid - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  coord - [in/out]? -
/// @param  elecE - [in/out]? -
/// @param  eval_bb - [in/out]? -
/// @param  eval_sc - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_sc_ha_coulombE(
  int const aaid,
  int const aa,
  int const aav,
  FArray2Da_float protein_coord,
  Ligand * ligand,
  float & elecE,
  bool eval_bb,
  bool eval_sc
)
{
  using namespace aaproperties_pack;
  using namespace etable;
  using namespace interface;
  using namespace ligand;
  using namespace param;
  using namespace template_pack;

  protein_coord.dimension( 3, MAX_ATOM() );

//mj local variables
  size_t hetatom_id,atom1;
  size_t start_id,end_id;
  float d2, E, limit2, scale_neighbor;

//mj set variables to 0
  elecE = 0.0;

//mj function body
  if ( !get_ligand_flag() || ligand->atom_vector.size() == 0 ) return;
  if ( !int_res(aaid) ) return;

  bool eval_exclude = get_enzyme_flag(); // do we need to call exclude_interaction?

//mj loop over all hetero atoms
  for ( hetatom_id = 0; hetatom_id < ligand->atom_vector.size(); ++hetatom_id )
    {
      // faster access to the current atom
      ligand::Ligand_AtomOP hetatm = ligand->atom_vector[hetatom_id];
      if ( hetatm->Is_Virtual() ) continue;  // skip fake atoms

      // hetero atom coordinates
      triplet::xyzVector_double hetatm_coord(hetatm->get_coordinates());

      //mj compute first and last atom to use depending on eval_bb and eval_sc
      start_id = eval_bb ? 1 : 5;
      end_id   = eval_sc ? natoms(aa,aav): 4;

      //mj loop over all amino acids atoms
      for ( atom1 = start_id; atom1 <= end_id; ++atom1 )
        {
          //mj check for covalent bond and skip those interactions
          if ( eval_exclude && exclude_interaction( (hetatom_id+1), atom1, aa, aav ) ) continue;

          // amino acid atom coordinates
          triplet::xyzVector_double aa_atom_coord ( protein_coord(1,atom1),
                                                    protein_coord(2,atom1),
                                                    protein_coord(3,atom1) );

          //mj compute distance squared
          d2=distance_squared(hetatm_coord, aa_atom_coord);

          //mj compute limt
          limit2 = square( fa_lj_radius(hetatm->get_ros_atom_type()) +
                           fa_lj_radius(fullatom_type(atom1,aa,aav)) );

          //mj compute energy
          simple_elec(atomic_charge(atom1,aa,aav),hetatm->get_partial_charge(),d2,
                      limit2,E);

          //mj sum up
          elecE += E;
        }
    }

  //mj neighbor info -- weight by residue burial to downweight exposed residues
  scale_neighbor = ( 1.0 / 24.0 ) * neighbors(aaid);
  if ( scale_neighbor > 1.0 ) scale_neighbor = 1.0;
  elecE *= scale_neighbor;
}

// =============================================================================
//     Computes WARSHEL electrostatics with distant dependent dielectric constant
//     E = 322.0637*q1*q2/r/e(r)
//     if ( r < 3 ) e(r) = 16.55
//     else         e(r) = 1 + 60*(1-exp(-0.1*r))
//     Warshel, A. Russell, S. T., Q. Rev. Biophys., 1984, 17, 283-422
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin warshel_elec
///
/// @brief
///     Computes WARSHEL electrostatics with distant dependent dielectric constant
///
/// @detailed
///     E = 322.0637*q1*q2/r/e(r)
///     if ( r < 3 ) e(r) = 16.55
///     else         e(r) = 1 + 60*(1-exp(-0.1*r))
///
/// @param  q1 - [in/out]? -
/// @param  q2 - [in/out]? -
/// @param  r2 - [in/out]? -
/// @param  limit2 - [in/out]? -
/// @param  E - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///     Warshel, A. Russell, S. T., Q. Rev. Biophys., 1984, 17, 283-422
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
warshel_elec(
  float q1,
  float q2,
  float r2,
  float limit2,
  float & E
)
{
  float r;
  if ( r2 <= limit2 ) {
    r = std::sqrt( limit2 );
  } else {
    r = std::sqrt( r2 );
  }

  float die;
  if ( r < 3.0 ) {
    die = 0.051387f;
  } else {
    die = 0.189404f - 0.186298f * std::exp( -0.1f * r );
  }

  E = q1 * q2 / r / die;
}

// =============================================================================
//     Computes simple distant dependent dielectric constant
//     E = 322.0637*q1*q2/r/e(r)
//     e(r) = 6*r
//     Brooks, B. R. et. al., J. Comput. Chem., 1983, 4, 187-217
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin simple_elec
///
/// @brief
///     Computes simple distant dependent dielectric constant
///
/// @detailed
///     E = 322.0637*q1*q2/r/e(r)
///     e(r) = 6*r
///
/// @param  q1 - [in/out]? -
/// @param  q2 - [in/out]? -
/// @param  r2 - [in/out]? -
/// @param  limit2 - [in/out]? -
/// @param  E - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///     Brooks, B. R. et. al., J. Comput. Chem., 1983, 4, 187-217
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
simple_elec(
  float q1,
  float q2,
  float r2,
  float limit2,
  float & E
)
{
//mj function body
  E = q1 * q2 * 53.676865f / ( r2 <= limit2 ? limit2 : r2 );
  //  if ( r2 <= limit2 ) {
  //    E = q1 * q2 / limit2 / 0.018630f;
  //  } else {
  //    E = q1 * q2 / r2 / 0.018630f;
  //  }

}


// =============================================================================
// Get ligand ligand energy
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin get_ligand_ligandE
///
/// @brief
///     Calculates Energy cetween between two ligands. Returns solvation energy,
/// van der Waals energies, hydrogen bond energy, and electrostatic energy
///
/// @detailed
///
/// @param [in] Ligand & molecule1
/// @param [in] Ligand & molecule2
/// @param [out] float & solvE
/// @param [out] float & atrE
/// @param [out] float & repE
/// @param [out] float & hbE
/// @param [out] float & elecE
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors kaufmann
///
/// @last_modified 5-26-2006
/////////////////////////////////////////////////////////////////////////////////
float
get_ligand_ligandE(
  Ligand & molecule1,
  Ligand & molecule2,
  float & solvE,
  float & atrE,
  float & repE,
  float & hbE,
  float & elecE
){

//kwk check ligand flag
if( !get_ligand_flag()){
  std::cerr << "get_ligand_ligandE() call outside of ligand mode." << std::endl;
  return 9999.0;
}
//kwk check that molecule1 and molecule2 are different
if( &molecule1 == &molecule2 ){
  std::cerr << "get_ligand_ligandE() called with same ligand object twice" << std::endl;
  return 9999.0;
}

//kwk The call to fast pair energy returns a distance
//kwk If the distance is less than 12 A for any of the atoms then
//kwk the rest of the energy functions will be calculated otherwise
//kwk otherwise the the function will exit returning zero for all the energies

solvE=0.0;
atrE=0.0;
repE=0.0;
hbE=0.0;
elecE=0.0;

FArray1D_float atom_1_coord(3),atom_2_coord(3);
bool is_neighbor=false;

for( ligand::Atom_Itr m1_atom=molecule1.atom_vector.begin();
  m1_atom!=molecule1.atom_vector.end(); m1_atom++){

  if( (*m1_atom)->Is_Virtual() ){ continue;}
  copy_to_FArray((*m1_atom)->get_coordinates(),atom_1_coord);

  for( ligand::Atom_Itr m2_atom=molecule2.atom_vector.begin();
    m2_atom!=molecule2.atom_vector.end(); m2_atom++){

    if( (*m2_atom)->Is_Virtual() ){ continue;}
    copy_to_FArray((*m2_atom)->get_coordinates(),atom_2_coord);
    float solE_tmp=0.0;
    float atrE_tmp=0.0;
    float repE_tmp=0.0;
    float distance_tmp=0.0;
    float cp_weight=1.0;
    fast_pairenergy(atom_1_coord,atom_2_coord,
            (*m1_atom)->get_ros_atom_type(),
            (*m2_atom)->get_ros_atom_type(),
            solE_tmp,atrE_tmp,repE_tmp,distance_tmp,cp_weight
          );

    solvE+=solE_tmp;
    atrE+=atrE_tmp;
    repE+=repE_tmp;
    if( distance_tmp < 12 ) {is_neighbor=true;}

  }//kwk m2_atom for loop
}//kwk m1_atom for loop

//kwk If no atoms are closer than 12 A exit the function
if( !is_neighbor ){ solvE=0.0; atrE=0.0; repE=0.0; return 0.0;}

//kwk calculate hbondE


//kwk calculate hb energy
// the bulk of this code is borrowed directly from ligand.cc
// the update_deriv option is set to false
// However in the event that this information is needed
// it can be readily added.

FArray1D_float proton_coord(3), donor_coord(3);
FArray1D_float accept_coord(3), acceptbase_coord(3);
int hybrid;
float energy;
FArray2D_float deriv;
bool update_deriv= { false };

//kwk first calculate with molecule1 acceptors molecule2 donors
for(int current_acceptor=1; current_acceptor<molecule1.hetero_atom_hbond_acc_count;
  current_acceptor++){

  //get acceptor atom and base coordinates
  copy_to_FArray(molecule1.atom_vector[
    molecule1.hetero_atom_hbond_acc(1,current_acceptor)]->get_coordinates(),
    accept_coord);

  copy_to_FArray(molecule1.atom_vector[
    molecule1.hetero_atom_hbond_acc(2,current_acceptor)]->get_coordinates(),
    acceptbase_coord);

  hybrid = aaproperties_pack::hybridization(molecule1.atom_vector[
    molecule1.hetero_atom_hbond_acc(1,current_acceptor)]->get_ros_atom_type());

  for(int current_donor=1; current_donor<molecule2.hetero_atom_hbond_don_count;
    current_donor++){

      copy_to_FArray(molecule2.atom_vector[
        molecule2.hetero_atom_hbond_don(1,current_donor)]->get_coordinates(),
        proton_coord);
      copy_to_FArray(molecule2.atom_vector[
        molecule2.hetero_atom_hbond_don(2,current_donor)]->get_coordinates(),
        donor_coord);


    // if not -O-H
        if ( molecule2.hetero_atom_hbond_don(4,current_donor) == -1 ||
          molecule2.fix_ligand_HO ) {
          //mj compute energy of hydrogen bond
          hb_energy_deriv(0,0,donor_coord(1),
            proton_coord(1),accept_coord(1),acceptbase_coord(1),
            acceptbase_coord(1),// in this case we don't care about 2nd bases
            hybrid,true,energy,update_deriv,deriv,0,0,false,false);

          hbE += energy;
        }

        // in case of -O-H optimize H position in ligand first
        else {

          ////////////////////////////////////////////////
          //
          //                            X
          //                   o--H
          //                   | /
          //                   |/ 0.98A
          //                   O 109.5grad
          //                   |
          //                   |
          //                   B
          //
          ////////////////////////////////////////////////

          // find optimized H positon
          double const sin109 = { 0.9426 * 0.98 };
          // sin( 70.5) * bondlength( O-H)
          double const cos109 = { 0.3338 * 0.98 };
          // cos( 70.5) * bondlength( O-H)
           static FArray1D_float opt_Hpos( 3 );
          triplet::xyzVector_double opt_Hpos_vect;
//kwk reset as xyzVectors    static FArray1D_float B_O_vect( 3 );
          triplet::xyzVector_double B_O_vect;
//kwk            static FArray1D_float O_X_vect( 3 );
          triplet::xyzVector_double O_X_vect;
//kwk            static FArray1D_float ort_vect( 3 );
          triplet::xyzVector_double ort_vect;
          triplet::xyzVector_double aatm_vect;

          aatm_vect=molecule1.atom_vector[molecule1.hetero_atom_hbond_acc(1,current_acceptor)]
            ->get_coordinates();
          // compute B-O and O-X vector

          B_O_vect= molecule2.atom_vector[molecule2.hetero_atom_hbond_don(2,current_donor)]
            ->get_coordinates()
            - molecule2.atom_vector[molecule2.hetero_atom_hbond_don(4,current_donor)]
            ->get_coordinates();

          O_X_vect= aatm_vect - molecule2.atom_vector[
            molecule2.hetero_atom_hbond_don(2,current_donor)]->get_coordinates();

          // normalize B-O and O-X vector
          B_O_vect.normalize();
          O_X_vect.normalize();
          double cosangle = inner_product( B_O_vect, O_X_vect );
          cosangle = std::min( 1.0, std::max( -1.0, cosangle ) );
          ort_vect = O_X_vect - cosangle * B_O_vect;
          ort_vect.normalize();
          // compute H position in B-O-X plane
          opt_Hpos_vect = molecule2.atom_vector[
            molecule2.hetero_atom_hbond_don(2,current_donor)]->get_coordinates()
                    + B_O_vect *  cos109
                    + ort_vect *  sin109;
          opt_Hpos(1)=opt_Hpos_vect.x();
          opt_Hpos(2)=opt_Hpos_vect.y();
          opt_Hpos(3)=opt_Hpos_vect.z();
          //mj compute energy of hydrogen bond
          copy_to_FArray(molecule2.atom_vector[
            molecule2.hetero_atom_hbond_don(2,current_donor)]->get_coordinates(),
            donor_coord);
          hb_energy_deriv(0,0,donor_coord(1), opt_Hpos(1),
            accept_coord(1), acceptbase_coord(1), acceptbase_coord(1),
            //
            hybrid,true,energy,update_deriv,deriv,0,0,false,false);

          hbE += energy;
        }
  }//current_donor loop
}//current_acceptor loop

//kwk now calculate with molecule2 acceptors molecule1 donors
for(int current_acceptor=1; current_acceptor<molecule2.hetero_atom_hbond_acc_count;
  current_acceptor++){

  //get acceptor atom and base coordinates
  copy_to_FArray(molecule2.atom_vector[
    molecule2.hetero_atom_hbond_acc(1,current_acceptor)]->get_coordinates(),
    accept_coord);

  copy_to_FArray(molecule2.atom_vector[
    molecule2.hetero_atom_hbond_acc(2,current_acceptor)]->get_coordinates(),
    acceptbase_coord);

  hybrid = aaproperties_pack::hybridization(molecule2.atom_vector[
    molecule2.hetero_atom_hbond_acc(1,current_acceptor)]->get_ros_atom_type());

  for(int current_donor=1; current_donor<molecule1.hetero_atom_hbond_don_count;
    current_donor++){

      copy_to_FArray(molecule1.atom_vector[
        molecule1.hetero_atom_hbond_don(1,current_donor)]->get_coordinates(),
        proton_coord);
      copy_to_FArray(molecule1.atom_vector[
        molecule1.hetero_atom_hbond_don(2,current_donor)]->get_coordinates(),
        donor_coord);


    // if not -O-H
        if ( molecule1.hetero_atom_hbond_don(4,current_donor) == -1 ||
          molecule1.fix_ligand_HO ) {
          //mj compute energy of hydrogen bond
          hb_energy_deriv(0,0,donor_coord(1),
            proton_coord(1),accept_coord(1),acceptbase_coord(1),
            acceptbase_coord(1),// in this case we don't care about 2nd bases
            hybrid,true,energy,update_deriv,deriv,0,0,false,false);
          if ( energy < hbonds::hbond_db::MAX_HB_ENERGY ) {hbE += energy;}
        }

        // in case of -O-H optimize H position in ligand first
        else {

          ////////////////////////////////////////////////
          //
          //                            X
          //                   o--H
          //                   | /
          //                   |/ 0.98A
          //                   O 109.5grad
          //                   |
          //                   |
          //                   B
          //
          ////////////////////////////////////////////////

          // find optimized H positon
          double const sin109 = { 0.9426 * 0.98 };
          // sin( 70.5) * bondlength( O-H)
          double const cos109 = { 0.3338 * 0.98 };
          // cos( 70.5) * bondlength( O-H)
           static FArray1D_float opt_Hpos( 3 );
          triplet::xyzVector_double opt_Hpos_vect;
//kwk reset as xyzVectors    static FArray1D_float B_O_vect( 3 );
          triplet::xyzVector_double B_O_vect;
//kwk            static FArray1D_float O_X_vect( 3 );
          triplet::xyzVector_double O_X_vect;
//kwk            static FArray1D_float ort_vect( 3 );
          triplet::xyzVector_double ort_vect;
          triplet::xyzVector_double aatm_vect;

          aatm_vect=molecule2.atom_vector[molecule2.hetero_atom_hbond_acc(1,current_acceptor)]
            ->get_coordinates();
          // compute B-O and O-X vector

          B_O_vect= molecule1.atom_vector[molecule1.hetero_atom_hbond_don(2,current_donor)]
            ->get_coordinates()
            - molecule1.atom_vector[molecule1.hetero_atom_hbond_don(4,current_donor)]
            ->get_coordinates();

          O_X_vect= aatm_vect - molecule1.atom_vector[
            molecule1.hetero_atom_hbond_don(2,current_donor)]->get_coordinates();

          // normalize B-O and O-X vector
          B_O_vect.normalize();
          O_X_vect.normalize();
          double cosangle = inner_product( B_O_vect, O_X_vect );
          cosangle = std::min( 1.0, std::max( -1.0, cosangle ) );
          ort_vect = O_X_vect - cosangle * B_O_vect;
          ort_vect.normalize();
          // compute H position in B-O-X plane
          opt_Hpos_vect = molecule1.atom_vector[
            molecule1.hetero_atom_hbond_don(2,current_donor)]->get_coordinates()
                    + B_O_vect *  cos109
                    + ort_vect *  sin109;
          opt_Hpos(1)=opt_Hpos_vect.x();
          opt_Hpos(2)=opt_Hpos_vect.y();
          opt_Hpos(3)=opt_Hpos_vect.z();
          //mj compute energy of hydrogen bond
          copy_to_FArray(molecule1.atom_vector[
            molecule1.hetero_atom_hbond_don(2,current_donor)]->get_coordinates(),
            donor_coord);
          hb_energy_deriv(0,0,donor_coord(1), opt_Hpos(1),
            accept_coord(1), acceptbase_coord(1), acceptbase_coord(1),
            //
            hybrid,true,energy,update_deriv,deriv,0,0,false,false);
          if ( energy < hbonds::hbond_db::MAX_HB_ENERGY ) {hbE += energy;}
        }
  }//current_donor loop
}//current_acceptor loop

//kwk end calculate hbond energy. note to self can this code be centralized some how? It now occurs in three different spots

//kwk calculate electrostatics energy
float elecE_tmp=0.0;
if( !param_pack::packer_logical::gen_born ){
  for(ligand::Atom_Itr c_atom=molecule1.atom_vector.begin();c_atom!=molecule1.atom_vector.end();
      c_atom++){
    if(!((*c_atom)->Is_Virtual())){
    for(ligand::Atom_Itr n_atom=molecule2.atom_vector.begin();
      n_atom!=molecule2.atom_vector.end();n_atom++){
      //compute distance
      if(!((*n_atom)->Is_Virtual())){
        float d2,limit2;
        d2=distance_squared((*c_atom)->get_coordinates(),
                  (*n_atom)->get_coordinates());
        //mj compute limit
        limit2 = ( etable::fa_lj_radius((*c_atom)->get_ros_atom_type()) +
                   etable::fa_lj_radius((*n_atom)->get_ros_atom_type()))*
                 ( etable::fa_lj_radius((*c_atom)->get_ros_atom_type()) +
                   etable::fa_lj_radius((*n_atom)->get_ros_atom_type()));

        //mj compute energy
        simple_elec((*c_atom)->get_partial_charge(),
                  (*n_atom)->get_partial_charge(),d2,limit2,elecE_tmp);

        elecE+=elecE_tmp;
        elecE_tmp=0.0;
      }}
    }}

}else{
//kwk if gen_born
FArray2D_float het_atom_coord1, het_atom_coord2;
FArray1D_float het_atom_charge1, het_atom_charge2;
FArray1D_int het_atom_type1, het_atom_type2;
molecule1.get_FArray2D_of_coordinates(molecule1.atom_vector,het_atom_coord1);
molecule1.get_FArray1D_of_charge(molecule1.atom_vector, het_atom_charge1);
molecule1.get_FArray1D_of_atom_type(molecule1.atom_vector, het_atom_type1);
molecule2.get_FArray2D_of_coordinates(molecule2.atom_vector,het_atom_coord2);
molecule2.get_FArray1D_of_charge(molecule2.atom_vector, het_atom_charge2);
molecule2.get_FArray1D_of_atom_type(molecule2.atom_vector, het_atom_type2);
float const tempgbE = gb_get_ligand_elecE(molecule1.atom_vector.size(), param::HETERO_ATOM_MAX(),
              het_atom_coord1( 1, 1 ), molecule1.hetero_born_radius( 1 ),
              het_atom_charge1( 1 ), het_atom_type1( 1 ),
              molecule2.atom_vector.size(), param::HETERO_ATOM_MAX(),
              het_atom_coord2( 1, 1 ),molecule2.hetero_born_radius( 1 ),
              het_atom_charge2( 1 ), het_atom_type2( 1 ), false );

elecE += tempgbE;
}//kwk end electrostatics
//kwk note to self need to fit these weights Wlig_hb * Wlig_cou * Wlig_cou * Wlig_sol * Wlig_atr * Wlig_rep *

return param_pack::pack_wts.Wlig_hb() * hbE
  + param_pack::pack_wts.Wlig_cou() * elecE
  + param_pack::pack_wts.Wlig_sol() * solvE
  + param_pack::pack_wts.Wlig_atr() * atrE
  + param_pack::pack_wts.Wlig_rep() * repE;
}// function end



// =============================================================================
//     Detect ligand protein interface residues
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin detect_ligand_interface
///
/// @brief
///     Detect ligand protein interface residues
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
detect_ligand_interface(
  Ligand & molecule,
  bool multiple_conformations
)
{
  using namespace ligand;
  using namespace interface;
  using namespace misc;
  using namespace runlevel_ns;

  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;

//mj function body
  if ( !molecule.recompute_ligand_interface ) return;
  molecule.recompute_ligand_interface = false;

//mj erase old interface
  for ( int i = 1; i <= total_residue; ++i ) {
    int_res(i) = false;
  }
	int_res_list(1).clear();

//mj detect new interface
  for ( int i = 1; i <= total_residue; ++i ) { // start loop over all amino acids
    detect_ligand_interface_res(i,molecule, multiple_conformations);

    if ( int_res(i) ) {                     // start only if residue detected
			int_res_list(1).push_back(i);
      if ( runlevel >= verbose ) {
        std::cout << "residue " << i <<
         " detected for ligand interface" << std::endl;
      }
    }                                     // end only if residue detected
  }

//mj output result
  if ( runlevel >= verbose ) {
    std::cout << "detect_ligand_interface: interface contains " <<
			int_res_list(1).size() << " residues!" << std::endl;
  }

//mj emergency stop
	if ( !get_enzyme_flag() && int(int_res_list(1).size()) == 0 ) {
    std::cout << "DANGER - no aminoacids are in the interface - EXIT" << std::endl;
//mjtmp    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }
//mjend

}

// =============================================================================
//     Detect ligand center
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin ligand_center
///
/// @brief
///     Detect ligand center
///
/// @detailed
///
/// @param  center_coordinates - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
ligand_center(
  FArray1Da_float center_coordinates,
  Ligand & molecule
)
{
  using namespace ligand;
  using namespace param;

  center_coordinates.dimension( 3 );

  //center_coordinates = 0.0;
  // kwk transforming coordinates in xyzVector
  triplet::xyzVector_float c_coord(float(0.0),float(0.0),float(0.0));
//  c_coord(float(0.0),float(0.0),float(0.0)); // kwk initializing to zero.
  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;

  int atom_count_tmp = 0;

  for ( size_t i = 0; i < molecule.atom_vector.size(); ++i ) {
    if ( !molecule.atom_vector[i]->Is_Virtual() ) {
      ++atom_count_tmp;
      c_coord += molecule.atom_vector[i]->get_coordinates();
    }
  }

  assert( ( atom_count_tmp > 0 ) );
  c_coord=c_coord/atom_count_tmp;
  for (int i=1;i<4;i++){center_coordinates(i)=c_coord(i);}
}

// =============================================================================
//     Detect ligand protein interface residues
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin detect_ligand_interface_res
///
/// @brief
///     Detect ligand protein interface residues
///
/// @detailed
///
/// @param  aaid - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
detect_ligand_interface_res(
  int aaid,
  Ligand & molecule,
  bool multiple_conformations
)
{
  using namespace interface;
  using namespace ligand;
  using namespace misc;
  using namespace param;

//mj local variables
  float dis2;
  bool neighbor;
  FArray1D_float het_coord(3);
  triplet::xyzVector_float h_coord;
//mj function body
  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;

//mj erase old interface
  if ( int_res(aaid) ) {
    int_res(aaid) = false;
  }


	if(multiple_conformations){
					for( size_t k=0; k< molecule.ligand_conformations.size(); k++){
	          molecule.change_to_ligand_conformation(k);
	          for(ligand::Atom_Itr c_atom=molecule.atom_vector.begin();
	              c_atom!=molecule.atom_vector.end(); c_atom++){
	            if ( !(*c_atom)->Is_Virtual() &&
                   ((*c_atom)->get_ros_atom_type() <= 21
                   || (*c_atom)->get_ros_atom_type() >= 27) ){
						    if( !int_res(aaid)){
	                copy_to_FArray((*c_atom)->get_coordinates(),het_coord);
	                hetero_atom_amino_acid_distance(res(aaid),full_coord(1,1,aaid),
                      het_coord(1),dis2,neighbor);
      	          if ( neighbor ) {
                    int_res(aaid) = true;
                  }
						    }
	            }
	          }
					}
	}else{
		//mj loop over all hetero atoms
    for ( size_t j = 0; j < molecule.atom_vector.size(); ++j )
      //if ( hetero_atom_type( j ) <= MAX_REALTYPES )
      //lin loop through heavy atoms
      if ( !molecule.atom_vector[j]->Is_Virtual() &&
          (molecule.atom_vector[j]->get_ros_atom_type() <= 21 || molecule.atom_vector[j]->get_ros_atom_type() >= 27) )
        if ( !int_res(aaid) ) {

          copy_to_FArray(molecule.atom_vector[j]->get_coordinates(),het_coord);
          hetero_atom_amino_acid_distance(res(aaid),full_coord(1,1,aaid),
                    het_coord(1),dis2,neighbor);
          if ( neighbor ) {
            int_res(aaid) = true;
          }

        }
	}
}

// =============================================================================
//     Detect ligand protein interface residues
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin detect_number_of_contacts
///
/// @brief
///     Detect ligand protein interface residues
///
/// @detailed
///
/// @param[in]   aaid - in -
/// @param[out]   contacts - out -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
detect_number_of_contacts(
	FArray1DB_int const & res,
  FArray3DB_float const & full_coord,
	int const aaid,
  int & contacts,
  Ligand & molecule
)
{
  using namespace ligand;
  using namespace param;

//mj local variables
  float dis2;
  bool neighbor;
  FArray1D_float het_coord(3);
  triplet::xyzVector_float h_coord;

//mj function body
  contacts = 0;
  if ( ! get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;

//mj loop over all hetero atoms
  for ( size_t j = 0; j < molecule.atom_vector.size(); ++j ){
    if ( !molecule.atom_vector[j]->Is_Virtual() ) {
     copy_to_FArray(molecule.atom_vector[j]->get_coordinates(),het_coord);
     hetero_atom_amino_acid_distance( res(aaid), full_coord(1,1,aaid),
       het_coord, dis2, neighbor );
    if ( neighbor ) ++contacts;
  }}

}

// =============================================================================
//     count number of atom atom contacts
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_number_of_ligand_contacts
///
/// @brief
///     count number of atom atom contacts
///
/// @detailed
///
/// @param[out]   number_lig_contacts - out -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
calc_number_of_ligand_contacts(
  int & number_lig_contacts,
  Ligand & molecule
)
{
  using namespace aaproperties_pack;
  using namespace interface;
  using namespace ligand;
  using namespace misc;
  using namespace runlevel_ns;

//mj local variables
  float dis2;
  float const dis5_cutoff = { 25.0 };

//mj function body
  number_lig_contacts = 0;
  if ( ! get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;
  detect_ligand_interface(molecule, false);

//mj count contacts
  for ( int i = 1; i <= total_residue; ++i ) { // start loop over all amino acids
    if ( int_res(i) ) {                          // start only if residue detected
      for ( int atom1 = 1, atom1e = nheavyatoms(res(i),res_variant(i));
       atom1 <= atom1e; ++atom1 ) { // start loop over all heavy atoms
        for ( size_t j = 0; j < molecule.atom_vector.size(); ++j ) {
         // start loop over all hetero atoms
          if ( molecule.atom_vector[j]->get_ros_atom_cent() > 0 ) { // only non hydrogen atoms
            triplet::xyzVector_double full_coord_atom(full_coord(1,atom1,i),
                                                      full_coord(2,atom1,i),
                                                      full_coord(3,atom1,i)); // transforming coordinates
            dis2=distance_squared(molecule.atom_vector[j]->get_coordinates(), full_coord_atom);
//            distance2_bk( full_coord(1,atom1,i), hetero_atom_coord(1,j), dis2 );
            if ( dis2 <= dis5_cutoff )  ++number_lig_contacts; // distance < 5A
          }
        }
      }
    }
  }

//mj output result
  if ( runlevel >= verbose ) {
    std::cout << "calc_number_of_ligand_contacts: interface contains " <<
     SS( number_lig_contacts ) << " contacts!" << std::endl;
  }

}


// =============================================================================
//     compute derivative
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin derivative_ligand_pos
///
/// @brief
///     compute derivative
///
/// @detailed
///
/// @param  TR - [in/out]? -
/// @param  dE_dTR - [in/out]? -
/// @param  nfree - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
derivative_ligand_pos(
  FArray1Da_float TR,
  FArray1Da_float dE_dTR,
  int nfree,
  Ligand & molecule
)
{
  using namespace aaproperties_pack;
  using namespace docking_min;
  using namespace interface;
  using namespace ligand;
  using namespace misc;
  using namespace param_pack;
  using namespace scorefxns;
  using namespace param;

  TR.dimension( nfree );
  dE_dTR.dimension( nfree );

//mj local variables
  float dE_dr; // derivative of energy with respect to atom-atom distance
  FArray1D_float dr_dx( 3 );
   // derivative of atom-atom distance along each coord. axis
  FArray1D_float q( 3 ); // position of j-atom relative to center of rotation
  triplet::xyzVector_double hetatom_id_coordinate;
  triplet::xyzVector_double dxyz;
  triplet::xyzVector_double full_coord_atom;
  FArray1D_float het_coord(3);

//     for numerical derivative
//      bool gfrag;                 // success indicator
//      FArray1D_float TR_pert(6);  // TR vector with a perturbation
//      float f_plus, f_minus;      // values of func with a perturbation
//      float lj_plus, lj_minus;    // values of LJ energy with a perturbation
//      FArray1D_float deriv(6);    // numerically computed derivative, cf dE_dTR
//      FArray1D_float lj_deriv(6); // numerically computed derivative of LJ score
//      float func;                 // (function) function being minimized
//      float epsilon;              // small perturbation
//      float vmin,vmax;            // functions

  bool neighbor;
  float dis2, dlj, datr, drep, dsol;
  if ( !get_ligand_flag() || molecule.atom_vector.size() == 0 ) return;

  docking_unpack_position(TR);
  detect_ligand_interface(molecule,false);

//mj initialize the derivative vector
  for ( int i = 1; i <= 6; ++i ) {
    dE_dTR(i) = 0.0;
  }

//mj loop over residues in the interface
  for ( int i = 1; i <= total_residue; ++i ) { // start loop over residues
    if ( int_res(i) ) {                     // start if in interface
      for ( size_t j = 0; j < molecule.atom_vector.size(); ++j )
        if ( !molecule.atom_vector[j]->Is_Virtual()) {
       // start loop over hetero atoms

//mj compute distance
        copy_to_FArray(molecule.atom_vector[j]->get_coordinates(),het_coord);
        hetero_atom_amino_acid_distance(res(i),full_coord(1,1,i),
            het_coord(1),dis2,neighbor);

//mj go on if close enough
        if ( neighbor ) {                // start if close enough
          for ( int iat = 1, iate = natoms( res(i), res_variant(i) ); iat <= iate; ++iat ) {

//mj compute atom atom distance
            full_coord_atom.assign(full_coord(1,iat,i),full_coord(2,iat,i),full_coord(3,iat,i));
            dis2 = distance_squared(full_coord_atom, molecule.atom_vector[j]->get_coordinates());
//            float const dxyz_1 = full_coord(1,iat,i)-hetero_atom_coord(1,j);
//            float const dxyz_2 = full_coord(2,iat,i)-hetero_atom_coord(2,j);
//            float const dxyz_3 = full_coord(3,iat,i)-hetero_atom_coord(3,j);
//            dis2 =
//             ( dxyz_1 * dxyz_1 ) + ( dxyz_2 * dxyz_2 ) + ( dxyz_3 * dxyz_3 );

//mj if distance smaller 8A
            if ( dis2 < 64 ) {

//mj lookup and interpolate the derivative(s)

              pairderiv(full_coord(1,iat,i),het_coord(1),
                fullatom_type(iat,res(i),res_variant(i)),
                molecule.atom_vector[j]->get_ros_atom_type(),
                dlj,datr,drep,dsol);

              dE_dr = drep * pack_wts.Wlig_rep() * fa_rep_weight
                  + datr * pack_wts.Wlig_atr() * fa_atr_weight
                  + dsol * pack_wts.Wlig_sol() * fa_solv_weight;

//mj calculate directional components x_i/r
              dxyz=full_coord_atom-molecule.atom_vector[j]->get_coordinates();
              dxyz.normalize();
//              one_r = 1.0/std::sqrt(dis2);
              dr_dx(1) = dxyz.x();
              dr_dx(2) = dxyz.y();
              dr_dx(3) = dxyz.z();

//mj calculate the distance from the pivot point
              q(1) = het_coord(1)-center_of_rotation(1);
              q(2) = het_coord(2)-center_of_rotation(2);
              q(3) = het_coord(3)-center_of_rotation(3);

//mj sum derivatives
              accumulate_dE_dTR( dE_dTR, dE_dr, dr_dx, q );
            }
          }
        }
      }
    }
  }
}

// =============================================================================
//     compute simple centroid based VDW score
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
/// @begin ligand_centroid_vdw
///
/// @brief
///     compute simple centroid based VDW score
///
/// @detailed
///
/// @param  ligand_centroid_VDW_E - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
ligand_centroid_vdw(
  float & ligand_centroid_VDW_E,
  Ligand & molecule
 )
{
  using namespace ligand;
  using namespace misc;
  using namespace runlevel_ns;

//mj local
  float dis2;
  triplet::xyzVector_double hetatom_id_coordinate;
  FArray1D_float het_coord(3);
//mj function body
  ligand_centroid_VDW_E = 0.0;

//mj loop over all amino acids
  for ( int aa_id = 1; aa_id <= total_residue; ++aa_id ) {

//mj loop over all heteroatoms
    for ( size_t hetatom_id = 0; hetatom_id < molecule.atom_vector.size(); ++hetatom_id ) {

//mj check that atom is not hydrogen and not virtual
      if ( molecule.atom_vector[hetatom_id]->get_ros_atom_cent() != -1 ) {

//mj centroid atom clashes

//mj compute distance
        copy_to_FArray(molecule.atom_vector[hetatom_id]->get_coordinates(),het_coord);

        distance2_bk(centroid(1,aa_id),het_coord,dis2);

//mj compare with table
        dis2 = vdw::atom_vdw(atom_type_cen(aa_id),molecule.atom_vector[hetatom_id]->get_ros_atom_cent()) -dis2;

//mj sum up energy
        if ( dis2 > 0.0 ) {
          ligand_centroid_VDW_E += ( dis2 * dis2 ) /
           vdw::atom_vdw(atom_type_cen(aa_id),molecule.atom_vector[hetatom_id]->get_ros_atom_cent());
        }

//mj backbone atom check
        for ( int bbatom_id = 5*aa_id-4; bbatom_id <= 5*aa_id; ++bbatom_id ) {
          if ( atom_type(bbatom_id) != 0 ) { // gly c-beta

//mj compute distance
            distance2_bk(position(1,bbatom_id),het_coord,dis2);

//mj compare with table
            dis2 =
             vdw::atom_vdw(atom_type(bbatom_id),molecule.atom_vector[hetatom_id]->get_ros_atom_cent()) -
             dis2;

//mj sum up energy
            if ( dis2 > 0.0 ) {
              ligand_centroid_VDW_E += ( dis2 * dis2 ) /
               vdw::atom_vdw(atom_type(bbatom_id),molecule.atom_vector[hetatom_id]->get_ros_atom_cent());
            }
          }         // glycine check
        }
      }
    }
  }

  ligand_centroid_VDW_E *= 100.0; // historical
  ligand_centroid_VDW_E *= 0.008; // moved from evaluate_envpair
  if ( runlevel >= verbose ) std::cout << "ligand vdw score: " <<
   SS( ligand_centroid_VDW_E ) << std::endl;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin ligand_bb_vdw
///
/// @brief
///     compute ligand - protein backbone bumpscore (using centroid LJ)
///
/// @detailed
///
/// this function computes the same thing as old ligand_bb_vdw (which was buged)
/// using the centroid LJ potential
/// This function if the default used in ligand mode when backbone clashes need to
/// be monitored.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Alex Zanghellini
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
ligand_bb_vdw(
  float & ligand_bb_VDW_E,
  Ligand & molecule
)
{
  using namespace ligand;
  using namespace param;
  using namespace misc;
  using namespace runlevel_ns;
  using namespace aaproperties_pack;

  float dis2;
  triplet::xyzVector_float hetatom_id_coordinate;
  FArray1D_float het_coord(3);

  ligand_bb_VDW_E = 0.0;

//lin loop over all amino acids
  for ( int aa_id = 1; aa_id <= total_residue; ++aa_id ) {

//lin loop over all heteroatoms
    for ( size_t hetatom_id = 0; hetatom_id < molecule.atom_vector.size(); ++hetatom_id ) {

      // az skip virtual atoms and Hydrogens ( hetero_atom_cent( i ) <= 0 )
      if ( molecule.atom_vector[hetatom_id]->get_ros_atom_cent() > 0 ) {
          copy_to_FArray(molecule.atom_vector[hetatom_id]->get_coordinates(),het_coord);
          // backbone atom check
          for ( int bbatom_id = 1; bbatom_id <= 5; ++bbatom_id ) {

            // in centroid, bbatom_id = 3 is Cb, and we don't
            // want these clash evaluations
            if ( bbatom_id != 3 ) {

              distance2_bk( Eposition(1,bbatom_id, aa_id), het_coord(1),dis2);

              //mj compare with table
              dis2 =
                vdw::atom_vdw(atom_type(bbatom_id),
                  molecule.atom_vector[hetatom_id]->get_ros_atom_cent()) -
                  dis2;

              //mj sum up energy
              if ( dis2 > 0.0 ) {
                ligand_bb_VDW_E += ( dis2 * dis2 ) /
                  vdw::atom_vdw(atom_type(bbatom_id),molecule.atom_vector[hetatom_id]->get_ros_atom_cent());
              }
            }
          }
      }
    }
  }

  ligand_bb_VDW_E *= 100.0; // historical
  ligand_bb_VDW_E *= 0.008; // moved from evaluate_envpair
  if ( runlevel >= verbose ) std::cout << "ligand bb vdw score: " <<
   SS( ligand_bb_VDW_E ) << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin ligand_bb_fa_vdw
///
/// @brief
///     compute backbone bump score:
///     the score is computed for all hetero atom heavy atom
///     and with backbone N,Ca,C,O. fullatom version of ligand_bb_vdw
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// adapted from Brian Kuhlman's sc_bump and previous ligand_bb_vdw,
/// which has a sever bug
///
/// @references
///
/// @authors Alex Zanghellini
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
ligand_bb_fa_vdw(
  float & ligand_bb_VDW_E,
  Ligand & molecule
)
{
  using namespace ligand;
  using namespace param;
  using namespace misc;
  using namespace runlevel_ns;
  using namespace aaproperties_pack;

  float dis2;
  float solvE1;
  float solvE2;
  float atrE;
  float repE;
  triplet::xyzVector_float hetatom_id_coordinate;
  FArray1D_float het_coord(3);

  ligand_bb_VDW_E = 0.0;

  //az loop over all residues
  for ( int aa_id = 1; aa_id <= total_residue; ++aa_id ) {

    //az for all hetero atoms ....
    for (  size_t hetatom_id = 0; hetatom_id < molecule.atom_vector.size(); ++hetatom_id ) {

      //az .. that are not hydrogens or virtual atoms (mapped as hydrogen, see read_pdb_hetero)
      if ( molecule.atom_vector[hetatom_id]->get_ros_atom_cent() > 0 ) {
        copy_to_FArray(molecule.atom_vector[hetatom_id]->get_coordinates(),het_coord);
        //az backbone atom checks.
        //az in fullcoord: N=1, Ca=2, C=3, O=4 (!= position and centroid data structures)

        // WARNING: possible performance issue here. The centroid code is clever in that
        // it computes the distance between atoms and only computes clash check of the distance
        // is lower than a cutoff. However, pairenergy has already a distance  cut-off (safe_max_dis2)

        for ( int bbatom_id = 1; bbatom_id <= 4; ++bbatom_id ) {

          int const var_id = 1;
          int const bbatom_type = fullatom_type(bbatom_id,res(aa_id),res_variant(var_id));

          pairenergy( full_coord(1,bbatom_id,aa_id), het_coord(1), bbatom_type, molecule.atom_vector[hetatom_id]->get_ros_atom_type(), solvE1, solvE2, atrE, repE, dis2 );

          ligand_bb_VDW_E += atrE+repE;
        }
      }
    }
  }

  if ( runlevel >= verbose ) std::cout << "ligand bb vdw score: " <<
   SS( ligand_bb_VDW_E ) << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_dihedral_energy
///
/// @brief Looks up the energy and derivative of a dihedral angle in
///PeriodicSplineReader given a std::pair of LigandAtomTypes and the dihedral
///
/// @detailed
///
/// @param atompairtype [in] ligand atom type needed to decide which dihedral
///                          function to query
/// @param dihedral     [in] value of dihedral to evaluate
/// @param dihedE       [out] return value for energy value
/// @param deriv_dihedE [out] return value for derivative of dihedral energy
///                           function
///
///
/// @global_read ligand_ns dihedral_spline
///
/// @global_write ligand_ns dihedral_spline may add a spline to account for
///                          queried atomtype pair.
///
/// @remarks
///
/// @references
///
/// @authors Kristian Kaufmann
///
/// @last_modified 3-14-2006
/////////////////////////////////////////////////////////////////////////////////

int
get_dihedral_energy(
  const std::pair< atom_chem::LigandAtomType, atom_chem::LigandAtomType > atompairtype,
  const double dihedral,
  double & dihedE,
  double & deriv_dihedE
){


  // iterating through already loaded splines.
  bool found_spline=false;
  std::list<std::pair<
    std::pair<atom_chem::LigandAtomType,atom_chem::LigandAtomType>,
    PeriodicSplineReader > >::iterator current_spline=ligand::dihedral_splines.begin();

  while(current_spline!=ligand::dihedral_splines.end() && !found_spline){

    if(((*current_spline).first.first==atompairtype.first &&
      (*current_spline).first.second == atompairtype.second ) ||
      ((*current_spline).first.first==atompairtype.second &&
      (*current_spline).first.second == atompairtype.first  )){
      found_spline=true;
      dihedE=(*current_spline).second.F(dihedral);
      deriv_dihedE=(*current_spline).second.dF(dihedral);
    }
    current_spline++;
  }


  // obtain LigandAtomType of concerned pair for query of
  // small_molecule_crystal_structure_dihedral.txt in order to load
  // spline of energy function.
  std::string first_atomtype=atom_chem::ligandatomtype_string[atompairtype.first ];
  std::string second_atomtype=atom_chem::ligandatomtype_string[atompairtype.second];
  if(!found_spline){

    utility::io::izstream iunit;
    iunit.clear();
    iunit.open(files_paths::data_path +
      "small_molecule_crystal_structure_dihedral.txt");
    if( iunit.fail()){
      std::cout << "small_molecule_crystal_structure_dihedral.txt failed";
      std::cout << std::endl << " to open. Please check that it is ";
      std::cout << "in rosetta_database." <<  std::endl;
      return -1;
    }

    // formatted input expected from
    //  small_molecule_crystal_structure_dihedral.txt
    // iunit expects two size_ts  followed by threelines defining a spline
    // as expected in utility PeriodicSplineReader object read function

    std::string spline_atomtype_one, spline_atomtype_two;

    found_spline=false;
    do{
      iunit >> spline_atomtype_one >> spline_atomtype_two >> skip;
      if( iunit.fail()){
        std::cout << "Encountered unexpected format while reading in dihedral ";
        std::cout << "spline data for atompair type" << std::endl;
        std::cout << spline_atomtype_one << " " << spline_atomtype_two << std::endl;
        std::cout << "Please check the format of ";
        std::cout << "small_molecule_crystal_structure_dihedral.txt"<<  std::endl;
        std::cout << "Encountered will searching for " << first_atomtype;
        std::cout << " " << second_atomtype << " dihedral spline" << std::endl;
        iunit.clear();
        iunit >> skip;
        iunit >> skip;
        iunit >> skip;
        iunit >> skip;
        iunit >> skip;
        utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
      }else{
        if((spline_atomtype_one==first_atomtype &&
            spline_atomtype_two==second_atomtype) ||
          (spline_atomtype_one==second_atomtype &&
            spline_atomtype_two==first_atomtype)){
          std::pair<std::pair<atom_chem::LigandAtomType,
              atom_chem::LigandAtomType>,PeriodicSplineReader > spline_now;

          (spline_now.first).first=atompairtype.first;
          (spline_now.first).second=atompairtype.second;
          if((spline_now.second).read(iunit)==-1){
            std::cout << "Failed to obtain statistics for given atom pair ";
            std::cout << first_atomtype << " " << spline_atomtype_two << std::endl;
            std::cout << "will try to continue as a rigid ligand" << std::endl;
            ligand::ligand_flexible=false;
            return -1;
          };
          ligand::dihedral_splines.push_back(spline_now);
          found_spline=true;
          dihedE=((ligand::dihedral_splines.back())).second.F(dihedral);
          deriv_dihedE=((ligand::dihedral_splines.back())).second.dF(dihedral);
        }else{
          iunit >> skip;
          iunit >> skip;
          iunit >> skip;
          iunit >> skip;
          iunit >> skip;
        }
        if(iunit.eof()){
          std::cout << "Failed to find statistics for given atom pair ";
          std::cout << first_atomtype << " " << spline_atomtype_two << std::endl;
          std::cout << "will try to continue as a rigid ligand" << std::endl;
          ligand::ligand_flexible=false;
          return -1;
        }
      }
    }while(!found_spline);
    iunit.clear();
    iunit.close();
  }
  return 1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_ligand_cartesian_conformation
///
/// @brief returns a LigandCartesianCoord object defining a one to one mapping
///  of the Cartesian conformation of the ligand passed into the function
///
/// @detailed
///
/// @param Ligand & molecule
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Kristian Kaufmann
///
/// @last_modified 4_25_2006
/////////////////////////////////////////////////////////////////////////////////



ligand::LigandCartesianCoord
get_ligand_cartesian_conformation(
  const Ligand & molecule
){
  ligand::LigandCartesianCoord conformation;
  for( std::vector<ligand::Ligand_AtomOP >::const_iterator current_atom=molecule.atom_vector.begin();
   current_atom!=molecule.atom_vector.end(); current_atom++){
    conformation.push_back((*current_atom)->get_coordinates());
  }
  return conformation;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_ligand_cartesian_conformation
///
/// @detailed
///
/// @param ligand::LigandCartesianCoord & conformation
/// @param Ligand & molecule
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Kristian Kaufmann
///
/// @last_modified 6_11_2006
/////////////////////////////////////////////////////////////////////////////////

void
set_ligand_cartesian_conformation(
	ligand::LigandCartesianCoord & conformation,
	Ligand & molecule
){
	if( molecule.atom_vector.size() == conformation.size() ){
	  ligand::LigandCartesianCoord::iterator c_coord_ptr=conformation.begin();
	  for( ligand::Atom_Itr c_atom_ptr=molecule.atom_vector.begin();
	    c_atom_ptr!=molecule.atom_vector.end();c_atom_ptr++){

	    (*c_atom_ptr)->set_coordinates(*c_coord_ptr);
	    c_coord_ptr++;
//	    std::cout << "kwk help cartesian coord" << (*c_atom_ptr)->get_coordinates() << std::endl;
	  }
	  molecule.calculate_internal_coordinates(true);
//	  utility::io::ozstream pdbout("ligand_conf.pdb");

//	  make_pdb_hetero(pdbout, molecule);
//	  pdbout.close();
	}else{
	  std::cout << "Error in setting ligand cartesian conformation" << std::endl;
	}
	return;

}
////////////////////////////////////////////////////////////////////////////////
/// @begin get_ligand_internal_conformation
///
/// @brief returns a LigandInternalCoord object defining a one to one mapping
///  of the internal conformation of the ligand passed into the function
///
/// @detailed
///
/// @param Ligand & molecule
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Kristian Kaufmann
///
/// @last_modified 5_10_2006
/////////////////////////////////////////////////////////////////////////////////



ligand::LigandInternalCoord
get_ligand_internal_conformation(
  const Ligand & molecule
){
  ligand::LigandInternalCoord conformation;
  for( std::vector<ligand::Ligand_BondAP >::const_iterator current_bond=molecule.bond_vector.begin();
   current_bond!=molecule.bond_vector.end(); current_bond++){
    conformation.push_back((*current_bond)->get_coordinates());
  }
  return conformation;
}

