// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 8705 $
//  $Date: 2006-06-06 16:15:21 -0700 (Tue, 06 Jun 2006) $
//  $Author: pbradley $

// unit headers
#include <epigraft/legacy/hiv_trimer.hh>

// package headers
#include <epigraft/epigraft_types.hh>
#include <epigraft/epi_graft.h>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/epigraft_io.hh>
#include <epigraft/ResidueRange.hh>

// rootstock headers
#include <rootstock/geometry.hh>

// Rosetta Headers
#include <aaproperties_pack.h>
#include <after_opts.h>
#include <are_they_neighbors.h>//use_atom
#include <design.h>
#include <disulfides.h>
#include <enzyme.h> //set_enable_ligaa_flag
#include <files_paths.h>
#include <fullatom.h>
#include <fullatom_setup.h>
#include <jumping_refold.h> //pose_to_misc
#include <jumping_util.h>
#include <ligand.h> // for ligand::multi_lig globals
#include <make_pdb.h> //output_fullatom_pdb
#include <minimize.h>
#include <min_debug_ns.h> //min_debug::nblist_score_delta_before
#include <misc.h>
#include <nblist.h>
#include <pack.h>
#include <PackerTask.h>
#include <param.h>
#include <param_aa.h>
#include <param_pack.h>
#include <pdb.h> //for pdb_res_num
#include <pose.h>
#include <pose_design.h>
#include <pose_io.h>
#include <pose_ligand.h>
#include <pose_rms.h>
//#include <pose_routines.h>
#include <pose_symmetric_docking.h>
#include <read_aa_ss.h>
#include <refold.h>
#include <runlevel.h>
#include <score.h>
#include <score_ns.h>
#include <symmetric_design.h>
#include <util_basic.h>
#include <pose_vdw.h>

#include <jumping_util.h>
#include <util_vector.h>

// ObjexxFCL Headers
#include <ObjexxFCL/DimensionExpressions.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>
#include <numeric/BodyPosition.hh>
#include <numeric/Quaternion.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/file/file_sys_util.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <sstream>


namespace epigraft {
namespace legacy {


///////////////////////////////////////////////////////////////////////////////
//make and score hiv trimers with glycans

void
hiv_trimer()
{

	using namespace pose_ns;

	// scorefxn
	Score_weight_map weight_map( score12 );

	float const save_fa_rep_weight = weight_map.get_weight( FA_REP );

	//use_atom used here for cb_dist measurements
	//returns cb atom number for all protein aa except gly
	//returns other chosen atom nums for all other aa
	static FArray1D_int const use_atom( param::MAX_AA(), atn_use_atom_initializer );


	//
	//set basic parameters controlling pose behavior
	//
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	bool const read_all_chains( true );
	//  bool const check_missing( false ); //use this for pose.set_coords throughout
	//  bool const coords_init( true ); //used in pose_from_misc

	//number of monomers in the oligommer
	int const N = 3;

	//
	//Before you read in pdbs, set disulf options automatically
	//
	disulfides::options::find_disulf = true;
	disulfides::options::norepack_disulf = true;

	//
	//Before you read in pdbs, set no_optH
	//
	files_paths::no_optH = truefalseoption("no_optH");


	//
	//options
	//

	//
	//minimize sidechains ?
	//
	bool do_minimize = true;
	//	do_minimize = truefalseoption("do_minimize");

	//
	//minimize, then repack?
	//
	bool do_minrep = true;
	//	do_minrep = truefalseoption("do_minrep");


	//
	//use symmetry for minimize/repack of trimers?
	//
	bool use_symmetry = true;
	//	use_symmetry = truefalseoption("use_symmetry");

	//output_pdbs
	bool output_pdbs = false;
	output_pdbs = truefalseoption("output_pdbs");

	// get translation distance
	Real trimer_distance;
	realafteroption( "trimer_distance", 0.0, trimer_distance );
//	if ( trimer_distance == 0.0 ) {
//		utility::exit( __FILE__, __LINE__, "ERROR: must set -trimer_distance!" );
//	}

	// load quaternions
	String q_filename;
	stringafteroption( "quaternions", "none", q_filename );
	SO3 so3;
	load_quaternions( q_filename, so3 );

	//file to output scores
	std::string trimer_scores_filename;
	stringafteroption( "trimer_scores_file", "none", trimer_scores_filename );
	if( trimer_scores_filename == "none" ) {
		std::cout << "Need -trimer_scores_file <filename> "
							<< "on command-line " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//
	//checkpoint file for running on clusters
	//
	std::string checkpoint_filename;
	std::string checkpoint_file_w_path;
	stringafteroption( "checkpoint", "none", checkpoint_filename );
	bool using_checkpoint = false;
	if ( checkpoint_filename == "none" ) {
		std::cout << "no checkpoint_filename, NOT using checkpointing" << std::endl;
	} else {
		using_checkpoint = true;
		checkpoint_file_w_path = files_paths::score_path + checkpoint_filename;
	}


	//
	//input pdb
	//
	//	std::string start_pdb;
	//	start_pdb = stringafteroption( "s" );


	//
	//Read list of start pdbs
	//
//	int n_start_pdb = 0;
//	FArray1D_string list_start_pdb;
//	read_list_scaff_pdb( n_start_pdb, list_start_pdb ); //using routine from epi_graft.cc

	// grab original monomer
	std::string original_monomer_pdb;
	stringafteroption( "monomer", "none", original_monomer_pdb );
	Pose original_monomer;
	pose_from_pdb( original_monomer, original_monomer_pdb, fullatom, ideal_pose, read_all_chains );

	// translate monomer to centroid of backbone heavy atoms
	numeric::xyzVector< Real > centroid = backbone_centroid( original_monomer );
	if ( truefalseoption( "force_old_centroid" ) ) { // for Kwong paper
		centroid = numeric::xyzVector< Real >( 19.07646917744047, 67.05515608019471, 106.7388577852724 );
	}
	original_monomer.transform_Ax_plus_b( numeric::xyzMatrix< Real >::identity(), -centroid );


//	for ( int i_start_pdb = 1; i_start_pdb <= n_start_pdb; i_start_pdb++ ) {
	Integer track = 1;
	for ( SO3::const_iterator s = so3.begin(), se = so3.end(); s != se; ++s ) {

		if ( using_checkpoint ) {
		  bool already_scored = check_conformation_already_scored( track, checkpoint_file_w_path );
		  if ( already_scored ) continue;
		}

		Integer const & q_id = s->first;
		numeric::Quaternion< Real > const & q = s->second;

//		std::string start_pdb = list_start_pdb( i_start_pdb );

		//
		//read in trimer pose
		//trimer_pose will be protein-only.
		//
		Pose trimer_pose;
//		int nres_trimer = 0;
//		pose_from_pdb( trimer_pose, start_pdb, fullatom, ideal_pose, read_all_chains );
		create_trimer( q, trimer_distance, original_monomer, trimer_pose );
		int nres_trimer = trimer_pose.total_residue();

		FArray1D_int trimer_pdb_res_num ( trimer_pose.pdb_info().pdb_res_num() );
		for ( int i = 1; i<nres_trimer;i++ ) std::cout << "pdb_res_num: " << I(5,i) << " " << I(5,trimer_pdb_res_num(i)) << std::endl;

		// flip symmetric sidechains
		{
			float const start_score( trimer_pose.score( weight_map ) );
			trimer_pose.refold_sidechains_from_chi();
			std::cout << "score-delta from sidechain flip: " <<
				trimer_pose.score( weight_map ) - start_score << std::endl;
		}

		int nres_monomer = 0;
		intafteroption( "nres_monomer",nres_monomer,nres_monomer );

		if ( nres_trimer != 3*nres_monomer ) {
			std::cout << "ERROR nres_monomer not consistent with trimer pdb" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}


		//
		//remember trimer scores
		//
		float const score_trimer = trimer_pose.get_0D_score( SCORE );
		float const fa_rep_trimer = trimer_pose.get_0D_score( FA_REP );
		float const fa_atr_trimer = trimer_pose.get_0D_score( FA_ATR );
		float const fa_sol_trimer = trimer_pose.get_0D_score( FA_SOL );
		float const fa_dun_trimer = trimer_pose.get_0D_score( FA_DUN );
		float const hbsc_trimer = trimer_pose.get_0D_score( HB_SC );
		float const fa_prob_trimer = trimer_pose.get_0D_score( FA_PROB );


		//make a copy of one monomer and score it!
		Pose monomer_pose;
		monomer_pose.simple_fold_tree( nres_monomer );
		monomer_pose.set_fullatom_flag( true, false );// set fullatom and do not repack
		//	monomer_pose.copy_segment( nres_monomer, trimer_symm_pose, 1 /*begin*/, 1 /*src_begin*/ );
		copy_segment_to_new_pose( trimer_pose, 1 /*begin_src*/, nres_monomer /*end_src*/, monomer_pose );

		//
		//remember monomer scores
		//
		monomer_pose.score( weight_map );
		float const score_monomer = monomer_pose.get_0D_score( SCORE );
		float const fa_rep_monomer = monomer_pose.get_0D_score( FA_REP );
		float const fa_atr_monomer = monomer_pose.get_0D_score( FA_ATR );
		float const fa_sol_monomer = monomer_pose.get_0D_score( FA_SOL );
		float const fa_dun_monomer = monomer_pose.get_0D_score( FA_DUN );
		float const hbsc_monomer = monomer_pose.get_0D_score( HB_SC );
		float const fa_prob_monomer = monomer_pose.get_0D_score( FA_PROB );


		//
		//remember disulfides in monomer
		//
		int n_disulf_monomer = disulfides::BOUNDARY::get_n_disulf_centroid();

		if ( n_disulf_monomer  == 0 ) std::cout << "WARNING WARNING WARNING NO DISULF DETECTED !!!! " << std::endl;

		FArray1D_int list_cys_resnums_in_disulf_monomer( 2*n_disulf_monomer );
		int n_cys_in_disulf_monomer = 0;
		for ( int i_disulf = 1; i_disulf <= n_disulf_monomer; i_disulf++ ) {
			int const resnum1 = disulfides::BOUNDARY::get_cys( disulfides::BOUNDARY::get_disulf_partner_a( i_disulf ) );
			int const resnum2 = disulfides::BOUNDARY::get_cys( disulfides::BOUNDARY::get_disulf_partner_b( i_disulf ) );
			++n_cys_in_disulf_monomer;
			list_cys_resnums_in_disulf_monomer( n_cys_in_disulf_monomer ) = resnum1;
			++n_cys_in_disulf_monomer;
			list_cys_resnums_in_disulf_monomer( n_cys_in_disulf_monomer ) = resnum2;
			std::cout << "disulf " << i_disulf << " resnum1, resnum2 " << resnum1 << "-" << resnum2
								<< " pdb_resnum1,pdb_resnum2: " << trimer_pdb_res_num(resnum1) << "-" << trimer_pdb_res_num(resnum2) << std::endl;
		}
		if ( n_cys_in_disulf_monomer != 2*n_disulf_monomer ) {
			std::cout << "ERROR counting disulf monomer " << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		//
		//read in the anchor resnums (ASN ) for the glycans on monomer
		//
		FArray1D_int list_anchor_resnums_monomer( nres_monomer );
		bool anchor_resnums_read_from_file = false;
		if ( truefalseoption ("list_anchor_resnums" ) ) {
			anchor_resnums_read_from_file = true;
			read_list_anchor_resnums( list_anchor_resnums_monomer );
		}
//		} else {
//			std::cout << "ERROR you must have -list_anchor_resnums_monomer <list> on command-line" << std::endl;
//			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
//		}

		//
		//setup list of natro positions in monomer ( can't be minimized/repacked )
		//
		//exclude disulfides
		//exclude Asn with glycan attached
		//exclude prolines!!!
		//

		FArray1D_int list_natro_positions_monomer( nres_monomer );
		FArray1D_bool natro_position_monomer( nres_monomer, false );
		int n_natro_positions_monomer = 0;
		//disulf
		for ( int i = 1; i <= n_cys_in_disulf_monomer; i++ ) {
			++n_natro_positions_monomer;
			list_natro_positions_monomer( n_natro_positions_monomer ) = list_cys_resnums_in_disulf_monomer( i );
			natro_position_monomer( list_natro_positions_monomer( n_natro_positions_monomer ) ) = true;
		}
		//Asn
		if ( anchor_resnums_read_from_file ) {
			for ( int i = 1, ie = list_anchor_resnums_monomer.size(); i <= ie; i++ ) {
				++n_natro_positions_monomer;
				list_natro_positions_monomer( n_natro_positions_monomer ) =	list_anchor_resnums_monomer( i );
				natro_position_monomer( list_natro_positions_monomer( n_natro_positions_monomer ) ) = true;
			}
		}
		//prolines
		for ( int ires = 1; ires <= nres_monomer; ires++ ) {
			if ( monomer_pose.res( ires ) == param_aa::aa_pro ) {
				++n_natro_positions_monomer;
				list_natro_positions_monomer( n_natro_positions_monomer ) = ires;
				natro_position_monomer( list_natro_positions_monomer( n_natro_positions_monomer ) ) = true;
				std::cout << "Excluding a pro from scmin/repack! " << ires << " pdb_res_num: " << trimer_pdb_res_num( ires ) << std::endl;
			}
		}
		list_natro_positions_monomer.redimension( n_natro_positions_monomer );


		//
		//compute ddGs for input pdb
		//

		float const	ddg_score_trimer   = score_trimer - 3*score_monomer;
		float const ddg_fa_rep_trimer  = fa_rep_trimer - 3*fa_rep_monomer;
		float const ddg_fa_atr_trimer  = fa_atr_trimer - 3*fa_atr_monomer;
		float const ddg_fa_sol_trimer  = fa_sol_trimer - 3*fa_sol_monomer;
		float const ddg_fa_dun_trimer  = fa_dun_trimer - 3*fa_dun_monomer;
		float const ddg_hbsc_trimer    = hbsc_trimer - 3*hbsc_monomer;
		float const ddg_fa_prob_trimer = fa_prob_trimer - 3*fa_prob_monomer;



		//
		//strip suffix and path from start_pdb to use in output filenames
		//
//		std::string tmp_name = start_pdb;
//		if ( has_suffix( tmp_name, ".gz" ) )
//			tmp_name.erase( tmp_name.length() - 3 ); // strip terminal .gz
//		if ( has_suffix( tmp_name, ".pdb" ) )
//			tmp_name.erase( tmp_name.length() - 4 ); // strip terminal .pdb
//		std::string start_pdb_name = tmp_name.substr( tmp_name.find_last_of( '/' ) + 1 );
		std::string start_pdb_name( "q_id" );

		std::string base_pdbfilename = start_pdb_name + "_" + string_of( q_id );
		//in case you run the same input multiple times, for example to see the effect of the hydrogen optimization,
		//you can distinguish the different output lines and output_pdbs because I am adding the i_start_pdb to the filename


		//
		//At this point the protein has been setup as a trimer
		//


		//write scores to trimer_scores_file
		//
		{//scope for output
			std::string filename = trimer_scores_filename;

			//want scorefile to live in score_path
			filename = files_paths::score_path + filename;

			// check if file exists
			utility::io::izstream infile ( filename );
			utility::io::ozstream outfile;
			if ( infile ) {
				//							utility::io::ozstream outfile ( filename,  std::ios::app );
				outfile.open_append( filename );
			} else {
				//							utility::io::ozstream outfile ( filename );
				outfile.open( filename );

				if ( ! outfile.good() ) {
					std::cout << "cant open trimer_scores_file for writing: "
										<< filename << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				//
				//only write the header the first time you open the file
				//

				outfile << SS( "       score: " )
								<< SS( "      score" )
								<< SS( "         atr" )
								<< SS( "         rep" )
								<< SS( "         sol" )
								<< SS( "         dun" )
								<< SS( "        hbsc" )
								<< SS( "        prob" )
								<< std::endl;
			}

			if ( ! outfile.good() ) {
				std::cout << "cant open trimer_scores_file for writing: "
									<< filename << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			outfile << SS( " total_input: " )
							<< F( 12, 2, score_trimer ) << " "
							<< F( 12, 2, fa_atr_trimer ) << " "
							<< F( 12, 2, fa_rep_trimer ) << " "
							<< F( 12, 2, fa_sol_trimer ) << " "
							<< F( 12, 2, fa_dun_trimer ) << " "
							<< F( 12, 2, hbsc_trimer   ) << " "
							<< F( 12, 2, fa_prob_trimer) << " "
							<< base_pdbfilename << " "
							<< std::endl;

			outfile << SS( "   ddg_input: " )
							<< F( 12, 2, ddg_score_trimer ) << " "
							<< F( 12, 2, ddg_fa_atr_trimer ) << " "
							<< F( 12, 2, ddg_fa_rep_trimer ) << " "
							<< F( 12, 2, ddg_fa_sol_trimer ) << " "
							<< F( 12, 2, ddg_fa_dun_trimer ) << " "
							<< F( 12, 2, ddg_hbsc_trimer ) << " "
							<< F( 12, 2, ddg_fa_prob_trimer ) << " "
							<< base_pdbfilename << " "
							<< std::endl;

			outfile.close();
			outfile.clear();
		}//scope for output


		//
		//Do minimization or minimization+repacking !
		//

		//do these matter?
		minimize_set_vary_phipsi( false );
		minimize_set_vary_chi( true );
		minimize_set_vary_omega( false );
		minimize_set_vary_rb_angle( false );
		minimize_set_vary_rb_trans( false );
		//							minimize_set_local_min( false, 0 ); // all non-move-list rsds minimized

		//
		//(1) trimer, no AB, no glycan... sidechain minimize only
		//

		FArray1D_bool interface_position_trimer( nres_trimer, false );
		FArray1D_int list_interface_positions_trimer( nres_trimer, 0 );
		bool glycan_attached = false;
		bool Ab_attached = false;

		FArray1D_bool natro_position_trimer( nres_trimer, false );
		for ( int i = 1; i <= nres_monomer; i++ ) {
			natro_position_trimer( i ) = natro_position_monomer( i );
			natro_position_trimer( i + nres_monomer ) = natro_position_monomer( i );   //monomer2
			natro_position_trimer( i + 2*nres_monomer ) = natro_position_monomer( i ); //monomer3
		}

		//							for ( int i = 1; i <=nres_trimer; i++ ) {
		//								std::cout << "natro_position_trimer: " << i << " " << natro_position_trimer( i ) << std::endl;
		//							}

		//
		//use trimer_pose here instead of trimer_symm_pose
		//until you update the function to ignore the pseudo residues
		//
		int const n_glycan_on_trimer = 0;
		int const nres_Ab = 0;
		identify_interface_positions( trimer_pose, glycan_attached, Ab_attached,
																	nres_monomer, n_glycan_on_trimer, nres_Ab,
																	natro_position_trimer,
																	interface_position_trimer,        /*output*/
																	list_interface_positions_trimer     /*output*/);

		//
		//print out interface positions for history
		//
		for ( int i = 1, ie = list_interface_positions_trimer.size(); i <= ie; i++ ) {
			std::cout << "interface_position: " << I(5,i) << " " << I(5,list_interface_positions_trimer(i))
								<< " pdb_res_num:" << I(5,trimer_pdb_res_num(list_interface_positions_trimer(i))) << std::endl;
		}


		bool save_use_nblist = get_use_nblist();
		bool save_try_rotamers = score_get_try_rotamers();
		score_set_try_rotamers( false );//no rotamer trials!
		set_use_nblist( true );
		trimer_pose.set_allow_bb_move( false );
		trimer_pose.set_allow_jump_move( false );
		trimer_pose.set_allow_chi_move( false );

		if ( use_symmetry ) {

			//
			//declare symmetric pose so minimize & repack will utilize symmetry
			//
			int const njump_monomer = 0;
			trimer_pose.setup_symm_info( nres_monomer, njump_monomer, N, "no_pseudo" );

			//
			//allow chi to move only at interface positions
			//
			for ( int i=1; i<= nres_trimer;i++ ) {
				if ( trimer_pose.symmetry_info().chi_independent( i ) ) {
					trimer_pose.set_allow_chi_move( i, interface_position_trimer( i ) );
				}
			}
		} else {
			std::cout << "won't minimize trimer unless you ask for -use_symmetry" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		//
		//must declare scores here so can output later, else we have scope problem.
		//

		//scmin scores
		float score_trimer_scmin = 0;
		double fa_rep_trimer_scmin = 0;
		float fa_atr_trimer_scmin = 0;
		float fa_sol_trimer_scmin = 0;
		float fa_dun_trimer_scmin = 0;
		float hbsc_trimer_scmin = 0;
		float fa_prob_trimer_scmin = 0;
		float ddg_score_trimer_scmin = 0;
		double ddg_fa_rep_trimer_scmin = 0;
		float ddg_fa_atr_trimer_scmin = 0;
		float ddg_fa_sol_trimer_scmin = 0;
		float ddg_fa_dun_trimer_scmin = 0;
		float ddg_hbsc_trimer_scmin = 0;
		float ddg_fa_prob_trimer_scmin = 0;

		//
		//Save a copy of trimer_pose pre-minimize
		//
		//		Pose pre_min_trimer_pose;
		//		pre_min_trimer_pose = trimer_pose;


		//
		//be able to quit this one if minimize goes awry
		//
		bool bad_min_breakout = false;

		//
		//Minimize but ramp the fa_rep_weight and the tolerance.
		//

		minimize_set_tolerance( 1e-4 );//1e-6...was 1e-3
		weight_map.set_weight( FA_REP, save_fa_rep_weight*0.01 );
		trimer_pose.main_minimize( weight_map, "dfpmin" ); //or "linmin"
		weight_map.set_weight( FA_REP, save_fa_rep_weight*0.05 );
		trimer_pose.main_minimize( weight_map, "dfpmin" ); //or "linmin"
		weight_map.set_weight( FA_REP, save_fa_rep_weight*0.25 );
		trimer_pose.main_minimize( weight_map, "dfpmin" ); //or "linmin"
		minimize_set_tolerance( 1e-5 );//1e-6...was 1e-4
		weight_map.set_weight( FA_REP, save_fa_rep_weight );
		trimer_pose.main_minimize( weight_map, "dfpmin" ); //or "linmin"

		//
		//total scores after scmin
		//
		trimer_pose.score ( weight_map );
		score_trimer_scmin   = trimer_pose.get_0D_score( SCORE );
		fa_rep_trimer_scmin  = trimer_pose.get_0D_score( FA_REP );
		fa_atr_trimer_scmin  = trimer_pose.get_0D_score( FA_ATR );
		fa_sol_trimer_scmin  = trimer_pose.get_0D_score( FA_SOL );
		fa_dun_trimer_scmin  = trimer_pose.get_0D_score( FA_DUN );
		hbsc_trimer_scmin    = trimer_pose.get_0D_score( HB_SC );
		fa_prob_trimer_scmin = trimer_pose.get_0D_score( FA_PROB );


		//
		//ddG scores after scmin
		//

		//make a copy of one monomer post-scmin and score it!
		Pose monomer_scmin_pose;
		monomer_scmin_pose.simple_fold_tree( nres_monomer );
		monomer_scmin_pose.set_fullatom_flag( true, false );// set fullatom and do not repack
		//	monomer_scmin_pose.copy_segment( nres_monomer, trimer_pose, 1 /*begin*/, 1 /*src_begin*/ );
		copy_segment_to_new_pose( trimer_pose, 1 /*begin_src*/, nres_monomer /*end_src*/, monomer_scmin_pose );
		monomer_scmin_pose.score( weight_map );

		ddg_score_trimer_scmin   = score_trimer_scmin - 3*monomer_scmin_pose.get_0D_score( SCORE );
		ddg_fa_rep_trimer_scmin  = fa_rep_trimer_scmin - 3*monomer_scmin_pose.get_0D_score( FA_REP );
		ddg_fa_atr_trimer_scmin  = fa_atr_trimer_scmin - 3*monomer_scmin_pose.get_0D_score( FA_ATR );
		ddg_fa_sol_trimer_scmin  = fa_sol_trimer_scmin - 3*monomer_scmin_pose.get_0D_score( FA_SOL );
		ddg_fa_dun_trimer_scmin  = fa_dun_trimer_scmin - 3*monomer_scmin_pose.get_0D_score( FA_DUN );
		ddg_hbsc_trimer_scmin    = hbsc_trimer_scmin - 3*monomer_scmin_pose.get_0D_score( HB_SC );
		ddg_fa_prob_trimer_scmin = fa_prob_trimer_scmin - 3*monomer_scmin_pose.get_0D_score( FA_PROB );

		if ( output_pdbs ) {
			std::string scmin_pdbfilename = files_paths::pdb_out_path + base_pdbfilename + ".scmin" + ".pdb.gz";
			trimer_pose.dump_scored_pdb( scmin_pdbfilename, weight_map );
		}

		//
		//write minimize scores to trimer_scores_file
		//
		{//scope for output
			std::string filename = trimer_scores_filename;

			//want scorefile to live in score_path
			filename = files_paths::score_path + filename;

			// check if file exists
			utility::io::izstream infile ( filename );
			utility::io::ozstream outfile;
			if ( infile ) {
				//							utility::io::ozstream outfile ( filename,  std::ios::app );
				outfile.open_append( filename );
			} else {
				//							utility::io::ozstream outfile ( filename );
				outfile.open( filename );

				if ( ! outfile.good() ) {
					std::cout << "cant open trimer_scores_file for writing: "
										<< filename << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				//
				//only write the header the first time you open the file
				//

				outfile << SS( "score:        " )
								<< SS( "      score" )
								<< SS( "         atr" )
								<< SS( "         rep" )
								<< SS( "         sol" )
								<< SS( "         dun" )
								<< SS( "        hbsc" )
								<< SS( "        prob" )
								<< SS( "  dist_Nterm" )
								<< std::endl;
			}//if infile

			if ( ! outfile.good() ) {
				std::cout << "cant open trimer_scores_file for writing: "
									<< filename << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			outfile << SS( " total_scmin: " )
							<< F( 12, 2, score_trimer_scmin ) << " "
							<< F( 12, 2, fa_atr_trimer_scmin ) << " "
							<< F( 12, 2, fa_rep_trimer_scmin ) << " "
							<< F( 12, 2, fa_sol_trimer_scmin ) << " "
							<< F( 12, 2, fa_dun_trimer_scmin ) << " "
							<< F( 12, 2, hbsc_trimer_scmin   ) << " "
							<< F( 12, 2, fa_prob_trimer_scmin) << " "
							<< base_pdbfilename << " "
							<< std::endl;

			outfile << SS( "   ddg_scmin: " )
							<< F( 12, 2, ddg_score_trimer_scmin ) << " "
							<< F( 12, 2, ddg_fa_atr_trimer_scmin ) << " "
							<< F( 12, 2, ddg_fa_rep_trimer_scmin ) << " "
							<< F( 12, 2, ddg_fa_sol_trimer_scmin ) << " "
							<< F( 12, 2, ddg_fa_dun_trimer_scmin ) << " "
							<< F( 12, 2, ddg_hbsc_trimer_scmin ) << " "
							<< F( 12, 2, ddg_fa_prob_trimer_scmin ) << " "
							<< base_pdbfilename << " "
							<< std::endl;

			outfile.close();
			outfile.clear();
		}//scope for output


		//
		//minrep scores
		//
		float score_trimer_minrep = 0;
		double fa_rep_trimer_minrep = 0;
		float fa_atr_trimer_minrep = 0;
		float fa_sol_trimer_minrep = 0;
		float fa_dun_trimer_minrep = 0;
		float hbsc_trimer_minrep = 0;
		float fa_prob_trimer_minrep = 0;
		float ddg_score_trimer_minrep = 0;
		double ddg_fa_rep_trimer_minrep = 0;
		float ddg_fa_atr_trimer_minrep = 0;
		float ddg_fa_sol_trimer_minrep = 0;
		float ddg_fa_dun_trimer_minrep = 0;
		float ddg_hbsc_trimer_minrep = 0;
		float ddg_fa_prob_trimer_minrep = 0;


		if ( do_minrep ) {
			std::cout << "now in repack step " << std::endl;

			if ( ! use_symmetry ) {
				std::cout << "won't repack without -use_symmetry" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			//we are using symmetry

			FArray1D_bool allow_repack_trimer( nres_trimer, false );
			FArray1D_float trimer_interface_com( 3, 0.0 );
			Pose pre_repack_pose;
			pre_repack_pose = trimer_pose;

			//
			// not rbmin so use previous chi_move positions for repack
			//
			//			for ( int i=1; i<=nres_trimer; i++ ) allow_repack_trimer( i ) = trimer_pose.get_allow_chi_move( i );

			//
			//allow repack only at interface positions
			//
			trimer_pose.set_allow_chi_move( false );
			for ( int i=1; i<= nres_trimer;i++ ) {
				if ( trimer_pose.symmetry_info().chi_independent( i ) ) {
					trimer_pose.set_allow_chi_move( i, interface_position_trimer( i ) );
					if ( interface_position_trimer( i ) ) {
						allow_repack_trimer( i ) =  true;
						std::cout << "allow_repack_trimer at " << I(5,i) << " pdb_res_num:" << I(5,trimer_pdb_res_num(i)) << std::endl;
					}
				}
			}



			//final repack.
			trimer_pose.repack( allow_repack_trimer, true /*include_current*/);

			std::cout << "After repack, n_disulf_trimer = " << disulfides::BOUNDARY::get_n_disulf_centroid() << std::endl;
			for ( int i_disulf = 1; i_disulf <= disulfides::BOUNDARY::get_n_disulf_centroid(); i_disulf++ ) {
				int const resnum1 = disulfides::BOUNDARY::get_cys( disulfides::BOUNDARY::get_disulf_partner_a( i_disulf ) );
				int const resnum2 = disulfides::BOUNDARY::get_cys( disulfides::BOUNDARY::get_disulf_partner_b( i_disulf ) );
				std::cout << "disulf " << i_disulf << " resnum1, resnum2 " << resnum1 << "-" << resnum2
									<< " pdb_resnum1,pdb_resnum2: " << trimer_pdb_res_num(resnum1) << "-" << trimer_pdb_res_num(resnum2) << std::endl;
			}


			//
			//final minimize... with tightest tolerance
			//
			minimize_set_tolerance( 1e-6 );//was 1e-5 at last stage of scmin
			weight_map.set_weight( FA_REP, save_fa_rep_weight );
			trimer_pose.main_minimize( weight_map, "dfpmin" ); //or "linmin"
			//		float const repack_allatom_rms( pose_calc_absolute_allatom_rmsd( trimer_pose, pre_repack_pose, 1, nres_trimer  ));
			//		std::cout << "repack_allatom_rms " << repack_allatom_rms << std::endl;

			//
			//minrep compute ddG
			//
			trimer_pose.score ( weight_map );
			score_trimer_minrep   = trimer_pose.get_0D_score( SCORE );
			fa_rep_trimer_minrep  = trimer_pose.get_0D_score( FA_REP );
			fa_atr_trimer_minrep  = trimer_pose.get_0D_score( FA_ATR );
			fa_sol_trimer_minrep  = trimer_pose.get_0D_score( FA_SOL );
			fa_dun_trimer_minrep  = trimer_pose.get_0D_score( FA_DUN );
			hbsc_trimer_minrep    = trimer_pose.get_0D_score( HB_SC );
			fa_prob_trimer_minrep = trimer_pose.get_0D_score( FA_PROB );

			//make a copy of one monomer post-minrep and score it!
			Pose monomer_minrep_pose;
			monomer_minrep_pose.simple_fold_tree( nres_monomer );
			monomer_minrep_pose.set_fullatom_flag( true, false );// set fullatom and do not repack
			//			monomer_minrep_pose.copy_segment( nres_monomer, trimer_pose, 1 /*begin*/, 1 /*src_begin*/ );
			copy_segment_to_new_pose( trimer_pose, 1 /*begin_src*/, nres_monomer /*end_src*/, monomer_minrep_pose );
			monomer_minrep_pose.score( weight_map );

			ddg_score_trimer_minrep   = score_trimer_minrep - 3*monomer_minrep_pose.get_0D_score( SCORE );
			ddg_fa_rep_trimer_minrep  = fa_rep_trimer_minrep - 3*monomer_minrep_pose.get_0D_score( FA_REP );
			ddg_fa_atr_trimer_minrep  = fa_atr_trimer_minrep - 3*monomer_minrep_pose.get_0D_score( FA_ATR );
			ddg_fa_sol_trimer_minrep  = fa_sol_trimer_minrep - 3*monomer_minrep_pose.get_0D_score( FA_SOL );
			ddg_fa_dun_trimer_minrep  = fa_dun_trimer_minrep - 3*monomer_minrep_pose.get_0D_score( FA_DUN );
			ddg_hbsc_trimer_minrep    = hbsc_trimer_minrep - 3*monomer_minrep_pose.get_0D_score( HB_SC );
			ddg_fa_prob_trimer_minrep = fa_prob_trimer_minrep - 3*monomer_minrep_pose.get_0D_score( FA_PROB );

			//
			//I did test what happens if you repack the start structure without minimizing
			//answer: the energy is much worse than if you repack after minimizing.
			//

			if ( output_pdbs ) {
				std::string minrep_pdbfilename = files_paths::pdb_out_path + base_pdbfilename + ".minrep" + ".pdb.gz";
				trimer_pose.dump_scored_pdb( minrep_pdbfilename, weight_map );
			}

		}//if do_minrep


		//
		//recover previous settings for nblist, try_rotamers
		//
		set_use_nblist ( save_use_nblist );
		score_set_try_rotamers( save_try_rotamers );

		//
		//write minrep scores to trimer_scores_file
		//
		{//scope for output
			std::string filename = trimer_scores_filename;

			//want scorefile to live in score_path
			filename = files_paths::score_path + filename;

			// check if file exists
			utility::io::izstream infile ( filename );
			utility::io::ozstream outfile;
			if ( infile ) {
				//							utility::io::ozstream outfile ( filename,  std::ios::app );
				outfile.open_append( filename );
			} else {
				//							utility::io::ozstream outfile ( filename );
				outfile.open( filename );

				if ( ! outfile.good() ) {
					std::cout << "cant open trimer_scores_file for writing: "
										<< filename << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				//
				//only write the header the first time you open the file
				//

				outfile << SS( "score:        " )
								<< SS( "      score" )
								<< SS( "         atr" )
								<< SS( "         rep" )
								<< SS( "         sol" )
								<< SS( "         dun" )
								<< SS( "        hbsc" )
								<< SS( "        prob" )
								<< SS( "  dist_Nterm" )
								<< std::endl;
			}//if infile

			if ( ! outfile.good() ) {
				std::cout << "cant open trimer_scores_file for writing: "
									<< filename << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			if ( do_minrep ) {
				outfile << SS( "total_minrep: " )
								<< F( 12, 2, score_trimer_minrep ) << " "
								<< F( 12, 2, fa_atr_trimer_minrep ) << " "
								<< F( 12, 2, fa_rep_trimer_minrep ) << " "
								<< F( 12, 2, fa_sol_trimer_minrep ) << " "
								<< F( 12, 2, fa_dun_trimer_minrep ) << " "
								<< F( 12, 2, hbsc_trimer_minrep   ) << " "
								<< F( 12, 2, fa_prob_trimer_minrep) << " "
								<< base_pdbfilename << " "
								<< std::endl;

				outfile << SS( "  ddg_minrep: " )
								<< F( 12, 2, ddg_score_trimer_minrep ) << " "
								<< F( 12, 2, ddg_fa_atr_trimer_minrep ) << " "
								<< F( 12, 2, ddg_fa_rep_trimer_minrep ) << " "
								<< F( 12, 2, ddg_fa_sol_trimer_minrep ) << " "
								<< F( 12, 2, ddg_fa_dun_trimer_minrep ) << " "
								<< F( 12, 2, ddg_hbsc_trimer_minrep ) << " "
								<< F( 12, 2, ddg_fa_prob_trimer_minrep ) << " "
								<< base_pdbfilename << " "
								<< std::endl;

				outfile.close();
				outfile.clear();
			}//do_minrep
		}//output

    //
    //checkpoint for clusters
    //
    if ( using_checkpoint ) {
    	update_conformation_already_scored( track, checkpoint_file_w_path );
    }

	} // foreach quaternion
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
//identify positions to repack/minimize between proteins or between protein-ligand (glycan)
void
identify_interface_positions(
														 pose_ns::Pose & pose,        //trimer of proteins at minimum
														 bool const & glycan_attached,    // is glycan also attached?
														 bool const & Ab_attached,  // is antibody (1 copy, to monomer 1) also attached?
														 int const & nres_monomer,
														 int const & nres_glycan,
														 int const & nres_Ab,
														 FArray1D_bool const & natro_position, // positions on protein_trimer that cannot be minimized/repacked
														 FArray1D_bool & interface_position,
														 FArray1D_int & list_interface_positions
														 )
{
	//
	//trimer is constructed in the following order:
	//(1) protein monomers 1-3
	//(2) if glycan is attached, it comes next ( 48 amino acids )
	//(3) if Ab is attached, it comes next
	//

	//	std::cout << "glycan_attached, Ab_attached " << glycan_attached << " " << Ab_attached << std::endl;

	int nres = pose.total_residue();//not const b/c can be reset if npseudo!=0;
	int const nres_protein_trimer = nres_monomer*3;

	//	std::cout << "glycan_attached, Ab_attached " << glycan_attached << " " << Ab_attached << std::endl;
	//	std::cout << "nres_protein_trimer, nres " << nres_protein_trimer << " " << nres << std::endl;

	//	float const dis_cutoff = 5.5; //angstroms
	float const dis_cutoff = 6.0; //angstroms

	//	interface_position = false;

	//
	//error checking
	//
	if ( !glycan_attached && !Ab_attached && nres!=nres_protein_trimer ) {
		//deal with case of pseudo residues
		if ( pose.symmetric() && nres == nres_protein_trimer + pose.symmetry_info().npseudo() ) {
			nres=nres_protein_trimer;
		} else {
			std::cout << "ERROR wrong nres " << nres << " nres_monomer " << nres_monomer << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	} else {
		if ( glycan_attached && !Ab_attached && nres!=nres_protein_trimer+nres_glycan ) {
			std::cout << "ERROR wrong nres w/glycan " << nres << " nres_monomer, nres_glycan " << nres_monomer << " " << nres_glycan << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {
			if ( !glycan_attached && Ab_attached && nres!=nres_protein_trimer+nres_Ab ) {
				std::cout << "ERROR wrong nres w/Ab " << nres << " nres_monomer, nres_Ab " << nres_monomer << " " << nres_Ab << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			} else {
				if ( glycan_attached && Ab_attached && nres!=nres_protein_trimer+nres_Ab+nres_glycan ) {
					std::cout << "ERROR wrong nres w/glycan+Ab " << nres << " nres_monomer, nres_glycan, nres_Ab "
										<< nres_monomer << " " << nres_glycan << " " << nres_Ab << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}

	//
	//protein-protein contacts first
	//
	//     check contacts between monomer1 and monomer2
	//     then add symmetrical positions to the list since we don't do symmetrical minimize/repack w/Ab breaking symmetry
	//

	//Need to exclude:
	//(1) Asn with attached glycan
	//(2) cysteines in disulfides


	for ( int i = 1; i <= nres_monomer; i++ ) {
		if ( natro_position( i ) ) {
			//			std::cout << "in identify, i is natro " << i << std::endl;
			continue;
		}

    int start_atm1, end_atm1;
    int const aa1 = pose.res( i );
    int const aav1 = pose.res_variant( i );
    if ( aa1 == param_aa::aa_gly ) {
      start_atm1 = 2;
      end_atm1 = 2;
    } else {
      start_atm1 = 5;
      end_atm1 = aaproperties_pack::nheavyatoms( aa1, aav1 );
    }

    for ( int j = nres_monomer+1; j<=2*nres_monomer; ++j ) {
      int start_atm2, end_atm2;
      int aa2 = pose.res( j );
      int aav2 = pose.res_variant( j );
      if ( aa2 == param_aa::aa_gly ) {
        start_atm2 = 2;
        end_atm2 = 2;
      } else {
        start_atm2 = 5;
        end_atm2 = aaproperties_pack::nheavyatoms( aa2, aav2 );
      }

      float dis;
      for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
        for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
          dis = distance( numeric::xyzVector_float( &( pose.full_coord()(1,atm1,i))),
                          numeric::xyzVector_float( &( pose.full_coord()(1,atm2,j))));
          if ( dis < dis_cutoff ) {
            interface_position( i ) = true;
            interface_position( j ) = true;
						//symmetry positions
						interface_position( i + nres_monomer ) = true;    //monomer 2
						interface_position( i + 2*nres_monomer ) = true;  //monomer 3
						interface_position( j - nres_monomer ) = true;    //monomer 1
						interface_position( j + nres_monomer )  = true;   //monomer 3
						//				std::cout << "identify_interface_position: " << i << " " << j << std::endl;
          }
        } // atm2 -- in monomer2
      } // atm1 -- in monomer1
    }//j -- residues in monomer2
	}//i -- residues in monomer1

	/*
	//debugging.
	for ( int i=1; i<=nres_protein_trimer; i++ ) {
		std::cout << "inside identify: interface_position( i ) " << i << " " << interface_position( i ) << std::endl;
	}
	*/

	int glycan2_start = 0;
	int glycan2_end = 0;
	int glycan3_start = 0;
	int glycan3_end = 0;

	if ( glycan_attached ) {
		//glycan residues come after 3 protein monomers in the pose
		//don't include residues from one monomer interacting with glycans from that same monomer

		int const n_glycan_per_monomer = int( nres_glycan/3 );
		glycan2_start = nres_protein_trimer+n_glycan_per_monomer+1;
		glycan2_end = nres_protein_trimer+2*n_glycan_per_monomer;
		glycan3_start = nres_protein_trimer+2*n_glycan_per_monomer+1;
		glycan3_end = nres_protein_trimer+3*n_glycan_per_monomer;

		//not used currently
		//		int const glycan1_start = nres_protein_trimer+1;
		//		int const glycan1_end = nres_protein_trimer+n_glycan_per_monomer;

		//
		//monomer1--glycan2/3
		//
		for ( int i = 1; i <= nres_monomer; i++ ) {
			if ( natro_position( i ) ) continue;

			int start_atm1, end_atm1;
			int const aa1 = pose.res( i );
			int const aav1 = pose.res_variant( i );
			if ( aa1 == param_aa::aa_gly ) {
				start_atm1 = 2;
				end_atm1 = 2;
			} else {
				start_atm1 = 5;
				end_atm1 = aaproperties_pack::nheavyatoms( aa1, aav1 );
			}

			//
			//monomer1-glycan2
			//
			for ( int j = glycan2_start; j<=glycan2_end; ++j ) {
				int start_atm2, end_atm2;
				int aa2 = pose.res( j );
				int aav2 = pose.res_variant( j );
				if ( aa2 == param_aa::aa_gly ) {
					start_atm2 = 2;
					end_atm2 = 2;
				} else {
					start_atm2 = 1;  //1 for glycan  5;
					end_atm2 = aaproperties_pack::nheavyatoms( aa2, aav2 );
				}

				float dis;
				for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
					for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
						dis = distance( numeric::xyzVector_float( &( pose.full_coord()(1,atm1,i))),
														numeric::xyzVector_float( &( pose.full_coord()(1,atm2,j))));
						if ( dis < dis_cutoff ) {
							interface_position( i ) = true;
							//							std::cout << "identify_interface_position: monomer1-glycan" << i << " " << j << std::endl;
							//Don't include glycan positions in minimization for now.
							//							interface_position( j ) = true;
							//
							//And here are the symmetry mates:
							//
							interface_position( i+nres_monomer ) = true;//note be careful assigning symmetry mates to the glycans...don't want monomer2/glycan2 etc.
							interface_position( i+2*nres_monomer ) = true;
							//interface_position( j + n_glycan_per_monomer ) = true; //glycan3
							//interface_position( j - n_glycan_per_monomer ) = true; //glycan1
							//							std::cout << "identify_interface_position: monomer2-glycan" << i+nres_monomer << " " << std::endl;
							//							std::cout << "identify_interface_position: monomer3-glycan" << i+2*nres_monomer << " " << std::endl;
						}
					} // atm2
				} // atm1
			}//j

			//
			//monomer1-glycan3
			//
			for ( int j = glycan3_start; j<=glycan3_end; ++j ) {
				int start_atm2, end_atm2;
				int aa2 = pose.res( j );
				int aav2 = pose.res_variant( j );
				if ( aa2 == param_aa::aa_gly ) {
					start_atm2 = 2;
					end_atm2 = 2;
				} else {
					start_atm2 = 1;  //1 for glycan  5;
					end_atm2 = aaproperties_pack::nheavyatoms( aa2, aav2 );
				}

				float dis;
				for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
					for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
						dis = distance( numeric::xyzVector_float( &( pose.full_coord()(1,atm1,i))),
														numeric::xyzVector_float( &( pose.full_coord()(1,atm2,j))));
						if ( dis < dis_cutoff ) {
							interface_position( i ) = true;
							//							std::cout << "identify_interface_position: monomer1-glycan" << i << " " << j << std::endl;
							//Don't include glycan positions in minimization for now.
							//							interface_position( j ) = true;
							//
							//And here are the symmetry mates:
							//
							interface_position( i+nres_monomer ) = true;//note be careful assigning symmetry mates to the glycans...don't want monomer2/glycan2 etc.
							interface_position( i+2*nres_monomer ) = true;
							//interface_position( j - 2*n_glycan_per_monomer ) = true; //glycan1
							//interface_position( j - n_glycan_per_monomer ) = true; //glycan2
							//							std::cout << "identify_interface_position: monomer2-glycan" << i+nres_monomer << " " << std::endl;
							//							std::cout << "identify_interface_position: monomer3-glycan" << i+2*nres_monomer << " " << std::endl;
						}
					} // atm2
				} // atm1
			}//j

		}//i
	}//glycan_attached


	if ( Ab_attached ) {
		int Ab_start, Ab_end;

		if ( glycan_attached ) {
			Ab_start = nres_protein_trimer + nres_glycan + 1;
			Ab_end = nres;
		} else {
			Ab_start = nres_protein_trimer + 1;
			Ab_end = nres;
		}

		//		std::cout << "Ab_start, Ab_end " << Ab_start << " " << Ab_end << std::endl;

		//
		//monomer2/3--Ab ( don't bother minimizing monomer1-Ab b/c that interaction is from xtal structure )
		//
		for ( int i = nres_monomer+1; i <= nres_protein_trimer; i++ ) {
			if ( natro_position( i ) ) continue;

			int start_atm1, end_atm1;
			int const aa1 = pose.res( i );
			int const aav1 = pose.res_variant( i );
			if ( aa1 == param_aa::aa_gly ) {
				start_atm1 = 2;
				end_atm1 = 2;
			} else {
				start_atm1 = 5;
				end_atm1 = aaproperties_pack::nheavyatoms( aa1, aav1 );
			}

			for ( int j = Ab_start; j<=Ab_end; ++j ) {
				if ( natro_position( j ) ) continue;

				int start_atm2, end_atm2;
				int aa2 = pose.res( j );
				int aav2 = pose.res_variant( j );
				if ( aa2 == param_aa::aa_gly ) {
					start_atm2 = 2;
					end_atm2 = 2;
				} else {
					start_atm2 = 5;
					end_atm2 = aaproperties_pack::nheavyatoms( aa2, aav2 );
				}

				float dis;
				for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
					for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
						dis = distance( numeric::xyzVector_float( &( pose.full_coord()(1,atm1,i))),
														numeric::xyzVector_float( &( pose.full_coord()(1,atm2,j))));
						if ( dis < dis_cutoff ) {
							interface_position( i ) = true;
							interface_position( j ) = true;
							//							float const x1 = pose.full_coord()(1,atm1,i);
							//							float const x2 = pose.full_coord()(1,atm2,j);
							//							float const y2 = pose.full_coord()(2,atm2,j);
							//							float const z2 = pose.full_coord()(3,atm2,j);
							//							std::cout << "identify_interface_position: monomer2/3-Ab " << i << " " << j << " "
							//												<< x1 << " " << x2 << " " << y2 << " " << z2 << std::endl;
						}
					} // atm2
				} // atm1
			}//j
		}//i

		if ( glycan_attached ) {
			//
			//Ab--glycan2/3 -- don't bother with Ab-glycan1 b/c that was done already with Molecular Dynamics
			//
			for ( int i = Ab_start; i <= Ab_end; i++ ) {
				if ( natro_position( i ) ) continue;

				int start_atm1, end_atm1;
				int const aa1 = pose.res( i );
				int const aav1 = pose.res_variant( i );
				if ( aa1 == param_aa::aa_gly ) {
					start_atm1 = 2;
					end_atm1 = 2;
				} else {
					start_atm1 = 5;
					end_atm1 = aaproperties_pack::nheavyatoms( aa1, aav1 );
				}

				for ( int j = glycan2_start; j<=glycan3_end; ++j ) {
					int start_atm2, end_atm2;
					int aa2 = pose.res( j );
					int aav2 = pose.res_variant( j );
					if ( aa2 == param_aa::aa_gly ) {
						start_atm2 = 2;
						end_atm2 = 2;
					} else {
						start_atm2 = 1;  //1 for glycan  5;
						end_atm2 = aaproperties_pack::nheavyatoms( aa2, aav2 );
					}

					float dis;
					for ( int atm1 = start_atm1; atm1 <= end_atm1; ++atm1 ) { // atoms pos1
						for ( int atm2 = start_atm2; atm2 <= end_atm2; ++atm2 ) { // atoms pos2
							dis = distance( numeric::xyzVector_float( &( pose.full_coord()(1,atm1,i))),
															numeric::xyzVector_float( &( pose.full_coord()(1,atm2,j))));
							if ( dis < dis_cutoff ) {
								interface_position( i ) = true;
								//Don't include glycan positions in minimization for now.
								//								interface_position( j ) = true;
								//												std::cout << "identify_interface_position: Ab-glycan2/3 " << i << " " << j << std::endl;
							}
						} // atm2
					} // atm1
				}//j
			}//i
		}//glycan_attached
	}//Ab_attached


	//
	//count interface positions
	//
	int n_interface_positions = 0;
	for ( int i = 1; i <= nres; i++ ) {
		if ( interface_position( i ) ) ++n_interface_positions;
	}

	//
	//dimension the list correctly and fill it
	//
	list_interface_positions.redimension( n_interface_positions );
	n_interface_positions = 0;
	for ( int i = 1; i <= nres; i++ ) {
		if ( interface_position( i ) ) {
			list_interface_positions( ++n_interface_positions ) = i;
			//			std::cout << "interface_position: " << I(5,n_interface_positions) << " " << I(5,i) << std::endl;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////


void
read_list_anchor_resnums(
												 FArray1D_int & list_anchor_resnums
												 )
{

	std::string filename;
  filename = stringafteroption( "list_anchor_resnums");

	utility::io::izstream infile ( filename );
  if ( ! infile ) {
		std::cout << "read_list_anchor_resnums:"
              << " cannot open file " << filename
              << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  int n_lig = 0;
	std::string line;
  while( getline( infile, line ) ) {
		std::string blank(line.size(),' ');
    if ( line == blank ) continue;
		std::istringstream line_stream( line );
    ++n_lig;
    list_anchor_resnums.redimension( n_lig );
    line_stream >> list_anchor_resnums( n_lig );
		std::cout << "n_lig, list_anchor_resnums " << I( 5, n_lig ) << " "
              << SS( list_anchor_resnums( n_lig ))  << std::endl;
  }

	//	if ( n_lig > param::MAX_N_LIG ) {
	//		std::cout << "ERROR in read_list_anchor_resnums:: n_lig = " << n_lig << " is greater than param::MAX_N_LIG = " << param::MAX_N_LIG << std::endl;
	//		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	//	}
	if ( n_lig < 1 ) {
		std::cout << "ERROR in read_list_anchor_resnums:: failed to read any anchors " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

///////////////////////////////////////////////////////////////////////////////

bool
integer_is_even( int & num )
{
	return ( num % 2 ) == 0;
}
///////////////////////////////////////////////////////////////////////


/// @brief read quaternion positions in id, w, x, y, z format
/// @param[out] list of pairs of quaternion id + quaternion
void
load_quaternions(
	std::string const & filename,
	SO3 & v
)
{
	using std::istringstream;
	using numeric::Quaternion;
	using utility::io::izstream;
	using epigraft::split_string;

	izstream in( filename );

	if ( in.fail() ) {
		utility::exit( __FILE__, __LINE__, "ERROR: cannot open quaternion file " + filename );
	}

	Integer id;
	Real w, x, y, z;
	String line;
	std::vector< String > entries;

	while ( getline( in, line ) ) {
		if ( line.c_str()[ 0 ] != '#' ) {
			// grab entries
			entries.clear();
			split_string( line, entries );

			// parse quaternion bits
            istringstream( entries[ 0 ] ) >> id;
            istringstream( entries[ 1 ] ) >> w;
            istringstream( entries[ 2 ] ) >> x;
            istringstream( entries[ 3 ] ) >> y;
            istringstream( entries[ 4 ] ) >> z;

            // store quaternion id and quaternion
			v.push_back( std::make_pair( id, numeric::Quaternion< Real >( w, x, y, z ) ) );
		}
	}

	in.close();
}


/// @brief compute centroid of backbone heavy atoms N,CA,C,O
numeric::xyzVector< Real >
backbone_centroid(
	Pose const & pose
)
{
	using numeric::xyzVector;

	xyzVector< Real > centroid( 0.0, 0.0, 0.0 );

	FArray3D_float const & fc = pose.full_coord();
	for ( Integer i = 1, ie = pose.total_residue(); i <= ie; ++i ) {
		centroid += xyzVector< Real >( &fc( 1, 1, i ) ); // N
		centroid += xyzVector< Real >( &fc( 1, 2, i ) ); // CA
		centroid += xyzVector< Real >( &fc( 1, 3, i ) ); // C
		centroid += xyzVector< Real >( &fc( 1, 4, i ) ); // O
	}

	centroid /= ( 4 * pose.total_residue() );

	return centroid;
}


/// @brief trimer translation based on distance
void
trimer_translation(
	Real const & distance,
	numeric::xyzVector< Real > & m1_t,
	numeric::xyzVector< Real > & m2_t,
	numeric::xyzVector< Real > & m3_t
)
{
	using numeric::conversions::radians;
	using std::sin;
	using std::cos;

	typedef numeric::xyzVector< Real > xyzVector;

	// monomer1 is directly specified by input d = 'x_translation_distance'
	// monomer2 is specified by x = -d * sin(30), y = +d * cos(30)
	// monomer3 is specified by x = -d * sin(30), y = -d * cos(30)
	m1_t = xyzVector( distance, 0.0, 0.0 );
	m2_t = xyzVector( -distance * sin( radians( 30.0 ) ),
	                   distance * cos( radians( 30.0 ) ),
	                   0.0 );
	m3_t = xyzVector( m2_t.x(), -m2_t.y(), 0.0 );
}


/// @brief create trimer transforms given a quaternion and trimer distance
void
trimer_transforms(
	numeric::Quaternion< Real > const & q,
	Real const & translation_distance,
	numeric::BodyPosition< Real > & monomer1_transform,
	numeric::BodyPosition< Real > & monomer2_transform,
	numeric::BodyPosition< Real > & monomer3_transform
)
{
	// used functions
	using rootstock::quaternion;
	using numeric::conversions::radians;

	// typedefs
	typedef numeric::BodyPosition< Real > BodyPosition;
	typedef numeric::Quaternion< Real > Quaternion;
	typedef numeric::xyzVector< Real > xyzVector;

	// rotation about 120 degrees for z, used to compute
	// protein + glycan coordinates for monomer2, monomer3
	// from monomer1
	xyzVector z_axis( 0.0, 0.0, 1.0 );
	Quaternion q_monomer2 = quaternion( z_axis, radians( static_cast< Real >( 120.0 ) ) ) * q;
	Quaternion q_monomer3 = quaternion( z_axis, radians( static_cast< Real >( 240.0 ) ) ) * q;

	// now handle translations
	xyzVector t_monomer1;
	xyzVector t_monomer2;
	xyzVector t_monomer3;
	trimer_translation( translation_distance, t_monomer1, t_monomer2, t_monomer3 );

	// now pass transformations to output
	monomer1_transform = BodyPosition( q, t_monomer1 );
	monomer2_transform = BodyPosition( q_monomer2, t_monomer2 );
	monomer3_transform = BodyPosition( q_monomer3, t_monomer3 );
}


/// @brief transform Pose full_coord coordinates into external FArray3D
/// @param[in] offset  offsets last dimension when filling in FArray3D
void
transform_full_coord(
	Pose const & pose,
	numeric::BodyPosition< Real > const & bp,
	FArray3D< Real > & full_coord,
	Size const & offset
)
{
	// typedefs
	typedef numeric::xyzVector< Real > xyzVector;

	assert( offset <= full_coord.size3() );

	FArray3D< Real > const & ifc = pose.full_coord(); // cache input full coord
	Size idx = 0; // linear indexing into FArray
	xyzVector xyz;

	for ( Size i = 1, ie = pose.total_residue(); i <= ie; ++i ) {
		Integer const aa = pose.res( i );
		Integer const aav = pose.res_variant( i );
		for ( Size j = 1, je = aaproperties_pack::natoms( aa, aav ); j <= je; ++j ) {
			xyz = xyzVector( &ifc( 1, j, i ) );
			bp.transform( xyz );
			idx = full_coord.index( 1, j, i + offset );
			full_coord[ idx ] = xyz.x();
			full_coord[ ++idx ] = xyz.y();
			full_coord[ ++idx ] = xyz.z();
		}
	}
}


/// @brief constructs a trimer Pose given appropriate transforms
void
create_trimer(
	numeric::Quaternion< Real > const & q,
	Real const & translation_distance,
	Pose const & monomer,
	Pose & trimer
)
{
	// using ...
	using pose_ns::Fold_tree;
	using epigraft::ResidueRange;

	// typedefs
	typedef numeric::BodyPosition< Real > BodyPosition;
	typedef numeric::xyzVector< Real > xyzVector;

	// create appropriate trimer transforms
	BodyPosition monomer1_transform;
	BodyPosition monomer2_transform;
	BodyPosition monomer3_transform;
	trimer_transforms( q, translation_distance, monomer1_transform, monomer2_transform, monomer3_transform );

	// prepare trimer fold tree
	Fold_tree ft;
	ft.add_edge( 1, monomer.total_residue(), Fold_tree::PEPTIDE );
	ft.add_edge( monomer.total_residue() + 1, monomer.total_residue() * 2, Fold_tree::PEPTIDE );
	ft.add_edge( monomer.total_residue() * 2 + 1, monomer.total_residue() * 3, Fold_tree::PEPTIDE );
	ft.add_edge( monomer.total_residue(), monomer.total_residue() + 1, 1 );
	ft.add_edge( monomer.total_residue() * 2, monomer.total_residue() * 2 + 1, 2 );
	ft.reorder( 1 );

	// prepare trimer pose
	trimer.set_fold_tree( ft );
	trimer.set_fullatom_flag( true, false ); // booleans: fullatom, repack
	FArray3D< Real > full_coord( 3, param::MAX_ATOM(), trimer.total_residue() ); // trimer full_coord

	// scratch data
	ResidueRange const monomer_range( 1, monomer.total_residue() );
	Integer counter = 1;
	Integer offset = 0;

	// duplicate data 3 times for trimer
	for ( Size i = 1; i <= 3; ++i ) {
		epigraft::transfer_identity_info_between_Poses( monomer, monomer_range, trimer, counter, true ); // boolean: keep_pdb_info
		counter += monomer.total_residue();
	}
	assert( counter == ( trimer.total_residue() + 1 ) );

	// monomer 1
	offset = 0;
	transform_full_coord( monomer, monomer1_transform, full_coord, offset );

	// monomer 2
	offset += monomer.total_residue();
	transform_full_coord( monomer, monomer2_transform, full_coord, offset );

	// monomer 3
	offset += monomer.total_residue();
	transform_full_coord( monomer, monomer3_transform, full_coord, offset );

	// finalize
	FArray3D_float Eposition( 3, param::MAX_POS, trimer.total_residue() );
	epigraft::full_coord_to_Eposition( trimer.total_residue(), full_coord, Eposition );
	trimer.set_coords( false, Eposition, full_coord, false ); // booleans: ideal_pose, check_missing

	// post modify pdb data so trimer chains get different pdb ids on output
	for ( Size i = 1, ie = monomer.total_residue(); i <= ie; ++i ) {
		trimer.pdb_info().set_pdb_chain( i, 'A' );
	}
	for ( Size i = 1 + monomer.total_residue(), ie = 2 * monomer.total_residue(); i <= ie; ++i ) {
		trimer.pdb_info().set_pdb_chain( i, 'B' );
	}
	for ( Size i = 1 + 2 * monomer.total_residue(), ie = 3 * monomer.total_residue(); i <= ie; ++i ) {
		trimer.pdb_info().set_pdb_chain( i, 'C' );
	}

	// use pdb info on output
	trimer.pdb_info().set_use_pdb_numbering( true );
}

}
}

