// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   MatchComponent.hh
/// @brief  Stores results for the component of a match, i.e. two pairs of endpoint residues
/// @brief  between loop and scaffold.  Plain-data class, almost everything is public.
/// @note   Meant to be used with and within MatchResult class.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_match_MatchComponent_HH_
#define INCLUDED_epigraft_match_MatchComponent_HH_

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_constants.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/conformation/DihedralInfo.hh>

// C++ headers
#include <set>


namespace epigraft {
namespace match {


/// @brief  Stores results for the component of a match, i.e. two pairs of endpoint residues
/// @brief  between loop and scaffold.  Plain-data class, almost everything is public.
/// @note   Meant to be used with and within MatchResult class.
class MatchComponent {
	
	public: // construct/destruct
	
		/// @brief default constructor
		inline
		MatchComponent() :
		    loop_id( -1 ),
		    overall_rms( MATCH_INFINITY ),
		    rms_over_length( MATCH_INFINITY ),
		    n_terminal_rms( MATCH_INFINITY ),
		    c_terminal_rms( MATCH_INFINITY ),
		    n_closure_angle( MATCH_INFINITY ),
		    c_closure_angle( MATCH_INFINITY ),
		    intra_clash( MATCH_INFINITY ),
		    cbeta_neighbors( -1 )
		{}
		
		/// @brief constructor
		inline
		MatchComponent(
			Integer const & loop_id_,
			ResidueRange const & loop_subrange_,
			ResidueRange const & native_loop_subrange_,
			ResidueRange const & scaffold_gap_range_
		) : loop_id( loop_id_ ),
		    loop_subrange( loop_subrange_ ),
		    native_loop_subrange( native_loop_subrange_ ),
		    scaffold_gap_range( scaffold_gap_range_ ),
		    overall_rms( MATCH_INFINITY ),
		    rms_over_length( MATCH_INFINITY ),
		    n_terminal_rms( MATCH_INFINITY ),
		    c_terminal_rms( MATCH_INFINITY ),
		    n_closure_angle( MATCH_INFINITY ),
		    c_closure_angle( MATCH_INFINITY ),
		    intra_clash( MATCH_INFINITY ),
		    cbeta_neighbors( -1 )
		{}
		
		/// @brief copy constructor
		inline
		MatchComponent(
			MatchComponent const & m
		) : loop_id( m.loop_id ),
		    loop_subrange( m.loop_subrange ),
		    native_loop_subrange( m.native_loop_subrange ),
		    scaffold_gap_range( m.scaffold_gap_range ),
		    dihedrals( m.dihedrals ),
		    overall_rms( m.overall_rms ),
		    rms_over_length( m.rms_over_length ),
		    n_terminal_rms( m.n_terminal_rms ),
		    c_terminal_rms( m.c_terminal_rms ),
		    n_closure_angle( m.n_closure_angle ),
		    c_closure_angle( m.c_closure_angle ),
		    intra_clash( m.intra_clash ),
		    cbeta_neighbors( m.cbeta_neighbors )
		{}
		
		/// @brief default destructor
		inline
		~MatchComponent() {}
		
	
	public: // assignment
	
		/// @brief copy assignment
		inline
		MatchComponent &
		operator =( MatchComponent const & m )
		{
			if ( this != &m ) {
				loop_id = m.loop_id;
				loop_subrange = m.loop_subrange;
				native_loop_subrange = m.native_loop_subrange;
				scaffold_gap_range = m.scaffold_gap_range;
				dihedrals = m.dihedrals;
				
				overall_rms = m.overall_rms;
				rms_over_length = m.rms_over_length;
				n_terminal_rms = m.n_terminal_rms;
				c_terminal_rms = m.c_terminal_rms;
				n_closure_angle = m.n_closure_angle;
				c_closure_angle = m.c_closure_angle;
				intra_clash = m.intra_clash;
				
				cbeta_neighbors = m.cbeta_neighbors;
			}
			return *this;
		}
		
	
	public: // operators
	
		/// @brief this.scaffold_gap_range < rvalue.scaffold_gap_range
		inline
		bool
		operator <( MatchComponent const & rvalue ) const
		{
			return scaffold_gap_range < rvalue.scaffold_gap_range;
		}
	
	
	public: // data

		// match data
		Integer loop_id;
		ResidueRange loop_subrange;
		ResidueRange native_loop_subrange;
		ResidueRange scaffold_gap_range;
		std::set< epigraft::conformation::DihedralInfo > dihedrals;

		// filter data
		Real overall_rms;
		Real rms_over_length;
		Real n_terminal_rms;
		Real c_terminal_rms;
		Real n_closure_angle;
		Real c_closure_angle;
		Real intra_clash;
		
		// additional statistics and filter data, not computed during
		// match process, only during post-processing
		Integer cbeta_neighbors; // counts number of cbeta neighbors within distance cutoff
};


} // match
} // epigraft


#endif /*INCLUDED_epigraft_match_MatchComponent_HH_*/
