// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   MatchResult.cc
/// @brief  Stores results for a match containing pairs of endpoint residues.  Plain-data class, almost
/// @brief  everything is public.
/// @note   information for the primary loop is always stored in the first (index 1) for utility::vector1 objects
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/match/MatchResult.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/conformation/DihedralInfo.hh>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <iomanip>
#include <set>
#include <sstream>
#include <string>


namespace epigraft {
namespace match {


// (private static) output data
utility::vector1< std::string > MatchResult::output_tags_ = utility::vector1< std::string >(); // names for output table columns


/// @brief report status for table output
std::string
MatchResult::to_string(
	std::string const & scaffold_filename,
	std::string const & output_pdb_prefix,
	std::string const & line_prefix,
	Size const nres_scaffold
) const
{
	using epigraft::conformation::DihedralInfo;

	if ( output_tags_.empty() ) {
		init_output_tags();
	}

	std::string empty( "-" );

	std::stringstream ss;
	ss.setf( std::ios::fixed, std::ios::floatfield ); // formatting for reals, precision is adjusted on case-by-case basis
	ss.setf( std::ios::right, std::ios::adjustfield ); // right justify everything

	for ( Integer i = 1, ie = components.size(); i <= ie; ++i ) {
		MatchComponent const & c = components[ i ];
		Integer counter = 0;

		// line prefix
		ss << line_prefix;

		// filename
		if ( i == 1 ) {
			add_column( counter, ss, scaffold_filename );
		} else {
			add_column( counter, ss, "*" );
		}

		// loop_id
		add_column( counter, ss, c.loop_id );

		// align_sys
		if ( i == 1 ) {
			add_column( counter, ss, system_name );
		} else {
			add_column( counter, ss, empty );
		}

		// gap_begin
		add_column( counter, ss, c.scaffold_gap_range.begin() );

		// gap_end
		add_column( counter, ss, c.scaffold_gap_range.end() );

		// native_loop_begin
		add_column( counter, ss, c.native_loop_subrange.begin() );

		// native_loop_end
		add_column( counter, ss, c.native_loop_subrange.end() );
		// dihedrals: D_1 to D_8 (possibly only to D_6, if handling single break)
		for ( std::set< DihedralInfo >::const_iterator d = c.dihedrals.begin(), de = c.dihedrals.end(); d != de; ++d ) {
			DihedralInfo const & dihedral = *d;

			std::stringstream dss;
			dss.precision( 8 );
			dss << dihedral.residue() << ":" << dihedral.type() << ":" << dihedral.angle();

			add_column( counter, ss, dss.str() );
		}

		// if less than eight dihedrals, pad with '-'
		if ( c.dihedrals.size() < 8 ) {
			for ( Integer d = c.dihedrals.size() + 1; d <= 8; ++d) {
				add_column( counter, ss, empty );
			}
		}

		// transformation matrix
		if ( i == 1 ) {
			ss.precision( 8 );

			// R11
			add_column( counter, ss, transformation_matrix[ 0 ] );

			// R21
			add_column( counter, ss, transformation_matrix[ 1 ] );

			// R31
			add_column( counter, ss, transformation_matrix[ 2 ] );

			// R12
			add_column( counter, ss, transformation_matrix[ 4 ] );

			// R22
			add_column( counter, ss, transformation_matrix[ 5 ] );

			// R32
			add_column( counter, ss, transformation_matrix[ 6 ] );

			// R13
			add_column( counter, ss, transformation_matrix[ 8 ] );

			// R23
			add_column( counter, ss, transformation_matrix[ 9 ] );

			// R33
			add_column( counter, ss, transformation_matrix[ 10 ] );

			// T_x
			add_column( counter, ss, transformation_matrix[ 12 ] );

			// T_y
			add_column( counter, ss, transformation_matrix[ 13 ] );

			// T_z
			add_column( counter, ss, transformation_matrix[ 14 ] );

		} else {
			for ( Integer j = 1; j <= 12; ++j ) {
				add_column( counter, ss, empty );
			}
		}

		ss.precision( 3 );

		// overall_rms
		add_column( counter, ss, c.overall_rms );

		// n_terminal_rms
		add_column( counter, ss, c.n_terminal_rms );

		// c_terminal_rms
		add_column( counter, ss, c.c_terminal_rms );

		// rms_over_length
		add_column( counter, ss, c.rms_over_length );

		// intra_clash
		add_column( counter, ss, c.intra_clash );

		// inter_clash
		if ( i == 1 ) {
			add_column( counter, ss, inter_clash );
		} else {
			add_column( counter, ss, empty );
		}

		// orientation
		add_column( counter, ss, c.n_closure_angle );
		add_column( counter, ss, c.c_closure_angle );

		// cbeta neighbors
		add_column( counter, ss, c.cbeta_neighbors );

		// number of residues in the scaffold
		add_column( counter, ss, nres_scaffold );

		// output_pdb
		add_column( counter, ss, output_pdb_prefix, "" );

		// if on primary line, write any additional output data
		if ( i == 1 && !additional_output_data_.empty() ) {
			for ( utility::vector1< std::string >::const_iterator o = additional_output_data_.begin(), oe = additional_output_data_.end(); o != oe; ++o ) {
				add_column( counter, ss, *o );
			}
		}

		ss << "\n";
	}

	return ss.str();
}


/// @brief header for table output
std::string
MatchResult::header_string()
{
	if ( output_tags_.size() < 1 ) {
		init_output_tags();
	}

	std::string spacer( "     " );
	std::stringstream ss;

//	ss << "# "; // comment delimiter -- NOTE: comment delimiter now included in 'filename' column, see input_output_tags()

	ss.setf( std::ios::right, std::ios::adjustfield ); // right justify everything
	for ( utility::vector1< std::string >::const_iterator t = output_tags_.begin(), te = output_tags_.end() - 1; t != te; ++t ) {
		ss << *t << spacer;
	}
	ss << *( output_tags_.end() - 1 ) << "\n";

	return ss.str();
}


/// @brief add output tag (to static class list)
void
MatchResult::add_output_tag(
	std::string const & tag
)
{
	if ( output_tags_.empty() ) {
		init_output_tags();
	}
	output_tags_.push_back( tag );
}


/// @brief remove last element from output tag (in static class list)
void
MatchResult::remove_last_output_tag()
{
	output_tags_.pop_back();
}


/// @brief format and add data column to stringstream, for .to_string()
template< typename T >
void
MatchResult::add_column(
	int & counter,
	std::stringstream & ss,
	T const & val,
	std::string const & spacer
) const
{
	counter++;
	ss << std::setw( output_tags_[ counter ].length() );
	ss << val;
	ss << spacer;
}


/// @brief init names for output table columns
void
MatchResult::init_output_tags()
{
	output_tags_.push_back( "#######filename" ); // includes comment delimiter
	output_tags_.push_back( "loop_id" );
	output_tags_.push_back( "align_sys" );
	output_tags_.push_back( "gap_begin" );
	output_tags_.push_back( "gap_end" );
	output_tags_.push_back( "native_loop_begin" );
	output_tags_.push_back( "native_loop_end" );
	output_tags_.push_back( "               D_1" );
	output_tags_.push_back( "               D_2" );
	output_tags_.push_back( "               D_3" );
	output_tags_.push_back( "               D_4" );
	output_tags_.push_back( "               D_5" );
	output_tags_.push_back( "               D_6" );
	output_tags_.push_back( "               D_7" );
	output_tags_.push_back( "               D_8" );
	output_tags_.push_back( "         R11" );
	output_tags_.push_back( "         R21" );
	output_tags_.push_back( "         R31" );
	output_tags_.push_back( "         R12" );
	output_tags_.push_back( "         R22" );
	output_tags_.push_back( "         R32" );
	output_tags_.push_back( "         R13" );
	output_tags_.push_back( "         R23" );
	output_tags_.push_back( "         R33" );
	output_tags_.push_back( "          T_x" );
	output_tags_.push_back( "          T_y" );
	output_tags_.push_back( "          T_z" );
	output_tags_.push_back( "overall_rms" );
	output_tags_.push_back( "n_terminal_rms" );
	output_tags_.push_back( "c_terminal_rms" );
	output_tags_.push_back( "rms_over_length" );
	output_tags_.push_back( "intra_clash" );
	output_tags_.push_back( "inter_clash" );
	output_tags_.push_back( "  n_angle" );
	output_tags_.push_back( "  c_angle" );
	output_tags_.push_back( "cbeta_neighbors" );
	output_tags_.push_back( "nres_scaffold" );
	output_tags_.push_back( "output_pdb_prefix" );
}

} // namespace match
} // namespace epigraft
