// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   MatchResult.hh
/// @brief  Stores results for a match containing pairs of endpoint residues.  Plain-data class, almost
/// @brief  everything is public.
/// @note   information for the primary loop is always stored in the first (index 1) for utility::vector1 objects
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_match_MatchResult_HH_
#define INCLUDED_epigraft_match_MatchResult_HH_

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_constants.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/ResidueRange.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <sstream>
#include <string>


namespace epigraft {
namespace match {


/// @brief  Stores results for a match containing pairs of endpoint residues.  Plain-data class, almost
/// @brief  everything is public.
/// @note   information for the primary loop is always stored in the first (index 1) for utility::vector1 objects
class MatchResult {


	public: // construct/destruct

		/// @brief default constructor
		inline
		MatchResult() :
			transformation_matrix( 4, 4, 0.0f ),
			overall_rms( MATCH_INFINITY ),
			inter_clash( MATCH_INFINITY )
		{}

		/// @brief constructor
		inline
		MatchResult(
			epigraft::match::align::AlignmentSystem::SystemType const & system_type_,
			epigraft::match::align::AlignmentSystem::AlignmentCenter const & alignment_center_,
			std::string const & system_name_
		) : system_type( system_type_ ),
			alignment_center( alignment_center_ ),
		    system_name( system_name_ ),
		    transformation_matrix( FOUR_BY_FOUR_IDENTITY_REAL ),
		    overall_rms( MATCH_INFINITY ),
		    inter_clash( MATCH_INFINITY )
		{}

		/// @brief copy constructor
		inline
		MatchResult(
			MatchResult const & m
		) : system_type( m.system_type ),
		    alignment_center( m.alignment_center ),
		    system_name( m.system_name ),
		    components( m.components ),
		    transformation_matrix( m.transformation_matrix ),
		    overall_rms( m.overall_rms ),
		    inter_clash( m.inter_clash ),
		    additional_output_data_( m.additional_output_data_ )
		{}

		/// @brief default destructor
		inline
		~MatchResult() {}


	public: // assignment

		/// @brief copy assignment
		inline
		MatchResult &
		operator =( MatchResult const & m )
		{
			if ( this != &m ) {
				system_type = m.system_type;
				alignment_center = m.alignment_center;
				system_name = m.system_name;
				components = m.components;
				transformation_matrix = m.transformation_matrix;
				overall_rms = m.overall_rms;
				inter_clash = m.inter_clash;
				additional_output_data_ = m.additional_output_data_;
			}
			return *this;
		}


	public: // additional output

		/// @brief add additional output data
		inline
		void
		add_additional_output_data(
			std::string const & data
		)
		{
			additional_output_data_.push_back( data );
		}

		/// @brief remove last additional output data
		inline
		void
		remove_last_additional_output_data()
		{
			additional_output_data_.pop_back();
		}

	public: // status

		/// @brief report status for table output
		std::string
		to_string(
			std::string const & scaffold_filename,
			std::string const & output_pdb_filename,
			std::string const & line_prefix = "",
			Size const nres_scaffold = 0
		) const;


		/// @brief header for table output
		static
		std::string
		header_string();


	public: // static

		/// @brief add output tag (to static class list)
		static
		void
		add_output_tag(
			std::string const & tag
		);

		/// @brief pop last element from output tag (in static class list)
		static
		void
		remove_last_output_tag();


	private: // for output

		/// @brief format and add data column to stringstream, for .to_string()
		template< typename T >
		void
		add_column(
			int & counter,
			std::stringstream & ss,
			T const & val,
			std::string const & spacer = "     "
		) const;

		/// @brief init names for output table columns
		static
		void
		init_output_tags();


	public: // data

		// NOTE: for utility::vector1 data types, primary loop information is always the first
		//       element!

		epigraft::match::align::AlignmentSystem::SystemType system_type;
		epigraft::match::align::AlignmentSystem::AlignmentCenter alignment_center;
		std::string system_name;

		utility::vector1< MatchComponent > components;

		FArray2D_float transformation_matrix; // 4x4 matrix encoding rotation and translation

		// filter data
		Real overall_rms;
		Real inter_clash;

		// Additional output data for match result to be tacked onto end of columns.
		// Unfortunately C++ has no standard polymorphic containers and I don't
		// feel like writing one up so everything is a string here.
		utility::vector1< std::string > additional_output_data_;


	private:

		// output data
		static utility::vector1< std::string > output_tags_; // names for output table columns
};

} // namespace match
} // namespace epigraft


#endif /*INCLUDED_epigraft_match_MatchResult_HH_*/
