// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   additional_filters.cc
/// @brief  functions for additional filtering and statistics of matches
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// unit headers
#include <epigraft/match/additional_filters.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/GraftOptions.hh>

// rosetta headers
#include <pose.h>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// utility headers
#include <utility/vector1.hh>

namespace epigraft{
namespace match {


/// @brief filters match results for given rms
void
strict_rms_filter(
	Real const & rms,
	utility::vector1< MatchResult > const & match_results,
	utility::vector1< MatchResult > & filtered_results
)
{
	using epigraft::match::align::AlignmentSystem;

	for ( utility::vector1< MatchResult >::const_iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {

		MatchResult const & match_result = *m;

		// check primary component
		switch ( match_result.system_type ) {
			case AlignmentSystem::N2C:
				if ( match_result.components[ 1 ].c_terminal_rms > rms ) {
					continue;
				}
				break;
			case AlignmentSystem::C2N:
				if ( match_result.components[ 1 ].n_terminal_rms > rms ) {
					continue;
				}
				break;
			default:
				if ( ( match_result.components[ 1 ].n_terminal_rms > rms || match_result.components[ 1 ].c_terminal_rms > rms )
				     && (match_result.components[ 1 ].overall_rms > rms ) ) {
					continue;
				}
				break;
		}

		// check secondary components
		bool accept = true;
		for ( utility::vector1< MatchComponent >::const_iterator c = match_result.components.begin() + 1, ce = match_result.components.end(); c < ce; ++c ) {
			MatchComponent const & component = *c;
			if ( component.n_terminal_rms > rms || component.c_terminal_rms > rms ) {
				accept = false;
				break;
			}
		}

		// passed rms filter, add to list
		if ( accept ) {
			filtered_results.push_back( match_result );
		}
	}
}


/// @brief filters match results to remove incomplete matches,
/// @brief ie if you want to match 3 epitope loops,
/// @brief then singleton and double matches will be removed
///vds
void
filter_out_incomplete_matches(
	std::size_t const & num_loops,
	utility::vector1< MatchResult > const & match_results,
	utility::vector1< MatchResult > & filtered_results
)
{
	for ( utility::vector1< MatchResult >::const_iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {

		MatchResult const & match_result = *m;
		if (match_result.components.size() == num_loops) { //vds change to a small number comparison?
			filtered_results.push_back( match_result );
		}
	}
}


/// @brief filters match results for C-beta positions on the alanine scaffold that
/// @brief are close to any heavy atoms of the given filter structure
/// @details cbeta number will be appended to output columns
void
cbeta_spatial_filter(
	GraftOptions const & options,
	Pose & ala_scaffold,
	utility::vector1< MatchResult > const & match_results,
	Pose & filter_structure,
	utility::vector1< MatchResult > & filtered_results
)
{
	// options mirrored for clarity
	Real DISTANCE_CUTOFF = options.spatial_filter_distance_cutoff;
	Real OCTREE_CUBE_SIZE = ( DISTANCE_CUTOFF > 6 ) ? DISTANCE_CUTOFF + 0.001 : 6.0; // add epsilon (0.001) for floating point error
	Size MIN_CBETA_CUTOFF = options.spatial_filter_min_cbeta;
	Size MAX_CBETA_CUTOFF = options.spatial_filter_max_cbeta;

	std::ostringstream oss; // for output

	rootstock::BoundingBox bb = bounding_box( ala_scaffold );

	for ( utility::vector1< MatchResult >::const_iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {
		MatchResult const & match_result = *m;

		// transform spatial filter structure
		Pose aligned_filter_structure;
		aligned_filter_structure = filter_structure;
		aligned_filter_structure.transform_GL( match_result.transformation_matrix );

		// make gapped pose and octree for distance check
		utility::vector1< bool > keep( ala_scaffold.total_residue(), true ); // keep the following residues but remove those in all match ranges
		for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
			for ( Integer residue = match_result.components[ c ].scaffold_gap_range.begin(), last_residue = match_result.components[ c ].scaffold_gap_range.end(); residue <= last_residue; ++residue ) {
				keep[ residue ] = false;
			}
		}
		Pose gapped_scaffold;
		fragment_Pose( ala_scaffold, keep, gapped_scaffold );
		rootstock::Octree< AtomPoint > octree_gapped( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
		fill_octree_CB( octree_gapped, gapped_scaffold );

		// count number of C-beta within a certain distance of spatial filter heavy atoms
		FArray3D_float const & fc = aligned_filter_structure.full_coord(); // cache
		std::set< AtomPoint > close_cbetas;
		for ( Integer res = 1, last_res = aligned_filter_structure.total_residue(); res <= last_res; ++res ) {
			int const aa( aligned_filter_structure.res( res ) );
			int const aav( aligned_filter_structure.res_variant( res ) );

			for ( int atom = 1; atom <= aaproperties_pack::nheavyatoms( aa, aav ); ++atom ) {
				std::vector< AtomPoint > nn = octree_gapped.near_neighbors( DISTANCE_CUTOFF, (Real)fc( 1, atom, res ), (Real)fc( 2, atom, res ), (Real)fc( 3, atom, res ) );
				close_cbetas.insert( nn.begin(), nn.end() );
			}
		}

		if ( MIN_CBETA_CUTOFF <= close_cbetas.size() && close_cbetas.size() <= MAX_CBETA_CUTOFF ) {
			// add new result
			filtered_results.push_back( match_result );

			// record number
			oss << close_cbetas.size();
			filtered_results.back().add_additional_output_data( oss.str() );

			// reset conversion buffer
			oss.str( "" );
		}
	}

}


/// @brief compute number of c-beta neighbors per component (heavy atoms only)
void
cbeta_neighbors_per_component(
	GraftOptions const & options,
	Pose & epitope,
	Pose & ala_scaffold,
	utility::vector1< MatchResult > & match_results
)
{
	Real CBETA_DISTANCE_CUTOFF = options.cbeta_neighbors_distance_cutoff;
	Real OCTREE_CUBE_SIZE = ( CBETA_DISTANCE_CUTOFF > 6 ) ? CBETA_DISTANCE_CUTOFF + 0.001 : 6.0; // add epsilon (0.001) for floating point error

	rootstock::BoundingBox bb = bounding_box( ala_scaffold );

	for ( utility::vector1< MatchResult >::iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {
		MatchResult & match_result = *m;

		// make aligned antigen
		Pose aligned_epitope;
		aligned_epitope = epitope;
		aligned_epitope.transform_GL( match_result.transformation_matrix );
		FArray3D_float const & aligned_epitope_full_coord = aligned_epitope.full_coord(); // cache

		// make gapped scaffold without components and octree for distance check
		utility::vector1< bool > keep( ala_scaffold.total_residue(), true ); // keep the following residues but remove those in all match ranges
		for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
			for ( Integer residue = match_result.components[ c ].scaffold_gap_range.begin(), last_residue = match_result.components[ c ].scaffold_gap_range.end(); residue <= last_residue; ++residue ) {
				keep[ residue ] = false;
			}
		}
		Pose gapped_scaffold;
		fragment_Pose( ala_scaffold, keep, gapped_scaffold );
		rootstock::Octree< AtomPoint > octree_gapped( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
		fill_octree_CB( octree_gapped, gapped_scaffold );

		// for each component, count number of C-beta within a certain distance of component heavy atoms
		for ( utility::vector1< MatchComponent >::iterator c = match_result.components.begin(), ce = match_result.components.end(); c != ce; ++c ) {
			MatchComponent & component = *c;

			// count cbetas
			std::set< AtomPoint > close_cbetas;
			for ( Integer res = component.loop_subrange.begin(), last_res = component.loop_subrange.end(); res <= last_res; ++res ) {
				int const aa( aligned_epitope.res( res ) );
				int const aav( aligned_epitope.res_variant( res ) );

				for ( int atom = 1; atom <= aaproperties_pack::nheavyatoms( aa, aav ); ++atom ) {
					std::vector< AtomPoint > nn = octree_gapped.near_neighbors( CBETA_DISTANCE_CUTOFF,
					                                                            (Real)aligned_epitope_full_coord( 1, atom, res ),
					                                                            (Real)aligned_epitope_full_coord( 2, atom, res ),
					                                                            (Real)aligned_epitope_full_coord( 3, atom, res ) );
					close_cbetas.insert( nn.begin(), nn.end() );
				}
			}

			component.cbeta_neighbors = close_cbetas.size();
		}
	}
}


} // namespace match
} // namespace epigraft
