// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   AlignmentSystem.hh
/// @brief  Abstract base class for alignment systems.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_match_align_AlignmentSystem_HH_
#define INCLUDED_epigraft_match_align_AlignmentSystem_HH_

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_functions.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/ResidueRange.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// utility headers
#include <utility/pointer/ReferenceCount.hh>

// C++ headers
#include <string>


namespace epigraft {
namespace match {
namespace align {


/// @brief  Abstract base class for alignment systems.
class AlignmentSystem : public utility::pointer::ReferenceCount {	

	private: // typedefs
	
		typedef utility::pointer::ReferenceCount Super;


	public: // enum
	
		/// @brief alignment system type
		enum SystemType { N2C, C2N, SUPERPOSITION, ENDPOINT, SPECIFIC_SUPERPOSITION };
		
		/// @brief alignment center
		enum AlignmentCenter { N, CA, C, DOUBLEBREAK };
		
		/// @brief rms type (currently only used in S, SS systems)
		enum RMSType { RMS, RMS_OVER_LENGTH };


	public: // construct/destruct
	
		/// @brief default constructor
		inline
		AlignmentSystem()
		{}
		
		/// @brief copy constructor
		inline
		AlignmentSystem(
			AlignmentSystem const & s
		) : ReferenceCount(),
		    system_type_( s.system_type_ ),
		    alignment_center_( s.alignment_center_ )
		{}
		
		/// @brief default destructor
		inline
		virtual
		~AlignmentSystem() {}


	public: // assignment
	
		/// @brief copy assignment
		inline
		AlignmentSystem &
		operator =( AlignmentSystem const & s )
		{
			if ( this != &s ) {
				system_type_ = s.system_type_;
				alignment_center_ = s.alignment_center_;
				Super::operator =( s );
			}
			return *this;
		}


	public: // accessors
	
		/// @brief alignment system type
		inline
		SystemType const &
		type() const
		{
			return system_type_;
		}
		
		/// @brief alignment center
		inline
		AlignmentCenter const &
		alignment_center() const
		{
			return alignment_center_;
		}


	public: // alignment
	
		/// @brief align loop onto scaffold, fills match component and transformation matrix, check if alignment passes set filters
		/// @note  match component is filled with rms values and other orientation/alignment filters
		virtual
		bool
		check_alignment(
			Pose const & scaffold,
			Pose const & loop,
			MatchComponent & match_component,
			FArray2D_float & transformation_matrix
		) = 0;

		
		/// @brief place loop into scaffold to create epitope-scaffold, loop must already be in correct orientation
		virtual
		void
		place_loop_into_scaffold(
			Pose const & scaffold,
			ResidueRange const & match_range,
			Pose const & loop,
			Pose & epitope_scaffold
		) = 0;
		

	public: // status methods

		/// @brief name of system
		virtual
		std::string
		name() const = 0;

		/// @brief report status
		virtual
		std::string
		to_string() const = 0;
		
	
	protected: // data specifying system type
	
		SystemType system_type_;
		AlignmentCenter alignment_center_;

};


} // namespace align
} // namespace match
} // namespace epigraft

#endif /*INCLUDED_epigraft_match_align_AlignmentSystem_HH_*/
