// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   C2N_C_Align.hh
/// @brief  "C2N" C-centered alignment system -- align by superposition of match residue -1 (CA, C)
/// @brief  and match residue (N)
/// @author Bill Schief (schief@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


#ifndef INCLUDED_epigraft_match_align_C2N_C_Align_HH_
#define INCLUDED_epigraft_match_align_C2N_C_Align_HH_

// base class header
#include <epigraft/match/align/AlignmentSystem.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/MatchComponent.hh>


namespace epigraft {
namespace match {
namespace align {


/// @brief  "C2N" C-centered alignment system -- align by superposition of match residue -1 (CA, C)
/// @brief  and match residue (N)
class C2N_C_Align : public AlignmentSystem {
	
	private: // typedefs
	
		typedef AlignmentSystem Super;

	
	public: // construct/destruct
	
		/// @brief default constructor
		C2N_C_Align() : Super()
		{
			initialize();
		}
		
		/// @brief constructor
		C2N_C_Align(
			Real const & max_chainbreak_rms
		) : Super(),
		    max_chainbreak_rms_( max_chainbreak_rms )
		{
			initialize();
		}
		
		/// @brief copy constructor
		C2N_C_Align(
			C2N_C_Align const & a
		) : Super( a ),
		    max_chainbreak_rms_( a.max_chainbreak_rms_ )
		{}
		
		/// @brief default destructor
		~C2N_C_Align() {}
		
	public: // copy assignment
	
		/// @brief copy assignment
		inline
		C2N_C_Align &
		operator =( C2N_C_Align const & a )
		{
			if ( this != &a ) {
				Super::operator =( a );
				max_chainbreak_rms_ = a.max_chainbreak_rms_;
			}
			return *this;
		}
	
	
	public: // alignment
	
		/// @brief align loop onto scaffold, fills match component and transformation matrix, check if alignment passes set filters
		/// @note  match component is filled with rms values and other orientation/alignment filters
		bool
		check_alignment(
			Pose const & scaffold,
			Pose const & loop,
			MatchComponent & match_component,
			FArray2D_float & transformation_matrix
		);

		/// @brief place loop into (gly) scaffold to create epitope-scaffold, loop must already be in correct orientation
		/// @note  the entire match range on the scaffold is removed, while the entire loop is kept
		/// @note  micromanagement: c-terminal 'CA', '1HA', '2HA', 'C', 'O' atom positions are taken from the (gly) scaffold
		void
		place_loop_into_scaffold(
			pose_ns::Pose const & scaffold,
			ResidueRange const & match_range,
			pose_ns::Pose const & loop,
			pose_ns::Pose & epitope_scaffold
		);

		/// @brief micromanage c-terminus: c-terminal 'CA', '1HA', '2HA', 'C', 'O' atom positions are taken from the (gly) scaffold
		static
		void
		micromanage_takeoff(
			FArray3D_float const & scaffold_full_coord,
			Integer const & scaffold_residue,
			FArray3D_float & epitope_scaffold_full_coord,
			Integer const & epitope_scaffold_residue
		);

	
	public: // status methods

		/// @brief name of system
		std::string
		name() const
		{
			return "C2N_C";
		}
		
		/// @brief report status
		std::string
		to_string() const
		{
			return "C2N_C_Align: to_string() method not implemented.";
		}


	public: // accessors
	
		/// @brief max chainbreak (on c-terminal side) rms filter
		inline
		void
		set_max_chainbreak_rms(
			Real rms
		)
		{
			max_chainbreak_rms_ = rms;
		}


	private: // methods
	
		/// @brief extract atoms from match residue -1 (CA, C) and match residue (N)
		void
		extract_CACN(
			pose_ns::Pose const & scaffold,
			Integer const & residue,
			FArray2D_float & crd
		);


	private: // initialize
	
		/// @brief initialize
		void
		initialize()
		{
			system_type_ = C2N;
			alignment_center_ = C;
		}


	private: // data
	
		Real max_chainbreak_rms_;
	
};

} // namespace align
} // namespace match
} // namespace epigraft

#endif /*INCLUDED_epigraft_match_align_C2N_C_Align_HH_*/
