// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   E_Align.hh
/// @brief  "E" alignment system -- align by superposition of endpoint (match) residues.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)

// unit headers
#include <epigraft/match/align/E_Align.hh>

// package headers
#include <epigraft/epigraft_functions.hh>
#include <epigraft/ResidueRange.hh>

// Rosetta headers
#include <aaproperties_pack.h>
#include <refold.h> // GL transformation functions

namespace epigraft {
namespace match {
namespace align {


/// @brief align loop onto scaffold, fills match component and transformation matrix, check if alignment passes set filters
/// @note  transformation matrix is always filled regardless of return value
/// @note  match component is filled with rms values and other orientation/alignment filters
bool
E_Align::check_alignment(
	Pose const & scaffold,
	Pose const & loop,
	MatchComponent & match_component,
	FArray2D_float & transformation_matrix
)
{
	ResidueRange const & match_range( match_component.scaffold_gap_range );
	ResidueRange const loop_range( 1, loop.total_residue() );
	
	FArray3D_float const & scaffold_full_coord = scaffold.full_coord();
	FArray3D_float const & loop_full_coord = loop.full_coord();
	
	FArray2D_double match_crd; // coordinates for _two_ endpoint match residues
	extract_2bb_from_full_coord( scaffold_full_coord, match_range, match_crd );
	
	FArray2D_double loop_crd; // coordinates for _two_ endpoint loop residues
	extract_2bb_from_full_coord( loop_full_coord, loop_range, loop_crd );
	
	// do superposition
	FArray2D_double match_crd_oriented, loop_crd_oriented; // oriented version of coordinates
	double overall_rms = MATCH_INFINITY;
	superimpose( match_crd, loop_crd, match_crd_oriented, loop_crd_oriented, overall_rms ); // match_crd stays fixed, loop_crd moves
	
	// conversion between double -> float due to inconsistency between types used in routines
	FArray2D_float f_loop_crd( loop_crd );
	FArray2D_float f_loop_crd_oriented( loop_crd_oriented );
	
	// construct transformation matrix; it's silly to find the superposition and then find the
	// matrix again (differently), but the Rosetta superposition and rms routines are a mess, so
	// this appears to be the cleanest way to do it until I can code up my own absolute orientation
	// routines
	get_GL_matrix( f_loop_crd.a( 1, 1 ), f_loop_crd.a( 1, 2 ), f_loop_crd.a( 1, 3 ),
	               f_loop_crd_oriented.a( 1, 1 ), f_loop_crd_oriented.a( 1, 2 ), f_loop_crd_oriented.a( 1, 3 ),
	               transformation_matrix ); // loop_crd moves onto loop_crd_oriented
	
	// check rms filter
	match_component.overall_rms = (float)overall_rms;
	if ( match_component.overall_rms > max_overall_rms_ ) {
		return false;
	}


	// data for terminal rms
	FArray2D_float match_t( 3, 6 );
	FArray2D_float loop_t( 3, 6 );
	Integer idx;

	// setup for n terminal filters
	for ( Integer i = 0; i < 12; ++i ) {
		match_t[ i ] = (float)match_crd[ i ]; // double -> float here
		loop_t[ i ] = (float)loop_crd[ i ]; // double -> float here
	}
	idx = scaffold_full_coord.index( 1, 1, match_range.begin() + 1 ); // 'N' for match + 1
	for ( Integer i = 12; i < 18; ++i, ++idx ) {  // 'N' and then 'CA' which is used for closure angle below
		match_t[ i ] = scaffold_full_coord[ idx ];
	}
	idx = loop_full_coord.index( 1, 1, loop_range.begin() + 1 ); // 'N' for loop + 1
	for ( Integer i = 12; i < 18; ++i, ++idx ) {  // 'N' and then 'CA' which is used for closure angle below
		loop_t[ i ] = loop_full_coord[ idx ];
	}
	transform_crd_via_GL( transformation_matrix, loop_t );

	// compute n-terminal filters
	compute_terminus_filters( match_t, loop_t, match_component.n_terminal_rms, match_component.n_closure_angle );
	
	
	// setup for c terminal filters
	for ( Integer i = 0; i < 12; ++i ) {
		match_t[ i ] = (float)match_crd[ i + 12 ];
		loop_t[ i ] = (float)loop_crd[ i + 12 ];
	}
	idx = scaffold_full_coord.index( 1, 3, match_range.end() - 1 ); // 'C' for match - 1
	match_t[ 12 ] = scaffold_full_coord[ idx ];
	match_t[ 13 ] = scaffold_full_coord[ idx + 1 ];
	match_t[ 14 ] = scaffold_full_coord[ idx + 2 ];
	idx = scaffold_full_coord.index( 1, 2, match_range.end() - 1 ); // 'CA' for match - 1 for closure angle below
	match_t[ 15 ] = scaffold_full_coord[ idx ];
	match_t[ 16 ] = scaffold_full_coord[ idx + 1 ];
	match_t[ 17 ] = scaffold_full_coord[ idx + 2 ];
	idx = loop_full_coord.index( 1, 3, loop_range.end() - 1 ); // 'C' for loop - 1
	loop_t[ 12 ] = loop_full_coord[ idx ];
	loop_t[ 13 ] = loop_full_coord[ idx + 1 ];
	loop_t[ 14 ] = loop_full_coord[ idx + 2 ];
	idx = loop_full_coord.index( 1, 2, loop_range.end() - 1 ); // 'CA' for loop - 1 for closure angle below
	loop_t[ 15 ] = loop_full_coord[ idx ];
	loop_t[ 16 ] = loop_full_coord[ idx + 1 ];
	loop_t[ 17 ] = loop_full_coord[ idx + 2 ];
	transform_crd_via_GL( transformation_matrix, loop_t );

	// compute n-terminal filters
	compute_terminus_filters( match_t, loop_t, match_component.c_terminal_rms, match_component.c_closure_angle );

	return true;
}


/// @brief place loop into scaffold to create epitope-scaffold, loop must already be in correct orientation
/// @note  the entire match range on the scaffold is removed, while the entire loop is kept
void
E_Align::place_loop_into_scaffold(
	pose_ns::Pose const & scaffold,
	ResidueRange const & match_range,
	pose_ns::Pose const & loop,
	pose_ns::Pose & epitope_scaffold
)
{
	epigraft::replace_segment_in_Pose( scaffold, match_range, loop, epitope_scaffold );
}


} // namespace align
} // namespace match
} // namespace epigraft
