// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   E_Align.hh
/// @brief  "E" alignment system -- align by superposition of endpoint (match) residues.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)

#ifndef INCLUDED_epigraft_match_align_E_Align_HH_
#define INCLUDED_epigraft_match_align_E_Align_HH_

// base class header
#include <epigraft/match/align/AlignmentSystem.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/MatchComponent.hh>


namespace epigraft {
namespace match {
namespace align {


/// @brief  "E" alignment system -- align by superposition of endpoint (match) residues.
class E_Align : public AlignmentSystem {
	
	private: // typedefs
	
		typedef AlignmentSystem Super;

	
	public: // construct/destruct
	
		/// @brief default constructor
		E_Align() : Super() 
		{
			initialize();
		}
		
		/// @brief constructor
		E_Align(
			Real const & max_overall_rms
		) : Super(),
		    max_overall_rms_( max_overall_rms )
		{
			initialize();
		}
		
		/// @brief copy constructor
		E_Align(
			E_Align const & a
		) : Super( a ),
		    max_overall_rms_( a.max_overall_rms_ )
		{}
		
		/// @brief default destructor
		~E_Align() {}


	public: // copy assignment
	
		/// @brief copy assignment
		inline
		E_Align &
		operator =( E_Align const & a )
		{
			if ( this != &a ) {
				Super::operator =( a );
				max_overall_rms_ = a.max_overall_rms_;
			}
			return *this;
		}
	
	
	public: // alignment
	
		/// @brief align loop onto scaffold, fills match component and transformation matrix, check if alignment passes set filters
		/// @note  match component is filled with rms values and other orientation/alignment filters
		bool
		check_alignment(
			Pose const & scaffold,
			Pose const & loop,
			MatchComponent & match_component,
			FArray2D_float & transformation_matrix
		);

		/// @brief place loop into scaffold to create epitope-scaffold, loop must already be in correct orientation
		/// @note  the entire match range on the scaffold is removed, while the entire loop is kept
		void
		place_loop_into_scaffold(
			Pose const & scaffold,
			ResidueRange const & match_range,
			Pose const & loop,
			Pose & epitope_scaffold
		);
	
	public: // status methods

		/// @brief name of system
		std::string
		name() const
		{
			return "E";
		}
		
		/// @brief report status
		std::string
		to_string() const
		{
			return "E_Align: to_string() method not implemented.";
		}


	public: // accessors
	
		/// @brief max overall rms filter
		inline
		void
		set_max_overall_rms(
			Real rms
		)
		{
			max_overall_rms_ = rms;
		}


	private: // initialize
	
		/// @brief initialize
		void
		initialize()
		{
			system_type_ = ENDPOINT;
			alignment_center_ = DOUBLEBREAK;
		}


	private: // data
	
		Real max_overall_rms_;
	
};


} // namespace align
} // namespace match
} // namespace epigraft

#endif /*INCLUDED_epigraft_match_align_E_Align_HH_*/
