// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   N2C_N_Align.cc
/// @brief  "N2C" N-centered alignment system -- align by superposition of match residue (C)
/// @brief  and match residue +1 (N, CA)
/// @author Bill Schief (schief@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/match/align/N2C_N_Align.hh>

// package headers
#include <epigraft/epigraft_functions.hh>
#include <epigraft/ResidueRange.hh>

// Rosetta headers
#include <aaproperties_pack.h>
#include <refold.h> // GL transformation functions


// ObjexxFCL headers
#include <ObjexxFCL/FArray2D.hh>


namespace epigraft {
namespace match {
namespace align {


/// @brief align loop onto scaffold, fills match component and transformation matrix, check if alignment passes set filters
/// @note  transformation matrix is always filled regardless of return value
/// @note  match component is filled with rms values and other orientation/alignment filters
bool
N2C_N_Align::check_alignment(
	Pose const & scaffold,
	Pose const & loop,
	MatchComponent & match_component,
	FArray2D_float & transformation_matrix
)
{
	ResidueRange const & match_range( match_component.scaffold_gap_range );
	ResidueRange const loop_range( 1, loop.total_residue() );
	
	FArray2D_float match_crd; // coordinates for n-terminal endpoint match residue
	extract_CNCA( scaffold, match_range.begin(), match_crd );
	
	FArray2D_float loop_crd; // coordinates for n-terminal endpoint loop residue
	extract_CNCA( loop, loop_range.begin(), loop_crd );

	// superposition on n-terminal match residue (C) and match residue +1 (N, CA)
	get_GL_matrix( loop_crd.a( 1, 1 ), loop_crd.a( 1, 2 ), loop_crd.a( 1, 3 ),
	               match_crd.a( 1, 1 ), match_crd.a( 1, 2 ), match_crd.a( 1, 3 ),
	               transformation_matrix ); // loop_crd moves onto match_crd
	
	// now load coordinates for c-terminal endpoint match residue and loop residue
	extract_1bb_from_Pose( scaffold, match_range.end(), match_crd );
	extract_1bb_from_Pose( loop, loop_range.end(), loop_crd );

	// append 'C', 'CA' from residue prior to c-terminal match residue for chainbreak scoring and other filtering
	match_crd.redimension( 3, 6 );
	for ( Integer i = 1; i <= 3; ++i ) {
		match_crd( i, 5 ) = scaffold.full_coord()( i, 3, match_range.end() - 1 ); // 'C'
		match_crd( i, 6 ) = scaffold.full_coord()( i, 2, match_range.end() - 1 ); // 'CA'
	}

	// append 'C', 'CA' from residue prior to c-terminal loop residue for chainbreak scoring and other filtering
	loop_crd.redimension( 3, 6 );
	for ( Integer i = 1; i <= 3; ++i ) {
		loop_crd( i, 5 ) = loop.full_coord()( i, 3, loop_range.end() - 1 ); // 'C'
		loop_crd( i, 6 ) = loop.full_coord()( i, 2, loop_range.end() - 1 ); // 'CA'
	}
	
	// do transformation and measure chain break
	transform_crd_via_GL( transformation_matrix, loop_crd );
	
	// compute filters
	compute_terminus_filters( match_crd, loop_crd, match_component.c_terminal_rms, match_component.c_closure_angle );
	
	if ( match_component.c_terminal_rms > max_chainbreak_rms_ ) {
		return false;
	}
	
	// fill meaningless numbers for 'N2C' align in match_result
	match_component.overall_rms = MATCH_INFINITY;
	match_component.n_terminal_rms = MATCH_INFINITY;
	match_component.n_closure_angle = MATCH_INFINITY;

	return true;
}


/// @brief place loop into (gly) scaffold to create epitope-scaffold, loop must already be in correct orientation
/// @note  the entire match range on the scaffold is removed, while the entire loop is kept
/// @note  micromanagement: n-terminal 'N', 'NH', 'CA', '1HA', '2HA' atom positions are taken from the (gly) scaffold
void
N2C_N_Align::place_loop_into_scaffold(
	pose_ns::Pose const & scaffold,
	ResidueRange const & match_range,
	pose_ns::Pose const & loop,
	pose_ns::Pose & epitope_scaffold
)
{
	epigraft::replace_segment_in_Pose( scaffold, match_range, loop, epitope_scaffold );
	
	// below we micromanage n-terminal 'N', 'NH', 'CA', '1HA', '2HA' atom positions from scaffold
	// TODO: this is expensive because Pose::set_coords recalculates a bunch of stuff, the
	//       alternative is to implant the body of replace_segment_in_Pose here with the
	//       micromanaging appended at the end.
	FArray3D_float full_coord( epitope_scaffold.full_coord() );
	
	micromanage_takeoff( scaffold.full_coord(), match_range.begin(), full_coord, match_range.begin() );
	
	FArray3D_float Eposition;
	full_coord_to_Eposition( epitope_scaffold.total_residue(), full_coord, Eposition );
	
	epitope_scaffold.set_coords( false, Eposition, full_coord, false );
}


/// @brief micromanage n-terminus: n-terminal 'N', 'NH', 'CA', '1HA', '2HA' atom positions from scaffold
void
N2C_N_Align::micromanage_takeoff(
	FArray3D_float const & scaffold_full_coord,
	Integer const & scaffold_residue,
	FArray3D_float & epitope_scaffold_full_coord,
	Integer const & epitope_scaffold_residue
)
{
	for ( Integer i = 1; i <= 3; ++i ) {
		epitope_scaffold_full_coord( i, 1, epitope_scaffold_residue ) = scaffold_full_coord( i, 1, scaffold_residue ); // 'N'
		epitope_scaffold_full_coord( i, 2, epitope_scaffold_residue ) = scaffold_full_coord( i, 2, scaffold_residue ); // 'CA' of gly
		epitope_scaffold_full_coord( i, 5, epitope_scaffold_residue ) = scaffold_full_coord( i, 5, scaffold_residue ); // 'NH' of gly
		epitope_scaffold_full_coord( i, 6, epitope_scaffold_residue ) = scaffold_full_coord( i, 6, scaffold_residue ); // '1HA' of gly
		epitope_scaffold_full_coord( i, 7, epitope_scaffold_residue ) = scaffold_full_coord( i, 7, scaffold_residue ); // '2HA' of gly
	}
}


/// @brief extract atoms from match residue (C) and match residue +1 (N, CA)
void
N2C_N_Align::extract_CNCA(
	pose_ns::Pose const & scaffold,
	Integer const & residue,
	FArray2D_float & crd
)
{
	crd.dimension( 3, 3 ); // 1st dimension: x,y,z | 2nd dimension: C, N, CA
	
	FArray3D_float const & full_coord = scaffold.full_coord();
	
	// 0-based linear indexing for speed
	Integer idx = full_coord.index( 1, 3, residue ); // 'C'
	for ( Integer i = 0; i < 3; ++i, ++idx ) {
		crd[ i ] = full_coord[ idx ];
	}
	
	idx = full_coord.index( 1, 1, residue + 1 ); // 'N' and 'CA'
	for ( Integer i = 3; i < 9; ++i, ++idx ) {
		crd[ i ] = full_coord[ idx ];
	}
}


} // namespace align
} // namespace match
} // namespace epigraft
