// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   SS_Align.cc
/// @brief  "SS" alignment system -- align by superposition of specific residues within a range of residues.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)

// unit headers
#include <epigraft/match/align/SS_Align.hh>

// package headers
#include <epigraft/epigraft_functions.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/match/match_constants.hh>

// Rosetta headers
#include <aaproperties_pack.h>
#include <refold.h> // GL transformation functions

namespace epigraft {
namespace match {
namespace align {


/// @brief align loop onto scaffold, fills match component and transformation matrix, check if alignment passes set filters
/// @note  transformation matrix is always filled regardless of return value
/// @note  match component is filled with rms values and other orientation/alignment filters
bool
SS_Align::check_alignment(
	Pose const & scaffold,
	Pose const & loop,
	MatchComponent & match_component,
	FArray2D_float & transformation_matrix
)
{
	assert( superposition_residues_.size() > 0 ); // must have some superposition residues!

	ResidueRange const & match_range( match_component.scaffold_gap_range );
	ResidueRange const loop_range( 1, loop.total_residue() );

	// range filter, number of residues must be the same so specified superposition residues are
	// included
	if ( match_range.length() != loop_range.length() ) {
		return false;
	}

	// range must contain all specified superposition residues
	for ( std::set< Integer >::const_iterator ri = superposition_residues_.begin(), rie = superposition_residues_.end(); ri != rie; ++ri ) {
		if ( !match_component.native_loop_subrange.contains( *ri ) ) {
			return false;
		}
	}

	// compute offset to find position of specified superposition residues in the loop
	Integer const offset_for_loop = match_component.native_loop_subrange.begin() - loop_range.begin();
	Integer const offset_for_scaffold = match_range.begin() - loop_range.begin();

	std::set< Integer > loop_superposition_residues;
	std::set< Integer > scaffold_superposition_residues;
	for ( std::set< Integer >::const_iterator ri = superposition_residues_.begin(), rie = superposition_residues_.end(); ri != rie; ++ri ) {
		Integer const loop_r = *ri - offset_for_loop;
		loop_superposition_residues.insert( loop_r );
		scaffold_superposition_residues.insert( loop_r + offset_for_scaffold );
	}

	FArray2D_double match_crd; // coordinates for scaffold superposition residues
	extract_bb_from_Pose( scaffold, scaffold_superposition_residues, match_crd );

	FArray2D_double loop_crd; // coordinates for loop superposition residues
	extract_bb_from_Pose( loop, loop_superposition_residues, loop_crd );

	// do superposition
	FArray2D_double match_crd_oriented, loop_crd_oriented; // oriented version of coordinates
	double overall_rms = MATCH_INFINITY;
	superimpose( match_crd, loop_crd, match_crd_oriented, loop_crd_oriented, overall_rms ); // match_crd stays fixed, loop_crd moves

	// conversion between double -> float due to inconsistency between types used in routines
	FArray2D_float f_loop_crd( loop_crd );
	FArray2D_float f_loop_crd_oriented( loop_crd_oriented );

	// construct transformation matrix; it's silly to find the superposition and then find the
	// matrix again (differently), but the Rosetta superposition and rms routines are a mess, so
	// this appears to be the cleanest way to do it until I can code up my own absolute orientation
	// routines
	get_GL_matrix( f_loop_crd.a( 1, 1 ), f_loop_crd.a( 1, 2 ), f_loop_crd.a( 1, 3 ),
	               f_loop_crd_oriented.a( 1, 1 ), f_loop_crd_oriented.a( 1, 2 ), f_loop_crd_oriented.a( 1, 3 ),
	               transformation_matrix ); // loop_crd moves onto loop_crd_oriented

	// check rms filter
	match_component.overall_rms = static_cast< float >( overall_rms );
	match_component.rms_over_length = match_component.overall_rms / static_cast< float >( match_component.loop_subrange.length() );
	switch( rms_type_ ) {
		case RMS: {
			if ( match_component.overall_rms > max_overall_rms_ ) {
				return false;
			}
			break;
		}
		case RMS_OVER_LENGTH: {
			if ( match_component.rms_over_length > max_rms_over_length_ ) {
				return false;
			}
			break;
		}
	}


	// n-terminal filters
	compute_terminus_filters( match_crd, loop_crd_oriented, match_component.n_terminal_rms, match_component.n_closure_angle ); // can be computed directly
	match_component.n_terminal_rms = MATCH_INFINITY;

	// data for filters
	FArray2D_float match_t( 3, 6 );
	FArray2D_float loop_t( 3, 6 );

	// c-terminal filters
	Integer j = 0;
	for ( Integer i = match_crd.size() - 12, ie = match_crd.size(); i < ie; ++i, ++j ) { // last bb residue
		match_t[ j ] = (float)match_crd[ i ];
		loop_t[ j ] = f_loop_crd_oriented[ i ];
	}
	for ( Integer i = match_crd.size() - 18, ie = match_crd.size() - 15; i < ie; ++i, ++j ) { // 'C' of prior to last bb residue
		match_t[ j ] = (float)match_crd[ i ];
		loop_t[ j ] = f_loop_crd_oriented[ i ];
	}
	for ( Integer i = match_crd.size() - 21, ie = match_crd.size() - 18; i < ie; ++i, ++j ) { // 'CA' of prior to last bb residue
		match_t[ j ] = (float)match_crd[ i ];
		loop_t[ j ] = f_loop_crd_oriented[ i ];
	}

	// c-terminal filters
	compute_terminus_filters( match_t, loop_t, match_component.c_terminal_rms, match_component.c_closure_angle ); // can be computed directly
	match_component.c_terminal_rms = MATCH_INFINITY;

	return true;
}


/// @brief place loop into scaffold to create epitope-scaffold, loop must already be in correct orientation
/// @note  the entire match range on the scaffold is removed, while the entire loop is kept
void
SS_Align::place_loop_into_scaffold(
	Pose const & scaffold,
	ResidueRange const & match_range,
	Pose const & loop,
	Pose & epitope_scaffold
)
{
	epigraft::replace_segment_in_Pose( scaffold, match_range, loop, epitope_scaffold );
}


} // namespace align
} // namespace match
} // namespace epigraft
