// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   S_Align.hh
/// @brief  "S" alignment system -- align by superposition in a range of residues.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)

#ifndef INCLUDED_epigraft_match_align_S_Align_HH_
#define INCLUDED_epigraft_match_align_S_Align_HH_

// base class header
#include <epigraft/match/align/AlignmentSystem.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/MatchComponent.hh>


namespace epigraft {
namespace match {
namespace align {


/// @brief  "S" alignment system -- align by superposition in a range of residues.
class S_Align : public AlignmentSystem {
	
	private: // typedefs
	
		typedef AlignmentSystem Super;
	
	
	public: // construct/destruct
	
		/// @brief default constructor
		S_Align() : Super()
		{
			initialize();
		}
		
		/// @brief constructor
		S_Align(
			RMSType const & rms_type,
			Real const & rms_threshold
		) : Super(),
		    rms_type_( rms_type )
		{
			switch( rms_type_ ) {
				case RMS: {
					max_overall_rms_ = rms_threshold;
					max_rms_over_length_ = -1.0;
					break;
				}
				case RMS_OVER_LENGTH: {
					max_rms_over_length_ = rms_threshold;
					max_overall_rms_ = -1.0;
					break;
				}
			}
			initialize();
		}
		
		/// @brief copy constructor
		S_Align(
			S_Align const & a
		) : Super( a ),
		    rms_type_( a.rms_type_ ),
		    max_overall_rms_( a.max_overall_rms_ ),
		    max_rms_over_length_( a.max_rms_over_length_ )
		{}
		
		/// @brief default destructor
		~S_Align() {}
		
	public: // copy assignment
	
		/// @brief copy assignment
		inline
		S_Align &
		operator =( S_Align const & a )
		{
			if ( this != &a ) {
				Super::operator =( a );
				rms_type_ = a.rms_type_;
				max_overall_rms_ = a.max_overall_rms_;
				max_rms_over_length_ = a.max_rms_over_length_;
			}
			return *this;
		}
	
	
	public: // alignment
	
		/// @brief align loop onto scaffold, fills match component and transformation matrix, check if alignment passes set filters
		/// @note  match component is filled with rms values and other orientation/alignment filters
		bool
		check_alignment(
			Pose const & scaffold,
			Pose const & loop,
			MatchComponent & match_component,
			FArray2D_float & transformation_matrix
		);

		/// @brief place loop into scaffold to create epitope-scaffold, loop must already be in correct orientation
		/// @note  the entire match range on the scaffold is removed, while the entire loop is kept
		void
		place_loop_into_scaffold(
			Pose const & scaffold,
			ResidueRange const & match_range,
			Pose const & loop,
			Pose & epitope_scaffold
		);
	
	public: // status methods

		/// @brief name of system
		std::string
		name() const
		{
			return "S";
		}
		
		/// @brief report status
		std::string
		to_string() const
		{
			return "S_Align: to_string() method not implemented.";
		}


	public: // accessors
	
		/// @brief current rms check type
		inline
		RMSType const &
		rms_type()
		{
			return rms_type_;
		}
		
		/// @brief set rms check type
		inline
		void
		set_rms_type(
			RMSType const & rms_type
		)
		{
			rms_type_ = rms_type;
		}
		
		/// @brief get max overall rms filter
		inline
		Real const &
		max_overall_rms()
		{
			return max_overall_rms_;
		}
		
		/// @brief max overall rms filter
		inline
		void
		set_max_overall_rms(
			Real const & rms
		)
		{
			max_overall_rms_ = rms;
		}
		
		/// @brief get max rms over length filter
		inline
		Real const &
		max_rms_over_length()
		{
			return max_rms_over_length_;
		}
		
		/// @brief set max rms over length filter
		inline
		void
		set_max_rms_over_length(
			Real const & rms
		)
		{
			max_rms_over_length_ = rms;
		}


	private: // initialize
	
		/// @brief initialize
		void
		initialize()
		{
			system_type_ = SUPERPOSITION;
			alignment_center_ = DOUBLEBREAK;
		}


	private: // data

		RMSType rms_type_;
		Real max_overall_rms_;
		Real max_rms_over_length_;
		
};


} // namespace align
} // namespace match
} // namespace epigraft

#endif /*INCLUDED_epigraft_match_align_S_Align_HH_*/
