// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   combi_match.cc
/// @brief  build multi-point matches from primary components and near neighbor criteria
/// @note   this is a rough match variant
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)


// unit headers
#include <epigraft/match/combi_match.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_constants.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/match_functions.hh>
#include <epigraft/conformation/FluidLanding.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/epigraft_functions.hh>

// Rosetta headers
#include <pose.h>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <stack>


namespace epigraft {
namespace match {


/// @brief  build multi-point matches from primary components and near neighbor criteria
/// @note   this is a rough match variant
void
combi_match(
	GraftOptions const & options,
	AntibodyComplex & native,
	utility::vector1< LoopInfo > const & loops_to_scan,
	Pose & scaffold,
	utility::vector1< MatchResult > & singleton_matches,
	utility::vector1< MatchResult > & multiple_matches
)
{
	// options, mirrored from GraftOptions object here for clarity
	Real CA_DIST_CUTOFF = options.combi_match_ca_distance;
	Real CA_OCTREE_CUBE_SIZE = ( CA_DIST_CUTOFF > 6.0 ) ? CA_DIST_CUTOFF : 6.0;
	Integer MINIMUM_WIDTH_BETWEEN_GAPS = options.moveable_closure_residues * 2 + 1; // minimum width between gaps is dictated by the number of _scaffold_ residues adjacent
	                                                                                // to break that are allowed to move; the * 2 multiple is for two sides, and the + 1 is
	                                                                                // for one required stable anchor residue

	// compute proper max closure rms: if attempting fluid ends on landing, then this number needs to
	// be modified as below
	Real MAX_CLOSURE_RMS = options.max_closure_rms;
	if ( options.fluidize_landing ) {
		MAX_CLOSURE_RMS += options.recovery_rms_epsilon;
	}


	// store allowed scaffold ranges depending upon primary components
	std::map< Integer, std::set< ResidueRange > > all_allowed_scaffold_ranges; // loop_id -> set of scaffold subranges
	for ( utility::vector1< MatchResult >::const_iterator m = singleton_matches.begin(), me = singleton_matches.end(); m != me; ++m ) {
		MatchComponent const & primary_component = m->components[ 1 ];

		// get loop/associated scaffold range set from map (creates if necessary); note that loop ids are sort of artifical right now and are equivalent to indices in loops_to_scan
		all_allowed_scaffold_ranges[ primary_component.loop_id ].insert( primary_component.scaffold_gap_range );
	}


	FArray3D_float const & scaffold_full_coord = scaffold.full_coord(); // for performance

	// construct octree around scaffold CA
	rootstock::BoundingBox bb;
	for ( Integer res = 1, nres = scaffold.total_residue(); res <= nres; ++res ) { // just 'CA' first
		bb.add( AtomPoint( res, 2, scaffold_full_coord( 1, 2, res ), scaffold_full_coord( 2, 2, res ), scaffold_full_coord( 3, 2, res ) ) ); // 'CA'
	}
	rootstock::Octree< AtomPoint > octree_s_CA( CA_OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
	fill_octree_CA( octree_s_CA, scaffold );

	// construct octree around scaffold, for constructing gapped scaffold octree later
	bb = bounding_box( scaffold );

	// stack for tracking matches to check
	std::stack< MatchResult > stack;

	// push all singleton match results onto stack
	for ( int i = 1, ie = singleton_matches.size(); i <= ie; ++i ) {
		stack.push( singleton_matches[ i ] );
	}


	while ( !stack.empty() ) {

		// grab result off of stack
		MatchResult match_result = stack.top();
		stack.pop();

		// reconstruct alignment of epitope
		Pose aligned_epitope;
		aligned_epitope = native.antigen();
		aligned_epitope.transform_GL( match_result.transformation_matrix );

		// run through all loops
		for ( Integer loop_id = 1, last_loop_id = loops_to_scan.size(); loop_id <= last_loop_id; ++loop_id ) {

			// skip if primary loop id
			if ( match_result.components[ 1 ].loop_id == loop_id ) {
				continue;
			}

			// skip if this secondary id is already in the list, prevents double counting, also catches
			// those matches that should not be allowed to grow further
			if ( match_result.components.size() > 1 && match_result.components.back().loop_id >= loop_id ) {
				continue;
			}

			LoopInfo const & current_loop = loops_to_scan[ loop_id ];

			// compute numbering offset in native (peptide) structure for this loop (epitope)
			Integer offset = compute_loop_offset( loops_to_scan, loop_id);

			// get allowed scaffold ranges
			std::set< ResidueRange> const & allowed_match_ranges = all_allowed_scaffold_ranges[ loop_id ];

			// run through each possible loop (epitope) sub-range
			for ( Integer subrange_id = 1, last_subrange_id = current_loop.native_subranges().size(); subrange_id <= last_subrange_id; ++subrange_id ) {
				ResidueRange const & native_subrange = current_loop.native_subranges()[ subrange_id ];
				ResidueRange const & internal_subrange = current_loop.internal_subranges()[ subrange_id ];
				ResidueRange loop_subrange( internal_subrange + offset );

				// find closest residues to n-terminus of loop subrange (by CA) in scaffold
				Integer ca_idx = aligned_epitope.full_coord().index( 1, 2, loop_subrange.begin() ); // 'CA'
				std::vector< AtomPoint > const viable_begin = octree_s_CA.near_neighbors( CA_DIST_CUTOFF, aligned_epitope.full_coord()[ ca_idx ], aligned_epitope.full_coord()[ ca_idx + 1 ], aligned_epitope.full_coord()[ ca_idx + 2 ] );

				// find closest residues to c-terminus of loop subrange (by CA) in scaffold
				ca_idx = aligned_epitope.full_coord().index( 1, 2, loop_subrange.end() ); // 'CA'
				std::vector< AtomPoint > const viable_end = octree_s_CA.near_neighbors( CA_DIST_CUTOFF, aligned_epitope.full_coord()[ ca_idx ], aligned_epitope.full_coord()[ ca_idx + 1 ], aligned_epitope.full_coord()[ ca_idx + 2 ] );

				// skip if an endpoint is completely missing
				if ( viable_begin.size() == 0 || viable_end.size() == 0 ) {
					continue;
				}

				// test endpoint combination
				for ( std::vector< AtomPoint >::const_iterator vb = viable_begin.begin(), vbe = viable_begin.end(); vb != vbe; ++vb ) {
					for ( std::vector< AtomPoint >::const_iterator ve = viable_end.begin(), vee = viable_end.end(); ve != vee; ++ve ) {

						ResidueRange secondary_match_range( vb->residue_id(), ve->residue_id() );

						// check to see if this is in allowed scaffold ranges
						if ( allowed_match_ranges.find( secondary_match_range ) == allowed_match_ranges.end() ) {
							continue; // not in allowed subranges
						}

						// skip if new graft site encompasses an existing graft (either primary or secondary)
						// or if two graft sites are too close
						bool reject = false;
						for ( Integer i = 1, ie = match_result.components.size(); i <= ie; ++i ) {
							ResidueRange const & existing_match_range = match_result.components[ i ].scaffold_gap_range;

							if ( secondary_match_range.contains( existing_match_range ) ) {
								reject = true; // encompasses
								break;
							}

							Integer width_between_gaps;
							if ( secondary_match_range.end() < existing_match_range.begin() ) {
								width_between_gaps = existing_match_range.begin() - secondary_match_range.end();
							} else {
								width_between_gaps = secondary_match_range.begin() - existing_match_range.end();
							}
							if ( width_between_gaps < MINIMUM_WIDTH_BETWEEN_GAPS ) {
								reject = true; // width between gaps too small
								break;
							}
						}
						if ( reject ) {
							continue;
						}

						// all filters have passed, prepare to clash check initial position
						utility::vector1< bool > keep( scaffold.total_residue(), true ); // keep the following residues but remove those in all match ranges
						for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
							for ( Integer residue = match_result.components[ c ].scaffold_gap_range.begin(), last_residue = match_result.components[ c ].scaffold_gap_range.end(); residue <= last_residue; ++residue ) {
								keep[ residue ] = false;
							}
						}
						// remember to also remove residues in the new secondary match range we're checking for
						for ( Integer residue = secondary_match_range.begin(), last_residue = secondary_match_range.end(); residue <= last_residue; ++residue ) {
							keep[ residue ] = false;
						}

						// get aligned loop
						Pose aligned_loop;
						extract_segment_from_Pose( aligned_epitope, loop_subrange, aligned_loop );
						FArray3D_float const & aligned_loop_full_coord = aligned_loop.full_coord(); // for performance

						// create secondary component so filters can be computed
						MatchComponent new_secondary_component( loop_id, loop_subrange, native_subrange, secondary_match_range );

						// compute filters (rms, closure angle, etc)
						compute_match_filters( scaffold_full_coord, new_secondary_component.scaffold_gap_range, aligned_loop_full_coord, ResidueRange( 1, aligned_loop.total_residue() ), new_secondary_component );

						// if rms filter not passed, reject this component
						if ( new_secondary_component.n_terminal_rms > MAX_CLOSURE_RMS || new_secondary_component.c_terminal_rms > MAX_CLOSURE_RMS ) {
							continue;
						}

						// gapped scaffold and octree for clash check
						Pose gapped_scaffold;
						fragment_Pose( scaffold, keep, gapped_scaffold );
						rootstock::Octree< AtomPoint > octree_gapped( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
						fill_octree( octree_gapped, gapped_scaffold );

						// intra clash check
						Real intra_clash = octree_inter_rep( octree_gapped, gapped_scaffold, aligned_loop, true ); // boolean: exclude terminal bb

						if ( intra_clash < options.max_intra_clash ) {

							Real inter_clash = 0.0;
							if ( native.Ab_exists() ) {
								// inter-clash check with re-aligned antibody
								Pose aligned_Ab;
								aligned_Ab = native.Ab();
								aligned_Ab.transform_GL( match_result.transformation_matrix );
								inter_clash = octree_inter_rep( octree_gapped, gapped_scaffold, aligned_Ab, false ); // boolean: exclude terminal bb
							}

							if ( inter_clash < options.max_inter_clash ) {

								// create new match result with additional secondary component
								MatchResult new_match_result = match_result;
								new_match_result.inter_clash = inter_clash;

								// store intra clash
								new_secondary_component.intra_clash = intra_clash;

								// add new secondary component
								new_match_result.components.push_back( new_secondary_component );

								// store in list of multiple matches
								multiple_matches.push_back( new_match_result );

								// use new_match_result as seed to build other matches
								stack.push( new_match_result );

							} // if inter_clash successful
						} // if intra_clash successful

					} // foreach viable_end
				} // foreach viable_begin

			} // foreach loop subrange
		} // foreach loop

	} // while stack not empty
}

} // match
} // epigraft
