// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   epigraft_match.cc
/// @brief  Searches for graft locations on a scaffold.
///
/// @note   Pose behavior in scarlet_match:
/// @note   fullatom = true
/// @note   ideal_pose = false // non-ideal backbone geometry
/// @note   read_all_chains = true
/// @note   check_missing = false
/// @note   coords_init = true //used in pose_from_misc
///
/// @author Bill Schief (schief@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/match/epigraft_match.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_constants.hh>
#include <epigraft/match/additional_filters.hh>
#include <epigraft/match/combi_match.hh>
#include <epigraft/match/find_singleton_matches.hh>
#include <epigraft/match/match_functions.hh>
#include <epigraft/match/match_io.hh>
#include <epigraft/match/refine_matches.hh>
#include <epigraft/match/rescore_matches.hh>
#include <epigraft/match/rough_match.hh>
#include <epigraft/match/try_fluid_ends.hh>
#include <epigraft/match/try_rb_move.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/match/align/C2N_C_Align.hh>
#include <epigraft/match/align/C2N_CA_Align.hh>
#include <epigraft/match/align/C2N_N_Align.hh>
#include <epigraft/match/align/E_Align.hh>
#include <epigraft/match/align/N2C_C_Align.hh>
#include <epigraft/match/align/N2C_CA_Align.hh>
#include <epigraft/match/align/N2C_N_Align.hh>
#include <epigraft/match/align/S_Align.hh>
#include <epigraft/match/align/SS_Align.hh>
#include <epigraft/conformation/FluidLanding.hh>
#include <epigraft/conformation/FluidTakeoff.hh>
#include <epigraft/conformation/FluidTakeoffResult.hh>
#include <epigraft/conformation/TransformGenerator.hh>
#include <epigraft/design/design_functions.hh>
#include <epigraft/design/design_io.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/epigraft_io.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/Checkpoint.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/LoopInfo.hh>

// Rosetta headers
#include <aaproperties_pack.h>
#include <after_opts.h>
#include <files_paths.h>
#include <param_aa.h>
#include <pose.h>
#include <pose_io.h>
#include <refold.h> // GL transformation routines
#include <rms.h>
#include <runlevel.h>
#include <score.h>

// rootstock headers
#include <rootstock/Octree.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// numeric headers
#include <numeric/xyzVector.hh>

// utility headers
#include <utility/file/file_sys_util.hh>
#include <utility/file/FileName.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/pointer/access_ptr.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/vector1.hh>

// C++ headers
#include <cstdio>
#include <iostream>
#include <set>
#include <stack>
#include <string>
#include <vector>


namespace epigraft {
namespace match {


/// @brief main for epigraft match
// TODO: cleanup slightly intertwined table and pdb output
void
epigraft_match(
	GraftOptions const & options
)
{
	using namespace rootstock;
	using namespace epigraft::match::align;
	using epigraft::design::parse_keep_natro_file;

	// override any rosetta global options
	if ( options.override_rosetta_pdb_output_path ) {
		files_paths::pdb_out_path = options.pdb_output_path;
	}

	// issue checks
	if ( options.moveable_closure_residues >= options.termini_residue_skip ) {
		utility::exit( __FILE__, __LINE__, "please make sure that # moveable_closure_residues is strictly less than # termini_residue_skip!" );
	}

	// load native complex from file
	AntibodyComplex native( options.native_complex_filename, options.nres_Ab, options.Ab_first, true ); // booleans: Ab_first, refold_sidechains_from_chi
	AntibodyComplex original_native; // used to keep a copy
	original_native = native;

	// load keep natro file if requested
	std::set < Integer > keep_natro_residues;
	if ( options.use_keep_natro ) {
		keep_natro_residues = parse_keep_natro_file( options.keep_natro_filename );
	}

	// convert native antigen to gly
	if ( options.use_epitope_sidechain_during_clash_check ) {

		// currently the keep natro routines don't handle insertion codes, so all
		// residue numbering is just integer
		utility::vector1< bool > mut_res( native.antigen().total_residue(), true );
		for ( Size i = 1, ie = native.antigen().total_residue(); i <= ie; ++i ) {
			mut_res[ i ] = ( keep_natro_residues.find( native.antigen().pdb_info().pdb_res_num( i ) ) == keep_natro_residues.end() );
		}
		epigraft::design::mutate_range( native.antigen(), mut_res, options.closure_residue_type, true ); // boolean: keep_gly

	} else {
		epigraft::design::mutate_range( native.antigen(), ResidueRange( 1, native.antigen().total_residue() ), options.closure_residue_type, true ); // boolean: keep_gly
	}

	// convert native antibody to hybrid ala-gly unless specified otherwise
	if ( !options.use_full_sidechain_inter_clash && native.Ab_exists() ) {
		epigraft::design::mutate_range( native.Ab(), ResidueRange( 1, native.Ab().total_residue() ), param_aa::aa_ala, true ); // boolean: keep_gly
	}

	// refresh antibody complex internals after changes
	native.refresh_complex();

	// load spatial filter structure if requested
	Pose spatial_filter_structure;
	if ( options.use_spatial_filter ) {
		// load spatial filter pdb
		pose_from_pdb( spatial_filter_structure, options.pdb_for_spatial_filter, true, false, true ); // for now we force the booleans: fullatom, ideal_pose, read_all_chains
	}

	// read list of epitope (loop) ranges
	// TODO: optional -- error checking to see if any of the subranges lie out of bounds relative to various criteria
	utility::vector1< LoopInfo > loops_to_scan;
	std::set< Integer > user_specified_superposition_residues;
	load_epitope_subranges( options.loop_ranges_filename, loops_to_scan, user_specified_superposition_residues, options.use_SS_align );

	// sanity check for epitope (loop) ranges and epitope input structure
	if ( !epitope_and_epitope_subranges_one_to_one( loops_to_scan, native.antigen() ) ) {
		utility::exit( __FILE__, __LINE__, "ERROR: please make sure the epitope in your antibody complex structure includes exactly the residues specified in the 'full_range'(s) of the loop ranges file" );
	}

	// if input is from a prior match run, load the match results
	std::map< std::string, utility::vector1< MatchResult > > pdb_to_existing_match_results;
	if ( options.use_input_from_match ) {
		match_results_from_file( options.input_filename, loops_to_scan, pdb_to_existing_match_results );
	}

	// load list of scaffold pdb filenames
	utility::vector1< std::string > scaffold_filenames;
	if ( !options.use_input_from_match ) { // default
			load_scaffold_filenames( options.input_filename, scaffold_filenames );
	} else { // input is from match, so get filenames from map

		for ( std::map< std::string, utility::vector1< MatchResult > >::const_iterator entry = pdb_to_existing_match_results.begin(), entry_e = pdb_to_existing_match_results.end(); entry != entry_e; ++entry ) {
			scaffold_filenames.push_back( entry->first ); // scaffold filename
		}

	}

	// create all alignment systems
	utility::vector1< AlignmentSystemOP > alignment_systems;

	// compute proper max closure rms: if attempting fluid ends or rb-move, then this number needs to
	// be modified as below
	Real MAX_CLOSURE_RMS = options.max_closure_rms;
	if ( options.fluidize_landing || options.fluidize_takeoff || options.rb_move ) {
		MAX_CLOSURE_RMS += options.recovery_rms_epsilon;
	}

	// "E" system
	if ( options.use_E_align ) {
		alignment_systems.push_back( AlignmentSystemOP( new E_Align( MAX_CLOSURE_RMS ) ) );
	}

	// "S" system
	if ( options.use_S_align ) {
		if ( options.max_rms_over_length < 0.0 ) {
			alignment_systems.push_back( AlignmentSystemOP( new S_Align( AlignmentSystem::RMS, MAX_CLOSURE_RMS ) ) );
		} else {
			alignment_systems.push_back( AlignmentSystemOP( new S_Align( AlignmentSystem::RMS_OVER_LENGTH, options.max_rms_over_length ) ) );
		}
	}

	// "SS" system
	if ( options.use_SS_align ) {
		if ( options.max_rms_over_length < 0.0 ) {
			alignment_systems.push_back( AlignmentSystemOP( new SS_Align( AlignmentSystem::RMS, MAX_CLOSURE_RMS, user_specified_superposition_residues ) ) );
		} else {
			alignment_systems.push_back( AlignmentSystemOP( new SS_Align( AlignmentSystem::RMS_OVER_LENGTH, options.max_rms_over_length, user_specified_superposition_residues ) ) );
		}
	}

	if ( !options.skip_N2C_align ) {
		// "N2C" CA-centered system
		alignment_systems.push_back( AlignmentSystemOP( new N2C_CA_Align( MAX_CLOSURE_RMS ) ) );

		// "N2C" N-centered system
		alignment_systems.push_back( AlignmentSystemOP( new N2C_N_Align( MAX_CLOSURE_RMS ) ) );

		// "N2C" C-centered system
		alignment_systems.push_back( AlignmentSystemOP( new N2C_C_Align( MAX_CLOSURE_RMS ) ) );
	}

	if ( !options.skip_C2N_align ) {
		// "C2N" CA-centered system
		alignment_systems.push_back( AlignmentSystemOP( new C2N_CA_Align( MAX_CLOSURE_RMS ) ) );

		// "C2N" N-centered system
		alignment_systems.push_back( AlignmentSystemOP( new C2N_N_Align( MAX_CLOSURE_RMS ) ) );

		// "C2N" C-centered system
		alignment_systems.push_back( AlignmentSystemOP( new C2N_C_Align( MAX_CLOSURE_RMS ) ) );
	}

	// create iterator index outside of loop due to potential checkpointing
	Integer scaffold_id = 1;

	// for checkpointing
	Checkpoint checkpoint( files_paths::score_path + options.checkpoint_filename );
	if ( options.use_checkpoint ) {
		checkpoint.recover_last_id_processed();

		if ( checkpoint.current_id() > scaffold_filenames.size() ) {
			// over count, so we're done
			utility::exit( __FILE__, __LINE__, "checkpoint indicates all files processed" );
		}

		scaffold_id = checkpoint.current_id() > 0 ? checkpoint.current_id() : 1;

		// start checkpointing
		checkpoint.open();
	}

	// NOTE: the order that these tags are added matters!  They must be in the same
	// order that the functions are called in, otherwise the output results will not
	// be consistent with the header.
	// if doing spatial C-beta check, add one additional output tag
	if ( options.use_spatial_filter ) {
		MatchResult::add_output_tag( "spatial_C_beta" );
	}

	// if doing various screen or refine, add additional output tags
	if ( options.screen_with_repack ) {
		MatchResult::add_output_tag( "inter_clash_before_repack" );
		MatchResult::add_output_tag( "inter_clash_after_repack" );
	}
	if ( options.superposition_minrepack_refine ) {
		// add ddG columns to MatchResult output tags
		MatchResult::add_output_tag( "ddG_before_minrepack" );
		MatchResult::add_output_tag( "ddG_after_minrepack" );
	}

	// open output file
	std::string output_filename = files_paths::score_path + options.output_filename;
	utility::io::ozstream outfile;
	utility::io::izstream testfile( output_filename );
	if ( !testfile.fail() ) {

		testfile.close();
		outfile.open_append( output_filename );

		// write new header since two additional output tags will be present
		if ( options.use_spatial_filter || options.superposition_minrepack_refine ) {
			outfile << MatchResult::header_string();
		}

	} else {
		testfile.close();
		outfile.open( output_filename );

		// new file, output header
		outfile << MatchResult::header_string();
	}

	// run over each scaffold
	for ( Integer last_scaffold_id = scaffold_filenames.size(); scaffold_id <= last_scaffold_id; ++scaffold_id ) {

		// checkpointing
		if ( options.use_checkpoint ) {
			checkpoint.set_current_id( scaffold_id );
			checkpoint.update();
		}

		std::string const & scaffold_filename = scaffold_filenames[ scaffold_id ];

		Pose scaffold;
		if ( !options.input_pdb_has_Ab ) {
			pose_from_pdb( scaffold, files_paths::start_path + scaffold_filename, true, false, true ); // boolean: fullatom, ideal_pose, read_all_chains
		} else {
			AntibodyComplex ac( files_paths::start_path + scaffold_filename, options.nres_Ab, options.Ab_first, false );
			scaffold = ac.antigen();
		}

		// convert scaffold to gly or hybrid ala-gly (governed by options.closure_residue_type)
		Pose gly_scaffold;
		gly_scaffold = scaffold;
		epigraft::design::mutate_range( gly_scaffold, ResidueRange( 1, gly_scaffold.total_residue() ), options.closure_residue_type, true ); // boolean: keep_gly

		// need all ala scaffold if using cbeta spatial filter
		Pose all_ala_scaffold;
		if ( options.use_spatial_filter || options.compute_cbeta_neighbors ) {
			// create all ala antigen, used during spatial filter to check against C-beta
			all_ala_scaffold = scaffold;
			epigraft::design::mutate_range( all_ala_scaffold, ResidueRange( 1, all_ala_scaffold.total_residue() ), param_aa::aa_ala );
		}

		std::cout << "Searching " << scaffold_filename << std::endl;

		// find singleton matches or grab existing matches
		utility::vector1< MatchResult > singleton_matches;
		if ( !options.use_input_from_match ) {

			find_singleton_matches( options, native, loops_to_scan, gly_scaffold, alignment_systems, singleton_matches );
			std::cout << "  found " << singleton_matches.size() << " singleton matches" << std::endl;

		} else {

			singleton_matches = pdb_to_existing_match_results[ scaffold_filename ]; // TODO: singleton_matches now becomes a misnomer because input can be a multi-match, correct this?
			rescore_matches( options, native, gly_scaffold, singleton_matches, scaffold_filename, user_specified_superposition_residues );
			std::cout << "  using " << singleton_matches.size() << " existing matches from input" << std::endl;

		}

		// apply spatial filter here so matches that don't satisfy the distances don't propogate to subsequent steps
		if ( options.use_spatial_filter ) {
			utility::vector1< MatchResult > spatially_filtered_matches;
			cbeta_spatial_filter( options, all_ala_scaffold, singleton_matches, spatial_filter_structure, spatially_filtered_matches );
			singleton_matches = spatially_filtered_matches;
			std::cout << "    " << singleton_matches.size() << " matches remain after spatial filtering" << std::endl;
		}

		// diversify by rb-move from initial set of singleton matches
		// do this _before_ takeoff fluidization, it doesn't makes sense to rb-move fluidized stuff
		// note that this alters the transformation matrix
		utility::vector1< MatchResult > all_singleton_matches; // this will eventually include singleton matches too
		if ( options.rb_move ) {
			std::cout << "    rb-move:";
			try_rb_move( options, native, gly_scaffold, singleton_matches, all_singleton_matches );
			std::cout << " expansion yields " << all_singleton_matches.size() << " matches" << std::endl;
		}


		// if specified, attempt recovery/optimization by fluidizing closed endpoint (single break only)
		// note that this alters the transformation matrix
		if ( options.fluidize_takeoff ) {
			std::cout << "    fluidizing takeoff...";
			try_fluidizing_takeoff( options, native, gly_scaffold, singleton_matches );
			std::cout << " done" << std::endl;
		}


		// Now append rb-move and original singleton matches, this is done after fluidizing takeoff so we
		// don't waste time in fluidizing non-existent takeoffs of the 'E' rb-move matches.
		// 'singleton_matches' is typically the smaller set, so we append it to the longer list.
		all_singleton_matches.insert( all_singleton_matches.end(), singleton_matches.begin(), singleton_matches.end() );
		singleton_matches.clear(); // safety


		// if specified, attempt recovery/optimization by fluid endpoints
		if ( options.fluidize_landing ) {
			std::cout << "    fluidizing landing...";
			try_fluidizing_landing( options, native, gly_scaffold, all_singleton_matches );
			std::cout << " done" << std::endl;
		}

		// filter singleton matches for those that satisfy the strict cutoff if any
		// recovery/optimization has been attempted
		utility::vector1< MatchResult > filtered_singleton_matches;
		if ( options.fluidize_landing || options.fluidize_takeoff || options.rb_move ) {

			strict_rms_filter( options.max_closure_rms, all_singleton_matches, filtered_singleton_matches );
			std::cout << "    " << filtered_singleton_matches.size() << " total singleton matches remain after strict rms filtering" << std::endl;

		} else {
			filtered_singleton_matches = all_singleton_matches;
		}
		all_singleton_matches.clear(); // safety

		// compute cbeta neighbors statistics, if requested
		if ( options.compute_cbeta_neighbors ) {
			cbeta_neighbors_per_component( options, native.antigen(), all_ala_scaffold, filtered_singleton_matches );
		}

		// post-match refinement/screen via repack
		if ( options.use_input_from_match && options.screen_with_repack ) {
			std::cout << "  doing screen with repack" << std::endl;
			screen_matches_with_repack( options, original_native, scaffold, filtered_singleton_matches, keep_natro_residues );
		}

		// only for superposition protocol designs, post-match refinement
		if ( options.use_input_from_match && options.superposition_minrepack_refine ) {
			std::cout << "  doing minrepack refine for superposition protocol matches" << std::endl;
			minrepack_superposition_matches( options, original_native, scaffold, filtered_singleton_matches, keep_natro_residues, false ); // boolean: output pdb
		}

		// output singleton matches into table
		if ( options.output_single_matches && !options.full_matches_only) {
			for ( Integer m = 1, me = filtered_singleton_matches.size(); m <= me; ++m ) {
				MatchResult const & match_result = filtered_singleton_matches[ m ];

				if ( match_result.inter_clash < options.max_inter_clash ) { // check here because rough match accepts all inter_clash
					std::ostringstream ss( "" );
					if ( options.output_aligned_loops || options.output_predesign_structure || options.output_predesign_structure_with_Ab ) {
						ss << base_pdb_filename( scaffold_filenames[ scaffold_id ] )
						   << "_single_"
						   << m;

						if ( options.output_aligned_loops ) {
							output_aligned_loops_pdb( original_native.antigen(), match_result, files_paths::pdb_out_path + ss.str() + ".aligned_loops.pdb.gz" );
						}

						if ( options.output_predesign_structure ) {
							output_predesign_structure_pdb( scaffold, original_native.antigen(), match_result, keep_natro_residues, files_paths::pdb_out_path + ss.str() + ".predesign.pdb.gz" );
						}

						if ( options.output_predesign_structure_with_Ab && native.Ab_exists() ) {
							output_predesign_structure_pdb_with_Ab( scaffold, original_native.antigen(), original_native.Ab(), match_result, keep_natro_residues, files_paths::pdb_out_path + ss.str() + ".predesign_with_Ab.pdb.gz" );
						}
					} else {
						ss << "-";
					}

					outfile << match_result.to_string( scaffold_filenames[ scaffold_id ], ss.str(), "", scaffold.total_residue() );
				}
			}

				outfile << std::flush;
		}


		// check multi-site by rough match or combi match (nearest neighbor criteria)
		if ( options.rough_match || options.combi_match ) {

			utility::vector1< MatchResult > multiple_matches;
			if ( options.rough_match ) {
				rough_match( options, native, loops_to_scan, gly_scaffold, filtered_singleton_matches, multiple_matches );
				std::cout << "  found " << multiple_matches.size() << " multiple matches through rough match" << std::endl;
			} else if (options.combi_match ) {
				combi_match( options, native, loops_to_scan, gly_scaffold, filtered_singleton_matches, multiple_matches );
				std::cout << "  found " << multiple_matches.size() << " multiple matches through combi match" << std::endl;
			}

			if ( options.fluidize_landing ) {
				std::cout << "    fluidizing landing...";
				try_fluidizing_landing( options, native, gly_scaffold, multiple_matches, 1 ); // skip fluidizing of primary component, this is already done above
				std::cout << " done" << std::endl;
			}

			// filter multiple matches for those that satisfy the strict cutoff if any
			// recovery/optimization has been attempted
			utility::vector1< MatchResult > filtered_multiple_matches;
			if ( options.fluidize_landing ) {

				// TODO: change to using a universal 'options.multi_match_closure_rms' to reduce number of
				//       rms floating around, e.g. below only rough_match_closure_rms is used, so if there are
				//       any additional multi-match routines implemented...?
				strict_rms_filter( options.rough_match_closure_rms, multiple_matches, filtered_multiple_matches );
				std::cout << "    " << filtered_multiple_matches.size() << " total multiple matches remain after strict rms filtering" << std::endl;

			} else {
				filtered_multiple_matches = multiple_matches;
			}
			multiple_matches.clear(); // safety

			// compute cbeta neighbors statistics, if requested
			if ( options.compute_cbeta_neighbors ) {
				cbeta_neighbors_per_component( options, native.antigen(), all_ala_scaffold, filtered_multiple_matches );
			}
			//vds control output of multi-matches based on how many loops got matches
			if ( options.full_matches_only ) {
				utility::vector1< MatchResult > full_matches;
				filter_out_incomplete_matches( loops_to_scan.size(), filtered_multiple_matches, full_matches );
				filtered_multiple_matches = full_matches;
				full_matches.clear();

				std::cout << "    " << filtered_multiple_matches.size() << " total multiple matches remain that match all " << loops_to_scan.size() << " desired loops" << std::endl;
			}

			// output multiple matches into table
			for ( Integer m = 1, me = filtered_multiple_matches.size(); m <= me; ++m ) {
				MatchResult const & match_result = filtered_multiple_matches[ m ];

				std::ostringstream ss( "" );
				if ( options.output_aligned_loops || options.output_predesign_structure || options.output_predesign_structure_with_Ab ) {
					ss << base_pdb_filename( scaffold_filenames[ scaffold_id ] )
					   << "_multi_"
					   << m;

					if ( options.output_aligned_loops ) {
						output_aligned_loops_pdb( original_native.antigen(), match_result, files_paths::pdb_out_path + ss.str() + ".aligned_loops.pdb.gz" );
					}

					if ( options.output_predesign_structure ) {
						output_predesign_structure_pdb( scaffold, original_native.antigen(), match_result, keep_natro_residues, files_paths::pdb_out_path + ss.str() + ".predesign.pdb.gz" );
					}

					if ( options.output_predesign_structure_with_Ab && native.Ab_exists() ) {
						output_predesign_structure_pdb_with_Ab( scaffold, original_native.antigen(), original_native.Ab(), match_result, keep_natro_residues, files_paths::pdb_out_path + ss.str() + ".predesign_with_Ab.pdb.gz" );
					}
				} else {
					ss << "-";
				}

				outfile << match_result.to_string( scaffold_filenames[ scaffold_id ], ss.str(), "", scaffold.total_residue() );
			}

				outfile << std::flush;
		}

	}

	// close output file
	outfile.close();

	// final update and close checkpoint
	if ( options.use_checkpoint) {
		checkpoint.set_current_id( scaffold_filenames.size() + 1 );
		checkpoint.update();
		checkpoint.close();
	}
}


/// @brief test functionality
void
test(
	GraftOptions const & options
)
{
//	N2C_N_Align align_sys( N2C_N_Align( options.max_closure_rms ) );
//
//	AntibodyComplex native( options.native_complex_filename, options.nres_Ab, true, true ); // booleans: Ab_first, refold_sidechains_from_chi
//
//	epigraft::convert_to_gly( native.antigen() );

//	typedef numeric::xyzVector< float > Vector;
//	using epigraft::conformation::TransformGenerator;
//
//	Pose scaffold;
//	pose_from_pdb( scaffold, "test.pdb.gz", true, false, true ); // boolean: fullatom, ideal_pose, read_all_chains
//
//	FArray3D_float const & fc = scaffold.full_coord();
//	TransformGenerator< Real > gen( Vector( fc[ 0 ], fc[ 1 ], fc[ 2 ] ), Vector( 1, 0, 0 ), Vector( 0, 1, 0 ), Vector( 0, 0, 1 ) );
//
//	FArray2D_float m( 4, 4, 0.0 );
//	for ( Integer i = 1; i < 3; ++i ) {
//		gen.rotate_x_degrees( 30.0 * i, m );
//		transform_Pose_via_GL( m, scaffold );
//
//		std::stringstream ss;
//		ss << "rX_" << ( 30 * i ) << ".pdb";
//		scaffold.dump_pdb( ss.str() );
//	}

	utility::vector1< LoopInfo > loops_to_scan;
	std::set< Integer > user_specified_superposition_residues;
	load_epitope_subranges( options.loop_ranges_filename, loops_to_scan, user_specified_superposition_residues, options.use_SS_align );

	std::map< std::string, utility::vector1< MatchResult > > test;
	match_results_from_file( "graft_results.out", loops_to_scan, test );

	for ( std::map< std::string, utility::vector1< MatchResult > >::iterator f = test.begin(), fe = test.end(); f != fe; ++f ) {
		for ( Integer m = 1, me = f->second.size(); m <= me; ++m ) {
			std::cout << f->second[ m ].to_string( f->first, "" );
			std::cout << "** " << f->second[ m ].components[ 1 ].loop_subrange.to_string() << std::endl;
		}
	}

//
//	epigraft::convert_to_gly( scaffold );
//
//	ResidueRange loop_subrange( 1, 12 ); // 669 - 680
//	ResidueRange match_range( 25, 74 );
//
//	Pose loop;
//	native.trimmed_antigen( loop, loop_subrange );
//
//	MatchResult match_result;
//	align_sys.check_alignment( scaffold, match_range, loop, loop_subrange, match_result );
//
//	Pose aligned_loop;
//	transform_Pose_via_GL( match_result.transformation_matrix, loop, aligned_loop ); // RT( loop ) -> aligned_loop
//
//
//	Pose loop_context;
//
//	using namespace epigraft::conformation;
//
//	create_loop_context( scaffold, match_range, aligned_loop, loop_context );
//	contextual_fold_tree( loop_context );
//
//	loop_context.dump_pdb( "loop_context.pdb" );
//
////	fluidize_endpoints_N2C( loop_context, 10, 5 );
//	fluidize_N2C( loop_context, 10, 5 );
}


} // namespace match
} // namespace epigraft
