// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   find_singleton_matches.cc
/// @brief  Find all singleton matches on a scaffold.
/// @author Bill Schief (schief@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/match/find_singleton_matches.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_constants.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/ResidueRange.hh>

// Rosetta headers
#include <pose.h>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// utility headers
#include <utility/pointer/owning_ptr.hh>
#include <utility/vector1.hh>


namespace epigraft {
namespace match {


/// @brief find all singleton matches
// TODO: reorganize filling of match result for clarity
void
find_singleton_matches(
	GraftOptions const & options,
	AntibodyComplex & native,
	utility::vector1< LoopInfo > const & loops_to_scan,
	Pose & scaffold,
	utility::vector1< AlignmentSystemOP > & alignment_systems,
	utility::vector1< MatchResult > & singleton_matches
)
{
	using epigraft::match::align::AlignmentSystem;

	// references to options here for clarity (smaller names), otherwise the 'for' definition below is a bit lengthy
	Integer const & termini_residue_skip = options.termini_residue_skip;
	Integer const & min_match_width = options.min_match_width;
	Integer const & max_match_width_delta = options.max_match_width_delta;

	// temporaries, for speed
	FArray3D_float const & scaffold_full_coord = scaffold.full_coord(); // TODO: consider calling full_coord_simple_return_packer() to prevent refold!
	FArray3D_float const & epitope_full_coord = native.antigen().full_coord(); // TODO: consider calling full_coord_simple_return_packer() to prevent refold!

	// compute bounding box here for constructing octree around gapped_scaffold later
	rootstock::BoundingBox bb = bounding_box( scaffold );

	for ( Integer loop_id = 1, last_loop_id = loops_to_scan.size(); loop_id <= last_loop_id; ++loop_id ) {
		// TODO: if scaffold already contains this loop, skip it

		// compute numbering offset in native (peptide) structure for this loop (epitope)
		Integer offset = compute_loop_offset( loops_to_scan, loop_id);

		LoopInfo const & current_loop = loops_to_scan[ loop_id ];
		//vds if current loop has been disallowed as primary in input file, skip it
		if ( current_loop.is_primary() ) {

			// run through each possible loop (epitope) sub-range
			for ( Integer subrange_id = 1, last_subrange_id = current_loop.native_subranges().size(); subrange_id <= last_subrange_id; ++subrange_id ) {
				ResidueRange const & native_subrange = current_loop.native_subranges()[ subrange_id ];
				ResidueRange const & internal_subrange = current_loop.internal_subranges()[ subrange_id ];
				ResidueRange loop_subrange( internal_subrange + offset );

				// loop sub-range CA distance for distance exclusion
				Real match_distance = distance( numeric::xyzVector< Real >( &( epitope_full_coord( 1, 2, loop_subrange.begin() ) ) ),
																				numeric::xyzVector< Real >( &( epitope_full_coord( 1, 2, loop_subrange.end() ) ) ) );

				// Setup minimum and maximum match distances.
				// This allows pruning of ranges before they are fed to the alignment systems.
				// Let 'l' be the C-alpha -> C-alpha distance of the endpoints of an epitope
				// loop subrange and 'g' the C-alpha -> C-alpha distance of the endpoints of
				// a scaffold gap.  Then a scaffold gap range is only tested if the following
				// holds:  l - e_min < g < l + e_max    where e_min, e_max are some epsilons
				// We set the epsilons to the largest values used based upon the requested
				// alignment systems, so if a lot of alignment systems are requested, then
				// for certain alignment systems the ranges will be over extended.  This may make
				// things less efficient for those alignment systems, but we don't have to do
				// any more in-depth modification to the code.
				Real min_match_distance_epsilon = 0.0;
				Real max_match_distance_epsilon = 0.0;
				for ( utility::vector1< AlignmentSystemOP >::iterator as = alignment_systems.begin(), ase = alignment_systems.end(); as != ase; ++as ) {
					AlignmentSystem & alignment_system = **as;

					switch ( alignment_system.type() ) {
						case AlignmentSystem::N2C: // dictated by triangle inequality
							min_match_distance_epsilon = std::max( match_distance, min_match_distance_epsilon );
							max_match_distance_epsilon = std::max( options.max_closure_rms, max_match_distance_epsilon );
							break;
						case AlignmentSystem::C2N: // dictated by triangle inequality
							min_match_distance_epsilon = std::max( match_distance, min_match_distance_epsilon );
							max_match_distance_epsilon = std::max( options.max_closure_rms, max_match_distance_epsilon );
							break;
						case AlignmentSystem::SUPERPOSITION: // want the loop to occupy approx the same amount of space
							min_match_distance_epsilon = std::max( static_cast< Real >( 3.0 ), min_match_distance_epsilon );
							max_match_distance_epsilon = std::max( static_cast< Real >( 3.0 ), max_match_distance_epsilon );
							break;
						case AlignmentSystem::ENDPOINT: // dictated by interval comparison
							min_match_distance_epsilon = std::max( static_cast< Real >( 2.0 * options.max_closure_rms ), min_match_distance_epsilon );
							max_match_distance_epsilon = std::max( static_cast< Real >( 2.0 * options.max_closure_rms ), max_match_distance_epsilon );
							break;
						case AlignmentSystem::SPECIFIC_SUPERPOSITION: // want the loop to occupy approx the same amount of space
							min_match_distance_epsilon = std::max( static_cast< Real >( 3.0 ), min_match_distance_epsilon );
							max_match_distance_epsilon = std::max( static_cast< Real >( 3.0 ), max_match_distance_epsilon );
							break;
						default: // do nothing on default
							break;
					}
				}

				// test for no epsilon setting and set a hardcoded 3.0, just in case
				if ( min_match_distance_epsilon == 0.0 && max_match_distance_epsilon == 0.0 ) {
					min_match_distance_epsilon = 3.0;
					max_match_distance_epsilon = 3.0;
				}

				Real min_match_distance_sq = match_distance - min_match_distance_epsilon; // epsilon contraction
				if ( min_match_distance_sq < 0 ) { // catch any sign issues
					min_match_distance_sq = 0.0;
				} else {
					min_match_distance_sq *= min_match_distance_sq; // re-create squared distance
				}
				Real max_match_distance_sq = match_distance + max_match_distance_epsilon; // epsilon expansion
				max_match_distance_sq *= max_match_distance_sq; // re-create squared distance

				// run through all pairs of residues, take into account minimum and maximum match widths
				Integer const max_right_residue = scaffold.total_residue() - termini_residue_skip;
				Integer const max_left_residue = max_right_residue - min_match_width + 1;

				for ( Integer left_residue = termini_residue_skip + 1; left_residue <= max_left_residue; ++left_residue ) {
					for ( Integer right_residue = min( left_residue + min_match_width - 1, max_left_residue ), right_prime = min( left_residue + loop_subrange.length() + max_match_width_delta - 1, max_right_residue ); right_residue <= right_prime; ++right_residue ) {

						ResidueRange match_range = ResidueRange( left_residue, right_residue );

						// gap CA distance check to exclude pairs
						// TODO: speed distance check by Octree...?  unclear in this case if there's a speed benefit
						// TODO: possible to speed distance check by pre-computing distance matrix and do table lookup
						Real pair_distance_sq = distance_squared( numeric::xyzVector< Real >( &( scaffold_full_coord( 1, 2, match_range.begin() ) ) ),
																											numeric::xyzVector< Real >( &( scaffold_full_coord( 1, 2, match_range.end() ) ) ) );

						if ( ( pair_distance_sq < min_match_distance_sq ) || ( pair_distance_sq > max_match_distance_sq ) ) { // gap much larger than loop sub-range distance
							continue;
						}

						// create trimmed loop (epitope)
						Pose loop;
						native.trimmed_antigen( loop, loop_subrange );

						for ( utility::vector1< AlignmentSystemOP >::iterator as = alignment_systems.begin(), ase = alignment_systems.end(); as != ase; ++as ) {
							epigraft::match::align::AlignmentSystem & alignment_system = **as;

							MatchResult match_result( alignment_system.type(), alignment_system.alignment_center(), alignment_system.name() );
							MatchComponent match_component( loop_id, loop_subrange, native_subrange, match_range );

							if ( !alignment_system.check_alignment( scaffold, loop, match_component, match_result.transformation_matrix ) ) {
								// alignment failed filters
								continue;
							}

							// perform full alignment of loop onto scaffold
							Pose aligned_loop;
							aligned_loop = loop;
							aligned_loop.transform_GL( match_result.transformation_matrix );

							// create gapped scaffold, remember that this removes _all_ residues in the given range
							Pose gapped_scaffold;
							remove_segment_from_Pose( scaffold, match_range, gapped_scaffold );

							// fast clash check through octree
							// TODO: potentially modify fill_octree to exclude a range, in which case we most likely won't
							//       have to create the gapped scaffold
							rootstock::Octree< AtomPoint > octree_gs( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
							fill_octree( octree_gs, gapped_scaffold );
							// intra_clash = inter-repulsion between gapped_scaffold versus loop (trimmed_epitope)
							match_component.intra_clash = octree_inter_rep( octree_gs, gapped_scaffold, aligned_loop, true ); // boolean: exclude terminal bb

							if ( match_component.intra_clash < options.max_intra_clash ) {

								match_result.inter_clash = 0.0;
								if ( native.Ab_exists() ) {
									// measure inter-clash
									Pose aligned_Ab;
									aligned_Ab = native.Ab();
									aligned_Ab.transform_GL( match_result.transformation_matrix );

									// fast clash check through octree
									match_result.inter_clash = octree_inter_rep( octree_gs, gapped_scaffold, aligned_Ab, false ); // boolean: exclude terminal bb
								}

								// rough match may resolve certain inter-clashes with scaffold (further slicing of
								// the scaffold), so we make sure and accept everything below if rough matching
								if ( options.rough_match || match_result.inter_clash < options.max_inter_clash ) {

									// everything satisfied, add the match component to the match result
									match_result.components.push_back( match_component );
									// add the match result to list of results
									singleton_matches.push_back( match_result );

								} // inter-clash satisfied
							} // intra-clash satisfied

						} // foreach alignment system

					} // foreach right residue
				} // foreach left residue

			} // foreach loop subrange

		} // if allowed primary
	} // foreach loop

}


} // namespace match
} // namespace epigraft
