// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   match_functions.hh
/// @brief  Various functions for match.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/match/align/C2N_C_Align.hh>
#include <epigraft/match/align/C2N_CA_Align.hh>
#include <epigraft/match/align/C2N_N_Align.hh>
#include <epigraft/match/align/N2C_C_Align.hh>
#include <epigraft/match/align/N2C_CA_Align.hh>
#include <epigraft/match/align/N2C_N_Align.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/conformation/DihedralInfo.hh>

// rosetta headers
#include <fold_tree.h>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <set>


namespace epigraft {
namespace match {


/// @brief create epitope-scaffold Pose (including proper fold-tree) for closure and design
/// @note  See similar procedure in EpitopeScaffold for more detailed control and setup for
/// @note  grafting.  This particular function is mainly for producing a predesign e-s for
/// @note  visual inspection after matching
/// @warning  Since the created Pose is just for visualizing the exact coordinates that
/// @warning  come out of the matching, it is not suitable for immediate plug-in to Rosetta
/// @warning  protocols.  Due to non-ideal bonds switched on, the bonds at the cutpoints are
/// @warning  completely incorrect.  You *MUST* call:
/// @warning  epitope_scaffold.insert_ideal_bonds_at_fold_tree_cutpoints() after this routine
/// @warning  if you are planning to use the epitope_scaffold Pose for anything.
void
make_predesign_epitope_scaffold(
	Pose const & original_scaffold,
	Pose const & epitope,
	MatchResult const & match_result,
	Pose & epitope_scaffold,
	Integer const & moveable_closure_residues
)
{
	using pose_ns::Fold_tree;
	using epigraft::conformation::DihedralInfo;
	using namespace epigraft::match::align;

	typedef std::set< MatchComponent > SortedMatchComponents;
	
	Integer const ALLOW_MOVE = moveable_closure_residues; // residues allowed to move prior to break
	
	// first construct aligned epitope
	Pose aligned_epitope;
	aligned_epitope = epitope;
	aligned_epitope.transform_GL( match_result.transformation_matrix );
	
	MatchComponent const & primary_component = match_result.components[ 1 ];

	// next build artifical scaffold with any takeoff optimized dihedral changes;
	// here we only want to change the takeoff optimization angles but keep the rest of the scaffold fixed
	Pose scaffold;
	scaffold = original_scaffold;
	if ( primary_component.dihedrals.size() > 0 ) {
		Fold_tree aft; // artificial fold tree that jumps from residue prior/after the takeoff change to just after/prior to the gap residue
		
		Integer first_optimization_residue = -1;
		switch ( match_result.system_type ) {
			case AlignmentSystem::N2C:
				first_optimization_residue = ( primary_component.dihedrals.begin() )->residue();
				if ( first_optimization_residue < primary_component.scaffold_gap_range.begin() ) {
					aft.add_edge( 1, first_optimization_residue - 1, Fold_tree::PEPTIDE );
					aft.add_edge( first_optimization_residue - 1, primary_component.scaffold_gap_range.begin(), Fold_tree::PEPTIDE );
					aft.add_edge( first_optimization_residue - 1, primary_component.scaffold_gap_range.begin() + 1, 1 ); // jump
					aft.add_edge( primary_component.scaffold_gap_range.begin() + 1, scaffold.total_residue(), Fold_tree::PEPTIDE );
					aft.reorder( 1 ); // fold forwards N->C
					scaffold.set_fold_tree( aft );
					
					// now set angles
					std::set< DihedralInfo >::const_iterator d = primary_component.dihedrals.begin(), de = primary_component.dihedrals.end();
					while ( d != de ) {
						if ( d->residue() < primary_component.scaffold_gap_range.begin() ) {
							d->apply( scaffold );
							++d;
						}
					}
				}
				break;
			case AlignmentSystem::C2N:
				first_optimization_residue = ( primary_component.dihedrals.rbegin() )->residue();
				if ( first_optimization_residue > primary_component.scaffold_gap_range.end() ) {
					aft.add_edge( scaffold.total_residue(), first_optimization_residue + 1, Fold_tree::PEPTIDE );
					aft.add_edge( first_optimization_residue + 1, primary_component.scaffold_gap_range.end(), Fold_tree::PEPTIDE );
					aft.add_edge( first_optimization_residue + 1, primary_component.scaffold_gap_range.end() - 1, 1 ); // jump
					aft.add_edge( primary_component.scaffold_gap_range.end() - 1, 1, Fold_tree::PEPTIDE );
					aft.reorder( scaffold.total_residue() ); // fold backwards C->N
					scaffold.set_fold_tree( aft );
					
					// now set angles
					std::set< DihedralInfo >::const_reverse_iterator d = primary_component.dihedrals.rbegin(), de = primary_component.dihedrals.rend();
					while ( d != de ) {
						if ( d->residue() > primary_component.scaffold_gap_range.begin() ) {
							d->apply( scaffold );
							++d;
						}
					}
				}
				break;
			default:
				break;
		}
	}
	
	// create set of components sorted on left endpoint of scaffold gap ranges
	SortedMatchComponents components;
	components.insert( match_result.components.begin(), match_result.components.end() );
	
	// count total number of residues;
	Integer total_residue = scaffold.total_residue();
	for ( SortedMatchComponents::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		MatchComponent const & component = *c;
		total_residue = total_residue + component.loop_subrange.length() - component.scaffold_gap_range.length();
	}
	
	// make fold tree for loop closure/design
	Fold_tree f;
	Integer start_vertex = 1; // initial vertex at N-terminus
	Integer offset = 0; // incrementally track differences after each gap/loop insertion
	Integer jump_label = 0;
	utility::vector1< ResidueRange > es_secondary_loops; // store for use in constructing parts of tree from epitope
	ResidueRange es_primary_loop;
	std::set< DihedralInfo > es_dihedrals; // as we set up the tree, track and renumber any dihedrals that need to be changed

	// handle parts of tree that come from scaffold
	for ( SortedMatchComponents::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		MatchComponent const & component = *c;
		
		Integer const first_branch_vertex = component.scaffold_gap_range.begin() + offset - ALLOW_MOVE - 1;
		Integer const second_branch_vertex = first_branch_vertex + ALLOW_MOVE + component.loop_subrange.length()
		                                     + ALLOW_MOVE + 1;
		
		f.add_edge( start_vertex, first_branch_vertex, Fold_tree::PEPTIDE );
		f.add_edge( first_branch_vertex, first_branch_vertex + ALLOW_MOVE, Fold_tree::PEPTIDE ); // moveable loop closure edge
		f.add_edge( first_branch_vertex, second_branch_vertex, ++jump_label );
		f.add_edge( second_branch_vertex, second_branch_vertex - ALLOW_MOVE, Fold_tree::PEPTIDE ); // moveable loop closure edge
		
		// store jump ranges for later use in constructing parts of tree from epitope
		if ( primary_component.scaffold_gap_range == component.scaffold_gap_range ) {
			es_primary_loop = ResidueRange( first_branch_vertex + ALLOW_MOVE + 1, second_branch_vertex - ALLOW_MOVE - 1 );
		} else {
			es_secondary_loops.push_back( ResidueRange( first_branch_vertex + ALLOW_MOVE + 1, second_branch_vertex - ALLOW_MOVE - 1 ) );
		}
		
		// dihedrals
		if ( component.dihedrals.size() > 0 ) { // dihedrals exist in component, so track their new numbering for epitope-scaffold
			for ( std::set< DihedralInfo >::const_iterator d = component.dihedrals.begin(), de = component.dihedrals.end(); d != de; ++d ) {
				DihedralInfo dihedral( *d ); // will be re-numbered dihedral for epitope-scaffold

				// notice here that dihedrals for the gap/match residue are not applied as they are only used to improve rms
				if ( dihedral.residue() < component.scaffold_gap_range ) { // must be completely left of the loop

					dihedral.set_residue( dihedral.residue() + offset );
					es_dihedrals.insert( dihedral );

				} else if ( dihedral.residue() > component.scaffold_gap_range ) { // must be completely right of the loop

					dihedral.set_residue( dihedral.residue() + offset + component.loop_subrange.length() - component.scaffold_gap_range.length() );
					es_dihedrals.insert( dihedral );

				}
				
			}
		}
		
		start_vertex = second_branch_vertex; // move start vertex
		offset += component.loop_subrange.length() - component.scaffold_gap_range.length();
	}
	f.add_edge( start_vertex, total_residue, Fold_tree::PEPTIDE ); // last edge to end of epitope-scaffold

	// handle parts of tree that come from epitope by anchoring at primary component and connecting to secondaries
	Integer const middle_of_primary = es_primary_loop.begin() + es_primary_loop.length() / 2; // middle of primary loop
	switch ( match_result.system_type ) { // handle primary component here
		case AlignmentSystem::N2C: // connect at N
			f.add_edge( es_primary_loop.begin() - 1, middle_of_primary, Fold_tree::PEPTIDE );
			f.add_edge( middle_of_primary, es_primary_loop.end(), Fold_tree::PEPTIDE );
			break;
		case AlignmentSystem::C2N: // connect at C
			f.add_edge( es_primary_loop.end() + 1, middle_of_primary, Fold_tree::PEPTIDE );
			f.add_edge( middle_of_primary, es_primary_loop.begin(), Fold_tree::PEPTIDE );
			break;
		default: // double break alignment systems (S, E, SS) -- connect from branch vertex to middle of loop
			f.add_edge( es_primary_loop.begin() - ALLOW_MOVE - 1, middle_of_primary, ++jump_label ); // scaffold -> primary loop
			f.add_edge( middle_of_primary, es_primary_loop.begin(), Fold_tree::PEPTIDE ); // grow to primary loop N-terminus
			f.add_edge( middle_of_primary, es_primary_loop.end(), Fold_tree::PEPTIDE ); // grow to primary loop C-terminus
			break;
	}
	for ( utility::vector1< ResidueRange >::const_iterator sl = es_secondary_loops.begin(), sle = es_secondary_loops.end(); sl != sle; ++sl ) { // now connect primary to secondaries
		ResidueRange const & es_secondary_loop = *sl;
		Integer const middle_of_secondary = es_secondary_loop.begin() + es_secondary_loop.length() / 2; // middle of secondary loop
		
		f.add_edge( middle_of_primary, middle_of_secondary, ++jump_label ); // connect primary -> secondary
		f.add_edge( middle_of_secondary, es_secondary_loop.begin(), Fold_tree::PEPTIDE ); // grow to secondary loop N-terminus
		f.add_edge( middle_of_secondary, es_secondary_loop.end(), Fold_tree::PEPTIDE ); // grow to secondary loop C-terminus
	}

	// finalize tree
	f.reorder( 1 ); // start from N-terminus

	// new data for epitope-scaffold
	epitope_scaffold.set_fold_tree( f );
	epitope_scaffold.set_fullatom_flag( true, false ); // force booleans: fullatom, repack
	FArray3D_float full_coord( 3, param::MAX_ATOM(), total_residue );

	// counter variables for tracking progress
	Integer counter = 1; // keeps track of next residue to insert into
	Integer scaffold_counter = 1;
	
	// fill epitope scaffold
	for ( SortedMatchComponents::const_iterator c = components.begin(), ce = components.end(); c != ce; ++c ) {
		MatchComponent const & component = *c;
		
		// first handle scaffold piece
		ResidueRange scaffold_transfer_range( scaffold_counter, component.scaffold_gap_range.begin() - 1 );
		transfer_identity_info_between_Poses( scaffold, scaffold_transfer_range, epitope_scaffold, counter );
		fill_full_coord_from_Pose( scaffold, scaffold_transfer_range, full_coord, counter ); // counter modified here
		
		scaffold_counter = component.scaffold_gap_range.end() + 1;
		
		// next handle epitope piece
		transfer_identity_info_between_Poses( aligned_epitope, component.loop_subrange, epitope_scaffold, counter );
		fill_full_coord_from_Pose( aligned_epitope, component.loop_subrange, full_coord, counter ); // counter modified here
	}
	// transfer rest of scaffold, if present
	if ( counter < total_residue + 1 ) {
		ResidueRange scaffold_transfer_range( scaffold_counter, scaffold.total_residue() );
		transfer_identity_info_between_Poses( scaffold, scaffold_transfer_range, epitope_scaffold, counter );
		fill_full_coord_from_Pose( scaffold, scaffold_transfer_range, full_coord, counter ); // counter modified here
	}
	
	assert( counter == total_residue + 1 ); // make sure number of transfered residues is correct

	// micromanage N2C and C2N primary components
	switch ( match_result.system_type ) {
		case AlignmentSystem::N2C:
			switch ( match_result.alignment_center ) {
				case AlignmentSystem::N:
					N2C_N_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.begin(), full_coord, es_primary_loop.begin() );
					break;
				case AlignmentSystem::CA:
					N2C_CA_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.begin(), full_coord, es_primary_loop.begin() );
					break;
				case AlignmentSystem::C:
					N2C_C_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.begin(), full_coord, es_primary_loop.begin() );
					break;
				default:
					break;
			}
			break;

		case AlignmentSystem::C2N:
			switch ( match_result.alignment_center ) {
				case AlignmentSystem::N:
					C2N_N_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.end(), full_coord, es_primary_loop.end() );
					break;
				case AlignmentSystem::CA:
					C2N_CA_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.end(), full_coord, es_primary_loop.end() );
					break;
				case AlignmentSystem::C:
					C2N_C_Align::micromanage_takeoff( scaffold.full_coord(), primary_component.scaffold_gap_range.end(), full_coord, es_primary_loop.end() );
					break;
				default:
					break;
			}
			break;

		default:
			break;
	}

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, total_residue );
	full_coord_to_Eposition( total_residue, full_coord, Eposition );

	// set coordinates
	epitope_scaffold.set_coords( false, Eposition, full_coord, false ); // internally recomputes all angles
	
	// now change any dihedral angles indicated
	if ( es_dihedrals.size() > 0 ) {
		for ( std::set< DihedralInfo >::const_iterator d = es_dihedrals.begin(), de = es_dihedrals.end(); d != de; ++d ) {
			d->apply( epitope_scaffold ); // apply dihedral
		}
	}

	// correct pdb info data
	char const scaffold_chain_id = scaffold.pdb_info().res_chain( 1 );
	for ( Integer i = 1, ie = epitope_scaffold.total_residue(); i <= ie; ++i ) {
		epitope_scaffold.pdb_info().set_pdb_chain( i, scaffold_chain_id );
		epitope_scaffold.pdb_info().set_pdb_res( i, i );
		epitope_scaffold.pdb_info().set_pdb_insert_let( i, ' ' );
	}
	
	epitope_scaffold.pdb_info().set_use_pdb_numbering( true ); // use pdb info on output
	
}


} // namespace match
} // namespace epigraft
