// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   match_functions.hh
/// @brief  Various functions for match.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_match_match_functions_HH_
#define INCLUDED_epigraft_match_match_functions_HH_

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/MatchResult.fwd.hh>

// numeric headers
#include <numeric/xyzVector.hh>


namespace epigraft {
namespace match {


/// @brief calculated n-terminal 5-point chainbreak rms, call using full_coord
/// @note  N,CA,C,O from residue and N from residue + 1
inline
Real
n_terminal_chainbreak(
	FArray3D_float const & scaffold_full_coord,
	Integer const & scaffold_residue,
	FArray3D_float const & loop_full_coord,
	Integer const & loop_residue
)
{
	FArray2D_float match_crd; // coordinates for n-terminal endpoint match residue
	extract_1bb_from_full_coord( scaffold_full_coord, scaffold_residue, match_crd );
	
	FArray2D_float loop_crd; // coordinates for n-terminal endpoint loop residue
	extract_1bb_from_full_coord( loop_full_coord, loop_residue, loop_crd );

	// append 'N' from residue after n-terminal match residue for chainbreak scoring and other filtering
	match_crd.redimension( 3, 5 );
	Integer n_idx = scaffold_full_coord.index( 1, 1, scaffold_residue + 1 ); // 'N'
	match_crd[ 12 ] = scaffold_full_coord[ n_idx ]; // x
	match_crd[ 13 ] = scaffold_full_coord[ n_idx + 1 ]; // y
	match_crd[ 14 ] = scaffold_full_coord[ n_idx + 2 ]; // z
	
	loop_crd.redimension( 3, 5 );
	n_idx = loop_full_coord.index( 1, 1, loop_residue + 1 ); // 'N'
	loop_crd[ 12 ] = loop_full_coord[ n_idx ]; // x
	loop_crd[ 13 ] = loop_full_coord[ n_idx + 1 ]; // y
	loop_crd[ 14 ] = loop_full_coord[ n_idx + 2 ]; // z
	
	// measure chainbreak
	return five_point_chainbreak( match_crd, loop_crd );
}


/// @brief calculated n-terminal 5-point chainbreak rms, call using Pose
/// @note  N,CA,C,O from residue and N from residue + 1
inline
Real
n_terminal_chainbreak(
	Pose const & scaffold,
	Integer const & scaffold_residue,
	Pose const & loop,
	Integer const & loop_residue
)
{
	return n_terminal_chainbreak( scaffold.full_coord(), scaffold_residue, loop.full_coord(), loop_residue );
}


/// @brief calculated c-terminal 5-point chainbreak rms, call using full_coord
/// @note  N,CA,C,O from residue and C from residue - 1
inline
Real
c_terminal_chainbreak(
	FArray3D_float const & scaffold_full_coord,
	Integer const & scaffold_residue,
	FArray3D_float const & loop_full_coord,
	Integer const & loop_residue
)
{
	FArray2D_float match_crd; // coordinates for c-terminal endpoint match residue
	extract_1bb_from_full_coord( scaffold_full_coord, scaffold_residue, match_crd );
	
	FArray2D_float loop_crd; // coordinates for c-terminal endpoint loop residue
	extract_1bb_from_full_coord( loop_full_coord, loop_residue, loop_crd );

	// append 'C' from residue prior to c-terminal match residue for chainbreak scoring and other filtering
	match_crd.redimension( 3, 5 );
	Integer c_idx = scaffold_full_coord.index( 1, 3, scaffold_residue - 1 ); // 'C'
	match_crd[ 12 ] = scaffold_full_coord[ c_idx ]; // x
	match_crd[ 13 ] = scaffold_full_coord[ c_idx + 1 ]; // y
	match_crd[ 14 ] = scaffold_full_coord[ c_idx + 2 ]; // z
	
	loop_crd.redimension( 3, 5 );
	c_idx = loop_full_coord.index( 1, 3, loop_residue - 1 ); // 'C'
	loop_crd[ 12 ] = loop_full_coord[ c_idx ]; // x
	loop_crd[ 13 ] = loop_full_coord[ c_idx + 1 ]; // y
	loop_crd[ 14 ] = loop_full_coord[ c_idx + 2 ]; // z
	
	// measure chainbreak
	return five_point_chainbreak( match_crd, loop_crd );
}


/// @brief calculated c-terminal 5-point chainbreak rms, call using Pose
/// @note  N,CA,C,O from residue and C from residue - 1
inline
Real
c_terminal_chainbreak(
	Pose const & scaffold,
	Integer const & scaffold_residue,
	Pose const & loop,
	Integer const & loop_residue
)
{
	return c_terminal_chainbreak( scaffold.full_coord(), scaffold_residue, loop.full_coord(), loop_residue );
}


/// @brief compute filters for a match
/// @note  FArray2D is a 3x6 array representing atoms where the first 4 atoms are the
/// @note  backbone atoms from a match residue and the last two atoms (either 'N' or 'C'
/// @note  and then 'CA') are from the appropriate residue adjacent to the match
template< typename T >
inline
void
compute_terminus_filters(
	FArray2D< T > const & scaffold_crd,
	FArray2D< T > const & loop_crd,
	Real & rms,
	Real & closure_angle
)
{	
	rms = five_point_chainbreak( scaffold_crd, loop_crd ); // bb and 'N'/'C'
	closure_angle = angle_of_degrees( &scaffold_crd( 1, 6 ), &scaffold_crd( 1, 2 ), &loop_crd( 1, 6 ), &loop_crd( 1, 2 ) ); // angle between 'CA'->'CA' vector
}


/// @brief compute n-terminal filters for a match
inline
void
compute_n_terminal_match_filters(
	FArray3D_float const & scaffold_full_coord,
	Integer const & scaffold_gap_range_begin,
	FArray3D_float const & loop_full_coord,
	Integer const & loop_subrange_begin,
	MatchComponent & match_component
)
{
	// data
	FArray2D_float match_crd;
	FArray2D_float loop_crd;
	Integer idx;
	
	// n-terminus
	extract_1bb_from_full_coord( scaffold_full_coord, scaffold_gap_range_begin, match_crd );
	extract_1bb_from_full_coord( loop_full_coord, loop_subrange_begin, loop_crd );

	match_crd.redimension( 3, 6 );
	loop_crd.redimension( 3, 6 );

	// append 'N' and 'CA' from residue after n-terminal match residue for filters
	idx = scaffold_full_coord.index( 1, 1, scaffold_gap_range_begin + 1 ); // 'N'
	match_crd[ 12 ] = scaffold_full_coord[ idx ]; // x
	match_crd[ 13 ] = scaffold_full_coord[ ++idx ]; // y
	match_crd[ 14 ] = scaffold_full_coord[ ++idx ]; // z
	// 'CA' directly follows
	match_crd[ 15 ] = scaffold_full_coord[ ++idx ]; // x
	match_crd[ 16 ] = scaffold_full_coord[ ++idx ]; // y
	match_crd[ 17 ] = scaffold_full_coord[ ++idx ]; // z

	// append 'N' and 'CA' from residue after n-terminal match residue for filters
	idx = loop_full_coord.index( 1, 1, loop_subrange_begin + 1 ); // 'N'
	loop_crd[ 12 ] = loop_full_coord[ idx ]; // x
	loop_crd[ 13 ] = loop_full_coord[ ++idx ]; // y
	loop_crd[ 14 ] = loop_full_coord[ ++idx ]; // z
	// 'CA' directly follows
	loop_crd[ 15 ] = loop_full_coord[ ++idx ]; // x
	loop_crd[ 16 ] = loop_full_coord[ ++idx ]; // y
	loop_crd[ 17 ] = loop_full_coord[ ++idx ]; // z

	// compute n-terminal filters
	compute_terminus_filters( match_crd, loop_crd, match_component.n_terminal_rms, match_component.n_closure_angle );
}


/// @brief compute c-terminal filters for a match
inline
void
compute_c_terminal_match_filters(
	FArray3D_float const & scaffold_full_coord,
	Integer const & scaffold_gap_end,
	FArray3D_float const & loop_full_coord,
	Integer const & loop_subrange_end,
	MatchComponent & match_component
)
{
	// data
	FArray2D_float match_crd;
	FArray2D_float loop_crd;
	Integer idx;
	
	// c-terminus
	extract_1bb_from_full_coord( scaffold_full_coord, scaffold_gap_end, match_crd );
	extract_1bb_from_full_coord( loop_full_coord, loop_subrange_end, loop_crd );

	match_crd.redimension( 3, 6 );
	loop_crd.redimension( 3, 6 );

	// append 'C' and 'CA' from residue prior to c-terminal match residue for filters
	idx = scaffold_full_coord.index( 1, 2, scaffold_gap_end - 1 ); // 'CA'
	match_crd[ 15 ] = scaffold_full_coord[ idx ]; // x
	match_crd[ 16 ] = scaffold_full_coord[ ++idx ]; // y
	match_crd[ 17 ] = scaffold_full_coord[ ++idx ]; // z
	// 'C' directly follows
	match_crd[ 12 ] = scaffold_full_coord[ ++idx ]; // x
	match_crd[ 13 ] = scaffold_full_coord[ ++idx ]; // y
	match_crd[ 14 ] = scaffold_full_coord[ ++idx ]; // z
	
	// append 'C' and 'CA' from residue prior to c-terminal match residue for filters
	idx = loop_full_coord.index( 1, 2, loop_subrange_end - 1 ); // 'CA'
	loop_crd[ 15 ] = loop_full_coord[ idx ]; // x
	loop_crd[ 16 ] = loop_full_coord[ ++idx ]; // y
	loop_crd[ 17 ] = loop_full_coord[ ++idx ]; // z
	// 'C' directly follows
	loop_crd[ 12 ] = loop_full_coord[ ++idx ]; // x
	loop_crd[ 13 ] = loop_full_coord[ ++idx ]; // y
	loop_crd[ 14 ] = loop_full_coord[ ++idx ]; // z

	// compute c-terminal filters
	compute_terminus_filters( match_crd, loop_crd, match_component.c_terminal_rms, match_component.c_closure_angle );
}


/// @brief compute filters for a match
inline
void
compute_match_filters(
	FArray3D_float const & scaffold_full_coord,
	ResidueRange const & scaffold_gap_range,
	FArray3D_float const & loop_full_coord,
	ResidueRange const & loop_subrange,
	MatchComponent & match_component
)
{
	// n-terminus
	compute_n_terminal_match_filters( scaffold_full_coord, scaffold_gap_range.begin(), loop_full_coord, loop_subrange.begin(), match_component );
	
	// c-terminus
	compute_c_terminal_match_filters( scaffold_full_coord, scaffold_gap_range.end(), loop_full_coord, loop_subrange.end(), match_component );
}


/// @brief CA centroid
inline
FArray2D_float
ca_centroid( Pose const & p ) {
	FArray3D_float const & fc = p.full_coord();
	Integer total_residue = p.total_residue();
	
	FArray2D_float xyz( 3, 1, 0.0 );
	
	for ( Integer res = 1; res <= total_residue; ++res ) {
		Integer idx = fc.index( 1, 2, res );
		xyz[ 0 ] += fc[ idx ]; // x
		xyz[ 1 ] += fc[ ++idx ]; // y
		xyz[ 2 ] += fc[ ++idx ]; // z
	}
	
	xyz[ 0 ] /= total_residue;
	xyz[ 1 ] /= total_residue;
	xyz[ 2 ] /= total_residue;
	
	return xyz;
}


/// @brief create epitope-scaffold Pose (including proper fold-tree) for closure and design
void
make_predesign_epitope_scaffold(
	Pose const & original_scaffold,
	Pose const & epitope,
	MatchResult const & match_result,
	Pose & epitope_scaffold,
	Integer const & moveable_closure_residues = 3
);


} // namespace match
} // namespace epigraft

#endif /*INCLUDED_epigraft_match_match_functions_HH_*/
