// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   match_io.cc
/// @brief  I/O for match.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/match/match_io.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/match_functions.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/conformation/DihedralInfo.hh>
#include <epigraft/design/EpitopeScaffold.hh>
#include <epigraft/design/GraftInfo.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/epigraft_io.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/GraftOptions.hh>

// ObjexxFCL headers
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// utility headers
#include <utility/file/FileName.hh>
#include <utility/io/izstream.hh>
#include <utility/vector1.hh>

// C++ headers
#include <map>
#include <set>
#include <sstream>
#include <string>


namespace epigraft {
namespace match {


/// @brief match result from file (from output table), fill in as much data as possible
/// @brief from given file
void
match_results_from_file(
	std::string const & filename,
	utility::vector1< LoopInfo > loops_to_scan,
	std::map< std::string, utility::vector1< MatchResult > > & pdb_to_match_results
)
{
	using std::istringstream;
	using epigraft::match::align::AlignmentSystem;
	using epigraft::conformation::DihedralInfo;

	utility::io::izstream infile ( filename );
	if ( !infile ) {
		std::ostringstream ss;
		ss << "match_results_from_file: cannot open file " << filename << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}

	std::string line;

	MatchResult * match_result = NULL;
	utility::vector1< std::string > entries;
	while ( getline( infile, line ) ) {

		entries.clear();
		split_string( line, entries ); // grab all columns from input file

		if ( entries[ 1 ].substr( 0 , 1 ) == "#" ) { // skip comment lines
			continue;
		}

		if ( entries.size() < 7 ) {
			std::ostringstream ss;
			ss << "match_results_from_file: following line in match results file missing required number of columns:" << std::endl;
			ss << line << std::endl;
			utility::exit( __FILE__, __LINE__, ss.str() );
		}

		// decide if this is a primary component or secondary component
		bool processing_primary = false;
		if ( *( entries.begin()->begin() ) != '*' ) { // primary
			processing_primary = true;

			pdb_to_match_results[ entries[ 1 ] ].push_back( MatchResult() ); // primary indicates new MatchResult

			match_result = &pdb_to_match_results[ entries[ 1 ] ].back();

			// alignment system
			match_result->system_name = entries[ 3 ];
			std::string sys = match_result->system_name.substr( 0, 3 );
			if ( sys == "N2C" ) {
				match_result->system_type = AlignmentSystem::N2C;

				std::string center = match_result->system_name.substr( 4, 2 );
				if ( center == "N" ) {
					match_result->alignment_center = AlignmentSystem::N;
				} else if ( center == "CA" ) {
					match_result->alignment_center = AlignmentSystem::CA;
				} else if ( center == "C" ) {
					match_result->alignment_center = AlignmentSystem::C;
				}
			} else if ( sys == "C2N" ) {
				match_result->system_type = AlignmentSystem::C2N;

				std::string center = match_result->system_name.substr( 4, 2 );
				if ( center == "N" ) {
					match_result->alignment_center = AlignmentSystem::N;
				} else if ( center == "CA" ) {
					match_result->alignment_center = AlignmentSystem::CA;
				} else if ( center == "C" ) {
					match_result->alignment_center = AlignmentSystem::C;
				}
			} else if ( sys == "E" ) {
				match_result->system_type = AlignmentSystem::ENDPOINT;
				match_result->alignment_center = AlignmentSystem::DOUBLEBREAK;
			} else if ( sys == "S" ) {
				match_result->system_type = AlignmentSystem::SUPERPOSITION;
				match_result->alignment_center = AlignmentSystem::DOUBLEBREAK;
			} else if ( sys == "SS" ) {
				match_result->system_type = AlignmentSystem::SPECIFIC_SUPERPOSITION;
				match_result->alignment_center = AlignmentSystem::DOUBLEBREAK;
			}

			if ( entries.size() >= 27 ) { // transformation matrix exists, fill it in
				// remember that FArrays are column-major:
				// | 0 4  8 12 |
				// | 1 5  9 13 |
				// | 2 6 10 14 |
				// | 3 7 11 15 |  ->  this last row is always | 0 0 0 1 |
				istringstream( entries[ 16 ] ) >> match_result->transformation_matrix[ 0 ];
				istringstream( entries[ 17 ] ) >> match_result->transformation_matrix[ 1 ];
				istringstream( entries[ 18 ] ) >> match_result->transformation_matrix[ 2 ];
				match_result->transformation_matrix[ 3 ] = 0.0;
				istringstream( entries[ 19 ] ) >> match_result->transformation_matrix[ 4 ];
				istringstream( entries[ 20 ] ) >> match_result->transformation_matrix[ 5 ];
				istringstream( entries[ 21 ] ) >> match_result->transformation_matrix[ 6 ];
				match_result->transformation_matrix[ 7 ] = 0.0;
				istringstream( entries[ 22 ] ) >> match_result->transformation_matrix[ 8 ];
				istringstream( entries[ 23 ] ) >> match_result->transformation_matrix[ 9 ];
				istringstream( entries[ 24 ] ) >> match_result->transformation_matrix[ 10 ];
				match_result->transformation_matrix[ 11 ] = 0.0;
				istringstream( entries[ 25 ] ) >> match_result->transformation_matrix[ 12 ];
				istringstream( entries[ 26 ] ) >> match_result->transformation_matrix[ 13 ];
				istringstream( entries[ 27 ] ) >> match_result->transformation_matrix[ 14 ];
				match_result->transformation_matrix[ 15 ] = 1.0;
			}
		} // primary

		// now store loop info in component
		MatchComponent component;
		istringstream( entries[ 2 ] ) >> component.loop_id;

		Integer gap_begin;
		istringstream( entries[ 4 ] ) >> gap_begin;
		Integer gap_end;
		istringstream( entries[ 5 ] ) >> gap_end;
		component.scaffold_gap_range = ResidueRange( gap_begin, gap_end );

		Integer native_loop_begin;
		istringstream( entries[ 6 ] ) >> native_loop_begin;
		Integer native_loop_end;
		istringstream( entries[ 7 ] ) >> native_loop_end;
		component.native_loop_subrange = ResidueRange( native_loop_begin, native_loop_end );

		// Important: component.loop_subrange is _not_ filled in here, this is done in external post-processing

		// for 'S' and 'SS' systems, make sure the number of residues is the same
		if ( processing_primary && ( match_result->system_type == AlignmentSystem::SUPERPOSITION ||
		     match_result->system_type == AlignmentSystem::SPECIFIC_SUPERPOSITION ) ) {
		     	if ( component.scaffold_gap_range.length() != component.native_loop_subrange.length() ) {
					std::ostringstream ss;
					ss << "match_results_from_file: S/SS system indicated in the following line, but there's a mismatch in the number of residues in gap versus loop:" << std::endl;
					ss << line << std::endl;
					utility::exit( __FILE__, __LINE__, ss.str() );
		     	}
		}

		// component dihedrals?
		if ( entries.size() >= 15 ) {
			for ( Integer i = 8; i <= 15; ++i ) {
				if ( entries[ i ] != "-" ) {
					replace( entries[ i ].begin(), entries[ i ].end(), ':', ' ' ); // dihedral information delimited by ':'

					istringstream dss( entries[ i ] );

					Integer residue;
					Integer a_t; // of type DihedralInfo::AngleType
					Real angle;

					dss >> residue;
					dss >> a_t;
					dss >> angle;

					component.dihedrals.insert( DihedralInfo( (DihedralInfo::AngleType)a_t, residue, angle ) );
				}
			}
		}

		match_result->components.push_back( component );

	} // foreach line

	infile.close(); // close input file

	// cache loop offsets for loop id computation below
	utility::vector1< int > loop_offsets;
	for ( Integer loop_id = 1, last_loop_id = loops_to_scan.size(); loop_id <= last_loop_id; ++loop_id ) {
		loop_offsets.push_back( compute_loop_offset( loops_to_scan, loop_id ) );
	}

	// now correct loop ids so they match those is available LoopInfo
	// and fill in proper loop subranges w/ respect to all loops
	// note that loop id is currently artificial and is just meant to match
	// the index of the loop in its vector
	for ( std::map< std::string, utility::vector1< MatchResult > >::iterator i = pdb_to_match_results.begin(), ie = pdb_to_match_results.end(); i != ie; ++i ) {
		for ( utility::vector1< MatchResult >::iterator m = i->second.begin(), me = i->second.end(); m != me; ++m ) {
			for ( utility::vector1< MatchComponent >::iterator component = m->components.begin(), component_e = m->components.end(); component != component_e; ++component ) {
				for ( Integer loop_id = 1, last_loop_id = loops_to_scan.size(); loop_id <= last_loop_id; ++loop_id ) {
					LoopInfo const & loop = loops_to_scan[ loop_id ];

					if ( loop.full_native_range().contains( component->native_loop_subrange ) ) {
						component->loop_id = loop_id;
						component->loop_subrange = loop.convert_native_to_internal( component->native_loop_subrange ) + loop_offsets[ loop_id ];
					}

				}
			}
		}
	}

}


/// @brief output pdb given a match result
void
output_aligned_loops_pdb(
	Pose const & epitope,
	MatchResult const & match_result,
	std::string const & output_pdb_filename
)
{
	// fragment epitope to get only desired loops
	utility::vector1< bool > keep( epitope.total_residue(), false );
	for ( Integer i = 1, ie = match_result.components.size(); i <= ie; ++i ) {

		ResidueRange const & loop_subrange = match_result.components[ i ].loop_subrange;

		for ( Integer residue = loop_subrange.begin(), last_residue = loop_subrange.end(); residue <= last_residue; ++residue ) {
			keep[ residue ] = true;
		}
	}

	Pose loops;
	fragment_Pose( epitope, keep, loops, true ); // boolean keep_pdb_info

	// transform loops
	Pose aligned_loops;
	aligned_loops = loops;
	aligned_loops.transform_GL( match_result.transformation_matrix );

	// use pdb numbering
	aligned_loops.pdb_info().set_use_pdb_numbering( true );

	// create directory if necessary
	utility::file::FileName fn( output_pdb_filename );
	recursively_create_directory( fn.path() );

	// dump pdb
	aligned_loops.dump_pdb( output_pdb_filename );
}


/// @brief output pdb of pre-design structure given a match result
void
output_predesign_structure_pdb(
	Pose const & scaffold,
	Pose const & epitope,
	MatchResult const & match_result,
	std::set< Integer > const & keep_natro_residues,
	std::string const & output_pdb_filename
)
{
	using epigraft::design::GraftInfo;
	using epigraft::design::EpitopeScaffold;

	GraftInfo graft_info( match_result );
	graft_info.lock();

	EpitopeScaffold epitope_scaffold( scaffold, epitope, graft_info, keep_natro_residues, true );

	// create directory if necessary
	utility::file::FileName fn( output_pdb_filename );
	recursively_create_directory( fn.path() );

	epitope_scaffold.pose().dump_pdb( output_pdb_filename );
}


/// @brief output pdb of pre-design structure given a match result with Antibody
void
output_predesign_structure_pdb_with_Ab(
	Pose const & scaffold,
	Pose const & epitope,
	Pose const & antibody,
	MatchResult const & match_result,
	std::set< Integer > const & keep_natro_residues,
	std::string const & output_pdb_filename
)
{
	using epigraft::design::GraftInfo;
	using epigraft::design::EpitopeScaffold;

	GraftInfo graft_info( match_result );
	graft_info.lock();

	EpitopeScaffold epitope_scaffold( scaffold, epitope, graft_info, keep_natro_residues, true );

	epitope_scaffold.set_Ab( antibody );
	epitope_scaffold.connect_Ab();

	// create directory if necessary
	utility::file::FileName fn( output_pdb_filename );
	recursively_create_directory( fn.path() );

	epitope_scaffold.pose().dump_pdb( output_pdb_filename );
}


} // namespace epigraft
} // namespace match
