// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   check_specific_sites.hh
/// @brief  Given a set of match positions, output results of statistics and filters (e.g. clash check).
/// @author Bill Schief (schief@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/match/check_specific_sites.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_constants.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/match/align/C2N_C_Align.hh>
#include <epigraft/match/align/C2N_CA_Align.hh>
#include <epigraft/match/align/C2N_N_Align.hh>
#include <epigraft/match/align/E_Align.hh>
#include <epigraft/match/align/N2C_C_Align.hh>
#include <epigraft/match/align/N2C_CA_Align.hh>
#include <epigraft/match/align/N2C_N_Align.hh>
#include <epigraft/match/align/S_Align.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/epigraft_io.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/Checkpoint.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/ResidueRange.hh>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// Rosetta headers
#include <pose.h>
#include <pose_io.h>
#include <after_opts.h>
#include <files_paths.h>

// utility headers
#include <utility/io/ozstream.hh>
#include <utility/pointer/access_ptr.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/vector1.hh>

// C++ Headers
#include <fstream>
#include <iostream>
#include <map>
#include <string>


namespace epigraft {
namespace match {


/// @brief the old check specific sites routine where the transformation was never stored
void
old_check_specific_sites(
	GraftOptions const & options
)
{
	using namespace epigraft::match::align;

	// load native complex from file, NOTE: assume Ab comes first!
	AntibodyComplex native( options.native_complex_filename, options.nres_Ab, true, true ); // booleans: Ab_first, refold_sidechains_from_chi
	epigraft::convert_to_gly( native.antigen() );

	// load old format results
	std::map< std::string, utility::vector1< MatchResult > > results_to_process;
	epigraft::load_old_format_match_results( options.input_from_match, results_to_process );

	// load alignment systems -- std::map provides cleaner implementation and reasonable performance on average
	//                           as opposed to a series of 'if' statements.  In naive tests 'if' wins when the
	//                           number of of key strings is small ( <= 4? ) or the sequence of strings to be
	//                           looked up is extremely repetitive, otherwise std::map performs as well or better.
	std::map< std::string, AlignmentSystemOP > alignment_systems;

	// TODO: see if AlignmentSystem::name() can be made static virtual

	// "E" system
	if ( options.use_E_align) {
		alignment_systems[ "E" ] = AlignmentSystemOP( new E_Align( options.max_closure_rms ) );
	}

	// "S" system
	if ( options.use_S_align) {
		alignment_systems[ "S" ] = AlignmentSystemOP( new S_Align( options.max_closure_rms ) );
	}

	// "N2C" CA-centered system
	alignment_systems[ "N2C_CA" ] = AlignmentSystemOP( new N2C_CA_Align( options.max_closure_rms ) );

	// "N2C" N-centered system
	alignment_systems[ "N2C_N" ] = AlignmentSystemOP( new N2C_N_Align( options.max_closure_rms ) );

	// "N2C" C-centered system
	alignment_systems[ "N2C_C" ] = AlignmentSystemOP( new N2C_C_Align( options.max_closure_rms ) );

	// "C2N" CA-centered system
	alignment_systems[ "C2N_CA" ] = AlignmentSystemOP( new C2N_CA_Align( options.max_closure_rms) );

	// "C2N" N-centered system
	alignment_systems[ "C2N_N" ] = AlignmentSystemOP( new C2N_N_Align( options.max_closure_rms ) );

	// "C2N" C-centered system
	alignment_systems[ "C2N_C" ] = AlignmentSystemOP( new C2N_C_Align( options.max_closure_rms ) );

	// create iterator outside of loop due to potential checkpointing
	std::map< std::string, utility::vector1< MatchResult > >::iterator s = results_to_process.begin();

	// for checkpointing
	Checkpoint checkpoint( files_paths::score_path + options.checkpoint_filename );
	if ( options.use_checkpoint ) {
		checkpoint.get_last_id_processed();

		if ( checkpoint.current_id() >= results_to_process.size() ) {
			// at max or somehow over count, so we're done
			utility::exit( __FILE__, __LINE__, "checkpoint indicates all files processed" );
		}

		for ( Integer i = 0, ie = checkpoint.current_id(); i < ie; ++i ) {
			++s; // increment iterator for each record already checked
		}

		// start checkpointing file
		checkpoint.start();
	}

	// open output file
	utility::io::ozstream outfile;
	open_output_file( options.output_filename, outfile );

	// write header
	// TODO: MOVE THIS
	outfile << "filename     alignment_system     gap_begin     gap_end     native_loop_begin     native_loop_end     rms     intra_clash     inter_clash     n_closure_orientation     c_closure_orientation" << std::endl;

	for ( std::map< std::string, utility::vector1< MatchResult > >::iterator last_s = results_to_process.end(); s != last_s; ++s ) {
		std::string scaffold_filename = (*s).first;
		Pose scaffold;
		pose_from_pdb( scaffold, files_paths::start_path + scaffold_filename, true, false, true ); // boolean: fullatom, ideal_pose, read_all_chains

		// convert scaffold to gly
		epigraft::convert_to_gly( scaffold );

		// run through and query all match results
		utility::vector1< MatchResult > & match_results = (*s).second;
		for ( Integer i = 1, ie = match_results.size(); i <= ie; ++i ) {
			MatchResult & match_result = match_results[ i ];

			// grab ranges
			ResidueRange const & loop_subrange = match_result.components[ 1 ].loop_subrange;
			ResidueRange const & match_range = match_result.components[ 1 ].scaffold_gap_range;

			// grab loop (trimmed epitope)
			Pose loop;
			native.trimmed_antigen( loop, loop_subrange );

			// grab alignment system
			AlignmentSystem & alignment_system = *(*alignment_systems.find( match_result.system_name )).second;

			// check alignment
			alignment_system.check_alignment( scaffold, loop, match_result.components[ 1 ], match_result.transformation_matrix );

			// perform full alignment of loop onto scaffold
			Pose aligned_loop;
			aligned_loop = loop;
			aligned_loop.transform_GL( match_result.transformation_matrix );

			// create gapped scaffold, remember that this removes _all_ residues in the given range
			Pose gapped_scaffold;
			remove_segment_from_Pose( scaffold, match_range, gapped_scaffold );

			// fast clash check through octree
			rootstock::BoundingBox bb = bounding_box( gapped_scaffold );
			rootstock::Octree< AtomPoint > octree_gs( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
			fill_octree( octree_gs, gapped_scaffold );
			match_result.components[ 1 ].intra_clash = octree_inter_rep( octree_gs, gapped_scaffold, aligned_loop, true ); // boolean: exclude terminal bb

			if ( match_result.components[ 1 ].intra_clash < options.max_intra_clash ) {
				Pose aligned_Ab;
				aligned_Ab = native.Ab();
				aligned_Ab.transform_GL( match_result.transformation_matrix );
				match_result.inter_clash = octree_inter_rep( octree_gs, gapped_scaffold, aligned_Ab, false ); // boolean: exclude terminal bb

				if ( match_result.inter_clash < options.max_inter_clash ) {
					// everything is ok, print result
					// TODO: move this
					outfile << scaffold_filename
					        << "     "
					        << match_result.system_name
					        << "     "
					        << I( 8, match_result.components[ 1 ].scaffold_gap_range.begin() )
					        << "     "
					        << I( 8, match_result.components[ 1 ].scaffold_gap_range.end() )
					        << "     "
					        << I( 8, match_result.components[ 1 ].native_loop_subrange.begin() )
					        << "     "
					        << I( 8, match_result.components[ 1 ].native_loop_subrange.end() )
					        << "     ";
					std::string direc = match_result.system_name.substr( 0, 3 );
					if ( direc == "N2C" ) {
						outfile << F( 7, 2, match_result.components[ 1 ].c_terminal_rms );
					} else {
						outfile << F( 7, 2, match_result.components[ 1 ].n_terminal_rms );
					}
					outfile << "     "
					        << F( 12, 2, match_result.components[ 1 ].intra_clash )
					        << "     "
					        << F( 12, 2, match_result.inter_clash )
					        << "     "
					        << F( 12, 2, match_result.components[ 1 ].n_closure_angle )
					        << F( 12, 2, match_result.components[ 1 ].c_closure_angle )
					        << std::endl;
				}
			}

			// checkpointing
			if ( options.use_checkpoint ) {
				checkpoint.update();
			}
		}
	}

	// close output file
	outfile.close();

	// close checkpointing
	if ( options.use_checkpoint) {
		checkpoint.stop();
	}
}

} // namespace match
} // namespace epigraft