// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   refine_matches.cc
/// @brief  functions for refining matches
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/match/refine_matches.hh>

// package headers
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/design/EpitopeScaffold.hh>
#include <epigraft/design/GraftInfo.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/match/align/S_Align.hh>
#include <epigraft/match/align/SS_Align.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/epigraft_functions.hh>
#include <epigraft/ResidueRange.hh>

// rosetta headers
#include <pose.h>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <map>
#include <set>
#include <sstream>
#include <string>


namespace epigraft{
namespace match {


/// @brief refine superposition protocol matches by min+repack
/// @details constructs temporary EpitopeScaffold object and calls
/// @details EpitopeScaffold::Ab_epitope_optimize() using both chi
/// @details and rb minimize, and then Epitope:Scaffold::interface_repack()
void
minrepack_superposition_matches(
	GraftOptions const & options,
	AntibodyComplex & native,
	Pose & scaffold,
	utility::vector1< MatchResult > & match_results,
	std::set< Integer > const & keep_natro_residues,
	bool const & output_pdb
)
{
	using namespace epigraft::match::align;
	using epigraft::design::EpitopeScaffold;
	using epigraft::design::GraftInfo;

	std::ostringstream before, after; // for output conversions

	// run through matches and min+repack
	for ( utility::vector1< MatchResult >::iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {

		MatchResult & match_result = *m;

		// skip matches that are not "S"
		if ( !( match_result.system_type == AlignmentSystem::SUPERPOSITION ||
		        match_result.system_type == AlignmentSystem::SPECIFIC_SUPERPOSITION ) ) {
			continue;
		}

		// make graft info
		GraftInfo graft_info( match_result );
		graft_info.set_superposition_defaults(); // force superposition protocol defaults
		graft_info.lock(); // lock graft info to perform range checks

		// make epitope scaffold
		// instantiate initial epitope scaffold object
		EpitopeScaffold epitope_scaffold( scaffold, native.antigen(), graft_info, keep_natro_residues, options.micromanage_termini );

		// set antibody
		epitope_scaffold.set_Ab( native.Ab() ); // antibody reoriented, but not connected
		epitope_scaffold.connect_Ab();

		// convert es to gly/ala as requested
		epitope_scaffold.convert_to_residue_type( options.closure_residue_type ); // gly by default, unless -close_as_ALA

		// recover native sidechains of epitope
		epitope_scaffold.recover_native_epitope_sidechains();

		// calculate ddG before minrepack
		Real ddG_before = epitope_scaffold.ddG();

		// do min repack
		epitope_scaffold.Ab_epitope_optimize( true, false ); // boolean: chi_minimize, rb_minimize
		epitope_scaffold.repack_interface();
		epitope_scaffold.Ab_epitope_optimize( true, false ); // boolean: chi_minimize, rb_minimize

		// ddG after minrepack
		Real ddG_after = epitope_scaffold.ddG();

		// we're done, record in match result
		before << ddG_before;
		after << ddG_after;
		match_result.add_additional_output_data( before.str() );
		match_result.add_additional_output_data( after.str() );

		// reset stringstreams
		before.str( "" );
		after.str( "" );

		if ( output_pdb ) {
			// TODO
		}
	}
}


/// @brief screen matches using repack
/// @details constructs temporary EpitopeScaffold object, converts
/// @details scaffold to either gly/hybrid ala-gly (with epitope sidechains), call
/// @details EpitopeScaffold::repack_interface(), and outputs the inter-clash
void
screen_matches_with_repack(
	GraftOptions const & options,
	AntibodyComplex & native,
	Pose & scaffold,
	utility::vector1< MatchResult > & match_results,
	std::set< Integer > const & keep_natro_residues
)
{
	using epigraft::design::EpitopeScaffold;
	using epigraft::design::GraftInfo;

	std::ostringstream before, after; // for output conversions

	// run through matches and min+repack
	for ( utility::vector1< MatchResult >::iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {

		MatchResult & match_result = *m;

		// make graft info
		GraftInfo graft_info( match_result );
		graft_info.lock(); // lock graft info to perform range checks

		// make epitope scaffold
		// instantiate initial epitope scaffold object
		EpitopeScaffold epitope_scaffold( scaffold, native.antigen(), graft_info, keep_natro_residues, options.micromanage_termini );

		// set antibody
		epitope_scaffold.set_Ab( native.Ab() ); // antibody reoriented, but not connected
		epitope_scaffold.connect_Ab();

		// convert es to gly/ala as requested
		epitope_scaffold.convert_to_residue_type( options.closure_residue_type ); // gly by default, unless -close_as_ALA

		// recover native sidechains of epitope
		epitope_scaffold.recover_native_epitope_sidechains();

		// ranges for clash checking
		ResidueRange const scaffold_range( 1, epitope_scaffold.n_res_scaffold() );
		ResidueRange const ab_range( epitope_scaffold.n_res_scaffold() + 1, epitope_scaffold.pose().total_residue() );

		// clash check before repack
		rootstock::BoundingBox bb_before = bounding_box( epitope_scaffold.pose(), ab_range );
		rootstock::Octree< AtomPoint > oc_before( OCTREE_CUBE_SIZE, bb_before.lower_corner(), bb_before.upper_corner() );
		fill_octree( oc_before, epitope_scaffold.pose(), ab_range );
		Real const inter_clash_before = octree_inter_rep( oc_before, epitope_scaffold.pose(), epitope_scaffold.pose(), scaffold_range );

		// do the interface repack
		epitope_scaffold.repack_interface();

		// clash check after repack
		rootstock::BoundingBox bb_after = bounding_box( epitope_scaffold.pose(), ab_range );
		rootstock::Octree< AtomPoint > oc_after( OCTREE_CUBE_SIZE, bb_after.lower_corner(), bb_after.upper_corner() );
		fill_octree( oc_after, epitope_scaffold.pose(), ab_range );
		Real const inter_clash_after = octree_inter_rep( oc_after, epitope_scaffold.pose(), epitope_scaffold.pose(), scaffold_range );

		// we're done, record in match result
		before << inter_clash_before;
		after << inter_clash_after;
		match_result.add_additional_output_data( before.str() );
		match_result.add_additional_output_data( after.str() );

		// reset stringstreams
		before.str( "" );
		after.str( "" );

	}
}


}
}
