// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   rescore_matches.cc
/// @brief  Takes a set of matches and rescore with respect to their existing internal information.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/match/rescore_matches.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_constants.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/match/align/E_Align.hh>
#include <epigraft/match/align/S_Align.hh>
#include <epigraft/match/align/SS_Align.hh>
#include <epigraft/match/align/N2C_C_Align.hh>
#include <epigraft/match/align/N2C_CA_Align.hh>
#include <epigraft/match/align/N2C_N_Align.hh>
#include <epigraft/match/align/C2N_C_Align.hh>
#include <epigraft/match/align/C2N_CA_Align.hh>
#include <epigraft/match/align/C2N_N_Align.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/GraftOptions.hh>

// rosetta headers
#include <files_paths.h>
#include <pose.h>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// utility headers
#include <utility/vector1.hh>
#include <utility/pointer/owning_ptr.hh>

// C++ headers
#include <set>


namespace epigraft {
namespace match {

/// @brief   Takes a set of matches and rescore with respect to their existing internal information.
/// @details Running this routine on existing match results is necessary if the input does not
/// @deatils necessarily have all information, e.g. regeneration of transformation matrix.
/// @details If input_pdb_has_Ab is true in options, then the rigid body orientation of the Ab-loop is
/// @details taken from the Ab in the input pdb.
// TODO: handle dihedrals?  currently they are skipped (existing dihedrals in match components are untouched as well)
// TODO: clean up in terms of passing in user specified superposition residues?
void
rescore_matches(
	GraftOptions const & options,
	AntibodyComplex & native,
	Pose & scaffold,
	utility::vector1< MatchResult > & match_results,
	std::string const & scaffold_filename,
	std::set< Integer > const & user_specified_superposition_residues
)
{
	using namespace epigraft::match::align;

	Real MAX_CLOSURE_RMS = MATCH_INFINITY;

	// load alignment systems -- std::map provides cleaner implementation and reasonable performance on average
	//                           as opposed to a series of 'if' statements.  In naive tests 'if' wins when the
	//                           number of of key strings is small ( <= 4? ) or the sequence of strings to be
	//                           looked up is extremely repetitive, otherwise std::map performs as well or better.
	std::map< std::string, AlignmentSystemOP > alignment_systems;

	// TODO: see if AlignmentSystem::name() can be made static virtual

	// "E" system
	alignment_systems[ "E" ] = AlignmentSystemOP( new E_Align( MAX_CLOSURE_RMS ) );

	// "S" system
	alignment_systems[ "S" ] = AlignmentSystemOP( new S_Align( AlignmentSystem::RMS, MAX_CLOSURE_RMS ) );

	// "SS" system
	alignment_systems[ "SS" ] = AlignmentSystemOP( new SS_Align( AlignmentSystem::RMS, MAX_CLOSURE_RMS, user_specified_superposition_residues ) );

	// "N2C" CA-centered system
	alignment_systems[ "N2C_CA" ] = AlignmentSystemOP( new N2C_CA_Align( MAX_CLOSURE_RMS ) );

	// "N2C" N-centered system
	alignment_systems[ "N2C_N" ] = AlignmentSystemOP( new N2C_N_Align( MAX_CLOSURE_RMS ) );

	// "N2C" C-centered system
	alignment_systems[ "N2C_C" ] = AlignmentSystemOP( new N2C_C_Align( MAX_CLOSURE_RMS ) );

	// "C2N" CA-centered system
	alignment_systems[ "C2N_CA" ] = AlignmentSystemOP( new C2N_CA_Align( MAX_CLOSURE_RMS ) );

	// "C2N" N-centered system
	alignment_systems[ "C2N_N" ] = AlignmentSystemOP( new C2N_N_Align( MAX_CLOSURE_RMS ) );

	// "C2N" C-centered system
	alignment_systems[ "C2N_C" ] = AlignmentSystemOP( new C2N_C_Align( MAX_CLOSURE_RMS ) );


	// scratch arrays
	FArray3D_float aligned_Ab_full_coord;
	if ( native.Ab_exists() ) {
		aligned_Ab_full_coord.dimension( native.Ab().full_coord() );
	}

	FArray3D_float const & scaffold_full_coord = scaffold.full_coord(); // for performance

	// bounding box and Octree for clash checks
	rootstock::BoundingBox bb = bounding_box( scaffold );


	for ( utility::vector1< MatchResult >::iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {
		MatchResult & match_result = *m;

		// check lower right corner of transformation matrix to see if there is an existing rb orientation
		if ( match_result.transformation_matrix[ 15 ] != 1.0 ) { // no orientation information exists, reconstruct everything from AlignmentSystem
			// grab alignment system
			AlignmentSystem & alignment_system = *( (*alignment_systems.find( match_result.system_name )).second );

			// create trimmed loop (epitope)
			Pose loop;
			native.trimmed_antigen( loop, match_result.components[ 1 ].loop_subrange );

			// fill primary component info using AlignmentSystem::check_alignment()
			alignment_system.check_alignment( scaffold, loop, match_result.components[ 1 ], match_result.transformation_matrix );
		}

		// obtain rigid body orientation from input pdb Ab
		if ( native.Ab_exists() && options.input_pdb_has_Ab ) {
			// read input pdb scaffold + Ab
			AntibodyComplex input_pdb_ac( files_paths::start_path + scaffold_filename, options.nres_Ab, options.Ab_first, false );

			// cache
			FArray3D_float const & native_Ab_full_coord = native.Ab().full_coord();
			FArray3D_float const & input_pdb_Ab_full_coord = input_pdb_ac.Ab().full_coord();

			// find transformation from native complex Ab to input pdb Ab
			get_GL_matrix( native_Ab_full_coord.a( 1, 1, 1 ), native_Ab_full_coord.a( 1, 1, 2 ), native_Ab_full_coord.a( 1, 1, 3 ),
			               input_pdb_Ab_full_coord.a( 1, 1, 1 ), input_pdb_Ab_full_coord.a( 1, 1, 2 ), input_pdb_Ab_full_coord.a( 1, 1, 3 ),
			               match_result.transformation_matrix ); // native_Ab moves onto input_pdb_Ab
		}

		// initial aligned epitope -- positioned with primary loop
		Pose aligned_epitope;
		aligned_epitope = native.antigen();
		aligned_epitope.transform_GL( match_result.transformation_matrix );

		// recompute match filters if rigid body orientation taken from input pdb Ab
		if ( options.input_pdb_has_Ab ) {
			MatchComponent & primary_component = match_result.components[ 1 ];
			compute_match_filters( scaffold_full_coord, primary_component.scaffold_gap_range, aligned_epitope.full_coord(), primary_component.loop_subrange, primary_component );
		}

		// aligned Ab -- positioned with primary loop
		if ( native.Ab_exists() ) {
			transform_full_coord_via_GL( match_result.transformation_matrix, native.Ab(), aligned_Ab_full_coord );
		}

		// create gapped scaffold (remove all loop match regions) and octree for inter-clash checks
		utility::vector1< bool > keep( scaffold.total_residue(), true ); // keep the following residues but remove those in all match ranges
		for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
			for ( Integer residue = match_result.components[ c ].scaffold_gap_range.begin(), last_residue = match_result.components[ c ].scaffold_gap_range.end(); residue <= last_residue; ++residue ) {
				keep[ residue ] = false;
			}
		}
		Pose gapped_scaffold;
		fragment_Pose( scaffold, keep, gapped_scaffold );
		rootstock::Octree< AtomPoint > octree_gapped( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
		fill_octree( octree_gapped, gapped_scaffold );

		// compute inter-clash
		match_result.inter_clash = 0.0;
		if ( native.Ab_exists() ) {
			match_result.inter_clash = octree_inter_rep( octree_gapped, gapped_scaffold, native.Ab(), aligned_Ab_full_coord, false ); // boolean: exclude terminal bb
		}

		// create vector of loop Poses for intra-clash checking
		// TODO: potentially expensive here, speed this up?
		utility::vector1< Pose * > aligned_loops; // remember to delete these manually, Pose is currently incompatible with smart pointers
		for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
			Pose * aligned_loop = new Pose();
			extract_segment_from_Pose( aligned_epitope, match_result.components[ c ].loop_subrange, *aligned_loop );
			aligned_loops.push_back( aligned_loop );
		}

		// now run through each component and compute intra clash and filter information
		for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
			MatchComponent & match_component = match_result.components[ c ];
			Pose & aligned_loop = *aligned_loops[ c ];

			// intra clash
			match_component.intra_clash = octree_inter_rep( octree_gapped, gapped_scaffold, aligned_loop, true ); // boolean: exclude terminal bb

			if ( c > 1 ) { // secondary components only, primary component handled below
				compute_match_filters( scaffold_full_coord, match_component.scaffold_gap_range, aligned_loop.full_coord(), ResidueRange( 1, aligned_loop.total_residue() ), match_component );
			} else {
				switch ( match_result.system_type ) {
					case AlignmentSystem::N2C:
						compute_c_terminal_match_filters( scaffold_full_coord, match_component.scaffold_gap_range.end(), aligned_loop.full_coord(), aligned_loop.total_residue(), match_component );
						break;
					case AlignmentSystem::C2N:
						compute_n_terminal_match_filters( scaffold_full_coord, match_component.scaffold_gap_range.begin(), aligned_loop.full_coord(), 1, match_component );
						break;
					default:
						compute_match_filters( scaffold_full_coord, match_component.scaffold_gap_range, aligned_loop.full_coord(), ResidueRange( 1, aligned_loop.total_residue() ), match_component );
						break;
				}
			}

		} // foreach component

		// memory management
		for ( utility::vector1< Pose * >::iterator loop = aligned_loops.begin(), last_loop = aligned_loops.end(); loop != last_loop; ++loop ) {
			delete *loop;
		}

	} // foreach match result
}


/// @brief Take a match and rescore it with respect to its existing internal information.
/// @note  If rescoring a set of matches, use rescore_matches() instead, as it's more efficient.
void
rescore_match(
	GraftOptions const & options,
	AntibodyComplex & native,
	Pose & scaffold,
	MatchResult & match_result,
	std::string const & scaffold_filename,
	std::set< Integer > const & user_specified_superposition_residues
)
{
	using namespace epigraft::match::align;

	Real MAX_CLOSURE_RMS = MATCH_INFINITY;

	// load alignment systems -- std::map provides cleaner implementation and reasonable performance on average
	//                           as opposed to a series of 'if' statements.  In naive tests 'if' wins when the
	//                           number of of key strings is small ( <= 4? ) or the sequence of strings to be
	//                           looked up is extremely repetitive, otherwise std::map performs as well or better.
	std::map< std::string, AlignmentSystemOP > alignment_systems;

	// TODO: see if AlignmentSystem::name() can be made static virtual

	// "E" system
	alignment_systems[ "E" ] = AlignmentSystemOP( new E_Align( MAX_CLOSURE_RMS ) );

	// "S" system
	alignment_systems[ "S" ] = AlignmentSystemOP( new S_Align( AlignmentSystem::RMS, MAX_CLOSURE_RMS ) );

	// "SS" system
	alignment_systems[ "SS" ] = AlignmentSystemOP( new SS_Align( AlignmentSystem::RMS, MAX_CLOSURE_RMS, user_specified_superposition_residues ) );

	// "N2C" CA-centered system
	alignment_systems[ "N2C_CA" ] = AlignmentSystemOP( new N2C_CA_Align( MAX_CLOSURE_RMS ) );

	// "N2C" N-centered system
	alignment_systems[ "N2C_N" ] = AlignmentSystemOP( new N2C_N_Align( MAX_CLOSURE_RMS ) );

	// "N2C" C-centered system
	alignment_systems[ "N2C_C" ] = AlignmentSystemOP( new N2C_C_Align( MAX_CLOSURE_RMS ) );

	// "C2N" CA-centered system
	alignment_systems[ "C2N_CA" ] = AlignmentSystemOP( new C2N_CA_Align( MAX_CLOSURE_RMS ) );

	// "C2N" N-centered system
	alignment_systems[ "C2N_N" ] = AlignmentSystemOP( new C2N_N_Align( MAX_CLOSURE_RMS ) );

	// "C2N" C-centered system
	alignment_systems[ "C2N_C" ] = AlignmentSystemOP( new C2N_C_Align( MAX_CLOSURE_RMS ) );


	// scratch arrays
	FArray3D_float aligned_Ab_full_coord;
	if ( native.Ab_exists() ) {
		aligned_Ab_full_coord.dimension( native.Ab().full_coord() );
	}

	FArray3D_float const & scaffold_full_coord = scaffold.full_coord(); // for performance

	// bounding box and Octree for clash checks
	rootstock::BoundingBox bb = bounding_box( scaffold );


	// check lower right corner of transformation matrix to see if there is an existing rb orientation
	if ( match_result.transformation_matrix[ 15 ] != 1.0 ) { // no orientation information exists, reconstruct everything from AlignmentSystem
		// grab alignment system
		AlignmentSystem & alignment_system = *( (*alignment_systems.find( match_result.system_name )).second );

		// create trimmed loop (epitope)
		Pose loop;
		native.trimmed_antigen( loop, match_result.components[ 1 ].loop_subrange );

		// fill primary component info using AlignmentSystem::check_alignment()
		alignment_system.check_alignment( scaffold, loop, match_result.components[ 1 ], match_result.transformation_matrix );
	}

	// obtain rigid body orientation from input pdb Ab
	if ( native.Ab_exists() && options.input_pdb_has_Ab ) {
		// read input pdb scaffold + Ab
		AntibodyComplex input_pdb_ac( files_paths::start_path + scaffold_filename, options.nres_Ab, options.Ab_first, false );

		// cache
		FArray3D_float const & native_Ab_full_coord = native.Ab().full_coord();
		FArray3D_float const & input_pdb_Ab_full_coord = input_pdb_ac.Ab().full_coord();

		// find transformation from native complex Ab to input pdb Ab
		get_GL_matrix( native_Ab_full_coord.a( 1, 1, 1 ), native_Ab_full_coord.a( 1, 1, 2 ), native_Ab_full_coord.a( 1, 1, 3 ),
		               input_pdb_Ab_full_coord.a( 1, 1, 1 ), input_pdb_Ab_full_coord.a( 1, 1, 2 ), input_pdb_Ab_full_coord.a( 1, 1, 3 ),
		               match_result.transformation_matrix ); // native_Ab moves onto input_pdb_Ab
	}

	// initial aligned epitope -- positioned with primary loop
	Pose aligned_epitope;
	aligned_epitope = native.antigen();
	aligned_epitope.transform_GL( match_result.transformation_matrix );

	// recompute match filters if rigid body orientation taken from input pdb Ab
	if ( options.input_pdb_has_Ab ) {
		MatchComponent & primary_component = match_result.components[ 1 ];
		compute_match_filters( scaffold_full_coord, primary_component.scaffold_gap_range, aligned_epitope.full_coord(), primary_component.loop_subrange, primary_component );
	}

	// aligned Ab -- positioned with primary loop
	if ( native.Ab_exists() ) {
		transform_full_coord_via_GL( match_result.transformation_matrix, native.Ab(), aligned_Ab_full_coord );
	}

	// create gapped scaffold (remove all loop match regions) and octree for inter-clash checks
	utility::vector1< bool > keep( scaffold.total_residue(), true ); // keep the following residues but remove those in all match ranges
	for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
		for ( Integer residue = match_result.components[ c ].scaffold_gap_range.begin(), last_residue = match_result.components[ c ].scaffold_gap_range.end(); residue <= last_residue; ++residue ) {
			keep[ residue ] = false;
		}
	}
	Pose gapped_scaffold;
	fragment_Pose( scaffold, keep, gapped_scaffold );
	rootstock::Octree< AtomPoint > octree_gapped( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
	fill_octree( octree_gapped, gapped_scaffold );

	// compute inter-clash
	match_result.inter_clash = 0.0;
	if ( native.Ab_exists() ) {
		match_result.inter_clash = octree_inter_rep( octree_gapped, gapped_scaffold, native.Ab(), aligned_Ab_full_coord, false ); // boolean: exclude terminal bb
	}

	// create vector of loop Poses for intra-clash checking
	// TODO: potentially expensive here, speed this up?
	utility::vector1< Pose * > aligned_loops; // remember to delete these manually, Pose is currently incompatible with smart pointers
	for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
		Pose * aligned_loop = new Pose();
		extract_segment_from_Pose( aligned_epitope, match_result.components[ c ].loop_subrange, *aligned_loop );
		aligned_loops.push_back( aligned_loop );
	}

	// now run through each component and compute intra clash and filter information
	for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
		MatchComponent & match_component = match_result.components[ c ];
		Pose & aligned_loop = *aligned_loops[ c ];

		// intra clash
		match_component.intra_clash = octree_inter_rep( octree_gapped, gapped_scaffold, aligned_loop, true ); // boolean: exclude terminal bb

		if ( c > 1 ) { // secondary components only, primary component handled below
			compute_match_filters( scaffold_full_coord, match_component.scaffold_gap_range, aligned_loop.full_coord(), ResidueRange( 1, aligned_loop.total_residue() ), match_component );
		} else {
			switch ( match_result.system_type ) {
				case AlignmentSystem::N2C:
					compute_c_terminal_match_filters( scaffold_full_coord, match_component.scaffold_gap_range.end(), aligned_loop.full_coord(), aligned_loop.total_residue(), match_component );
					break;
				case AlignmentSystem::C2N:
					compute_n_terminal_match_filters( scaffold_full_coord, match_component.scaffold_gap_range.begin(), aligned_loop.full_coord(), 1, match_component );
					break;
				default:
					compute_match_filters( scaffold_full_coord, match_component.scaffold_gap_range, aligned_loop.full_coord(), ResidueRange( 1, aligned_loop.total_residue() ), match_component );
					break;
			}
		}

	} // foreach component

	// memory management
	for ( utility::vector1< Pose * >::iterator loop = aligned_loops.begin(), last_loop = aligned_loops.end(); loop != last_loop; ++loop ) {
		delete *loop;
	}

}


} // namespace match
} // namespace epigraft
