// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   try_fluid_ends.cc
/// @brief  functions for attempting recover/optimization by fluidizing endpoints
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// unit headers
#include <epigraft/match/try_fluid_ends.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/conformation/FluidLanding.hh>
#include <epigraft/conformation/FluidLandingResult.hh>
#include <epigraft/conformation/FluidTakeoff.hh>
#include <epigraft/conformation/FluidTakeoffResult.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/epigraft_functions.hh>

// utility headers
#include <utility/vector1.hh>

// C++ headers
#include <iostream>


namespace epigraft {
namespace match {


/// @brief attempt recovery/optimization of *primary* match by fluidizing takeoff terminus (for single break only)
void
try_fluidizing_takeoff(
	GraftOptions const & options,
	AntibodyComplex & native,
	Pose & scaffold,
	utility::vector1< MatchResult > & match_results
)
{
	using namespace epigraft::conformation;
	using epigraft::match::align::AlignmentSystem;

	// options here for clarity
	Real const DIHEDRAL_LEFT = options.dihedral_deviation;
	Real const DIHEDRAL_RIGHT = options.dihedral_deviation;
	Real const DIHEDRAL_STEP = options.dihedral_step;
	Real const INTRA_CLASH_INCREASE = options.allowed_intra_clash_increase;
	Real const MAX_CLOSURE_RMS = options.max_closure_rms + options.recovery_rms_epsilon; // epsilon added here because additional optimizations can be performed (e.g. takeoff fluidization)

	// construct octree around scaffold, for constructing gapped scaffold octree later
	rootstock::BoundingBox bb = bounding_box( scaffold );

	// run through all matches and attempt dihedral movement
	for ( utility::vector1< MatchResult >::iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {
		MatchResult & match_result = *m;

		switch ( match_result.system_type ) {
			case AlignmentSystem::SUPERPOSITION:
				continue; // skip superposition match
			case AlignmentSystem::ENDPOINT:
				continue; // skip endpoint match
			case AlignmentSystem::SPECIFIC_SUPERPOSITION:
				continue; // skip specific superposition match
			default:
				break; // case ok, fall through to below
		}

		// component
		MatchComponent & primary = match_result.components[ 1 ];

		// ranges
		ResidueRange const & loop_subrange = primary.loop_subrange;
		ResidueRange const & match_range = primary.scaffold_gap_range;

		// reconstruct aligned loop
		Pose loop;
		native.trimmed_antigen( loop, loop_subrange );
		Pose aligned_loop;
		aligned_loop = loop;
		aligned_loop.transform_GL( match_result.transformation_matrix );

		// setup
		FluidTakeoff takeoff( scaffold, match_range, aligned_loop, DIHEDRAL_LEFT, DIHEDRAL_RIGHT, DIHEDRAL_STEP );
		utility::vector1< FluidTakeoffResult > fluid_results;

		// try hinging
		Real original_rms;
		switch ( match_result.system_type ) {
			case AlignmentSystem::N2C:
				takeoff.fluidize_n_terminus( MAX_CLOSURE_RMS, fluid_results );
				original_rms = primary.c_terminal_rms; // save for later
				break;
			case AlignmentSystem::C2N:
				takeoff.fluidize_c_terminus( MAX_CLOSURE_RMS, fluid_results );
				original_rms = primary.n_terminal_rms; // save for later
				break;
			default:
				std::cerr << "try_fluidizing_takeoff(): should not be here! given match result type not handled here, so continuing on" << std::endl;
				continue;
		}

		// get best dihedrals, check against original chainbreak rms, which is stored above
		Integer best_index = -1;
		Real best_rms = original_rms;
		for ( Integer h = 1, he = fluid_results.size(); h <= he; ++h ) {
			FluidTakeoffResult const & result = fluid_results[ h ];
			if ( result.rms() < best_rms ) {
				best_index = h;
				best_rms = result.rms();
			}
		}

		// if there's an improvement, try clash checking
		if ( best_index > -1 ) {
			// gapped scaffold and octree for lash check
			Pose gapped_scaffold;
			remove_segment_from_Pose( scaffold, match_range, gapped_scaffold );
			rootstock::Octree< AtomPoint > octree_gapped( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
			fill_octree( octree_gapped, gapped_scaffold );

			// intra-clash check with re-aligned loop
			aligned_loop.transform_GL( fluid_results[ best_index ].transformation_matrix() ); // can re-use old aligned_loop here so use Pose method for transform
			Real intra_clash = octree_inter_rep( octree_gapped, gapped_scaffold, aligned_loop, true ); // boolean: exclude terminal bb

			if ( intra_clash - primary.intra_clash < INTRA_CLASH_INCREASE ) {

				Real inter_clash = 0.0;
				FArray2D_float RT( 4, 4 ); // new transformation matrix since we need to orient the Ab from its original position
				compose_GL( fluid_results[ best_index ].transformation_matrix(), match_result.transformation_matrix, RT );

				if ( native.Ab_exists() ) {
					// inter-clash check with re-aligned antibody
					Pose aligned_Ab;
					aligned_Ab = native.Ab();
					aligned_Ab.transform_GL( RT );
					inter_clash = octree_inter_rep( octree_gapped, gapped_scaffold, aligned_Ab, false ); // boolean: exclude terminal bb
				}

				if ( inter_clash < match_result.inter_clash ) { // inter-clash has decreased, keep this conformation
					// save dihedrals in match result
					FluidTakeoffResult best_fluid_result = fluid_results[ best_index ];
					primary.dihedrals.insert( best_fluid_result.dihedrals().begin(), best_fluid_result.dihedrals().end() );

					// save new rb orientation w/ respect to fluidized takeoff
					match_result.transformation_matrix = RT;

					// record data
					switch ( match_result.system_type ) {
						case AlignmentSystem::N2C:
							primary.c_terminal_rms = best_fluid_result.rms();
							primary.c_closure_angle = best_fluid_result.closure_angle();
							break;
						case AlignmentSystem::C2N:
							primary.n_terminal_rms = best_fluid_result.rms();
							primary.n_closure_angle = best_fluid_result.closure_angle();
							break;
						default:
							std::cerr << "try_fluidizing_takeoff(): should not be here! given match result type not handled here, so continuing on" << std::endl;
							continue;
					}
				}
			}

		} // if rms improved by fluidizing

	} // foreach match result
}


/// @brief attempt recovery/optimization of all breaks by fluidizing landing termini (single and double breaks)
void
try_fluidizing_landing(
	GraftOptions const & options,
	AntibodyComplex & native,
	Pose & scaffold,
	utility::vector1< MatchResult > & match_results,
	Integer const & skip_component
)
{
	using namespace epigraft::conformation;
	using epigraft::match::align::AlignmentSystem;

	// options here for clarity
	Real const DIHEDRAL_LEFT = options.dihedral_deviation;
	Real const DIHEDRAL_RIGHT = options.dihedral_deviation;
	Real const DIHEDRAL_STEP = options.dihedral_step;
	Real MAX_CLOSURE_RMS = options.max_closure_rms; // use strict rms, as landing fluidization is the endgame
	if ( skip_component > 0 && options.rough_match ) { // primary component is being skipped implying this is a multi-match; use rough match closure rms instead
		MAX_CLOSURE_RMS = options.rough_match_closure_rms; // use strict rms, as landing fluidization is the endgame
	}

	// run through all matches
	for ( utility::vector1< MatchResult >::iterator m = match_results.begin(), me = match_results.end(); m != me; ++m ) {
		MatchResult & match_result = *m;

		// reconstruct aligned epitope (all loops)
		Pose aligned_epitope;
		aligned_epitope = native.antigen();
		aligned_epitope.transform_GL( match_result.transformation_matrix );

		// fluid result containers
		utility::vector1< FluidLandingResult > nt_fluid_results;
		utility::vector1< FluidLandingResult > ct_fluid_results;

		// run through all components
		for ( utility::vector1< MatchComponent >::iterator c = match_result.components.begin() + skip_component, ce = match_result.components.end(); c < ce; ++c ) { // iterate over vector, so '<' is ok
			MatchComponent & component = *c;

			FluidLanding landing( scaffold, component.scaffold_gap_range, aligned_epitope, component.loop_subrange, DIHEDRAL_LEFT, DIHEDRAL_RIGHT, DIHEDRAL_STEP );

			// TODO: it seems like there may be a visible performance hit here, investigate
			//       and consider ways to remove the 'if' branch for primary/secondaries
			if ( c == match_result.components.begin() ) { // handle primary
				switch ( match_result.system_type ) {
					case AlignmentSystem::N2C:
						landing.fluidize_c_terminus( MAX_CLOSURE_RMS, ct_fluid_results );
						break;
					case AlignmentSystem::C2N:
						landing.fluidize_n_terminus( MAX_CLOSURE_RMS, nt_fluid_results );
						break;
					default: // ENDPOINT, SUPERPOSITION, and SPECIFIC_SUPERPOSITION
						landing.fluidize_n_terminus( MAX_CLOSURE_RMS, nt_fluid_results );
						landing.fluidize_c_terminus( MAX_CLOSURE_RMS, ct_fluid_results );
				}
			} else { // handle secondaries: always double break
				landing.fluidize_n_terminus( MAX_CLOSURE_RMS, nt_fluid_results );
				landing.fluidize_c_terminus( MAX_CLOSURE_RMS, ct_fluid_results );
			}

			if ( nt_fluid_results.size() > 0 ) {
				Integer best_index = index_of_best_fluid_result( nt_fluid_results, component.n_terminal_rms );
				if ( best_index > -1 ) {
					FluidLandingResult const & best_fluid_result = nt_fluid_results[ best_index ];
					component.dihedrals.insert( best_fluid_result.dihedrals().begin(), best_fluid_result.dihedrals().end() );

					// record data
					component.n_terminal_rms = best_fluid_result.rms();
					component.n_closure_angle = best_fluid_result.closure_angle();
				}

				// clear container
				nt_fluid_results.clear();
			}

			if ( ct_fluid_results.size() > 0 ) {
				Integer best_index = index_of_best_fluid_result( ct_fluid_results, component.c_terminal_rms );
				if ( best_index > -1 ) {
					FluidLandingResult const & best_fluid = ct_fluid_results[ best_index ];
					component.dihedrals.insert( best_fluid.dihedrals().begin(), best_fluid.dihedrals().end() );

					// record data
					component.c_terminal_rms = best_fluid.rms();
					component.c_closure_angle = best_fluid.closure_angle();
				}

				// clear container
				ct_fluid_results.clear();
			}

			// clear containers
		} // foreach match component

	} // foreach match result

}


/// @brief get best fluid result w/ respect to rms
Integer
index_of_best_fluid_result(
	utility::vector1< epigraft::conformation::FluidLandingResult > const & results,
	Real const & original_rms
)
{
	Integer best_index = -1;
	Real best_rms = original_rms;
	for ( Integer f = 1, fe = results.size(); f <= fe; ++f ) {
		epigraft::conformation::FluidLandingResult const & result = results[ f ];
		if ( result.rms() < best_rms ) {
			best_index = f;
			best_rms = result.rms();
		}
	}

	return best_index;
}


} // match
} // epigraft
