// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   try_rb_move.cc
/// @brief  functions for attempting recover/optimization by rb-move
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// unit headers
#include <epigraft/match/try_rb_move.hh>

// package headers
#include <epigraft/match/match_types.hh>
#include <epigraft/match/match_constants.hh>
#include <epigraft/match/align/AlignmentSystem.hh>
#include <epigraft/match/MatchComponent.hh>
#include <epigraft/match/MatchResult.hh>
#include <epigraft/match/match_functions.hh>
#include <epigraft/conformation/TransformGenerator.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/AtomPoint.hh>
#include <epigraft/GraftOptions.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/epigraft_functions.hh>

// rootstock headers
#include <rootstock/BoundingBox.hh>
#include <rootstock/Octree.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// numeric headers
#include <numeric/xyzVector.hh>

// utility headers
#include <utility/vector1.hh>


namespace epigraft {
namespace match {


/// @brief attempt recovery/optimization by rb-move, this operates on the primary loop
/// @note  all matches found here are designated 'ENDPOINT' (E) matches
/// @note  original matches are excluded in rb_results list
void
try_rb_move(
	GraftOptions const & options,
	AntibodyComplex & native,
	Pose & scaffold,
	utility::vector1< MatchResult > const & match_results,
	utility::vector1< MatchResult > & rb_results
)
{
	using epigraft::match::align::AlignmentSystem;

	// typedefs for convenience
	typedef utility::vector1< FArray2D_float > Transforms;

	// first check and see if Ab is being used, if not then exit as try_rb_move
	// is currently not supported without Ab
	if ( !native.Ab_exists() ) {
		std::cout << "Epi-Graft Match: WARNING -- skipping try_rb_move, as there is no Ab for reference!" << std::endl;
		return;
	}

	Real const INTRA_CLASH_INCREASE = options.allowed_intra_clash_increase;
	Real const CUBE_SIDE = options.rb_cube_side_length;
	Real const CUBE_STEP = options.rb_translation_step;
	Real const ROTATION_DEVIATION = options.rb_angle_deviation;
	Real const ROTATION_STEP = options.rb_angle_step;

	FArray2D_float scratch( 4, 4, 0.0 ); // scratch transformation matrix
	FArray2D_float RT( 4, 4, 0.0 ); // working transformation matrix (to be applied to coordinates)

	FArray3D_float Ab_transformed_fc; // scratch Ab full_coord array
	Ab_transformed_fc.dimension( native.Ab().full_coord() );

	FArray3D_float loop_transformed_fc; // scratch loop full_coord for intra-clash checking

	FArray3D_float const & scaffold_full_coord = scaffold.full_coord(); // for performance

	// construct octree around scaffold, for constructing gapped scaffold octree later
	rootstock::BoundingBox bb = bounding_box( scaffold );

	// reference (before alignment) for computing z-axis vector for transforms
	FArray2D_float reference_crd_before_alignment = ca_centroid( native.Ab() );

	for ( Integer m = 1, me = match_results.size(); m <= me; ++m ) {
		MatchResult const & match_result = match_results[ m ];
		MatchComponent const & primary = match_result.components[ 1 ];

		// initial aligned epitope -- positioned with primary loop
		Pose aligned_epitope;
		aligned_epitope = native.antigen();
		aligned_epitope.transform_GL( match_result.transformation_matrix );

		// initial aligned Ab -- positioned with primary loop
		Pose aligned_Ab;
		aligned_Ab = native.Ab();
		aligned_Ab.transform_GL( match_result.transformation_matrix );

		// grab points for computing x- and z-axis vectors
		FArray3D_float const & ae_fc = aligned_epitope.full_coord_simple_return_packer();
		Point const first_CA( &ae_fc( 1, 2, primary.loop_subrange.begin() ) ); // use 'CA'
		Point const second_CA( &ae_fc( 1, 2, primary.loop_subrange.end() ) ); // use 'CA'
		Point const origin( midpoint( first_CA, second_CA ) ); // origin of coordinate frame around aligned loop

		FArray2D_float reference_crd = reference_crd_before_alignment;
		transform_crd_via_GL( match_result.transformation_matrix, reference_crd );
		Point const reference( &reference_crd[ 0 ] ); // aligned reference towards Ab for computing z-axis

		// compute x- and z-axis vectors
		Vector const x_axis( second_CA - first_CA );
		Vector const z_axis( perpendicular_foot( reference, first_CA, second_CA ) - reference );

		// create transform generator
		epigraft::conformation::TransformGenerator< Real > const gen( origin, x_axis, z_axis );

		// generate translations
		Transforms translations;
		gen.push_translate_cube( CUBE_SIDE, CUBE_STEP, translations );

		// generate rotations -- first pass: identity (this is split out so we don't generate from -ROT to +ROT and triple count);
		// TODO: Maybe do the following: it would be possible to generate a sequence of continuous
		//       transformations to do this instead of relying on transforming a fixed frame each time.
		//       This would speed up rotation generation, because only 2 matrices for each rotation would
		//       be necessary, RT( rotation_deviation ) for resetting and RT( rotation_step ) for moving.
		//       If this is going to be stored in a single container, remember to not multiple count
		//       due to the need for resetting to the original position.
		Transforms rotations;
		gen.identity( RT );
		rotations.push_back( RT );

		// generate rotations -- second pass: negative angles
		for ( Real angle = -ROTATION_DEVIATION; angle < 0; angle += ROTATION_STEP ) {
			gen.rotate_x_degrees( angle, RT );
			rotations.push_back( RT );
			gen.rotate_y_degrees( angle, RT );
			rotations.push_back( RT );
			gen.rotate_z_degrees( angle, RT );
			rotations.push_back( RT );
		}

		// generate rotations -- third pass: positive angles
		for ( Real angle = ROTATION_STEP; angle <= ROTATION_DEVIATION; angle += ROTATION_STEP ) {
			gen.rotate_x_degrees( angle, RT );
			rotations.push_back( RT );
			gen.rotate_y_degrees( angle, RT );
			rotations.push_back( RT );
			gen.rotate_z_degrees( angle, RT );
			rotations.push_back( RT );
		}

		// create gapped scaffold (remove all loop match regions) and octree for inter-clash checks
		utility::vector1< bool > keep( scaffold.total_residue(), true ); // keep the following residues but remove those in all match ranges
		for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
			for ( Integer residue = match_result.components[ c ].scaffold_gap_range.begin(), last_residue = match_result.components[ c ].scaffold_gap_range.end(); residue <= last_residue; ++residue ) {
				keep[ residue ] = false;
			}
		}
		Pose gapped_scaffold;
		fragment_Pose( scaffold, keep, gapped_scaffold );
		rootstock::Octree< AtomPoint > octree_gapped( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
		fill_octree( octree_gapped, gapped_scaffold );

		// create vector of loop Poses for intra-clash checking
		// TODO: potentially expensive here, speed this up?
		utility::vector1< Pose * > aligned_loops; // remember to delete these manually, Pose is currently incompatible with smart pointers
		for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
			Pose * aligned_loop = new Pose();
			extract_segment_from_Pose( aligned_epitope, match_result.components[ c ].loop_subrange, *aligned_loop );
			aligned_loops.push_back( aligned_loop );
		}

		// run through each translation
		for ( Transforms::const_iterator t = translations.begin(), te = translations.end(); t != te; ++t ) {
			FArray2D_float const & translation = *t;

			// try each rotation
			for ( Transforms::const_iterator r = rotations.begin(), re = rotations.end(); r != re; ++r ) {
				FArray2D_float const & rotation = *r;

				// compute transformation matrix
				compose_GL( rotation, translation, RT );

				// make new match result
				MatchResult rb_match_result( match_result );

				// run through each loop and intra-clash check
				bool all_intra_clashes_ok = true;
				for ( Integer c = 1, ce = match_result.components.size(); c <= ce; ++c ) {
					MatchComponent const & match_component = match_result.components[ c ];
					Pose & aligned_loop = *aligned_loops[ c ];

					loop_transformed_fc.dimension( aligned_loop.full_coord() );
					transform_full_coord_via_GL( RT, aligned_loop, loop_transformed_fc );
					Real intra_clash = octree_inter_rep( octree_gapped, gapped_scaffold, aligned_loop, loop_transformed_fc, true ); // boolean: exclude terminal bb

					if ( intra_clash - match_component.intra_clash > INTRA_CLASH_INCREASE ) {
						all_intra_clashes_ok = false;
						break; // one of the loops now clashes too much with scaffold, reject this rb-orientation below
					}

					// store new component data in new match result
					MatchComponent & rb_match_component = rb_match_result.components[ c ];
					rb_match_component.intra_clash = intra_clash;

					// new filters (rms, closure angle, etc)
					compute_match_filters( scaffold_full_coord, rb_match_component.scaffold_gap_range, loop_transformed_fc, ResidueRange( 1, aligned_loop.total_residue() ), rb_match_component );

					// unused data
					rb_match_component.overall_rms = MATCH_INFINITY;
				}

				// all intra-clashes ok? if so, inter-clash check
				if ( all_intra_clashes_ok ) {
					transform_full_coord_via_GL( RT, aligned_Ab, Ab_transformed_fc );
					Real inter_clash = octree_inter_rep( octree_gapped, gapped_scaffold, native.Ab(), Ab_transformed_fc, false ); // boolean: exclude terminal bb

					if ( inter_clash < match_result.inter_clash ) {
						// note that strict '<' above excludes the original match_result position

						// inter-clash has improved, save and report new transformation matrix
						compose_GL( RT, match_result.transformation_matrix, rb_match_result.transformation_matrix ); // compose to obtain complete transform
						rb_match_result.inter_clash = inter_clash;

						rb_match_result.system_type = AlignmentSystem::ENDPOINT;
						rb_match_result.system_name = "E";
						rb_match_result.overall_rms = MATCH_INFINITY; // postprocessing procedure takes care of this

						rb_results.push_back( rb_match_result );
					}
				}

			} // rotation
		} // translation

		// memory management
		for ( utility::vector1< Pose * >::iterator loop = aligned_loops.begin(), last_loop = aligned_loops.end(); loop != last_loop; ++loop ) {
			delete *loop;
		}
	}
}


/// @brief perpendicular foot
/// @note  Finds perpendicular foot along vector vo->ve for a triangle defined by two vectors vo->ve and vo->rp.
/// @note  Depending upon the angle between vo->ve and vo->rp, it might not be the perpendicular foot.
/// @note  If the angle is 90 degrees, then vo is returned.  If the angle is obtuse, then the point that drops
/// @note  down from rp to make a right angle is returned.
inline
Point
perpendicular_foot(
	Point const & rp,
	Point const & vo,
	Point const & ve
)
{
	Vector vo_ve = ve - vo; // vo -> ve
	Vector vo_rp = rp - vo; // vo -> rp

	Real rp_vo_ve = angle_of( vo_rp, vo_ve );

	if ( rp_vo_ve > numeric::constants::d::pi_over_2 ) { // obtuse triangle

		rp_vo_ve = numeric::constants::d::pi - rp_vo_ve;
		Angle vo_rp_u = numeric::constants::d::pi_over_2 - rp_vo_ve; // let u denote perpendicular foot

		Length scale = vo_rp.length() * std::sin( vo_rp_u ); // length of desired vector along vo -> ve
		vo_ve.normalize();
		return -scale * vo_ve + vo; // obtuse triangle needs reversal of direction

	} else if ( rp_vo_ve < numeric::constants::d::pi_over_2 ) { // acute triangle

		Angle vo_rp_u = numeric::constants::d::pi_over_2 - rp_vo_ve; // let u denote perpendicular foot

		Length scale = vo_rp.length() * std::sin( vo_rp_u ); // length of desired vector along vo -> ve
		vo_ve.normalize();
		return scale * vo_ve + vo;

	}

	return vo; // already right triangle
}


} // match
} // epigraft
