// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   epigraft_match.cc
/// @brief  Searches for graft locations on a scaffold.
/// @author Bill Schief (schief@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/epigraft_match.hh>

// package headers
#include <epigraft/epi_graft.h> // split away after refactoring
#include <epigraft/epigraft_functions.hh>
#include <epigraft/AtomPoint.hh>

// Rosetta headers
#include <aaproperties_pack.h>
#include <after_opts.h>
#include <disulfides.h> // cys_res_in_disulf
#include <files_paths.h>
#include <pose.h>
#include <pose_io.h>
#include <refold.h> // get_GL_matrix
#include <rms.h> // charlie rms routines
#include <runlevel.h>
#include <score.h>

// rootstock headers
#include <rootstock/Octree.hh>

// ObjexxFCL headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// utility headers
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ headers
#include <iostream>
#include <string>


// temporary globals
rootstock::Distance const OCTREE_CUBE_SIZE = 6.0;
rootstock::Distance const BUMP_DISTANCE = 1.5;

void
epi_graft_match()
{
	using namespace pose_ns;
	using namespace rootstock;

	// scorefxn
	Score_weight_map weight_map( score12 );

	// set basic parameters controlling pose behavior
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	bool const read_all_chains( true );
	bool const check_missing( false ); //use this for pose.set_coords throughout
	bool const coords_init( true ); //used in pose_from_misc

	// Before you read in pdbs, set disulf options automatically
	disulfides::options::find_disulf = true;
	disulfides::options::norepack_disulf = true;

	// Before you read in pdbs, set no_optH ( though need to update svn to get this functionality )

	//	files_paths::no_optH = truefalseoption("no_optH");

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	//options
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	// check specific scaffolds w/specific graft sites, graft ranges, directions, etc.
	// ( this is useful for clash_check to make it fast )
	bool check_specific_sites;
	check_specific_sites=truefalseoption("check_specific_sites");

  int max_loopsize_difference = 10;
  intafteroption( "max_loopsize_difference", max_loopsize_difference, max_loopsize_difference );


	// max_closure_rms for output
	float max_closure_rms = 6.0;
	realafteroption( "max_closure_rms", max_closure_rms, max_closure_rms );

	// Do you want clash check after rms measurement?
	bool clash_check = true;
	clash_check = truefalseoption("clash_check");

	// max # of intra_bumps (atoms within BUMP_DISTANCE)
	int max_intra_bumps = -1;
	//	int max_intra_bumps = 40;
	//	intafteroption( "max_intra_bumps", max_intra_bumps, max_intra_bumps );
	//	if ( max_intra_bumps < 0 ) {
	//		utility::exit( __FILE__, __LINE__, "-max_intra_bumps must be positive!" );
	//	}

	// intra clash between scaff_w_gap and graft: Using a loose threshold initially prior to closing the chainbreak(s)
	float max_intra_clash = 10000.; // needs to be tested
	realafteroption("max_intra_clash", max_intra_clash, max_intra_clash);

	// inter clash between Ab and scaff_w_graft_gly: Using a loose threshold initially prior to closing the chainbreak(s)
	float max_inter_clash = 30000.; // needs to be tested
	realafteroption("max_inter_clash", max_inter_clash, max_inter_clash);

	// Do the input 'scaff' pdbs include Ab+scaff ?
	bool input_pdbs_include_Ab = false;
	input_pdbs_include_Ab=truefalseoption("input_pdbs_include_Ab");

	// if input pdbs do include Ab, is the Ab first or second?
	bool scaff_first = true;
	bool Ab_first = false;
	if ( input_pdbs_include_Ab ) {
		if ( truefalseoption("Ab_first") ) {
			scaff_first = false;
			Ab_first =  true;
		}
	}


	// always needed
	int n_scaff = 0;
	FArray1D_string list_scaff_pdb( n_scaff );

	// optional for check_specific_sites
	FArray1D_string list_specific_graft_dir( n_scaff );
	FArray1D_string list_specific_graft_align_system( n_scaff );
	FArray2D_int list_specific_graft_resnums( 4, n_scaff );      //1,2 are graft_resnums in 1-x numbering, 3,4 are in native epitope numbering
	FArray2D_int list_specific_scaff_gap_resnums( 2, n_scaff ); //1,2 are scaff_gap_resnums in 1-nres_scaff numbering


	if ( check_specific_sites ) {
		read_list_scaff_info_for_epi_graft_design( list_specific_graft_dir,
												   list_specific_graft_align_system,
												   list_specific_graft_resnums,
												   list_specific_scaff_gap_resnums,
												   list_scaff_pdb
												 );
		n_scaff = list_scaff_pdb.size();
	} else {
		// Read in list of scaffold pdbs
		read_list_scaff_pdb( n_scaff, list_scaff_pdb );
	}


	//file to output scores
	std::string output_filename;
	stringafteroption( "output_file", "none", output_filename );
	if( output_filename == "none" ) {
		std::cout << "Need -output_file <filename> "
              << "on command-line " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// open the output file
	// ( leave it open since we write scores about 1/s or faster )
	utility::io::ozstream outfile;
	open_output_file( output_filename, outfile );

	// put header in output file
	outfile	<< SS("scaff")
					<< SS("   rng")
					<< SS(" dir ")
					<< SS(" sys")
					<< SS("epi1_nat epi2_nat")
					<< SS("epi1 epi2")
					<< SS("sca1 sca2")
					<< SS("    rms");
	if ( clash_check ) {
		outfile << SS(" bump_intra")
						<< SS(" bump_inter")
						<< SS(" farep_intra")
						<< SS(" farep_inter");
	}
	outfile << SS(" nres_scaff")
					<< SS(" scaff_filename")
					<< std::endl;

	// checkpoint file for running on clusters
	std::string checkpoint_filename;
	std::string checkpoint_file_w_path;
	stringafteroption( "checkpoint", "none", checkpoint_filename );
	bool using_checkpoint = false;
	if ( checkpoint_filename == "none"  && runlevel_ns::runlevel > runlevel_ns::quiet ) {
		std::cout << "no checkpoint_filename, NOT using checkpointing" << std::endl;
	} else {
		using_checkpoint = true;
		checkpoint_file_w_path = files_paths::score_path + checkpoint_filename;
	}

	// option to output pdbs oriented relative to nat_cmplx
	bool output_pdbs = false;
	output_pdbs = truefalseoption("output_pdbs");

	// read in the native complex of epitope-Ab
	Pose nat_cmplx_pose;
	std::string native_complex = stringafteroption("native_complex");
	std::string native_complex_w_path = files_paths::pdb_path_1 + native_complex;
	pose_from_pdb( nat_cmplx_pose, native_complex_w_path, fullatom, ideal_pose,
								 read_all_chains );

//	nat_cmplx_pose.dump_pdb( "nat_cmplx_pose.pdb" );

	// flip symmetric sidechains
	{
		float const start_score( nat_cmplx_pose.score( weight_map ) );
		nat_cmplx_pose.refold_sidechains_from_chi();
		if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
			std::cout << "score-delta from sidechain flip: " <<
				nat_cmplx_pose.score( weight_map ) - start_score << std::endl;
		}
	}

	int const nres_nat_cmplx( nat_cmplx_pose.total_residue() );
	int nres_Ab(0);

	//*** native complex is assumed to have Ab first, followed by epitope ***
	if ( runlevel_ns::runlevel > runlevel_ns::quiet ) {
		std::cout << "*** NOTE: native complex is assumed to have Ab first, followed by epitope ***" << std::endl;
	}
	// set length of antibody:
	// get from commandline
	intafteroption( "nres_Ab", 0, nres_Ab );
	if ( nres_Ab == 0 ) {
		std::cout << "must specify -nres_Ab <nres> on commandline" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// save nres for full-length peptide graft
	//
	int const nres_nat_pep = nres_nat_cmplx - nres_Ab;

	// Read in list of graft ranges ( only needed if not check_specific_sites )
	int n_graft_ranges = 0;
	FArray2D_int list_graft_ranges( 4, n_graft_ranges );
	if ( ! check_specific_sites ) {
		read_list_graft_ranges( n_graft_ranges, list_graft_ranges );

		// Error checking ( requires nres_nat_pep )
		for ( int i = 1; i <= n_graft_ranges; i++ ) {
			if ( list_graft_ranges( 1, i ) > nres_nat_pep || list_graft_ranges( 2, i ) > nres_nat_pep) {
				std::cout << "ERROR list_graft_range low or high  > nres_nat_pep "
									<< list_graft_ranges( 1, i ) << ' ' << list_graft_ranges( 2, i )
									<< nres_nat_pep << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
	}



	// make a pose that is Ab only and score it
	Pose Ab_pose;
	Ab_pose.simple_fold_tree( nres_Ab );
	Ab_pose.set_fullatom_flag( fullatom, false /*repack*/);
	// Ab_pose.copy_segment( nres_Ab, nat_cmplx_pose, 1, 1 /*begin_src*/);
	copy_segment_to_new_pose( nat_cmplx_pose /*pose_in*/, 1, nres_Ab, Ab_pose /*pose_out*/ );
	Ab_pose.score( weight_map );
	// Ab_pose.dump_pdb("Ab_pose.pdb");
	float const fa_rep_Ab ( Ab_pose.get_0D_score( FA_REP ) );

	// make a pose that is epitope only and score it
	Pose nat_pep_pose;
	nat_pep_pose.simple_fold_tree( nres_nat_pep );
	nat_pep_pose.set_fullatom_flag( fullatom, false /*repack*/);
	// nat_pep_pose.copy_segment( nres_nat_pep, nat_cmplx_pose, 1, nres_Ab+1 /*begin_src*/ );
	copy_segment_to_new_pose( nat_cmplx_pose, nres_Ab+1, nres_nat_cmplx, nat_pep_pose );
	nat_pep_pose.score( weight_map );
	float const fa_rep_nat_pep ( nat_pep_pose.get_0D_score( FA_REP ) );

	// make a pose with all-atom Ab and glycine for epitope
	Pose nat_cmplx_gly_pose;
	nat_cmplx_gly_pose = nat_cmplx_pose;
	int gly_begin = nres_Ab + 1;
	int gly_end = nres_nat_cmplx;
	convert_resnum_range_to_gly( nat_cmplx_gly_pose, gly_begin,  gly_end );
	nat_cmplx_gly_pose.score( weight_map );
	float const fa_rep_nat_complex_gly ( nat_cmplx_gly_pose.get_0D_score( FA_REP ) );

	// nat_cmplx_gly_pose.dump_pdb( "nat_cmplx_gly_pose.pdb");

	// make a pose that is just the native epitope (peptide) all-gly
	Pose nat_pep_gly_pose;
	nat_pep_gly_pose.simple_fold_tree( nres_nat_pep );
	nat_pep_gly_pose.set_fullatom_flag( fullatom, false /*repack*/);
	// nat_pep_gly_pose.copy_segment( nres_nat_pep, nat_cmplx_gly_pose, 1, nres_Ab+1 /*begin_src*/ );
	copy_segment_to_new_pose( nat_cmplx_gly_pose, nres_Ab+1, nres_nat_cmplx, nat_pep_gly_pose );
	nat_pep_gly_pose.score( weight_map );
	// nat_pep_gly_pose.dump_pdb( "nat_pep_gly_pose.pdb");
	float const fa_rep_nat_peptide_gly ( nat_pep_gly_pose.get_0D_score( FA_REP ) );

	// atomnums in full_coord or Eposition
	int atomnum_N,atomnum_CA,atomnum_C,atomnum_O,atomnum_CB;
	if ( fullatom ) {
		atomnum_N =  1;
		atomnum_CA = 2;
		atomnum_C =  3;
		atomnum_O =  4;
		atomnum_CB = 5;
	} else {
		atomnum_N =  1;
		atomnum_CA = 2;
		atomnum_C =  4;
		atomnum_O =  5;
		atomnum_CB = 3;
	}

	////////////////////////////////////////////////////////////////////
	//native: now measure clash between epitope backbone and Ab all-atom
	////////////////////////////////////////////////////////////////////

	float const ddg_fa_rep_nat_gly ( fa_rep_nat_complex_gly - fa_rep_nat_peptide_gly - fa_rep_Ab );

	std::cout << "native complex w/all-gly epitope:: "
						<< "farep_cmplx, farep_Ab, farep_peptide, ddg_farep "
						<< fa_rep_nat_complex_gly << " "
						<< fa_rep_Ab << " "
						<< fa_rep_nat_peptide_gly << " "
						<< ddg_fa_rep_nat_gly << std::endl;

	// count # of sites scored.
	int n_graft_sites_scored = 0;

	// yab: for structure caching; don't load twice if files between i and i+1 are the same
	std::string prior_filename( "" );
	Pose prior_structure;

	for ( int i_scaff = 1; i_scaff <= n_scaff; i_scaff++ ) {

		if ( using_checkpoint ) {
			bool already_scored = check_conformation_already_scored( i_scaff, checkpoint_file_w_path );
			if ( already_scored ) continue;
		}

		////////////////////////////////////////////////////////////////////
		//		load scaffold into pose
		//
		std::string scaff_base_filename = list_scaff_pdb( i_scaff );
		std::string scaff_filename = files_paths::start_path + scaff_base_filename;

		// strip terminal .pdb.gz or .pdb to use base_filename in output
		if ( has_suffix( scaff_base_filename, ".pdb.gz" ) ) scaff_base_filename.erase( scaff_base_filename.length() - 7 ); // strip terminal .pdb.gz
		if ( has_suffix( scaff_base_filename, ".pdb" ) ) scaff_base_filename.erase( scaff_base_filename.length() - 4 ); // strip terminal .pdb

		utility::io::izstream infile ( scaff_filename );
		if ( ! infile ) {
			std::cout	<< " cannot open file " << scaff_filename
								<< std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		std::cout << "successfully read input pdb " << SS( scaff_filename ) << std::endl;

		// Deal with possibility that input "scaff" pdb includes Ab in complex with scaff
		// and Ab can come before or after the scaff

		// yab: structure caching
		Pose input_structure;
		if ( prior_filename != scaff_filename ) {

			pose_from_pdb( input_structure, scaff_filename, fullatom, ideal_pose, read_all_chains );

			prior_structure = input_structure; // cache
			prior_filename = scaff_filename;

		} else {
			input_structure = prior_structure;
		}
		int nres_input = input_structure.total_residue();

		Pose scaff_pose;
		int nres_scaff = 0;
		if ( !input_pdbs_include_Ab ) {
			nres_scaff = nres_input;
			scaff_pose = input_structure;
		} else {
			nres_scaff = nres_input - nres_Ab;
			int begin_src, end_src;
			if ( scaff_first ) {
				begin_src = 1; // by default scaff is first
				end_src = nres_scaff;
			} else {
				begin_src = nres_Ab + 1; //scaff comes after Ab
				end_src = nres_input;//scaff comes after Ab
			}
			scaff_pose.simple_fold_tree( nres_scaff );
			scaff_pose.set_fullatom_flag( fullatom, false /*repack*/ );
			//			scaff_pose.copy_segment( nres_scaff, input_structure, 1, begin_src );
			copy_segment_to_new_pose( input_structure, begin_src, end_src, scaff_pose );
		}

//		scaff_pose.dump_pdb("scaff_pose.pdb");

		// delta_resnum is:
		//         1. minimum sequence separation between start,stop of graft
		//              and
		//         2. boundary condition at n,c term
		//      ( graft can't start stop within delta_resnum of either terminus )
		int const delta_resnum = 3;

		// allow grafting of sub-ranges of input graft peptide

		// n_graft_ranges is set by reading in all possible ranges, unless we are check_specific_sites
		if ( check_specific_sites ) n_graft_ranges = 1;

		for ( int i_graft_range = 1; i_graft_range <= n_graft_ranges; i_graft_range++ ) {

			// The basic assumption is that the scaff will nearly close the graft already
			// and that the residues that can be aligned in space are
			//
			//    graft_resnum_1   <---->  scaff_gap_resnum_1
			//    graft_resnum_2   <---->  scaff_gap_resnum_2
			//
			// When we finally put the graft on, we can only keep one member of
			// of each aligned pair, because they will overlap in space.

			int graft_resnum_1, graft_resnum_2;
			if ( check_specific_sites ) {
				graft_resnum_1 = list_specific_graft_resnums( 1, i_scaff );//n2c or c2n is taken care of below.
				graft_resnum_2 = list_specific_graft_resnums( 2, i_scaff );
			} else {
				graft_resnum_1 = list_graft_ranges( 1, i_graft_range );
				graft_resnum_2 = list_graft_ranges( 2, i_graft_range );
			}

			// loop over residues in scaffold -- ires_scaff is start position for graft
			int ires_scaff_first = delta_resnum;
			int ires_scaff_last = nres_scaff - delta_resnum;

			if ( check_specific_sites ) {
				if ( list_specific_graft_dir( i_scaff ) == "n2c" ) {
					ires_scaff_first = list_specific_scaff_gap_resnums( 1, i_scaff );//scaff_gap_1 always < scaff_gap_2 but graft always ires -> jres
				} else {
					ires_scaff_first = list_specific_scaff_gap_resnums( 2, i_scaff );
				}
				ires_scaff_last = ires_scaff_first; //only check one specific ires_scaff
			}
			for ( int ires_scaff = ires_scaff_first; ires_scaff <= ires_scaff_last; ires_scaff++ ) {
				//			for ( int ires_scaff = delta_resnum; ires_scaff <= nres_scaff - delta_resnum; ires_scaff++ ) {

				// whether n2c or c2n we are initiating the transform at ires_scaff

				int const transform_start_resnum_scaff = ires_scaff;

				// graft_direction is n2c or c2n  ( default is to test both )
				int i_dir_first = 1;
				int i_dir_last = 2;
				if ( check_specific_sites ) {
					if ( list_specific_graft_dir( i_scaff ) == "n2c" ) {
						i_dir_first = 1;
					} else {
						i_dir_first = 2;
					}
					i_dir_last = i_dir_first; //only check one specific direction
				}

				for ( int i_dir = i_dir_first; i_dir <= i_dir_last; i_dir++ ) {

					assert( i_dir >= 1 && i_dir <= 2 );
					std::string graft_dir;
					if ( i_dir == 1 ) graft_dir = "n2c";
					if ( i_dir == 2 ) graft_dir = "c2n";


					// setup a breakout so we can avoid CA or C checks at j_res if N shows too much clash
					// ...this will not prevent same check at c2n if n2c shows too much clash.
					// ...this is not necessary b/c we have "check_specific_sites"
//					FArray1D_bool bad_clash_breakout( nres_scaff, false );

					// alignment system is N, CA, or C  ( default is to test all 3 )
					int i_align_sys_first = 1;
					int i_align_sys_last = 3;
					if ( check_specific_sites ) {
						if ( list_specific_graft_align_system( i_scaff ) == "N" ) {
							i_align_sys_first = 1;
						} else {
							if ( list_specific_graft_align_system( i_scaff ) == "CA" ) {
								i_align_sys_first = 2;
							} else {
								i_align_sys_first = 3;
							}
						}
						i_align_sys_last = i_align_sys_first;  //only check one specific align_sys
					}

					for ( int i_align_sys = i_align_sys_first; i_align_sys <= i_align_sys_last; i_align_sys++ ) {

						assert( i_align_sys >= 1 && i_align_sys <= 3 );
						std::string graft_align_system;
						if ( i_align_sys == 1 ) graft_align_system = "N";
						if ( i_align_sys == 2 ) graft_align_system = "CA";
						if ( i_align_sys == 3 ) graft_align_system = "C";

						// Align ires_scaff from scaff onto the appropriate subrange in graft

						FArray1D_float coords_pep_cent( 3 ),coords_pep_forw( 3 ),coords_pep_back( 3 );
						FArray1D_float coords_scaff_cent( 3 ),coords_scaff_forw( 3 ),coords_scaff_back( 3 );

						get_alignment_coords( nat_pep_gly_pose,
																	scaff_pose,
																	graft_align_system,
																	graft_dir,
																	graft_resnum_1,
																	graft_resnum_2,
																	transform_start_resnum_scaff,
																	transform_start_resnum_scaff,
																	coords_pep_cent,    //outputs from here down
																	coords_pep_forw,
																	coords_pep_back,
																	coords_scaff_cent,
																	coords_scaff_forw,
																	coords_scaff_back );

						////////////////////////////////////////////////////////////////////
						// get matrix to reorient peptide coordinates onto scaffold coordinates
						// P2 mapped to Q2...P2->P3 vector mapped to Q2->Q3 vector
						//

						FArray2D_float Mgl( 4, 4 );
						get_GL_matrix( coords_pep_forw,   coords_pep_cent,   coords_pep_back,
													 coords_scaff_forw, coords_scaff_cent, coords_scaff_back, Mgl );

						////////////////////////////////////////////////////////////////////
						//make a copy of the peptide gly coordinates
						//then re-orient these coords
						//

						FArray3D_float coords_pep_gly_oriented ( nat_pep_gly_pose.full_coord() );

						// apply the transformation to the peptide
						// apply to coords_pep_gly_oriented
						for ( int ires = 1; ires <= nres_nat_pep; ires++ ) {
							int const aa( nat_pep_gly_pose.res( ires ) );
							int const aav( nat_pep_gly_pose.res_variant( ires ));
							for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
								GL_rot_in_place( Mgl, coords_pep_gly_oriented( 1, iatom, ires ) );
							}
						}

						///////////////////////////////////////////////////////////////////////////////////////////////
						//
						//Now loop over rest of scaffold ( either n2c or c2n ) and measure chainbreak_rms over 5 atoms
						//
						///////////////////////////////////////////////////////////////////////////////////////////////

						int jres_min, jres_max;
						if ( graft_dir == "n2c" ) {
							jres_min = ires_scaff + delta_resnum;  // n2c:  jres > ires
							jres_max = nres_scaff - delta_resnum;  // n2c:  jres > ires

							//
							//for n2c, set the jres_max using max_loopsize_difference...
							//
							int jres_max_alt = ires_scaff + graft_resnum_2 - graft_resnum_1 + max_loopsize_difference;
							jres_max = std::min( jres_max, jres_max_alt );

						} else {
							jres_min = delta_resnum;               //c2n: jres < ires
							jres_max = ires_scaff - delta_resnum;  //c2n: jres < ires

							//
							//for c2n, set the jres_min using max_loopsize_difference...
							//
							int jres_min_alt = ires_scaff - ( graft_resnum_2 - graft_resnum_1 ) - max_loopsize_difference;
							jres_min = std::max( jres_min, jres_min_alt );

						}

						//safety
						//						if ( jres_min > jres_max ) {
						//							std::cout << "WARNING: jres_min > jres_max..not searching unless check_specific_sites " << std::endl;
						//						}

						// override jres_min,jres_max for specific sites
						if ( check_specific_sites ) {
							if ( list_specific_graft_dir( i_scaff ) == "n2c" ) {
								jres_min = list_specific_scaff_gap_resnums( 2, i_scaff );//scaff_gap_1 always < scaff_gap_2 but graft always ires -> jres
							} else {
								jres_min = list_specific_scaff_gap_resnums( 1, i_scaff );
							}
							jres_max = jres_min;
						}
						for ( int jres_scaff = jres_min; jres_scaff <= jres_max; jres_scaff++ ) {

							++n_graft_sites_scored;

							// now measure rms over 5 closure atoms
							int const n_closure_atom = 5;
							FArray1D_int closure_atom( n_closure_atom );
							FArray1D_int closure_resnum_scaff( n_closure_atom );
							FArray1D_int closure_resnum_graft( n_closure_atom );

							get_closure_atom_resnum( fullatom,
																			 graft_dir,
																			 graft_resnum_1,
																			 graft_resnum_2,
																			 jres_scaff,
																			 closure_atom,
																			 closure_resnum_scaff,
																			 closure_resnum_graft );

							// measure rms and report finding...
							// just output info in format that can be input to the actual grafting/design stage.

							float dist2 = 0.0;
							for ( int i = 1; i <= n_closure_atom; i++ ) {
								for ( int k = 1; k <= 3; k++ ) {
									dist2  += square( coords_pep_gly_oriented( k, closure_atom( i ), closure_resnum_graft( i ) )
																		- scaff_pose.full_coord()( k, closure_atom( i ), closure_resnum_scaff( i ) ) );
								}
							}
							float one_over_n_atoms = 1.0 / n_closure_atom;
							float closure_rms = std::sqrt( dist2 * one_over_n_atoms );

							// Note: ires_scaff and jres_scaff determine scaff_gap_resnum_1, scaff_gap_resnum_2

							// by convention we write out the scaff_gap always from n2c
							int scaff_gap_resnum_1;
							int scaff_gap_resnum_2;
							if ( graft_dir == "n2c" ) {
								scaff_gap_resnum_1 = ires_scaff;
								scaff_gap_resnum_2 = jres_scaff;
							} else {
								scaff_gap_resnum_1 = jres_scaff;
								scaff_gap_resnum_2 = ires_scaff;
							}

							if (! clash_check ) {

								// report closure_rms if below max_closure_rms, then continue to next graft_site
								if ( closure_rms <= max_closure_rms ) {
									outfile << I( 6, i_scaff ) << ' '
													<< I( 6, i_graft_range ) << ' '
													<< SS( graft_dir ) << ' '
													<< SS( right_string_of( graft_align_system, 4) )
													<< ' ' << I( 8, list_graft_ranges( 3, i_graft_range ) )
													<< ' ' << I( 8, list_graft_ranges( 4, i_graft_range ) )
													<< ' ' << I( 4, graft_resnum_1 ) << ' ' << I( 4, graft_resnum_2 )
													<< ' ' << I( 4, scaff_gap_resnum_1 ) << ' ' << I( 4, scaff_gap_resnum_2 )
													<< ' ' << F( 7, 2, closure_rms )
                          << ' ' << I( 11, nres_scaff )
													<< ' ' << SS( scaff_filename ) << std::endl;
								}
								continue;
							}

							///////////////////////////////////////////////////////////////////////////////////////////////////////
							//
							// clash check now, "intra" and "inter"
							//
							///////////////////////////////////////////////////////////////////////////////////////////////////////

							// Don't bother with clash_check if rms is no good
							if ( closure_rms >  max_closure_rms ) {
								if ( ! check_specific_sites ) continue;
							}

							// Parameters derived from those read in

							int const nres_graft = graft_resnum_2 - graft_resnum_1 + 1;


							// nres_gap_in_scaff tells how many residues can fit in gap if
							// both gap residues from the scaff are deleted.
							// or if the terminal residues on the graft are deleted.
							int const nres_gap_in_scaff = scaff_gap_resnum_2 - scaff_gap_resnum_1 + 1;

							// It is OK to have a graft that is longer or shorter in # of residues  than the gap in scaffold
							if ( ! ( nres_gap_in_scaff == nres_graft )  && runlevel_ns::runlevel > runlevel_ns::quiet ) {
								std::cout << "WARNING: size mismatch between graft: " << nres_graft
													<< " and gap: " << nres_gap_in_scaff << std::endl;
							}

							// construct a gly-only version of scaffold  -- scaff_gly_pose
							Pose scaff_gly_pose;
							scaff_gly_pose = scaff_pose;
							gly_begin = 1;
							gly_end = nres_scaff;
							convert_resnum_range_to_gly( scaff_gly_pose, gly_begin, gly_end );
//							scaff_gly_pose.dump_pdb("scaff_gly_pose.pdb");

							//  construct gly-only version with graft-site removed -- scaff_gly_w_gap_pose
							//
							//  Note:  scaff_gap_resnum_1 < gap < scaff_gap_resnum_2
							//         gap includes scaff_gap_resnum_x
							//
							Pose scaff_gly_w_gap_pose;
							int gap_begin = scaff_gap_resnum_1;
							int gap_end = scaff_gap_resnum_2;
							make_pose_with_gap( scaff_gly_pose, gap_begin, gap_end, scaff_gly_w_gap_pose );
							//							scaff_gly_w_gap_pose.dump_pdb("scaff_gly_w_gap_pose.pdb");
							int const nres_scaff_w_gap = scaff_gly_w_gap_pose.total_residue();

							if ( nres_scaff_w_gap != ( nres_scaff - ( gap_end - gap_begin + 1 ) ) ) {
								std::cout << "error on nres_scaff_w_gap " << nres_scaff_w_gap << " "
													<< nres_scaff << " " << gap_end << " " << gap_begin << std::endl;
								utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
							}

							// score scaff_gly_w_gap and save the fa_rep
							scaff_gly_w_gap_pose.score( weight_map );
							float const fa_rep_scaff_gly_w_gap ( scaff_gly_w_gap_pose.get_0D_score( FA_REP ) );


							// yab: construct bounding box around scaffold_gly_w_gap
							BoundingBox bb = epigraft::bounding_box( scaff_gly_w_gap_pose );

							// yab: construct octree around scaffold_gly_w_gap
							Octree< epigraft::AtomPoint > octree_sgwg( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() ); // need a better variable name!
							epigraft::fill_octree( octree_sgwg, scaff_gly_w_gap_pose );


							////////////////////////////////////////////////////////////////////
							//The next 3 steps are:
							//
							//(1) Orient the full-length all-gly peptide onto the graftsite (gap in scaff)
							//       must do the orienting on full-length peptide b/c input resnums
							//       control which atoms to superpose
							//(2) trim native peptide to the right residue range for this graft
							//(3) evalute repE interaction between peptide and scaff.
							//
							////////////////////////////////////////////////////////////////////

							//
							//(1)  Orient the full-length all-gly peptide onto the graftsite (gap in scaff)
							//
							//align 3 atoms on graft peptide with 3 atoms on scaff
							//the three atoms we use depend on the "graft_align_sys" and the "graft_dir"
							//the 3 atoms are referred to as "cent","forw", and "back"

							get_alignment_coords( nat_pep_gly_pose,
																		scaff_gly_pose,
																		graft_align_system,
																		graft_dir,
																		graft_resnum_1,
																		graft_resnum_2,
																		scaff_gap_resnum_1,
																		scaff_gap_resnum_2,
																		coords_pep_cent,    //outputs from here down
																		coords_pep_forw,
																		coords_pep_back,
																		coords_scaff_cent,
																		coords_scaff_forw,
																		coords_scaff_back );


							////////////////////////////////////////////////////////////////////
							// get matrix to reorient peptide coordinates onto scaffold coordinates
							// P2 mapped to Q2...P2->P3 vector mapped to Q2->Q3 vector
							//

							get_GL_matrix( coords_pep_forw,   coords_pep_cent,   coords_pep_back,
														 coords_scaff_forw, coords_scaff_cent, coords_scaff_back, Mgl );

							////////////////////////////////////////////////////////////////////
							//make a copy of the peptide gly coordinates
							//then re-orient these coords
							//

							//							FArray3D_float coords_pep_gly_oriented ( nat_pep_gly_pose.full_coord() );
							coords_pep_gly_oriented = nat_pep_gly_pose.full_coord();

							// apply the transformation to the peptide
							// apply to coords_pep_gly_oriented
							for ( int ires = 1; ires <= nres_nat_pep; ires++ ) {
								int const aa( nat_pep_gly_pose.res( ires ) );
								int const aav( nat_pep_gly_pose.res_variant( ires ));
								for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
									GL_rot_in_place( Mgl, coords_pep_gly_oriented( 1, iatom, ires ) );
								}
							}

							////////////////////////////////////////////////////////////////////
							//make a pose of just the oriented peptide so we can dump it and look at it
							//
							Pose pep_gly_oriented_pose;
							pep_gly_oriented_pose.simple_fold_tree( nres_nat_pep );
							pep_gly_oriented_pose.set_fullatom_flag( true, false );// set fullatom and do not repack
							FArray3D_float Epos_pep_gly_oriented( 3, param::MAX_POS, nres_nat_pep );//MAX_POS = 5
							for ( int i = 1; i <= nres_nat_pep; ++i ) {
								pep_gly_oriented_pose.set_phi        ( i, nat_pep_gly_pose.phi(i) );
								pep_gly_oriented_pose.set_psi        ( i, nat_pep_gly_pose.psi(i) );
								pep_gly_oriented_pose.set_omega      ( i, nat_pep_gly_pose.omega(i) );
								pep_gly_oriented_pose.set_secstruct  ( i, nat_pep_gly_pose.secstruct(i) );
								pep_gly_oriented_pose.set_name       ( i, nat_pep_gly_pose.name(i) );
								pep_gly_oriented_pose.set_res        ( i, nat_pep_gly_pose.res( i ) );
								pep_gly_oriented_pose.set_res_variant( i, nat_pep_gly_pose.res_variant( i ) );
								for ( int k = 1; k <= 3; ++k ) {
									Epos_pep_gly_oriented(k,1,i) = coords_pep_gly_oriented(k,1,i);
									Epos_pep_gly_oriented(k,2,i) = coords_pep_gly_oriented(k,2,i);
									Epos_pep_gly_oriented(k,4,i) = coords_pep_gly_oriented(k,3,i);
									Epos_pep_gly_oriented(k,5,i) = coords_pep_gly_oriented(k,4,i);
									Epos_pep_gly_oriented(k,3,i) = coords_pep_gly_oriented(k,5,i);
								}
							}
							pep_gly_oriented_pose.set_coords( ideal_pose, Epos_pep_gly_oriented, coords_pep_gly_oriented, check_missing );
//							pep_gly_oriented_pose.dump_pdb( "pep_gly_oriented_pose.pdb" );

							////////////////////////////////////////////////////////////////////
							//1. trim the full length all-gly peptide to the correct
							//   sub-range for the graft (call this the "graft_gly")
							//2. make a pose for oriented graft_gly
							//
							//
							//Note: General rule is to include the terminal residues of the graft

							Pose graft_gly_pose;
							graft_gly_pose.simple_fold_tree( nres_graft );
							graft_gly_pose.set_fullatom_flag( true, false /*repack*/ );
//							graft_gly_pose.copy_segment( nres_graft, pep_gly_oriented_pose, 1, graft_resnum_1+1 /*begin_src*/ );
							copy_segment_to_new_pose( pep_gly_oriented_pose, graft_resnum_1, graft_resnum_1 + nres_graft - 1, graft_gly_pose );
//							graft_gly_pose.dump_pdb( "graft_gly_pose.pdb" );


							// yab: count bumps through octree using no-termini oriented gly peptide
							Size intra_bumps = epigraft::count_bumps( octree_sgwg, graft_gly_pose, BUMP_DISTANCE );

							// skip if too many bumps
							if ( intra_bumps > (rootstock::Size)max_intra_bumps ) {
								continue;
							}

							// Score the graft_gly_pose to get fa_rep

							graft_gly_pose.score( weight_map );
							float const fa_rep_graft_gly ( graft_gly_pose.get_0D_score( FA_REP ) );

							// next we want to make a pose that has the graft_notermini + scaff_w_gap in all-gly
							// then score fa_rep for that pose and substract fa_rep for scaff_w_gap alone

							int const nres_scaff_w_graft = nres_graft + nres_scaff_w_gap;

							if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
								std::cout << "nres_scaff_w_graft, nres_graft, nres_scaff_w_gap = "
													<< nres_scaff_w_graft << " "
													<< nres_graft << " "
													<< nres_scaff_w_gap << std::endl;
								std::cout << "nres_scaff, scaff_gap_resnum_1, scaff_gap_resnum_2 "
													<< nres_scaff << " " << scaff_gap_resnum_1 << " "
													<< scaff_gap_resnum_2 << std::endl;
							}

							Pose scaff_w_graft_gly_pose;
//							make_scaff_w_graft_gly_pose ( scaff_gly_pose,
//																						graft_gly_pose,
//																						scaff_gap_resnum_1,
//																						scaff_gap_resnum_2,
//																						nres_scaff_w_graft,
//																						scaff_w_graft_gly_pose );//output

							new_make_scaff_w_graft_gly_pose ( scaff_gly_pose,
							                              graft_gly_pose,
																						graft_align_system,
																						graft_dir,
																						1,
																						graft_gly_pose.total_residue(),
																						scaff_gap_resnum_1,
																						scaff_gap_resnum_2,
																						nres_scaff_w_graft,
																						scaff_w_graft_gly_pose );//output

							//							pose_to_misc( scaff_w_graft_gly_pose ); //necessary?
							scaff_w_graft_gly_pose.score( weight_map );
							float const fa_rep_scaff_w_graft_gly ( scaff_w_graft_gly_pose.get_0D_score( FA_REP ) );
							//							scaff_w_graft_gly_pose.dump_pdb("scaff_w_graft_gly_pose.pdb");

							////////////////////////////////////////////////////////////////////
							//scaffold: now measure "intra" clash between graft and scaff backbones only
							////////////////////////////////////////////////////////////////////

							float const fa_rep_intra_gly ( fa_rep_scaff_w_graft_gly - fa_rep_graft_gly - fa_rep_scaff_gly_w_gap );

//							std::cout << "fa_rep_intra_gly, fa_rep_scaff_w_graft_gly, fa_rep_graft_gly, fa_rep_scaff_gly_w_gap "
//												<< fa_rep_intra_gly << " " << fa_rep_scaff_w_graft_gly << " " << fa_rep_graft_gly << " "
//												<< fa_rep_scaff_gly_w_gap << std::endl;

							///////////////////////////////////////////////////////////////////////////////
							//Ask if this intra clash is below the loose threshold
							//
							// if not satisfy loose_max, continue to next jres_scaff
							// ...this avoids spending time on the inter-clash check
							//
							if ( fa_rep_intra_gly > max_intra_clash ) {
								continue;
							}


							///////////////////////////////////////////////////////////////////////////////
							//Now want to measure fa_rep_inter_gly
							//  This is the clash between all-atom Ab and all-gly scaff_w_gap
							//
							//For this we must orient the Ab the same way we have oriented the graft
							//Then we can either (1) make a pose of oriented Ab+ scaff_w_gap and score it
							//            or     (2) directly measure repE between Ab and scaff_w_gap (faster?)
							//

							// make a copy of the Ab coords to orient
							FArray3D_float coords_Ab_oriented ( Ab_pose.full_coord() );

							// apply the transformation to the Ab ( use same matrix "Mgl" applied to epitope )
							for ( int ires = 1; ires <= nres_Ab; ires++ ) {
								int const aa( Ab_pose.res( ires ) );
								int const aav( Ab_pose.res_variant( ires ));
								for ( int iatom = 1; iatom <= aaproperties_pack::natoms( aa, aav ); ++iatom ) {
									GL_rot_in_place( Mgl, coords_Ab_oriented( 1, iatom, ires ) );
								}
							}

							// make a pose of Ab_oriented
							Pose Ab_oriented_pose;
							Ab_oriented_pose = Ab_pose;

							// make an Eposition array that is consistent with
							// transformed fcoord array.
							FArray3D_float Epos_Ab_oriented( 3, param::MAX_POS, nres_Ab );//MAX_POS = 5

							for ( int i = 1; i <= nres_Ab; ++i ) {
								for ( int k = 1; k <= 3; ++k ) {
									Epos_Ab_oriented(k,1,i) = coords_Ab_oriented(k,1,i);
									Epos_Ab_oriented(k,2,i) = coords_Ab_oriented(k,2,i);
									Epos_Ab_oriented(k,4,i) = coords_Ab_oriented(k,3,i);
									Epos_Ab_oriented(k,5,i) = coords_Ab_oriented(k,4,i);
									Epos_Ab_oriented(k,3,i) = coords_Ab_oriented(k,5,i);
								}
							}
							// Put oriented coords in the pose
							Ab_oriented_pose.set_coords( ideal_pose, Epos_Ab_oriented, coords_Ab_oriented, check_missing );


							// yab: count bumps through octree using oriented Ab
							Size inter_bumps = epigraft::count_bumps( octree_sgwg, Ab_oriented_pose, BUMP_DISTANCE );


							// make a pose of Ab (all-atom) + scaff_w_graft (all-gly)
							// ( this can be revised to use construct_pose_complex_from_p1_p2 )
							// must test it out to make sure gives same result

							Pose Ab_scaff_w_graft_gly_pose;
							make_Ab_scaff_w_graft_gly_pose( Ab_oriented_pose,
																							scaff_w_graft_gly_pose,
																							Ab_scaff_w_graft_gly_pose );

							// Score the Ab_scaff_w_graft_gly_pose to get fa_rep

							Ab_scaff_w_graft_gly_pose.score( weight_map );
							float const fa_rep_Ab_scaff_w_graft_gly ( Ab_scaff_w_graft_gly_pose.get_0D_score( FA_REP ) );

							if ( runlevel_ns::runlevel > runlevel_ns::standard ) {
								std::cout << "Ab_scaff_w_graft_gly_pose.show_scores():" << std::endl;
								std::cout << Ab_scaff_w_graft_gly_pose.show_scores() << std::endl;
							}

							////////////////////////////////////////////////////////////////////
							//measure "inter" clash between Ab all-atom and scaff_w_graft backbone
							////////////////////////////////////////////////////////////////////

							float const fa_rep_inter_gly ( fa_rep_Ab_scaff_w_graft_gly - fa_rep_Ab - fa_rep_scaff_w_graft_gly );

							///////////////////////////////////////////////////////////////////////////////
							//Ask if this inter clash is below the loose threshold
							//
							// if not satisfy loose_max, continue to next graft/scaff combo
							//
							/*
							if ( fa_rep_inter_gly > max_fa_rep_inter_gly_before_close ) {
								std::cout << "fa_rep_inter_gly too high, "
													<< "going to next graft/scaff combo"
													<< std::endl;
								continue;
							}
							*/


							outfile << I( 6, i_scaff ) << ' '
											<< I( 6, i_graft_range ) << ' '
											<< SS( graft_dir ) << ' '
											<< SS( right_string_of( graft_align_system, 4) );
							if ( check_specific_sites ) {
								outfile << ' ' << I( 8, list_specific_graft_resnums( 3, i_scaff ) )  //native epitope numbering
												<< ' ' << I( 8, list_specific_graft_resnums( 4, i_scaff ) );
							} else {
								outfile << ' ' << I( 8, list_graft_ranges( 3, i_graft_range ) )      //native epitope numbering
												<< ' ' << I( 8, list_graft_ranges( 4, i_graft_range ) );
							}
							outfile	<< ' ' << I( 4, graft_resnum_1 ) << ' ' << I( 4, graft_resnum_2 )
											<< ' ' << I( 4, scaff_gap_resnum_1 ) << ' ' << I( 4, scaff_gap_resnum_2 )
											<< ' ' << F( 7, 2, closure_rms )
											<< ' ' << I( 11, intra_bumps )
											<< ' ' << I( 11, inter_bumps )
											<< ' ' << F( 12, 2, fa_rep_intra_gly )
											<< ' ' << F( 12, 2, fa_rep_inter_gly )
											<< ' ' << I( 11, nres_scaff )
											<< ' ' << SS( scaff_filename ) << std::endl;

							if ( output_pdbs ) {

								int graft_range_native_num_1, graft_range_native_num_2;
								int graft_range_sequential_num_1, graft_range_sequential_num_2;
								if ( check_specific_sites ) {
									graft_range_sequential_num_1 = list_specific_graft_resnums( 1, i_scaff );
									graft_range_sequential_num_2 = list_specific_graft_resnums( 2, i_scaff );
									graft_range_native_num_1 = list_specific_graft_resnums( 3, i_scaff );
									graft_range_native_num_2 = list_specific_graft_resnums( 4, i_scaff );
								} else {
									graft_range_sequential_num_1 = list_graft_ranges( 1, i_graft_range );
									graft_range_sequential_num_2 = list_graft_ranges( 2, i_graft_range );
									graft_range_native_num_1 = list_graft_ranges( 3, i_graft_range );
									graft_range_native_num_2 = list_graft_ranges( 4, i_graft_range );
								}


								std::string pdb_filename;
								pdb_filename = files_paths::pdb_out_path + scaff_base_filename + "_" + string_of( i_scaff ) + "_"
									+ graft_dir + "_" + graft_align_system + "_"
									+ string_of( graft_range_native_num_1 ) + "-" + string_of( graft_range_native_num_2 ) + "_"
									+ string_of( graft_range_sequential_num_1 ) + "-" + string_of( graft_range_sequential_num_2 ) + "_"
									+ string_of( scaff_gap_resnum_1 ) + "-" + string_of( scaff_gap_resnum_2 )
									+ "_pre-close.pdb";
								//									+ string_of( closure_rms ) + "_" + string_of( fa_rep_intra_gly ) + "_"
								//									+ string_of( fa_rep_inter_gly ) + ".pdb";
								Ab_scaff_w_graft_gly_pose.pdb_info().set_use_pdb_numbering( true );
								Ab_scaff_w_graft_gly_pose.dump_pdb( pdb_filename );
							}


							//
							// set the bad_clash_breakout to avoid checking this ires_scaff, jres_scaff, dir combination again
							//
							//							if ( fa_rep_inter_gly > max_fa_rep_inter_gly_before_close ||
							//									 fa_rep_intra_gly > max_intra_clash ) {
							//								bad_clash_breakout( jres_scaff ) = true;
							//							}

						}// jres_scaff
					}// i_align_sys
				}// i_dir
			}// ires_scaff
		}// i_graft_range

		if ( using_checkpoint ) {
			update_conformation_already_scored( i_scaff, checkpoint_file_w_path );
		}

	}// i_scaff

	// Close the output file
	outfile.close();
	outfile.clear();

}
