// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   epigraft_rough_match.cc
/// @brief  Searches for additional graft locations on an existing epitope-scaffold.
/// @author Bill Schief (schief@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// unit headers
#include <epigraft/old/epigraft_rough_match.hh>

// package headers
#include <epigraft/epi_graft.h> // split away after refactoring
#include <epigraft/epigraft_functions.hh>
#include <epigraft/epigraft_io.hh>
#include <epigraft/AntibodyComplex.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/ResidueRange.hh>
#include <epigraft/ScaffoldInfo.hh>
#include <epigraft/old/old_functions.hh>

// rosetta headers
#include <after_opts.h>
#include <files_paths.h>
#include <pose.h>
#include <pose_io.h>
#include <score.h>

// rootstock headers
#include <rootstock/Octree.hh>

// ObjexxFCL headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// numeric headers
#include <numeric/xyzVector.hh>

// utility headers
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/vector1.hh>

// C++ headers
#include <iostream>
#include <sstream>
#include <string>

namespace epigraft {

// temporary globals
rootstock::Distance const OCTREE_CUBE_SIZE = 6.0;

/// @brief rough match protocol
void
epigraft_rough_match()
{

	using namespace pose_ns;
	
	// scorefxn
	Score_weight_map weight_map( score12 );
	
	// set basic parameters controlling Pose behavior
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	bool const read_all_chains( true );
	bool const check_missing( false ); // use this for pose.set_coords throughout
//	bool const coords_init( true ); // used in pose_from_misc

	// atom numbers
	int atomnum_N, atomnum_CA, atomnum_C, atomnum_O, atomnum_CB;
	if ( fullatom ) {
	  atomnum_N =  1;
	  atomnum_CA = 2;
	  atomnum_C =  3;
	  atomnum_O =  4;
	  atomnum_CB = 5;
	} else { // Eposition numbering
	  atomnum_N =  1;
	  atomnum_CA = 2;
	  atomnum_C =  4;
	  atomnum_O =  5;
	  atomnum_CB = 3;
	}

	// file to output scores
	std::string output_filename;
	stringafteroption( "output_file", "none", output_filename );
	if( output_filename == "none" ) {
		std::cout << "Need -output_file <filename> "
	            << "on command-line " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// open the output file
	// ( leave it open since we write scores about 1/s or faster )
	utility::io::ozstream outfile;
	open_output_file( output_filename, outfile );


	// put header in output file
	outfile << SS("scaff")
	        << SS("  loop")
	        << SS("   rng")
	        << SS("epi1_nat epi2_nat")
	        << SS(" epi1  epi2")
	        << SS(" sca1  sca2")
	        << SS("chainbreak_rms")
	        << SS("chbk_rms_nterm")
	        << SS("chbk_rms_cterm")
	        << SS("loop1_id")
	        << SS("loop1_epi1_nat loop1_epi2_nat")
	        << SS("loop1_epi1 loop1_epi2")
	        << SS("intra_clash")
	        << SS(" scaff_filename") << std::endl;


	// checkpoint file for running on clusters
	std::string checkpoint_filename;
	stringafteroption( "checkpoint", "none", checkpoint_filename );
	bool using_checkpoint = false;
	if ( checkpoint_filename == "none" ) {
		std::cout << "no checkpoint_filename, NOT using checkpointing" << std::endl;
	} else {
		using_checkpoint = true;
		checkpoint_filename = files_paths::score_path + checkpoint_filename;
	}

	// option to output pdbs oriented relative to nat_cmplx
	bool output_pdbs = truefalseoption( "output_pdbs" );

	// parameters controlling matching
	float ca_dist_cutoff = 6.0;
	realafteroption( "rough_match_ca_dist_cutoff", ca_dist_cutoff, ca_dist_cutoff );

	int max_loopsize_difference = 10;
	intafteroption( "max_loopsize_difference", max_loopsize_difference, max_loopsize_difference );
	
	// run clash check?
	bool clash_check = false;
	clash_check = truefalseoption("clash_check");
	
	// intra clash
	float max_intra_clash = 10000.0f;
	realafteroption( "max_intra_clash", max_intra_clash, max_intra_clash );

	// load epitope (loop) ranges
	utility::vector1< LoopInfo > loops_to_scan;
	epigraft::old::load_epitope_subranges( stringafteroption( "epi_loop_info" ), loops_to_scan );

	// load input scaffold info
	utility::vector1< ScaffoldInfo > scaffolds_to_scan;
	load_scaffold_info( stringafteroption( "list_superpos_scaff_info" ), scaffolds_to_scan );


	// set length of Ab from commandline (required!)
	int nres_Ab( 0 );
	intafteroption( "nres_Ab", 0, nres_Ab );
	if ( nres_Ab == 0 ) {
		std::cout << "must specify -nres_Ab <nres> on commandline" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// set lower limit on epitope sub-range to match, must be greater than 1 otherwise
	// Pose fold tree crashes!
	int min_match_length( 3 );
	intafteroption( "min_match_length", 3, min_match_length );


	// load native complex from file, NOTE: assume Ab comes first!
	std::string native_complex_filename = stringafteroption( "native_complex" );
	AntibodyComplex native( native_complex_filename, nres_Ab, true, true ); // booleans: Ab_first, refold_sidechains_from_chi
	

	// must specify whether Ab or scaff comes first for query complexes
	bool Ab_first = truefalseoption("Ab_first");
	bool scaff_first = truefalseoption("scaff_first");
	if ( Ab_first && scaff_first ) {
		std::cout << "ERROR you cannot specify both Ab_first and scaff_first" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else {
		if ( !(Ab_first || scaff_first)) {
			std::cout << "ERROR must specify either Ab_first or scaff_first" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	
	// now run through each scaffold in list
	for ( int scaffold_id = 1, last_scaffold_id = scaffolds_to_scan.size(); scaffold_id <= last_scaffold_id; ++scaffold_id ) {

		// checkpointing
		if ( using_checkpoint ) {
			bool already_scored = check_conformation_already_scored( scaffold_id, checkpoint_filename );
			if ( already_scored ) {
				continue;
			}
		}

		// current scaffold info
		ScaffoldInfo const & scaffold_info = scaffolds_to_scan[ scaffold_id ];

		// load Ab-scaff complex into pose
		std::string Ab_scaff_filename = files_paths::start_path + scaffold_info.input_filename();

		utility::io::izstream infile ( Ab_scaff_filename );
		if ( !infile ) {
			std::cout	<< " cannot open file " << Ab_scaff_filename << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		AntibodyComplex decoy( Ab_scaff_filename, nres_Ab, Ab_first );
		
		// enforce ordering of Ab, otherwise superposition will fail
		if ( !compare_chain_order( native.Ab(), decoy.Ab() ) ) {
			utility::exit( __FILE__, __LINE__, Ab_scaff_filename + " : Ab chain ordering incorrect, check your input files!" );
		}

		// make oriented Pose
		Pose decoy_complex_oriented;
		decoy_complex_oriented = decoy.complex();

		// alignment, decoy_complex_oriented moves on top of native.complex()
		double rms = superimpose_using_ncaco( native.complex(), 1, nres_Ab, decoy_complex_oriented, 1, nres_Ab );
		if ( rms > 1.0 ) {
			utility::exit( __FILE__, __LINE__, Ab_scaff_filename + " : Ab superposition rms > 1.0, check your input files!" );
		}

		// reconstruct re-oriented/aligned decoy
		AntibodyComplex aligned_decoy( decoy_complex_oriented, nres_Ab, true ); // boolean: Ab always first within protocol

//		decoy_complex_oriented.dump_pdb("decoy_complex_oriented_pose_" + string_of( s + 1 ) + "_.pdb");

		// record id of existing epitope on scaffold, Bill prints this on output
		int existing_epitope_id = -1;
		for ( int loop_id = 1, last_loop_id = loops_to_scan.size(); loop_id <= last_loop_id; ++loop_id ) {
			
			LoopInfo epitope_info = loops_to_scan[ loop_id ];
			
			if ( epitope_info.contains_native_subrange( scaffold_info.native_epitope_range() ) ) {
				existing_epitope_id = epitope_info.id();
				break;
			}
		}
		
		// run through each possible epitope (loop)
		for ( int loop_id = 1, last_loop_id = loops_to_scan.size(); loop_id <= last_loop_id; ++loop_id ) {
			
			LoopInfo const & current_loop = loops_to_scan[ loop_id ];
			
			// if scaffold already contains this epitope, skip it
			if ( current_loop.contains_native_subrange( scaffold_info.native_epitope_range() ) ) {
				continue;
			}
			
			// compute numbering offset in native (peptide) structure for this epitope (loop)
			int offset = 0;
			for ( int o = 1; o < loop_id; ++o) {
				offset += loops_to_scan[ o ].nres();
			}
			
			// **The following is the new offset calculation procedure.  It's currently commented out until
			//   we can figure out exactly what we want in the input subranges file and how the pdb file
			//   should be ordered.  Talk to yab or menis for more details.
			//
			// compute numbering offset in native (peptide) structure for this epitope (loop)
			// we should re-standardize the input format -- the following routine will not work correctly 
			// if there is no single longest subrange!
//			int offset = 0;
//			for ( int k = 1, ke = loops_to_scan.size(); k <= ke; ++k ) {
//				if ( k != loop_id ) {
//					
//					LoopInfo const & other_loop = loops_to_scan[ k ];
//					
//					if ( current_loop.longest_native_subrange().overlaps( other_loop.longest_native_subrange() ) ) {
//						if ( current_loop.longest_native_subrange().begin() >= other_loop.longest_native_subrange().begin() ) {
//							// here the two ranges overlap and the left endpoint of the current loop lies to the right
//							// of the left endpoint of the other loop
//							offset += current_loop.longest_native_subrange().begin() - other_loop.longest_native_subrange().begin();
//						}
//						
//					} else if ( current_loop.longest_native_subrange().begin() > other_loop.longest_native_subrange().begin() ) {
//						// here the two ranges are disjoint and the current loop lies completely on the right side
//						// of the other loop
//						offset += other_loop.longest_native_subrange().length();
//					}
//				}
//			}
			
			// run through each possible epitope (loop) sub-range
			for ( int subrange_id = 1, last_subrange_id = current_loop.native_subranges().size(); subrange_id <= last_subrange_id; ++subrange_id ) {
				
				ResidueRange const & native_subrange = current_loop.native_subranges()[ subrange_id ];
				ResidueRange const & internal_subrange = current_loop.internal_subranges()[ subrange_id ];
				ResidueRange epitope_subrange( internal_subrange + offset );
				
				// if epitope sub-range is too small then skip
				if ( epitope_subrange.length() < min_match_length ) {
					continue;
				}
				
				std::cout << ( scaffold_id ) << " " << ( loop_id ) << " " << ( subrange_id ) << " "
				          << internal_subrange.begin() << " " << internal_subrange.end() << " "
				          << epitope_subrange.begin() << "-" << epitope_subrange.end() << " "
				          << offset << std::endl;
				
				// next, scan scaffold for potential graft sites
				utility::vector1< int > viable_begin;
				utility::vector1< int > viable_end;
				FArray3D_float const & da_fc = aligned_decoy.antigen().full_coord(); // for speed
				FArray3D_float const & na_fc = native.antigen().full_coord(); // for speed
				
				// loop over antigen and collect potential graft sites by distance check
				for ( int residue = 1, last_residue = aligned_decoy.antigen().total_residue(); residue <= last_residue; ++residue ) {
					
					// if residue is within site of existing epitope on scaffold, skip it
					if ( scaffold_info.epitope_range().contains( residue ) ) {
						continue;
					}
					
					// distance calculations against CA, use xyzVector pointer constructor
					float ca_dist_begin = distance( numeric::xyzVector_float( &( da_fc( 1, 2, residue ))),
					                                numeric::xyzVector_float( &( na_fc( 1, 2, epitope_subrange.begin() ))));

					float ca_dist_end = distance( numeric::xyzVector_float( &( da_fc( 1, 2, residue ))),
					                              numeric::xyzVector_float( &( na_fc( 1, 2, epitope_subrange.end() ))));
					
					if ( ca_dist_begin < ca_dist_cutoff ) {
						viable_begin.push_back( residue );
					}
					
					if (ca_dist_end < ca_dist_cutoff ) {
						viable_end.push_back( residue );
					}
				}
				
				// skip if an endpoint is completely missing
				if ( ( viable_begin.size() == 0 ) || ( viable_end.size() == 0 ) ) {
					continue;
				}
				
				// test endpoint combinations
				for ( utility::vector1< int >::const_iterator vb = viable_begin.begin(), vbe = viable_begin.end(); vb != vbe; ++vb ) {
					for ( utility::vector1< int >::const_iterator ve = viable_end.begin(), vee = viable_end.end(); ve != vee; ++ve ) {

						// enforce n2c direction
						if ( (*vb) >= (*ve) ) {
							continue;
						}
						
						ResidueRange match_range( *vb, *ve );
						
						// skip if new graft site is too large w/ respect to the graft
						int delta_length = match_range.length() - epitope_subrange.length();
						if ( ( delta_length > 0 ) && ( delta_length > max_loopsize_difference ) ) {
							continue;
						}
						
						// skip if new graft site encompasses existing graft
						if ( match_range.contains( scaffold_info.epitope_range() ) ) {
							continue;
						}
						
						// all filters have passed, let's measure chainbreak
						float nterm_rms = -1.0f;
						float cterm_rms = -1.0f;
						float chainbreak_rms = -1.0f;
						chainbreak( native.antigen(), epitope_subrange, aligned_decoy.antigen(), match_range,
						            nterm_rms, cterm_rms, chainbreak_rms );
						
						// prepare to check intra clash for gapped scaffold
						Pose gapped_scaffold;
						aligned_decoy.gapped_antigen( gapped_scaffold, match_range );
						
						Pose trimmed_epitope;
						native.trimmed_antigen( trimmed_epitope, epitope_subrange );
						
						// convert scaffold to all-gly, right now input to rough match is already
						// all-gly so NO CONVERSION is done
//						convert_to_gly( gapped_scaffold );  // gapped scaffold now all-gly
						
						// put scaffold and epitope together
						Pose epitope_scaffold;
						insert_epitope( aligned_decoy.antigen(), trimmed_epitope, match_range, epitope_scaffold );
						
						// do clash checking
						float intra_clash = 0.0f;
						if ( clash_check ) {
							// fast clash check through octree
							rootstock::BoundingBox bb = epigraft::bounding_box( gapped_scaffold );
							rootstock::Octree< AtomPoint > octree_gs( OCTREE_CUBE_SIZE, bb.lower_corner(), bb.upper_corner() );
							epigraft::fill_octree( octree_gs, gapped_scaffold );
							// intra_clash = inter-repulsion between gapped_scaffold versus trimmed_epitope
							intra_clash = epigraft::octree_inter_rep( octree_gs, gapped_scaffold, trimmed_epitope, false ); // boolean: exclude query termini

							if ( intra_clash > max_intra_clash ) { // no runtime savings, but save on output
								continue;
							}
						}
						
						// output information
						outfile	<< I( 6, scaffold_id ) << ' '  // scaffold
						        << I( 6, loop_id ) << ' '  // epitope
						        << I( 6, subrange_id ) << ' '  // epitope sub-range (loop)
						        << I( 8, native_subrange.begin() ) << ' '                       // begin epitope_range in native epitope numbering
						        << I( 8, native_subrange.end() ) << ' '                         // end   epitope_range in native epitope numbering
						        << I( 5, internal_subrange.begin() ) << ' '                       // begin epitope_range in 1-nres_range numbering
						        << I( 5, internal_subrange.end() ) << ' '                         // end   epitope_range in 1-nres_range numbering
						        << I( 5, match_range.begin() ) << ' '               // scaff numbering 1-nres_scaff
						        << I( 5, match_range.end() ) << ' '                 // scaff numbering 1-nres_scaff
//						        << I( 10, epitope_subrange.begin() + native.Ab().total_residue() ) << ' '           // epitope resnum within nat_cmplx_pose: Ab-epitope cmplx w/Ab first (1-nres_nat_cmplx)
//						        << I( 10, epitope_subrange.end() + native.Ab().total_residue() ) << ' '             // epitope resnum within nat_cmplx_pose: Ab-epitope cmplx w/Ab first (1-nres_nat_cmplx)
//						        << I( 10, match_range.begin() + aligned_decoy().Ab().total_residue() ) << ' '      // scaff resnum within Ab-scaff cmplx w/Ab first (1-nres_Ab_scaff)
//						        << I( 10, match_range.end() + aligned_decoy().Ab().total_residue() ) << ' '        // scaff resnum within Ab-scaff cmplx w/Ab first (1-nres_Ab_scaff)
						        << F( 14, 2, chainbreak_rms ) << ' '                         // chainbreak_rms
						        << F( 14, 2, nterm_rms ) << ' '                              // chainbreak_rms at nterm of graft only
						        << F( 14, 2, cterm_rms ) << ' '                              // chainbreak_rms at cterm of graft only
						        // add information about the loop that was already superposed or grafted on input
						        << I( 8, existing_epitope_id ) << ' '                      // loop_id for loop1 already superposed or grafted
						        << I( 14, scaffold_info.native_epitope_range().begin() ) << ' '    // loop1: begin epitope_range in native epitope numbering
						        << I( 14, scaffold_info.native_epitope_range().end() ) << ' '      // loop1:   end epitope_range in native epitope numbering
						        << I( 10, scaffold_info.epitope_range().begin() ) << ' '   // loop1: begin epitope_range in 1-nres_range numbering
						        << I( 10, scaffold_info.epitope_range().end() ) << ' ';     // loop1:   end epitope_range in 1-nres_range numbering
						if ( clash_check ) {
//							outfile << I( 11, intra_bumps ) << ' '        // intra-bump between gapped scaffold and trimmed epitope
							outfile << F( 11, 2, intra_clash ) << ' ';    // intra-clash between gapped scaffold and trimmed epitope
						} else {
							outfile << I( 11, -1 ) << ' '       // no clash check, so no intra-bumps 
							        << F( 11, 2, -1.0f ) << ' ';   // no clash check, so no intra-clash
						}
						outfile << SS( scaffold_info.input_filename() ) << std::endl;

						// output pdbs if requested
						if ( output_pdbs ) {
							
							// get input filename
							std::string scaff_base_filename = scaffold_info.input_filename();

							// strip terminal .pdb.gz or .pdb to use base_filename in output
							if ( has_suffix( scaff_base_filename, ".pdb.gz" ) ) {
								scaff_base_filename.erase( scaff_base_filename.length() - 7 ); // strip terminal .pdb.gz
							}
							if ( has_suffix( scaff_base_filename, ".pdb" ) ) {
								scaff_base_filename.erase( scaff_base_filename.length() - 4 ); // strip terminal .pdb
							}

							std::string pdb_filename;
							pdb_filename = files_paths::pdb_out_path + scaff_base_filename + "_rmatch_" + string_of( scaffold_id ) + "_"
							                                         + string_of( loop_id ) + "_" + string_of( subrange_id ) + "_"
							                                         + string_of( native_subrange.begin() ) + "-"
							                                         + string_of( native_subrange.end() ) + "_"
							                                         + string_of( match_range.begin() ) + "-"
							                                         + string_of( match_range.end() ) + "_"
							                                         + string_of( chainbreak_rms ) + "_"
							                                         + string_of( existing_epitope_id );

							// dump out a pose that has scaffold first, this will make it easier for visulization.
							aligned_decoy.complex().pdb_info().set_use_pdb_numbering( true );
							aligned_decoy.complex().dump_pdb( pdb_filename + "_.pdb" );

						} // if output_pdbs

					} // for each ve (viable end)
				}  // for each vb (viable begin
				
			} // for each epitope sub-range
			
		} // for each epitope

		if ( using_checkpoint ) {
			update_conformation_already_scored( scaffold_id, checkpoint_filename );
		}

	} // for each scaffold

	// close the output file
	outfile.close();
	outfile.clear();

}


/// @brief load input scaffold info
void
load_scaffold_info(
	std::string filename,
	utility::vector1< ScaffoldInfo > & v
)
{
	utility::io::izstream infile( filename );
	if ( !infile ) {
		std::cout << "load_scaffold_info: cannot open file " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	
	std::string line;
	while ( getline( infile, line ) ) { // read each scaffold

		strip_whitespace(line);
		if ( line == "" ) {
			continue;
		}
		
		std::istringstream ss( line );
		
		std::string scaffold_filename;
		int a, b; // (rosetta) epitope range
		int m, n; // native epitope range
		
		ss >> scaffold_filename >> a >> b >> m >> n;
		
		v.push_back( ScaffoldInfo( scaffold_filename, ResidueRange( a, b ), ResidueRange( m, n ) ) );
	}
	
	// close file
	infile.close();
	
	// report to user
	std::cout << "load_scaffold_info: found " << v.size() << " input scaffolds as follows:" << std::endl;
	for ( int i = 1, ie = v.size(); i <= ie; ++i ) {
		std::cout << "   " << v[ i ].to_string();
	}
}


/// @brief calculate chainbreak score
/// @param[out] nterm_rms
/// @param[out] cterm_rms
/// @param[out] chainbreak_rms
void
chainbreak(
	pose_ns::Pose const & p1,
	ResidueRange const & rr1,
	pose_ns::Pose const & p2,
	ResidueRange const & rr2,
	float & nterm_rms,
	float & cterm_rms,
	float & chainbreak_rms
)
{
	int atomnum_N, atomnum_CA, atomnum_C, atomnum_O, atomnum_CB;
	// always use fullatom numbering!  fullatom flag always true for grafting protocols
	atomnum_N =  1;
	atomnum_CA = 2;
	atomnum_C =  3;
	atomnum_O =  4;
	atomnum_CB = 5;
	
	// Eposition numbering below, for reference
	// atomnum_N =  1;
	// atomnum_CA = 2;
	// atomnum_C =  4;
	// atomnum_O =  5;
	// atomnum_CB = 3;
	
	//closure atoms
	FArray1D_int closure_atom_cterm( 5 );
	closure_atom_cterm( 1 ) = atomnum_C;
	closure_atom_cterm( 2 ) = atomnum_N;
	closure_atom_cterm( 3 ) = atomnum_CA;
	closure_atom_cterm( 4 ) = atomnum_C;
	closure_atom_cterm( 5 ) = atomnum_O;

	FArray1D_int closure_atom_nterm( 5 );
	closure_atom_nterm( 1 ) = atomnum_N;
	closure_atom_nterm( 2 ) = atomnum_CA;
	closure_atom_nterm( 3 ) = atomnum_C;
	closure_atom_nterm( 4 ) = atomnum_O;
	closure_atom_nterm( 5 ) = atomnum_N;

	float dist2_nterm ( 0.0 ), dist2_cterm( 0.0 );
	int atom_count( 0 );
	for ( int i = 1; i <= 5; i++ ) {
		atom_count++;
		
		// residue numbers
		int p1_n = rr1.begin();
		int p1_c = rr1.end();
		int p2_n = rr2.begin();
		int p2_c = rr2.end();
		
		if ( i == 1 ) { // take atom from prior residue
			p1_c--;
			p2_c--;
		} else if ( i == 5 ) { // take atom from next residue
			p1_n++;
			p2_n++;
		}
		
		for ( int k = 1; k <= 3; k++ ) {
			dist2_nterm += square( p2.full_coord()( k, closure_atom_nterm( i ), p2_n )
			                        - p1.full_coord()( k, closure_atom_nterm( i ), p1_n ) );
			dist2_cterm += square( p2.full_coord()( k, closure_atom_cterm( i ), p2_c )
			                        - p1.full_coord()( k, closure_atom_cterm( i ), p1_c ) );
		}
	}
	
	float one_over_atom_count;

	// rms at nterm
	one_over_atom_count = 1.0 / atom_count;
	nterm_rms = std::sqrt( dist2_nterm * one_over_atom_count );

	// rms at cterm
	cterm_rms = std::sqrt( dist2_cterm * one_over_atom_count );

	// total rms
	float dist2_total = dist2_nterm + dist2_cterm;
	one_over_atom_count = 1.0 / ( 2.0 * atom_count );
	chainbreak_rms = std::sqrt( dist2_total * one_over_atom_count );

}

/// @brief compare chain order between two poses
bool
compare_chain_order(
	pose_ns::Pose const & p1,
	pose_ns::Pose const & p2
)
{
	utility::vector1< char > p1_chains;
	utility::vector1< char > p2_chains;

	char current_chain = '~';
	for ( int i = 1, ie = p1.total_residue(); i <= ie; i++ ) {
		
		char c = p1.pdb_info().res_chain( i );
		
		if ( current_chain != c) {
			current_chain = c;
			p1_chains.push_back( c );
		}
	}
	
	current_chain = '~';
	for ( int i = 1, ie = p2.total_residue(); i <= ie; i++ ) {
		
		char c = p2.pdb_info().res_chain( i );
		
		if ( current_chain != c) {
			current_chain = c;
			p2_chains.push_back( c );
		}
	}
	
	return p1_chains == p2_chains;
}

} // namespace epigraft

