// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   old_functions.cc
/// @brief  Old epigraft functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// unit headers
#include <epigraft/old/old_functions.hh>

// package headers
#include <epigraft/epigraft_types.hh>
#include <epigraft/LoopInfo.hh>
#include <epigraft/ResidueRange.hh>

// ObjexxFCL headers
#include <ObjexxFCL/string.functions.hh>

// utility headers
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/vector1.hh>

// C++ headers
#include <string>


namespace epigraft {
namespace old {

/// @brief load epitope ranges
void
load_epitope_subranges(
	std::string const & filename,
	utility::vector1< LoopInfo > & v
)
{
	utility::io::izstream infile( filename );
	if ( !infile ) {
		std::stringstream ss;
		ss << "load_epitope_ranges: cannot open file " << filename << std::endl;
		utility::exit( __FILE__, __LINE__, ss.str() );
	}
	
	Integer id = -1;
	std::string dummy;
	std::string line;
	while ( getline( infile, line ) ) {
		
		ObjexxFCL::strip_whitespace(line);
		if ( line == "" ) {
			continue;
		}
		
		if ( line.substr( 0, 5 ) == "loop:" ) { // new epitope
			std::istringstream loop_stream( line );
			loop_stream >> dummy >> id;
			
			// look for number of residues
			std::string nres_line;
			getline( infile, nres_line );
			
			if ( nres_line.substr( 0, 5 ) != "nres:" ) { // not found, throw error
				std::stringstream ss;
				ss << "Error! load_epitope_ranges: missing nres for loop " << id << std::endl;
				ss << "                            instead got: " << nres_line << std::endl;
				utility::exit( __FILE__, __LINE__, ss.str() );
			}
			
			Integer nres = -1;
			std::istringstream nres_stream( nres_line );
			nres_stream >> dummy >> nres;
			
			// we have enough information to create LoopInfo
			LoopInfo l_info( id, nres );
			
			// look for number of ranges
			std::string nranges_line;
			getline( infile, nranges_line );
			
			if ( nranges_line.substr( 0, 8 ) != "nranges:" ) { // not found, throw error
				std::stringstream ss;
				ss << "Error! load_epitope_ranges: missing nranges for loop " << id << std::endl;
				ss << "                            instead got: " << nranges_line << std::endl;
				utility::exit( __FILE__, __LINE__, ss.str() );
			}
			
			Integer nranges = -1;
			std::istringstream nranges_stream( nranges_line );
			nranges_stream >> dummy >> nranges;
			
			// now cycles through all specified ranges
			for ( Integer i = 1; i <= nranges; ++i ) {
				std::string range_line;
				getline( infile, range_line );
				
				if (range_line.substr( 0, 6 ) != "range:" ) { // not found, throw error
					std::stringstream ss;
					ss << "Error! load_epitope_ranges: malformed range for loop " << id << std::endl;
					ss << "                            instead got: " << range_line << std::endl;
					utility::exit( __FILE__, __LINE__, ss.str() );
				}
				
				Integer a, b; // epitope range
				Integer m, n; // native (epitope) range
				std::istringstream range_stream( range_line );
				range_stream >> dummy >> a >> b >> m >> n;
				
				// add range to current LoopInfo
				l_info.add_native_subrange( ResidueRange( a, b ), ResidueRange( m, n ) );
			}
			
			// sanity check to see if longest subrange is equivalent to the number of residues
			// if not, raise an error flag... note that this does not catch the case where somehow
			// there are multiple subranges that equal the number of residues in the loop, which
			// probably shouldn't happen; we should re-standardize the input file if we want more
			// general input
			if ( l_info.full_native_range().length() != l_info.nres() ) {
				std::stringstream ss;
				ss << "loop " << id << " : nres != length of longest subrange, is there a mistake?";
				utility::exit( __FILE__, __LINE__, ss.str() );
			}
			
			// we're done, add LoopInfo to vector
			v.push_back( l_info );

		} // end new epitope
	} // end master while getline
	
	
	// close input file
	infile.close();
	
	// report to user
	std::cout << "load_epitope_ranges: read " << v.size() << " loops as follows:" << std::endl;
	for ( Integer i = 1, ie = v.size(); i <= ie; i++ ) {
		std::cout << v[ i ].to_string();
	}
}


} // namespace old
} // namespace epigraft
