// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   BoundingBox.hh
/// @brief  Bounding box class; purpose is to avoid coding repetitive min/max and retrieval calls.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef BOUNDINGBOX_HH_
#define BOUNDINGBOX_HH_

// Type headers
#include <rootstock/rootstock_types.hh>


namespace rootstock {

/// @brief  Bounding box class.
class BoundingBox
{
	public: // construct/destruct

		/// @brief default constructor
		inline
		BoundingBox() {}

		/// @brief constructor
		inline
		BoundingBox(
			PointPosition const & pp
		) : lower_( pp ),
		    upper_( lower_ )
		{}

		/// @brief copy constructor
		inline
		BoundingBox(
			BoundingBox const & rval
		) :
			lower_( rval.lower_ ),
			upper_( rval.upper_ )
		{}

		/// @brief default destructor
		inline
		~BoundingBox() {}


	public: // assignment

		inline
		BoundingBox &
		operator =( BoundingBox const & rval )
		{
			if ( this != &rval ) {
				lower_ = rval.lower_;
				upper_ = rval.upper_;
			}
			return *this;
		}


	public: // methods

		/// @brief add a point, expands bounds if necessary
		inline
		void
		add(
			PointPosition const & pp
		)
		{
			lower_.min( pp );
			upper_.max( pp );
		}

		/// @brief get lower corner
		inline
		PointPosition const &
		lower_corner() const
		{
			return lower_;
		}

		/// @brief get upper corner
		inline
		PointPosition const &
		upper_corner() const
		{
			return upper_;
		}

		/// @brief clear corners
		inline
		void
		clear()
		{
			lower_ = PointPosition();
			upper_ = PointPosition();
		}


	private: // data

		PointPosition lower_; // lower corner
		PointPosition upper_; // upper corner
};

} // namespace rootstock

#endif /*BOUNDINGBOX_HH_*/
