// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   ClosedInterval.hh
/// @brief  Template to handle closed intervals.  Immutable.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_rootstock_ClosedInterval_HH_
#define INCLUDED_rootstock_ClosedInterval_HH_

// Type headers
#include <rootstock/rootstock_types.hh>

// C++ headers
#include <sstream>
#include <string>

namespace rootstock {

/// @brief Template to handle closed intervals.  Immutable.
template< typename T >
class ClosedInterval {

	private: // friends
		template< typename > friend class ClosedInterval;

	public: // construct/destruct

		/// @brief default constructor
		inline
		ClosedInterval() {}

		/// @brief constructor
		inline
		ClosedInterval(
			T const & left,
			T const & right
		) : left_( left ),
		    right_( right )
		{
			assert( left <= right ); // ordering
		}

		/// @brief copy constructor
		inline
		ClosedInterval(
			ClosedInterval const & m
		) : left_( m.left_ ),
		    right_( m.right_ )
		{}

		/// @brief copy constructor
		template< typename U >
		inline
		ClosedInterval(
			ClosedInterval< U > const & m
		) : left_( m.left_ ),
		    right_( m.right_ )
		{}

		/// @brief default destructor
		inline
		virtual
		~ClosedInterval() {}


	public: // assignment

		/// @brief copy assignment
		inline
		ClosedInterval &
		operator =( ClosedInterval const & m )
		{
			if ( this != &m ) {
				left_ = m.left_;
				right_ = m.right_;
			}
			return *this;
		}

		/// @brief copy assignment
		template< typename U >
		inline
		ClosedInterval &
		operator =( ClosedInterval< U > const & m )
		{
			if ( this != &m ) {
				left_ = m.left_;
				right_ = m.right_;
			}
			return *this;
		}


	public: // friend operators

		/// @brief ClosedInterval + T
		/// @note  Addition of template type T is enforced, e.g. so if
		/// @note  ClosedInterval is defined with 'int' you cannot add 'double'
		/// @note  without casting.  This can be changed by making a template
		/// @note  function, but it's not clear we want that behavior.
		friend
		inline
		ClosedInterval
		operator +( ClosedInterval const & m, T const & t )
		{
			return ClosedInterval( m.left_ + t, m.right_ + t );
		}

		/// @brief T + ClosedInterval
		/// @note  Addition of template type T is enforced, e.g. so if
		/// @note  ClosedInterval is defined with 'int' you cannot add 'double'
		/// @note  without casting.  This can be changed by making a template
		//  @note  function, but it's not clear we want that behavior.
		friend
		inline
		ClosedInterval
		operator +( T const & t, ClosedInterval const & m )
		{
			return ClosedInterval( t + m.left_, t + m.right_ );
		}

		/// @brief ClosedInterval - T
		/// @note  Addition of template type T is enforced, e.g. so if
		/// @note  ClosedInterval is defined with 'int' you cannot add 'double'
		/// @note  without casting.  This can be changed by making a template
		/// @note  function, but it's not clear we want that behavior.
		friend
		inline
		ClosedInterval
		operator -( ClosedInterval const & m, T const & t )
		{
			return ClosedInterval( m.left_ - t, m.right_ - t );
		}

		/// @brief T - ClosedInterval
		/// @note  Addition of template type T is enforced, e.g. so if
		/// @note  ClosedInterval is defined with 'int' you cannot add 'double'
		/// @note  without casting.  This can be changed by making a template
		//  @note  function, but it's not clear we want that behavior.
		friend
		inline
		ClosedInterval
		operator -( T const & t, ClosedInterval const & m )
		{
			return ClosedInterval( t - m.left_, t - m.right_ );
		}

		/// @brief T < ClosedInterval
		friend
		inline
		bool
		operator <( T const & s, ClosedInterval const & m ) {
			return s < m.left_ && s < m.right_;
		}

		/// @brief ClosedInterval < T
		friend
		inline
		bool
		operator <( ClosedInterval const & m, T const & s ) {
			return m.left_ < s && m.right_ < s;
		}

		/// @brief T > ClosedInterval
		friend
		inline
		bool
		operator >( T const & s, ClosedInterval const & m ) {
			return s > m.left_ && s > m.right_;
		}

		/// @brief ClosedInterval > T
		friend
		inline
		bool
		operator >( ClosedInterval const & m, T const & s ) {
			return m.left_ > s && m.right_ > s;
		}


	public: // member operator

		/// @brief this == ClosedInterval, lexicographic order
		inline
		virtual
		bool
		operator ==(ClosedInterval const & rvalue ) const
		{
			return ( left_ == rvalue.left_ ) && ( right_ == rvalue.right_ );
		}

		/// @brief this != ClosedInterval, lexicographic order
		inline
		virtual
		bool
		operator !=(ClosedInterval const & rvalue ) const
		{
			return ( left_ != rvalue.left_ ) || ( right_ != rvalue.right_ );
		}

		/// @brief this < ClosedInterval, lexicographic order
		inline
		virtual
		bool
		operator <(ClosedInterval const & rvalue ) const
		{
			return (
			 ( left_ < rvalue.left_ ? true :
			 ( rvalue.left_ < left_ ? false : // left_ == rvalue.left_
			 ( right_ < rvalue.right_ ) ) ) );
		}

		/// @brief this <= ClosedInterval, lexicographic order
		inline
		virtual
		bool
		operator <=(ClosedInterval const & rvalue ) const
		{
			return (
			 ( left_ < rvalue.left_ ? true :
			 ( rvalue.left_ < left_ ? false : // left_ == rvalue.left_
			 ( right_ <= rvalue.right_ ) ) ) );
		}

		/// @brief this > ClosedInterval, lexicographic order
		inline
		virtual
		bool
		operator >(ClosedInterval const & rvalue ) const
		{
			return (
			 ( left_ > rvalue.left_ ? true :
			 ( rvalue.left_ > left_ ? false : // left_ == rvalue.left_
			 ( right_ > rvalue.right_ ) ) ) );
		}

		/// @brief this >= ClosedInterval, lexicographic order
		inline
		virtual
		bool
		operator >=(ClosedInterval const & rvalue ) const
		{
			return (
			 ( left_ > rvalue.left_ ? true :
			 ( rvalue.left_ > left_ ? false : // left_ == rvalue.left_
			 ( right_ >= rvalue.right_ ) ) ) );
		}


	public: // accessors

		/// @brief get left endpoint
		inline
		T const &
		left() const
		{
			return left_;
		}

		/// @brief get left endpoint
		inline
		T const &
		begin() const
		{
			return left_;
		}

		/// @brief get right endpoint
		inline
		T const &
		right() const
		{
			return right_;
		}

		/// @brief get right endpoint
		inline
		T const &
		end() const
		{
			return right_;
		}


	public: // query

		/// @brief length of interval
		inline
		T
		length() const
		{
			return right_ - left_;
		}

		/// @brief is this interval contained within interval m?
		inline
		bool
		within(
			ClosedInterval const & m
		) const
		{
			return(
			 ( left_ < m.left_ ? false :    // left endpoint lies outside
			 ( right_ > m.right_ ? false :  // right endpoint lies outside
			 ( true ) ) ) );
		}

		/// @brief does this interval contain interval m?
		inline
		bool
		contains(
			ClosedInterval const & m
		) const
		{
			return(
			 ( m.left_ < left_ ? false :    // left endpoint of m lies outside
			 ( m.right_ > right_ ? false :  // right endpoint of m lies outside
			 ( true ) ) ) );
		}

		/// @brief does this interval contain point t?
		inline
		bool
		contains(
			T const & t
		) const
		{
			return ( left_ <= t ) && ( t <= right_ );
		}

		/// @brief do the intervals overlap?
		inline
		bool
		overlaps(
			ClosedInterval const & m
		) const
		{
			return !( left_ > m.right_ || m.left_ > right_ );
		}

		/// @brief are these two intervals equivalent?
		inline
		bool
		equals(
			ClosedInterval const & m
		) const
		{
			return ( left_ == m.left_ ) && ( right_ == m.right_ );
		}


	public: // status

		inline
		std::string
		to_string() const
		{
			std::stringstream ss;
			ss << "[" << left_ << ", " << right_ << "]";
			return ss.str();
		}


	protected: // member data

		T left_;
		T right_;

};

} // namespace rootstock

#endif /*INCLUDED_rootstock_ClosedInterval_HH_*/
