// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   IntegerUnionFind.hh
/// @brief  Simple, fast integer-only union-find.
/// @note   An equivalent template class could be constructed but this requires a standardized
/// @note   hash map class, which doesn't seem to exist yet across different compiler
/// @note   implementations.  std::map could be used in place, but then find() operations are
/// @note   log(n) instead of O(1).
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INTEGERUNIONFIND_HH_
#define INTEGERUNIONFIND_HH_

// package headers
#include <rootstock/rootstock_types.hh>


namespace rootstock {


// struct for internal use only
struct UFRecord {
	typedef rootstock::SSize SSize;
	SSize p; // parent node
};


/// @brief  Simple, fast integer-only union-find.
class IntegerUnionFind
{

	public: // typedefs

		typedef rootstock::SSize SSize;


	public: // construct/destruct

		/// @brief constructor
		/// @note  creates records indexed from 0 .. (n-1)
		IntegerUnionFind(
			SSize n
		)
		{
			uf_ = new UFRecord[ n ];
		}

		/// @brief default constructor
		IntegerUnionFind();

		/// @brief default destructor
		~IntegerUnionFind()
		{
			delete []uf_;
		}


	public: // methods

		/// @brief add element
		inline
		void
		add(
			SSize i
		)
		{
			uf_[ i ].p = -1;
		}

		/// @brief find root of set
		inline
		SSize
		find(
			SSize i
		)
		{
			SSize j, t;
			j = i;

			// find root of set
			while ( uf_[ j ].p >= 0 )
			{
					j = uf_[ j ].p;
			}

			// path compression
			while ( uf_[ i ].p >= 0 )
			{
				t = i;
				i = uf_[ i ].p;
				uf_[ t ].p = j;
			}

			return j;
		}

		/// @brief union sets, returns true if union occurred, otherwise false
		/// @note  'union' is reserved keyword in C/C++ so can't use that as method name
		inline
		bool
		do_union(
			SSize i,
			SSize j
		)
		{
			i = find( i );
			j = find( j );

			if ( i == j ) {
				return false;
			} else {
				if ( uf_[ i ].p >= uf_[ j ].p ) {
					// swap, i must be the larger set
					Size tmp = i;
					i = j;
					j = tmp;
				}

				uf_[ i ].p = uf_[ i ].p + uf_[ j ].p;
				uf_[ j ].p = i;

				return true;
			}
		}


	private: // data

		UFRecord * uf_; // array of UFRecords

};

} // namespace rootstock

#endif /* INTEGERUNIONFIND_HH_ */
