// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   geometry.hh
/// @brief  Various geometry functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_rootstock_geometry_HH_
#define INCLUDED_rootstock_geometry_HH_

// type headers
#include <rootstock/rootstock_types.hh>

// numeric headers
#include <numeric/conversions.hh>
#include <numeric/Quaternion.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

namespace rootstock {


/// @brief   computes orientation of tetrahedron 'abcd', positive if 'd' lies below CCW 'abc'
/// @details Computes orientation of tetrahedron 'abcd' by essentially computing the
/// @details signed volume (4x4 determinant).  Let 'abc' be an oriented plane, where
/// @details 'a', 'b', and 'c' appear in counter-clockwise order when viewed from above the
/// @details plane.  Then if 'd' is below 'abc', the function returns a positive value.
/// @details If 'd' is above 'abc', the function returns a negative value.  When the points
/// @details are coplanar the function returns zero.
/// @return  6 * signed volume
/// @warning This function is not robust, do not use in geometric algorithms!
/// @warning For proper computation, see keywords: robust predicates, exact
/// @warning arithmetic, adaptive arithmetic, filtered arithmetic,
/// @warning symbolic perturbation, arbitrary precision arithmetic
template< typename T >
T
orient3D(
	numeric::xyzVector< T > const & a,
	numeric::xyzVector< T > const & b,
	numeric::xyzVector< T > const & c,
	numeric::xyzVector< T > const & d
)
{
	using numeric::xyzMatrix;
	
	// computes the following determinant:
	// | a_x  a_y  a_z  1 |     | a_x - d_x, a_y - d_y, a_z - d_z |
	// | b_x  b_y  b_z  1 |  =  | b_x - d_x, b_y - d_y, b_z - d_z | = result
	// | c_x  c_y  c_z  1 |     | c_x - d_x, c_y - d_y, c_z - d_z |
	// | d_x  d_y  d_z  1 |
	
	xyzMatrix< T > m = xyzMatrix< T >::rows_constructor( a - d,
	                                                     b - d,
	                                                     c - d );
	
	return m.det();
}


/// @brief construct quaternion from axis-angle pair
/// @brief input axis does not have to be unit vector since this is enforced within
/// @brief the function
template< typename T >
numeric::Quaternion< T >
quaternion(
	numeric::xyzVector< T > axis, // pass by value since axis might be modified
	T const & angle
)
{
	if ( !axis.is_unit() ) {
		axis.normalize();
	}
	
	T s = std::sin( angle / 2.0 );
	return numeric::Quaternion< T >( std::cos( angle / 2.0 ), axis.x() * s, axis.y() * s, axis.z() * s );
}


/// @brief construct quaternion from x,y,z euler angles
template< typename T >
numeric::Quaternion< T >
quaternion(
	T const & x_angle,
	T const & y_angle,
	T const & z_angle
)
{
	using numeric::conversions::radians;

	// unit vectors specifying standard axes
	static numeric::xyzVector< T > x_axis( 1.0, 0.0, 0.0 );
	static numeric::xyzVector< T > y_axis( 0.0, 1.0, 0.0 );
	static numeric::xyzVector< T > z_axis( 0.0, 0.0, 1.0 );

	// specified rotations
	numeric::Quaternion< T > x_rot = quaternion( x_axis, radians( x_angle ) );
	numeric::Quaternion< T > y_rot = quaternion( y_axis, radians( y_angle ) );
	numeric::Quaternion< T > z_rot = quaternion( z_axis, radians( z_angle ) );

	// return composite xyz rotation for monomer1
	return z_rot * y_rot * x_rot;
}


} // namespace rootstock

#endif /*INCLUDED_rootstock_geometry_HH_*/
