// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   FixedFrameTransformSequence.cc
/// @brief  Creates and maintain a sequence of rigid body transforms.
/// @brief  Transforms are meant to be applied against a single fixed frame/state.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

// Unit headers
#include <rootstock/FixedFrameTransformSequence.hh>


namespace rootstock {

////////////////////////////////
/* Public: Construct/Destruct */
////////////////////////////////

/// @brief constructor
FixedFrameTransformSequence::FixedFrameTransformSequence() {}

/// @brief destructor
FixedFrameTransformSequence::~FixedFrameTransformSequence() {}


///////////////////
/* Public: Moves */
///////////////////

/// @brief roll/wedge move
void
FixedFrameTransformSequence::push_rotate_X(
	Point const & vo,
	Point const & ve,
	Angle const & angle,
	Size const & step
)
{
	Point m = midpoint(vo, ve); // midpoint of vo->ve, use as origin
	
	Vector axis = ve - vo; // rotation axis
	axis.normalize();
	
	Angle ai = angle / step; // angle increment
	
	for ( Size s = 1; s <= step; ++s ) {
		push_back( FixedFrameTransform( m , RigidTransform( quaternion( axis, s * ai ) ) ) );
	}
}

/// @brief pitch/pendulum move
void
FixedFrameTransformSequence::push_rotate_Y(
	Point const & rp,
	Point const & vo,
	Point const & ve,
	Angle const & angle,
	Size const & step
)
{
	Point m = midpoint(vo, ve); // midpoint of vo->ve, use as origin
	
	Point u = perpendicular_foot( rp, vo, ve ); // perpendicular foot, point u
	
	Vector u_rp = rp - u; // vector along the altitude
	
	Vector vo_ve = ve - vo;
	
	Vector axis = vo_ve.cross_product( u_rp ); // rotation axis in plane
	axis.normalize();
	
	Angle ai = angle / step; // angle increment

	for ( Size s = 1; s <= step; ++s ) {
		push_back( FixedFrameTransform( m , RigidTransform( quaternion( axis, s * ai ) ) ) );
	}
}

/// @brief yaw/rock move
void
FixedFrameTransformSequence::push_rotate_Z(
	Point const & rp,
	Point const & vo,
	Point const & ve,
	Angle const & angle,
	Size const & step
)
{
	Point m = midpoint(vo, ve); // midpoint of vo->ve, use as origin
	
	Point u = perpendicular_foot( rp, vo, ve ); // perpendicular foot, point u
	
	Vector axis = rp - u; // rotation axis u->rp
	axis.normalize();
	
	Angle ai = angle / step; // angle increment
	
	for ( Size s = 1; s <= step; ++s ) {
		push_back( FixedFrameTransform( m , RigidTransform( quaternion( axis, s * ai ) ) ) );
	}
}

/// @brief move along x-axis
void
FixedFrameTransformSequence::push_translate_X(
	Point const & vo,
	Point const & ve,
	Distance const & d,
	Size const & step
)
{
	Vector xx = ve - vo; // x-axis
	xx.normalize();
	
	Distance di = d / step;
	
	for ( Size i = 1; i <= step; ++i ) {
		push_back( FixedFrameTransform( Translation( 0, 0, 0 ), RigidTransform( Quaternion(), i * di * xx ) ) );
	}
}

/// @brief move along y-axis
void
FixedFrameTransformSequence::push_translate_Y(
	Point const & rp,
	Point const & vo,
	Point const & ve,
	Distance const & d,
	Size const & step
)
{
	Vector xx = ve - vo; // x-axis
	
	Point u = perpendicular_foot( rp, vo, ve ); // perpendicular foot, point u
	
	Vector zz = rp - u; // vector along the altitude, z-axis
	
	Vector yy = xx.cross_product( zz ); // axis in plane, y-axis
	yy.normalize();
	
	Distance di = d / step;
	
	for ( Size i = 1; i <= step; ++i ) {
		push_back( FixedFrameTransform( Translation( 0, 0, 0 ), RigidTransform( Quaternion(), i * di * yy ) ) );
	}
}

/// @brief move along z-axis
void
FixedFrameTransformSequence::push_translate_Z(
	Point const & rp,
	Point const & vo,
	Point const & ve,
	Distance const & d,
	Size const & step
)
{
	Point u = perpendicular_foot( rp, vo, ve ); // perpendicular foot, point u
	
	Vector zz = rp - u; // vector along the altitude, z-axis
	zz.normalize();
	
	Distance di = d / step;
	
	for ( Size i = 1; i <= step; ++i ) {
		push_back( FixedFrameTransform( Translation( 0, 0, 0 ), RigidTransform( Quaternion(), i * di * zz ) ) );
	}
}

/// @brief cube move
/// @note  move generation will also include the original position
void
FixedFrameTransformSequence::push_cube(
	Point const & rp,
	Point const & vo,
	Point const & ve,
	Length const & side,
	Size const & step
)
{
	Point m = midpoint(vo, ve); // midpoint of vo->ve, use as origin
	
	Point u = perpendicular_foot( rp, vo, ve ); // perpendicular foot, point u
	
	Vector zz = rp - u; // vector along the altitude, z-axis
	zz.normalize();
	
	Vector xx = ve - vo; // x-axis
	xx.normalize();
	
	Vector yy = xx.cross_product( zz ); // axis in plane, y-axis
	yy.normalize();
	
	Length si = side / step; // side increment
	
	SSize sstep = step;	
	for ( SSize ix = -sstep; ix <= sstep; ++ix ) {
		for ( SSize iy = -sstep; iy <= sstep; ++iy ) {
			for ( SSize iz = -sstep; iz <= sstep; ++iz ) {
				
				push_back( FixedFrameTransform( Translation( 0, 0, 0 ),
				                        RigidTransform( Quaternion(),
				                                        ( ix * si * xx ) + ( iy * si * yy ) + ( iz * si * zz ) ) ) );
				                                        
			}
		}
	}
}

} // namespace rootstock