// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   FixedFrameTransformSequence.hh
/// @brief  Creates and maintain a sequence of rigid body transforms.
/// @brief  Transforms are meant to be applied against a single fixed frame/state.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef FIXEDFRAMETRANSFORMSEQUENCE_HH_
#define FIXEDFRAMETRANSFORMSEQUENCE_HH_

// Package headers
#include <rootstock/rootstock_types.hh>

// Numeric headers
#include <numeric/BodyPosition.hh>
#include <numeric/constants.hh>
#include <numeric/Quaternion.hh>
#include <numeric/xyzTriple.hh>
#include <numeric/xyzVector.hh>

// C++ headers
#include <cmath>
#include <utility>
#include <vector>


namespace rootstock {

typedef numeric::BodyPosition< Real > RigidTransform;
typedef RigidTransform::Translation Translation;
typedef std::pair< Translation, RigidTransform > FixedFrameTransform;

/// @brief  Creates and maintain a sequence of rigid body transforms.  Subclasses std::vector< T >, see source for type.
class FixedFrameTransformSequence : public std::vector< FixedFrameTransform >
{
	
	public: // typedefs

		typedef numeric::xyzVector< Real > Point;
		typedef numeric::xyzVector< Real > Vector;
		typedef numeric::Quaternion< Real > Quaternion;
	
	public: // construct/destruct

		/// @brief constructor
		FixedFrameTransformSequence();

		/// @brief destructor
		~FixedFrameTransformSequence();
	
	public: // moves
	
		/// @brief roll/wedge move
		void
		push_rotate_X(
			Point const & vo,
			Point const & ve,
			Angle const & angle,
			Size const & step
		);

		/// @brief pitch/pendulum move
		void
		push_rotate_Y(
			Point const & rp,
			Point const & vo,
			Point const & ve,
			Angle const & angle,
			Size const & step
		);

		/// @brief yaw/rock move
		void
		push_rotate_Z(
			Point const & rp,
			Point const & vo,
			Point const & ve,
			Angle const & angle,
			Size const & step
		);

		/// @brief move along x-axis
		void
		push_translate_X(
			Point const & vo,
			Point const & ve,
			Distance const & d,
			Size const & step
		);

		/// @brief move along y-axis
		void
		push_translate_Y(
			Point const & rp,
			Point const & vo,
			Point const & ve,
			Distance const & d,
			Size const & step
		);

		/// @brief move along z-axis
		void
		push_translate_Z(
			Point const & rp,
			Point const & vo,
			Point const & ve,
			Distance const & d,
			Size const & step
		);

		/// @brief cube move
		void
		push_cube(
			Point const & rp,
			Point const & vo,
			Point const & ve,
			Length const & side,
			Size const & step
		);

	private: // triangle geometry
	
		/// @brief perpendicular foot
		/// @note  Finds perpendicular foot along vector vo->ve for a triangle defined by two vectors vo->ve and vo->rp.
		/// @note  Depending upon the angle between vo->ve and vo->rp, it might not be the perpendicular foot.
		/// @note  If the angle is 90 degrees, then vo is returned.  If the angle is obtuse, then the point that drops
		/// @note  down from rp to make a right angle is returned.
		Point
		perpendicular_foot(
			Point const & rp,
			Point const & vo,
			Point const & ve
		)
		{
			Vector vo_ve = ve - vo; // vo -> ve
			Vector vo_rp = rp - vo; // vo -> rp
			
			Real rp_vo_ve = angle_of( vo_rp, vo_ve );
			
			if ( rp_vo_ve > numeric::constants::d::pi_over_2 ) { // obtuse triangle

				rp_vo_ve = numeric::constants::d::pi - rp_vo_ve;
				Angle vo_rp_u = numeric::constants::d::pi_over_2 - rp_vo_ve; // let u denote perpendicular foot
				
				Length scale = vo_rp.length() * std::sin( vo_rp_u ); // length of desired vector along vo -> ve
				vo_ve.normalize();
				return -scale * vo_ve + vo; // obtuse triangle needs reversal of direction
				
			} else if ( rp_vo_ve < numeric::constants::d::pi_over_2 ) { // acute triangle
				
				Angle vo_rp_u = numeric::constants::d::pi_over_2 - rp_vo_ve; // let u denote perpendicular foot
				
				Length scale = vo_rp.length() * std::sin( vo_rp_u ); // length of desired vector along vo -> ve
				vo_ve.normalize();
				return scale * vo_ve + vo;
				
			}
			
			return vo; // already right triangle
		}

	private: // methods
	
		/// @brief construct quaternion from axis-angle pair
		/// @note  input axis does not have to be unit vector since this is enforced within the function
		Quaternion
		quaternion(
			Vector axis, // pass by value since axis might be modified
			Angle const & angle
		)
		{
			if ( !axis.is_unit() ) {
				axis.normalize();
			}
			Real s = std::sin( angle / 2.0 );
			return Quaternion( std::cos( angle / 2.0 ), axis.x() * s, axis.y() * s, axis.z() * s );
		}
		
};

} // namespace rootstock

#endif /*FIXEDFRAMETRANSFORMSEQUENCE_HH_*/
