// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/io/ozstream.cc
/// @brief  Output file stream wrapper for uncompressed and compressed files
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author David Kim (dekim@u.washington.edu)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


// Unit header
#include <utility/io/ozstream.hh>

// Project headers
#include <utility/file/file_sys_util.hh>

// C++ headers
#include <cstdlib>


namespace utility {
namespace io {


	/// @brief Open a file
	void
	ozstream::open(
		std::string const & filename_a,
		std::ios_base::openmode open_mode // Ignored for gzip files
	)
	{
		using std::ios;
		using std::ios_base;
		using utility::file::file_extension;
		using utility::file::trytry_ofstream_open;
		using zlib_stream::zip_ostream;

		// Close the file if open and reset the state
		close();

		// Open the ofstream and set the compression state and file name
		filename_ = filename_a;
		if ( ( open_mode & ios::ate ) || ( open_mode & ios::app )
		 || ( ( open_mode & ios::in ) && ( open_mode & ios::out ) ) ) {

			// prepare new character buffer -- must do this before file is opened
			allocate_assign_char_buffer();

			// Unsupported for gzip files: Use ofstream
			trytry_ofstream_open( of_stream_, filename_a, open_mode );

			compression_ = UNCOMPRESSED;

		} else if ( file_extension( filename_a ) == "gz" ) { // gzip file

			trytry_ofstream_open( of_stream_, filename_a, ios_base::out|ios_base::binary );
			if ( of_stream_ ) { // Open succeeded
				compression_ = GZIP;
			} else { // Leave stream state so that failure can be detected
				compression_ = NONE;
			}

		} else { // Uncompressed file

			// prepare new character buffer -- must do this before file is opened
			allocate_assign_char_buffer();

			trytry_ofstream_open( of_stream_, filename_a, ios_base::out );
			if ( of_stream_ ) { // Open succeeded
				compression_ = UNCOMPRESSED;
			} else { // Leave stream state so that failure can be detected
				compression_ = NONE;
			}

		}

		// Attach zip_ostream to ofstream if gzip file
		if ( compression_ == GZIP ) {
			// zip_stream_p_ deleted by close() above so don't have to here
			zip_stream_p_ = new zip_ostream( of_stream_, true, static_cast< size_t >( Z_DEFAULT_COMPRESSION ), zlib_stream::DefaultStrategy, 15, 8, buffer_size_ );
			if ( ( !zip_stream_p_ ) || ( !( *zip_stream_p_ ) ) ||
			 ( !zip_stream_p_->is_gzip() ) ) { // zip_stream not in good state
				delete zip_stream_p_; zip_stream_p_ = 0;
				of_stream_.close();
				// Set failbit so failure can be detected
				of_stream_.setstate( ios_base::failbit );
			}
		}
	}


	/// @brief Open a text file or gzip'd file for appending
	void
	ozstream::open_append( std::string const & filename_a )
	{
		using std::cout;
		using std::endl;
		using std::exit;
		using std::ios_base;
		using utility::file::file_extension;
		using utility::file::trytry_ofstream_open;
		using zlib_stream::zip_ostream;

		// Close the file if open and reset the state
		close();

		// Open the ofstream and set the compression state and file name
		filename_ = filename_a;
		if ( file_extension( filename_a ) == "gz" ) { // gzip file

			trytry_ofstream_open( of_stream_, filename_a, ios_base::out|ios_base::binary|ios_base::app );
			if ( of_stream_ ) { // Open succeeded
				compression_ = GZIP;
			} else { // Leave stream state so that failure can be detected
				compression_ = NONE;
			}

		} else { // Uncompressed file

			// prepare new character buffer -- must do this before file is opened
			allocate_assign_char_buffer();

			trytry_ofstream_open( of_stream_, filename_a, ios_base::out|ios_base::app );
			if ( of_stream_ ) { // Open succeeded
				compression_ = UNCOMPRESSED;
			} else { // Leave stream state so that failure can be detected
				compression_ = NONE;
			}

		}

		// Attach zip_ostream to ofstream if gzip file
		if ( compression_ == GZIP ) {
			// zip_stream_p_ deleted by close() above so don't have to here
			zip_stream_p_ = new zip_ostream( of_stream_, true, static_cast< size_t >( Z_DEFAULT_COMPRESSION ), zlib_stream::DefaultStrategy, 15, 8, buffer_size_ );
			if ( ( !zip_stream_p_ ) || ( !( *zip_stream_p_ ) ) ||
			 ( !zip_stream_p_->is_gzip() ) ) { // zip_stream not in good state
				delete zip_stream_p_; zip_stream_p_ = 0;
				of_stream_.close();
				// Set failbit so failure can be detected
				of_stream_.setstate( ios_base::failbit );
			}
		}
	}


} // namespace io
} // namespace utility
